!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                                                                   !!
!!                   GNU General Public License                      !!
!!                                                                   !!
!! This file is part of the Flexible Modeling System (FMS).          !!
!!                                                                   !!
!! FMS is free software; you can redistribute it and/or modify       !!
!! it and are expected to follow the terms of the GNU General Public !!
!! License as published by the Free Software Foundation.             !!
!!                                                                   !!
!! FMS is distributed in the hope that it will be useful,            !!
!! but WITHOUT ANY WARRANTY; without even the implied warranty of    !!
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     !!
!! GNU General Public License for more details.                      !!
!!                                                                   !!
!! You should have received a copy of the GNU General Public License !!
!! along with FMS; if not, write to:                                 !!
!!          Free Software Foundation, Inc.                           !!
!!          59 Temple Place, Suite 330                               !!
!!          Boston, MA  02111-1307  USA                              !!
!! or see:                                                           !!
!!          http://www.gnu.org/licenses/gpl.txt                      !!
!!                                                                   !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
module atmos_tracer_driver_mod
! <CONTACT EMAIL="wfc@gfdl.noaa.gov">
!   William Cooke
! </CONTACT>

! <REVIEWER EMAIL="mjh@gfdl.noaa.gov">
!   Matt Harrison
! </REVIEWER>

! <REVIEWER EMAIL="bw@gfdl.noaa.gov">
!   Bruce Wyman
! </REVIEWER>

! <HISTORY SRC="http://www.gfdl.noaa.gov/fms-cgi-bin/cvsweb.cgi/FMS/"/>

! <OVERVIEW>
!     This code allows the user to easily add tracers to the FMS framework.
! </OVERVIEW>

! <DESCRIPTION>
!
!     This code allows a user to easily implement tracer code in the FMS
!     framework.  The tracer and tracer tendency arrays are supplied along
!     with longtitude,  latitude, wind, temperature, and pressure
!     information which allows a  user to implement sources and sinks of the
!     tracer which depend on these parameters.
!
!     In the following example, radon being implemented in the atmosphere
!     will be used as an example of how to implement a tracer in the FMS
!     framework.
!
!     Within the global scope of tracer_driver_mod a use
!     statement should be inserted for each tracer to be added.
!<PRE>      use radon_mod, only : radon_sourcesink, radon_init, radon_end </PRE>
!     
!     An integer parameter, which  will be used as an identifier for the
!     tracer, should be assigned.
!<PRE>
!      integer :: nradon 
!</PRE>
!     Within tracer_driver_init a call to the tracer manager is needed in
!     order  to identify which tracer it has set the tracer as.
!<PRE>
!      nradon = get_tracer_index(MODEL_ATMOS,'radon')
!</PRE>
!     Here MODEL_ATMOS is a parameter defined in field_manager. 
!          'radon' is the name of the tracer within the field_table.
!     
!     If the tracer exists then the integer returned will be positive and it
!     can be used to call the initialization routines for the individual
!     tracers.
!<PRE>
!      if (nradon > 0) then
!           call radon_init(Argument list)
!      endif
!</PRE>     
!
!     Within tracer_driver the user can also use the identifier to surround
!     calls to the source-sink routines for the tracer of interest.
!
!<PRE>
!      if (nradon > 0 .and. nradon <= nt) then
!          call radon_sourcesink (Argument list)
!          rdt(:,:,:,nradon)=rdt(:,:,:,nradon)+rtnd(:,:,:)
!      endif
!</PRE>
!
!     It is the users responsibility to add the tendency generated by the
!     sourcesink routine.
!      
!     Within tracer_driver_end the user can add calls to the
!     terminators for the appropriate source sink routines.
!
!<PRE>      call radon_end</PRE>
!
!     This may simply be a deallocation statement or a routine to send
!     output to the logfile stating that the termination routine has been
!     called.
! </DESCRIPTION>


!-----------------------------------------------------------------------

use types_mod, only : r8
use              fms_mod, only : file_exist, &
                                 write_version_number, &
                                 error_mesg, &
                                 FATAL, &
                                 stdlog
use     time_manager_mod, only : time_type
use   tracer_manager_mod, only : get_tracer_index,   &
                                 get_tracer_names,   &
                                 get_tracer_indices
use    field_manager_mod, only : MODEL_ATMOS
use atmos_tracer_utilities_mod, only : wet_deposition,     &
                                 dry_deposition,     &
                                 atmos_tracer_utilities_init
use        constants_mod, only : grav
use      atmos_radon_mod, only : atmos_radon_sourcesink,   &
                                 atmos_radon_init,         &
                                 atmos_radon_end
use atmos_carbon_aerosol_mod, only : atmos_blackc_sourcesink,  &
                                 atmos_organic_sourcesink, &
                                 atmos_carbon_aerosol_init,&
                                 atmos_carbon_aerosol_end
use atmos_sulfur_hex_mod, only : atmos_sf6_sourcesink,     &
                                 atmos_sulfur_hex_init,    &
                                 atmos_sulfur_hex_end

implicit none
private
!-----------------------------------------------------------------------
!----- interfaces -------

public  atmos_tracer_driver, atmos_tracer_driver_init, atmos_tracer_driver_end

!-----------------------------------------------------------------------
!----------- namelist -------------------
!-----------------------------------------------------------------------
!
!  When initializing additional tracers, the user needs to make the
!  following changes.
!
!  Add an integer variable below for each additional tracer. 
!  This should be initialized to zero. 
!
!-----------------------------------------------------------------------

integer :: nradon=0  ! tracer number for radon
integer :: nbcphobic=0
integer :: nbcphilic=0
integer :: nocphobic=0
integer :: nocphilic=0
integer :: nclay    =0
integer :: nsilt    =0
integer :: nseasalt =0
integer :: nsf6     =0

character(len=6), parameter :: module_name = 'tracer'

logical :: module_is_initialized = .FALSE.


integer, allocatable :: local_indices(:) 
! This is the array of indices for the local model. 
! local_indices(1) = 5 implies that the first local tracer is the fifth
! tracer in the tracer_manager.
  
!-----------------------------------------------------------------------
type(time_type) :: Time

!---- version number -----
character(len=128) :: version = '$Id: atmos_tracer_driver.f90,v 1.2 2004/12/03 22:14:09 jla Exp $'
character(len=128) :: tagname = '$Name:  $'
!-----------------------------------------------------------------------

contains

!#######################################################################

! <SUBROUTINE NAME="atmos_tracer_driver">
!   <OVERVIEW>
!     A routine which allows tracer code to be called.
!   </OVERVIEW>
!   <DESCRIPTION>
!     This subroutine calls the source sink routines for atmospheric
!     tracers. This is the interface between the dynamical core of the 
!     model and the tracer code. It should supply all the necessary 
!     information to a user that they need in order to calculate the 
!     tendency of that tracer with respect to emissions or chemical losses.
!
!   </DESCRIPTION>
!   <TEMPLATE>
!     call atmos_tracer_driver (is, ie, js, je, Time, lon, lat, land, phalf, pfull, r,  &
!                           u, v, t, q, u_star, rdt, rm, rdiag, kbot)
!   </TEMPLATE>
!   <IN NAME="is, ie, js, je" TYPE="integer">
!     Local domain boundaries.
!   </IN>
!   <IN NAME="Time" TYPE="type(time_type)">
!     Model time.
!   </IN>
!   <IN NAME="lon" TYPE="real(r8)" DIM="(:,:)">
!     Longitude of the centre of the model gridcells
!   </IN>
!   <IN NAME="lat" TYPE="real(r8)" DIM="(:,:)">
!     Latitude of the centre of the model gridcells
!   </IN>
!   <IN NAME="land" TYPE="logical" DIM="(:,:)">
!     Land/sea mask.
!   </IN>
!   <IN NAME="phalf" TYPE="real(r8)" DIM="(:,:,:)">
!     Pressures on the model half levels.
!   </IN>
!   <IN NAME="pfull" TYPE="real(r8)" DIM="(:,:,:)">
!     Pressures on the model full levels.
!   </IN>
!   <IN NAME="r" TYPE="real(r8)" DIM="(:,:,:,:)">
!     The tracer array in the component model.
!   </IN>
!   <IN NAME="u" TYPE="real(r8)" DIM="(:,:,:)">
!     Zonal wind speed.
!   </IN>
!   <IN NAME="v" TYPE="real(r8)" DIM="(:,:,:)">
!     Meridonal wind speed.
!   </IN>
!   <IN NAME="t" TYPE="real(r8)" DIM="(:,:,:)">
!     Temperature.
!   </IN>
!   <IN NAME="q" TYPE="real(r8)" DIM="(:,:,:)">
!     Specific humidity. This may also be accessible as a
!                        portion of the tracer array.
!   </IN>
!   <IN NAME="u_star" TYPE="real(r8)" DIM="(:,:)">
!     Friction velocity :: 
!     The magnitude of the wind stress is density*(ustar**2)
!     The drag coefficient for momentum is u_star**2/(u**2+v**2)
!   </IN>
!   <INOUT NAME="rdt" TYPE="real(r8)" DIM="(:,:,:,:)">
!     The tendency of the tracer array in the compenent
!     model. The tendency due to sources and sinks computed
!     in the individual tracer routines should be added to
!     this array before exiting tracer_driver.
!   </INOUT>
!   <IN NAME="rm" TYPE="real(r8)" DIM="(:,:,:,:)">
!     The tracer array in the component model for the previous timestep.
!   </IN>
!   <INOUT NAME="rdiag" TYPE="real(r8)" DIM="(:,:,:,:)">
!     The array of diagnostic tracers. As these may be changed within the
!     tracer routines for diagnostic purposes, they need to be writable.
!   </INOUT>
!   <IN NAME="kbot" TYPE="integer, optional" DIM="(:,:)">
!     Integer array describing which model layer intercepts the surface.
!   </IN>
 subroutine atmos_tracer_driver (is, ie, js, je, Time, lon, lat, land, phalf, pfull, r,  &
                           u, v, t, q, u_star, rdt, rm, rdiag, kbot)

!-----------------------------------------------------------------------
integer, intent(in)                           :: is, ie, js, je
type(time_type), intent(in)                   :: Time
real(r8), intent(in),    dimension(:,:)           :: lon, lat, u_star
real(r8), intent(in),    dimension(:,:)           :: land
real(r8), intent(in),    dimension(:,:,:)         :: phalf, pfull, u, v, t, q
real(r8), intent(in),    dimension(:,:,:,:)       :: r
real(r8), intent(inout), dimension(:,:,:,:)       :: rdt
real(r8), intent(in),    dimension(:,:,:,:)       :: rm
real(r8), intent(inout), dimension(:,:,:,:)       :: rdiag
integer, intent(in), dimension(:,:), optional :: kbot
!-----------------------------------------------------------------------
real(r8), dimension(size(r,1),size(r,2),size(r,3)) :: rtnd, pwt
real(r8), dimension(size(r,1),size(r,2),size(r,3)) :: rtndphob, rtndphil
real(r8), dimension(size(r,1),size(r,2)) :: dsinku
integer :: k, kd, nt

!-----------------------------------------------------------------------

!   <ERROR MSG="tracer_driver_init must be called first." STATUS="FATAL">
!     Tracer_driver_init needs to be called before tracer_driver.
!   </ERROR>
      if (.not. module_is_initialized)  &
      call error_mesg ('Tracer_driver','tracer_driver_init must be called first.', FATAL)

!-----------------------------------------------------------------------
      kd=size(r,3); nt=size(r,4)

  
! Flux into the layer is assumed to be kg(tracer)/m2/s
! Tracers are transported as mixing ratio.
! Therefore need to convert flux to a mixing ratio tendency 
! kg(tracer)/kg(air)/s = kg(tracer)/m2/s / [dz(m) * density of air(kg(air)/m3)]
! dz = dp/(rho*gravity)
! so dz*density of air = dp/gravity
! kg(tracer)/kg(air)/s = flux(tracer) / (dp/gravity)
      do k=1,kd
         pwt(:,:,k)=(phalf(:,:,k+1)-phalf(:,:,k))/grav
      enddo
!!WARNING pwt is dp/grav!!
! Go do the dry deposition of the tracers

  do k=1,nt
    call dry_deposition(k, is, js, u(:,:,kd), v(:,:,kd), t(:,:,kd), &
                        pwt(:,:,kd), pfull(:,:,kd), u_star, &
                        (land > 0.5), dsinku, r(:,:,kd,k), Time)
    rdt(:,:,kd,k) = rdt(:,:,kd,k) - dsinku
  enddo

!
!--------------- compute radon source-sink tendency --------------------
   if (nradon > 0) then
         if (nradon > nt) call error_mesg ('Tracer_driver', &
                            'Number of tracers .lt. number for radon', FATAL)
         call atmos_radon_sourcesink (lon,lat,land,pwt,r(:,:,:,nradon),  &
                                rtnd, Time, kbot)
      rdt(:,:,:,nradon)=rdt(:,:,:,nradon)+rtnd(:,:,:)
   endif

   if (nbcphobic > 0 .and. nbcphilic > 0 ) then
         if (nbcphobic > nt .or. nbcphilic > nt) &
            call error_mesg ('Tracer_driver', &
            'Number of tracers .lt. number for black carbon', FATAL)
         call atmos_blackc_sourcesink (lon,lat,land,pwt, &
                                 r(:,:,:,nbcphobic), rtndphob, &
                                 r(:,:,:,nbcphilic), rtndphil, &
                                 Time,is,ie,js,je)
      rdt(:,:,:,nbcphobic)=rdt(:,:,:,nbcphobic)+rtndphob(:,:,:)
      rdt(:,:,:,nbcphilic)=rdt(:,:,:,nbcphilic)+rtndphil(:,:,:)
   endif


   if (nocphobic > 0) then
         if (nocphobic > nt ) call error_mesg ('Tracer_driver', &
                            'Number of tracers .lt. number for organic carbon', FATAL)
         call atmos_organic_sourcesink (lon,lat,land,pwt,r(:,:,:,nocphobic),  &
                                  rtnd,Time,is,ie,js,je,kbot)
      rdt(:,:,:,nocphobic)=rdt(:,:,:,nocphobic)+rtnd(:,:,:)
   endif

   if (nocphilic > 0) then
         if (nocphilic > nt ) call error_mesg ('Tracer_driver', &
                            'Number of tracers .lt. number for organic carbon', FATAL)
         call atmos_organic_sourcesink (lon,lat,land,pwt,r(:,:,:,nocphilic),  &
                                 rtnd, Time,is,ie,js,je, kbot)
      rdt(:,:,:,nocphilic)=rdt(:,:,:,nocphilic)+rtnd(:,:,:)
   endif

   if (nsf6 > 0) then
         if (nsf6 > nt ) call error_mesg ('Tracer_driver', &
                            'Number of tracers .lt. number for sulfur hexafluoride', FATAL)
         call atmos_sf6_sourcesink (lon,lat,land,pwt,r(:,:,:,nsf6),  &
                                 rtnd, Time,is,ie,js,je, kbot)
      rdt(:,:,:,nsf6)=rdt(:,:,:,nsf6)+rtnd(:,:,:)
   endif

 end subroutine atmos_tracer_driver
! </SUBROUTINE>

!#######################################################################

! <SUBROUTINE NAME="atmos_tracer_driver_init">
!   <OVERVIEW>
!     Subroutine to initialize the tracer driver module.
!   </OVERVIEW>
!   <DESCRIPTION>
!   The purpose of the arguments here are for passing on to the individual
!   tracer code. The user may wish to provide initial values which can be
!   implemented in the initialization part of the tracer code. Remember that
!   the tracer manager will provide a simple fixed or exponential profile if
!   the user provides data for this within the field table. However if a more
!   complicated profile is required then it should be set up in the
!   initialization section of the user tracer code.
!   </DESCRIPTION>
!   <TEMPLATE>
!     call atmos_tracer_driver_init (lonb,latb, r, mask, axes, Time)
!   </TEMPLATE>
!   <IN NAME="lonb" TYPE="real(r8)" DIM="(:)">
!     The longitudes for the local domain.
!   </IN>
!   <IN NAME="latb" TYPE="real(r8)" DIM="(:)">
!     The latitudes for the local domain.
!   </IN>
!   <IN NAME="mask" TYPE="real(r8), optional" DIM="(:,:,:)">
!      optional mask (0. or 1.) that designates which grid points
!           are above (=1.) or below (=0.) the ground dimensioned as
!           (nlon,nlat,nlev).
!   </IN>
!   <IN NAME="Time" TYPE="type(time_type)">
!     Model time.
!   </IN>
!   <IN NAME="axes" TYPE="integer" DIM="(4)">
!     The axes relating to the tracer array dimensioned as
!      (nlon, nlat, nlev, ntime)
!   </IN>
!   <INOUT NAME="r" TYPE="real(r8)" DIM="(:,:,:,:)">
!     Tracer fields dimensioned as (nlon,nlat,nlev,ntrace). 
!   </INOUT>
 subroutine atmos_tracer_driver_init (lonb, latb, r, axes, Time, mask)

!-----------------------------------------------------------------------
           real(r8), intent(in),    dimension(:)               :: lonb, latb
           real(r8), intent(inout), dimension(:,:,:,:)         :: r
type(time_type), intent(in)                                :: Time
        integer, intent(in)                                :: axes(4)
           real(r8), intent(in),    dimension(:,:,:), optional :: mask

!-----------------------------------------------------------------------
!
!  When initializing additional tracers, the user needs to make changes 
!
!-----------------------------------------------------------------------

      if (module_is_initialized) return

      call write_version_number (version, tagname)

!If we wish to automatically register diagnostics for wet and dry 
! deposition, do it now.
      call atmos_tracer_utilities_init(lonb, latb, axes, Time)

!----- set initial value of radon ------------

      nradon = get_tracer_index(MODEL_ATMOS,'radon')
      if (nradon > 0) then
        call atmos_radon_init(r, axes, Time, mask)
      endif


      nbcphobic = get_tracer_index(MODEL_ATMOS,'bcphob')

      nbcphilic = get_tracer_index(MODEL_ATMOS,'bcphil')

      nocphobic = get_tracer_index(MODEL_ATMOS,'ocphob')

      nocphilic = get_tracer_index(MODEL_ATMOS,'ocphil')

      if (nbcphobic > 0) then
        call atmos_carbon_aerosol_init(lonb, latb, r, axes, Time, mask)
      endif
      
      nsf6 = get_tracer_index(MODEL_ATMOS,'sf6')

      if (nsf6 > 0) then
        call atmos_sulfur_hex_init (lonb, latb, r, axes, Time, mask)
      endif


      module_is_initialized = .TRUE.

 end subroutine atmos_tracer_driver_init
! </SUBROUTINE>

!#######################################################################

! <SUBROUTINE NAME="atmos_tracer_driver_end">
!   <OVERVIEW>
!     Subroutine to terminate the tracer driver module.
!   </OVERVIEW>
!   <DESCRIPTION>
!     Termination routine for tracer_driver. It should also call
!     the destructors for the individual tracer routines.
!   </DESCRIPTION>
!   <TEMPLATE>
!     call atmos_tracer_driver_end
!   </TEMPLATE>
 subroutine atmos_tracer_driver_end

!-----------------------------------------------------------------------


      write (stdlog(),'(/,(a))') 'Exiting tracer_driver, have a nice day ...'

      call atmos_radon_end
      call atmos_sulfur_hex_end
      call atmos_carbon_aerosol_end

      module_is_initialized = .FALSE.

!-----------------------------------------------------------------------

 end subroutine atmos_tracer_driver_end
! </SUBROUTINE>

!######################################################################

end module atmos_tracer_driver_mod



