\name{flat.write}
\alias{flat.write}
\author{Tim Hoar \email{thoar@ucar.edu} \url{http://www.cgd.ucar.edu/~thoar} }
\title{Fortran "flat" write}
\description{
  writes R data into Fortran unformatted binary records (native to the writing
platform, naturally).  
}
\synopsis{
   flat.write(datmat=NULL, file=NULL, 
                 ioflag=3, miss=-999., unit=95, type="real")
}
\usage{
   flat.write(datmat, file)
   flat.write(datmat, file, ioflag)
   flat.write(datmat, file, ioflag, type)
}

\arguments{

   \item{datmat}{The array to export.}

   \item{file}{The file name.}

   \item{ioflag}{[optional] integer to describe the read state of the file\cr
   ioflag == 1   open, read and close the file\cr
   ioflag == 2   open, read and leave the file open for further reading\cr
   ioflag == 3   read and leave the file open [default]\cr
   ioflag == 4   read and close the file}

   \item{miss}{[optional] real number to replace NaN\cr
               miss = -999. [default]}

   \item{unit}{[optional] integer to describe the Fortran unit. This is only
   needed if you intend to read/write from multiple files simultaneously. 
   Fortran associates a file with a unique (hopefully) integer and uses 
   the integer for everything.  Stay away from unit == 5, unit == 6, 
   which can have special meaning./cr
   unit == 95 [default]}

   \item{type}{[optional] character string to describe the Fortran variable
   type.\cr
   type == "real" [default]\cr
   type == "real*8"\cr
   type == "integer"}

}
\value{
   \item{A scalar containing the number of items written ...}
}
\details{You should be able to confirm everything was written the way you 
expected by comparing the actual file size with the expected file size.  
Every time you "write" a record, the file size will increase by 8bytes + 
whatever you write. (4 bytes for a "beginning of record" and 4 more for 
an "end of record")\cr
\cr
Functionally, what the following examples are trying to do is replicate
portions of the following Fortran code:\cr

      integer dim1,dim2             ! scalars -- 1x1 "matrices"\cr
      parameter (dim1=2, dim2=3)\cr
      real     a(dim1,dim2)         ! 32bit matrix\cr
      real*8   b(dim1,dim2)         ! 64bit matrix\cr
      integer  d(dim1,dim2)         ! matrix of integers\cr
      ...\cr
      open(9,file='Flat.ieee',form='unformatted')\cr
      ...\cr
      write(9)dim1,dim2\cr
      write(9)a\cr
      write(9)b\cr
      write(9)d
}

\examples{
# EXAMPLE 1: WRITING a matrix (of integers)
NDarray <- array(c(111,121,112,122,113,123,
                   211,221,212,222,213,223,
                   311,321,312,322,313,323,
                   411,421,412,422,413,423),c(2,3,4))
datmat <- NDarray[,,1]
A <- flat.write(datmat,"matrix.ieee", ioflag=1, type="integer")

# EXAMPLE 2: WRITING a couple of things in a row.
dims  <- dim(NDarray)
nrows <- dims[1]
ncols <- dims[2]
a <- NDarray[,,1]
b <- NDarray[,,2]
d <- NDarray[,,3]
flat.write(c(nrows,ncols), "Flat.ieee",ioflag=2,type="integer");  # open, read, leave open
flat.write(a, "Flat.ieee", ioflag=3);     # write a 'real' matrix, leave open
flat.write(b, "Flat.ieee", ioflag=3, type="real*8");    # write a 'real*8'
flat.write(d, "Flat.ieee", ioflag=4, type="integer");   # write a 'real*8', close
}

\seealso{ \code{\link{flat.read}, \link{direct.read}, \link{direct.write} } }

}

\note{
   VERY IMPORTANT If you "open, write, ... close" and then RE-OPEN to 
   write some more -- THIS WILL NOT WORK! Every time you "open", you 
   start at the beginning!\cr

   IMPORTANT If you have multiple files open (for reading or writing),
   YOU had best keep track of which unit is associated with a file. Doing 
   otherwise is going to produce undesirable results, guaranteed.
}
\keyword{IO}
