//************************************************************************************//
// Module       :   gtmatrix.cpp
// Date         :   7/9/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a template matrix object, whose data is composed of
//                a regular array of contiguous data, ordered like a Fortran
//                matrix in row-major order (row data changing fastest over column data).
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <typeinfo>
#include "gtmatrix.hpp"
#include "gcomm.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// constructor (1)
//************************************************************************************
template<class T> GTMatrix<T>::GTMatrix<T>()
:
n1                    (0),
n2                    (0),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE)
{
  

  data.Resize(1);
  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }
} // end of constructor 1 



//************************************************************************************
//************************************************************************************
// constructor (2)
//************************************************************************************
template<class T>  GTMatrix<T>::GTMatrix<T>(const GINT   size1, const GINT   size2)
:
n1                    (size1),
n2                    (size2),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE)
{

  data.Resize(n1*n2);

  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }

  Zero();
} // end of constructor 2


//************************************************************************************
//************************************************************************************
// constructor (3)
//************************************************************************************
template<class T>  GTMatrix<T>::GTMatrix<T>(T *array, GINT   m1, GINT   m2)
:
n1                    (m1),
n2                    (m2),
dtype                 (G_GDOUBLE),
cdtype                (GC_GDOUBLE)
{



  if      ( typeid(T) == typeid(GUSHORT ) ) {
   dtype  = G_GUSHORT ;
   cdtype = GC_GUSHORT ; }
  else if ( typeid(T) == typeid(GSHORT ) ) {
    dtype  = G_GSHORT ;
    cdtype = GC_GSHORT ;}
  else if ( typeid(T) == typeid(GINT ) ) {
    dtype  = G_GINT ;
    cdtype = GC_GINT ; }
  else if ( typeid(T) == typeid(GDOUBLE) ) {
    dtype  = G_GDOUBLE;
    cdtype = GC_GDOUBLE;}
  else if ( typeid(T) == typeid(GQUAD) ) {
    dtype  = G_GQUAD;
    cdtype = GC_GQUAD; }

  // build matrix data structure:
  data.Resize(n1*n2);
  memcpy(data.Data(), array, n1*n2*G_TYPESZ[dtype]);

  Zero();

} // end of constructor 3


//************************************************************************************
//************************************************************************************
// Copy constructor:
//************************************************************************************
template<class T> GTMatrix<T>::GTMatrix<T>(const GTMatrix<T> &m)
{



  // copy member data:
  n1      = m.n1;
  n2      = m.n2;
  dtype   = m.dtype;
  cdtype  = m.cdtype;
  
  data.Resize(n1*n2);

  data = m.data;
} // end of copy constructor


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
template<class T> GTMatrix<T>::~GTMatrix<T>()
{
   DeleteDynamic();
}

//************************************************************************************
//************************************************************************************
// Assignment operator method
//************************************************************************************
template<class T> GTMatrix<T> GTMatrix<T>::operator=(const GTMatrix<T> &m)
{
  

  if ( &m != this ) 
  {
    if ( m.n1 != n1 || m.n2 != n2 )
    {
      cout << "GTMatrix<T>::=: incompatible matrices" << endl;
      exit(1);
    }
    // copy member data:
    n1      = m.n1;
    n2      = m.n2;
    dtype   = m.dtype;
    cdtype  = m.cdtype;
    data    = m.data;
  }

  return *this;

} // end = operator


//************************************************************************************
//************************************************************************************
// Assignment operator method
//************************************************************************************
template<class T> void  GTMatrix<T>::operator=(T m)
{
  GINT   i;

  for ( i=0; i<n1*n2; i++ ) { 
      data[i] = m; 
  }

} // end = operator


//************************************************************************************
//************************************************************************************
// METHOD     : operator *
// DESCRIPTION: multiplies this by constant, and returns
//              result, without destroying *this data
// RETURNS    : product matrix
//************************************************************************************
template<class T> GTMatrix<T> GTMatrix<T>::operator*(T a) 
{
  GINT          i;
  T             *adata;
  GTMatrix<T>   aprod(n1,n2);

  adata = aprod.Data();
  memcpy(adata, data.Data(), n1*n2*G_TYPESZ[dtype]);
  for ( i=0; i<n1*n2; i++ ) {
    adata[i] *= a; 
   }

  return aprod;

} // end of * operator (for constant ref.)


//************************************************************************************
//************************************************************************************
// Matrix-vector product:
//************************************************************************************
template<class T> GTVector<T> GTMatrix<T>::operator*(GTVector<T>  local_a)
{

  GTVector<T>   *aret;

  aret = new GTVector<T>(n1);

  switch (dtype) {
    case G_GDOUBLE:
      MTK::fmatvec_prod((GTMatrix<GDOUBLE>&)*this,(GTVector<GDOUBLE>&)local_a,(GTVector<GDOUBLE>&)*aret);
      break;
    case G_GQUAD:
      MTK::qmatvec_prod((GTMatrix<GQUAD>&)*this,(GTVector<GQUAD>&)local_a,(GTVector<GQUAD>&)*aret);
      break;
    default:
      cout << " GTMatrix<T>::operator*(vector): invalid data type" << endl;
      exit(1);
  }
  return *aret;

} // end of operator *


//************************************************************************************
//************************************************************************************
// METHOD     : operator *
// DESCRIPTION: multiplies this by m1, and returns
//              result
// RETURNS    : product matrix
//************************************************************************************
template<class T> GTMatrix<T> GTMatrix<T>::operator*(GTMatrix<T> m) 
{
  GTMatrix<T> *mret;

  if ( this->n2 != m.dim(1) ) {
    cout << "GTMatrix<T>::*: (Matrix) incompatible matrix"<< endl;
    exit(1);
  }

  mret = new GTMatrix<T>(n1,m.dim(2));

  switch (dtype) {
    case G_GDOUBLE:
      MTK::fmatmat_prod(*((GTMatrix<GDOUBLE>*)this),(GTMatrix<GDOUBLE>&)m,(GTMatrix<GDOUBLE>&)*mret);
      break;
    case G_GQUAD:
//    MTK::qmatmat_prod(*((GTMatrix<GQUAD>*)this),(GTMatrix<GQUAD>&)m,(GTMatrix<GQUAD>&)*mret);
      cout << " GTMatrix<T>::operator*(matrix): Use MTK!" << endl;
      exit(1);
    default:
      cout << " GTMatrix<T>::operator*(matrix): invalid data type" << endl;
      exit(1);
  }
  return *mret;

} // end of operator * (GTMatrix<T>)


//************************************************************************************
//************************************************************************************
template<class T> GTMatrix<T> GTMatrix<T>::operator+(GTMatrix<T> a) 
{

  GTMatrix<T>  *asum;

  if ( this->n1 != a.dim(1) || this->n2 !=a.dim(2) ) {
    cout << "GTMatrix<T>::+: incompatible matrices"<< endl;
    exit(1);
  }

  asum  = new GTMatrix<T>(n1,n2);

  switch (dtype) {
    case G_GDOUBLE:
      MTK::fvec_add((GTVector<GDOUBLE>&)data,*((GTVector<GDOUBLE>*)a.VData()),*((GTVector<GDOUBLE>*)asum->VData()));
      break;
    case G_GQUAD:
      MTK::qvec_add((GTVector<GQUAD>&)data,*((GTVector<GQUAD>*)a.VData()),*((GTVector<GQUAD>*)(asum->VData())));
      exit(1);
    default:
      cout << " GTMatrix<T>::operator+(matrix): invalid data type" << endl;
      exit(1);
  }

  return *asum;

}


//************************************************************************************
//************************************************************************************
template<class T> GTMatrix<T> GTMatrix<T>::operator-(GTMatrix<T>  a) 
{
  GTMatrix<T>  *asum;
  
  if ( this->n1 != a.dim(1) || this->n2 !=a.dim(2) ) {
    cout << "GTMatrix<T>::-: incompatible matrices"<< endl;
    exit(1);
  } 

  asum  = new GTMatrix<T>(n1,n2);

  switch (dtype) {
    case G_GDOUBLE:
      MTK::fvec_sub((GTVector<GDOUBLE>&)data,*((GTVector<GDOUBLE>*)a.VData()),*((GTVector<GDOUBLE>*)asum->VData()));
      break;
    case G_GQUAD:
      MTK::qvec_sub((GTVector<GQUAD>&)data,*((GTVector<GQUAD>*)a.VData()),*((GTVector<GQUAD>*)(asum->VData())));
      exit(1);
    default:
      cout << " GTMatrix<T>::operator+(matrix): invalid data type" << endl;
      exit(1);
  }

  return *asum;

}


//************************************************************************************
//************************************************************************************
// << operator method 
//************************************************************************************
ostream &operator<<(ostream &str, GTMatrix<GDOUBLE> &a)
{

  GINT     i, j, N1, N2;
  GDOUBLE  *adata;
   
  adata = a.Data();
  N1    = a.dim(1);
  N2    = a.dim(2);
#if 0
  str << "{ ";
  for ( i=a.GetIndex(1).beg(); i<a.GetIndex(1).beg()+a.dim(1); i++ )
  {
    str << "{ ";
    for ( j=a.GetIndex(2).beg(); j<a.GetIndex(2).beg()+a.dim(2)-1; j++ )
      str  
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j)
//        << setw(1)
          << ", ";

      str
//        << setiosflags(ios::scientific)
//        << setw(11)
//        << setprecision(4)
//        << setiosflags(ios::fixed)
          << a(i,j) ;
//        << setw(1) ;
    if ( i < a.GetIndex(1).beg()+a.dim(1)-1 ) str << " }, ";
    else str << " }";
  }
#else
  str << endl;
  for ( i=0; i<N1; i++ ) {
    for ( j=0; j<N2 ; j++ ) {
      str
#if 0
          << setiosflags(ios::scientific)
          << setw(18)
          << setprecision(15)
          << setiosflags(ios::fixed)
#endif
          << (fabs(adata[i+j*N1]) < TINY*10.0 ? 0.0 : adata[i+j*N1])
          << " ";
    }
    str << endl;
  }
#endif

  return str;
} // end of << operator 


#if defined(GBUFF_DEF_GQUAD)
//************************************************************************************
//************************************************************************************
// << operator method
//************************************************************************************
ostream &operator<<(ostream &str, GTMatrix<GQUAD> &a)
{
  GINT   i, j, N1, N2;
  GQUAD *adata;
   
  adata = a.Data();
  N1    = a.dim(1);
  N2    = a.dim(2);

  str << endl;
  for ( j=0; j<N2 ; j++ ) {
    for ( i=0; i<N1; i++ ) {
      str
#if 0
          << setiosflags(ios::scientific)
          << setw(18)
          << setprecision(15)
          << setiosflags(ios::fixed)
#endif
          << (GDOUBLE)(fabs((GQUAD)adata[i*j*N1]) < TINY ? 0.0 : adata[i+j*N1])
          << " ";
    }
    str << endl;
  }
  return str;
} // end of << operator
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : Data
// DESCRIPTION: 
// RETURNS    :  pointer to the data area
//************************************************************************************
template<class T> T *GTMatrix<T>::Data() 
{
  return data.Data();
} // end of method Data


//************************************************************************************
//************************************************************************************
// METHOD     : VData
// DESCRIPTION: 
// RETURNS    :  pointer to the data area
//************************************************************************************
template<class T> GTVector<T> *GTMatrix<T>::VData() 
{
  return &data;
} // end of method VData


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
//************************************************************************************
template<class T> void GTMatrix<T>::DeleteDynamic()
{
} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
template<class T> GBOOL GTMatrix<T>::Resize(GINT   new1, GINT   new2)
{
  

  if ( n1 == new1 && n2 == new2 ) return TRUE;

  DeleteDynamic();

  n1 = new1;
  n2 = new2;

  data.Resize(n1*n2);

  return TRUE;

} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : tsize
// DESCRIPTION: total size of data storage area
//              in direction 
// RETURNS    : GINT   size
//************************************************************************************
template<class T> GINT   GTMatrix<T>::tsize(GINT   idir)
{

  if      ( idir == 1 )
    return n1;
  else if ( idir == 2 )
    return n2;
  else
    return 0;

} // end of method tsize


//************************************************************************************
//************************************************************************************
// METHOD     : dim
// DESCRIPTION: array dimension (usable)
//              in direction idir 
// RETURNS    : GINT   size
//************************************************************************************
template<class T> GINT GTMatrix<T>::dim(GINT   idir) 
{

  if      ( idir == 1 )
    return n1;
  else if ( idir == 2 )
    return n2;
  else
    return 0;
} // end of method dim


//************************************************************************************
//************************************************************************************
// METHOD     : Zero
// DESCRIPTION: Zeros out data elemnts
// RETURNS    : none
//************************************************************************************
template<class T> void GTMatrix<T>::Zero()
{ 
  

  data = 0.0;
}  // end of method Zero



//************************************************************************************
//************************************************************************************
// METHOD     : Transpose (1)
// DESCRIPTION: computes transpose of *this, but
//              does not destroy data.
// RETURNS    : transpose of this
//************************************************************************************
template<class T> GBOOL  GTMatrix<T>::Transpose(GTMatrix<T> &trans)
{

  GINT   i, j, k;
  T      *tdata;

  if ( trans.dim(2) !=  n1 || trans.dim(1) != n2 ) {
    cout << "GTMatrix<T>::Transpose: incompatible matrix"<< endl;
    exit(1);

  }

  tdata = trans.Data();
  for ( j=0; j<n1; j++ ) {
    k = j*n2;
    for ( i=0; i<n2; i++ ) {
       tdata[i+k] = data[j+i*n1];
    }
  }
  return TRUE;
 
} // end of method Transpose (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Transpose (2)
// DESCRIPTION: computes transpose of this, but
//              does not destroy data. A copy is made and
//              returned.
// RETURNS    : transpose of this
//************************************************************************************
template<class T> GTMatrix<T>  GTMatrix<T>::Transpose()
{

  GTMatrix<T> *t;

  t = new GTMatrix<T>(n1,n2);
  if ( !Transpose(*t) ) {
    cout << "GTMatrix<T>::Transpose(2): failed" << endl;
    exit(1);
  }

  return *t;

} // end of method Transpose (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Inverse (1)
// DESCRIPTION: computes inverse of this, copying the
//              result to mret
// RETURNS    : inverse of this
//************************************************************************************
template<class T> GBOOL  GTMatrix<T>::Inverse(GTMatrix<T> &mret)
{

  GINT        i, j, *indx;
  GBOOL       bRet=TRUE;
  T           **A, *col,  d;
  

  if ( mret.dim(1) !=  n1 || mret.dim(2) != n2 ) {
    cout << "GTMatrix<T>::Inverse: incompatible matrix"<< endl;
    exit(1);
  }
  if ( n1 != n2 ) {
    cout << "GTMatrix<T>::Inverse: matrix not square"<< endl;
    exit(1);
  }

  A     = new T * [n1];
  col   = new T [n1];
  indx  = new GINT   [n2];
  for ( i=0; i<n1; i++ ) A[i] = new T [n2];

  
 
  mret = 0.0;
  for ( i=0; i<n1; i++ ) mret(i,i) = 1.0;

  for ( j=0; j<n2; j++ ) 
    for ( i=0; i<n1; i++ )
      A[i][j] = data[i+j*n1];

  if ( !ludcmp(A, n1, indx, &d) ) {
     cout << "GTMatrix::Inverse: ludcmp failed" << endl;
     bRet = FALSE;
  }

#if 0
  GMatrix LU(n1,n1), L(n1,n1), U(n1,n1);
  for ( i=0; i<n1; i++ )
    for ( j=0; j<n1; j++ )
      LU(i,j) = A[i][j];
  for ( i=0; i<n1; i++ )
    for ( j=i; j<n1; j++ )
      U(i,j) = A[i][j] ;
  for ( i=0; i<n1; i++ )
  {
    for ( j=0; j< i; j++ )
      L(i,j) = A[i][j];
    L(i,i) = 1.0;
  }

  cout << "GTMatrix: A= " << LU << endl;
  cout << "GTMatrix: L= " << L << endl;
  cout << "GTMatrix: U= " << U << endl;
  cout << "GTMatrix: L*U= " << (L*U) << endl;
  cout << "GTMatrix: d = " << d << "  indx= " <<  endl;
  for ( i=0; i<n1 && bRet; i++ )
    cout << indx[i] << " ";
  cout << endl;
#endif

  for ( j=0; j<n2 && bRet; j++ ) {
    for ( i=0; i<n1; i++ ) {
      col[i] = mret(i,j); //mdata[i][j];
    }
    if ( !(bRet=lubksb(A, n1, indx, col)) ) {
       cout << "GTMatrix::Inverse: lubjsb failed" << endl;
       bRet = FALSE;
       break;
    }
    for ( i=0; i<n1; i++ ) {
      mret(i,j) = col[i];
    }
  }

  for ( i=0; i<n1; i++ ) delete [] A[i];
  delete [] A;
  delete [] col;
  delete [] indx;

  return bRet;

} // end of method Inverse


//************************************************************************************
//************************************************************************************
// METHOD     : Inverse (2)
// DESCRIPTION: computes inverse of this, but
//              does not destroy data. A copy is made and
//              returned.
// RETURNS    : inverse of this
//************************************************************************************
template<class T> GTMatrix<T>  GTMatrix<T>::Inverse()
{

  GTMatrix<T> *mi;

  mi = new GTMatrix<T>(n1,n2);

  if ( !Inverse(*mi) ) {
    cout << "GTMatrix<T>::Inverse(2): failed" << endl;
    exit(1);
  }

  return *mi;

} // end of method Inverse (2)


//************************************************************************************
//************************************************************************************
// METHOD     : isSymmetric 
// DESCRIPTION: determines if matrix is symmetric
// RETURNS    : TRUE or FALSE 
//************************************************************************************
template<class T> GBOOL  GTMatrix<T>::isSymmetric()
{
  GINT   i, j, k, m;
  GBOOL  bRet;

  if ( n1 != n2 ) return FALSE;

  // NOTE: should be symmetric w.r.t some tolerance!!!
  for ( j=1; j<n2; j++ ) {
    m = j*n2;
    for ( i=j+1,bRet=TRUE; i<n1-1; i++ ) {
      k = n2*i;
      bRet = bRet && data[i+m] == data[i+k];
    }
  }
  return bRet;
 
} // end of method isSymmetric


//************************************************************************************
//************************************************************************************
// METHOD     : wludcmp
// DESCRIPTION: Taken largely from Numerical Recipes
// RETURNS    : GBOOL flag
//************************************************************************************
template<class T> GBOOL  GTMatrix<T>::wludcmp(T **&a, GINT   n, GINT   *&indx, T *d)
{
  if ( a == NULL || indx == NULL ) return FALSE;

  GINT   i, imax=n-1, j, k;
  GBOOL bRet=TRUE;
  T     big, dum, sum, temp;
  T     *vv;

  vv = new T [n]; 
  *d=1.0;
  for (i=0;i<n && bRet;i++) {
    big=0.0;
    for (j=0;j<n;j++)
      if ((temp=fabs(a[i][j])) > big) big=temp;
    if (big == 0.0){
      bRet = FALSE; 
      break;
    }
    vv[i]=1.0/big;
  }

  for (j=0;j<n && bRet;j++) {
    for (i=0;i<j;i++) {
      sum=a[i][j];
      for (k=0;k<i;k++) sum -= a[i][k]*a[k][j];
      a[i][j]=sum;
    }
    big=0.0;
    for (i=j;i<n;i++) {
      sum=a[i][j];
      for (k=0;k<j;k++) sum -= a[i][k]*a[k][j];
      a[i][j]=sum;
      if ( (dum=vv[i]*fabs(sum)) >= big) {
        big=dum;
        imax=i;
      }
    }
    if (j != imax) {
      for (k=0;k<n;k++) {
        dum=a[imax][k];
        a[imax][k]=a[j][k];
        a[j][k]=dum;
      }
     *d = -(*d);
     vv[imax]=vv[j];
    }
    indx[j]=imax;
    if (a[j][j] == 0.0) a[j][j]=TINYTINY;
    if (j != n-1) {
      dum=1.0/(a[j][j]);
      for (i=j+1;i<n;i++) a[i][j] *= dum;
    }
  }
  delete [] vv;


  return bRet;

} // end of method wludcmp


//************************************************************************************
//************************************************************************************
// METHOD     : lubksb
// DESCRIPTION: Taken largely from Numerical Recipes
// RETURNS    : GBOOL flag
//************************************************************************************
template<class T> GBOOL  GTMatrix<T>::lubksb(T **&a, GINT   nd, GINT   *&indx, T b[])
{

    GINT  n=nd-1; 
    GINT  i, ii = -1, ip, j;
    T   sum;

    for ( i=0; i <= n; i++ ) {
        ip = indx[i];
        sum = b[ip];
        b[ip] = b[i];
        if (ii > -1) {
            for (j = ii; j < i; j++) sum -= a[i][j]*b[j];
        }
        else { 
            if (sum) ii = i;
        }
        b[i] = sum;
    }
    for ( i=n; i>=0; i-- ) {
        sum=b[i];
        if ( i < n )
        for (j = i+1; j <= n; j++) sum -= a[i][j]*b[j];
        b[i] = sum /(a[i][i]);
    }

  return TRUE;

} // end of method lubksb


//************************************************************************************
//************************************************************************************
// METHOD     : ludcmp
// DESCRIPTION: LU decomposition method provided by
//              Warren Jasper
// RETURNS    : GBOOL flag
//************************************************************************************
template<class T> GBOOL  GTMatrix<T>::ludcmp(T **&a, GINT   nd, GINT   *&indx, T *d)
{
    GINT   i, imax, j, k;
    GINT   n=nd-1;
    GBOOL bRet = TRUE;
    T     big, dum, sum, temp;
    T     *vv = new T [nd];

    *d = 1.0;
    imax = -1;

    for ( i = 0; i <= n; i++ ) {
        big = 0.0;
        for ( j = 0; j <= n; j++ )
            if ((temp = fabs( a[i][j] )) > big) big = temp;
            if (big == 0.0) {
              bRet = FALSE;  
              break;
            }
        vv[i] = 1.0 / big;
    }
    for ( j = 0; j <= n; j++ ) {
        for ( i = 0; i < j; i++ ) {
            sum = a[i][j];
                for (k = 0; k < i; k++) sum -= a[i][k]*a[k][j];
                a[i][j] = sum;
        }
        big = 0.0;
        for ( i = j; i <= n; i++ ) {
            sum = a[i][j];
            for ( k = 0; k < j; k++ )
                sum -= a[i][k]*a[k][j];
            a[i][j] = sum;
            if ( (dum = vv[i]*fabs(sum)) >= big) {
                big = dum;
                imax = i;
            }
        }
        if (j != imax) {
            if ( imax < 0 ) return FALSE;
            for ( k = 0; k <= n; k++ ) {
                dum = a[imax][k];
                a[imax][k] = a[j][k];
                a[j][k] = dum;
            }
            *d = -(*d);
            vv[imax] = vv[j];
        }
        indx[j] = imax;
        if (a[j][j] == 0.0) a[j][j] = TINYTINY;
        if (j != n) {
            dum = 1.0 / (a[j][j]);
            for (i = j+1; i <= n; i++) a[i][j] *= dum;
        }
    }
    delete [] vv;

    return bRet;

} // end of method ludcmp

//************************************************************************************
//************************************************************************************
// METHOD     : isamax
// DESCRIPTION: BLAS routine that finds the index of element having max.
//              absolute value.
// ARGUMENTS  : n   : Number of elements to check.
//              sx  : Vector to be checked.
//              incx: Every incx-th element is checked.

// RETURNS    : int
//************************************************************************************
GINT  isamax( GINT  n, GDOUBLE *sx, GINT  incx )
{
  GDOUBLE  smax = 0.0e0;
  GINT   i, istmp = 0;

  if( n <= 1 ) return( istmp );
  if( incx != 1 ) {
    /* Code for increment not equal to 1. */
    if( incx < 0 ) sx = sx + ((-n+1)*incx + 1);
    istmp = 0;
    smax  = fabs( *sx );
    sx += incx;
    for( i=1; i<n; i++, sx+=incx )
      if( fabs( *sx ) > smax ) {
        istmp = i;
        smax  = fabs( *sx );
      }
    return( istmp );
  }
  /* Code for increment equal to 1. */
  istmp = 0;
  smax  = fabs(*sx);
  sx++;
  for( i=1; i<n; i++, sx++ )
    if( fabs( *sx ) > smax ) {
      istmp = i;
      smax  = fabs( *sx );
    }
  return( istmp );
}  // end of method isamax


