//************************************************************************************//
// Module       : gpartitioner.cpp
// Date         : 6/9/05 (DLR)
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                partitioning of data/elements among processors.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "gpartitioner.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GPartitioner::GPartitioner(GElemList *gelems, GFieldList **gfields, GINT nfields)
:
nprocs_            (GComm::WorldSize()),
this_rank_         (GComm::WorldRank()),
bInitialized_      (FALSE),
bdelete_partmap_   (TRUE),
nfieldsv_          (nfields),
nfieldsp_          (0),
npartmax_          (0),
maxvnodes_         (0),
maxpnodes_         (0),
maxedges_          (0),
maxfaces_          (0),
maxverts_          (0),
maxnto_            (0),
nto_               (0),
nfrom_             (0),
lbuff_             (0),
gelemsv_           (gelems),
gelemsp_           (NULL),
gfieldsv_          (gfields),
gfieldsp_          (NULL),
gbasisv_           (NULL),
gbasisp_           (NULL),
szbasisv_          (NULL),
szbasisp_          (NULL),
recvbuff_          (NULL),
partmap_           (NULL)
{
  char     *serr = "GPartitioner::GPartitioner(1) : ";
  GINT     j;

  if ( gelemsv_ == NULL ) {
    cout << serr << "NULL v-element list" << endl;
    exit(1);
  }
  if ( gfieldsv_ == NULL || nfieldsv_ <= 0 ) {
    cout << serr << "NULL v-field list" << endl;
    exit(1);
  }

  // Add primary fields to fieldgroup structure:
  fieldgroups_.add();
  fieldgroups_.member()->nfields = nfields;
  fieldgroups_.member()->pelems  = gelems;
  fieldgroups_.member()->pfields = new GFieldList * [nfields];

  // Make copy of primary field group for use when only this
  // is required:
  for ( j=0; j<nfields; j++ ) {
    fieldgroups_.member()->pfields[j] = gfields[j];
  }

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
GPartitioner::~GPartitioner()
{

  if ( bdelete_partmap_ && partmap_ != NULL ) {
    delete [] partmap_;
  }

}

//************************************************************************************
//************************************************************************************
// METHOD     : Init
// DESCRIPTION: Initializes object by setting partition map.
// ARGUMENTS  : pm  : pointer to stGPartitionMap array. There should be nmax * nprocs 
//                    structure elements in this array. Each proc will send at most 
//                    nmax elements.  An element is invalid for send if key = procid = PMDEFAULT_KEY/PROC,
//                    and for a given proc, if there are fewer than nmax elements to send,
//                    the stGPartitionMap for the nmax-no. valid elems must be set to PMDEFAULT_KEY_PROC. 
//
//             pm has the form:
//
//              / (0)      elem_key=XXX, proc_to == 0   \
//              | (1)      elem_key=YYY, proc_to == 1   |
//    nmax      | (2)      elem_key=ZZZ, proc_to == 2   |   proc 0
//              |                  ...                  |
//              \ (nmax-1) elem_key=KKK, proc_to == N   /
//
//              / (nmax)   elem_key=XXX, proc_to == 9   \
//              | (nmax+1) elem_key=YYY, proc_to == 7   |
//     nmax     | (nmax+2) elem_key=-1 , proc_to == -1  |   proc 1
//              |                  ...                  |
//              \ (2nmax-1)elem_key=-1 , proc_to == -1  /
// 
//                            .
//                            .
//                            .
//              / (...)   elem_key=XXX, proc_to == 9   \
//              | (...)   elem_key=-1 , proc_to == -1  |
//     nmax     | (...)   elem_key=-1 , proc_to == -1  |   proc Nprocs-1
//              |                 ...                  |
//              \ (...)   elem_key=-1 , proc_to == -1  /
//              nmax: max number of elements to send 
// RETURNS    : none.
//************************************************************************************
GBOOL GPartitioner::Init(stGPartitionMap pm[], GINT nmax)
{
  char     *serr = "GPartitioner::Init: ";
  GINT     i, is, j, n, nsend;
  GBOOL    bsend;
  GSBuffer irecv;
  Elem2D   *elem;

  // First, compute local quantities:

  // ... first, how many sends from this proc to different procs:
  is = this_rank_ * nmax;
  n = 0;
  nsend = 0;
  while ( nsend < nmax 
       && pm[is+nsend].elem_key   != PMDEFAULT_KEY
       && pm[is+nsend].parent_key != PMDEFAULT_KEY
       && pm[is+nsend].proc_to    != this_rank_  ) {
     nsend++;
   }

  index_to_.Resize(nsend);
  iproc_to_.Resize(nsend);

  // ... which procs is data being sent from, handles for asynch recvs:
  irecv.Resize(nprocs_);
  nfrom_ = 0;
  for ( i=0; i<nprocs_; i++ ) {
    if ( i == this_rank_ ) continue; // don't consider those 'sent' from this proc
    for ( j=0,bsend=FALSE; j<nmax && !bsend; j++ ) {
      n = i*nmax+j;
      bsend = pm[n].elem_key != PMDEFAULT_KEY && pm[n].proc_to != PMDEFAULT_PROC && pm[n].proc_to == this_rank_;
    }
    if ( bsend ) irecv[nfrom_++] = i;
  }

  iproc_from_.Resize(nfrom_);
  rhandle_.Resize(nfrom_);
  for ( i=0; i<nfrom_; i++ ) iproc_from_[i] = irecv[i];

  // ... proc ids where this proc sends data:
  gelemsv_->start(NULL);
  for ( i=0, n=0; i<gelemsv_->size(); i++ ) {
    elem = gelemsv_->member();
    for ( j=is; j<is+nsend; j++ ) {
      if ( pm[j].elem_key ==  elem->GetID() ) {
        index_to_[n] = i;
        iproc_to_[n] = pm[is+j].proc_to;
        n++;
      }
      gelemsv_->next();
    }
  }
    
  if ( n != nsend ) {
    cout << serr << "element keys not unique?" << endl;
    exit(1);
  }
  partmap_  = pm;
  npartmax_ = nmax;
  bdelete_partmap_ = FALSE;

  // Allocate recv and send buffers:
  if ( !CreateSendRecvBuffs() ) {
    cout << serr << "buffer allocation failed" << endl;
    exit(1);
  }

  bInitialized_ = TRUE;

  return TRUE;

} // end of method Init


//************************************************************************************
//************************************************************************************
// METHOD     : CreateSendRecvBuffs;
// DESCRIPTION: set up the send/recv buffers based on partition map.
// ARGUMENTS  : none. 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::CreateSendRecvBuffs()
{ 
  char       *serr = "GPartitioner::CreateSendRecvBuffs: ";
  GINT       i, j, ng, nfields, nlmax;
  GINT       lmaxlevels=0, lnto=0, nto;
  GIBuffer   lmax, gmax;
  Elem2D     *velem, *pelem;
  GElemList  *gelems;
  GFieldList **gfields;

  if ( partmap_ == NULL ) {
    cout << serr << "NULL partition map" << endl;
    return FALSE;
  }

  // Get local buffer size components:
  lnto = 0;                      // number procs to send local data _to_
  for ( i=this_rank_; i<npartmax_&&partmap_[i].elem_key!=PMDEFAULT_KEY; i++ ) {
    lnto   += ( partmap_[i].proc_to == this_rank_ ? 0 : 1 );
  }
 
  nlmax = fieldgroups_.size()+3;
  lmax.Resize(nlmax); 
  gmax.Resize(nlmax); 
  lmax = 0;
  nto_ = lnto;  
  gelems  = fieldgroups_[0]->pelems; gelems->start(NULL);
  // From primary elems, find nodes, # bdy indices, # vertices
  for ( i=0; i<gelems->size(); i++ ) { 
    velem     = gelems->member();
    lmax[0]   = MAX(lmax[0], velem->GetNumNodes()); 
    lmax[1]   = MAX(lmax[1], velem->GetBdyIndices()->dim()); 
    lmax[2]   = MAX(lmax[2], velem->GetNumVertices()); 
    gelems->next();
  }
  lmax[3] = lnto;

  // Secondary grid are assumed to reside _within_ primary grid.
  // Thus, find # nodes only:
  for ( ng=1; ng<fieldgroups_.size(); ng++ ) {
    gelems  = fieldgroups_[0]->pelems; gelems->start(NULL);
    for ( i=0; i<gelems->size(); i++ ) {
      velem      = gelems->member();
      lmax[3+ng] =  MAX(lmax[3+ng],velem->GetNumNodes()); 
      gelems->next();
    }
  }
  maxlevels_ = (*gfieldsv_[0])[0]->GetNumTimeLevels();

  // Get global max of various sizes:
  GComm::Allreduce(lmax.Data(), gmax.Data(), nlmax, GC_GINT, G_OP_MAX);
   maxvnodes_ = gmax[0];
   maxedges_  = gmax[1];
   maxverts_  = gmax[2];
   maxnto_    = gmax[3];
// maxp0nodes_ = gmax[4]; maxp1nodes_ = gmax[5] ...

  // Compute length for all data corresp to one element:
  lbuff_ =  1                    // no. elements in primary grid/fieldgroup
        +   1                    // no. fields on primary grid
        +   1                    // elemtype  
        +   1                    // key
        +   1                    // root key
        +   3                    // v-exp. orders 
        +   1                    // number vertices
        +   1                    // number edges
        +   1                    // number faces
        +   maxverts_*GDIM       // vertices (if any)
        +   maxverts_            // bdy conditions on vertices
        +   maxedges_            // bdy conditions on edges
        +   maxfaces_            // bdy conditions on faces
        +   nfieldsv_            // number fields
        +   maxlevels_*maxvnodes_*nfieldsv_
                                 // node values for all field time levels
        +   fieldgroups_.size() - 1;
  for ( ng=1; ng<fieldgroups_.size(); ng++ ) {   
    nfields    = fieldgroups_[ng]->nfields;
    maxpnodes_ = gmax[3+ng];
    lbuff_ +=  nfields           // number p-fields
           +   3                 // p-exp. orders 
           +   maxpnodes_        // number p-field-nodes for each time level 
           +   maxlevels_*maxpnodes_*nfields; // nodal field values
  } // end, fieldgroup loop   
  lbuff_  =  lbuff_*maxnto_ + 1; // total size = size for each element times no. elements + no. grids

  // Allocate send/recv buffers:
  sendbuff_.Resize(lbuff_);
  if ( recvbuff_ != NULL ) delete [] recvbuff_;
  if ( nfrom_ > 0 ) {
    recvbuff_ = new GDBuffer [nfrom_];
    for ( j=0; j<nfrom_; j++ ) recvbuff_[j].Resize(lbuff_);
  }
                                 
  return TRUE;

} // end of method CreateSendRecvBuffs


//************************************************************************************
//************************************************************************************
// METHOD     : PostReceives 
// DESCRIPTION: posts receive messages for asynchronous retrieval of data
// ARGUMENTS  : none. 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::PostReceives()
{ 
  char     *serr = "GPartitioner::PostReceives : ";
  GINT     k, len;
  GDOUBLE *buff;

  for ( k=0; k<nfrom_; k++ ) {
    buff        = recvbuff_[k].Data();
    len         = recvbuff_[k].dim();
    GComm::ARecv((void**)&buff, 1, &len, NULL, GC_GDOUBLE, &(iproc_from_[k]),rhandle_[k]);
  }
                                 
  return TRUE;

} // end of method PostRecvs


//************************************************************************************
//************************************************************************************
// METHOD     : WaitOnRecvs 
// DESCRIPTION: waits for receives from all other procs
// ARGUMENTS  : none. 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::WaitOnRecvs()
{ 
  char     *serr = "GPartitioner::WaitOnRecvs : ";
  GINT     k;
  GBOOL    bRet;

  for ( k=0, bRet=TRUE; k<nfrom_ && bRet; k++ ) {
    bRet &= GComm::AWaitOnRecv(rhandle_[k]);
  }

  return bRet;

} // end of method WaitOnRecvs


//************************************************************************************
//************************************************************************************
// METHOD     : SendData
// DESCRIPTION: Sends all element-related data
// ARGUMENTS  : none. 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::SendData()
{ 
  char     *serr = "GPartitioner::SendData: ";
  GINT     k;
  GBOOL    bRet, bPacked;
  GDOUBLE  *buff;
  

  for ( k=0, bRet=TRUE,bPacked=TRUE; k<nto_ && bRet; k++ ) {
    // create send message: 
    if ( !(bPacked=Pack(sendbuff_.Data(),sendbuff_.dim(),index_to_.Data(),index_to_.dim())) ) break;
    buff = sendbuff_.Data();
    bRet  = GComm::BSend((void**)&buff, 1, &lbuff_, NULL, GC_GDOUBLE, &(iproc_to_[k]));
  }

  if ( !bRet    ) cout << serr << "BSend failed " << endl;
  if ( !bPacked ) cout << serr << "Pack comm msg failed " << endl;

  return bRet && bPacked;

} // end of method SendData


//************************************************************************************
//************************************************************************************
// METHOD     : SetFieldGroup
// DESCRIPTION: Sets field groups in addition to the primary group/mesh specified on
//              construction.
// ARGUMENTS  : gelems : GElemList pointer to elements
//              gfields: array of pointers to GFieldLists representing the group fields 
//              nfields: number of fields for each element in gelems list
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::SetFieldGroup(GElemList *gelems, GFieldList **gfields, GINT nfields)
{
  char     *serr = "GPartitioner::SetFieldGroup: ";
  GINT     j;

  if ( gelems == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }
  if ( gfields == NULL || nfields <= 0 ) {
    cout << serr << "NULL field list" << endl;
    exit(1);
  }
  
  // Add fields to fieldgroup structure:
  fieldgroups_.add();
  fieldgroups_.member()->nfields = nfields;
  fieldgroups_.member()->pelems  = gelems;
  fieldgroups_.member()->pfields = new GFieldList * [nfields];
  
  // Set the field group structure's pointers to field lists:
  // is required:
  for ( j=0; j<nfields; j++ ) {
    fieldgroups_.member()->pfields[j] = gfields[j];
  }

  return TRUE;
} // end of method SetFieldGroup



//************************************************************************************
//************************************************************************************
// METHOD     : SetBasisPools
// DESCRIPTION: Sets v- and p- basis pools
// ARGUMENTS  : vbasis  : buffer of v-basis pointers
//              szvbasis: sizes of basis fcns in vbasis 
//              pbasis  : buffer of p-basis pointers
//              szpbasis: sizes of basis fcns in pbasis 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::SetBasisPools(pGLLBuffer *vbasis, GIBuffer *szvbasis,
                                  pGLBuffer *pbasis, GIBuffer *szpbasis)
{ 
  char     *serr = "GPartitioner::SetBasisPools: ";

  if ( vbasis && !szvbasis || pbasis && !szpbasis ) {
    cout << serr << "invalid basis pool definitions" << endl;
    exit(1);
  }

  gbasisv_  = vbasis;
  szbasisv_ = szvbasis;
  gbasisp_  = pbasis;
  szbasisp_ = szpbasis;

  return TRUE;

} // end of method SetBasisPools


//************************************************************************************
//************************************************************************************
// METHOD     : Pack
// DESCRIPTION: packs field data into comm packet
// ARGUMENTS  : cdata   : comm data packet (returned). Must be allocated with 
//                        enough space; no checking performed here.
//              ndata   : max size of comm data packet
//              index   : array of indices of field/element data in lists
//              nindices: fields to pack for each element
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::Pack(GDOUBLE *cdata, GINT ndata, GINT index[], GINT nindices)
{ 
  char       *serr = "GPartitioner::Pack: ";
  GINT       i, j, k, l, n=0, nbdy, nfields, ng;
  GBOOL      bRet=TRUE;
  Point      *vert;
  GIBuffer   *ibdy;
  GVector    *u;
  Elem2D     *elem, *pelem;
  Field2D    *fld;
  GElemList  *gelems;
  GFieldList **gfields;

  nfields = fieldgroups_[0]->nfields;

  cdata[0] = (GDOUBLE) fieldgroups_.size();        n++;      // number of grids/fieldgroups
  
  // Pack primary grid fields
  cdata[n] = (GDOUBLE) nindices;                   n++;      // number of primary grid elements
  cdata[n] = (GDOUBLE) nfields;                    n++;      // number of fields on primary grid
  for ( i=0; i<nindices; i++ ) {
    elem     = (*gelemsv_)[index[i]];
    cdata[n] = (GDOUBLE) elem->ElemType();         n++;      // elem type
    cdata[n] = (GDOUBLE) elem->GetID();            n++;      // elem key
    cdata[n] = (GDOUBLE) elem->GetRootID();        n++;      // elem root key
    for ( k=0; k<3; k++,n++ )                                // v-exp. orders
    cdata[n] = (GDOUBLE) elem->GetOrder(k+1);
    cdata[n] = (GDOUBLE) elem->GetNumVertices();   n++;      // number of vertices
    cdata[n] = (GDOUBLE) elem->GetNumEdges   ();   n++;      // number of edges
    cdata[n] = (GDOUBLE) elem->GetNumFaces   ();   n++;      // number of faces
    for ( k=0; k<elem->GetNumVertices(); k++ ) {
      vert = elem->GetSpVertices(k);
      for ( j=0; j<GDIM; j++,n++ ) {
        cdata[n] = (*vert)[j];                               // elem vertices
      }
    }
    for ( j=0; j<elem->GetNumVertices(); j++,n++ ) 
      cdata[n] = (GDOUBLE) elem->GetVertType(j);             // vertex bc type
    for ( j=0; j<elem->GetNumEdges(); j++,n++ ) 
      cdata[n] = (GDOUBLE) elem->GetEdgeType(j);             // bdy types of edges
    for ( j=0; j<elem->GetNumFaces(); j++,n++ ) 
      cdata[n] = (GDOUBLE) elem->GetFaceType(j);             // bdy types of faces
    cdata[n] = (GDOUBLE) nfieldsv_;                n++;      // number of v-fields
    for ( j=0; j<nfieldsv_; j++ ) {
      fld = (*gfieldsv_[j])[index[i]];
      for ( l=0; l<maxlevels_; l++ ) {
        u = fld->GetExpCoeffs(l);
        cdata[n] = fld->GetTime(l);                n++;       // v-field time level stamp
        for ( k=0; k<u->dim(); k++,n++ ) cdata[n] = (*u)[k];  // v-field vals at this time level
      } 
    }  

    // Pack secondary grid fields:
    for ( ng=1; ng<fieldgroups_.size(); ng++ ) {
      gelems   = fieldgroups_[ng]->pelems;
      nfields  = fieldgroups_[ng]->nfields;
      gfields  = fieldgroups_[ng]->pfields;
      cdata[n] = (GDOUBLE) nfields;                   n++;     // number of p-fields
      elem     = (*gelems)[index[i]];  
      for ( k=0; k<3 && nfieldsp_; k++,n++ )                   // p-exp. orders
      cdata[n] = (GDOUBLE) elem->GetOrder(k+1);
      for ( j=0; j<nfields; j++ ) {
        fld = (*gfields[j])[index[i]];
        for ( l=0; l<maxlevels_; l++ ) {
          u = fld->GetExpCoeffs(l);
          cdata[n] = fld->GetTime(l);                 n++;     // p-field time level stamp
          for ( k=0; k<u->dim(); k++,n++ ) cdata[n] = (*u)[k]; // p-field vals at this time level, if any
        } 
      }  
    }  
    if ( n <= lbuff_ ) {
      cout << serr << "insufficient comm buffer size" << endl;
      exit(1);
    }

  } // end of fld-index loop


  return bRet;

} // end of method Pack



//************************************************************************************
//************************************************************************************
// METHOD     : Unpack (1)
// DESCRIPTION: unpacks field data from comm packet
// ARGUMENTS  : cdata  : comm data packet (returned)
//              ndata  : max size of comm data packet
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::Unpack(GDOUBLE *cdata, GINT ndata)
{ 
  char       *serr = "GPartitioner::Unpack(1): ";
  GINT       i, infields, j, k, l, m, n, nelems, nfields, nfg, ng ;
  GINT       ntmp, ntmpf, P[GDIM], nfieldv, nfieldp=0;
  GINT       ind[GDIM], nverts, nedges, nfaces, nnodes;
  GBOOL      bRet=TRUE;
  Point      *verts=NULL;
  ELEMTYPE   etype;
  GKEY       ekey, rkey;
  GIBuffer   *ibdy;
  GVector    *u;
  GLLBasis   *gllb[GDIM];
  GLBasis    *glb[ GDIM];
  Field2D    *fld;
  GElemList  *gelems;
  GFieldList **gfields;

  if ( gbasisv_ == NULL || szbasisv_ == NULL ) {
    cout << serr << "v-basis pool not specified" << endl;
    exit(1);
  }

  ntmp    = (*gelemsv_    )[0]->GetTempMgr()->GetNumBlocks();
  ntmpf   = (*gfieldsv_[0])[0]->GetTempMgr()->GetNumBlocks();
  nfields = fieldgroups_[0]->nfields;
  n = 0;
  nfg      = (GINT)cdata[n++];                               // number of elements & assoc. fields to add
  if ( nfg != fieldgroups_.size() ) {
    cout << serr << "invalid fieldgroup data" << endl;
    return FALSE;
  }
  nelems   = (GINT)cdata[n++];                               // number of elements & assoc. fields to add
  if ( nelems <= 0 ) {
    cout << serr << "invalid comm data" << endl;
    exit(1);
  }
  infields = (GINT)cdata[n++];                                // number of fields for each element
  if ( nelems != infields ) {
    cout << serr << "invalid comm data" << endl;
    exit(1);
  }
  
  for ( i=0; i<nelems; i++ ) {
    etype  = (ELEMTYPE) cdata[n++];                          // elem type
    ekey   = (GKEY) cdata[n++];                              // elem key
    rkey   = (GKEY) cdata[n++];                              // elem root key
    for ( k=0; k<3; k++,n++ )                                // v-exp. orders
    P[k]   = (GINT) cdata[n];                           
    nverts = (GINT) cdata[n++];                              // number of vertices
    nedges = (GINT) cdata[n++];                              // number of edges
    nfaces = (GINT) cdata[n++];                              // number of faces
    if ( verts != NULL ) delete [] verts;
    verts = new Point [nverts];
    for ( k=0,n=8; k<nverts; k++ ) {
      for ( j=0; j<GDIM; j++,n++ ) {
        verts[k][j] = cdata[n];                              // elem vertices
      }
    }
    for ( j=0; j<GDIM; j++ ) {
      if ( !szbasisv_->contains(P[j],m) ) {
        cout << serr << "required basis unavailable" << endl;
        exit(1);
      }
      gllb[j] = (*gbasisv_)[m];
    }
    gelemsv_->add(etype,ntmp);
    gelemsv_->member()->SetBasis(gllb[0], gllb[1]);
    gelemsv_->member()->SetVertices(verts,nverts);
    gelemsv_->member()->SetRootID(ekey);
    gelemsv_->member()->SetID    (rkey);
    gelemsv_->member()->SolveFE();

    for ( j=0; j<nverts; j++,n++ ) 
      gelemsv_->member()->GetVertType(j) = (BDYTYPE)cdata[n];// set elem vertex bcs
    for ( j=0; j<nedges; j++,n++ ) 
      gelemsv_->member()->GetEdgeType(j) = (BDYTYPE)cdata[n];// set elem edge bcs
    for ( j=0; j<nfaces; j++,n++ ) 
      gelemsv_->member()->GetFaceType(j) = (BDYTYPE)cdata[n];// set elem face bcs

    gelemsv_->member()->ComputeBdyInfo();

    nfieldv = cdata[n++];                                    // get number v-fields
    for ( j=0; j<nfieldv; j++ ) {
      gfieldsv_[j]->add(maxlevels_, gelemsv_->member(), ntmpf); 
      fld = gfieldsv_[j]->member();
      for ( l=0; l<maxlevels_; l++ ) {
        u = fld->GetExpCoeffs(l);
        fld->SetTime(l,cdata[n++]);                          // set v-field time level stamp
        for ( k=0; k<u->dim(); k++,n++ ) u[k] = cdata[n];    // set v-field at this time level
      }
    }
    for ( ng=1; ng<nfg; ng++ ) {
      nfieldp = cdata[n++];                                  // get number p-fields for this grid/fieldgroup
      gelems  = fieldgroups_[ng]->pelems;
      gfields = fieldgroups_[ng]->pfields;
      for ( j=0; j<3 && nfieldp; j++,n++ ) 
        P[j] = cdata[n];                                     // get p-exp.order
      if ( nfieldp > 0 && (gbasisp_ == NULL || szbasisp_ == NULL) ) {
        cout << serr << "p-basis pool not specified" << endl;
        exit(1);
      }
      for ( j=0; j<GDIM && nfieldp; j++ ) {                  // get bases objects to use from pool
        if ( !szbasisp_->contains(P[j],m) ) {
          cout << serr << "required p-basis unavailable" << endl;
          exit(1);
      }
        glb[j] = (*gbasisp_)[m];
      }
      gelems->add(etype,ntmp);                                // set p-element
      gelems->member()->SetVertices(verts, nverts);
      gelems->member()->SetBasis(glb[0], glb[1]);
      gelems->member()->SetRootID(gelemsv_->member()->GetRootID());
      gelems->member()->SetID(gelemsv_->member()->GetID());
      gelems->member()->SolveFE();
  
      for ( j=0; j<nfieldp; j++ ) {              
        gfields[j]->add(maxlevels_, gelems->member(), ntmpf);
        fld = gfields[j]->member();
        for ( l=0; l<maxlevels_; l++ ) {
          u = fld->GetExpCoeffs(l);
          fld->SetTime(l,cdata[n++]);                          // set p-field time level stamp
          for ( k=0; k<u->dim(); k++,n++ ) u[k] = cdata[n];    // set p-field at this time level
        } 
      }

      if ( n <= lbuff_ ) {
        cout << serr << "insufficient comm buffer size" << endl;
        exit(1);
      }
    }

  } // end of fld-index loop

  if ( verts != NULL ) delete [] verts;

  return bRet;

} // end of method Unpack (1)

//************************************************************************************
//************************************************************************************
// METHOD     : Unpack (2)
// DESCRIPTION: unpacks field data from all comm packets
// ARGUMENTS  : none 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::Unpack()
{ 
  char       *serr = "GPartitioner::Unpack(2): ";
  GINT       i;
  GBOOL      bRet;

  for ( i=0, bRet=TRUE; i<nfrom_ && bRet; i++ ) {
    bRet = bRet && Unpack(recvbuff_[i].Data(),recvbuff_[i].dim());
  }

  return bRet;

} // end of method Unpack (2)


//************************************************************************************
//************************************************************************************
// METHOD     : DoPartition
// DESCRIPTION: carries out partitioning
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::DoPartition()
{
  char       *serr = "GPartitioner::DoPartition: ";

  if ( !bInitialized_ ) {
    cout << serr << "object not initialized" << endl;
    exit(1);
  }

  if ( !PostReceives() ) {
    cout << serr << "post of RECVs failed" << endl;
    exit(1);
  }

  if ( !SendData() ) {
    cout << serr << "data SEND failed" << endl;
    exit(1);
  }

  if ( !WaitOnRecvs() ) {
    cout << serr << "WAIT on RECVs failed" << endl;
    exit(1);
  }

  if ( !Unpack() ) {
    cout << serr << "unpack of comm packets failed" << endl;
    exit(1);
  }

  if ( !CleanFieldLists() ) {
    cout << serr << "cleanup of field lists failed" << endl;
    exit(1);
  }

  // If partition successful, next call requires re-initialization:
  bInitialized_ = FALSE;
  
  return TRUE;
    
} // end of method DoPartition


//************************************************************************************
//************************************************************************************
// METHOD     : CleanFieldLists
// DESCRIPTION: removes extraneous elements/fields from their lists after SEND.
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GPartitioner::CleanFieldLists()
{
  char       *serr = "GPartitioner::CleanFieldLists: ";
  GINT       i, is, j, k, nfields, ng;
  Elem2D     *elem;
  Field2D    *fld;
  GElemList  *gelems;
  GFieldList **gfields;

  // location in partition map for elems this proc sent to other procs:
  is = this_rank_ * npartmax_;  

  // Delete from element lists and field lists the elements and 
  // corresp. fields that were already sent off-proc:
  for ( ng=0; ng<fieldgroups_.size(); ng++ ) {
    nfields = fieldgroups_[ng]->nfields;
    gelems  = fieldgroups_[ng]->pelems;
    gfields = fieldgroups_[ng]->pfields;

    // delete sent fields/elems:
    gelems->start(NULL);
    for ( i=0; i<gelems->size(); i++ ) {
      elem = gelems->member();
      for ( j=is; j<is+npartmax_; j++ ) {
        if ( partmap_[j].elem_key == elem->GetID()
          && partmap_[j].proc_to  != this_rank_ ) {
          for ( k=0; k<nfields; k++ ) {
            gfields[k]->del(elem);
          }
          gelems->del(elem);
        }
      } // end, partmap loop
      gelems->next();
    } // end, element loop 

    // Lastly, reorder the lists so that we can access them
    // using random access:
    gelems ->renumber();
    for ( k=0; k<nfields; k++ ) gfields[k]->renumber();
  } // end, fieldgroup/grid loop

  return TRUE;

} // end of method CleanFieldLists
