//************************************************************************************//
// Module       : iconn_amr.cpp
// Date         : 8/25/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                defining a simple (I)nterpolation-based (Conn)ectivity-type
//                (AMR) method.
// Derived From : 
// Modifications:
//************************************************************************************//
#include "iconn_amr.hpp"
#include "gutils.hpp"
#include <string>
#include <stdlib.h>

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
IConnAMR::IConnAMR()
:
nprocs_            (GComm::WorldSize()),
this_rank_         (GComm::WorldRank()),
nglobDomain_       (0),
bInitialized_      (FALSE),
bSynchMortars_     (TRUE),
bDoJInterp_        (TRUE),
bDoJtInterp_       (TRUE),
bDoDataExch_       (TRUE),
bProjection_       (FALSE),
bPreMask_             (FALSE),
bGlobalDomain_     (FALSE),
hDSOp_             (NULL_HANDLE),
globDomain_        (NULL),
gelems_            (NULL),
keygen_            (NULL),
glop_              (NULL)
{
  GINT  i;

  nprocs_     = GComm::WorldSize();
  this_rank_  = GComm::WorldRank();

  // Set 'fuzziness' in spatial points:
  db_midpnt_.Bracket (1.0e-12);     
  db_vertex_.Bracket (1.0e-12);

  glop_       = new GlOp();
  
  for ( i=0; i<GDIM; i++ ) {
     xm_  [i] = new GVector();
     xreg_[i] = new GDBuffer();
  }

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
IConnAMR::IConnAMR(GElemList *in_gelems)
:
nprocs_          (GComm::WorldSize()),
this_rank_       (GComm::WorldRank()),
nglobDomain_     (0),
bInitialized_    (FALSE),
bSynchMortars_   (TRUE),
bDoJInterp_      (TRUE),
bDoJtInterp_     (TRUE),
bDoDataExch_     (TRUE),
bGlobalDomain_   (FALSE),
hDSOp_           (NULL_HANDLE),
globDomain_      (NULL),
gelems_          (in_gelems),
keygen_          (NULL),
glop_            (NULL)
{
  GINT  i;

  nprocs_     = GComm::WorldSize();
  this_rank_  = GComm::WorldRank();

  // Set 'fuzziness' in spatial points:
  db_midpnt_.Bracket(1.0e-12);
  db_vertex_.Bracket(1.0e-12);

  glop_       = new GlOp();
  for ( i=0; i<GDIM; i++ ) {
     xm_  [i] = new GVector();
     xreg_[i] = new GDBuffer();
  }

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
IConnAMR::~IConnAMR()
{
  GINT  i;

  if ( globDomain_  ) delete [] globDomain_; globDomain_ = NULL;
  if ( glop_        ) delete    glop_      ; glop_       = NULL;
  for ( i=0; i<GDIM; i++ ) {
    if ( xm_  [i] ) delete xm_  [i];  
    if ( xreg_[i] ) delete xreg_[i];  
  }

}

//************************************************************************************
//************************************************************************************
// METHOD     : InitDB
// DESCRIPTION: Initialize the DBs with data from the element list
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
GBOOL IConnAMR::InitDB()
{
  GINT    i, j, nv=0, eid, compid, anc=0;
  GBYTE    comptype, mortnum, iendpnt;
  GKEY    ekey=0, rkey=0, pkey=0;
  Point3D *pts, vpt;
  Elem2D  *elem;

  if ( !gelems_ || !gelems_->size() ) {
    cout << "IConnAMR::InitDB: element list empty" << endl;
    return FALSE;
  }
  
  // Build the vertex DB, and the midpoint DB with 
  // information from the elements:
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem       = gelems_->member();
    pts        = elem->GetSpVertices();
    ekey       = elem->GetID();
    rkey       = elem->GetRootID();
    eid        = i;
    comptype   = GELEM_VERTEX;

    for ( j=0; j<elem->GetNumVertices(); j++ ) { // vertex db
      mortnum = (GBYTE)j;
      iendpnt = j==0 || j== 1 ? 0 : 1;
      vpt = pts[j];
      SET_COMP(&compid,&comptype,&mortnum,&iendpnt);
      db_vertex_.add(&vpt, &pkey, &ekey, &rkey, &compid, &anc,
                     &eid, &this_rank_, 1);
      nv++;
    }
    pts        = elem->GetSpMidpoints();
    comptype   = GELEM_PARENT_EDGE;
    iendpnt    = 0;
    for ( j=0; j<elem->GetNumEdges(); j++ ) {    // midpoint db
      mortnum = (GBYTE)j;
      vpt = pts[j];
      SET_COMP(&compid,&comptype,&mortnum,&iendpnt);
      db_midpnt_.add(&vpt, &pkey, &ekey, &rkey, &compid, &anc,
                     &eid, &this_rank_, 1);
    }
    gelems_->next();
  }
#if 0
  cout << endl << endl << endl << " Before Synch..." << endl;
  cout << " rank: " << GComm::WorldRank() << " Vertex_DB:" << endl;
  cout << db_vertex_ << endl << endl;;
  cout << " rank: " << GComm::WorldRank() << " Midpoint_DB:" << endl;
  cout << db_midpnt_ << endl << endl;;
#endif

  // Synch-up dbs on all processors:
  db_vertex_.synch();
  db_midpnt_.synch();

#if 0
  cout << endl << endl << endl << " After Synch..." << endl;
  cout << " rank: " << GComm::WorldRank() << " Vertex_DB:" << endl;
  cout << db_vertex_ << endl << endl;;
  cout << " rank: " << GComm::WorldRank() << " Midpoint_DB:" << endl;
  cout << db_midpnt_ << endl << endl;;
#endif

  return TRUE;

} // end of method InitDB

//************************************************************************************
//************************************************************************************
// METHOD     : SetElements
// DESCRIPTION: Set element list
// ARGUMENTS  : in_elems: pointer to element list.
// RETURNS    : none.
//************************************************************************************
void IConnAMR::SetElements(GElemList *in_elems)
{
  gelems_ = in_elems;
} // end of method SetElements

//************************************************************************************
//************************************************************************************
// METHOD     : Init
// DESCRIPTION: Carries out all required activities related to the domain
//              decomposition.
// ARGUMENTS  : in_elems: GElemList containing linked list of all elements
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::Init()
{
  char *serr = "IConnAMR::Init: ";
  if ( gelems_ == NULL ) {
    cout << serr << "element list must be non-NULL. Call SetElements?" << endl;
    exit(1);
  }

#if defined(IC_TRACE_OUTPUT)
  cout << "&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&" << endl;
  cout << endl << endl << serr << "initializing databases..." << endl;
#endif

  bInitialized_ = FALSE;

  // Empty db contents:
  db_vertex_ .CleanAll();
  db_midpnt_ .CleanAll();
  if ( !InitDB() )  {
    cout << serr << "database initialization failed" << endl;
    return FALSE;
  }
#if defined(IC_TRACE_OUTPUT)
  cout << serr << "mortaring..." << endl;
#endif

  if ( !DoMortaring() )  {
    cout << serr << "mortaring failed" << endl;
    return FALSE;
  }
#if defined(IC_TRACE_OUTPUT)
  cout << serr << "mortaring done." << endl;
#endif

#if defined(IC_TRACE_OUTPUT)
  cout << serr << "finding mortar ids...." << endl;
#endif
  if ( !FindMortarIDs() ) {
    cout << serr << "creation of mortar ids failed" << endl;
    return FALSE;
  }
#if defined(IC_TRACE_OUTPUT)
  cout << serr << "mortar ids found." << endl;
#endif
  
#if defined(IC_TRACE_OUTPUT)
  cout << serr << "Doing GlOp::Init..." << endl;
#endif
  if ( hDSOp_ != NULL_HANDLE ) glop_->FreeHandle(hDSOp_);

  if ( (hDSOp_=glop_->Init(&mortarids_, maxid_)) == NULL_HANDLE ) {
    cout << serr << "connectivity determination failed" << endl;
    return FALSE;
  }
  bInitialized_ = TRUE;
#if defined(IC_TRACE_OUTPUT)
  cout << serr << "GlOp done." << endl;
#endif

#if defined(IC_TRACE_OUTPUT)
  cout << endl << endl << serr << "Initialization complete" << endl;
  cout << "&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&" << endl;
#endif

  return TRUE;

} // end of method Init


//************************************************************************************
//************************************************************************************
// METHOD     : GSOp
// DESCRIPTION: Carries out gather/scatter operation
// ARGUMENTS  : veclist : GVector list to perform operation on. Size of list
//                        must be the same as for the element list provided in the
//                        Init methods.
//              op      : G_OP g/s operation to perform
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::GSOp(GVecList &veclist, G_OP op)
{
  char *serr = "IConnAMR::GSOp: ";
  if ( veclist.size() != gelems_->size() ) {
    cout << serr << "incompatible vector list length" << endl;
    return FALSE;
  }

  if ( !bInitialized_ ) {
    cout << serr << "not initialized" << endl;
    return FALSE;
  }

  // Apply J^T:
  if ( !Host2Mortar(veclist) ) {  
    cout << serr << "SetMortars failed" << endl;
    return FALSE;
  }

  // Do Q~ Q~^T
  if ( bDoDataExch_ && !glop_->DoOp(flat_mortars_.Data(), flat_mortars_.dim(), op, hDSOp_) ) { 
    cout << serr << "GlOp::DSOp failed" << endl;
    return FALSE;
  }

  // Apply J:
  if ( !Mortar2Host(veclist) ) { 
    cout << serr << "SetMortars failed" << endl;
    return FALSE;
  }

  return TRUE;

} // end of method GSOp


//************************************************************************************
//************************************************************************************
// METHOD     : DoMortaring
// DESCRIPTION: Set geometric bdys and other data for mortars, using
//              db_midpnt_ and db_vertex_ DBs.
//              NOTE: method currently valid only for 2D
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::DoMortaring()
{
  GINT          i, j, k, idir, m, m0, m1, m2, m4, m5, m6;
  GINT          ne, nm, iv[4][2]={{0,1},{1,2},{3,2},{0,3}};
  GDOUBLE       sgn;
  GBOOL         bConforming, bParentEdge, bBdyEdge, bPeriodic[2*GDIM];
  BDYTYPE       bctype;
  Point3D       *vertex, *midpoint, vP[2], gvertex[4], gmidpoint[4];
  Elem2D        *elem;
  GVector       *imult;
  GIBuffer      *eindices;
  GVector       *x[GDIM];
  GNeighborList *eneighbor;
  GMortar1D     *mortar;

#if defined(IS3D)
  cout << "IConnAMR::DoMortaring: 3D currently unsupported" << endl;
  exit(1);
#endif

  if ( !bGlobalDomain_ || !globDomain_ ) {
    cout << "IConnAMR::DoMortaring: global domain not specified" << endl;
    exit(1);
  }

  // Re-initialize elemental volatiles:
#if 1
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = gelems_->member();
    imult     = elem->GetNodalMultiplicity();
    mortar    = elem->GetEdgeMortar();
    vertex    = elem->GetSpVertices();
    *imult    = 1.0;
    for ( j=0; j<GDIM; j++ ) {
      x [j] = elem->GetSpNodes(j+1);
    }
    for ( j=0; j<elem->GetNumVertices(); j++ ) {
      elem->GetVNeighbor(j)->empty();
      elem->GetVVNeighbor(j)->empty();
    }
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      elem->GetENeighbor(j)->empty();
      eindices = elem->GetEdgeIndices(j);
      mortar[j].SetBdyPoints(vertex[iv[j][0]], vertex[iv[j][1]]);
      mortar[j].SetHostGrid(x, GDIM, eindices->Data(), eindices->dim());
    }
    gelems_->next();
  }
#endif

  // Consider 1d mortars, and set the mortar endpoints
  // for each edge of each element, and determine neighbors.
  // If vertex is PERIODIC, determine edge that it corresp.to,
  // to determine direction in which it is PERIODIC. 
  // If have a PERIODIC point, must find its periodic ghost point in 
  // order to find neighbors: 
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem     = gelems_->member();

#if 0
    cout << "IConnAMR::DoMortaring: elem[" << i << "]=" << *elem << endl;
#endif
    eneighbor = elem->GetENeighbor();
    vertex    = elem->GetSpVertices();
    midpoint  = elem->GetSpMidpoints();
    mortar    = elem->GetEdgeMortar();
    imult     = elem->GetNodalMultiplicity();
    ne        = elem->GetNumEdges();
    for ( j=0; j<ne  ; j++ ) gmidpoint[j] = midpoint[j];
    for ( j=0; j<ne; j++ ) {
      k           = (j+1) % ne;
      bBdyEdge    = elem->bGlobalBdyEdge(j);
      bctype      = elem->GetEdgeType(j);
      eindices    = elem->GetEdgeIndices(j);
      bPeriodic[j]=  elem->bGlobalBdyEdge(j) && elem->GetEdgeType(j) == PERIODIC ? TRUE : FALSE;
      gvertex  [j] = vertex[j];
      gvertex  [k] = vertex[k];
#if 0
  cout << "IConnAMR::DoMortaring: elem[" << i << "][" << j << "]: periodic: " << bPeriodic[j] << " bBdyEdge: " << bBdyEdge << endl;
#endif
      if ( bBdyEdge && bctype == PERIODIC ) bBdyEdge = FALSE;   
      for ( m=0; m<GDIM; m++ ) {
        xm_[m]->Resize(mortar[j].GetMortarField()->dim());
        elem->GetSpNodes(m+1)->GetSection(eindices->Data(),eindices->dim(),*xm_[m]);
      }
      sgn = j == 0 || j == 3 ? 1.0 : -1.0;
      idir = (j+1)%2;
      for ( m=0; m<GDIM; m++ ) gmidpoint[j][m] = 0.5*(vertex[j][m] + vertex[k][m]);
      if ( bPeriodic[j] ) {
        gvertex  [j][idir] = gvertex  [j][idir] + sgn*gdL_[idir];
        gvertex  [k][idir] = gvertex  [k][idir] + sgn*gdL_[idir];
        gmidpoint[j][idir] = gmidpoint[j][idir] + sgn*gdL_[idir];
//      for ( mm=0; mm<xm_[idir]->dim(); mm++ ) (*xm_[idir])[mm] += sgn*gdL_[idir];
      }
      m0          = db_midpnt_  .multiplicity(&gmidpoint[j]) + (GINT )bPeriodic[j];
      m1          = db_vertex_  .multiplicity(&gvertex  [j]) + (GINT )bPeriodic[j];
      m2          = db_vertex_  .multiplicity(&gvertex  [k]) + (GINT )bPeriodic[j];
//    m3          = db_vertex_  .multiplicity(&gmidpoint[j]);
      m4          = db_midpnt_  .multiplicity(&gvertex  [j]);
      m5          = db_midpnt_  .multiplicity(&gvertex  [k]);
      m6          = db_vertex_  .multiplicity(&gmidpoint[j]); 
      bConforming = m0 == 2 && m1 > 1 && m2 > 1;
//    bParentEdge = m0 == 1 && m1 > 1 && m2 > 1 && m3 > 1 ;
      bParentEdge = m6 == 2;

      eneighbor[j].empty();
      if ( !FindEdgeNeighbors(gvertex[j], m1, gvertex[k], m2,
            gmidpoint[j], m0, m6, vertex[j], vertex[k], bPeriodic,
            i, bBdyEdge, eneighbor[j]) ) {
        cout << "*********************************************************************************" << endl;
        cout << "IConnAMR::DoMortaring: FindEdgeNeighbors failed" << endl;
        cout << "                       edge[" << i << "][" << j << "]: P0=" << gvertex[j] 
             << " P1=" << gvertex[k] << " PMid=" << gmidpoint[j] << endl;
        cout << " m0=" << m0 << " m1=" << m1 << " m2=" << m2 << " m6=" << m6 << endl << endl; 
        cout << "*********************************************************************************" << endl;
        return FALSE;
      }

#if defined(IC_DEBUG_OUTPUT)
      cout << "DoMortaring: eneighbors[" << i << "][" << j << "]=" << eneighbor[j] << endl;
#endif

      if ( bBdyEdge ) continue;  // if non-periodic bdy, there are no 'neighbors'
      if ( bConforming || bParentEdge ) {
#if 0
        cout << "IConnAMR::DoMortaring: elem[" << i << "]; mortar[" << j << "]  endpoints=" 
             << vertex[iv[j][0]] << " " << vertex[iv[j][1]] << "--is conforming" << endl;
#endif
        mortar[j].SetBdyPoints(vertex[iv[j][0]], vertex[iv[j][1]]);
        mortar[j].SetHostGrid(xm_, GDIM, NULL, 0); 
//      if( bPeriodic[j] ) mortar[j].isConforming() = TRUE;
      }
      else {                             // is a child edge
        // If at least one vertex isn't in the midpoint DB, there's an error: 
#if 0
        // or both are, there's an error:
        if ( (m4 != 1 && m5 != 1) || (m4 == 1 && m5 == 1) ) {
          cout << "IConnAMR::DoMortaring: nonconforming database info corrupted" << endl; 
          return FALSE;
        }
#endif
        if ( (m4 != 1 && m5 != 1) ) {
          cout << "IConnAMR::DoMortaring: nonconforming database info corrupted" << endl; 
          return FALSE;
        }
        ComputeMortarBdy(vertex[j], vertex[k], j, m4, vP[0], vP[1]);

#if defined(IC_DEBUG_OUTPUT)
        cout << "IConnAMR::DoMortaring: elem[" << i << "]; mortar[" << j << "]  endpoints=" 
             << vP[0] << " " << vP[1] << "--is NOT conforming" << endl;
#endif
        mortar[j].SetBdyPoints(vP[0], vP[1]);
//      mortar[j].SetHostGrid(xm_, GDIM, eindices->Data(), eindices->dim());
        mortar[j].SetHostGrid(xm_, GDIM, NULL, 0); 
        
      } // end, else
    } // end, edge-loop

    gelems_->next();
  } // end, element loop

  return DoPeriodicMortars();

} // end of method DoMortaring


//************************************************************************************
//************************************************************************************
// METHOD     : SetGlobalDomain
// DESCRIPTION: Sets points defining global domain.
//              Currently used only when defining PERIODIC bc's.
// ARGUMENTS  : domain       : array of points defining domain
//              ndomain      : number of points in arrat
//              igPeriodicity: buffer containing integers, s.t., >0 means that
//                             global bdy is periodic.
// RETURNS    : none.
//************************************************************************************
void IConnAMR::SetGlobalDomain(Point *domain, GINT  ndomain, GIBuffer &igPeriodicity)
{
  GINT     j, nfaces;

  nfaces = (GINT) (pow(2.0, (GINT)(2-GDIM))*ndomain + 2*(GDIM-2));
  if ( igPeriodicity.dim() != nfaces ) {
    cout << "IConnAMR::SetGlobalDomain: number of flags inconsistent with required number of faces" << endl;
    exit(1);
  }
  
  // For rectangular domain:
  if ( globDomain_ != NULL ) delete [] globDomain_; globDomain_ = NULL;
  globDomain_ = new Point [ndomain];

  for ( j=0; j<ndomain; j++ ) {
    globDomain_[j] = domain[j];
  }
  nglobDomain_   = ndomain;
  bGlobalDomain_ = TRUE;

  for ( j=0; j<GDIM; j++ ) {
    gdL_[j] = PDISTANCE(globDomain_[(j+1)%ndomain],globDomain_[j%ndomain]);
  }

  bgPeriodicity_.Resize(nfaces);
  for ( j=0; j<nfaces; j++ ) bgPeriodicity_[j] = (GBOOL)igPeriodicity[j];
#if 0
cout << "gP0=" << globDomain_[j+1] << " gP1=" << globDomain_[j] << 
        "gdL[" << j << "]=" << gdL_[j] << endl;
#endif

} // end of method SetGlobalDomain


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeMortarBdy
// DESCRIPTION: Computes mortar bdy for nonconforming edge, given edge vertices,
//              and multiplicity, mm, of first edge vertex, vj
// ARGUMENTS  : vj   : vertex corresponding to jedge
//              vk   : v_j+1
//              jedge: edge id
//              mm   : multiplicity corresp to vj in midpoint database
//              vP1  : first mortar bdy point
//              vP2  : second mortar bdy point
// RETURNS    : none
//************************************************************************************
void IConnAMR::ComputeMortarBdy(Point  &vj, Point &vk, GINT  jedge, GINT  mm, Point &vP1, Point &vP2)
{
  switch (jedge) { // consider j_th edge of 'child' element:
    case 0: // bottom edge, so adjacent edge is a top edge
      if ( mm == 1 ) { // bottom left corner is midpoint of adjacent edge
        vP1    = vj; vP2 = vk;
        vP1.x1 = 2.0*vP1.x1 - vP2.x1;
      }
      else {           // bottom right corner is midpoint of adjacent edge
        vP1    = vj; vP2 = vk;
        vP2.x1 = 2.0*vP2.x1 - vP1.x1;
      }
      break;
    case 1: // right  edge, so adjacent edge is a left edge
      if ( mm == 1 ) { // bottom right corner is midpoint of adjacent edge
        vP1    = vj; vP2 = vk;
        vP1.x2 = 2.0*vP1.x2 - vP2.x2;
      }
      else {           // top right corner is midpoint of adjacent edge
        vP1    = vj; vP2 = vk;
        vP2.x2 = 2.0*vP2.x2 - vP1.x2;
      }
      break;
    case 2: // top    edge, so adjacent edge is a bottom edge
      if ( mm == 1 ) { // top right corner is midpoint of adjacent edge
        vP1    = vk; vP2 = vj;
        vP2.x1 = 2.0*vP2.x1 - vP1.x1;  
      }
      else {           // top left corner is midpoint of adjacent edge
        vP1    = vk; vP2 = vj;
        vP1.x1 = 2.0*vP1.x1 - vP2.x1;  
      }
      break;
    case 3: // left   edge, so adjacent edge is a right edge
      if ( mm == 1 ) { // top left corner is midpoint of adjacent edge
        vP1    = vk; vP2 = vj;
        vP2.x2 = 2.0*vP2.x2 - vP1.x2;  
      }
      else {           // bottom left corner is midpoint of adjacent edge
        vP1    = vk; vP2 = vj;
        vP1.x2 = 2.0*vP1.x2 - vP2.x2;  
      }
      break;
    default:
      cout << "IConnAMR::ComputeMortarBdy: invalid edge id" << endl;
      exit(1);
  }
} // end of method ComputeMortar


//************************************************************************************
//************************************************************************************
// METHOD     : FindEdgeNeighbors
// DESCRIPTION: Fills neighbor list for edge specified by two endpoints and
//              its midpoint. Also, virtual vertices are detected here, and 
//              included as edge neighbors if they exist. List is emptied before filling.
// ARGUMENTS  : v1    : Point (Point3D) vertex object for one end of edge
//              m1    : multiplicity of v1.
//              v2    : Point (Point3D) vertex object for other end of edge
//              m2    : multiplicity of v2.
//              mid   : Point (Point3D) vertex object for edge midpoint
//              mm    : multiplicity of midpoint in midpt db.
//              md    : multiplicity of midpoint in vertex db.
//              hostid: this edge's element id
//              bGB   : flag stating if this is a global bdy or not
//              list  : GNeighborList object containing edge neighbor information from DB
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::FindEdgeNeighbors(Point &v1 , GINT  m1, Point &v2, GINT  m2,
                                  Point &mid, GINT  mm, GINT  md, Point &pv1, Point &pv2,
                                  GBOOL bPeriodic[], GINT  hostid, GBOOL bGB, 
                                  GNeighborList &elist)
{
  GINT    i, j, ic, ne=4, nsibmort, nrec[2], nmem=2;
  GINT    compid;
  GBYTE   comptype, imort, ipoint;
  GBOOL   bConforming;
  Point   s4[2], vj[2], ept, *pvert[2]={&pv1,&pv2};
  VDBData **record[2]={NULL,NULL}, **sibmortrec=NULL;
  Elem2D  *ehost;

  bConforming = mm == 2 && m1 > 1 && m2 > 1;
//bParentEdge = mm == 1 && m1 > 1 && m2 > 1;
//bParentEdge = md == 2 ;

  if ( bGB ) return TRUE;         // no neighbor; if PERIODIC, is handled outside method
  ehost = (*gelems_)[hostid];

  if ( ehost == NULL ) {
    cout << "IConnAMR::FindEdgeNeighbors: NULL host element" << endl;
    return FALSE;
  }

  // Specified edge is conforming...
  if ( bConforming ) {
    if ( db_midpnt_.record(mid, record[0], nrec[0]) == NULL ) {
      cout << "IConnAMR::FindEdgeNeighbors: conforming edges not found" << endl;
      return FALSE;
    }
#if 0
    if ( nrec[0] != mm ) {
      cout << "IConnAMR::FindEdgeNeighbors: inconsistent number of conforming edges " << endl;
      return FALSE;
    }
#endif
    for ( i=0; i<nrec[0]; i++ ) {
      if ( hostid != record[0][i]->hostid() || this_rank_ != record[0][i]->proc  () ) {
        elist.add();
        elist.member()->proc       () = record[0][i]->proc       ();  // neighbor's proc id
        elist.member()->elemid     () = record[0][i]->hostid     ();  // neighbor's element's local id
        elist.member()->elemkey    () = record[0][i]->hostkey    ();  // neighbor's element's local id
        elist.member()->elemrootkey() = record[0][i]->hostrootkey();  // neighbor's element's local id
        elist.member()->compid     () = record[0][i]->localid    ();  // neighbor's edge id 
        elist.member()->ancillary  () = 0                          ;  // neighbor's ancillary data
        
      }
    }
    for ( i=0; i<nmem; i++ ) {
      if ( record[i] ) delete [] record[i]; record[i] = NULL;
    }
    return TRUE;
  }

  // Check if specified edge is a 'child' edge;
  // s.t. one of the endpoints is in the midpoint DB.
  db_midpnt_.record(v1, record[0], nrec[0]);
  db_midpnt_.record(v2, record[1], nrec[1]);
#if 0
  // If both endpoints are in midpoint DB, it's an error:
  if ( nrec[0] > 0 && nrec[1] > 0 ) {
    cout << "IConnAMR::FindEdgeNeighbors: nonconforming database error" << endl;
    return FALSE;
  }
#endif
//cout << "DoMortaring: nrec[0]=" << nrec[0] << " nrec[1]=" << nrec[1] << endl;
  
  if ( nrec[0] || nrec[1] ) {
    // Edge is a 'child' edge, endpoint should
    // appear precisely once in midpoint DB, and 
    // this midpoint should belong to the 'parent' neighbor.
    for ( i=0; i<nmem; i++ ) {
      if ( nrec[i] > 0 ) {
        if ( nrec[i] != 1  ) {
          cout << "IConnAMR::FindEdgeNeighbors: nonconforming edge inconsistency (1)" << endl;
          return FALSE;
        }
        if ( record[i] ) {
        elist.add();   
        elist.member()->proc       () = record[i][0]->proc       ();   // neighbor proc id
        elist.member()->elemid     () = record[i][0]->hostid     ();   // neighbor element's local id
        elist.member()->elemkey    () = record[i][0]->hostkey    ();   // neighbor element's GKEY
        elist.member()->elemrootkey() = record[i][0]->hostrootkey();   // neighbor element's root key
        elist.member()->compid     () = record[i][0]->localid    ();   // neighbor edge id
        elist.member()->ancillary  () = 0                          ;   // neighbor's ancillary data
        }

        db_vertex_.record(*pvert[i], sibmortrec, nsibmort);
        if ( nsibmort != 2 ) {
          cout << "IConnAMR::FindEdgeNeighbors: nonconforming edge inconsistency (2)" << endl;
          return FALSE;
        }
        // Must find child edge's sibling , which will complete its mortar:
        for ( ic=0; ic<nsibmort; ic++ ) {
          if ( hostid != sibmortrec[ic]->hostid() || this_rank_ != sibmortrec[ic]->proc  () ) {
            elist.add();
            elist.member()->proc       () = sibmortrec[ic]->proc       ();  // neighbor's proc id
            elist.member()->elemid     () = sibmortrec[ic]->hostid     ();  // neighbor's element's local id
            elist.member()->elemkey    () = sibmortrec[ic]->hostkey    ();  // neighbor's element's GKEY
            elist.member()->elemrootkey() = sibmortrec[ic]->hostrootkey();  // neighbor's element's root key
            compid                        = sibmortrec[ic]->localid    ();  // neighbor's vertex id
            comptype                      = GELEM_CHILD_EDGE; imort = COMPNUM(&(record[i][0]->localid()));
            ipoint                        = 0;
            imort                         = (imort + 2) % ne;
            SET_COMP(&compid,&comptype,&imort,&ipoint);                    // create co-mortar edge id
            elist.member()->compid     () = compid;                        // set component id
            elist.member()->ancillary  () = 0;                             // neighbor's ancillary data
        
          }
        }
        if ( sibmortrec ) delete [] sibmortrec; sibmortrec = NULL;
      }
    }
    for ( i=0; i<nmem; i++ ) {
      if ( record[i] ) delete [] record[i]; record[i] = NULL;
    }
    return TRUE;
  }
   
  // Edge must be a 'parent' edge, so it must have precisely 
  // 2 neighbors; thus the 2 neighbors must have their centers
  // at the quarter-lengths of the specified edge:
  vj[0] = v1; vj[1] = v2; 
  for ( i=0; i<nmem; i++ ) {
    for ( j=0; j<GDIM; j++ ) {
      s4[i][j] = 0.5*(vj[i][j]+mid[j]);
    }
    db_midpnt_.record(s4[i], record[i], nrec[i]);
#if 0
    for ( j=0; j<GDIM; j++ ) {
      s4[i][j] = (1.0-f[i])*v1[j] + f[i]*v2[j];
    }
    db_midpnt_.record(s4[i], record[i], nrec[i]);
    if ( nrec[i] != 1 ) {
      for ( j=0; j<GDIM; j++ ) {
        s4[i][j] = (1.0-f[i])*v2[j] + f[i]*v1[j];
      }
      db_midpnt_.record(s4[i], record[i], nrec[i]);
    }
#endif
    if ( nrec[i] != 1 ) {
      cout << "  rank= " << GComm::WorldRank() << ": IConnAMR::FindEdgeNeighbors: nonconforming edge inconsistency (3)" << endl;
#if defined(IC_DEBUG_OUTPUT)
cout << "  nrec[" << i << "]=" << nrec[i] << endl;
cout << "  v1 = " << v1  << " v2=" << v2 << endl;
cout << "i=" << i << "  s4[0] = " << s4[0] << " s4[1]=" << s4[1] << " db_mid=" << db_midpnt_ << endl << endl
     << endl << " db_vert=" << db_vertex_ << endl << endl << endl;
#endif
      return FALSE;
    }
    elist.add();   
    elist.member()->proc       () = record[i][0]->proc       ();       // neighbor proc id
    elist.member()->elemid     () = record[i][0]->hostid     ();       // neighbor element's local element id
    elist.member()->elemkey    () = record[i][0]->hostkey    ();       // neighbor element's key
    elist.member()->elemrootkey() = record[i][0]->hostrootkey();       // neighbor element's root key 
    elist.member()->compid     () = record[i][0]->localid    ();       // neighbor edge's component id
    elist.member()->ancillary  () = 0                          ;       // neighbor's ancillary data
  }

  for ( i=0; i<nmem; i++ ) {
    if ( record[i] ) delete [] record[i]; record[i] = NULL;
  }
  return TRUE;

} // end of method FindEdgeNeighbors


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoJInterp
// DESCRIPTION: Sets flag to do J-interpolations
// ARGUMENTS  : bInterp : if TRUE, apply J, J^T; else, don't
// RETURNS    : none
//************************************************************************************
void IConnAMR::SetDoJInterp(GBOOL bInterp)
{
  bDoJInterp_ = bInterp;
} // end of SetDoiJInterp


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoJtInterp
// DESCRIPTION: Sets flag to do Jt-interpolations
// ARGUMENTS  : bInterp : if TRUE, apply J, J^T; else, don't
// RETURNS    : none
//************************************************************************************
void IConnAMR::SetDoJtInterp(GBOOL bInterp)
{
  bDoJtInterp_ = bInterp;
} // end of SetDoJtInterp


//************************************************************************************
//************************************************************************************
// METHOD     : DoProjection
// DESCRIPTION: Sets flag for doing a projection
// ARGUMENTS  : bProj : if TRUE, do projection; else, don't
// RETURNS    : none
//************************************************************************************
void IConnAMR::DoProjection(GBOOL bProj)
{
  bProjection_ = bProj;
} // end of DoProjection


//************************************************************************************
// METHOD     : DoMult
// DESCRIPTION: Sets flag for using multiplicity in projection
// ARGUMENTS  : bProj : if TRUE, use mult; else, don't
// RETURNS    : none
//************************************************************************************
void IConnAMR::DoMult(GBOOL bUseMult)
{ 
  bUseMult_ = bUseMult;
} // end of DoMult


//************************************************************************************
//************************************************************************************
// METHOD     : UseMask
// DESCRIPTION: Sets flag for using mask before applying J
// ARGUMENTS  : bPreMask : if TRUE, use mask; else, don't
// RETURNS    : none
//************************************************************************************
void IConnAMR::PreMask(GBOOL bPreMask)
{
  bPreMask_ = bPreMask;
} // end of UseMask


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoExchange
// DESCRIPTION: Sets flag to do data exchange
// ARGUMENTS  : bExch : if TRUE, do data exchage, else don't
// RETURNS    : none
//************************************************************************************
void IConnAMR::SetDoExchange(GBOOL bExch)
{
  bDoDataExch_ = bExch;
} // end of SetDoExchange


//************************************************************************************
//************************************************************************************
// METHOD     : SetKeyGen
// DESCRIPTION: Sets mortar id generator object
// ARGUMENTS  : keygen: pointer to key generator object
// RETURNS    : none
//************************************************************************************
void IConnAMR::SetKeyGen(GKeyGen *keygen)
{ 
  keygen_ = keygen;
} // end of SetKeyGen


//************************************************************************************
//************************************************************************************
// METHOD     : FindMortarIDs
// DESCRIPTION: Creates mortar node ids  for use in GlOp operator
// ARGUMENTS  : none
// RETURNS    : 
//************************************************************************************
GBOOL IConnAMR::FindMortarIDs()
{
  char         *serr = "IConnAMR::FindMortarIDs: ";
  GINT         i, j, k, m, n, is;
  GNODEID      lmaxid=0;
  GDOUBLE      *x[GDIM];
  Point        p, rlen;
  GNIDBuffer   ibuff;
  Elem2D       *elem;

  if ( !glop_) {
    cout << serr << "NULL connectivity operator" << endl;
    return FALSE;
  }
  if ( !keygen_) {
    cout << serr << "key generator not set" << endl;
    return FALSE;
  }
  if ( !bGlobalDomain_ || !globDomain_ ) {
    cout << serr << "global domain not set" << endl;
    return FALSE;
  }

  for ( i=0,n=0; i<gelems_->size(); i++ ) {
    elem = (*gelems_)[i];
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      n += elem->GetEdgeMortar(j)->GetMortarField()->dim();
    }
  } 
  mortarids_.Resize(n);

  for ( i=0,is=0; i<gelems_->size(); i++ ) {
    elem = (*gelems_)[i];
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      n    = elem->GetEdgeMortar(j)->GetSpGrid(1)->dim(); // same dim for each dir
#if 1
      for ( k=0; k<GDIM; k++ ) { 
        x   [k] = elem->GetEdgeMortar(j)->GetSpGrid(k+1)->Data();
      }
#else
      RegularizeMortar(xreg_, elem->GetEdgeMortar(j));
      for ( k=0; k<GDIM; k++ ) { 
        x   [k] = xreg_[k]->Data();
      }
#endif
      ibuff.Resize(n);
      keygen_->key(ibuff.Data(), sizeof(GNODEID),  x, n);
      for ( m=0; m<n; m++ ) {
        *(mortarids_.Data()+is+m) = (GNODEID)fabs((GDOUBLE)(*(ibuff.Data()+m)));
//      mortarids_[m+is] = ibuff[m];
        lmaxid = MAX(lmaxid, (mortarids_[is+m]));
#if 0
        mortarids_[m+is] = ibuff[m];
        lmaxid = MAX(lmaxid, mortarids_[is+m]);
#endif
      }
#if defined(IC_DEBUG_OUTPUT)
      if ( icycle_ == 60 && (i == 21 || i == 22 || i == 40 || i == 43 ) )  {
        for ( m=0; m<n; m++ ) {
          for ( k=0; k<GDIM; k++ ) p[k] = x[k][m];
            cout << GComm::WorldRank() << " " << serr << "Px        [" << i << "][" << j << "][" << m << "]=" << p 
//               << " ibuff[" << i << "][" << j << "][" << m << "]=" << ibuff[m]
                 << " mortarid[" << i << "][" << j << "][" << m << "]=" << mortarids_[is+m]<< endl;
        }
      }

      
#endif
      is += n;
    }
  } 

  GComm::Allreduce(&lmaxid, &maxid_, 1, GC_GNODEID, G_OP_MAX);
#if defined(IC_DEBUG_OUTPUT)
  cout << serr <<  "maxid     =" << maxid_     << endl;
#endif
  
#if 0
GINT    gnids, lnids, nt ;
GNODEID icmp;
GNIDBuffer gids, lids;
lnids = mortarids_.dim();
GComm::Allreduce(&lnids, &gnids, 1, GC_GINT, G_OP_MAX);
nt = GComm::WorldSize()*gnids;
gids.Resize   (nt); gids = -1;
lids.Resize(gnids); lids = -1;
for ( j=0; j<lnids; j++ ) lids[j] = mortarids_[j];
GComm::Allgather(lids.Data(),gnids,GC_GNODEID,gids.Data(),gnids,GC_GNODEID);
for ( i=0; i<nt; i++ ) {
icmp = gids[i];
if ( icmp == -1 ) break;
if ( icmp < 0 ) {
cout << serr << "Negative mortarid[" << i << "]=" << icmp << endl;
}
for ( j=0, n=0; j<nt && gids[j]!=-1; j++ ) {
  n += (i!=j && icmp==gids[j]) ? 1 : 0;
}
if ( n > 4 ) {
cout << serr << " mortarid[" << i << "]=" << icmp << " mult=" << n << endl;
}
}
#endif

  return TRUE;

} // end of method FindMortarIDs 


//************************************************************************************
//************************************************************************************
// METHOD     : Host2Mortar
// DESCRIPTION: Apply J^T: interpolate (where necessary) from host element to 
//              mortars; create 'flat' mortar data array.
// ARGUMENTS  : veclist: vector list, where, for each element, corresp. vector 
//              covers entire element.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::Host2Mortar(GVecList &veclist)
{
  GINT      i, j, k, is;
  GVector   *u, *mf, *imult, *mask;
  GIBuffer  *ie;
  GMortar1D *mortar;
  Elem2D    *elem;

  //  mortarids_ set in FindMortarIDs method...
  flat_mortars_.Resize(mortarids_.dim());

#if defined(IC_DEBUG_OUTPUT)
  cout << "IConnAMR::Host2Mortar: begin**************************************" << endl;
  cout << "IConnAMR::Host2Mortar: bProjection: " << bProjection_ << endl;
#endif
  // If a projection (to maintain H1), scale by multiplicity:
  for ( i=0, is=0; i<gelems_->size() && bProjection_; i++ ) {
    u         = veclist[i];
    imult     = (*gelems_)[i]->GetNodalMultiplicity();
#if 0
    cout << "IConnAMR::Host2Mortar: imult[" << i << "]=" << *imult << endl;
#endif
    for ( k=0; k<u->dim(); k++ ) (*u)[k] *= (*imult)[k];
  }

  // Multiply by mask, if required:
  for ( i=0; i<gelems_->size() && bPreMask_; i++ ) {
    u         = veclist[i];
    mask      = (*gelems_)[i]->GetMask();
    MTK::fvec_point_prod_rep(*u, *mask);
  }

  for ( i=0; i<gelems_->size() && bDoJtInterp_; i++ ) {
    elem      = (*gelems_)[i];
    u         = veclist[i];
    mortar    = elem->GetEdgeMortar();
    imult     = elem->GetNodalMultiplicity();
    ie        = elem->GetEdgeIndices();

    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      if ( mortar[j].isConforming() ) continue;
      mf = mortar[j].GetMortarField();
      mortar[j].Host2Mortar(u, ie[j].Data(), ie[j].dim(), 0);
      if ( bProjection_ ) (*mf) = 0.0;
      u->Set(0.0, ie[j].Data(), ie[j].dim());
#if 0
      cout << "IConnAMR::Host2Mortar: mortar_field[" << i << "][" << j << "]=" << *mf << endl;
#endif
    }
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      if ( !mortar[j].isConforming() ) continue;
      mortar[j].Host2Mortar(u, ie[j].Data(), ie[j].dim(), 0);
      u->Set(0.0, ie[j].Data(), ie[j].dim());
#if 0
      mf = mortar[j].GetMortarField();
      cout << "IConnAMR::Host2Mortar: mortar_field[" << i << "][" << j << "]=" << *mf << endl;
#endif
    }
  }

#if defined(IC_DEBUG_OUTPUT)
//cout << endl << endl;
#endif
  // Pack 'flat' mortar list:
  for ( i=0, is=0; i<gelems_->size(); i++ ) {
    elem      = (*gelems_)[i];
    mortar    = elem->GetEdgeMortar();
    for ( j=0; j<elem->GetNumEdges() && bDoJtInterp_; j++ ) {
      mf = mortar[j].GetMortarField();
      for ( k=0; k<mf->dim(); k++ ) flat_mortars_[k+is] = (*mf)[k];
      is += mf->dim();
    }
  }
#if 0
  cout << "IConnAMR::Host2Mortar: flat_mortars_=" << flat_mortars_ << endl;
#endif

#if defined(IC_DEBUG_OUTPUT)
  cout << "IConnAMR::Host2Mortar: end****************************************" << endl;
#endif
#if defined(IC_DEBUG_OUTPUT)
//cout << endl << endl;
#endif
  return TRUE; 

} // end of method Host2Mortar


//************************************************************************************
//************************************************************************************
// METHOD     : Mortar2Host
// DESCRIPTION: Apply J: interpolate (where necessary) from mortars to host element 
// ARGUMENTS  : veclist: vector list, where, for each element, corresp. vector 
//              covers entire element.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::Mortar2Host(GVecList &veclist)
{
  GINT      i, j, k, is ;
  GVector   *u, *mf, *mask;
  GIBuffer  *ie;
  GMortar1D *mortar;
  Elem2D    *elem;
#if defined(IC_DEBUG_OUTPUT)
  GVector edge;
#endif

#if defined(IC_DEBUG_OUTPUT)
  cout << endl << endl << "IConnAMR::Mortar2Host: begin**************************************" << endl;
  cout << "IConnAMR::Mortar2Host: bProjection: " << bProjection_ << endl;
#endif
#if 0
  cout << "IConnAMR::Host2Mortar: flat_mortars_=" << flat_mortars_ << endl;
#endif
  // Reset structured mortar mortar data to prepare for interpolation to
  // elemental (local) DOFs:
  for ( i=0,is=0; i<gelems_->size() && bDoJInterp_; i++ ) {
    elem      = (*gelems_)[i];
    mortar    = elem->GetEdgeMortar();
    ie        = elem->GetEdgeIndices();
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      mf = mortar[j].GetMortarField();
      for ( k=0; k<mf->dim(); k++ ) (*mf)[k] = flat_mortars_(k+is);
#if 0
      cout << "IConnAMR::Mortar2Host: after DSS: mortar[" << i << "][" << j << "]=" << *mf << endl;
#endif
      is += mf->dim();
    }
  }

  // Multiply by mask, if required:
  for ( i=0; i<gelems_->size() && bPreMask_; i++ ) {
    elem      = (*gelems_)[i];
    u         = veclist[i];
    mortar    = elem->GetEdgeMortar();
    mask      = elem->GetMask();
    ie        = elem->GetEdgeIndices();
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      mf = mortar[j].GetMortarField();
      for ( k=0; k<mf->dim(); k++ ) (*mf)[k] *= (*mask)[ie[j][k]];
    }
  }

#if defined(IC_DEBUG_OUTPUT)
      cout << endl << "Begin with...conforming faces: " << endl;
#endif
  // Apply J: do interpolations from mortar (global) DOFs to elemental nodes:
  // ...do conforming faces first:
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = (*gelems_)[i];
    u         = veclist[i];
    mortar    = elem->GetEdgeMortar();
    ie        = elem->GetEdgeIndices();
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      if ( !mortar[j].isConforming() ) continue;
      mortar[j].Mortar2Host(u, ie[j].Data(), ie[j].dim(), 0);
#if 0
      mf = mortar[j].GetMortarField();
      cout << "IConnAMR::Mortar2Host: mortar[" << i << "][" << j << "]=" << *mf << endl;
#endif
    }
  }
#if defined(IC_DEBUG_OUTPUT)
      cout << endl << "Finish with...nonconforming faces: " << endl;
#endif
  // ...do nonconforming faces:
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = (*gelems_)[i];
    u         = veclist[i];
    mortar    = elem->GetEdgeMortar();
    ie        = elem->GetEdgeIndices();
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
      if ( mortar[j].isConforming() ) continue;
      mortar[j].Mortar2Host(u, ie[j].Data(), ie[j].dim(), 0);
#if 0
      mf = mortar[j].GetMortarField();
      cout << "IConnAMR::Mortar2Host: mortar[" << i << "][" << j << "]=" << *mf << endl;
#endif
#if 0
      edge.Resize(ie[j].dim());
      u->GetSection(ie[j].Data(), ie[j].dim(), edge);
      cout << "IConnAMR::Mortar2Host: edge[" << i << "][" << j << "]=" << edge << endl;
#endif
    }
  }
#if defined(IC_DEBUG_OUTPUT)
  cout << "IConnAMR::Mortar2Host: end****************************************" << endl << endl << endl;
#endif

  return TRUE;

} // end of method Mortar2Host 


//************************************************************************************
//************************************************************************************
// METHOD     : SetSynch
// DESCRIPTION: Sets synch flag
// ARGUMENTS  : synch: synch, TRUE or FALSE
// RETURNS    : none
//************************************************************************************
void IConnAMR::SetSynch(GBOOL synch)
{
  bSynchMortars_ = synch;
} // end of SetSynch


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : DoPeriodicMortars
// DESCRIPTION: Reconciles common mortar points if there are periodic bdys
//              NOTE: after a return from this method, the mortar bdy points along
//                    the global bdy are no longer consistent with their
//                    mortar spatial grid components.
// ARGUMENTS  : none
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::DoPeriodicMortars()
{
  GINT      i, j, k, m, jj, jk, md, ne, nv;
  GDOUBLE     sgn, xory;
  GBOOL     bPeriodic[2], bOnBdy;
  GMortar1D *mortar;
  Elem2D    *elem;

  if ( !bGlobalDomain_ ) {
    cout << "IConnAMR::DoPeriodicMortars: global domain not set" << endl;
    return FALSE;
  }
#if defined(IC_DEBUG_OUTPUT)
  for ( k=0; k<GDIM; k++ ) {  
    cout << "IConnAMR::DoPeriodicMortars: gdL[" << k << "]=" << gdL_[k] << endl;
  }
#endif

  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = gelems_->member();
    mortar    = elem->GetEdgeMortar();
    nv        = elem->GetNumVertices();
    ne        = elem->GetNumEdges();

    // Adjust mortar vertices if edges are periodic (
    // NOTE: good only for 2d quads):
    for ( j=0; j<ne; j++ ) {
      bPeriodic[0] = elem->GetEdgeType(1) == PERIODIC || elem->GetEdgeType(3) == PERIODIC;
      bPeriodic[1] = elem->GetEdgeType(0) == PERIODIC || elem->GetEdgeType(2) == PERIODIC;
      if ( !bPeriodic[0] && !bPeriodic[1] ) continue;

      md = mortar[j].GetMortarField()->dim();
      for ( m=0; m<2; m++ ) {  // cycle over mortar end-points:
        for ( k=0; k<GDIM; k++ ) {  
          if ( !bPeriodic[k] ) continue;
          xory = (*(mortar[j].GetSpGrid(k+1)))[m*(md-1)];         
          for ( jj=0, bOnBdy=FALSE; jj<nglobDomain_ && !bOnBdy; jj+=2 ) {  // check diag points:
            bOnBdy = xory == globDomain_[jj][k]; jk = jj;
          }
          sgn = bOnBdy && jk==0 ? 1.0 : 0.0; 
          (*(mortar[j].GetSpGrid(k+1)))[m*(md-1)]  = xory + sgn*gdL_[k];
        }      
      }      
    }
    gelems_->next();
  }

  return TRUE;
} // end of method DoPeriodicMortars
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : DoPeriodicMortars
// DESCRIPTION: Reconciles common mortar points if there are periodic bdys
//              Note: Make sure that Mortar::SetHostGrid has been called
//                    prior to entry, so that the interp. matrices are
//                    properly constructed. The purpose here is to make sure
//                    that periodic bdys are properly reflected in mortar
//                    coordinates being equal in the periodic coord directions.
// ARGUMENTS  : none
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::DoPeriodicMortars()
{
  GINT      i, j, k, m, md, ne, nv;
  GBOOL     bOnBdy;
  Point     v0, v1, vn[GDIM];
  GMortar1D *mortar;
  Elem2D    *elem;

  if ( !bGlobalDomain_ ) {
    cout << "IConnAMR::DoPeriodicMortars: global domain not set" << endl;
    return FALSE;
  }
#if defined(IC_DEBUG_OUTPUT)
  for ( k=0; k<GDIM; k++ ) {  
    cout << "IConnAMR::DoPeriodicMortars: gdL[" << k << "]=" << gdL_[k] << endl;
  }
#endif

//ndv = bgPeriodicity_.dim();

#if 0
  // Consider interior mortar points:
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = gelems_->member();
    mortar    = elem->GetEdgeMortar();
    nv        = elem->GetNumVertices();
    ne        = elem->GetNumEdges();
    for ( j=0; j<ne; j++ ) {
      md = mortar[j].GetMortarField()->dim();
      mg = mortar[j].GetSpGrid(id[j]+1);
      for ( m=0; m<md && elem->GetEdgeType(j) == PERIODIC; m++ ) {  
        (*mg)[m] += vsgn[j][id[j]]*gdL_[id[j]];
      }
    }
    gelems_->next();
 }
#endif

#if 1
  // Consider mortars that aren't
  // entirely PERIODIC but that have an endpoint
  // lying on a PERIODIC bdy:
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = gelems_->member();
    mortar    = elem->GetEdgeMortar();
    nv        = elem->GetNumVertices();
    ne        = elem->GetNumEdges();

    for ( j=0; j<nv; j++ ) {
      md = mortar[j].GetMortarField()->dim();
      // consider all mortar points:
      for ( m=0; m<md; m++ ) {  
        for ( k=0; k<GDIM; k++ ) v0[k] = (*(mortar[j].GetSpGrid(k+1)))[m];         
        v1 = v0;
#if defined(IC_DEBUG_OUTPUT)
        cout << "DoPeriodicMortars: mortar[" << i << "][" << j << "][" << m << "]_init =" <<
        v0 << endl;
#endif
        bOnBdy = GUtils::PointOnSeg(globDomain_[0], globDomain_[1], v0);
        if ( bOnBdy && bgPeriodicity_[0] && bgPeriodicity_[2] ) v1[1] += gdL_[1];
        bOnBdy = GUtils::PointOnSeg(globDomain_[0], globDomain_[3], v0);
        if ( bOnBdy && bgPeriodicity_[1] && bgPeriodicity_[3] ) v1[0] += gdL_[0];
        for ( k=0; k<GDIM; k++ ) (*(mortar[j].GetSpGrid(k+1)))[m] = v1[k];         
#if defined(IC_DEBUG_OUTPUT)
        cout << "DoPeriodicMortars: mortar[" << i << "][" << j << "][" << m << "]_final=" << v1 << endl;
#endif
      } 
    }
    gelems_->next();
  }
#endif

#if 0
  // Adjust mortar vertices if edges are periodic (
  // NOTE: good only for 2d quads):
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem      = gelems_->member();
    mortar    = elem->GetEdgeMortar();
    ne        = elem->GetNumEdges();
    for ( j=0; j<ne; j++ ) {
      bPeriodic[0] = elem->GetEdgeType(1) == PERIODIC || elem->GetEdgeType(3) == PERIODIC;
      bPeriodic[1] = elem->GetEdgeType(0) == PERIODIC || elem->GetEdgeType(2) == PERIODIC;
      if ( !bPeriodic[0] && !bPeriodic[1] ) continue;

      md = mortar[j].GetMortarField()->dim();
      for ( m=0; m<2; m++ ) {  // cycle over mortar end-points:
        for ( k=0; k<GDIM; k++ ) {  
          if ( !bPeriodic[k] ) continue;
          xory = (*(mortar[j].GetSpGrid(k+1)))[m*(md-1)];         
          for ( jj=0, bOnBdy=FALSE; jj<nglobDomain_ && !bOnBdy; jj+=2 ) {  // check diag points:
            bOnBdy = xory == globDomain_[jj][k]; jk = jj;
          }
          sgn = bOnBdy && jk==0 ? 1.0 : 0.0; 
          (*(mortar[j].GetSpGrid(k+1)))[m*(md-1)]  = xory + sgn*gdL_[k];
          vn[m][k] = (*(mortar[j].GetSpGrid(k+1)))[m*(md-1)] ;
        }      
      }      
      cout << "DoPeriodicMortars: mortar[" << i << "][" << j << "]: v0=" 
           << vn[0] << " v1=" << vn[1] << endl;
    }
    gelems_->next();
  }
#endif

  return TRUE;
} // end of method DoPeriodicMortars


//************************************************************************************
//************************************************************************************
// METHOD     : WrapPeriodicVPoint
// DESCRIPTION: Takes a vertex point, and 'wraps' it to its logical periodic
//              neighbor, but not in both directions!
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::WrapPeriodicVPoint(Point &vpt, GINT  ivert, Elem2D &elem)
{
  GINT    jj, jk, k;
  GBOOL   bOnBdy;
  GDOUBLE   sgn[2];
        
#if 0
  sgn[0] = ivert == 0 || ivert == 3 ? 1.0 : 0.0;
  sgn[1] = ivert == 0 || ivert == 1 ? 1.0 : 0.0;
  for ( j=0; j<GDIM; j++ ) {
    if ( elem.GetEdgeType(idir[ivert][j]) == PERIODIC ) {
      vpt[j] += ( sgn[j]*gdL_[j] );
    }
  }
#endif

  for ( k=0; k<GDIM; k++ ) {  
    for ( jj=0, bOnBdy=FALSE; jj<nglobDomain_ && !bOnBdy; jj+=2 ) {  // check diag points:
      bOnBdy = vpt [k]== globDomain_[jj][k]; jk = jj;
    }
    sgn[k] = bOnBdy && jk==0 ? 1.0 : 0.0; 
    vpt[k] = vpt[k] + sgn[k]*gdL_[k];
  }      

  return TRUE;
} // end of WrapPeriodicVPoint


//************************************************************************************
//************************************************************************************
// METHOD     : WrapPeriodicEPoint
// DESCRIPTION: Takes an edge point, and 'wraps' it to its logical periodic
//              neighbor
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL IConnAMR::WrapPeriodicEPoint(Point &vpt, GINT  iedge, Elem2D &elem)
{
  GINT    idir[4]= {1,0,1,0};
  GDOUBLE   sgn;

  sgn = iedge == 0 || iedge == 3 ? 1.0 : 0.0;
  if ( elem.GetEdgeType(iedge) == PERIODIC ) {
    vpt[idir[iedge]] +=  ( sgn*gdL_[idir[iedge]] );
  }

  return TRUE;
} // end of WrapPeriodicEPoint

//************************************************************************************
//************************************************************************************
// METHOD     : RegularizeMortar
// DESCRIPTION: Takes mortar coords and regularizes them, storing the
//              results in xreg buffer array.
//              NOTE: this method good, stricly speaking, only for 1D mortars.
// ARGUMENTS  : 
//              xreg : buffer array of length GDIM, with each buffer allocated
//                     here to length of mortar coords. 
//              mortar: mortar to be regularized
//             
// RETURNS    : none
//************************************************************************************
void IConnAMR::RegularizeMortar(GDBuffer *xreg[], GMortar1D *mortar)
{ 
  GINT     j, k, nbdy, nc;
  GDOUBLE  del=0.0, ifact;
  GVector  *xm[GDIM];
  Point    *bdy;

  if ( mortar == NULL ) {
    cout << "IConnAMR::RegularizeMortar: NULL mortar data" << endl;
    exit(1);
  }
  bdy = mortar->GetBdyPoints(nbdy);
  for ( k=0; k<GDIM; k++ ) {
    xm[k] = mortar->GetSpGrid(k+1);
    nc    = xm[k]->dim();
    ifact = 1.0 / ((GDOUBLE)nc-1.0);
    xreg[k]->Resize(nc);
    if ( nc > 1 ) del = ( bdy[1][k] - bdy[0][k] ) * ifact;
    for ( j=1; j<nc-1 && del>0.0; j++ ) {
      (*xreg[k])[j] = bdy[0][k] + j*del;
//    cout << "IConnAMR::Regularize: xreg[" << k << "][" << j << "]=" << (*xreg[k])[j] << " x_mort=" << (*xm[k])[j] << endl; 
    }
    for ( j=1; j<nc-1 && del<=0.0; j++ ) {
      (*xreg[k])[j] = (*xm[k])[j];
//    cout << "IConnAMR::Regularize: xreg[" << k << "][" << j << "]=" << (*xreg[k])[j] << " x_mort=" << (*xm[k])[j] << endl; 
    }
    (*xreg[k])   [0] = (*xm[k])   [0];
    (*xreg[k])[nc-1] = (*xm[k])[nc-1];
  }

} // end of method RegularizeMortar


//************************************************************************************
//************************************************************************************
// METHOD     : SetCycle
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void IConnAMR::SetCycle(GINT icycle)
{
  icycle_ = icycle;
} // end of SetCycle


