//************************************************************************************//
// Module       : morton_keyken.cpp
// Date         : 8/25/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                defining a Morton-type key-generator, as used in GASpAR.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "morton_keygen.hpp"
#include "gcutils.hpp"
#include <string>

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
Morton_KeyGen::Morton_KeyGen(MORTON_TYPE type)
:GKeyGen(),
itype_       (type),
btakelog_    (FALSE),
ksz_         (sizeof(GKEY)),
logFact_     (log10(1/TINY)),
delmax_      (1.0),
idelmax_     (1.0),
idel_        (1.0),
bb           (NULL)
{
  idX_.x1 = idX_.x2 = idX_.x3 = SFPTINY;
  bb = new BitBlock(BITSPERBYTE*sizeof(GKEY));
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
Morton_KeyGen::~Morton_KeyGen()
{
  if ( bb != NULL ) delete bb;
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetType
// DESCRIPTION: Morton-ordering method
// ARGUMENTS  : intype: type.
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::SetType(MORTON_TYPE intype)
{
  itype_ = intype;
} // end of method SetType


//************************************************************************************
//************************************************************************************
// METHOD     : SetOrigin
// DESCRIPTION: Set geometric origin for Morton-ordering
// ARGUMENTS  : inP0: Point3D type for coordinate origin (Cartesian)
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::SetOrigin(Point3D &inP0)
{
  P0_ = inP0;
} // end of method SetOrigin


//************************************************************************************
//************************************************************************************
// METHOD     : SetBox
// DESCRIPTION: Set geometric extent
// ARGUMENTS  : inP0: Point3D type for coordinate origin (Cartesian)
//              inP1: Point type for coordinate diagonal
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::SetBox(Point &inP0, Point &inP1)
{
  GINT i;

  P0_ = inP0;
  P1_ = inP1;
  for ( i=0,delmax_=0.0; i<GDIM; i++ ) {
    dX_    [i] =       fabs(P1_[i] - P0_[i] + TINY);
    idX_   [i] = 1.0 / dX_[i];
    delmax_    = MAX(delmax_,dX_[i]);
  }
  logFact_ = log10(1.0 + delmax_/TINY);    
  idelmax_ = 1.0/delmax_;
    
 
} // end of method SetBox


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoLog
// DESCRIPTION: Set flag indicating whether or not to take log of difference before
//              integralizing.
// ARGUMENTS  : bDoLog: TRUE or FALSE flag
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::SetDoLog(GBOOL bDoLog)
{
  btakelog_ = bDoLog;
} // end of method SetDoLog


//************************************************************************************
//************************************************************************************
// METHOD     : SetIntegralLen
// DESCRIPTION: Set integral length for Morton-ordering. This is the smallest
//              length in each Cartesian direction such that it integralizes by
//              unity a quantity P-P0, where P is an arbitrary point, and P0 is
//              the coord origin, set in SetOrigin.
// ARGUMENTS  : indX: Point3D type for integral length(s) (Cartesian)
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::SetIntegralLen(Point3D &indX)
{
  GINT    k;
  GDOUBLE dmin=SEHUGE;

  for ( k=0; k<GDIM; k++ ) {
    idX_[k] = 1.0/indX[k];
    dmin    = MIN(dmin,indX[k]);
  }
  idel_ = 1.0/dmin;

} // end of method SetIntegralLen


//************************************************************************************
//************************************************************************************
// METHOD     : key (1)
// DESCRIPTION: Computes Morton-ordered key
//              MORTON_TYPE is defined as follows:
//                          Let X = x7 x6 x5 x4 x3 x2 x1 x0  and
//                              Y = y7 y6 y5 y4 y3 y2 y1 y0 be the X,Y values of a point,
//              where x0, ... x7 represent the bits 0-7 of an 8-bit float, and same for Y.
//              Then MORTON_FULL_INTERLEAVE means that the computed key will be
//                  KEY = y7x7 y6x6 ... x2x2 y1x1 y0x0
//              while for MORTON_PARTIAL_INTERLEAVE the key is s.t.:
//                  KEY = y7y6y5...y2y1y0x7x6...x2x1x0.
//              with an obvious extension to 3D
//              Let X = x7 x6 x5 x4 x3 x2 x1 x0  and
//                              Y = y7 y6 y5 y4 y3 y2 y1 y0 be the X,Y values of a point,
//              where x0, ... x7 represent the bits 0-7 of an 8-bit float, and same for Y.
//              Then MORTON_FULL_INTERLEAVE means that the computed key will be
//                  KEY = y7x7 y6x6 ... x2x2 y1x1 y0x0
//              while for MORTON_PARTIAL_INTERLEAVE the key is s.t.:
//                  KEY = y7y6y5...y2y1y0x7x6...x2x1x0.
//              with an obvious extension to 3D
// ARGUMENTS  : 
//              id   : Array of length 'n' containing the resultant keys. This
//                     must be allocated by caller and be of size 'n'.
//              idsz : size of 'id' array elements in bytes
//              point: Array of length 'n' of Cartesian points for which to 
//                     generate a Morton-type key.
//              n    : Number of points for which to generate keys.
//
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::key(void *id, GINT idsz, Point3D point[], GINT  n)
{
  char      *serr = " Morton_KeyGen::key: ";
  GINT      i, j, k, ib, nbits, tbits;
  GLONG     ix[3]={0,0,0};
  GUSHORT   ui;
  GDOUBLE   del;

  nbits = BITSPERBYTE * idsz / GDIM;
  tbits = GDIM * nbits;
  logFact_ = fabs(pow(2.0,nbits-1) / log10(delmax_/TINY));
  if ( GDIM > 3 || GDIM < 1 ) {
    cout << "Morton_KeyGen::key (1): spatial dimension" << endl;
    exit(1);
  }
  if ( point == NULL || id == NULL ) {
    cout << "Morton_KeyGen::key (1): input data NULL" << endl;
    exit(1);
  }
  if ( idsz != ksz_ ) {
    if ( bb != NULL ) delete bb; bb = NULL;
    bb = new BitBlock(BITSPERBYTE*idsz);
    ksz_ = idsz;
  } 

#if 0
  cout << serr << "idel    =" << idel_ << endl;
  cout << serr << "delmax  =" << delmax_ << endl;
  cout << serr << "logfact =" << logFact_<< endl;
#endif
  // interleave integer bits into key:
  if ( itype_ == MORTON_FULL_INTERLEAVE ) { // full-interleaving
    for ( i=0; i<n; i++ ) {
      bb->Reset();
      for ( k=0; k<GDIM; k++ ) {
//      ix[k] = (GLONG)fabs((point[i][k]-P0_[k])*idX_[k]+0.5);
        del   = fabs(point[i][k]-P0_[k]+TINY);
        ix[GDIM-1-k] = btakelog_ ? (GLONG)(logFact_*log10(delmax_/del)) 
                                 : (GLONG)(             del *idel_+0.5);
//      cout << serr << " del=" << del << " ix[" << k << "]=" << ix[k] << endl;
      }
      for ( j=0,ib=0; j<nbits; j++ ) {
        for ( k=0; k<GDIM; k++,ib++ ) {
          ui    = ( (ix[k] >> ib) & ~(~0 << 1) );                     // shift & mask with 0000001
          bb->SetBits(ib, ui);
        }
      }
      bb->TransferBits(((GBYTE*)id+i*idsz), 1, idsz);
//    cout << serr << " index[" << i << "]=" << (GNODEID)(*((GBYTE*)id+i*idsz)) << endl;
#if defined(_GBYTESWAP)
      GCUtils::swap(((GBYTE*)id+i*idsz),idsz);
#endif
    } // end, point[] interleaving
  } // end, MORTON_FULL_INTERLEAVE
  else  {                                 // partial interleaving
    for ( i=0; i<n; i++ ) {
      bb->Reset();
      for ( k=0; k<GDIM; k++ ) {
//      ix[k] = (GLONG)fabs((point[i][k]-P0_[k])*idX_[k]+0.5);
        del   = fabs(point[i][k]-P0_[k]+TINY);
        ix[GDIM-1-k] = btakelog_ ? (GLONG)(logFact_*log10(delmax_/del)) 
                                 : (GLONG)(             del *idel_+0.5);
//      ix[k] = btakelog_ ? (GLONG)(logFact_*log10(delmax_/del)) 
//                        : (GLONG)(      del *idel_+0.5);
//      ix[k] = btakelog_ ? (GLONG)(log10(del)*idX_[k]+0.5) 
//                        : (GLONG)(      del *idX_[k]+0.5);
//      cout << serr << " del=" << del << " ix[" << k << "]=" << ix[k] << endl;
      }
      for ( k=0,ib=0; k<GDIM; k++ ) {
        for ( j=0; j<nbits; j++,ib++ ) {
          ui    = ( (ix[k] >> ib) & ~(~0 << 1) );                     // shift & mask with 0000001
          bb->SetBits(ib, ui);
        }
      }
      bb->TransferBits(((GBYTE*)id+i*idsz), 1, idsz);
//    cout << serr << " index[" << i << "]=" << (GNODEID)(*((GBYTE*)id+i*idsz)) << endl;
#if defined(_GBYTESWAP)
      GCUtils::swap(((GBYTE*)id+i*idsz),idsz);
#endif
    } // end, point[] interleaving
  } // end, MORTON_PARTIAL_INTERLEAVE

} // end of method key


//************************************************************************************
//************************************************************************************
// METHOD     : key (2)
// DESCRIPTION: Computes Morton-ordered key
//              MORTON_TYPE is defined as follows:
//                          Let X = x7 x6 x5 x4 x3 x2 x1 x0  and
//                              Y = y7 y6 y5 y4 y3 y2 y1 y0 be the X,Y values of a point,
//              where x0, ... x7 represent the bits 0-7 of an 8-bit float, and same for Y.
//              Then MORTON_FULL_INTERLEAVE means that the computed key will be
//                  KEY = y7x7 y6x6 ... x2x2 y1x1 y0x0
//              while for MORTON_PARTIAL_INTERLEAVE the key is s.t.:
//                  KEY = y7y6y5...y2y1y0x7x6...x2x1x0.
//              with an obvious extension to 3D
//              Let X = x7 x6 x5 x4 x3 x2 x1 x0  and
//                              Y = y7 y6 y5 y4 y3 y2 y1 y0 be the X,Y values of a point,
//              where x0, ... x7 represent the bits 0-7 of an 8-bit float, and same for Y.
//              Then MORTON_FULL_INTERLEAVE means that the computed key will be
//                  KEY = y7x7 y6x6 ... x2x2 y1x1 y0x0
//              while for MORTON_PARTIAL_INTERLEAVE the key is s.t.:
//                  KEY = y7y6y5...y2y1y0x7x6...x2x1x0.
//              with an obvious extension to 3D
// ARGUMENTS  : 
//              id   : Array of length 'n' containing the resultant keys. This
//                     must be allocated by caller and be of size 'n'.
//              idsz : size of 'id' array elements in bytes
//              x    : Array of length 'GDIM' of pointers to 'n' x, y, z coords
//              n    : Number of points for which to generate keys.
//
// RETURNS    : none.
//************************************************************************************
void Morton_KeyGen::key(void *id, GINT idsz, GDOUBLE *x[], GINT  n)
{
  GINT      j, k;
  Point     p;

  for ( j=0; j<n; j++ ) {
    for ( k=0; k<GDIM; k++ ) p[k] = x[k][j];
    key(((GBYTE*)id+j*idsz), idsz, &p, 1);
  }

} // end of method key


