//************************************************************************************//
// Module       : ntree_adapt.cpp
// Date         : 9/25/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                defining an N-tree-based adaption scheme.
// Derived From : IConnAMR
// Modifications:
//************************************************************************************//
#include "ntree_adapt.hpp"
#include "gtbuffer.hpp"
#include "gpmsibpart.hpp"
#include <string>

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
NTreeAdapt::NTreeAdapt(GINT  maxlev)
:IConnAMR(),
nprocs_          (GComm::WorldSize()),
this_rank_       (GComm::WorldRank()),
bDoLoadBalance_  (FALSE),
bPartitioning_   (FALSE),
bLimitByLevel_   (FALSE),
ndcomp_          ((GINT )pow(2.0,GDIM)),
nlevel_limit_    (0),
nfields_         (0),
dcomp_dir_       (0),
max_levels_      (maxlev),
npart_           (0),
max_key_mult_    (0),
indcomp_         (1.0/pow(2.0,GDIM)),
refq_type_       (0),
gfields_         (NULL),
partitioner_     (NULL),
partmap_         (NULL),
pmglobal_        (NULL),
pmsibling_       (NULL)
{
  AGINT          *blk_ptr ;
  GINT           *n_types ;
  GC_DATATYPE    *sz_types;
  stRefineQuery  tstruct;

  blk_ptr  = new AGINT       [NRQUERYST_MEM];
  n_types  = new GINT        [NRQUERYST_MEM];
  sz_types = new GC_DATATYPE [NRQUERYST_MEM];
  // Register the stRefineQuery data structure type for communication:
  n_types [0] = n_types[1] = n_types[2] = n_types[3] = 1;
  sz_types[0] = GC_GKEY;
  sz_types[1] = GC_GKEY;
  sz_types[2] = GC_GKEY;
  sz_types[3] = GC_GINT;

  GComm::Address(&(tstruct.elem_key)      , &blk_ptr[0]);
  GComm::Address(&(tstruct.parent_key)    , &blk_ptr[1]);
  GComm::Address(&(tstruct.root_key)      , &blk_ptr[2]);
  GComm::Address(&(tstruct.iflag)         , &blk_ptr[3]);
  if ( !GComm::DataTypeFromStruct(blk_ptr, sz_types, n_types, NRQUERYST_MEM, &refq_type_) ) {
    cout << "NTreeAdapt::NTreeAdapt (1): Comm refine query datatype registration failed" << endl;
    exit(1);
  }

  delete [] blk_ptr; delete [] n_types; delete [] sz_types;

  max_key_mult_ = (GKEY)(pow((GDOUBLE)ndcomp_,max_levels_));
  if ( max_key_mult_ <= 0 ) {
    cout << "NTreeAdapt::NTreeAdapt (2): illegal key multiplier: max_key_mult_=" << max_key_mult_ 
         << " ndcomp_=" << ndcomp_ << " max_levels=" << max_levels_ << endl;
    exit(1);
  }
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
NTreeAdapt::NTreeAdapt(GElemList *in_gelems, GFieldList *in_fields[], GINT  nfields, GINT  maxlev)
:IConnAMR(in_gelems),
nprocs_          (GComm::WorldSize()),
this_rank_       (GComm::WorldRank()),
bDoLoadBalance_  (FALSE),
bPartitioning_   (FALSE),
bLimitByLevel_   (FALSE),
ndcomp_          ((GINT )pow(2.0,GDIM)),
nlevel_limit_    (0),
nfields_         (nfields),
dcomp_dir_       (0),
max_levels_      (maxlev),
npart_           (0),
indcomp_         (1.0/pow(2.0,GDIM)),
refq_type_       (0),
gfields_         (NULL),
partitioner_     (NULL),
partmap_         (NULL),
pmglobal_        (NULL),
pmsibling_       (NULL)
{
  GINT           j;
  AGINT          *blk_ptr ;
  GINT           *n_types ;
  GC_DATATYPE    *sz_types;
  stRefineQuery  tstruct;

  if ( in_gelems == NULL || in_fields == NULL || nfields <= 0 ) {
    cout << "NTreeAdapt::NTreeAdapt (2): NULL fields not allowed on construction" << endl;
    exit(1);
  }

  // Add primary fields to fieldgroup structure:
  fieldgroups_.add(); 
  fieldgroups_.member()->nfields = nfields;
  fieldgroups_.member()->pelems  = in_gelems;
  fieldgroups_.member()->pfields = new GFieldList * [nfields];
  gfields_ = new GFieldList * [nfields];

  // Make copy of primary field group for use when only this
  // is required:
  for ( j=0; j<nfields; j++ ) {
    gfields_[j] = in_fields[j]; 
    fieldgroups_.member()->pfields[j] = in_fields[j]; 
  }

  blk_ptr  = new AGINT       [NRQUERYST_MEM];
  n_types  = new GINT        [NRQUERYST_MEM];
  sz_types = new GC_DATATYPE [NRQUERYST_MEM];

  // Register the stRefineQuery data structure type for communication:
  n_types [0] = n_types[1] = n_types[2] = 1;
  sz_types[0] = GC_GKEY;
  sz_types[1] = GC_GKEY;
  sz_types[2] = GC_GINT ;

  GComm::Address(&(tstruct.elem_key)      , &blk_ptr[0]);
  GComm::Address(&(tstruct.parent_key)    , &blk_ptr[1]);
  GComm::Address(&(tstruct.iflag)         , &blk_ptr[2]);
  if ( !GComm::DataTypeFromStruct(blk_ptr, sz_types, n_types, NRQUERYST_MEM, &refq_type_) ) {
    cout << "NTreeAdapt::NTreeAdapt (2): Comm query datatype registration failed" << endl;
    exit(1);
  }

  max_key_mult_ = (GKEY)(pow((GDOUBLE)ndcomp_,max_levels_));
  if ( max_key_mult_ <= 0 ) {
    cout << "NTreeAdapt::NTreeAdapt (1): illegal key multiplier: max_key_mult_=" << max_key_mult_ 
         << " ndcomp_=" << ndcomp_ << " max_levels=" << max_levels_ << endl;
    exit(1);
  }

  pmsibling_   = new GPMSibPart(fieldgroups_[0]->pelems);

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
NTreeAdapt::~NTreeAdapt()
{
  if ( gfields_   ) delete [] gfields_;
  if ( partmap_   ) delete [] partmap_;
  if ( pmsibling_ ) delete pmsibling_;
  GComm::DataTypeFree(&refq_type_);

}


//************************************************************************************
//************************************************************************************
// METHOD     : SetFieldGroup
// DESCRIPTION: Sets fields on secondary grid. Each time this method is called, 
//              the new field/element group is added to a list of existing
//              groups. User must make sure that the specified element and field
//              lists are unique; no checking is done here. If this method is used
//              with constructor (1), then the first specified field group _must_
//              be the primary group, with subsequent calls the secondary groups.
// ARGUMENTS  : elems     : pointer to GElemList
//              in_fields : array of pointers to GFieldList objects, of length nfields
//              fnields   : length of 'fields' pointer array
//                           
// RETURNS    : none
//************************************************************************************
void NTreeAdapt::SetFieldGroup(GElemList *in_elems, GFieldList *in_fields[], GINT  nfields)
{   
  GINT      j;

  fieldgroups_.add(); 
  fieldgroups_.member()->nfields = nfields;
  fieldgroups_.member()->pelems  = in_elems;
  fieldgroups_.member()->pfields = new GFieldList * [nfields];
  for ( j=0; j<nfields; j++ ) fieldgroups_.member()->pfields[j] = in_fields[j];       

  if ( fieldgroups_.size() == 1 ) {  // have set primary grid's field group
    if ( pmsibling_   != NULL ) delete pmsibling_;
    gelems_      = fieldgroups_.member()->pelems;
    pmsibling_   = new GPMSibPart(gelems_);

    // Make copy of primary field group for use when only this
    // is required or easier to use than fieldgroup structure:
    if ( gfields_ ) delete [] gfields_;
    gfields_ = new GFieldList * [nfields];

    for ( j=0; j<nfields; j++ ) {
      gfields_[j] = in_fields[j]; 
    }
  }

} // end of method SetFieldGroup
  


//************************************************************************************
//************************************************************************************
// METHOD     : Adapt
// DESCRIPTION: Public interface to adaption methods. Performs consistent grid adaption.
//              Connectivity is reestablished between refined/coarsened elements. 
//              ElemList and FieldList structures will be modified in this call.
// ARGUMENTS  : hrefine: buffer containing current local element (field) indices 
//                       that are tagged for refining
//              coarsen: buffer containing current local element (field) indices 
//                       that are tagged for coarsening
//                         
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::Adapt(GIBuffer  &hrefine, GIBuffer  &coarsen)
{
  char    *serr = "NTreeAdapt::Adapt: ";
  GINT    i, j, n[2], gn[2];
  GBOOL   bOk=TRUE;

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "Calling AdaptAll..." <<  endl;
#endif
GComm::Synch();
  if ( !(bOk=AdaptAll(hrefine,coarsen)) ) {
    cout << serr << "elemental refinement/coarsening failed: NB" << endl;
    return FALSE;
  } 

  if ( gelems_->size() != gfields_[0]->size() ) {
    cout << serr << "inconsistent lists after adaption" << endl;
    return FALSE;
  }

  n[0] = hrefine.dim();
  n[1] = coarsen.dim();
  GComm::Allreduce(n, gn, 2, GC_GINT, G_OP_SUM);
  if ( gn[0] == 0 && gn[1] == 0 ) return TRUE;  // nothing to re-init

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "Calling Init..." <<  endl;
#endif

  // Finally, re-initialize the GS operator:
  // NOTE: It's BAD to call Init if we don't have to!
  if ( !(bOk=Init()) ) { //(gn[0] != 0 || gn[1] != 0))) ) {
    cout << serr << "re-initialization of GS operator failed" << endl;
    return bOk;
//  exit(1);
  } 

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "Calling CheckPeriodic..." <<  endl;
#endif
  if ( !CheckPeriodic(i,j) ) {
    cout << "###################################################################" << endl;
    cout << serr << "(2): Edge[" << i << "][" << j << "] should have at least one neighbor!!" << endl;
    cout << "###################################################################" << endl;
    return FALSE; //exit(1);
  }

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "done." <<  endl;
#endif
  return bOk;

} // end of method Adapt

//************************************************************************************
//************************************************************************************
// METHOD     : AdaptAll
// DESCRIPTION: Performs grid adaption for one level of refinement. ElemList and 
//              FieldList structures will be modified in this call. Numbering of
//              elements and fields in their lists will be consecutive on exit.
//              All fields are adapted as required, and load balancing is
//              accounted for here.
// ARGUMENTS  : hrefine : buffer containing current local element (field) indices 
//                        that are tagged for refining
//              coarsen : buffer containing current local element (field) indices 
//                        that are tagged for coarsening
//                         
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::AdaptAll(GIBuffer  &hrefine, GIBuffer  &coarsen)
{
  char       *serr = "NTreeAdapt:AdaptAll: ";
  GIBuffer   icoarsen;
  GKEYBuffer kcoarsen;

  if ( !gelems_ || !gfields_ || !bInitialized_ ) {
    cout << serr << "improperly initialized"  << endl;
    return FALSE;
  }

  // Check lists for consistency:
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "before refine consistency check: hrefine=" << hrefine << endl;
#endif
  CheckRefinement(hrefine, coarsen);
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "after refine consistency check: hrefine=" << hrefine << endl;
#endif
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "before coarsen consistency check:coarsen=" << coarsen << endl;
#endif
  CheckCoarsening(hrefine, coarsen);

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "after coarsen consistency check: coarsen=" << coarsen << endl;
#endif

  // Element indices will change in refinement, so need to convert coarsen 
  // indices to element keys for later identification in partitioner:
  ElemIndexToKey(kcoarsen, coarsen);  

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "Doing refinement..." << endl;
#endif
  // Refinement is inherently local, so just do it:
  if ( !DoRefinement(hrefine) ) {
    cout << serr << "refinement failed"  << endl;
    return FALSE;
  }
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "Refinement done." << endl;
#endif

  // Coarsening may not be local, so must treat it as global.
  // If we load balance, then we send siblings to proc where parent
  // will live, so that we can preclude necessity of extra communication.
  // If we don't load balance, then we must make sure that siblings
  // get to proc where parent will live, and this is determined simply
  // by minimizing the communication required to move siblings--a possibly
  // different process for determining the partition map than for
  // global load balancing. At the end of this step, all siblings to be coarsened
  // must reside on the same proc, and those on this proc are provided by the 
  // index buffer icoarsen:
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " Re-partitioning..." << endl;
#endif
  if ( bPartitioning_ ) {
    if ( bDoLoadBalance_ ) {
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " ...Doing load balance:" << endl;
#endif
      if ( !DoPartition(icoarsen,kcoarsen,pmglobal_) ) {
        cout << serr << "load balancing failed"  << endl;
        return FALSE;
      }
      bDoLoadBalance_ = FALSE;   // don't attempt this at every adapt step
    }
    else {
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " ...consolidating siblings:" << endl;
#endif
      if ( !DoPartition(icoarsen,kcoarsen,pmsibling_) ) {
        cout << serr << "sibling collection failed"  << endl;
        return FALSE;
      }
    }
  }
  else {
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " Before Restrict2OneSibMem: coarsen=" << coarsen  << endl;
#endif
    Restrict2OneSibMem(icoarsen,kcoarsen);   
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " After Restrict2OneSibMem: icoarsen=" << icoarsen  << endl;
#endif
  }
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " Doing coarsening..." << endl;
#endif

//if ( hrefine.dim() == 0 && coarsen.dim() == 0 ) return TRUE;  // nothing to do

  // By this point, all siblings should reside on the processor
  // where their parent will live, and we can carry out coarsening
  // by using the element keys:
  if ( !DoCoarsening(icoarsen) ) {
    cout << serr << "coarsening failed"  << endl;
    return FALSE;
  }
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << " Coarsening done." << endl;
#endif

  return TRUE;

} // end of method AdaptAll


//************************************************************************************
//************************************************************************************
// METHOD     : DoRefinement
// DESCRIPTION: Performs grid adaption for one level of refinement. ElemList and 
//              FieldList structures will be modified in this call. Numbering of
//              elements and fields in their lists will be consecutive an exit.
// ARGUMENTS  : irefine : buffer containing current local element (field) indices 
//                        that are tagged for refining
//                         
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::DoRefinement(GIBuffer  &irefine) 
{
  char       *serr = "NTreeAdapt:DoRefinement: ";
  GINT       i, j, k, m, nfields;
  GBOOL      bOk=TRUE;
  Elem2D     **pelem_refine =NULL, *pnew=NULL;
  Field2D    ***pfield_refine=NULL;
  GElemList  *gelems;
  GFieldList **gfields;


//if ( irefine.dim() <= 0 ) return TRUE;

  for ( m=0,bOk=TRUE; m<fieldgroups_.size() && bOk; m++ ) {
    nfields = fieldgroups_[m]->nfields;
    gelems  = fieldgroups_[m]->pelems ;
    gfields = fieldgroups_[m]->pfields;
    pelem_refine  = new Elem2D  *  [irefine.dim()];
    pfield_refine = new Field2D ** [irefine.dim()];
    for ( i=0; i<irefine.dim(); i++ )
      pfield_refine[i] = new Field2D * [nfields];
    // Do refinement. Remember, the irefine buffers
    // refer to the element lists as they were on entry; they 
    // will be modified as a result of the refinement. 
    // For this reason, we get a list of pointers to the elements
    // that the refine lists refer to, rather than rely
    // on the element list indices themselves:
    // [NOTE: better to use buffers with Resize method, so that check
    //        is done to reallocate only if needed.]
    for ( i=0; i<irefine.dim(); i++ ) {
      pelem_refine [i] = (*gelems )[irefine(i)];
      for ( k=0; k<nfields; k++ )
        pfield_refine[i][k] = (*gfields[k])[irefine(i)];
    }
  
    for ( i=0, bOk=TRUE; i<irefine.dim() && bOk; i++ ) { 
       bOk = RefineElemental(pelem_refine[i], pfield_refine[i], gelems, gfields, nfields);
    }
    if ( !bOk ) {
      cout << serr << "refinement step failed." << endl;  // on elem[" << irefine(i) << "]" << endl;
      return FALSE;
    }

    // Now, delete the refined parent elements, and the
    // coarsened siblings:
    for ( i=0; i<irefine.dim(); i++ ) {        // delete refined parents
      for ( k=0; k<nfields; k++ ) {            
        if ( pelem_refine[i] ) gfields[k]->del(pelem_refine[i]);  
      }
#if defined(NT_DEBUG_OUTPUT)
      cout << serr << "deleting pelem=" << pelem_refine[i] << endl;
#endif
      if ( pelem_refine[i] ) gelems->del(pelem_refine [i]);       
    }

    // Renumber element and field lists so that ramdon-access
    // calls can be made into them:
    gelems ->renumber();
    for ( j=0; j<nfields; j++ ) gfields[j]->renumber();
  
    // Do cursory check that field and element lists are the same size:
    for ( j=0; j<nfields; j++ ) bOk = bOk && gelems->size() == gfields[j]->size();
    if ( !bOk ) {
      cout << serr << "incompatible list sizes." <<  endl;
    }
    if ( pelem_refine   ) delete [] pelem_refine ; pelem_refine = NULL;
    for ( i=0; i<irefine.dim(); i++ ) {
      if ( pfield_refine  ) delete [] pfield_refine[i];
    }
    if ( pfield_refine  ) delete [] pfield_refine ; pfield_refine = NULL;
  
  } // end of fieldgroup loop 
  // Free up temp memory:

#if defined(NT_DEBUG_OUTPUT)
  if ( bOk ) 
    cout << serr << "done." <<  endl;
  else 
    cout << serr << "failed." << endl;

#endif

  return bOk;
} // end of method DoRefinement


//************************************************************************************
//************************************************************************************
// METHOD     : DoCoarsening
// DESCRIPTION: Performs grid adaption for one level of coarsening. ElemList and 
//              FieldList structures will be modified in this call. Numbering of
//              elements and fields in their lists will be consecutive an exit.
// ARGUMENTS  : kcoarsen: buffer containing keys of local elements to be refined.
//                        NOTE: method assumes that all siblings are local on entry.
//                         
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::DoCoarsening(GIBuffer  &coarsen)
{
  char       *serr = "NTreeAdapt:DoCoarsening: ";
  GINT       i, j, k, m, n, nfields, nsiblings;
  GBOOL      bOk=TRUE;
  Elem2D     **pelem_coarsen =NULL, **esiblings=NULL,
             **allsibs=NULL, *pnew=NULL;
  Field2D    ***pfield_coarsen=NULL;
  GElemList  *gelems;
  GFieldList **gfields;

#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "coarsen list:" << coarsen << endl;
#endif
  for ( m=0,bOk=TRUE; m<fieldgroups_.size() && bOk; m++ ) {
    nfields = fieldgroups_[m]->nfields;
    gelems  = fieldgroups_[m]->pelems ;
    gfields = fieldgroups_[m]->pfields;
    // Do coarsening. Element keys are assumed to be the same
    // for all fields on that (primary) element.
  
    if ( coarsen.dim() > 0 ) {
      pelem_coarsen  = new Elem2D  *  [coarsen.dim()];
      pfield_coarsen = new Field2D ** [coarsen.dim()];
      for ( j=0; j<coarsen.dim(); j++ )
        pfield_coarsen[j] = new Field2D * [nfields];
      for ( i=0; i<coarsen.dim(); i++ ) {
        pelem_coarsen [i] = (*gelems )[coarsen(i)];
        for ( j=0; j<nfields; j++ )
          pfield_coarsen[i][j] = (*gfields[j])[coarsen(i)];
      }
    }

    // Assume the coarsen check methods provide the
    // coarsen indices for each *group* of siblings, that is
    // at most, only one sibling from each parent--because
    // the remaining siblings will be gathered here:
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "after EPointersToIndices: coarsen list:" << coarsen << endl;
#endif

    if ( coarsen.dim() > 0 && bOk ) {
      allsibs = new Elem2D  * [coarsen.dim()*ndcomp_];
      for ( j=0; j<coarsen.dim()*ndcomp_; j++ ) allsibs[j] = NULL;
      for ( i=0, bOk=TRUE; i<coarsen.dim() && bOk; i++ ) {
#if defined(NT_DEBUG_OUTPUT)
        cout << serr << "Getting siblings for element[" << coarsen[i] << "]: " << nsiblings << endl;
#endif
        GetSiblings(pelem_coarsen[i], esiblings, nsiblings, gelems);
        for ( j=0; j<MIN(nsiblings,ndcomp_); j++ ) {
          allsibs[j+i*ndcomp_] = esiblings[j];
        }
#if defined(NT_DEBUG_OUTPUT)
        cout << serr << "number siblings for element[" << coarsen[i] << "]: " << nsiblings << endl;
#endif

        if ( nsiblings == ndcomp_ ) {
          bOk = CoarsenElemental(esiblings, nsiblings, pnew, gelems, gfields, nfields);
#if defined(NT_DEBUG_OUTPUT)
          cout << serr << "CoarsenElemental[" << coarsen(i) << "]: " 
           << bOk << endl;
#endif
        }
        else {
        cout << serr << "in coarsening element " << coarsen(i) << " insufficient number of siblings!" << endl;
          for ( j=0; j<ndcomp_; j++ ) allsibs[j+i*ndcomp_] = NULL;
        }
      }
  
      if ( !bOk ) {
        cout << serr << "coarsen step failed on elem[" << coarsen(i) << "]" << endl;
      }
    }
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "before delete: num elements=" << gelems_->size() << endl;
#endif


    for ( k=0; k<nfields; k++ ) {
      for ( j=0; j<coarsen.dim()*ndcomp_; j++ ) {      // delete coarsened sibling fields
        if ( allsibs[j] ) gfields[k]->del(allsibs[j]);
      }
    }
    for ( j=0; j<coarsen.dim()*ndcomp_; j++ ) {        // delete coarsened sibling elems
      if ( allsibs[j] ) gelems->del(allsibs[j]);
    }

    // Renumber the lists, so that array-like random-access 
    // calls can be made into them:
    gelems ->renumber();
    for ( j=0; j<nfields; j++ ) gfields[j]->renumber();
  
    if ( bOk ) {
      for ( j=0; j<nfields; j++ ) bOk = bOk && gelems->size() == gfields[j]->size();
      if ( !bOk ) {
        cout << serr << "incompatible list sizes." <<  endl;
      }
    }

    // Free up temp memory:
    if ( pelem_coarsen  ) delete [] pelem_coarsen; pelem_coarsen= NULL;
    if ( esiblings      ) delete [] esiblings    ; esiblings    = NULL; 
    if ( allsibs        ) delete [] allsibs      ; allsibs      = NULL; 
    for ( i=0; i<coarsen.dim(); i++ ) {
      if ( pfield_coarsen ) delete [] pfield_coarsen[i];
    }
    if ( pfield_coarsen ) delete [] pfield_coarsen; pfield_coarsen= NULL;
  } // end of fieldgroup loop
#if defined(NT_DEBUG_OUTPUT)
  if ( bOk ) 
    cout << serr << "done." <<  endl;
  else 
    cout << serr << "failed." << endl;

#endif

  return bOk;
} // end of method DoCoarsening



//************************************************************************************
//************************************************************************************
// METHOD     : DoPartition
// DESCRIPTION: Carries out partitioning, as specified by mapper object, pm. 
// ARGUMENTS  : icoarsen: output buffer containing indices of siblings to be coarsened, 
//                        in updated element list, created here, and returned. Only
//                        one sibling in a group is to be represented in this list.
//              kcoarsen: input buffer containing keys of elements in current local element
//                        list. This will not be valid, in general, after call; i.e.,
//                        elements represented by keys may not be in local list, since
//                        they may be migrated off proc after this call. 
//              pm      : partition mapper object, specified by caller. Must be non-NULL.
//  
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::DoPartition(GIBuffer  &icoarsen, GKEYBuffer  &kcoarsen, GPartMapper *pm)
{
  char       *serr = "NTreeAdapt::DoPartition: " ;
  GINT       ind, j, m, n, nfields, nfound, ng;
  GIBuffer   ictmp;
  GKEYBuffer kptmp;
  GElemList  *gelems;
  GFieldList **gfields;
 
  if ( partitioner_ == NULL || pm == NULL ) {
    cout << serr << "NULL parititoner/mapper: partitioner=" << partitioner_ << " mapper=" << pm << endl;
    exit(1);
  } 

  pm->SetCoarsenables(&kcoarsen);
  if ( !pm->GenerateMap(partmap_,npart_) ) {
    cout << serr << "partition map generation failed" << endl;
    return FALSE;
  }
#if 0
  for ( j=0; j<nprocs_*npart_; j++ ) {
    cout << serr << " partmap[" << j << "]=" << partmap_[j] << endl;
  }
#endif
  if ( !partitioner_->Init(partmap_,npart_) ) {
    cout << serr << "partitioner initialization failed" << endl;
    return FALSE;
  }
  if ( !partitioner_->DoPartition() ) {
    cout << serr << "partition failed" << endl;
    return FALSE;
  }

  // Re-index fields, elements:
  for ( ng=0; ng<fieldgroups_.size(); ng++ ) {
    nfields = fieldgroups_[ng]->nfields;
    gelems  = fieldgroups_[ng]->pelems;
    gfields = fieldgroups_[ng]->pfields;
    gelems ->renumber();
    for ( j=0; j<nfields; j++ ) gfields[j]->renumber();
  }

  // Determine the keys of local elements expected to be coarsened:
  for ( j=0,n=0; j<nprocs_*npart_; j++ ) {
    n += partmap_[j].bcoarsened && partmap_[j].proc_to==this_rank_ ? 1 : 0;
  }
  kcoarsen.Resize(n);
  kptmp   .Resize(n);
  ictmp   .Resize(n);
  for ( j=0, n=0; j<nprocs_*npart_; j++ ) {
    if ( partmap_[j].bcoarsened && partmap_[j].proc_to==this_rank_ ) {
      kcoarsen[n]=partmap_[j].elem_key;
      kptmp   [n]=partmap_[j].parent_key;
      n++;
    }
  }
 
  // Convert the element keys to indices into (re-indexed) element list: 
  nfound = ElemKeyToIndex(ictmp, kcoarsen);
  if ( nfound != kcoarsen.dim() ) {
    cout << serr << "expected siblings not found after partition: expcted " 
         <<  kcoarsen.dim() << " found " << nfound << endl;
    return FALSE;
  }

  // Find unique parent keys of sibs to be coarsened:
  kcoarsen = -1;
  for ( j=0,n=0; j<kptmp.dim(); j++ ) {
    if ( !kcoarsen.contains(kptmp[j],ind) ) kcoarsen[n++] = kptmp[j];
  }

  // With parent keys, find corresp. indices into gelems list.
  // Only one sibling index should be represented in the index list:
  icoarsen.Resize(n);
  for ( j=0,m=0; j<n; j++ ) {
    if ( kptmp.contains(kcoarsen[j],ind) ) icoarsen[m++] = ictmp[ind];
  }

  return TRUE;
  
} // end of method DoPartition



//************************************************************************************
//************************************************************************************
// METHOD     : SetPartitioner
// DESCRIPTION: Sets data partitioner object
// ARGUMENTS  : partitioner : grid partitioner of type GPartitioner *
//  
// RETURNS    : none.
//************************************************************************************
void NTreeAdapt::SetPartitioner(GPartitioner *partitioner)
{   
  char *serr = "NTreeAdapt::SetPartitioner: " ;

  if ( partitioner== NULL ) {
    cout << serr << "NULL parititoner" << endl;
    exit(1);
  }

  partitioner_ = partitioner;

} // end of method SetPartitioner



//************************************************************************************
//************************************************************************************
// METHOD     : SetLoadBalanceMapper
// DESCRIPTION: Sets load balance mapper object
// ARGUMENTS  : pm : load balance eapper of derived type GPartMapper *
//  
// RETURNS    : none.
//************************************************************************************
void NTreeAdapt::SetLoadBalanceMapper(GPartMapper *pm)
{
  char *serr = "NTreeAdapt::SetLoadBalanceMapper: " ;

  if ( pm == NULL ) {
    cout << serr << "NULL parititon mapper" << endl;
    exit(1);
  }

  pmglobal_ = pm;


} // end of method SetLoadBalanceMapper



//************************************************************************************
//************************************************************************************
// METHOD     : CheckConsistency
// DESCRIPTION: Makes input lists consistent with refinement criteria. Input
//              buffers will be modified as a result of this call.
// ARGUMENTS  : hrefine: buffer containing current local element (field) indices
//                       that are tagged for refining
//              coarsen: buffer containing current local element (field) indices
//                       that are tagged for coarsening
//
// RETURNS    : void
//************************************************************************************
void NTreeAdapt::CheckConsistency(GIBuffer  &hrefine, GIBuffer  &coarsen)
{
  GINT   nr, nc, nr1, nc1;
  GBOOL  bChange=TRUE;

  nr = hrefine.dim(); nc = coarsen.dim();
  while ( bChange ) {
    nr1 = nr; nc1 = nc;
    CheckRefinement(hrefine, coarsen);
    CheckCoarsening(hrefine, coarsen);
    nr = hrefine.dim(); nc = coarsen.dim();
    bChange = ( nr != nr1 || nc != nc1 );
  }

} // end of method CheckConsistency


//************************************************************************************
//************************************************************************************
// METHOD     : CheckRefinement
// DESCRIPTION: Performs check of refinement tags, primarily, but coarsen
//              list may be altered.
// ARGUMENTS  : hrefine: buffer containing current local element (field) indices
//                       that are tagged for refining
//              coarsen: buffer containing current local element (field) indices
//                       that are tagged for coarsening
//
// RETURNS    : void
//************************************************************************************
void NTreeAdapt::CheckRefinement(GIBuffer  &hrefine, GIBuffer  &coarsen)
{
  char           *serr = "NTreeAdapt::CheckRefinement: ";
  GINT           i, j, k, l, m, nk, gnk, n0, nr, nrl, ncl, gdim, ind;
  GINT           elev, nlev, *idist=NULL;
  GBOOL          bRefine, bContinue, bConf;
//GKEYBuffer     knew, gknew;
  GIBuffer       hnew, iadd;
  GKEY           kelem, ekey;
  Elem2D         *elem;
  GMortar1D      *mortar;
  GNeighborList  *eneighbor;
  stRefineQuery  *gcquery=NULL, *grquery=NULL;
#if defined(NT_DEBUG_OUTPUT)
   GINT           nloop=0;
#endif

#if defined(NT1_DEBUG_OUTPUT)
   cout << "<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<" << endl;
#endif
  if ( !bInitialized_ ) {
    cout << serr << "object not initialized" << endl;
    exit(1);
  }
  if ( !bGlobalDomain_ || globDomain_ == NULL ) {
    cout << serr << "global domain not specified" << endl;
    exit(1);
  }

#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << " On input: hrefine=" << hrefine << endl;
#endif

  if ( bLimitByLevel_ ) {
    // If we limit refinement but allow the calculation to 
    // proceed, then remove those elements from list that
    // would exceed level limit:
    hnew .Resize(hrefine.dim()); 
    for ( j=0, nr=0; j<hrefine.dim(); j++ ) {
      elem      = (*gelems_)[hrefine[j]];
      if ( RefineLevel(elem) <  nlevel_limit_ ) hnew[nr++] = hrefine[j]; 
    }
    hrefine.Resize(nr);
    for ( j=0; j<nr; j++ ) hrefine   [j] = hnew[j];
  }

  n0 = hrefine.dim();

  // For refinement, a given element must be tagged if it has a
  // neighbor that is already refined, and is also
  // itself tagged to be refined again:
  iadd .Resize(gelems_->size());
  bContinue = TRUE;
  while ( bContinue ) {  // loop until there are no more refinements added...
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "........................................Reduction loop count = " << nloop << endl;
#endif
    // First, get distinct (local) ids:
    hrefine.distinct(idist, nr);
    hnew   .Resize(nr); 
    for ( j=0; j<nr; j++ ) hnew(j) = hrefine(idist[j]);
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "Doing BuildGQueryLists..." << endl;
#endif
    // Build refinement query list:
    BuildGQueryList(grquery, nrl, hnew   , NULL, 0);
#if defined(NEW_CONDITION)
    BuildGQueryList(gcquery, ncl, coarsen, NULL, 0);
#endif
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "BuildGQueryLists done." << endl;
#endif
    iadd = -1;
    for ( i=0, nk=0; i<gelems_->size(); i++ ) {
      if ( hnew.contains(i,ind) ) continue;
      elem      = (*gelems_)[i];
      eneighbor = elem->GetENeighbor();
      mortar    = elem->GetEdgeMortar();
      for ( j=0, bRefine=FALSE; j<elem->GetNumEdges() && !bRefine; j++ ) {
        bConf = mortar[j].isConforming() ;
//      if ( !bConf ) continue;
#if defined(NEW_CONDITION)
        // If neighbor of element being checked is tagged to be coarsened, add 
        // neighbor to refinement list. Coarsen check should then remove neighbor
        // element tagged for coarsening from coarsen list:
        if ( eneighbor[j].size() == 1 || !mortar[j].isConforming() ) {
          ekey = eneighbor[j][0].elemkey();
          bRefine =  bRefine || ( 
                     gelems_->member(ekey) != NULL && InQueryList(gcquery,ncl,ekey,NULL,0) 
                                                   && InQueryList(grquery,nrl,elem->GetID(),NULL,0)
                      );
        }  
        else {
#endif
          // Determine if neighbor is to be refined again;
          // if neighbor not being refined, nothing to consider:
          elev = RefineLevel(elem);
          for ( m=0; m<eneighbor[j].size() && eneighbor[j].size() > 1; m++ ) {
            ekey  = eneighbor[j][m].elemkey();
            nlev  = RefineLevel(eneighbor[j][m].elemrootkey(), ekey);
            bRefine = bRefine || ( InQueryList(grquery,nrl,ekey,NULL,0)
                              &&   elev < nlev ); 
          }
#if defined(NEW_CONDITION)
        }
#endif
      }
      if ( bRefine && !hnew.contains(i,ind) ) iadd(nk++) = i;
    } // end, element loop
    hrefine.Resize(nr+nk);
    for ( j=0; j<nr; j++ ) hrefine   [j] = hnew[j];
    for ( j=0; j<nk; j++ ) hrefine[j+nr] = iadd[j];
//  cout << "NTreeAdapt::CheckRefinement: hrefine=" << hrefine << endl;
    GComm::Allreduce(&nk, &gnk, 1, GC_GINT , G_OP_MAX);
    bContinue = gnk > 0 ;
    if ( grquery ) delete [] grquery; grquery = NULL;
    if ( gcquery ) delete [] gcquery; gcquery = NULL;
#if defined(NT_DEBUG_OUTPUT)
    nloop++;
#endif

  } // end of consistency loop
//cout << "NTreeAdapt::CheckRefinement: hrefine=" << hrefine << endl;
#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "Number elements added to refine list: =" << hrefine.dim()-n0 << endl;
#endif

  if ( bLimitByLevel_ ) {
    // If we limit refinement but allow the calculation to 
    // proceed, then remove those elements from list that
    // would exceed level limit:
    hnew .Resize(hrefine.dim()); 
    for ( j=0, nr=0; j<hrefine.dim(); j++ ) {
      elem      = (*gelems_)[hrefine[j]];
      if ( RefineLevel(elem) <  nlevel_limit_ ) hnew[nr++] = hrefine[j]; 
    }
    hrefine.Resize(nr);
    for ( j=0; j<nr; j++ ) hrefine   [j] = hnew[j];
  }

  if ( idist ) delete [] idist; idist = NULL;
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "done.: hrefine=" << hrefine << endl;
#endif
} // end of method CheckRefinement


//************************************************************************************
//************************************************************************************
// METHOD     : CheckCoarsening
// DESCRIPTION: Performs check of coarsening tags, primarily. Refine tags may be
//              checked, but not altered.
// ARGUMENTS  : hrefine: buffer containing current local element (field) indices
//                       that are tagged for refining
//              coarsen: buffer containing current local element (field) indices
//                       that are tagged for coarsening
//
// RETURNS    : void
//************************************************************************************
void NTreeAdapt::CheckCoarsening(GIBuffer  &hrefine, GIBuffer  &coarsen)
{
  char           *serr = "NTreeAdapt::CheckCoarsening: ";
  GINT           i, j, m, ilevel, nglevel, n0, nc, nr, nn, ind;
  GINT           ncl, nrl,  *idist=NULL, *ilist, ldim, nlev, elev, nsib;
  GKEY           nkey;
  GBOOL          bkeep, bnonConf, binrql, bincql;
  GIBuffer       hnew, cnew, ikeep, ikeepf, ibeg, iend ;
  GKEYBuffer     pkeybuff;  
  Elem2D         *elem, **esibs=NULL;
  GNeighborList  *eneighbor;
  GMortar1D      *mortar;
  stRefineQuery  *grquery=NULL, *gcquery=NULL;

  if ( !bInitialized_ ) {
    cout << serr << "object not initialized" << endl;
    exit(1);
  }

//if ( coarsen.dim() <= 0 ) return; 
  n0 = coarsen.dim();

  cnew .Resize(coarsen.dim());
  cnew = coarsen;
  cnew .distinct(idist, ldim);
  coarsen.Resize(ldim);
  for ( i=0; i<ldim; i++ ) coarsen(i) = cnew(idist[i]);
  if ( idist ) delete [] idist; idist = NULL;

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "Beginning check of coarsen indices..." << endl;
#endif

  // For coarsening, first check that an element *can* be
  // coarsened; i.e., that the element doesn't reside 
  // at its root node in the tree. Remove those elements
  // tagged for coarsening that cannot be coarsened:
  for ( i=0,nr=0; i<coarsen.dim(); i++ ) {
    elem  = (*gelems_)[coarsen(i)];
    if ( elem->GetID() > elem->GetRootID() ) cnew(nr++) = coarsen(i);
  }
  GComm::Synch();
  coarsen.Resize(nr);
  for ( i=0; i<nr; i++ ) coarsen(i) = cnew(i);
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "after root limit check: coarsen=" << coarsen << endl;
#endif
#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "removed from coarsen list after root elem check=" << n0-coarsen.dim() << endl;
#endif

//if ( coarsen.dim() <= 0 ) goto cleanup; 

#if defined(NEW_CONDITION)
  // Make certain that element index doesn't appear in 
  // both a refinement and coarsen request. If so, then
  // remove it from refinement and coarsening list.
  // [NOTE: CheckRefinement method may add an element to refinement list
  // for compatibility, and this is intended to nullify coarsening of that element.
  // Also, we don't have to re-do CheckRefinement for consistency.]:
  cnew.Resize(coarsen.dim());
  hnew.Resize(hrefine.dim());
  for ( i=0, nc=0; i<coarsen.dim(); i++ ) {
    if ( !hrefine.contains(coarsen(i),ind) ) cnew(nc++) = coarsen(i);
  }
  for ( i=0, nr=0; i<hrefine.dim(); i++ ) {
    if ( !coarsen.contains(hrefine(i),ind) ) hnew(nr++) = hrefine(i);
  }
  coarsen.Resize(nc);  
  hrefine.Resize(nr);  
  for ( i=0; i<nc; i++ )  coarsen(i) = cnew(i);
  for ( i=0; i<nr; i++ )  hrefine(i) = hnew(i);
//if ( coarsen.dim() <= 0 ) goto cleanup; 
#else
  // If index appears in both a refine and coarsen list, remove
  // it from the coarsen list:
  cnew.Resize(coarsen.dim());
  for ( i=0, nc=0; i<coarsen.dim(); i++ ) {
    if ( !hrefine.contains(coarsen(i),ind) ) cnew(nc++) = coarsen(i);
  }
  coarsen.Resize(nc);  
  for ( i=0; i<nc; i++ )  coarsen(i) = cnew(i);
#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "removed from coarsen list after refinelist duplication check=" << n0-coarsen.dim() << endl;
#endif
#endif
  
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "after cross-check of refinement list: coarsen=" << coarsen << endl;
#endif
#if defined(NT1_DEBUG_OUTPUT)
cout << serr << "after cross-check of refinement list: coarsen=" << coarsen << endl;
#endif

  // Check that all siblings of coarsened elements are in the list:
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "before sibling check, coarsen=" << coarsen << endl;
#endif
  cnew.Resize(coarsen.dim());
  cnew = coarsen;
  nr = AllSibsCoarsened(cnew, NULL, 0, ikeep); 
  coarsen.Resize(nr);
  for ( i=0; i<nr; i++ ) {
    coarsen[i] = cnew[ikeep[i]];
  }

#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "removed from coarsen list after 1st all-sibs check=" << n0-coarsen.dim() << endl;
#endif

#if defined(NT_DEBUG_OUTPUT)
  cout << serr <<  "ikeep=" << ikeep << endl;
  cout << serr << "after sibling check, coarsen=" << coarsen << endl;
#endif

  // Cannot allow a group of siblings to be coarsened, if
  // doing so would mean that the new parent element would 
  // have a face that has more than 1 level of refinement on it:
  BuildGQueryList(grquery, nrl, hrefine, NULL, 0);

  cnew.Resize(coarsen.dim());
  ikeep.Resize(coarsen.dim());
  cnew = coarsen;
  ikeep = -1;
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "before refine level consistency check, before re-order, coarsen=" << coarsen << endl;
#endif

  // Re-order remaining coarsen indices from finest refinement level to
  // coarsest. Retrieve the global max number of levels on grid, and the
  // start and stop indices in 'cnew' for each level (these may be the same 
  // when the level is not represented locally). Then cycle through each
  // refinement level to deterimine which elements should be kept
  // for coarsening, updating with each level the global query list that
  // contains the current element information for those elements still
  // allowed to be coarsened.
  OrderLevelDecreasing(cnew, nglevel);        // also get global number of levels
  nglevel++;
  ibeg.Resize(nglevel); iend.Resize(nglevel);
  hnew.Resize(cnew.dim());
  GetLevelRanges(ibeg, iend, cnew, nglevel);
#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "max_levels: " << nglevel << endl;
  cout << serr << "ibeg=" << ibeg << endl;
  cout << serr << "iend=" << iend << endl;
  cout << serr << "cnew=" << cnew << endl;
//cout << serr << "before refine level consistency check, after GetLevelRang, coarsen=" << cnew << endl;
#endif
  
  // Consider refinement levels from highest to lowest, as in cnew:
  for ( ilevel=nglevel-1, nr=0; ilevel>=0; ilevel-- ) { 
    ilist = ilevel == (nglevel-1) ? NULL : ikeep.Data(); 
    hnew  = -1;
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "ibeg[" << ilevel << "]=" << ibeg[ilevel] << " iend[" << ilevel << "]=" << iend[ilevel] << endl;
#endif
    BuildGQueryList(gcquery, ncl, cnew, ilist, nr);
    for ( i=ibeg[ilevel]; i<=iend[ilevel]; i++ ) { 
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "level=" << ilevel << " checking element no." << cnew[i] << endl;
#endif
      elem      = (*gelems_)[cnew[i]];
      mortar    = elem->GetEdgeMortar();
      eneighbor = elem->GetENeighbor();
      elev      = RefineLevel(elem);       
      for ( j=0, bkeep=TRUE; j<elem->GetNumEdges() && bkeep; j++ ) {
        nn       = eneighbor[j].size();
        bnonConf = !mortar[j].isConforming() ;
        // Determine if refined neighbors are to be coarsened or not--
        // if they are not, or if neighbor is to be refined,
        // then element is no good;
        for ( m=0; m<nn && bkeep; m++ ) { 
          nkey   = eneighbor[j][m].elemkey();
          nlev   = RefineLevel(eneighbor[j][m].elemrootkey(), nkey);       
          binrql = InQueryList(grquery,nrl,nkey,NULL,0);
          bincql = InQueryList(gcquery,ncl,nkey,NULL,0);
          nsib   = NumberSiblings(gcquery,ncl,ParentKey(eneighbor[j][m].elemrootkey(), nkey));
          bkeep  = bkeep && ( (bnonConf            ) 
//                         || (nn    > 1     &&   bincql)
//                         || (nn   == 1     &&  !binrql) 
//                         || (nn   == 1     && (!binrql || bincql)) 
//                         || (elev == nlev  &&  !binrql) 
                           || (elev  < nlev  &&   bincql && nsib == ndcomp_)
                           || (elev == nlev  &&  !binrql) 
                           );
        }
#if defined(NT1_DEBUG_OUTPUT)
        if ( !bkeep ) {
          cout << serr << "elem[" << cnew[i] << "] removed from c.list because its neighbors are not all in c.list" << endl;
        }
#endif
      }
      if ( bkeep ) ikeep(nr++) = i;
    } // end, local level loop, i
    // Remove all those that passed test, but don't have all siblings:

#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "removed from coarsen list after level test=" << n0-nr << endl;
#endif

#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "level[" << ilevel << "], before sib-check, coarsen=" << endl;
    for ( i=0; i<nr; i++ ) cout << cnew[ikeep[i]] << " " ; cout << endl;
#endif
    nn = AllSibsCoarsened(cnew, ikeep.Data(), nr, ikeepf);  
    ikeep = -1; 
    for ( i=0, nr=0; i<nn; i++ ) if ( !ikeep.contains(ikeepf[i],ind) ) ikeep[nr++] = ikeepf[i];
    
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "level[" << ilevel << "], after sib-check, coarsen=" << endl;
    for ( i=0; i<nr; i++ ) cout << cnew[ikeep[i]] << " " ; cout << endl;
#endif
  } // end, global refine level loop, ilevel
  // Find the modified coarsen list:
  coarsen.Resize(nr); 
  for ( i=0; i<nr; i++ ) {
//  if ( !coarsen.contains(ikeep[i],ind) ) coarsen[i] = cnew[ikeep[i]];
    coarsen[i] = cnew[ikeep[i]];
  } 
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "after refine level consistency check, coarsen=" << coarsen<< endl;
#endif

#if defined(NT1_DEBUG_OUTPUT)
  cout << serr << "_total removed from coarsen list=" << n0-coarsen.dim() << endl;
#endif

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "coarsen=" << coarsen << endl;
  cout << serr << "done." << endl;
#endif

cleanup:
  // delete local quantities:
  if ( grquery ) delete [] grquery;
  if ( gcquery ) delete [] gcquery;
  if ( idist   ) delete [] idist;
  if ( esibs   ) delete [] esibs;

  return;
} // end of method CheckCoarsening 


//************************************************************************************
//************************************************************************************
// METHOD     : ParentKey (1)
// DESCRIPTION: Computes parent key for element
// ARGUMENTS  : elem: element pointer, must not be NULL 
// RETURNS    : key.
//************************************************************************************
GKEY NTreeAdapt::ParentKey(Elem2D *elem)
{
  GWORD ihi, ilo;
  GKEY  ckey, pkey, rkey;

  if ( elem == NULL ) {
    cout << "NTreeAdapt::ParentKey: NULL element" << endl;
    exit(1);
  }
#if 0
  pkey = (GKEY)( ((GDOUBLE)(elem->GetID()))*indcomp_);
  pkey = MAX(elem->GetRootID(), pkey);
#endif
  
  rkey  = elem->GetRootID();
  ckey  = elem->GetID();
  ihi   = HIWORD(&ckey);  // h, s.t I_0 = 2^(dh)
  ilo   = LOWORD(&ckey);  // b, s.t.I   = I_0 + b
  ihi   --;               // reduce level by 1
  ilo   = ilo / pow(2.0,GDIM);
  SET_HW(&pkey,&ihi);
  SET_LW(&pkey,&ilo);

  return pkey;
} // end of method ParentKey (1)


//************************************************************************************
//************************************************************************************
// METHOD     : ParentKey (2)
// DESCRIPTION: Computes parent key for element
// ARGUMENTS  : rkey: element root key
//              ckey: element key
// RETURNS    : key.
//************************************************************************************
GKEY NTreeAdapt::ParentKey(GKEY rkey, GKEY ckey)
{
  GWORD ihi, ilo;
  GKEY  pkey;

  ihi   = HIWORD(&ckey);  // h, s.t I_0 = 2^(dh)
  ilo   = LOWORD(&ckey);  // b, s.t.I   = I_0 + b
  ihi   --;               // reduce level by 1
  ilo   = ilo / pow(2.0,GDIM);
  SET_HW(&pkey,&ihi);
  SET_LW(&pkey,&ilo);
    
  return pkey;
} // end of method ParentKey (2)


//************************************************************************************
//************************************************************************************
// METHOD     : ChildKey
// DESCRIPTION: Computes child key for element given parent
// ARGUMENTS  : elem: element pointer, must not be NULL 
// RETURNS    : key.
//************************************************************************************
GKEY NTreeAdapt::ChildKey(Elem2D *elem, GINT i)
{
  GINT  lp;
  GWORD ihi, ilo;
  GKEY  ckey, pkey, rkey;
    
  if ( elem == NULL ) {
    cout << "NTreeAdapt::ChildKey: NULL element" << endl;
    exit(1);
  }
  rkey  = elem->GetRootID();
  pkey  = elem->GetID();
  ihi   = HIWORD(&pkey);  // h, s.t I_0 = 2^(dh)
  ilo   = LOWORD(&pkey);  // b, s.t.I   = I_0 + b
  lp    = ihi - HIWORD(&rkey);
  ihi   ++;               // increase refine level by 1
  ilo   = (GWORD)pow(2.0,GDIM)*(ilo+1) + i;
  SET_HW(&ckey,&ihi);
  SET_LW(&ckey,&ilo);
    
  return ckey;
} // end of method ChildKey


//************************************************************************************
//************************************************************************************
// METHOD     : RefineLevel (1)
// DESCRIPTION: Computes element's refinement level
// ARGUMENTS  : elem: element pointer, must not be NULL 
// RETURNS    : key.
//************************************************************************************
GINT NTreeAdapt::RefineLevel(Elem2D *elem)
{
  GINT  lp;
  GKEY  ekey, rkey;
    
  if ( elem == NULL ) {
    cout << "NTreeAdapt::ChildKey: NULL element" << endl;
    exit(1);
  }
  rkey  = elem->GetRootID();
  ekey  = elem->GetID();
  lp    = HIWORD(&ekey) - HIWORD(&rkey);
    
  return lp;
} // end of method RefineLevel (1)


//************************************************************************************
//************************************************************************************
// METHOD     : RefineLevel (2)
// DESCRIPTION: Computes element's refinement level
// ARGUMENTS  : rkey: element root key
//              ckey: element key
// RETURNS    : key.
//************************************************************************************
GINT NTreeAdapt::RefineLevel(GKEY rkey, GKEY ckey)
{
  return (HIWORD(&ckey) - HIWORD(&rkey));
} // end of method RefineLevel (2)


//************************************************************************************
//************************************************************************************
// METHOD     : RefineElemental
// DESCRIPTION: Refines element specified by the input element pointer. 
//              Element list will be modified as a result of this call, but
//              NOTE: parent element/field is not deleted here. 
// ARGUMENTS  : 
//              parent_elem : pointer to corresp. parent element
//              parent_field: pointer to parent field/element
//              gelems      : grid element list whose members parent_elem points to
//              gfields     : field lists whose members parent_field points to
//              nfields     : number of parent_field's  and gfields to refine
//
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::RefineElemental(Elem2D *parent_elem, Field2D *parent_field[], GElemList *gelems, GFieldList *gfields[], GINT  nfields)
{
  char          *serr = "NTreeAdapt::RefineElemental: ";
  GINT          i, j, k, ntime, NN;
  GKEY          maxid, cid;
  GBOOL         bOk;
  ElemListSt    *stElem;
  GIBuffer      iremap;
  Elem2D        *child_elem;
  Field2D       *child_field;
  
  if ( gelems == NULL || nfields == 0 || !bInitialized_ ) {
    cout << serr << "object not initialized properly" << endl;
    return FALSE;
  }

  if ( parent_elem == NULL ) {
    cout << serr << "NULL parent elemen" << endl;
    return FALSE;
  }
 
  for ( j=0; j<nfields; j++ ) {
    if ( parent_field[j] == NULL || parent_elem != parent_field[j]->GetElement() ) {
      cout << serr << "parent field NULL or has inconsistent element" << endl;
      return FALSE;
    }
  }

  if ( (stElem=gelems->find(parent_elem)) == NULL ) {
    cout << serr << "unable to locate parent element in list" << endl;
    return FALSE;
  }
//maxid = max_key_mult_ * ( parent_elem->GetRootID() + 1 ) - 1;

  // Create children elements/fields:
#if 0
      cout << serr << " parent=" << *parent_elem << endl;
#endif
  for ( i=0, bOk=TRUE; i<ndcomp_ && bOk; i++ ) {
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "creating child[" << i << "]..." << endl;
#endif
    gelems->add(parent_elem->ElemType(),parent_elem->GetTempMgr()->GetNumBlocks());
    child_elem  = gelems->member();

#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "child element created: " << child_elem << endl;
#endif
    // Set element key and root key:
//  cid                     = ndcomp_ * parent_elem->GetID() + i;
    cid = ChildKey(parent_elem,i);
    child_elem->SetID(cid);
    child_elem->SetRootID(parent_elem->GetRootID());
    child_elem->SetParentID(parent_elem->GetID());
//  bOk                     = bOk && ( child_elem->GetID() <=  maxid ) 
    bOk                     = bOk && ( child_elem->GetID() > 0 );
#if 0
    cout << serr << "parent_id=" << parent_elem->GetID()
         << " computed from child:" << child_elem->GetParentID()  
         << " child_id: " << cid << " child_root_id: " << child_elem->GetRootID() 
         << " parent_root: " << parent_elem->GetRootID() << endl;
#endif

    if ( !bOk) {
       cout << serr << "failed to set child id for child[" << i << "]" << endl;
       cout << "                           : child_id = " << cid << " parent_root=" << parent_elem->GetRootID() << " parent_id=" << parent_elem->GetID() << " maxkeymultiplier=" << max_key_mult_  
            <<  endl;
       break;
    }
    // Set basis:
    for ( j=0,NN=1; j<GDIM && bOk; j++ ) {
      child_elem->SetBasis(parent_elem->GetBasisObj(j+1), j+1);
      NN *= (parent_elem->GetBasisObj(j+1)->GetOrder()+1);
    }
    // Enable dealiasing:
    for ( j=0; j<GDIM; j++ ) {
      child_elem->SetDBasis(parent_elem->GetDBasisObj(j+1),j+1);
    }

    // Set child vertices:
    bOk = bOk && SetChildVertices(parent_elem, child_elem, i);
#if defined(NT_DEBUG_OUTPUT)
    if ( !bOk) cout << serr << "failed to set child vertices for child[" << i << "]" << endl;
#endif
    // Compute child element data:
    bOk = bOk && child_elem->SolveFE();
#if defined(NT_DEBUG_OUTPUT)
    if ( !bOk) cout << serr << "failed to solve for child[" << i << "]" << endl;
#endif

    // Compute field data on child element:
    for ( j=0; j<nfields&& bOk; j++ ) {
      ntime = parent_field[j]->GetNumTimeLevels();
      gfields[j]->add(ntime, child_elem);
      child_field = gfields[j]->member();

      // Set child boundary information:
      bOk = bOk && SetChildBdy(parent_field[j], child_field, i);
#if 0
      if ( j == 0 ) 
      cout << serr << "child_elem[" << i << "]=" << *child_elem << endl;
      if ( !bOk) cout << serr << "failed to set bdy info for child[" << i << "]" << endl;
      cout << serr << "mapping parent element into child[" << i << "]" << endl;
#endif
      // Map all parent timelevel data into child, including time stamps:
      bOk = bOk && parent_field[j]->Map2NewElem(child_elem, child_field->GetExpCoeffs(), ntime, &iremap);
      bOk = bOk && ( iremap.dim() == NN ); 
      for ( k=0; k<ntime; k++ ) child_field->SetTime(k,parent_field[j]->GetTime(k));
#if defined(NT_DEBUG_OUTPUT)
      if ( !bOk) cout << serr << "failed to map parent into child[" << i << "]" << endl;
#endif
    }
  }
  
  return bOk;
} // end of method RefineElemental


//************************************************************************************
//************************************************************************************
// METHOD     : SetChildVertices
// DESCRIPTION: Sets child vertices from parent
// ARGUMENTS  : parent_elem: pointer to parent element
//              child_elem : pointer to child element
//              ichild     : which child we are setting
//
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::SetChildVertices(Elem2D *parent_elem, Elem2D *child_elem, GINT  ichild)
{ 
  GINT  i, nv; 
  Point *cvert, *pvert, *pmid, p_ctr;

  // NOTE: this method is good only for 2D quads.

  pvert        = parent_elem->GetSpVertices ();
  pmid         = parent_elem->GetSpMidpoints();
  nv           = parent_elem->GetNumVertices();
  cvert        = new Point [nv];
  p_ctr[0] = 0.5*(pvert[0][0] + pvert[1][0]);
  p_ctr[1] = 0.5*(pvert[0][1] + pvert[3][1]);

  // Good only for 2D regular quads:
  if ( dcomp_dir_ == 0 ) {      // 4x-decomposition
    cvert       [ichild] = pvert[ichild];
    cvert[(ichild+2)%nv] = p_ctr;
    cvert[(ichild+1)%nv] = pmid[(ichild+nv)%nv];
    cvert[(ichild+3)%nv] = pmid[(ichild+nv-1)%nv];
    
  }
  else if ( dcomp_dir_ == 1 ) { // 2x-decomposition in x
    if ( ichild == 0 ) {        // left element
      cvert[0] = pvert[0];
      cvert[1] = pmid [0];
      cvert[2] = pmid [2];
      cvert[3] = pvert[3];
    }
    else if ( ichild == 1 ) {   // right element
      cvert[0] = pmid [0];
      cvert[1] = pvert[1];
      cvert[2] = pvert[2];
      cvert[3] = pmid [2];
    }
  }
  else if ( dcomp_dir_ == 2 ) { // 2x-decomposition in y
    if ( ichild == 0 ) {        // bottom element
      cvert[0] = pvert[0];
      cvert[1] = pvert[1];
      cvert[2] = pmid [1];
      cvert[3] = pmid [3];
    }
    else if ( ichild == 1 ) {   // top element
      cvert[0] = pmid [3];
      cvert[1] = pmid [1];
      cvert[2] = pvert[2];
      cvert[3] = pvert[3];
    }
  }
#if 0
  for ( i=0; i<nv; i++ ) {
    cout << "NTreeAdapt::SetChildVertices: vertex[" << ichild << "][" << i << "]=" << cvert[i] << endl;
  }
#endif
  child_elem->SetVertices(cvert,nv);
  delete [] cvert; cvert = NULL;

  return TRUE;
} // end of method SetChildVertices


//************************************************************************************
//************************************************************************************
// METHOD     : SetChildBdy
// DESCRIPTION: Sets child bdy indices/ types and conditions from parent.
// ARGUMENTS  : parent_field: pointer to parent field
//              child_field : pointer to child field
//              i           : which child we are setting
//
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::SetChildBdy(Field2D *parent_field, Field2D *child_field, GINT  i)
{ 
  GINT          j, nv, ne;
  GBOOL         bOk, *pbBdyEdge, *cbBdyEdge;
  GIBuffer      *cbdy_indices,*iremap=NULL;
  GVector       *cubdy, *xnew[GDIM]={NULL};
  Elem2D        *parent_elem=NULL , *child_elem=NULL;

  if ( parent_field == NULL || child_field == NULL ) {
    cout << "NTreeAdapt::SetChildBdy: NULL input fields" << endl;
    return FALSE;
  }

  // NOTE: this method is good only for 2D quads.
  parent_elem  = parent_field->GetElement();
  child_elem   = child_field ->GetElement();
  nv           = parent_elem->GetNumVertices();
  ne           = parent_elem->GetNumEdges();
  pbBdyEdge    = new GBOOL [ne];
  cbBdyEdge    = new GBOOL [ne];
  for ( j=0; j<ne; j++ ) {
    pbBdyEdge[j]    = parent_elem->bGlobalBdyEdge(j);
  }

  cubdy         = child_field->GetBdyValues();

  // Set boundary indices/types (good only in 2D quads):
  for ( j=0; j<ne; j++ ) {
//  cbBdyEdge[j] = ( j==i || j==(i+ne-1)%ne ? pbBdyEdge[i] || pbBdyEdge[(i+ne-1)%ne] : FALSE );
    cbBdyEdge[j] = FALSE;
    if ( dcomp_dir_ == 0 )                          // refinement in x and y dirs
      if ( j == i || j == (i+ne-1)%ne ) cbBdyEdge[j] = pbBdyEdge[j];
    else if ( dcomp_dir_ == 1  || dcomp_dir_ == 2 ) // refinement in x or in y
      if ( j != (i+1) ) cbBdyEdge[j] = pbBdyEdge[j];
    child_elem->bGlobalBdyEdge(j) = cbBdyEdge[j];
  }
  for ( j=0; j<ne; j++ ) {
    if ( cbBdyEdge[j] ) {
      child_elem->GetEdgeType       (j) = parent_elem->GetEdgeType(j);
      child_elem->GetVertType       (j) = parent_elem->GetVertType(j);
      child_elem->GetVertType((j+1)%ne) = parent_elem->GetVertType((j+1)%ne);
    }
#if 0
    if ( cbBdyEdge[j] && cbBdyEdge[(j+ne-1)%ne] ) {
      child_elem->GetVertType       (j) = parent_elem->GetVertType(j);
      cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; VertTypes[" << j << "]: " << child_elem->GetVertType(j) << endl;
    }
#endif
  }
#if 0
  for ( j=0; j<ne; j++ ) {
    cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; Edge[" << j << "] Global?: " << cbBdyEdge[j] << endl;
    cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; EdgeTypes[" << j << "]: " << child_elem->GetEdgeType(j) << endl;
    cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; VertTypes[" << j << "]: " << child_elem->GetVertType(j) << endl;
  }
#endif

  child_elem->ComputeBdyInfo();
  cbdy_indices = child_elem->GetBdyIndices();

#if 0
  cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; bdy_indices[" << i << "]=" << *cbdy_indices << endl;
  cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; bdy_types[" << i << "]=" << *(child_elem->GetBdyTypes()) << endl;
#endif

  // Set child (DIRICHLET only) boundary values from parent boundary:
  cubdy->Resize(cbdy_indices->dim());
  xnew[0] = child_elem->GetSpNodes(1);
  xnew[1] = child_elem->GetSpNodes(2);
  bOk = parent_elem->Map2NewCoords(parent_field->GetExpCoeffs(0), xnew, GDIM, NULL, 
                                   cbdy_indices->dim(), cubdy, iremap);
#if 0
  cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; bdy_values[" << i << "]=" << *cubdy << endl;
  //cout << "NTreeAdapt::SetChildBdy: child[" << i << "]; i_remapped[" << i << "]=" << *iremap<< endl;
#endif
  if ( pbBdyEdge ) delete [] pbBdyEdge; pbBdyEdge = NULL;
  if ( cbBdyEdge ) delete [] cbBdyEdge; cbBdyEdge = NULL;
  if ( iremap    ) delete iremap      ; iremap    = NULL;

  return bOk;
} // end of method SetChildBdy


//************************************************************************************
//************************************************************************************
// METHOD     : CoarsenElemental
// DESCRIPTION: Coarsens sibling element/fields, by reconstituting siblings to parent
//              Element list will be modified as a result of this call, but
//              NOTE: sibling element/fields are not deleted here.
// ARGUMENTS  : esiblings: sibling list 
//              nsiblings: number of siblings
//              p_elem   : new parent
//              gelems   : grid element list whose members esiblings points to
//              gfields  : field lists to be coarsened
//              nfields  : number gfields to refine
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::CoarsenElemental(Elem2D *esiblings[], GINT  nsiblings, Elem2D *&p_elem, GElemList *gelems, GFieldList *gfields[], GINT nfields)
{ 
  char       *serr = "NTreeAdapt::CoarsenElemental: ";
  GINT       i, j, ntime ;
  GBOOL      bOk;
  GIBuffer   iremap;
  Field2D    *p_field, **fsiblings;
  FieldListSt *stField;

  if ( gelems == NULL || gfields == NULL || !bInitialized_ ) {
    cout << serr << "object not initialized properly" << endl;
    return FALSE;
  }
  if ( nsiblings != ndcomp_ ) {
    cout << serr << "incorrect sibling decomposition" << endl;
    return FALSE;
  }
  if ( bDoLoadBalance_ && bPartitioning_ ) {
    cout << serr << "load balancing not turned on" << endl;
    exit(1);
  }

  // Reorder siblings in natural order (required for
  // setting parent vertices and bcs):
  bOk   = ReorderSiblings(esiblings,nsiblings);
  
  gelems ->add(esiblings[0]->ElemType(),esiblings[0]->GetTempMgr()->GetNumBlocks());
  p_elem  = gelems->member();

  p_elem->SolveFE();

  // Set element key and root key:
  p_elem->SetID      (esiblings[0]->GetParentID());
  p_elem->SetRootID  (esiblings[0]->GetRootID());
  p_elem->SetParentID(ParentKey(p_elem));
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "pelem_root=" << p_elem->GetRootID() 
       << " pelem_id= " <<  p_elem->GetID    () 
       << " sibling_root=" << esiblings[0]->GetRootID() 
       << " sibling_parent=" << (esiblings[0]->GetParentID()) <<  endl;
#endif
  // Set basis:
  for ( i=0; i<GDIM && bOk; i++ ) {
    p_elem->SetBasis(esiblings[0]->GetBasisObj(i+1), i+1);
  }
  // Enable dealiasing:
  for ( j=0; j<GDIM; j++ ) {
    p_elem->SetDBasis(esiblings[0]->GetDBasisObj(j+1),j+1);
  }


  // Set parent vertices:
  bOk = bOk && SetParentVertices(esiblings, nsiblings, p_elem);
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "SetParentVertices: " << bOk << endl;
#endif

  // Compute parent element data:
  bOk = bOk && p_elem->SolveFE();
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "pelem_SolveFE: " << bOk << endl;
#endif

  // Do fields data now:
  fsiblings = new Field2D * [nsiblings];
  for ( j=0; j<nfields; j++ ) {
    gfields[j]->start(NULL);
    ntime   = gfields[j]->member()->GetNumTimeLevels();

    for ( i=0; i<nsiblings && bOk; i++ ) {
      fsiblings[i] = NULL;
      if ( (bOk=(stField=gfields[j]->find(esiblings[i])) != NULL) ) {
        fsiblings[i] = stField->member;
      }
    }
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "siblings_found: " << bOk << endl;
#endif

    gfields[j]->add(ntime, p_elem);
    p_field = gfields[j]->member();

    // Set parent boundary information:
    bOk = bOk && SetParentBdy(fsiblings, nsiblings, p_field);
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "SetParentBdy: " << bOk << endl;
#endif

    // Map all child timelevel data into parent:
    for ( i=0; i<nsiblings && bOk; i++ ) {
//cout << serr << "i=" << i << " fsib=" << *(fsiblings[i]->GetExpCoeffs(0)) << endl;
      bOk = bOk && fsiblings[i]->Map2NewElem(p_elem, p_field->GetExpCoeffs(), ntime, &iremap);
//cout << serr << "i=" << i << " iremap=" << iremap << endl << " p_field=" << *(p_field->GetExpCoeffs(0)) << endl;
#if defined(NT_DEBUG_OUTPUT)
      if ( !bOk ) {
        cout << serr << "remap from sibling[" << i 
             << "] to parent failed" << endl; 
      }
#endif
    }
  }
     
  if ( fsiblings ) delete [] fsiblings; fsiblings = NULL;
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "done. " << bOk << endl;
#endif

  return bOk;

} // end of method CoarsenElemental


//************************************************************************************
//************************************************************************************
// METHOD     : DoLoadBalance
// DESCRIPTION: Sets/gets flag for indicating that load balancing is to be done
// ARGUMENTS  : none
// RETURNS    : reference to bDoLoadBalance_ flag
//************************************************************************************
void NTreeAdapt::DoLoadBalance(GBOOL bal)
{
  bDoLoadBalance_ = bal;
} // end of method DoLoadBalance


//************************************************************************************
//************************************************************************************
// METHOD     : DoPartitioning
// DESCRIPTION: Sets/gets flag for indicating that paretitioning of any kind
//              is to be done.
// ARGUMENTS  : none
// RETURNS    : reference to bPartitioning_ flag
//************************************************************************************
void NTreeAdapt::DoPartitioning(GBOOL doit)
{
  bPartitioning_ = doit;
} // end of method DoPartitioning


//************************************************************************************
//************************************************************************************
// METHOD     : GetSiblings
// DESCRIPTION: Gets siblings of specified element
// ARGUMENTS  : 
//              e        : element for which to find siblings
//              esiblings: array of pointer to elements in list that 
//                         are siblings to e. This is allocated here, so
//                         caller responsible for deletion.
//              nsiblings: number of siblings to e
//              gelems   : list of element to which e points
// RETURNS    : none; method will exit if object not initialized
//************************************************************************************
void NTreeAdapt::GetSiblings(Elem2D *e, Elem2D **&esiblings, GINT  &nsiblings, GElemList *gelems)
{
  GINT   i, lref, le;
  GKEY   pkey, ekey;
  Elem2D *elem;

  if ( gelems == NULL ) {
    cout << "NTreeAdapt::GetSiblings: object not initialized properly" << endl;
    exit(1);
  }
  if ( esiblings != NULL ) {
    delete [] esiblings; esiblings = NULL;
  }

  pkey      = e->GetParentID();
  lref      = RefineLevel(e); //GNTREEADAPT_LEVEL(e->GetID(),e->GetiRootID());
#if defined(NT_DEBUG_OUTPUT)
  cout << "NTreeAdapt::GetSiblings: Finding siblings for parent_key=" << pkey << endl;
#endif
  nsiblings = 0;
  gelems->start(NULL);
#if defined(NT_DEBUG_OUTPUT)
  i =0;
#endif
  while ( (elem=gelems->member()) != NULL ) {
    le         = RefineLevel(elem); //GNTREEADAPT_LEVEL(elem->GetID(),elem->GetiRootID());
    nsiblings += ( (ekey=elem->GetParentID()) == pkey
                  &&                  le  == lref ? 1 : 0 );
#if defined(NT_DEBUG_OUTPUT)
    if ( ekey == pkey ) {
    cout << "NTreeAdapt::GetSiblings: pkey=" << pkey << " ekey=" << ekey 
         << " root=" << elem->GetRootID() << " nsiblings=" << nsiblings 
         << " Sibling elem id = [" << i << "]" << endl;
    }
    i++;
#endif
    gelems->next();
  }
  esiblings = new Elem2D * [nsiblings];
 
  i = 0;
  gelems->start(NULL);
  while ( (elem=gelems->member()) != NULL && i < nsiblings ) {
    le         = RefineLevel(elem); //GNTREEADAPT_LEVEL(elem->GetID(),elem->GetiRootID());
    if ( elem->GetParentID() == pkey && le == lref) {
      esiblings[i] = elem;
      i++;
    }
    gelems->next();
  }
  
#if defined(NT_DEBUG_OUTPUT)
cout << "NTreeAdapt::GetSiblings: done. " << endl;
#endif
} // end of method GetSiblings


//************************************************************************************
//************************************************************************************
// METHOD     : SetParentVertices
// DESCRIPTION: Sets parent element vertices given sibling elements
// ARGUMENTS  : 
//              esiblings: array of pointer to elements in list that
//                         are siblings to parent, in natural order. 
//              nsiblings: number of siblings
//              parent   : parent element 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::SetParentVertices(Elem2D *esiblings[], GINT  nsiblings, Elem2D *parent)
{ 
  GINT  i, nv;
  Point *pvert, **cvert;

  if ( gelems_ == NULL || gfields_ == NULL || !bInitialized_ ) {
    cout << "NTreeAdapt::SetParentVertices: object not initialized properly" << endl;
    return FALSE;
  }
  if ( nsiblings != ndcomp_ ) {
    cout << "NTreeAdapt::SetParentVertices: incorrect sibling decomposition" << endl;
    return FALSE;
  }
  if ( parent == NULL ) {
    cout << "NTreeAdapt::SetParentVertices: NULL parent" << endl;
    return FALSE;
  }

  nv    = parent->GetNumVertices();
  cvert = new Point * [ndcomp_];
  pvert = new Point        [nv];
  for ( i=0; i<ndcomp_; i++ ) {
    cvert[i] = esiblings[i]->GetSpVertices();
//  for ( j=0; j<4; j++ )
//    cout << "SetParentVertices: cvert[" << i << "]=" << cvert[i][j] << endl;

  }

  // Set parent vertices: good for 2D quads.
  // We can do this in this way because of the
  // (1) way the children are ordered; (2) way
  // the vertices of each child are ordered:
  for ( i=0; i<nv; i++ ) {
    pvert[i] = cvert[i][i];
//cout << "SetParentVertices: pvert[" << i << "]=" << pvert[i] << endl;
  }
  
  parent->SetVertices(pvert, nv);

  if ( cvert ) delete [] cvert; cvert = NULL;
  if ( pvert ) delete [] pvert; pvert = NULL;

  return TRUE;
} // end of method SetParentVertices


//************************************************************************************
//************************************************************************************
// METHOD     : SetParentBdy
// DESCRIPTION: Sets parent field bdy given sibling fields
// ARGUMENTS  : 
//              fsiblings: array of pointers to fields in list that
//                         are siblings to parent. 
//              nsiblings: number of siblings
//              parent   : parent field
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::SetParentBdy(Field2D *fsiblings[], GINT  nsiblings, Field2D *parent)
{ 
  GINT       i, j, m, ne;
  GBOOL      bOk;
  GIBuffer   *pbdy_indices, iremap;
  GVector    *pubdy, ptmp;
  Elem2D     **celem, *pelem;

  if ( gelems_ == NULL || gfields_ == NULL || !bInitialized_ ) {
    cout << "NTreeAdapt::SetParentBdy: object not initialized properly" << endl;
    return FALSE;
  }
  if ( nsiblings != ndcomp_ ) {
    cout << "NTreeAdapt::SetParentBdy: incorrect sibling decomposition" << endl;
    return FALSE;
  }
  if ( parent == NULL ) {
    cout << "NTreeAdapt::SetParentBdy: NULL parent" << endl;
    return FALSE;
  }

  // NOTE: Following good only for 2D quads:

  ne           = fsiblings[0]->GetElement()->GetNumEdges();
  pelem        = parent->GetElement();
  pubdy        = parent->GetBdyValues();
  pbdy_indices = pelem ->GetBdyIndices();
  celem        = new Elem2D * [nsiblings];
  for ( i=0; i<nsiblings; i++ ) {
    celem[i] = fsiblings[i]->GetElement();
  }
  for ( i=0; i<nsiblings; i++ ) {
    for ( j=0; j<ne; j++ ) {
      if ( j == i ) {
        pelem->GetEdgeType   (j) = celem[j]->GetEdgeType(j);
        pelem->GetVertType   (j) = celem[j]->GetVertType(j);
//      pelem->bGlobalBdyEdge(j) = pelem->GetEdgeType   (j) != NONE;  //celem[i]->bGlobalBdyEdge(i);
        pelem->bGlobalBdyEdge(j) = celem[i]->bGlobalBdyEdge(j);
      }
    }
  }
#if 0
  for ( j=0; j<ne; j++ ) {
    if ( pelem->bGlobalBdyEdge(j) ) {
      pelem->GetEdgeType(j) = celem[j]->GetEdgeType(j);
      pelem->GetVertType(j) = celem[j]->GetVertType(j);
    }
  }
#endif
  pelem->ComputeBdyInfo();

#if 0
  cout << "NTreeAdapt::SetParentBdy: pelem=" << *pelem << endl;
  cout << "NTreeAdapt::SetParentBdy: pbdy_indices =" << *pbdy_indices << endl;
#endif
   
  // Set child (DIRICHLET only) boundary values from parent boundary:
  pubdy->Resize(pbdy_indices->dim());
  for ( i=0, m=1; i<GDIM; i++ ) m *= (pelem->GetOrder(i+1)+1);
  ptmp.Resize(m);
  ptmp = 0.0;
  for ( i=0, bOk=TRUE; i<nsiblings && pubdy->dim() && bOk; i++ ) {
    bOk = fsiblings[i]->Map2NewCoords(pelem->GetSpNodes(1),pelem->GetSpNodes(2),
                                       pbdy_indices, &ptmp, 0, &iremap);
  }
  for ( i=0; i<pbdy_indices->dim(); i++ ) (*pubdy)[i] = ptmp[(*pbdy_indices)[i]];
#if defined(NT_DEBUG_OUTPUT)
  cout << "NTreeAdapt::SetParentBdy: completed." << endl;
#endif
  if ( celem  ) delete [] celem; celem  = NULL;

  return TRUE;
  
} // end of method SetParentBdy


//************************************************************************************
//************************************************************************************
// METHOD     : ReorderSiblings
// DESCRIPTION: Re-orders siblings to 'natural order'
// ARGUMENTS  :
//              esiblings: array of pointer to elements in list that
//                         are siblings to parent.
//              nsiblings: number of siblings
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::ReorderSiblings(Elem2D *esiblings[], GINT  nsiblings)
{
  GINT    i, ir, *ind;
  Elem2D  **etmp;

  if ( nsiblings != ndcomp_ ) {
    cout << "NTreeAdapt::ReorderSiblings: incorrect sibling decomposition" << endl;
    return FALSE;
  }
  if ( nsiblings != ndcomp_ ) {
    cout << "NTreeAdapt::ReorderSiblings: incorrect sibling decomposition/NULL parent" << endl;
    return FALSE;
  }
  
  etmp = new Elem2D * [nsiblings];
  ind  = new GINT     [nsiblings];
  for ( i=0; i<nsiblings; i++ ) etmp[i] = esiblings[i];

#if 0
  for ( i=0; i<nsiblings; i++ ) {
    cout << "ReorderSibs: before reorder: sibling[" << i << "]=" << *esiblings[i];
  }
#endif
  for ( i=0; i<nsiblings; i++ ) {
     if ( (ir=((etmp[i]->GetID())%ndcomp_)) < 0 || ir >= ndcomp_ ) return FALSE;
#if defined(NT_DEBUG_OUTPUT)
     cout << "NTreeAdapt::ReorderSiblings: sib[" << i << "]_id=" << etmp[i]->GetID() << " ir=" <<ir << endl;
#endif
     ind[ir] = ir;    
     esiblings[ir] = etmp[i];
  }

  // Array ind[n], n in (0,ndcomp_-1) gives the indices of the original
  // siblings
//for ( i=0; i<nsiblings; i++ ) esiblings[i] = etmp[ind[i]];

#if 0
  for ( i=0; i<nsiblings; i++ ) {
    cout << "NTreeAdapt::ReorderSiblings: sibling[" << i << "]=" << *esiblings[i];
  }
#endif
  
  if ( ind  ) delete [] ind ; 
  if ( etmp ) delete [] etmp; 

  return TRUE;
} // end of method ReorderSiblings


//************************************************************************************
//************************************************************************************
// METHOD     : SetRefineDir
// DESCRIPTION: Set refinement direction. All subsequet refinements will be carried
//              out in this direction until this is changed. Default is 0.
// ARGUMENTS  : 
//              idir : direction of splitting, in [0, GDIM]
// RETURNS    : none
//************************************************************************************
void NTreeAdapt::SetRefineDir(GINT  idir)
{
  if ( idir < 0 || idir > GDIM ) {
    cout << "NTreeAdapt::SetRefineDir: invalid refinement direction" << endl;
    exit(1);
  }
  dcomp_dir_ = idir;
  if ( dcomp_dir_ == 0 ) ndcomp_ = 4;
  else                   ndcomp_ = 2;
} // end of method SetRefineDir


//************************************************************************************
//************************************************************************************
// METHOD     : EPointersToIndices
// DESCRIPTION: Convert elem pointers into gelemlist indices
//              Comparisons are made with gelems_ member element pointers,
//              so the gelems_ data should be renumbered before entry.
// ARGUMENTS  : 
//              ibuff : return buffer of indices into gelems_ list
//              pelem : array of element pointers to match with gelems_ data
//              npelem: number of pointers in pelem array
// RETURNS    : none
//************************************************************************************
void NTreeAdapt::EPointersToIndices(GIBuffer  &ibuff, Elem2D *pelem[], GINT  npelem)
{
  char     *serr = "NTreeAdapt::EPointersToIndices: ";
  GINT      i, j, nb, ind;
  GIBuffer  btmp;
  Elem2D    *e;

#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "starting..." << endl;
#endif
  btmp.Resize(npelem); btmp = -1;
  for ( j=0, nb=0; j<npelem; j++ ) {
    gelems_->start(NULL);
    for ( i=0; i<gelems_->size(); i++ ) { 
      e = gelems_->member();
      if ( e == pelem[i] && !btmp.contains(i,ind) ) btmp(nb++) = i;
      gelems_->next();
    }
  }
  ibuff.Resize(nb);
  for ( j=0; j<nb; j++ ) ibuff[j] = btmp[j]; 
#if defined(NT_DEBUG_OUTPUT)
  cout << serr << "done." << endl;
#endif

} // end of method EPointersToIndices


//************************************************************************************
//************************************************************************************
// METHOD     : CheckPeriodic
// DESCRIPTION: 
// ARGUMENTS  : i : element index of failing element, returned
//              j : edge index, of failing edge, returned 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::CheckPeriodic(GINT  &i, GINT  &j)
{
  GINT          k;
  GNeighborList *eneighbor;
  Elem2D        *elem; 
  
  gelems_->start(NULL);
  for ( i=0; i<gelems_->size(); i++ ) {
    elem = gelems_->member();
    eneighbor = elem->GetENeighbor();
    for ( j=0; j<elem->GetNumEdges(); j++ ) {
        k = (j+2)%elem->GetNumEdges();
      if ( elem->GetEdgeType(j) == PERIODIC
        && elem->GetEdgeType(k) != PERIODIC
        && eneighbor[j].size() < 1
        && gelems_->size() > 1 ) {
        return FALSE;
      }
    }
    gelems_->next();
  }

  return TRUE;

} // end of method CheckPeriodic


//************************************************************************************
//************************************************************************************
// METHOD     : SetLevelLimit
// DESCRIPTION: Sets max levels for which refinement will be done. In no case
//              can this limit exceed MAX_REFINE_LEVELS 
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void NTreeAdapt::SetLevelLimit(GINT  nlimit)
{
  if ( nlimit > MAX_REFINE_LEVELS ) {
    cout << "NTreeAdapt::SetLevelLimit invalid level limit" << endl;
    exit(1);
  }
  if ( nlimit <= 0 ) {
    nlevel_limit_  = MAX_REFINE_LEVELS;
    bLimitByLevel_ = FALSE; 
    return;
  }
  nlevel_limit_  = nlimit;
  bLimitByLevel_ = TRUE; 
} // end of method SetLevelLimit

//************************************************************************************
//************************************************************************************
// METHOD     : OrderLevelDecreasing
// DESCRIPTION: Reorders the input buff of indices to elementlist s.t.
//              elements in list go from most refined to least refined;
//              i.e., from higher refinement level to lower.
// ARGUMENTS  : GIBuffer  buff: returned with new ordering
//              nglevel      : global number of refinement levels
// RETURNS    : none
//************************************************************************************
void NTreeAdapt::OrderLevelDecreasing(GIBuffer  &buff, GINT  &nglevel)
{
  char       *serr = "NTreeAdapt::OrderLevelDecreasing: ";
  GINT       i, j, nbuff, itmp, ili, ilj, lmax;
  Elem2D     *elemi, *elemj;

  nbuff = buff.dim();

  // Sort buff levels from highest to lowest level.
  // NOTE: this can be done much more efficiently (e.g., using heapsort):
  lmax = 0;
  for ( j=0; j<nbuff; j++ ) {
    elemj = (*gelems_)[buff[j]];
    ilj   = RefineLevel(elemj); //GNTREEADAPT_LEVEL(elemj->GetID(),elemj->GetiRootID());
#if defined(NT_DEBUG_OUTPUT)
    cout << serr << "elemj_id=" << elemj->GetID() 
         << " elemj_rootid=" << elemj->GetRootID() 
         << " level_num=" << ilj <<  endl;
#endif
    lmax  = MAX(lmax,ilj);
    for ( i=j+1; i<nbuff; i++ ) {
      elemi = (*gelems_)[buff[i]];
      ili   = RefineLevel(elemi); //GNTREEADAPT_LEVEL(elemi->GetID(),elemi->GetiRootID());
      if ( ili > ilj ) {
        itmp = buff[j]; buff[j] = buff[i]; buff[i] = itmp;
        elemj   = (*gelems_)[buff[j]];
        ilj     = RefineLevel(elemj); //GNTREEADAPT_LEVEL(elemj->GetID(),elemj->GetiRootID());
      }
    }
  }

  for ( j=0; j<nbuff; j++ ) {
    elemj = (*gelems_)[buff[j]];
    ilj   = RefineLevel(elemj); //GNTREEADAPT_LEVEL(elemj->GetID(),elemj->GetiRootID());
    lmax  = MAX(lmax,ilj);
  }

#if 0
  cout << " OrderLevelDecreasing : levels=" << endl;
  for ( j=0; j<nbuff; j++ ) {
    elemj = (*gelems_)[buff[j]];
    cout << "elem[" << buff[j] << "]; eid=" << elemj->GetID()
         << " rootid=" << elemj->GetiRootID() << " ilevel=" << RefineLEvel(elemj) << " "
         << endl;
  }
exit(1);
#endif
  GComm::Allreduce(&lmax, &nglevel, 1, GC_GINT , G_OP_MAX);
} // end of method OrderLevelDecreasing


//************************************************************************************
//************************************************************************************
// METHOD     : GetLevelRanges
// DESCRIPTION: Gets the refinement level ranges of input element id list. Ranges
//              are indices in the the element id list buffer, and specify
//              the start (ibeg) and end (iend) of the refinement level range.
//              There will be nglevel elements in the ibeg/iend buffers, so that
//              all refinement levels will have a range pair. The global number
//              of refinement levels, nglevel, must be supplied.
// ARGUMENTS  : buff          : buffer of element indices, ordered according to
//                              decreasing refinement level 
//              ibeg          : buffer of indices of length nglevel, that
//                              contain the start in buff of the element indices 
//                              at a given refinement level. These are ordered in 
//                              the reverse of buff.
//              iend          : buffer of indices of length nglevel, that
//                              contain the end in buff of the element indices 
//                              at a given refinement level. These are ordered in 
//                              the reverse of buff.
//               nglevel      : global number of refinement levels
// RETURNS    : none
//************************************************************************************
void NTreeAdapt::GetLevelRanges(GIBuffer  &ibeg, GIBuffer  &iend, GIBuffer  &buff, GINT  &nglevel)
{
  char       *serr = "NTreeAdapt::GetLevelRanges: ";
  GINT       i, j, ltmp, il;
  GIBuffer   ilevel;

  // Find the beg/end pairs for each level in 
  // local element index list. Pairs will be ordered
  // in decreasing refinement level order:
  for ( j=nglevel-1, i=0; j>=0; j-- ) { // j indexes level number
    ltmp    = j + 1;
    ibeg[j] = buff.dim(); 
    iend[j] = 0;
    while ( i < buff.dim() 
//       && (il=GNTREEADAPT_LEVEL((*gelems_)[buff[i]]->GetID(),(*gelems_)[buff[i]]->GetiRootID())) >  ltmp ) { 
         && (il=RefineLevel((*gelems_)[buff[i]])) >  ltmp ) { 
      i++; 
    }
    if ( i < buff.dim() ) ibeg[j] = i; 
    while ( i < buff.dim() 
//       && (il=GNTREEADAPT_LEVEL((*gelems_)[buff[i]]->GetID(),(*gelems_)[buff[i]]->GetiRootID())) == ltmp ) { 
         && (il=RefineLevel((*gelems_)[buff[i]])) ==  ltmp ) { 
      i++; 
    }
    if ( i <= buff.dim() ) iend[j] = i-1;
  }
#if 0
  cout << serr << "level start/stop indices into coarsen buff:" << " ibeg=" << ibeg
       << " iend=" << iend << endl; 
  exit(1);
#endif
} // end of method GetLevelRanges


//************************************************************************************
//************************************************************************************
// METHOD     : InQueryList
// DESCRIPTION: Is specified key in the query key list?
// ARGUMENTS  : lrquery : list (array) of stRefineQuery structures
//              nquery : number in the lrquery array
//              key    : key to find in list
//              isrch  : which indices of lrquery to search. If NULL, all
//                       indices are searched.
//              nsrch     : number of isrch indices to use.
// RETURNS    : TRUE if in list; else FALSE
//************************************************************************************
GBOOL NTreeAdapt::InQueryList(stRefineQuery *lrquery, GINT  nquery, GKEY key, GINT  *isrch, GINT  nsrch)
{
  GINT       j=0;

  if ( lrquery == NULL && nquery > 0 ) {
    cout << "NTreeAdapt::InQueryList: NULL query list" << endl;
    exit(1);
  }

  if ( key == NULL_HOSTKEY ) return FALSE;

  if ( isrch == NULL ) {
    while ( j<nquery && key != lrquery[j].elem_key ) j++;
  }
  else {
    while ( j<nsrch && key != lrquery[isrch[j]].elem_key ) j++;
  }

  return (j < nquery);

} // end of method InQueryList


//************************************************************************************
//************************************************************************************
// METHOD     : NumberSiblings
// DESCRIPTION: Number of siblings in query list that correspond to
//              specified parent
// ARGUMENTS  : lquery : list (array) of stRefineQuery structures
//              nquery : number in the lrquery array
//              pkey   : parent key of siblings 
// RETURNS    : TRUE if in list; else FALSE
//************************************************************************************
GINT  NTreeAdapt::NumberSiblings(stRefineQuery lquery[], GINT  nquery, GKEY pkey)
{
  char  *serr = "NTreeAdapt::NumberSiblings: ";
  GINT  j, num;

#if 0
    cout << serr << "^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
#endif
  for ( j=0, num=0; j<nquery; j++ ) {
    num += ( lquery[j].parent_key == pkey 
          && lquery[j].parent_key != NULL_HOSTKEY ? 1 : 0 );
#if 0
    if ( lquery[j].parent_key == pkey && lquery[j].parent_key != NULL_HOSTKEY )
      cout << serr << " lquery.elem_key=" << lquery[j].elem_key << " lquery.parent_key= " <<  lquery[j].parent_key << endl;
#endif
  }
#if 0
    cout << serr << "^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
#endif

  return num;

} // end of method NumberSiblings


//************************************************************************************
//************************************************************************************
// METHOD     : ElemKeyToIndex
// DESCRIPTION: Provides indices into current element list given element keys
// ARGUMENTS  : 
//              index: buffer of indices corresp. to specified element keys 
//              key  : buffer containing keys for elements specified by indices, index
// RETURNS    : number of keys found in element list

//************************************************************************************
GINT NTreeAdapt::ElemKeyToIndex(GIBuffer &index, GKEYBuffer &key)
{
  char  *serr = "NTreeAdapt::ElemKeyToIndex: ";
  GINT  i, j, n;

  index.Resize(key.dim());
  index = -1;
  for ( j=0, n=0; j<key.dim(); j++ ) {
    gelems_->start(NULL);
    for ( i=0; i<gelems_->size(); i++ ) {
//    cout << serr << "key[" << j << "]=" << key[j] << " eid[" << i << "]=" << (*gelems_)[i]->GetID() << endl;
      if ( gelems_->member()->GetID() == key[j] ) {
         index[n] = i;  
         n++;
         break;                                 // find only first occurrence of key in list
      }
      gelems_->next();
    } // end, element loop
  } // end, list of coarsen keys

  return n;

} // end of method ElemKeyToIndex


//************************************************************************************
//************************************************************************************
// METHOD     : ElemIndexToKey
// DESCRIPTION: Provides element keys provided indices into current element list
// ARGUMENTS  : 
//              key  : buffer containing keys for elements specified by indices, index
//              index: buffer of indices corresp. to specified element keys
// RETURNS    : none
//************************************************************************************
void NTreeAdapt::ElemIndexToKey(GKEYBuffer &key, GIBuffer &index)
{
  GINT  i;

  key.Resize(index.dim());
  for ( i=0; i<index.dim(); i++ ) {
     key[i] = (*gelems_)[index[i]]->GetID();
  }

} // end of method ElemIndexToKey



//************************************************************************************
//************************************************************************************
// METHOD     : AllSibsCoarsened
// DESCRIPTION: Determines indices of coarsen list that have elements for
//              all siblings in the list.
// ARGUMENTS  : coarsen : initial list of element indices to be coarsened. Note that
//                        element indices must be unique; the same element index must
//                        not appear more than once.
//              isrch   : list of coarsen indices to consider. If NULL, entire
//                        list will be considered. Note that this search list must
//                        point to unique point in the 'coarsen' buffer, or we'll
//                        get incorrect count for the number of siblings.
//              nsrch   : number of coarsen indices to consider
//              ikeep   : indices into coarsen list that have all sibs 
//                        represented; is reallocated (if necessary) here.
// RETURNS    : number of ikeep elements that contain valid data 
//************************************************************************************
GINT  NTreeAdapt::AllSibsCoarsened(GIBuffer  &coarsen, GINT  *isrch, GINT  nsrch, GIBuffer  &ikeep)
{
  char           *serr = "NTreeAdapt::AllSibsCoarsened: " ;
  GINT           i, ldim, ncl, np, nr, nsib;
  stRefineQuery  *gcquery=NULL;
  Elem2D         *elem;

  ldim = isrch==NULL ? coarsen.dim() : nsrch;

  // Make sure that all siblings of all elements in coarsen list
  // are also in the coarsen list:
  ikeep.Resize(ldim); ikeep = -1;

  np = BuildGQueryList(gcquery, ncl, coarsen, isrch, nsrch);
#if 0
  cout << "&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&" << endl;
  cout << "&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&" << endl;
  cout << serr << "query list=" << endl;
  GINT ip, k; 
  for ( GINT ip=0; ip<GComm::WorldSize(); ip++ ) {
    for ( i=0; i<np; i++ ) {
      k = ip*np + i;
      cout << "proc: " << ip << " record[" << i << "]:  e_id=" << gcquery[k].elem_key 
           << " parent_id=" << gcquery[k].parent_key 
           << " root_id=" << gcquery[k].root_key
           << " ilevel=" << gcquery[k].iflag << endl;
    }  
  }  
  cout << "&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&" << endl;
  cout << "&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&" << endl;
#endif

  // Check global coarsen list for number of siblings for each element:
  if ( isrch == NULL ) {
    for ( i=0, nr=0; i<ldim; i++ ) {   
      elem  = (*gelems_)[coarsen[i]];
       if ( (nsib=NumberSiblings(gcquery, ncl, elem->GetParentID())) 
          == ndcomp_ ) ikeep(nr++) = i; 
#if defined(NT1_DEBUG_OUTPUT)
       else  {cout << serr << " pkey=" << elem->GetParentID() << " elem_key[" << coarsen[i] << "]: " << elem->GetID() << " : nsib=" << nsib << endl;}
#endif
    }
  }
  else {
    for ( i=0, nr=0; i<ldim; i++ ) {   
      elem  = (*gelems_)[coarsen[isrch[i]]];
       if ( (nsib=NumberSiblings(gcquery, ncl, elem->GetParentID())) 
          == ndcomp_ ) ikeep(nr++) = isrch[i]; 
    }
  }
  if ( gcquery != NULL ) delete [] gcquery;

  return nr;

} // end of method AllSibsCoarsened


//************************************************************************************
//************************************************************************************
// METHOD     : BuildFQueryList
// DESCRIPTION: Builds global query list given coarse or refine list
// ARGUMENTS  : querylist : array of query structures, allocated here, so caller
//                          is responsible for deletion.
//              nquerylist: number of elements in querylist
//              rclist    : initial list of element indices to be refined/coarsened, 
//                          and queried globally.
//              isrch     : array containing valid indices of rclist
//                          to consider. If NULL, entire rclist is used.
//              nsrch     : number of isrch indices to use.
// RETURNS    : GINT: number of records for each processor
//************************************************************************************
GINT NTreeAdapt::BuildGQueryList(stRefineQuery *&gquerylist, GINT  &nquerylist, 
                                 GIBuffer  &rclist, GINT  *isrch, GINT  nsrch)
{ 
  GINT          i, ldim, gdim;
  Elem2D        *elem;
  stRefineQuery *lquery=NULL;

  if ( gquerylist != NULL ) delete [] gquerylist; gquerylist = NULL;

  ldim = isrch == NULL ? rclist.dim() : nsrch;
  GComm::Allreduce(&ldim, &gdim, 1, GC_GINT , G_OP_MAX);
  nquerylist = gdim <= 0 ? 0 : nprocs_*gdim;
  if ( gdim <=  0 ) return 0;

  lquery     = new stRefineQuery [gdim];
  gquerylist = new stRefineQuery [nquerylist];
  for ( i=ldim; i<gdim; i++ ) {  // initialize local structure
    lquery[i].elem_key   = NULL_HOSTKEY;
    lquery[i].parent_key = NULL_HOSTKEY;
    lquery[i].root_key   = NULL_HOSTKEY;
    lquery[i].iflag      = FALSE;
  }
  if ( isrch == NULL ) {
    for ( i=0; i<ldim; i++ ) {   // set local query structure array
      elem  = (*gelems_)[rclist[i]];
      lquery[i].elem_key   = elem->GetID();
      lquery[i].parent_key = elem->GetParentID(); 
      lquery[i].root_key   = elem->GetRootID();
      lquery[i].iflag      = RefineLevel(elem); //GNTREEADAPT_LEVEL(elem->GetID(),elem->GetiRootID());
    }
  }
  else {
    for ( i=0; i<ldim; i++ ) {   // set local query structure array
      elem  = (*gelems_)[rclist[isrch[i]]];
      lquery[i].elem_key   = elem->GetID();
      lquery[i].parent_key = elem->GetParentID(); 
      lquery[i].root_key   = elem->GetRootID();
      lquery[i].iflag      = RefineLevel(elem); //GNTREEADAPT_LEVEL(elem->GetID(),elem->GetiRootID());
    }
  }
  GComm::Allgather(lquery, gdim, refq_type_, gquerylist, gdim, refq_type_);

  if ( lquery != NULL ) delete [] lquery;
  return gdim;

} // end of method BuildQueryList


//************************************************************************************
//************************************************************************************
// METHOD     : Restrict2OneSibMem
// DESCRIPTION: Restrict input indices, so that on output only
//              a single index representing one sibling from among
//              a group of siblings, is given.
// ARGUMENTS  : outind: output array of indices properly sized
//              inkeys: input array of element keys to convert 
// RETURNS    : none.
//************************************************************************************
void NTreeAdapt::Restrict2OneSibMem(GIBuffer &outind, GKEYBuffer &inkeys)
{
  char       *serr = "NTreeAdapt::Restrict2OneSibMem: ";
  GINT       ldim, *idist=NULL;
  GKEYBuffer pkeybuff;
  GIBuffer   inind;

  if ( inkeys.dim() == 0 ) {
    outind.Resize(0);
    return;
  }
  inind.Resize(inkeys.dim());
  ElemKeyToIndex(inind, inkeys);

  // Remove all but one sibling element from input index list
  pkeybuff.Resize(inind.dim());
  for ( GINT i=0; i<inind.dim(); i++ ) {  // get parents for all elems in input list
    pkeybuff[i] = (*gelems_)[inind[i]]->GetParentID(); 
  }
  pkeybuff.distinct(idist, ldim);         // find unique indices with unique parents in list
  outind.Resize(ldim);
  for ( GINT i=0; i<ldim; i++ ) {
    outind[i] = inind[idist[i]]; 
  }

  delete [] idist;

} // end of method Restrict2OneSibMem
