//************************************************************************************//
// Module       : vdb.hpp
// Date         : 8/5/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a voxel database object
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <math.h>
#include <string.h>
#include "vdb.hpp"
#include "gcomm.hpp"
#include "gtbuffer.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
VDB::VDB()
:
nprocs_               (1),
this_rank_            (0),
eps_                  (TINY*10.0),
keygen_               (NULL),
vdbst_type_           (0)
{
  nprocs_    = GComm::WorldSize();
  this_rank_ = GComm::WorldRank();

  // Register the GVDBst data structure type for communication:
  AGINT       blk_ptr [NVDBST_MEM];
  GINT        n_types [NVDBST_MEM];
  GC_DATATYPE sz_types[NVDBST_MEM];
  GVDBst      tstruct;

  n_types [0] = n_types[1] = n_types[2] = n_types[3] = n_types[4] = 1; n_types [5] = GDIM;
  sz_types[0] = GC_GKEY;
  sz_types[1] = GC_GKEY;
  sz_types[2] = GC_GINT ;
  sz_types[3] = GC_GINT ;
  sz_types[4] = GC_GINT ;
  sz_types[5] = GC_GDOUBLE;

  GComm::Address(&(tstruct.hostkey)    , &blk_ptr[0]);
  GComm::Address(&(tstruct.hostrootkey), &blk_ptr[1]);
  GComm::Address(&(tstruct.hostid)     , &blk_ptr[2]);
  GComm::Address(&(tstruct.id)         , &blk_ptr[3]);
  GComm::Address(&(tstruct.ancillary)  , &blk_ptr[4]);
  GComm::Address(&(tstruct.x)          , &blk_ptr[5]);
  if ( !GComm::DataTypeFromStruct(blk_ptr, sz_types, n_types, NVDBST_MEM, &vdbst_type_) ) {
    cout << "VDB::VDB (1): Comm datatype registration failed" << endl;
    exit(1);
  }

} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Destructor
VDB::~VDB()
{
    GComm::DataTypeFree(&vdbst_type_);
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
VDB::VDB(const VDB &a)
{
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : add (1)
// DESCRIPTION: Adds point to DB. If point==NULL, nothing is done.
// ARGUMENTS  : point     : Point3D array; default is pointer to one element
//              key       : GKey array of length 'num'. May be NULL; if so, and 
//                          if keygen_ is not set (via SetGenKey), it defaults to 0.
//                          If keygen_ is set, then this array isn't used.
//              hostkey   : cross-reference host key of 'point'; e.g., element id for 'point'
//              hostid    : cross-reference host id of 'point'; e.g., element id for 'point'
//              localid   : local id for point (may corresp. to an edge, vertex, or face, for e.g.)
//              iproc     : processor id (rank) array, of length 'num'
//              num       : number of records to add
// RETURNS    : none
//************************************************************************************
void VDB::add(Point3D *point, GKEY *key, GKEY *hostkey, GKEY *hostrootkey, 
              GINT  *localid, GINT  *ancillary, GINT  *hostid, GSHORT  *iproc, GINT  num )
{
  GINT       i;
  GKEY       *genkey;

  if ( point == NULL || iproc == NULL ) return;

  genkey = new GKEY[num];
  if ( keygen_ != NULL ) keygen_->key(genkey, G_TYPESZ[G_GKEY], point, num);

  for ( i=0; i<num; i++ ) {
    vdb_list_.add();
    if ( keygen_ == NULL ) {
      vdb_list_.member()->key  ()    = key==NULL ? 0 : key[i];
    }
    else {
      vdb_list_.member()->key  ()    = genkey[i];
    }
    vdb_list_.member()->proc       () = iproc      [i];
    vdb_list_.member()->ancillary  () = ancillary  [i];
    vdb_list_.member()->hostkey    () = hostkey    [i];
    vdb_list_.member()->hostrootkey() = hostrootkey[i];
    vdb_list_.member()->hostid     () = hostid     [i];
    vdb_list_.member()->localid    () = localid    [i];
    vdb_list_.member()->point      () = point      [i]; 
    vdb_list_.member()->point      ().Bracket(eps_);
  }
  delete [] genkey ;

} // end of method add(1)
 

//************************************************************************************
//************************************************************************************
// METHOD     : del (1)
// DESCRIPTION: Deletes point record(s) from DB. If point==NULL, nothing is done.
//              DB list is set to start upon exit.
// ARGUMENTS  : point     : Point3D array; default is pointer to one element
//              num       : number of records to delete
// RETURNS    : TRUE if at least one element found and deleted; else FALSE 
//************************************************************************************
GBOOL VDB::del(Point3D *point, GINT  num)
{
  GINT       i, n=0;
//GBOOL      bDel=FALSE;
  VDBData    *m;

  if ( point == NULL ) return TRUE;

  for ( i=0; i<num; i++ ) {
    vdb_list_.start(NULL);
    while ( vdb_list_.size() && (m=vdb_list_.member()) ) {   // do deletions while not empty & successful
      if ( (m->point() == point[i]) ) {
//      bDel = vdb_list_.del(m) ? TRUE : FALSE;
//      cout << "VDB::del: deleted index: " << n << endl;
        n++;
//      break;
      }
      vdb_list_.next();
    }
  }
     
  vdb_list_.renumber();
  vdb_list_.start(NULL);

  return (n>0);

} // end of method del(1)
 

//************************************************************************************
//************************************************************************************
// METHOD     : del (2)
// DESCRIPTION: Deletes record(s) from DB, given key. If key==NULL, nothing is done.
//              DB list is set to start upon exit.
// ARGUMENTS  : key       : GKEY array; default is pointer to one element
//              num       : number of records to delete
// RETURNS    : TRUE if found and deleted; else FALSE 
//************************************************************************************
GBOOL VDB::del(GKEY *key, GINT  num)
{
  GINT       i, j;
  GBOOL      bRet=FALSE;
  VDBData    *m;

  if ( key == NULL ) return TRUE;

  for ( i=0; i<num; i++ ) {
    vdb_list_.start(NULL);
    for ( j=0; j<vdb_list_.size(); j++ ) {
      if ( (m=vdb_list_.member()) && (m->key() == key[i]) ) {
        vdb_list_.del(m);
        bRet = TRUE;
//      break;
      }
      vdb_list_.next();
    }
  }
     
  vdb_list_.renumber();
  vdb_list_.start(NULL);

  return bRet;

} // end of method del(2)
 

//************************************************************************************
//************************************************************************************
// METHOD     : size
// DESCRIPTION: Gets current number of records in DB
// ARGUMENTS  : none
// RETURNS    : no. of records
//************************************************************************************
GINT  VDB::size()
{

  return vdb_list_.size();

} // end of method size

 
//************************************************************************************
//************************************************************************************
// METHOD     : multiplicity (1)
// DESCRIPTION: Computes multiplicity of specified point in DB; i.e., the
//              number of records that point is in in DB. If point==NULL, this
//              is an error.
// ARGUMENTS  : point     : Point3D 
// RETURNS    : GINT  multiplicity value
//************************************************************************************
GINT  VDB::multiplicity(Point3D *point)
{
  GINT  j, mult=0;

  if ( point == NULL ) {
    cout << "VDB::multiplicity(1): record == NULL; invalid query" << endl;
    exit(1);
  }

  for ( j=0, mult=0; j<vdb_list_.size(); j++ ) {
    mult += ( vdb_list_[j].point() == *point ? 1 : 0 );
  }

  return mult;

} // end of method multiplicity (1)


//************************************************************************************
//************************************************************************************
// METHOD     : multiplicity (2) 
// DESCRIPTION: Computes multiplicity of specified key in DB; i.e., the
//              number of records that key is in in DB. If key==NULL, this
//              is an error.
// ARGUMENTS  : key   : GKEY key
// RETURNS    : GINT  multiplicity value
//************************************************************************************
GINT  VDB::multiplicity(GKEY *key)
{
  GINT  j, mult;
    
  if ( key == NULL ) {
    cout << "VDB::multiplicity(2): record == NULL; invalid query" << endl;
    exit(1);
  }

  for ( j=0, mult=0; j<vdb_list_.size(); j++ ) {
    mult += ( vdb_list_[j].key() == *key ? 1 : 0 );
  }

  return mult;

} // end of method multiplicity (2)



//************************************************************************************
//************************************************************************************
// METHOD     : multiplicity (3)
// DESCRIPTION: Computes multiplicity of each specified point in DB; i.e., the
//              number of records that each point is in in DB. If point==NULL, this
//              is an error.
// ARGUMENTS  : point     : Point3D array; default is pointer to one element
//              mult      : Multiplicity array, allocated here to be of length 'num'.
//                          Caller responsible for deletion.
//              num       : number of points specified
// RETURNS    : none.
//************************************************************************************
void VDB::multiplicity(Point3D *point, GINT  *&mult, GINT  num)
{
  GINT  i, j;

  if ( point == NULL ) {
    cout << "VDB::multiplicity(3): record == NULL; invalid query" << endl;
    exit(1);
  }

  if ( mult != NULL ) delete [] mult;
  mult = new GINT  [num];
  memset(mult,'0',num*sizeof(GINT ));

  for ( i=0; i<num; i++ ) {
    mult[i] = 0;
    for ( j=0; j<vdb_list_.size(); j++ ) {
      mult[i] += ( vdb_list_[j].point() == point[i] ? 1 : 0 );
    }
  }

} // end of method multiplicity (3)

 
//************************************************************************************
//************************************************************************************
// METHOD     : multiplicity (4)
// DESCRIPTION: Computes multiplicity of each specified key in DB; i.e., the
//              number of records that each key is in in DB. If key ==NULL, this
//              is an error.
// ARGUMENTS  : key       : host GKEY array; default is pointer to one element
//              mult      : Multiplicity array, allocated here to be of length 'num'.
//                          Caller responsible for deletion.
//              num       : number of keys specified
// RETURNS    : none.
//************************************************************************************
void VDB::multiplicity(GKEY *key, GINT  *&mult, GINT  num)
{
  GINT  i, j;

  if ( key == NULL ) {
    cout << "VDB::multiplicity(4): record == NULL; invalid query" << endl;
    exit(1);
  }

  if ( mult != NULL ) delete [] mult;
  mult = new GINT  [num];
  memset(mult,'0',num*sizeof(GINT ));

  for ( i=0; i<num; i++ ) {
    mult[i] = 0;
    for ( j=0; j<vdb_list_.size(); j++ ) {
      mult[i] += ( vdb_list_[j].key() == key[i] ? 1 : 0 );
    }
  }

} // end of method multiplicity (4)
 

//************************************************************************************
//************************************************************************************
// METHOD     : procs (1)
// DESCRIPTION: Finds proc list for specified points from DB. For each record
//              (point) found, there is a list of proc ids.
//              If point==NULL, this is considered an error. 
// ARGUMENTS  : point     : Point3D array; default is pointer to one element
//              iprocs    : Array of pointers to GSBuffer  's, allocated here 
//                          to be of length 'num'. Each member of array is a buffer
//                          containing the proc ids. Caller responsible for deletion.
//                          If iprocs is non-NULL on entry, it is an error.
//              num       : number of points specified
// RETURNS    : none.
//************************************************************************************
void VDB::procs(Point3D *point, GSBuffer   **&iprocs, GINT  num)
{
  GINT      i, j, n;
  GIBuffer  ieq(num);

  if ( point == NULL ) {
    cout << "VDB::procs(1): record == NULL; invalid query" << endl;
    exit(1);
  }
  if ( iprocs != NULL ) {
    cout << "VDB::procs(1): invalid buffer initialization" << endl;
    exit(1);
  }
  
  iprocs = new GSBuffer   * [num];
  if ( iprocs == NULL ) {
    cout << "VDB::procs(1): unable to allocate buffer" << endl;
    exit(1);
  }

  for ( i=0; i<num; i++ ) {  
    iprocs[i] = new GSBuffer   (0);
    for ( j=0, n=0; j<vdb_list_.size(); j++ ) {  // find no. common records, and indices
      if ( vdb_list_[j].point() == point[i] ) {
        ieq(n) = j;
        n ++;
      }
    }
    iprocs[i]->Resize(n);
    for ( j=0; j<ieq.dim(); j++ ) {              // get the proc ids for the common records
      (*iprocs[i])(j) = vdb_list_[j].proc();
    }
  }

} // end of method procs (1)

 
//************************************************************************************
//************************************************************************************
// METHOD     : procs (2)
// DESCRIPTION: Finds proc list for specified keys from DB. For each record
//              (key) found, there is a list of proc ids.
//              If key ==NULL, this is considered an error. 
// ARGUMENTS  : key       : host GKEY array; default is pointer to one element
//              iprocs    : Array of pointers to GSBuffer  's, allocated here 
//                          to be of length 'num'. Each member of array is a buffer
//                          containing the proc ids. Caller responsible for deletion.
//                          If iprocs is non-NULL on entry, it is an error.
//              num       : number of keys specified
// RETURNS    : none.
//************************************************************************************
void VDB::procs(GKEY *key, GSBuffer   **&iprocs, GINT  num)
{
  GINT      i, j, n;
  GIBuffer  ieq(num);

  if ( key == NULL ) {
    cout << "VDB::procs(2): record == NULL; invalid query" << endl;
    exit(1);
  }
  if ( iprocs != NULL ) {
    cout << "VDB::procs(2): invalid buffer initialization" << endl;
    exit(1);
  }
  
  iprocs = new GSBuffer   * [num];
  if ( iprocs == NULL ) {
    cout << "VDB::procs(2): unable to allocate buffer" << endl;
    exit(1);
  }

  for ( i=0; i<num; i++ ) {  
    iprocs[i] = new GSBuffer   (0);
    for ( j=0, n=0; j<vdb_list_.size(); j++ ) {  // find no. common records, and indices
      if ( vdb_list_[j].key() == key[i] ) {
        ieq(n) = j;
        n ++;
      }
    }
    iprocs[i]->Resize(n);
    for ( j=0; j<ieq.dim(); j++ ) {              // get the proc ids for the common records
      (*iprocs[i])(j) = vdb_list_[j].proc();
    }
  }

} // end of method procs (2)
 

//************************************************************************************
//************************************************************************************
// METHOD     : synch
// DESCRIPTION: Synchronizes DB on all processors
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void VDB::synch()
{
  GINT       i, k, max_rec, num=vdb_list_.size();
  GSHORT      j;
  Point3D    pt;
  GVDBst     *db, *gdb;

  // Remove all non-local data. If all procs do this,
  // then synching makes sense!
  CleanNonLocal();
  
  // Find max number of records:
  GComm::Allreduce(&num, &max_rec, 1, GC_GINT , G_OP_MAX); // get max record len

  if ( num == 0 ) return;                                  // nothing to synch 

  db     = new GVDBst [max_rec]; 
  gdb    = new GVDBst [max_rec*GComm::WorldSize()]; 

  // NOTE: we only need to synch the coordinates, localid, and host id 
  //       info, since we can compute the point's key locally....
  for ( i=0; i<num; i++) {
    for ( k=0; k<GDIM; k++ ) db[i].x    [k] = vdb_list_[i].point()[k];
    db[i].hostkey     =  vdb_list_[i].hostkey();
    db[i].hostrootkey =  vdb_list_[i].hostrootkey();
    db[i].hostid      =  vdb_list_[i].hostid ();
    db[i].id          =  vdb_list_[i].localid();
    db[i].ancillary   =  vdb_list_[i].ancillary();
  }
  // Set NULL records, if any:
  for ( i=num; i<max_rec; i++) {
    for ( k=0; k<GDIM; k++ ) db[i].x    [k] = 0.0;
    db[i].hostkey     =  NULL_HOSTKEY;
    db[i].hostrootkey =  NULL_HOSTKEY;
    db[i].hostid      =  NULL_HOSTKEY;
    db[i].id          =  NULL_HOSTKEY;
    db[i].ancillary   =  0;
  }

  // Gather all VDB data together:
  GComm::Allgather(db, max_rec, vdbst_type_, gdb, max_rec, vdbst_type_);

  // Now, add other procs' data to local DB by
  // extracting info by proc:
  for ( j=0; j<nprocs_; j++ ) {
    if ( j == this_rank_ ) continue;
    for ( i=j*max_rec; i<j*max_rec+max_rec; i++) {
      if ( gdb[i].hostkey == NULL_HOSTKEY ) break;  // don't add NULL records
      for ( k=0; k<GDIM; k++ ) pt  [k] = gdb[i].x    [k];
      add(&pt, NULL, &gdb[i].hostkey, &gdb[i].hostrootkey, 
          &gdb[i].id, &gdb[i].ancillary, &gdb[i].hostid, &j, 1);  // add to the DB 
    }
  }
  delete [] db;
  delete [] gdb;

} // end of method synch


//************************************************************************************
//************************************************************************************
// METHOD     : SetKeyGen
// DESCRIPTION: Sets key-generato object. If set, then this is
//              used to set the key in the VDBData, and it overrides
//              the value(s) set via the ::add method.
//              If key ==NULL, this is considered an error. 
// ARGUMENTS  : inkeygen  : pointer to non-NULL GKeyGen object.
// RETURNS    : none.
//************************************************************************************
void VDB::SetKeyGen(GKeyGen *inkeygen)
{
  keygen_ = inkeygen;
} // end of method SetKeyGen


//************************************************************************************
//************************************************************************************
// METHOD     : CleanAll
// DESCRIPTION: Removes all data from DB. 
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void VDB::CleanAll()
{

  vdb_list_.empty();

} // end of method CleanAll


//************************************************************************************
//************************************************************************************
// METHOD     : CleanNonLocal
// DESCRIPTION: Removes all non-local data from DB. Should best be used
//              prior to actually synching-up the DB. 
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void VDB::CleanNonLocal()
{
  GINT       i;

  for ( i=0; i<vdb_list_.size(); i++ ) {
    if ( vdb_list_[i].proc() != this_rank_ )  del(&vdb_list_[i].point(), 1);
  }

} // end of method CleanNonLocal


//************************************************************************************
//************************************************************************************
// METHOD     : record (1)
// DESCRIPTION: Gets individual record in DB by local index reference.
// ARGUMENTS  : GINT  irec: local index in [0, vdb_list.size()-1]
//
// RETURNS    : pointer to VDBData record; else NULL on failure to find it.
//************************************************************************************
VDBData *VDB::record(GINT  irec)
{
  if ( irec < 0 || irec >= vdb_list_.size() ) return NULL;
  if ( vdb_list_.size() <= 0                ) return NULL;

  return vdb_list_.member(irec);

} // end of record (1)


//************************************************************************************
//************************************************************************************
// METHOD     : record (2)
// DESCRIPTION: Gets all records in DB corresponding to the specified Point
// ARGUMENTS  : point  : pointer to the point of interest
//              vdbelems: array containing pointers to the VDBData records containing
//                        containing the 'point'. This array is allocated here, so the
//                        caller is responsible for deleting it.
//              num     : the number of records found containig 'point'. Should be the
//                        multiplicity.
//
// RETURNS    : pointer to first VDBData record; else NULL on failure to find it.
//************************************************************************************
VDBData *VDB::record(Point3D &point, VDBData **&vdbelems, GINT  &num)
{
  GINT  i, n;

  if ( vdbelems != NULL ) {
    delete [] vdbelems; vdbelems = NULL;
  }
  num = multiplicity(&point);
  if ( num <= 0 ) return NULL;

  vdbelems = new VDBData * [num];

  vdb_list_.start(NULL);
  for ( i=0,n=0; i<vdb_list_.size() && n<num;  i++ ) {
    if ( vdb_list_.member()->point() == point )  {
      vdbelems[n++] = vdb_list_.member();
    }
    vdb_list_.next();
  }
  if ( n != num ) {
    delete [] vdbelems; vdbelems = NULL;
    return NULL;
  }
  return vdbelems[0];
} // end of record (2)


//************************************************************************************
//************************************************************************************
// METHOD     : record (3)
// DESCRIPTION: Gets all records in DB corresponding to the specified GKEY
// ARGUMENTS  : key     : host GKEY of record interest
//              vdbelems: array containing pointers to the VDBData records containing
//                        containing the 'key'. This array is allocated here, so the
//                        caller is responsible for deleting it.
//              num     : the number of records found containig 'key'. Should be the
//                        multiplicity.
// 
// RETURNS    : pointer to first VDBData record; else NULL on failure to find it.
//************************************************************************************
VDBData *VDB::record(GKEY &key, VDBData **&vdbelems, GINT  &num)
{
  GINT  i, n;


  if ( vdbelems != NULL ) {
    delete [] vdbelems; vdbelems = NULL;
  }

  num = multiplicity(&key);
  if ( num <= 0 ) return NULL;

  vdbelems = new VDBData * [num];

  vdb_list_.start(NULL);
  for ( i=0,n=0; i<vdb_list_.size() && n<num; i++ ) {
    if ( vdb_list_.member()->key() == key ) { 
      vdbelems[n++] = vdb_list_.member();
    }
    vdb_list_.next();
  }
  if ( n != num ) {
    delete [] vdbelems; vdbelems = NULL;
    return NULL;
  }
  return vdbelems[0];
} // end of record (3)


//************************************************************************************
//************************************************************************************
// METHOD     : Bracket
// DESCRIPTION: set fuzziness in point position specifications
// ARGUMENTS  : eps: GDOUBLE magnitude
//
// RETURNS    :
//************************************************************************************
void VDB::Bracket(GDOUBLE eps)
{
  eps_ = eps;
} // end of Bracket


//************************************************************************************
//************************************************************************************
// METHOD     :  << operator method (1)
// DESCRIPTION: output stream operator. 
// ARGUMENTS  :
//
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, VDB &a)
{
  GINT    i=0;
  VDBData *m;

  // NOTE: because vdb_list_ is fully dynamic, we should not use
  //       the ::[] operator becuase it refers to quantities that
  //       may no longer exist!
  a.vdb_list_.start(NULL);
  while ( (m=a.vdb_list_.member()) != NULL && i < a.vdb_list_.size() ) {
    str << "(" << i << ") " << *m << endl;
    a.vdb_list_.next();
    i++;
  }
  a.vdb_list_.start(NULL);

  return str;
} // end of operator <<


//************************************************************************************
//************************************************************************************
// METHOD     : duplicates
// DESCRIPTION: gets indices for duplicate records searching by point
// ARGUMENTS  : point  : Point as query to DB
//              indices: array of record indices if there are duplicates. This
//                       is allocated here, so caller is responsible for deleting.
//              num    : number of duplicate records found
//
// RETURNS    : TRUE if there are duplicate records; else FALSE
//************************************************************************************
GBOOL VDB::duplicates(VDBData &rec, GINT  *&indices, GINT  &num)
{
  GINT    i;

  if ( indices != NULL ) delete [] indices; indices = NULL;
   
  for ( i=0, num=0; i<vdb_list_.size(); i++ ) {
    num += ( vdb_list_[i] == rec ) ? 1 : 0;
  }
  
  if ( num == 0 ) return FALSE;

  indices = new GINT  [num];
  for ( i=0, num=0; i<vdb_list_.size(); i++ ) {
    if ( vdb_list_[i] == rec ) indices[num++] = i; 
  }

  return TRUE;

} // end of duplicates 



