//======================================================================================
// Name         : gaspar_restart.cpp
// Date         : 11/10/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : GASpAR dump/restart module. 
// Modifications:
//======================================================================================
#include "gaspar_t.h"

#include "guser.h"


//************************************************************************************
//************************************************************************************
// METHOD     : GRestart
// DESCRIPTION: Sets GASpAR variables from a restart file. If bRegrid_==TRUE,
//              Restart occurs with a new mesh provided by interpolating from old mesh
//              in restart file to new mesh contained in uelems data structure, which
//              is pre-defined prior to entry. If bRegrid_==FALSE, then the grid 
//              contained in the restart file is used.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GRestart()
{
  char       *serr = "GRestart: ";
  GINT       i, j, k, num, ni, nf, nt;
  GBOOL      bRet;
  char       slabel[FILE_NAME_MAX+1];
  GDBuffer   tag;
  Elem2D     *elem, *pelem;
  Field2D    *fu, *fp;
  GLLBasis   *bgll[3]={NULL,NULL,NULL};
  GLBasis    *bgl [3]={NULL,NULL,NULL};
  GNBasis    **bg=NULL;
  GBinReader rgbin;
  
  if ( !rgbin.Open(fnrst_) ) {
    cout << serr << "Cannot open file named " << fnrst_
         << " Reader error: " << rgbin.Error() << endl;
    exit(1);
  }

  // Set other system quantities:
  bRet = GSetSystemVar(rgbin, num);

#if 0
  if ( !bRegrid_ ) {
    // Empty lists:
    uelems.empty();
    for ( k=0; k<nEvolvedFields_; k++ ) {
      pgfields_[k]->empty();
    }
    bRet = GSetGridFromFile(rgbin); // nelems_ set here
    if ( !bRet ) { 
      cout << serr << "GSetGridFromFile failed" << endl;
      exit(1);
    }
  }
#endif

  // Set field quantities:
  for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->start(NULL);
  for ( i=0; i<nelems_ && bRet; i++ ) {
    elem  = uelems[i];
    for ( k=0; k<nEvolvedFields_ && bRet; k++ ) {
      switch ( iBasisTypes_[k] ) {
        case GBASIS_GLL: 
          for ( j=0; j<GDIM; j++ ) if ( bgll[j] == NULL )  bgll[j] = new GLLBasis();
          bg = (GNBasis**)bgll;
          break;
        case GBASIS_GL: 
          for ( j=0; j<GDIM; j++ ) if ( bgl[j] == NULL )  bgl[j] = new GLBasis();
          bg = (GNBasis**)bgl;
          break;
      }
#if 0
      if ( !bRegrid_ ) pgfields_[k]->add(ntimelevels_, elem);
#endif
      fu = (*pgfields_[k])[i];
      // If maintining multiple time levels, set them:
      for ( j=0; j<fu->GetNumTimeLevels() && bRet; j++ ) {
        if ( j==0 ) sprintf(slabel, "%s", sDSLabel_[k]);
        else        sprintf(slabel, "%s%s%d", sDSLabel_[k], sDSLabel_suff, j);
        bRet = bRet && GSetFieldVar(rgbin, slabel, bg, fu, j, nf, ni);  
        bRet = bRet && (nf <= ni );
      }
    } // end of field-index loop
    for ( k=0; k<nEvolvedFields_ && bRet; k++ ) pgfields_[k]->next();
  } // end of element-index loop

#if 0
  if ( bLinAdvection_ ) { // cadv_ must be specified in guser file; not stored in dump file
    for ( i=0; i<nelems_ && bRet; i++ ) {
      elem  = uelems[i];
      for ( k=0; k<GDIM; k++ ) {
        if ( !bRegrid_ ) cadv_[k]->add(ntimelevels_,uelems[i],ntmplevels_);
      }
    } // end of element-index loop
  }
#endif

  if ( !bRet ) {
    cout << serr << "GSetFieldVar failed: i_element: " << i << " nf=" << nf << " ni=" << ni << endl;
  }

  rgbin.Close();

  return bRet;

} // end of method GRestart


//************************************************************************************
//************************************************************************************
// METHOD     : GSetFieldVar
// DESCRIPTION: Sets GASpAR field variables from restart file
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GSetFieldVar(GBinReader &rgbin, char *dslabel, GNBasis *b[],  Field2D *field, 
                  GINT  iLevel, GINT  &nfldnodes, GINT  &ninterp)
{
  char      *serr = "GSetFieldVar: ";
  GINT      i, j, n, NN, nds, nt, rank, *idims;
  GBOOL     bEqualRegions;
  GDOUBLE   *tag;
  ELEMTYPE  etype;
  Point3D   *fldvert, *vert;
  GVector    u(1), *ui, *unew, *x[3], *xi[3];
  GIBuffer  *ii;
 
  fldvert = field->GetElement()->GetSpVertices();
 
  nfldnodes = 0;
  ninterp   = 0;

  ui = new GVector(1);
  ii = new GIBuffer  (1);
  for ( i=0; i<3; i++ ) {
    xi[i] = new GVector(1);
    x [i] = new GVector(1);
  }
 
  nds     = rgbin.GetNumDataSets();
  for ( n=0; n<nds; n++ ) {
    if ( strcmp(dslabel,rgbin.GetLabel(n)) != 0  ) continue;
    rank    = rgbin.GetRank(n);
    if ( rank != nd_ || rank != field->GetElement()->Dim() ) {
      cout << serr << "Incompatible data rank" << endl;
      return FALSE;
    }
    vert    = rgbin.GetVert(n);
    if ( !GbOverlap(fldvert, vert, nvert_, bEqualRegions) )  continue;
    idims   = rgbin.GetDims(n);
    etype   = rgbin.GetElemTypes(n);
    for ( j=0,NN=1; j<rank; j++ ) NN *= idims[j];
    for ( j=0; j<rank; j++ ) {
      b[j]->SetOrder(idims[j]-1);
      b[j]->Solve();
    }
#if 0
    for ( j=0; j<rank; j++ ) {
      x[j]->Resize(rgbin.GetCoordDims(n,j+1));
      if ( rgbin.GetGridData(n, j+1, x[j]->Data(), x[j]->dim() )==NULL ) {
        cout << serr << "Cannot retrieve " << j+1 << "-grid data from file " << fnrst_
             << " Reader error: " << rgbin.Error() << endl;
       return FALSE;
      }
    }
#endif
    // Find indices of enclosing patch of new grid; interpolate
    // new points onto restart data element:
    if ( !GCollectInterpPts(rank, field->GetElement(), nvert_, vert, b, xi, ii) ) {
      cout << serr << "Could not assemble interpolation points" << endl;
      exit(1);
    }
#if 0
cout << "SetFldVar: " << " num overlapping pts:" << ii->dim() << endl;
#endif
    if ( ii->dim() == 0 ) continue;

    ui->Resize(ii->dim());
    u.Resize(NN);
    nt  = rgbin.GetNumTags(n);
    tag = rgbin.GetTags(n);
    field->SetTime(iLevel,tag[nt-1]);  // dump field time is _always last_
    if ( rgbin.GetFieldData(n, u.Data(), u.dim() )==NULL ) {
      cout << serr << "Cannot retrieve field data from file " << fnrst_
           << " Reader error: " << rgbin.Error() << endl;
      exit(1);
    }

    // if all field points are enclosed by this dataset, then
    // simply assign the values to the new exp. coeffs:
//cout << "bEaualRegions=" << bEqualRegions << endl;
    if ( field->GetExpCoeffs(iLevel)->dim() == ii->dim()   
      &&                          ii->dim() == u.dim  () 
      &&                                 bEqualRegions ) {
      unew = field->GetExpCoeffs(iLevel);
      *unew = u;
      ninterp   = u.dim();
      break;
    }

#if 0
  cout << "GSetFieldVar: ii_dim=" << ii->dim() << endl;
for ( j=0; j<ii->dim(); j++ ) {
  cout << "GSetFieldVar: ii[" << j << "]= " << (*ii)(j) << endl;
}
#endif

#if 0
cout << "GSetFieldVar: ui[" << n << "] = " << *ui    << endl;
cout << "GSetFieldVar: xi[1][" << n << "] = " << *xi[1]    << endl;
cout << "GSetFieldVar: ui_dim=" << ui->dim() << endl << endl;
#endif
    if ( !GInterpData(rank, b, u, xi, ui) ) {
      cout << serr << "Could not interpolate points" << endl;
      exit(1);
    }

    unew = field->GetExpCoeffs(iLevel);
    for ( j=0; j<ii->dim(); j++ ) {
      (*unew)((*ii)(j)) = (*ui)(j); 
    }
    ninterp   += ii->dim();
#if 0
if ( u.dim() == ui->dim() ) {
cout << "GSetFieldVar: unew=" << *unew    << endl;
cout << "GSetFieldVar: u[" << n << "] = " << u    << endl;
cout << "GSetFieldVar: ui[" << n << "] = " << *ui    << endl;
cout << "GSetFieldVar: ii=" << *ii << endl;
}
#endif
  }

  nfldnodes  = ( (field->GetElement()->GetOrder(1)+1)*(field->GetElement()->GetOrder(2)+1 ) );
#if 0
cout << "GSetFieldVar: cleaning up........................." << endl;
#endif
  if ( ui ) delete ui;
  if ( ii ) delete ii;
  for ( i=0; i<3; i++ ) {
    if ( xi[i] ) delete xi[i];
    if ( x [i] ) delete x [i];
  }

  return TRUE;
} // end of method GSetFieldVar


//************************************************************************************
//************************************************************************************
// METHOD     : GSetVarDataB
// DESCRIPTION: Sets GASpAR variable data from restart file. This method
//              accepts a buffer whose length can be altered to accommodate
//              the dimensions of the stored data.
// ARGUMENTS  : rgbin  : GBin reader object
//              dslabel: dataset label
//              data   : dataset buffer
// RETURNS    :
//************************************************************************************
GBOOL GSetVarDataB(GBinReader &rgbin, char *dslabel, GDBuffer &data)
{
  char      *serr = "GSetVarData: ";
  GINT      j, n, NN, nds, rank, *idims;
 
  nds     = rgbin.GetNumDataSets();
  for ( n=0; n<nds; n++ ) {
    if ( strcmp(dslabel,rgbin.GetLabel(n)) != 0 ) continue;
    rank    = rgbin.GetRank(n);
    idims   = rgbin.GetDims(n);
    for ( j=0,NN=1; j<rank; j++ ) NN *= idims[j];
    data.Resize(NN);
    if ( rgbin.GetFieldData(n, data.Data(), data.dim())==NULL ) {
      cout << serr << "Cannot retrieve field data from file " << fnrst_
           << " Reader error: " << rgbin.Error() << endl;
      exit(1);
    }
  }

  return TRUE;
} // end of method GSetVarDataB


//************************************************************************************
//************************************************************************************
// METHOD     : GSetVarDataA
// DESCRIPTION: Sets GASpAR variable data from restart file. This method
//              requires data buffer to previously allocated of fixed length.
// ARGUMENTS  : rgbin  : GBin reader object
//              dslabel: dataset label
//              data   : dataset data already allocated of length ndata
//              ndata  : length of 'data' pointer
// RETURNS    :
//************************************************************************************
GBOOL GSetVarDataA(GBinReader &rgbin, char *dslabel, GDOUBLE *data, GINT ndata)
{
  char      *serr = "GSetVarData: ";
  GINT      j, n, NN, nds, rank, *idims;
 
  nds     = rgbin.GetNumDataSets();
  for ( n=0; n<nds; n++ ) {
    if ( strcmp(dslabel,rgbin.GetLabel(n)) != 0 ) continue;
    rank    = rgbin.GetRank(n);
    idims   = rgbin.GetDims(n);
    for ( j=0,NN=1; j<rank; j++ ) NN *= idims[j];
    if ( ndata < NN ) {
      cout << serr << "Insufficient allocation for data" << endl;
      exit(1);
    }
    if ( rgbin.GetFieldData(n, data, ndata)==NULL ) {
      cout << serr << "Cannot retrieve field data from file " << fnrst_
           << " Reader error: " << rgbin.Error() << endl;
      exit(1);
    }
  }

  return TRUE;
} // end of method GSetVarDataA



//************************************************************************************
//************************************************************************************
// METHOD     : GSetSystemVar
// DESCRIPTION: Sets GASpAR field variables from restart file.
//              NOTE: this method and the GDump method are
//                    inherently related. If changes are made in
//                    one place, make sure they are consistent in
//                    both.
// ARGUMENTS  : rgbin: GBinReader object currently reading data
//              num  : returns with the total number of system variables that 
//                     changed. 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GSetSystemVar(GBinReader &rgbin, GINT  &num)
{
  GINT  i, j, n, nm;
  GDOUBLE *fmeta=NULL;
  
  nm    = rgbin.GetNumMeta();
  fmeta = rgbin.GetMeta   ();
  num   = 0;

  if ( nm < ndmpmeta_ ) return TRUE;
  
  
  icycle_           = (GINT )       fmeta [0] ; 
  icycle_max_       = (GINT )       fmeta [1] ; 
  icycle_out_beg_   = (GINT )       fmeta [2] ; 
  icycle_out_end_   = (GINT )       fmeta [3] ; 
  icycle_out_skip_  = (GINT )       fmeta [4] ;
  icycle_out_last_  = (GINT )       fmeta [5] ; 
  icycle_dmp_beg_   = (GINT )       fmeta [6] ; 
  icycle_dmp_skip_  = (GINT )       fmeta [7] ; 
  icycle_log_skip_  = (GINT )       fmeta [8] ; 
  dt_               = (GDOUBLE)     fmeta [9] ; 
  time_             = (GDOUBLE)     fmeta[10] ; 
  time_max_         = (GDOUBLE)     fmeta[11] ; 
  time_out_beg_     = (GDOUBLE)     fmeta[12] ; 
  time_out_end_     = (GDOUBLE)     fmeta[13] ; 
  time_out_skip_    = (GDOUBLE)     fmeta[14] ; 
  time_out_last_    = (GDOUBLE)     fmeta[15] ; 
  iStopCond_        = (STOPCONDTYPE)fmeta[16] ; 
  iOutType_         = (OUTPUTTYPE)  fmeta[17] ; 
  bUPC              = (GBOOL)       fmeta[18] ; 
  bPPC              = (GBOOL)       fmeta[19] ; 
  upc_type_         = (GPC)         fmeta[20] ; 
  stokes_type_      = (STOKES_TYPE) fmeta[21] ; 
  nu_[0]            = (GDOUBLE)     fmeta[22] ;
  nu_[1]            = (GDOUBLE)     fmeta[23] ;
  rho_              = (GDOUBLE)     fmeta[24] ;
  ntimelevels_      = (GINT )       fmeta[25] ;
  ngVertices_       = (GINT )       fmeta[26] ;
  bFixedTimeStep_   = (GBOOL)       fmeta[27] ;
  iEvolType_        = (TIME_EVOLTYPE)fmeta[28] ;
  iorderadv_        = (GINT )       fmeta[29] ;
  iorderBDF_        = (GINT )       fmeta[30] ;
  iorderAM_         = (GINT )       fmeta[31] ;
  bDoAdapt_         = (GBOOL)       fmeta[32] ;
  ap_tol_           = (GDOUBLE)     fmeta[33] ;
  ap_mult_          = (GDOUBLE)     fmeta[34] ;

  num = ndmpmeta_ + ngVertices_*GDIM;
  for ( i=0; i<ngVertices_; i++ ) {
    btype_[i] = (BDYTYPE)fmeta[num+i] ;
  } 
  num += ngVertices_;
  if ( !GSetVarDataB(rgbin, sTHLabel_, dthist_) ) {
    cout << "GSetSystemVar: unable to retrieve time history buffer" << endl;
    exit(1); 
  }
  num += dthist_.dim();

  if ( !GSetVarDataB(rgbin, sPBLabel_, fgPeriodic_ ) ) {
    cout << "GSetSystemVar: unable to periodicity buffer" << endl;
//  return FALSE; 
  }
  igPeriodic_.Resize(fgPeriodic_.dim());
  for ( j=0; j<fgPeriodic_.dim(); j++ ) igPeriodic_[j] = (GINT)fgPeriodic_[j];
  num += igPeriodic_.dim();

  if ( !GSetVarDataB(rgbin, sSpNLabel_, apunorm_) ) {
    cout << "GSetSystemVar: unable to retrieve spectral norm buffer" << endl;
    exit(1); 
  }
  num += apunorm_.dim();
  if ( !GSetVarDataB(rgbin, sDerivNLabel_, dunorm_) ) {
    cout << "GSetSystemVar: unable to retrieve derivative norm buffer" << endl;
    exit(1); 
  }
  num += dunorm_.dim();

#if 0
  if ( pgfields_  != NULL ) delete [] pgfields_;
  if ( pgbdyvals_ != NULL ) delete [] pgbdyvals_;
  pgfields_   = new GFieldList * [nEvolvedFields_];
  pgbdyvals_  = new GVecList   * [nEvolvedFields_];
  if      ( nEvolvedFields_ == 1 ) {
    pgfields_ [0] = &u1;  
    pgbdyvals_[0] = &u1bdyvals;
  }
  else if ( nEvolvedFields_ == 2 ) {
    pgfields_ [0] = &u1;
    pgfields_ [1] = &u2;
    pgbdyvals_[0] = &u1bdyvals;
    pgbdyvals_[1] = &u2bdyvals;
  }
  else {  
    pgfields_ [0] = &u1;
    pgfields_ [1] = &u2;
    pgfields_ [2] = &u3;
    pgbdyvals_[0] = &u1bdyvals;
    pgbdyvals_[1] = &u2bdyvals;
    pgbdyvals_[2] = &u3bdyvals;
  }
#endif


  return TRUE;
} // end of method GSetSystemVar


//************************************************************************************
//************************************************************************************
// METHOD     : GInterpData
// DESCRIPTION:   Uses interpolation operators of bases evaluated at
//                desired nodes, together with tensor product multiplication
//                to find values of u interpolated at the new xi.
// ARGUMENTS  :
//              rank   : System rank. Also dimensions of arrays:
//              b[]    : GNBasis * array, of length rank.
//              u      : GVector of interpolants. Must be of length
//                       (b[0].Order+1)*(b[1].Order+1)*(b[2].Order+1)*...
//              xi[]   : GVector * array of length rank
//              ui     : GVector of interpolated vals. Will be of length
//                        xi[0].dim * xi[1].dim * ...
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GInterpData(GINT  rank, GNBasis *b[],  GVector &u, GVector *xi[], GVector *ui)
{
  char      *serr = "GInterpData: ";
  GINT      i, j, k, m, NN, nb[3]={0};
  GBOOL     bOK=TRUE;
  GDOUBLE     yi;
  GVector   *utmp;
  GMatrix   *I[3]={NULL};

  if ( rank < 1 || rank > 3 ) return FALSE;
  NN = xi[0]->dim();
  for ( j=0; j<rank && bOK; j++ ) {
    nb[j] = b[j]->GetOrder() + 1;
    I[j] = new GMatrix(xi[j]->dim(), nb[j]);
    bOK = bOK && ( xi[j]->dim() == NN );
    bOK = bOK && I[j] != NULL;
    bOK = bOK && b[j]->EvalBasis(xi[j],I[j]);
//cout << "GInterpData: I[" << j << "]=" << *I[j] << endl;
  }
  if ( !bOK ) {
    cout << serr << "Unable to create interpolation matrices." << endl;
    exit(1);
  }

  ui->Resize(NN);
  *ui = 0.0;
  if ( rank == 1 ) {
    MTK::fmatvec_prod(*I[0], u, *ui);
    for ( j=0; j<rank; j++ ) 
      if ( I[j] ) delete I[j];
  }
  else if ( rank == 2 ) {
    for ( k=0; k<NN; k++ ) {
      for ( j=0; j<nb[1]; j++ ) {
        yi = (*I[1])(k,j);
        for ( i=0; i<nb[0]; i++ ) {
          m = i + nb[0]*j;
          (*ui)(k) += (yi*(*I[0])(k,i)*u(m));
        }
      }
    }
    for ( j=0; j<rank; j++ ) 
      if ( I[j] ) delete I[j];
  }
  else if ( rank == 3 ) {
//  MTK::D3_X_D2_X_D1(*I[0], *I[1], *I[2],  u, *ui);
    cout << serr << "There is only support for Rank=2" << endl;
    return FALSE;
  }
  
  return TRUE;

} // end of method GInterpData


//************************************************************************************
//************************************************************************************
// METHOD     : GCollectInterpPts
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GCollectInterpPts(GINT  rank, Elem2D *e, GINT  nvert, Point3D *vert, 
                       GNBasis *b[], GVector *xi[], GIBuffer  *ii) 
{
  char      *serr = "GCollectInterpPts: ";
  GINT      i, j, kin, m, NN, igdims[3]={0};
  GDOUBLE     L, x, y, z;
  GBOOL     bRet=TRUE;
  GVector  *xtmp[3]={NULL}, *xnew[3]={NULL};
  GVector  *xitmp[3]={NULL};
  GIBuffer  *itmp=NULL;

  if ( rank < 1 || rank > 2 ) {
    cout << serr << "Invalid rank. Rank=3 unsupported" << endl;
    return FALSE;
  }

  for ( i=0,NN=1; i<rank; i++ ) {
    igdims[i] = e->GetOrder(i+1) + 1;
    NN *= igdims[i];
    xnew  [i] = e->GetSpNodes(i+1);
  }

  for ( i=0; i<rank; i++ ) {
    xtmp [i] = new GVector(NN);
    xitmp[i] = new GVector(NN);
  }
  itmp     = new GIBuffer  (NN);

  if ( rank == 1 ) {
    kin = 0;
    for ( i=0; i<igdims[0]; i++ ) {
      x = (*xnew[0])(i);
      if ( x >= vert[0].x1 && x <= vert[1].x1 ) {
        L = fabs(vert[0].x1 - vert[1].x1);
        (*xtmp [0])(kin) = x;
        (*xitmp[0])(kin) = (2.0*x - vert[0].x1 - vert[1].x1 ) / L;
        (*itmp)(kin)     = i;
        kin++;
      }
    }
  }
  else if ( rank == 2 ) {
    kin = 0;
    for ( j=0; j<igdims[1]; j++ ) {
      for ( i=0; i<igdims[0]; i++ ) {
        m = i + igdims[0]*j;
        x = (*xnew[0])(m);
        y = (*xnew[1])(m);
        if ( GPoint_in_region(nvert, vert, x, y) ) {
          (*xtmp[0])(kin) = x;
          (*xtmp[1])(kin) = y;
          (*itmp)   (kin) = m;
          kin++;
        }
#if 0
else {
cout << "Collect point: " << x << ", " << y << " not in polygon:" ;
for ( m=0; m<nvert; m++ ) {
  cout << "(" << vert[m].x1 << ", " << vert[m].x2 << "); ";
}
cout << endl;

}
#endif
      }
    }

    for ( j=0; j<rank; j++ ) {
      xi[j]->Resize(kin);
    }

    ii->Resize(kin);
    memcpy(ii->Data(), itmp->Data(), kin*sizeof(GINT ));

#if 0
cout << " GCollectInterpPts: Vertices: " << endl;
for ( m=0; m<nvert; m++ ) {
  cout << "(" << vert[m].x1 << ", " << vert[m].x2 << "); ";
}
cout << endl;
#endif

    bRet = GXToXi(nvert, vert, *b[0], *b[1], kin, xtmp[0]->Data(), xtmp[1]->Data(),
                xi[0]->Data(), xi[1]->Data());
  }
#if 0
  for ( j=0; j<xi[0]->dim(); j++ ) {
    cout << "Collect: x, y= " << (*xtmp[0])(j) << " " << (*xtmp[1])(j) << " xi_1, xi_2= " <<
                                 (*xi  [0])(j) << " " << (*xi  [1])(j) <<
                                  endl;
  }
#endif
  
  for ( j=0; j<rank; j++ ) {
    if ( xtmp [j] ) delete xtmp [j];
    if ( xitmp[j] ) delete xitmp[j];
  }
  if ( itmp ) delete itmp;

  return bRet;

} // end of method GCollectInterpPts


//************************************************************************************
//************************************************************************************
// METHOD     : GPoint_in_region
// DESCRIPTION: Determines if input point, (x, y), lies within
//              the region specified by the vertices, spVertices.
// ARGUMENTS  :
// RETURNS    : TRUE if point is in the polygon; else FALSE
GBOOL GPoint_in_region(GINT  nVertices, Point3D *spVertices, GDOUBLE x, GDOUBLE y)
{
  GINT  n, np1;
  GDOUBLE dotProd;
  GBOOL bIn;
  Point3D Norm, dR;

  // Check boundaries:
  
  for ( n=0, bIn=TRUE; n<nVertices && bIn; n++ )
  {
      np1 = (n + 1) % nVertices;

      // Assume vertices in counterclockwise order, and
      // compute normal to edge segment n,np1. This vector
      // points 'inward' in a r.h. sense:
      Norm.x1 = -(spVertices[np1].x2 - spVertices[n].x2);
      Norm.x2 =  (spVertices[np1].x1 - spVertices[n].x1);

      // Compute vector dR =  R_test - R_VertexPoint:
      dR.x1   = x - spVertices[n].x1;
      dR.x2   = y - spVertices[n].x2;

      // If dR dot Norm < 0, then point is outside of polygon:
      dotProd = dR.x1*Norm.x1 + dR.x2*Norm.x2;
      bIn = (dotProd >= 0.0);
  }

  return bIn;
} // end of method GPoint_in_region (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GXToXi2d()
// DESCRIPTION: converts the real space values into parent-domain values
// ARGUMENTS  : Point *x = pointer to real space points;
//              Point *xi= pointer to parent domain points
//              GINT  num= number of points to invert
//
//              NOTE: there will be problems if the number of
//              points in each array is not the same.
// RETURNS    : TRUE on success; FALSE on failure if the
//              solution is imaginary, or undefined, or if input
//              point lies outside of element.
GBOOL GXToXi(GINT  nVertices, Point3D spVertices[], GNBasis &basis1, GNBasis &basis2,
            GINT  num, GDOUBLE *ax1, GDOUBLE *ax2, GDOUBLE *xi1, GDOUBLE *xi2)
{

  GINT  i;
  GDOUBLE a[2], b[2], c[2], d[2], A, B, C, ch;
  GDOUBLE x1, x2, del0, del1, ximin, ximax, xip, xim, xi1tmp, xi2tmp;

  if ( nVertices < 4 ) return FALSE;

  ximin = basis1.GetXimin()<basis2.GetXimin()?basis1.GetXimin():basis2.GetXimin();
  ximax = basis1.GetXimax()>basis2.GetXimax()?basis1.GetXimax():basis2.GetXimax();


  a[0] = spVertices[0].x1 - spVertices[1].x1 + spVertices[2].x1 - spVertices[3].x1;
  a[1] = spVertices[0].x2 - spVertices[1].x2 + spVertices[2].x2 - spVertices[3].x2;

  b[0] =-spVertices[0].x1 + spVertices[1].x1 + spVertices[2].x1 - spVertices[3].x1;
  b[1] =-spVertices[0].x2 + spVertices[1].x2 + spVertices[2].x2 - spVertices[3].x2;

  c[0] =-spVertices[0].x1 - spVertices[1].x1 + spVertices[2].x1 + spVertices[3].x1;
  c[1] =-spVertices[0].x2 - spVertices[1].x2 + spVertices[2].x2 + spVertices[3].x2;

  d[0] = spVertices[0].x1 + spVertices[1].x1 + spVertices[2].x1 + spVertices[3].x1;
  d[1] = spVertices[0].x2 + spVertices[1].x2 + spVertices[2].x2 + spVertices[3].x2;

  for ( i=0; i<num; i++ )
  {
    x1   = ax1[i];
    x2   = ax2[i];
    A    = a[0]*c[1] - a[1]*c[0];
    del0 = 4.0*x1-d[0];
    del1 = 4.0*x2-d[1];
    B    = a[1]*del0 - a[0]*del1 + a[0]*d[1]-a[1]*d[0] + b[0]*c[1]-b[1]*c[0];
    C    =-b[0]*del1 + b[1]*del0;

    if ( a[0] == 0.0 && a[1] == 0.0 ) {
      xi2tmp = ( del1 - (b[1]/(b[0]+TINY)) * del0 ) / ( c[1] - c[0]*(b[1]/(b[0]+TINY)) ); 
      xi1tmp = ( del0 - c[0]*xi2tmp ) / ( b[0] + TINY);
    }
    // xi_2 is solution to quadratic equation:
    //  A xi^2 + B xi + C = 0:
    else if ( A > TINY ) // quadratic solution
    {
      ch   = B*B - 4.0*A*C;

      if ( ch < 0 )
      {
        cout << "GXToXi(1): xi_2 not real!" << endl;
        return FALSE;
      }

      xip   = ( -B + sqrt(ch) )  / ( 2.0*A);
      xim   = ( -B - sqrt(ch) )  / ( 2.0*A);
      if ( (xip > ximax || xip < ximin) &&
           (xim > ximax || xim < ximin) )
      {
        cout << "GXToXi(1): xi_2 out of range" << endl;
        return FALSE;
      }
      if      ( xip > ximax || xip < ximin ) xi2tmp = xim;
      else if ( xim > ximax || xim < ximin ) xi2tmp = xip;

      xi1tmp = ( 4.0*x1 - c[0]*xi2tmp - d[0] ) / ( a[0]*xi2tmp + b[0] );
    }
    else  // linear solution (A=0)
    {
      if ( fabs(B) < TINY )
      {
        cout << "GXToXi(1): xi_2 singular!" << endl;
        return FALSE;
      }
      xi2tmp = -C/B;
      xi1tmp = ( 4.0*x1 - c[0]*xi2tmp - d[0] ) / ( a[0]*(xi2tmp) + b[0] );
    }

#if 0
    if ( xi1tmp > ximax ) xi1tmp == ximax;
    if ( xi1tmp < ximin ) xi1tmp == ximin;
    if ( xi2tmp > ximax ) xi2tmp == ximax;
    if ( xi2tmp < ximin ) xi2tmp == ximin;
#endif
//  xi1tmp = MAX(ximin,MIN(ximax,xi1tmp));
//  xi2tmp = MAX(ximin,MIN(ximax,xi2tmp));
    if ( xi2tmp > ximax || xi2tmp < ximin )
    {
      if ( xi2tmp > ximax ) cout << "xi2tmp > ximax" << endl;
      if ( xi2tmp < ximin ) cout << "xi2tmp < ximin" << endl;
      cout << "GXToXi(1): Quad:  xi_2 out of range! (x1, x2) = (" << x1 <<  ", " << x2 
      << ");  (xi_1,xi_2)= (" << xi1tmp << "," << xi2tmp << "); xi/ximax= (" <<
      xi1tmp/ximax << "," << xi2tmp/ximax << ")"  << endl;
      return FALSE;
    }
    if ( xi1tmp > ximax || xi1tmp < ximin )
    {
      cout << "GXToXi(1): Quad:  xi_1 out of range! (x1, x2) = (" << x1 << ", " << x2 
      << ");  (xi_1,xi_2)= (" << xi1tmp << "," << xi2tmp << "); xi/ximax= ("  <<
      xi1tmp/ximax << "," << xi2tmp/ximax << ")"  << endl;
      return FALSE;
    }
    xi1[i] = xi1tmp;
    xi2[i] = xi2tmp;
  }

  return TRUE;
} // end of method GXToXi


//************************************************************************************
//************************************************************************************
// METHOD     : GbOverlap
// DESCRIPTION: determines of there is overlap between two regions as defined
//              by their vertex lists
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GbOverlap(Point3D *v1, Point3D *v2, GINT  nvert, GBOOL &bEqual)
{
  GINT  i;
  GBOOL bRet=FALSE;


  bEqual = FALSE;

  if ( nvert < nvert_) return FALSE;

  if      ( nd_ == 1 ) {
    bRet   = v2[0].x1 <= v1[1].x1;  
    bEqual = v2[0].x1 == v1[1].x1;  
  }
  else if ( nd_ == 2 ) {
    for ( i=0, bRet=FALSE,bEqual=TRUE; i<nvert; i++ ) {
      bRet   = bRet   || GPoint_in_region(nvert, v1, v2[i].x1, v2[i].x2);
      bEqual = bEqual && ( v1[i].x1 == v2[i].x1 ) 
                      && ( v1[i].x2 == v2[i].x2 );
    }
  }
  else if ( nd_ == 3 ) {
    cout << "GbOverlap: 3D not supported" << endl;
    exit(1);
  }
  return bRet;
} // end of method GbOverlap


//************************************************************************************
//************************************************************************************
// METHOD     : GDump
// DESCRIPTION: Creates or overwrites a dumpfile. 
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GDump(const char *fn, GBOOL deleteFirst)
{
  char       *serr = "GDump: ";
  GINT       i, j, k, n, N[3], NN, nm, nt=2, iret;
  GDOUBLE    *fmeta=NULL, T0, T1 ;
  GBOOL      bRet;
  char       slabel[FILE_NAME_MAX], cmdstr[CMD_STR_MAX];
  Point      *vert;
  GVector    *uu, *ub;
  GDBuffer   tag;
  Elem2D     *e;
#if defined(MPI_IO_DEFAULT)
  GBinWriter wgbin(TRUE,TRUE,GComm::WorldRank()/NPROCESSES_PER_NODE); // Collective, independent writes
#else
  GBinWriter wgbin;
#endif

  T0 = STK::CTimer(); T1 = STK::Timer (); 
  GComm::Synch();
  sprintf(cmdstr,"cp -f %s %s%%", fn, fn); 
  if ( rank == 0  && ( (iret=system(cmdstr)) == -1 || iret == 127 ) ) {
    cout << serr << "Could not copy dump file, " << fn << endl;
  }
  GComm::Synch();

  if ( deleteFirst ) {           // not considered to be a normal output file
    icycle_dmp_last_ = icycle_;
  }

  if ( !wgbin.Open(fn,gios::inout|gios::create|gios::trunc) ) {
     cout << serr << "Open failed for file " << fn << 
             "; Writer error: " << wgbin.Error() << endl;
     iGError_ = GERR_DUMP;
     GDoLog(0,"#GDump failure: open failure");
     return FALSE;
  }

  // Create and set meta data:
  fmeta = new GDOUBLE [ndmpmeta_+ngVertices_*(GDIM+1)];

  fmeta [0] = (GDOUBLE)icycle_;
  fmeta [1] = (GDOUBLE)icycle_max_;
  fmeta [2] = (GDOUBLE)icycle_out_beg_;
  fmeta [3] = (GDOUBLE)icycle_out_end_;
  fmeta [4] = (GDOUBLE)icycle_out_skip_;
  fmeta [5] = (GDOUBLE)icycle_out_last_;
  fmeta [6] = (GDOUBLE)icycle_dmp_beg_;
  fmeta [7] = (GDOUBLE)icycle_dmp_skip_;
  fmeta [8] = (GDOUBLE)icycle_log_skip_;
  fmeta [9] = (GDOUBLE)dt_;
  fmeta[10] = (GDOUBLE)time_;
  fmeta[11] = (GDOUBLE)time_max_;
  fmeta[12] = (GDOUBLE)time_out_beg_;
  fmeta[13] = (GDOUBLE)time_out_end_;
  fmeta[14] = (GDOUBLE)time_out_skip_;
  fmeta[15] = (GDOUBLE)time_out_last_;
  fmeta[16] = (GDOUBLE)iStopCond_;
  fmeta[17] = (GDOUBLE)iOutType_;
  fmeta[18] = (GDOUBLE)bUPC;
  fmeta[19] = (GDOUBLE)bPPC;
  fmeta[20] = (GDOUBLE)upc_type_;
  fmeta[21] = (GDOUBLE)stokes_type_;
  fmeta[22] = (GDOUBLE)nu_[0];
  fmeta[23] = (GDOUBLE)nu_[1];
  fmeta[24] = (GDOUBLE)rho_;
  fmeta[25] = (GDOUBLE)ntimelevels_;
  fmeta[26] = (GDOUBLE)ngVertices_;
  fmeta[27] = (GDOUBLE)bFixedTimeStep_;
  fmeta[28] = (GDOUBLE)iEvolType_;
  fmeta[29] = (GDOUBLE)iorderadv_;
  fmeta[30] = (GDOUBLE)iorderBDF_;
  fmeta[31] = (GDOUBLE)iorderAM_;
  fmeta[32] = (GDOUBLE)bDoAdapt_;
  fmeta[33] = (GDOUBLE)ap_tol_;
  fmeta[34] = (GDOUBLE)ap_mult_;
  // Write global bdy vertices:
  n = ndmpmeta_;
  for ( i=0; i<ngVertices_; i++ )  {
    for ( j=0; j<GDIM; j++ ) fmeta[n+j+i*GDIM] = gd_[i][j];
  }
  // Write bdy-types: assumes no. bdys == no. vertices--only good for 2d
  n += ngVertices_*GDIM;
  for ( i=0; i<ngVertices_; i++ )  {
    fmeta[n+i] = (GDOUBLE)btype_[i];
  }

  // Set meta data:
  wgbin.SetMeta(ndmpmeta_+ngVertices_*(GDIM+1), fmeta, "GASPAR_DUMP_FILE");

  // Write dt-history
  nm = dthist_.dim();
  bRet = wgbin.SetDims(1, &nm) && 
         wgbin.WriteData(1, &nm, dthist_.Data(), 0, NULL, sTHLabel_ ,NULL);
  if ( !bRet ) {
    cout << serr << "dthist_ write failure: GBin write error: " << wgbin.Error() << endl;
    iGError_ = GERR_DUMP;
    GDoLog(0,"#GDump failure: dthist write failure");
    return FALSE;
  }

  nm = igPeriodic_.dim();
  fgPeriodic_.Resize(nm);
  for ( j=0; j<nm; j++ ) fgPeriodic_[j] = (GDOUBLE)igPeriodic_[j];
  bRet = wgbin.SetDims(1, &nm) &&
         wgbin.WriteData(1, &nm, fgPeriodic_.Data(), 0, NULL, sPBLabel_ ,NULL);
  if ( !bRet ) {
    cout << serr << "igPeriodic_ write failure: GBin write error: " << wgbin.Error() << endl;
    iGError_ = GERR_DUMP;
    GDoLog(0,"#GDump failure: igPeriodic write failure");
    return FALSE;
  }
  // Write spectral norms:
  nm = apunorm_.dim();
  bRet = wgbin.SetDims(1, &nm) &&
         wgbin.WriteData(1, &nm, apunorm_.Data(), 0, NULL, sSpNLabel_ ,NULL); 
  if ( !bRet ) {
    cout << serr << "GBin write error: " << wgbin.Error() << endl;
    iGError_ = GERR_DUMP;
    GDoLog(0,"#GDump failure: spectral norms write failure");
    return FALSE;
  }
  // Write derivative  norms:
  nm = dunorm_.dim();
  bRet = wgbin.SetDims(1, &nm);
         wgbin.WriteData(1, &nm, dunorm_.Data(), 0, NULL, sDerivNLabel_ ,NULL);
  if ( !bRet ) {
    cout << serr << "GBin write error: " << wgbin.Error() << endl;
    iGError_ = GERR_DUMP;
    GDoLog(0,"#GDump failure: derivative norms write failure");
    return FALSE;
  }

  // Write datasets:
  for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->start(NULL);
  wgbin.InitCommList(nelems_);
  for ( i=0, bRet=TRUE; i<nelems_ && bRet; i++ ) {
    for ( k=0; k<nEvolvedFields_; k++ ) {
      uu   = pgfields_[k]->member()->GetExpCoeffs(0);
      ub   = pgfields_[k]->member()->GetBdyValues();  
      e    = pgfields_[k]->member()->GetElement();
      vert = e->GetSpVertices();
      for ( j=0, NN=1; j<3; j++ ) {
        N[j] = e->GetOrder  (1) + 1;
        NN  *= N[j];
      }
      nt   = e->GetNumEdges()+e->GetNumVertices()+7;
      tag.Resize(nt);               nm=0;
      tag[0] = time_;               nm++;
      tag[1] = (GDOUBLE)icycle_;    nm++;
      tag[2] = e->GetNumVertices(); nm++;
      for ( j=0; j<e->GetNumVertices(); j++,nm++ ) 
        tag[nm] = (GDOUBLE) (e->GetVertType(j));
      tag[nm] = e->GetNumEdges();   nm++;
      for ( j=0; j<e->GetNumEdges(); j++,nm++ ) tag[nm] = (GDOUBLE) (e->GetEdgeType(j));
      tag  [nm] = (GDOUBLE)(e->GetRootID()); nm++;
      tag  [nm] = (GDOUBLE)(e->GetID());     nm++;
      if ( !wgbin.SetDims(nd_, N) ) bRet = FALSE;
      if ( !wgbin.SetVertices(nvert_, vert, e->ElemType()) ) bRet = FALSE;
      for ( j=0; j<GDIM && bRet; j++ ) { 
        if ( e->GetSpNodes(j+1) != NULL ) { 
          sprintf(slabel, "%s%d", "X", j);
          bRet = wgbin.SetCoord(j+1, NN, e->GetSpNodes(j+1)->Data(), slabel); 
        }
      }
      for ( j=0; j<pgfields_[k]->member()->GetNumTimeLevels() && bRet; j++ ) { 
        uu   = pgfields_[k]->member()->GetExpCoeffs(j);
        if ( j == 0 ) sprintf(slabel, "%s"    , sDSLabel_[k]   );
        else          sprintf(slabel, "%s%s%d", sDSLabel_[k], sDSLabel_suff, j);
        tag[nm] = pgfields_[k]->member()->GetTime(j);  // NOTE: timestamp is always last tag!
        if ( !wgbin.WriteData(nd_, N, uu->Data(), nt, tag.Data(), slabel ,NULL) ) bRet = FALSE;
      }
#if 0
    if ( ub != NULL && ub->Data() != NULL &&
         !wgbin.WriteData(nd_, N, ub->Data(), 0, NULL, sDSLabel_[GSV1B],NULL) ) bRet = FALSE;
#endif
    } // end of field-index loop
    for ( k=0; k<nEvolvedFields_; k++ ) pgfields_[k]->next();
    wgbin.EndLoop(i);
  } // end of field-index loop

  wgbin.Close();
  if ( fmeta ) delete [] fmeta; 

  if ( !bRet ) {
    cout << serr << "GBin write error: " << wgbin.Error() << endl;
    iGError_ = GERR_DUMP;
    GDoLog(0,"#GDump failure: data write failure");
    return FALSE;
  }
  TCDmp_ += (STK::CTimer()-T0); TWDmp_ += ((STK::Timer ()-T1)/1.0e6); // in sec
  return bRet;

} // end of method GDump


//************************************************************************************
//************************************************************************************
// METHOD     : GSetGridFromFile
// DESCRIPTION: Sets GASpAR grid from restart file
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GSetGridFromFile(GBinReader &rgbin)
{
  char      *serr = "GSetGridFromFile: ";
  GINT      i, ig, ind, j, n, nds, nm, nt, num, rank, *idims;
  GBOOL     bRet=TRUE;
  GDOUBLE   *tags, *fmeta=NULL;
  ELEMTYPE  etype;
  Point3D   *vert;
  Elem2D    *elem;
  GElemList *pelems;

  // Get global grid bounds:
//nm    = rgbin.GetNumMeta();
  fmeta = rgbin.GetMeta   ();
  if ( gd_ != NULL ) delete [] gd_; gd_ = NULL;
  gd_ = new Point [ngVertices_];
  num = ndmpmeta_;
  for ( i=0; i<ngVertices_; i++ )  {
    for ( j=0; j<GDIM; j++ ) gd_[i][j] = fmeta[num+j+i*GDIM];
  }
 
  // Get mesh:
  nds     = rgbin.GetNumDataSets();
  for ( ig=0; ig<fieldgroups_.size(); ig++ ) {
    for ( n=0, bRet=TRUE; n<nds && bRet; n++ ) {
//    if ( strcmp(sDSLabel_[0],rgbin.GetLabel(n)) != 0 ) continue;
      if ( strcmp(fieldgroups_[ig]->name,rgbin.GetLabel(n)) != 0 ) continue;
      pelems = fieldgroups_[ig]->pelems;
      rank    = rgbin.GetRank(n);
      if ( rank != nd_ ) {
        cout << serr << "Incompatible data rank" << endl;
        return FALSE;
      }
      vert    = rgbin.GetVert(n);
      idims   = rgbin.GetDims(n);
      etype   = rgbin.GetElemTypes(n);
      nt      = rgbin.GetNumTags(n);
      tags    = rgbin.GetTags(n);
  
      pelems->add(etype, ntmplevels_); elem = pelems->member();
      elem->SetVertices(vert, elem->GetNumVertices());
      switch ( fieldgroups_[ig]->basis_type ) {
        case GBASIS_GLL:
          for ( j=0; j<rank; j++ ) {
            if ( !gllpool_.findorder(idims[j]-1) ) gllpool_.add(idims[j]-1);
            gllpool_.member()->Solve();
            elem->SetBasis   (gllpool_.member(),j+1);
          }
        break;
        case GBASIS_GL:
          for ( j=0; j<rank; j++ ) {
            if ( !glpool_.findorder(idims[j]-1) ) glpool_.add(idims[j]-1);
            glpool_.member()->Solve();
            elem->SetBasis   (glpool_.member(),j+1);
          }
        break;
      }
      // NOTE: IMPORTANT!!: these tags should match 1-for-1 with those
      //       set in GDump method.
      nm = 3;
      for ( j=0; j<tags[2]; j++ ) {
        elem->GetVertType(j) = (BDYTYPE)tags[j+nm]; 
      }
      nm += (GINT )tags[2] + 1;
      for ( j=0; j<tags[nm-1]; j++ ) {
        elem->GetEdgeType(j) = (BDYTYPE)tags[j+nm]; 
        elem->bGlobalBdyEdge(j) = ( elem->GetEdgeType(j) != NONE ) ;
      }
      nm += (GINT )tags[nm-1];
      elem->SetRootID((GKEY)tags[nm++]); 
      elem->SetID    ((GKEY)tags[nm++]); 
      elem->ComputeBdyInfo();
      if ( !elem->SolveFE() ) {
        cout << serr << "Error in grid solve: grid id: " << ig << endl;
        return FALSE;
      }
    } // end, loop over data set grids
  } // end, field group loop

  // Operate on the primary grid for the following quantities:
  // NOTE: primary grid must be the _first_ grid represented in the 
  //       fieldgroups list.
  pelems       = fieldgroups_[0]->pelems;
  nelems_      = pelems->size();
//gMinLength_  = GUtils::ComputeMinLength(uelems);
  GUtils::ComputeGlobalDOFs(*pelems, gndofs_, gMinElemLength_, gMaxElemLength_);
  GUtils::GetGlobalPeriodicity(igPeriodic_, gd_, ngVertices_, *pelems);
  GComm::Allreduce(&nelems_, &ngelems_, 1, GC_GINT , G_OP_SUM);

  return bRet;
} // end of method GSetGridFromFile
