//======================================================================================
// Name         : gaspar_t.cpp
// Date         : 9/10/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : GASpAR main driver 
// Modifications:
//======================================================================================
#define _GASPAR_H_INIT
#include "gaspar_t.h"
#include "guser.h"


int main(int argc, char* argv[])
{
  char      *serr = "GASpAR::main: ";
  GINT      iargc=(GINT )argc, nc=0;
  char      smsg[FILE_NAME_MAX];
  GDOUBLE   dtt;
#if defined(G_MAIN_LOOP_CPUTIME)
  GDOUBLE   TW=0, TC=0, T0, T1;
  GBOOL     bExist;
  ifstream  iscpu;
  ofstream  oscpu;
  iscpu.open("gaspar.time");
  bExist = !iscpu.fail(); iscpu.close();
  oscpu.open("gaspar.time",ios::app);
  if ( !bExist ) 
    oscpu << " NE  T_cpu_per_cyc (s)  T_wall_per_cyc (s)  T_cpu (s)  T_wall (s)" << endl;
#endif

#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Doing InitComm..." << endl;
#endif
  GComm::InitComm(&argc, &argv); 
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "InitComm done." << endl;
#endif

  // Start GASpAR:
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Doing GInit..." << endl;
#endif
  if ( !GInit(iargc, argv) ) {
    cout << "Main: GInit failed" << endl;
    if ( GComm::WorldSize() == 1 ) GOutput();
    exit(1);
  }


#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "GInit done." << endl;
#endif

#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Initializing time levels..." << endl;
#endif
  if ( bAutoTimeLevelFill_ && !bRestart_ && !GInitTimeLevels() ) {
    cout << "Main: GInitTimeLevels failed" << endl;
    exit(1);
  }
  time_out_last_ = time_;
  time_out_beg_  = time_;
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Initializing time levels done." << endl;
#endif
  

#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Creating log output..." << endl;
#endif
  GDoLog(0,"#*************************************************************************************");
  GDoLog(0,"#*************************************************************************************");
  if ( bRestart_ )
    GDoLog(0,"#**RESTART** Parameter Summary: \n#" );
  else
    GDoLog(0,"#Parameter Summary: \n#" );
  GDoLog(2,"","header","static");
  GDoLog(1,"","dynamic");
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Initial log output done." << endl;
#endif
  bRestart_ = FALSE;
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Creating initial data output..." << endl;
#endif
  if ( GOutCond() ) GOutput();
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Initial data output done." << endl;
#endif

  // Enter time loop:
  while ( !GStopCond() && !iGError_ ) {
#if defined(G_MAIN_LOOP_CPUTIME)
    T0 = STK::CTimer(); T1 = STK::Timer();
#endif
    bElemListChange_ = FALSE;
    if ( iGError_ == GERR_ADAPT ) {
      GOutput();
      break; 
    }
    GTimeStep();
    if ( iStopCond_ == BY_TIME ) {
      if ( (dtt=MIN(fabs(time_max_-time_),dt_)) <= dtMin_ ) break;
      dt_ = dtt;
    }
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Doing GAdaptGrid..." << endl;
#endif

    glop->SetCycle(icycle_);
    GAdaptGrid() ;

#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "GAdaptGrid done." << endl;
    cout << serr << "Doing GUserTimeDep..." << endl;
#endif
    GUserTimeDep(time_, dt_);
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "GUserTimeDep done." << endl;
    cout << serr << "Doing GStep..." << endl;
#endif
    GStep();
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "GStep done." << endl;
#endif
    icycle_++; time_ += dt_;
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "Doing GTimeUpdate..." << endl;
#endif
    GTimeUpdate();
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "GTimeUpdate done." << endl;
    cout << serr << "Doing GOutput..." << endl;
#endif
    if ( GOutCond() ) GOutput();
#if defined(MAIN_TRACE_OUTPUT)
    cout << serr << "GOutput done." << endl;
#endif
    
    // Do logging:
    if ( icycle_ == 0 || icycle_log_skip_ > 0 && icycle_ % icycle_log_skip_ == 0 ) {
#if defined(MAIN_TRACE_OUTPUT)
      cout << serr << "Doing logging (1)..." << endl;
#endif
      GDoLog(1,"","dynamic");
#if defined(MAIN_TRACE_OUTPUT)
      cout << serr << "logging (1) done." << endl;
#endif
    }
     
    // Do dynamic logging:
    if ( icycle_dmp_skip_ > 0 && icycle_ % icycle_dmp_skip_ == 0  ) {
#if defined(MAIN_TRACE_OUTPUT)
      cout << serr << "Doing logging (2)..." << endl;
#endif
      GDoLog(0,"#***>","dynamic");
#if defined(MAIN_TRACE_OUTPUT)
      cout << serr << "logging (2) done." << endl;
      cout << serr << "Doing GDump..." << endl;
#endif
      GDump(fndmp_,TRUE);
#if defined(MAIN_TRACE_OUTPUT)
      cout << serr << "GDump done." << endl;
#endif
    }
//  if ( bElemListChange_ ) { strcpy(fnout_,"tmp"); GOutput(); break; }
    nc++;
#if defined(G_MAIN_LOOP_CPUTIME)
    TC += (STK::CTimer()-T0)         ;  // total time
    TW += ((STK::Timer ()-T1)/1.0e6) ;
#endif
  } // end of time-loop
 
#if defined(G_MAIN_LOOP_CPUTIME)
  TC -= (TCLog_ + TCOut_ + TCDmp_);       // subtract out all cumulative I/O times
  TW -= (TWLog_ + TWOut_ + TWDmp_);     
  cout  
        << nelems_                               << " " 
        << (nc > 1 ? TC/(nc-1) : TC)             << " "
        << (nc > 1 ? TW/(nc-1) : TW)             << " "
        << TC                                    << " " 
        << TW                                    << endl;
  oscpu 
        << nelems_                               << " " 
        << (nc > 1 ? TC/(nc-1) : TC)             << " "
        << (nc > 1 ? TW/(nc-1) : TW)             << " "
        << TC                                    << " " 
        << TW                                    << endl;
  oscpu.close();
#endif

  GUserTerm();
  GDoLog (1,"","dynamic");
  if ( iStopCond_ == BY_CYCLE || iGError_ > 0  ) GOutput();


  GWrapup();

  GComm::TermComm(); 
 
} // end of function main


//************************************************************************************
//************************************************************************************
// METHOD     : GStep
// DESCRIPTION:
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GStep()
{
  char     *serr = "GStep: ";
  GSHORT   nv;
  GINT    bStep[2]={1,1}, bStepg[2]={1,1};
  GBOOL   bRet;

  bStep[0] = stepper->Step(dt_);

//bStep[1]  = bStep[1] && GErrChk();
  GComm::Allreduce(bStep, bStepg, 2, GC_GINT , G_OP_MIN);
  bStepError_ = (GBOOL) (!bStepg[0]);
  bConvError_ = (GBOOL) (!bStepg[1]);

  bRet = ((GBOOL)bStepg[0] && (GBOOL)bStepg[1]);
  if ( !bRet ) {
    GDoLog(0,"#GStep failure");
    iGError_ = GERR_STEP;
  }

  return bRet;
  
} // end of method GStep


//************************************************************************************
//************************************************************************************
// METHOD     : GInit
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GInit(int argc, char *argv[])
{

  char       *serr = "GInit: " ;
  GINT       i, j, k, nf;
  GBinReader rgbin;


  rank  = GComm::WorldRank();
  nproc = GComm::WorldSize();
  sprintf(fnout_    ,"results");
  sprintf(fnmesh_   ,"mesh.dat");
  sprintf(fnlog_    ,"gaspar.log");
  sprintf(fncmd_    ,"gaspar.dat");
  sprintf(fndmp_    ,"gaspar.dmp");  
  sprintf(fnrst_    ,"gaspar.dmp"); 
  sprintf(fnuser_   ,"gaspar.user"); 
  sprintf(suserblk_ ,"UserBlk"); 

  for ( i=0; i<GDIM; i++ ) {
    nu_     [i] = 1.0;
  }
  igPeriodic_.Resize((GINT)pow(2.0,GDIM)-2*GDIM+4); 
  igPeriodic_ = 1;
  nrefinelevels_ = MAX_REFINE_LEVELS;

  dX_ = new Point();
  glop = new NTreeAdapt(nrefinelevels_);
  glop->SetElements(&uelems);
  glop->DoPartitioning(FALSE);
  igNumElems_.Resize(GComm::WorldSize());
  
//morton_ = new Morton_KeyGen(Morton_KeyGen::MORTON_FULL_INTERLEAVE);
  morton_ = new Morton_KeyGen(Morton_KeyGen::MORTON_PARTIAL_INTERLEAVE);

  if ( !GCmdLineParse(argc, argv) ) {
    cout << serr << "command line parse error " << endl;
    exit(1);
  }
  if ( bReadCmdFile_ && !GCmdFileParse() ) {
    cout << serr << "command file parse error " << endl;
    exit(1);
  }
  if ( !GCmdLineParse(argc, argv) ) {
    cout << serr << "command line parse error " << endl;
    exit(1);
  }
 
  if ( !GUserConfig() ) {
    cout << serr << "GUserConfig failed" << endl;
    return FALSE;
  }
 
  // Create variable names:
#if defined(BURGERS)
  sDSLabel_ = new char * [nEvolvedFields_];
  for ( k=0; k<nEvolvedFields_; k++ ) {
    sDSLabel_[k] = new char[8];
    if      ( k <  GDIM ) sprintf(sDSLabel_[k],"V%d",k+1);
  }
#endif

#if defined(NAVIER_STOKES)
  nEvolvedFields_ = GDIM + 1;
  sDSLabel_ = new char * [nEvolvedFields_];
  for ( k=0; k<nEvolvedFields_; k++ ) {
    sDSLabel_[k] = new char[8];
    if      ( k <  GDIM ) sprintf(sDSLabel_[k],"V%d",k+1);
    else if ( k == GDIM ) sprintf(sDSLabel_[k],"P");
  }
#endif

  ntimelevels_ = iorderBDF_;
  ntimelevels_ = MAX(iorderadv_, iorderBDF_);
  if ( iEvolType_ == TE_AMFE ) {
    ntimelevels_ = MAX(ntimelevels_, iorderAM_);
  }

  if ( bLinAdvection_ ) {
    for ( j=0; j<3; j++ ) cadv_[j] = new GFieldList();
  }
  
  dthist_.Resize(10); dthist_ = 0.0;

  if ( !GCreateFieldArrays() ) {
    cout << serr <<  "GCreateFieldArrays failed" << endl;
    return FALSE;
  }
  
  // Get mesh:
  if ( !bRestart_ || bRegrid_ ) {
    // Use command line params to generate mesh file:
    if ( !GInitMesh () ) {
      cout << serr << "GInitMesh failed" << endl;
      return FALSE;
    }
  } 
  else {
    if ( !rgbin.Open(fnrst_) ) {
      cout << serr << "Cannot open file named " << fnrst_
           << " Reader error: " << rgbin.Error() << endl;
      exit(1);
    }

    // Get mesh from restart file:
    if ( !GSetGridFromFile(rgbin) ) { 
      cout << serr << "GSetGridFromFile failed" << endl;
      exit(1);
    }
    rgbin.Close();
  }

  // Create hydro variables (if necessary):
  if ( !GCreateVar() ) {
    cout << serr <<  "GCreateVar failed" << endl;
    return FALSE;
  }

  // Configure load balancing objects:
  nf = nEvolvedFields_;
#if defined(NAVIER_STOKES)
  nf--;  // don't include pressure
#endif
  partitioner_ = new GPartitioner(&uelems,pgfields_,nf); // set primary fields; secondaries (e.g. pressure) set later
  partmapper_  = new GPMElemID(&uelems);
  glop->SetPartitioner(partitioner_);
  glop->SetLoadBalanceMapper(partmapper_);

  // Initialize gather/scatter operator:
  if ( !GSetKeyGen() ) {
    cout << serr << "could not initialize key generator" << endl;
    return FALSE;
  }

  if ( !GInitGS() ) {
    cout << serr << "could not initialize gather-scatter" << endl;
    return FALSE;
  }

  
#if 0
  // Do globalization of boundary nodes:
  if ( !GGlobalizeBdy() ) {
    cout << "GInit: GlobalizeBdy failed" << endl;
    return FALSE;
  }
#endif

  // ###########################################################################################
  //                                  Problem initialization:
  // ###########################################################################################

  if ( !bRestart_ ) {
    if ( !GUserInit(argc, argv) ) {
      cout << serr << "NSUserInit failed" << endl;
      return FALSE;
    }
  }
  else {  // restart
    if ( !GRestart() ) {
      cout << serr << "restart failed" << endl;
      return FALSE;
    }
    if ( !GCmdLineParse(argc, argv) ) { // reset command line set params
      cout << serr << "command line parse error " << endl;
      exit(1);
    }
    if ( bReadCmdFile_ && !GCmdFileParse() ) {
      cout << serr << "command file parse error " << endl;
      exit(1);
    }
    if ( !GCmdLineParse(argc, argv) ) {
      cout << serr << "command line parse error " << endl;
      exit(1);
    }
    if ( !GUserStart() ) {
      cout << serr << "UserStart failed" << endl;
      return FALSE;
    }
  }

  // Create solver (stepper) operators, and initialize:
  if ( !GCreateStepper() ) {
    cout << serr << "GCreateStepper failed" << endl;
    return FALSE;
  }
  stepper->SetTime(time_);
  stepper->SetNSubcycles((GINT)Courant_+2);

  if ( !GSetBdyCond() )  {
    cout << serr << "SetBdyCond failed" << endl;
    return FALSE;
  }

  // Do global assembly of quantities:
  if ( !GGlobalAssembly() ) {
    cout << serr << "GlobalAssembly failed" << endl;
    return FALSE;
  }

  if ( bRestart_ ) {
    GDoLog(0,"#***> Restart loaded.");
  }


  if ( bDoAdapt_ && ( !bSpectralAP_ && !b1DerivativeAP_ && !b2DerivativeAP_  ) ) {
    cout << serr << "Adaption requires a a-posterioir error estimator" << endl;
    exit(1);
  }

  if ( !GInitLogger() ) {
    cout << serr << "logger initialization failure" << endl;
    exit(1);
  }

  if ( !bRestart_ ) return GInitNorms();

  return TRUE;
} // end of method GInit


//************************************************************************************
//************************************************************************************
// METHOD     : GInitNorms
// DESCRIPTION: Compute initial solution norms for normalizing the 
//              adaptivity criteria
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GInitNorms()
{

  GINT      i, j, m;
  GDOUBLE   ul, gul, gLMax, gL[GDIM];
  GVector   *u;

#if 1
  gL[0] = gd_[1][0] - gd_[0][0];
  gL[1] = gd_[3][1] - gd_[0][1];
  if ( GDIM == 3 ) gL[2] = gd_[4][0] - gd_[0][0];
  for ( j=0, gLMax=0.0; j<GDIM; j++ ) {
    gLMax = MAX(gLMax,gL[j]);
  }


  // Find solution norm at t0 for adatptivity criteria:
  for ( m=0; m<naPostFields_; m++ ) {
    pafields_[m]->start(NULL);
    for ( i=0,ul=0.0; i<nelems_; i++ ) {
      u     = pafields_[m]->member()->GetExpCoeffs(0); 
      ul = MAX(ul,u->MaxA());
      pafields_[m]->next();
    }
    GComm::Allreduce(&ul, &gul, 1, GC_GDOUBLE, G_OP_MAX);
    apunorm_[2*m  ] = 1.0 / gul;
    apunorm_[2*m+1] = 1.0;
    dunorm_ [2*m  ] = 1.0 / gul;  //gLMax / gul;
    dunorm_ [2*m+1] = dunorm_[2*m];  //gLMax*gLMax * dunorm_[2*m];
    pafields_[m]->next();
  }
#else
  apunorm_= 1.0;
  dunorm_ = 1.0;
#endif
  return TRUE;
} // end of method GInitNorms


//************************************************************************************
//************************************************************************************
// METHOD     : GInitMesh
// DESCRIPTION: Initializes primary mesh with call to mesh file, or by strict construction.
//              NOTE: the basis pointer arrays that come out of this routine are assumed
//                    to be constant for all elements; i.e., the degree is the same
//                    for all elements.  
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GInitMesh()
{
  char       *serr = "GInitMesh: " ;
  GINT       i, j, iret;
  GINT       gne_max, ind;
  GBOOL      bRet;
  char       cmdstr[2048];
  Point3D    v2[4], *vert;
  MeshReader  mr;

  if ( !bExtMesh && rank == 0 ) {
    sprintf(cmdstr,"./gdd -quiet -d %d -Nx %d -Ny %d -xNx %d -xNy %d -Np %d -P0 %f %f -P1 %f %f -b %f %f %f %f %d -b %f %f %f %f %d -b %f %f %f %f %d -b %f %f %f %f %d  -nodes -o By_Proc -f mesh.dat %s",
            nd_, Ne[0], Ne[1], xN[0]+1, xN[1]+1,  nproc,
            P0_[0], P0_[1], P1_[0], P1_[1],
            B0_[0][0], B0_[0][1], B1_[0][0],  B1_[0][1], btype_[0],
            B0_[1][0], B0_[1][1], B1_[1][0],  B1_[1][1], btype_[1],
            B0_[2][0], B0_[2][1], B1_[2][0],  B1_[2][1], btype_[2],
            B0_[3][0], B0_[3][1], B1_[3][0],  B1_[3][1], btype_[3],
            sgdd_
            ); 
    cout << serr << "cmdstr:" << cmdstr << endl;
    if ( (iret=system(cmdstr)) == -1 || iret == 127 ) { 
      cout << serr << "gdd execution falied" << endl;
      return FALSE;
    }
    sprintf(fnmesh_,"mesh.dat");
  }

  if ( !bPGrid_ ) {
    // Read mesh file:
    if ( !GUtils::InitMesh(fnmesh_ , rank, gllpool_,
                   uelems, ngelems_, vnode_ids, vmaxid, 
                   gd_, ngVertices_, 1        , nrefinelevels_) ) {
      cout << serr << "InitMesh failed" << endl;
      exit(1);
    }
  }
  else {
    // Create PERIODIC grid:
    if ( !GUtils::InitPMesh2d(Ne[0], Ne[1], xN[0], xN[1], 
                   gllpool_,
                   uelems, ngelems_, vnode_ids, vmaxid, 
                   gd_, ngVertices_, 1, nrefinelevels_) ) {
      cout << serr << "InitMesh failed" << endl;
      exit(1);
    }
  }
  nelems_          = uelems.size();
  bElemListChange_ = FALSE;

  // Gather nelems from all procs here:
  GComm::Allgather(&nelems_, 1, GC_GINT, igNumElems_.Data(), 1, GC_GINT);
  GComputeLoadBal();

  // Create pool to be used for dealiasing. There must
  // one for each member of the gllpool structure, and in the
  // same order:
  gllpool_.start(NULL);
  for ( j=0; j<gllpool_.size(); j++ ) {
    gllpool32_.add(1.5*(gllpool_.member()->GetOrder()));
    gllpool32_.member()->Solve();
    gllpool_.next();
  }

  // Set the dealiasing bases in each element:
  uelems.start(NULL);
  for ( i=0; i<uelems.size(); i++ ) {
    for ( j=0; j<GDIM; j++ ) {
      if ( !gllpool_.findorder(uelems[i]->GetOrder(j+1),ind) ) {
        cout << serr << "requested basis not in pool" << endl;
        return FALSE;
      }
      uelems.member()->SetDBasis(gllpool32_[ind],j+1);
      uelems.member()->SolveFE();
    }
    uelems.next();
  }  

#if defined(NAVIER_STOKES)
  // From primary (GLL) grid, set the secondary grid basis pool:
  gllpool_.start(NULL);
  for ( j=0; j<gllpool_.size(); j++ ) {
    glpool_.add(gllpool_.member()->GetOrder()-2);
    glpool_.member()->Solve();
    gllpool_.next();
  }

  // Generate secondary mesh:
  for ( i=0; i<uelems.size(); i++ ) {
    vert = uelems[i]->GetSpVertices();
    pelems.add(uelems[i]->ElemType(),1);
    pelems.member()->SetVertices(vert, uelems[i]->GetNumVertices());
    for ( j=0; j<GDIM; j++ ) {
      if ( !glpool_.findorder(uelems[i]->GetOrder(j+1)-2,ind) ) {
        cout << serr << "requested basis not in pool" << endl;
        return FALSE;
      }
      pelems.member()->SetBasis(glpool_[ind],j+1);
    }
    pelems.member()->SetRootID(uelems[i]->GetRootID());
    pelems.member()->SetID    (uelems[i]->GetID    ());
    if ( !pelems.member()->SolveFE() ) {
      cout << serr << "error in pressure grid solve" << endl;
      return FALSE;
    }
  }

#endif

//gMinelemLength_ = GUtils::C mputeMinLength(uelems);
  GUtils::ComputeGlobalDOFs(uelems, gndofs_, gMinElemLength_, gMaxElemLength_);
  GUtils::GetGlobalPeriodicity(igPeriodic_, gd_, ngVertices_, uelems);
  bRet = GSetKeyGen();

  return bRet;

} // end of method GInitMesh


//************************************************************************************
//************************************************************************************
// METHOD     : GSetKeyGen
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GSetKeyGen()
{
  Point p=gd_[0];

  if ( !GUtils::SetMorton(morton_, &uelems, &p, dX_, 
                gd_, ngVertices_, nrefinelevels_) ) {
    cout << "GSetKeyGen: Warning: unable to compute integral length" << endl;
    exit(1);
  }
  *dX_= 1.0e-8;
  morton_->SetIntegralLen(*dX_);
  morton_->SetBox(gd_[0],gd_[2]);
  morton_->SetDoLog(FALSE);
  glop->SetGlobalDomain(gd_,ngVertices_,igPeriodic_);
  glop->SetKeyGen(morton_);
  glop->SetLevelLimit(nrefinelevels_);

  return TRUE;

} // end of method GSetKeyGen


//************************************************************************************
//************************************************************************************
// METHOD     : GCreateVar
// DESCRIPTION: Creates F.D. variables
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GCreateVar()
{
  GINT       i, j, k, NN;
  GBOOL      bRet=TRUE;
  Point      *vert;

  uelems.start(NULL);
  for ( i=0; i<uelems.size(); i++ ) {
    NN = uelems[i]->GetNumNodes();
    u1  .add(ntimelevels_,uelems[i],ntmplevels_);
    u2  .add(ntimelevels_,uelems[i],ntmplevels_);

    if ( bLinAdvection_ ) {
      for ( j=0; j<GDIM; j++ ) {
        cadv_[j]->add(ntimelevels_,uelems[i],ntmplevels_);
      }
    }

    utmp        .add(NULL, TRUE);   utmp        [i] = new GVector (NN);
    uptmp       .add(NULL, FALSE);  uptmp       [i] = NULL;
    poldElements.add(uelems[i]); 
    u1bdyvals   .add(NULL, FALSE);  // will point to areas in corresp fields
    u2bdyvals   .add(NULL, FALSE);  // will point to areas in corresp fields
    u1bdyvals [i] = u1[i]->GetBdyValues();
    if ( nEvolvedFields_ > 1 )
    u2bdyvals [i] = u2[i]->GetBdyValues();


#if defined(NAVIER_STOKES)
    if ( pelems[i] != NULL ) {
      NN = pelems[i]->GetNumNodes();
      ptmp        .add(NULL, TRUE);   ptmp        [i] = new GVector (NN);
      p   .add(ntimelevels_, pelems[i], ntmplevels_);
    }
#endif
  } // end of element list loop

  if ( bFilter_ ) {
    filter_ = new GLinOpList ();
    for ( i=0; i<uelems.size(); i++ ) {
      filter_->add(NULL,TRUE); 
      (*filter_)[i] = new GIFilter(uelems[i],filter_alpha_,filter_delta_);
    }
  }
  
  return bRet;

}  // end of method GCreateVar


//************************************************************************************
//************************************************************************************
// METHOD     : GCreateFieldArrays
// DESCRIPTION: Creates variable arrays
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GCreateFieldArrays()
{
  GINT       j;

  // Set up nEvolvedFields_-dependent quantities:
  errcg_  = new GDOUBLE[nEvolvedFields_+2];
  gerrcg_ = new GDOUBLE[nEvolvedFields_+2];
  ncgit_  = new GINT   [nEvolvedFields_+2];
  gncgit_ = new GINT   [nEvolvedFields_+2];
  naPostFields_ = nEvolvedFields_;
  if ( pgfields_  != NULL ) delete [] pgfields_;
  if ( pafields_  != NULL ) delete [] pafields_;
  if ( pgbdyvals_ != NULL ) delete [] pgbdyvals_;
  pgfields_   = new GFieldList * [nEvolvedFields_];
  pafields_   = new GFieldList * [naPostFields_];
  pgbdyvals_  = new GVecList   * [nEvolvedFields_];
  apunorm_.Resize(2*naPostFields_);
  apunorm_ = 1.0;
  dunorm_ .Resize(2*naPostFields_);
  dunorm_  = 1.0;
  iBasisTypes_.Resize(nEvolvedFields_);
#if defined(BURGERS)
  if      ( nEvolvedFields_ == 1 ) {
    pgfields_ [0] = &u1; 
    pafields_ [0] = &u1; 
    pgbdyvals_[0] = &u1bdyvals;
  }
  else if ( nEvolvedFields_ == 2 ) {
    pgfields_ [0] = &u1;
    pgfields_ [1] = &u2;
    pafields_ [0] = &u1;
    pafields_ [1] = &u2;
    pgbdyvals_[0] = &u1bdyvals;
    pgbdyvals_[1] = &u2bdyvals;
  }
  else { 
    pgfields_ [0] = &u1;
    pgfields_ [1] = &u2;
    pgfields_ [2] = &u3;
    pafields_ [0] = &u1;
    pafields_ [1] = &u2;
    pafields_ [2] = &u3;
    pgbdyvals_[0] = &u1bdyvals;
    pgbdyvals_[1] = &u2bdyvals;
    pgbdyvals_[2] = &u3bdyvals;
  }
  fieldgroups_.add();
  fieldgroups_.member()->nfields = nEvolvedFields_;
  strcpy(fieldgroups_.member()->name,sDSLabel_[0]);
  fieldgroups_.member()->basis_type = GBASIS_GLL;
  fieldgroups_.member()->pelems  = &uelems;
  fieldgroups_.member()->pfields = new GFieldList * [nEvolvedFields_];
  for ( j=0; j<nEvolvedFields_; j++ ) {
    fieldgroups_.member()->pfields[j] = pgfields_[j];
  }
  for ( j=0; j<nEvolvedFields_; j++ ) iBasisTypes_[j] = GBASIS_GLL;
#endif

#if defined(NAVIER_STOKES)
  if ( nEvolvedFields_ == 2 ) {
    pgfields_ [0] = &u1;
    pgfields_ [1] = &p;
    pafields_ [0] = &u1;
    pafields_ [1] = &p;
    pgbdyvals_[0] = &u1bdyvals;
    pgbdyvals_[1] = NULL; 
  }
  else if (nEvolvedFields_ == 3 ) {
    pgfields_ [0] = &u1;
    pgfields_ [1] = &u2;
    pgfields_ [2] = &p;
    pafields_ [0] = &u1;
    pafields_ [1] = &u2;
    pafields_ [2] = &p;
    pgbdyvals_[0] = &u1bdyvals;
    pgbdyvals_[1] = &u2bdyvals;
    pgbdyvals_[2] = NULL; 
  }
  fieldgroups_.add(); // add GLL grid: GDIM comps of velocity:
  fieldgroups_.member()->nfields = nEvolvedFields_-1;
  strcpy(fieldgroups_.member()->name,sDSLabel_[0]);
  fieldgroups_.member()->basis_type = GBASIS_GLL;
  fieldgroups_.member()->pelems  = &uelems;
  fieldgroups_.member()->pfields = new GFieldList * [nEvolvedFields_-1];
  for ( j=0; j<nEvolvedFields_-1; j++ ) {
    fieldgroups_.member()->pfields[j] = pgfields_[j];
  }
  fieldgroups_.add(); // add GL grid: pressure
  fieldgroups_.member()->nfields = 1;
  strcpy(fieldgroups_.member()->name,sDSLabel_[nEvolvedFields_-1]);
  fieldgroups_.member()->basis_type = GBASIS_GL;
  fieldgroups_.member()->pelems  = &pelems;
  fieldgroups_.member()->pfields = new GFieldList * [1];
  fieldgroups_.member()->pfields[0] = pgfields_[nEvolvedFields_-1];
  for ( j=0; j<nEvolvedFields_-1; j++ ) iBasisTypes_[j] = GBASIS_GLL;
  iBasisTypes_[nEvolvedFields_-1] = GBASIS_GL;
#endif

  // Field pointers for Courant-limited timesteps:
  ncourflds_ = bLinAdvection_ ? GDIM : nEvolvedFields_;
#if defined(NAVIER_STOKES)
  ncourflds_--;
#endif
  if ( !bLinAdvection_ ) {
    for ( j=0; j<ncourflds_; j++ ) courflds_[j] = pgfields_[j];
  }
  else {
    for ( j=0; j<ncourflds_; j++ ) courflds_[j] = cadv_    [j];
  }

  return TRUE;

}  // end of method GCreateFieldArrays




//************************************************************************************
//************************************************************************************
// METHOD     : GInitGS
// DESCRIPTION: Initializes gather scatter operator
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GInitGS()
{
  GINT       i, na, ne;
  GBOOL      bRet;
  GFieldList **f, *press;

  // With mesh file, and elem list and fields, can 
  // now initialize comm object:
  if ( pgfields_ == NULL ) {
    cout << "GInitGS: invalid field pointers" << endl;
    return FALSE;
  }
  na = bLinAdvection_ ? GDIM : 0;
  f  = new GFieldList * [nEvolvedFields_+na];
  for ( i=0; i<nEvolvedFields_; i++ ) f[i] = pgfields_[i];
  for ( i=nEvolvedFields_; i<nEvolvedFields_+na; i++ ) f[i] = cadv_[i-nEvolvedFields_];
  ne = nEvolvedFields_ + na;
#if defined(NAVIER_STOKES)
  ne = GDIM;  // remove pressure
#endif
  glop->SetFieldGroup(&uelems, f, ne); 
#if defined(NAVIER_STOKES)
  press = &p;
  glop->SetFieldGroup(&pelems, &press, 1); 
#endif
  bRet = glop->Init();

  delete [] f;

  return bRet;

} // end of method GInitGS


//************************************************************************************
//************************************************************************************
// METHOD     : GCreateStepper
// DESCRIPTION: Creates Stokes operators
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GCreateStepper()
{
  GINT        k;
  GFieldList  *uf[GDIM+1];
  
  if ( glop == NULL ) return FALSE;

#if defined(BURGERS)
  if ( bLinAdvection_ ) 
    stepper = new BurgersSolver(&uelems, pgfields_, nEvolvedFields_, cadv_, iadv_, nadv_);
  else
    stepper = new BurgersSolver(&uelems, pgfields_, nEvolvedFields_, NULL, iadv_, nadv_);
  if ( stepper == NULL ) return FALSE;
#endif
#if defined(NAVIER_STOKES)
  uf[0] = &u1; uf[1] = &u2; uf[2] = &p;
  stepper = new NavierStokes(uf, GDIM+1, &uelems, &pelems, stokes_type_);
  if ( stepper == NULL ) return FALSE;
  stepper->SetDen(rho_);
  stepper->SetDoDealiasing(bDoDealiasing_);
#endif

  GComm::Synch();
  stepper->SetComm(glop);
//stepper->SetCommHandle(hDSOp) ;
  for ( k=0; k<nEvolvedFields_; k++ ) {
    stepper->GetSolver(k+1)->SetTolerance(u1tol);
    stepper->GetSolver(k+1)->SetMaxIterations(MAX(gndofs_,u1iter));
  }
#if defined(NAVIER_STOKES)
  stepper->GetSolver(nEvolvedFields_)->SetTolerance(ptol);
  stepper->GetSolver(nEvolvedFields_)->SetMaxIterations(MAX(gndofs_,piter));
#endif
  for ( k=0; k<GDIM; k++ ) stepper->SetVisc(nu_[k], k+1);
  stepper->SetDoAdvection(bDoAdvection_);



#if 0
  for ( i=0; i<nelems_ ; i++ ) {
    for ( j=0; j<nsi_force_; j++ ) {
      u1.member(i)->GetSIForceList()->add(si_force_type_[j]);
      if ( !u1.member(i)->GetSIForceList()->member()->ParamSet(fncmd_, sforce_blk_[j]) ) {
         cout << "GCreateStepper: Unable to set parameters for force; file:" << fncmd_
              << " block: " << sforce_blk_[j] << endl;
      }
    }
  }
#endif
  if ( bUPC ) {
#if defined(BURGERS)
    stepper->SetPreconditioner(upc_type_); // sets precond for all fields
#endif
#if defined(NAVIER_STOKES)
    for ( k=0; k<GDIM; k++ ) 
    stepper->GetStokes()->SetPreconditioner(k+1,upc_type_);
#endif
  }
  
  stepper->SetTimestepHistory(&dthist_);
  stepper->SetAdvOrder(iorderadv_);
  stepper->SetBDFOrder(iorderBDF_);
#if defined(BURGERS)
  stepper->SetAMOrder(iorderAM_);
#endif
#if defined(NAVIER_STOKES)
  if ( bPPC ) {
    stepper->GetStokes()->SetPreconditioner(GDIM+1, GPC_BLOCKJAC_LAP);
  }
  stepper->SetEvolType((NavierStokes::TIME_EVOLTYPE) iEvolType_);
#elif defined(BURGERS)
  stepper->SetEvolType((BurgersSolver::TIME_EVOLTYPE)iEvolType_);
#endif

  stepper->SetFilter(filter_);


  return TRUE;
}  // end of method GCreateStepper


//************************************************************************************
//************************************************************************************
// METHOD     : GSetBdyCond
// DESCRIPTION: Sets bdy conditions
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GSetBdyCond()
{ 
  GINT  k;

  if ( stepper == NULL ) return TRUE;

  for ( k=0; k<nEvolvedFields_; k++ ) {
    stepper->SetVBdyData(k+1, pgbdyvals_[k]);
  }

  return TRUE;
} // end of method GSetBdyCond


//************************************************************************************
//************************************************************************************
// METHOD     : GCmdFileParse
// DESCRIPTION: Parses command command file
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GCmdFileParse()
{

  GINT        ssi=(GINT )stokes_type_, pci=(GINT )upc_type_, osi=(GINT )iStopCond_;
  GINT        itorder=iorderBDF_, iaorder=iorderadv_, imorder=iorderAM_;
  GDOUBLE     fdt=dt_;
  ParamReader mcmd(TRUE);


  // If config file doesn't exist, this is not considered an error:
  if ( !mcmd.Access(fncmd_) ) {
    cout << "CmdFileParse: Command file, '" << fncmd_ << "' does not exist; using defaults" << endl;
    return TRUE;
  }

   mcmd.SetBuffSize(SHRT_MAX);

  // Configure ParamReader object with  all parameters that _may_
  // be changed via cmd file:
  mcmd.SetParams("%f %b %b %d %d %i %i %f %f %d %f %f %f %i %i %i %i %i %s %s %s %b %b %s %s %d %f %f %f %f %f %f %b %i %i %i %b %f %b %b %f %i %b %b %b %i, %b %f"
                                     , "Courant_Number"  
                                     , "Use_V-Preconditioner"
                                     , "Use_P-Preconditioner"
                                     , "V-Preconditioner_Type"
                                     , "Stokes_Solver_Spltting_Type"
                                     , "Max_Number_of_Velocity_Iterations" 
                                     , "Max_Number_of_Pressure_Iterations" 
                                     , "Velocity_Solver_Tolerance" 
                                     , "Pressure_Tolerance" 
                                     , "Output_Time_Specification"
                                     , "Output_Time_Begin"
                                     , "Output_Time_End"
                                     , "Output_Time_Delta"
                                     , "Output_Cycle_Begin"
                                     , "Output_Cycle_End"
                                     , "Output_Cycle_Delta"
                                     , "Logging_Cycle_Delta"
                                     , "Dump_Cycle_Delta"
                                     , "Output_File_Prefix"
                                     , "Log_File_Name"
                                     , "Mesh_File_Name"
                                     , "Do_Restart"
                                     , "Do_RestartRegrid"
                                     , "Default_Dump_File_Name"
                                     , "Restart_Using_File_Name"
                                     , "Time_Evolution_Scheme"
                                     , "a-Posteriori_Tolerance" 
                                     , "a-Posteriori_ToleranceMult"
                                     , "a-Posteriori_ToleranceD1" 
                                     , "a-Posteriori_ToleranceMultD1"
                                     , "a-Posteriori_ToleranceD2" 
                                     , "a-Posteriori_ToleranceMultD2"
                                     , "Is_External_Mesh"
                                     , "Time_Derivative_Order"
                                     , "Advection_Order"
                                     , "Diffusion_Order"
                                     , "Fixed_TimeStep"
                                     , "Time_Step"
                                     , "Scale_TimeStep"
                                     , "Do_Adaption"
                                     , "Decay_Rate_Tolerance"
                                     , "Num_Spectral_Fit_Points"
                                     , "Do_Spectral_Check_AP"
                                     , "Do_1Derivative_Check_AP"
                                     , "Do_2Derivative_Check_AP"
                                     , "AMR_Cycle_Delta"
                                     , "Use_Fixed_Timestep"
                                     , "Timestep"
                   );
  // Get parameters that are in file, and reset the associated
  // values:
  if ( !mcmd.GetParams(fncmd_, main_blk_
                                       , &Courant_
                                       , &bUPC
                                       , &bPPC
                                       , &pci
                                       , &ssi
                                       , &u1iter
                                       , &piter
                                       , &u1tol
                                       , &ptol
                                       , &osi
                                       , &time_out_beg_
                                       , &time_out_end_
                                       , &time_out_skip_
                                       , &icycle_out_beg_
                                       , &icycle_out_end_
                                       , &icycle_out_skip_
                                       , &icycle_log_skip_
                                       , &icycle_dmp_skip_
                                       ,  fnout_
                                       ,  fnlog_
                                       ,  fnmesh_
                                       , &bRestart_ 
                                       , &bRegrid_
                                       ,  fndmp_
                                       ,  fnrst_
                                       , &iEvolType_
                                       , &ap_tol_
                                       , &ap_mult_
                                       , &ap_tol1_
                                       , &ap_mult1_
                                       , &ap_tol2_
                                       , &ap_mult2_
                                       , &bExtMesh
                                       , &itorder
                                       , &iaorder
                                       , &imorder
                                       , &bFixedTimeStep_
                                       , &fdt
                                       , &bScaledTimeStep_
                                       , &bDoAdapt_
                                       , &sig_tol_
                                       , &nSpFit_
                                       , &bSpectralAP_
                                       , &b1DerivativeAP_
                                       , &b2DerivativeAP_
                                       , &icycle_amr_skip_
                                       , &bFixedTimeStep_
                                       , &dt_
                                               ) ) {
    cout << "CmdFileParse: Error reading parameter file, " << fncmd_<< ": " << main_blk_ <<  endl;
    cout << "ParamReader error: " << mcmd.Error() << endl;
    return FALSE;
  }
  upc_type_    = (GPC)pci;
  stokes_type_ = (STOKES_TYPE)ssi;
  iStopCond_   = (STOPCONDTYPE)osi;
  u2iter       = u1iter;
  u2tol        = u1tol;
  time_max_    = time_out_end_;
  icycle_max_  = icycle_out_end_;
  iorderBDF_   = itorder;
  iorderadv_   = iaorder;
  iorderAM_    = imorder;
  
  if ( bFixedTimeStep_ ) dt_ = fdt;

  if ( !bRestart_ ) {
    mcmd.SetParams("%f %f %f %s %s"
                                       , "1-Viscosity" 
                                       , "2-Viscosity" 
                                       , "Density" 
                                       , "User_Parameter_File_Name"
                                       , "User_Parameter_Block_File_Name"
                     );
    if ( !mcmd.GetParams(fncmd_, aux_blk_
                                         , &nu_[0]
                                         , &nu_[1]
                                         , &rho_
                                         ,  fnuser_
                                         ,  suserblk_
                                                 ) ) {
      cout << "CmdFileParse: Error reading parameter file, " << fncmd_<< ": " << aux_blk_ <<  endl;
      cout << "ParamReader error: " << mcmd.Error() << endl;
      return FALSE;
    }
  }

  return TRUE;
} // end of method GCmdFileParse


//************************************************************************************
//************************************************************************************
// METHOD     : GCmdLineParse
// DESCRIPTION: Parses command line args.
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GCmdLineParse(int argc, char *argv[])
{
  GSHORT   i, j, n=0;
  GBOOL    isok=FALSE;

  // Find dimensionality; first one found is 
  // selected:
  j = 0;
  while ( j<argc && strcmp(argv[j],"-d")!=0 ) j++;

  nBdyCond_ = 0; 
  j = 1;
  while ( j<argc ) {
    if      ( strcmp(argv[j],"-Nx")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       Ne[0] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-Ny")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       Ne[1] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-xNx")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       xN[0] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-xNy")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       xN[1] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-P0")==0 ) {
       isok = TRUE;
       if ( j+nd_ >= argc ) return FALSE;
//     for ( i=1; i<=nd_; i++ ) isok = isok && isdigit(*argv[j+i]);
       if ( !isok ) return FALSE;
       for ( i=1; i<=nd_; i++ ) P0_[i-1] = atof(argv[j+i]);
//     bP0 = TRUE;
       B0_[0][0] = P0_[0]; B0_[0][1] = P0_[1]; B1_[0][0] = P1_[0]; B1_[0][1] = P0_[1];
       B0_[1][0] = P1_[0]; B0_[1][1] = P0_[1]; B1_[1][0] = P1_[0]; B1_[1][1] = P1_[1];
       B0_[2][0] = P1_[0]; B0_[2][1] = P1_[1]; B1_[2][0] = P0_[0]; B1_[2][1] = P1_[1];
       B0_[3][0] = P0_[0]; B0_[3][1] = P1_[1]; B1_[3][0] = P0_[0]; B1_[3][1] = P0_[1];
       j += (nd_+1);
    }
    else if ( strcmp(argv[j],"-P1")==0 && j+1 < argc  ) {
       isok = TRUE;
       if ( j+nd_ >= argc ) return FALSE;
//     for ( i=1; i<=nd_; i++ ) isok = isok && isdigit(*argv[j+i]);
       if ( !isok ) return FALSE;
       for ( i=1; i<=nd_; i++ ) P1_[i-1] = atof(argv[j+i]);
//     bP1 = TRUE;
       B0_[0][0] = P0_[0]; B0_[0][1] = P0_[1]; B1_[0][0] = P1_[0]; B1_[0][1] = P0_[1];
       B0_[1][0] = P1_[0]; B0_[1][1] = P0_[1]; B1_[1][0] = P1_[0]; B1_[1][1] = P1_[1];
       B0_[2][0] = P1_[0]; B0_[2][1] = P1_[1]; B1_[2][0] = P0_[0]; B1_[2][1] = P1_[1];
       B0_[3][0] = P0_[0]; B0_[3][1] = P1_[1]; B1_[3][0] = P0_[0]; B1_[3][1] = P0_[1];
       j += (nd_+1);
    }
    else if ( strcmp(argv[j],"-spectralap")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       bSpectralAP_ = (GBOOL)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-deriv1ap")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       b1DerivativeAP_ = (GBOOL)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-deriv2ap")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       b2DerivativeAP_ = (GBOOL)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-apmult")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ap_mult_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-apmult1")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ap_mult1_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-apmult2")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ap_mult2_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-aptol")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ap_tol_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-aptol1")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ap_tol1_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-aptol2")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ap_tol2_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-u1tol")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       u1tol = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-u2tol")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       u2tol = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-utol")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       u1tol = u2tol = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ptol")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       ptol = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-u1iter")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       u1iter = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-u2iter")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       u2iter = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-uiter")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       u1iter = u2iter = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-piter")==0 && j+1 < argc  ) {
//     if ( !isdigit(*argv[j+1]) ) return FALSE;
       piter = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-cycles")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_max_ = atoi(argv[j+1]);
       icycle_out_end_ = icycle_max_;
       iStopCond_ = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-time")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_max_     = atof(argv[j+1]);
       time_out_end_ = time_max_;
       iStopCond_ = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-uf")==0 && j+1 < argc  ) {
       strncpy(fnuser_,argv[j+1], FILE_NAME_MAX);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ub")==0 && j+1 < argc  ) {
       strncpy(suserblk_,argv[j+1], FILE_NAME_MAX);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-if")==0 && j+1 < argc  ) {
       strncpy(fncmd_,argv[j+1], FILE_NAME_MAX);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-of")==0 && j+1 < argc  ) {
       strncpy(fnout_,argv[j+1], FILE_NAME_MAX);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-lf")==0 && j+1 < argc  ) {
       strncpy(fnlog_,argv[j+1], FILE_NAME_MAX);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-r")==0  ) {
       bRestart_  = TRUE;
       j += 1 ;
    }
    else if ( strcmp(argv[j],"-rg")==0  ) {
       bRestart_ = TRUE;
       bRegrid_  = TRUE;
       j += 1 ;
    }
    else if ( strcmp(argv[j],"-pgrid")==0  ) {
       bPGrid_ = TRUE;
       j += 1 ;
    }
    else if ( strcmp(argv[j],"-noadvect")==0  ) {
       bDoAdvection_ = FALSE;
       j += 1 ;
    }
    else if ( strcmp(argv[j],"-no_outongridchg")==0  ) {
       bOutOnGridChange_ = FALSE;
       j += 1 ;
    }
    else if ( strcmp(argv[j],"-linadvect")==0  ) {
       bLinAdvection_ = TRUE;
       j += 1 ;
    }
    else if ( strcmp(argv[j],"-bal")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       bDoLoadBalancing_ = (GBOOL)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-filter")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       bFilter_ = (GBOOL)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-filter_strength")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       filter_alpha_ = (GDOUBLE)atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-filter_delta")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       filter_delta_ = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-nr")==0  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nrefinelevels_ = (atoi(argv[j+1]));
       bDoAdapt_ = nrefinelevels_ == 0 ? FALSE : TRUE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-rc")==0  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       bReadCmdFile_ = (GBOOL) (atoi(argv[j+1]));
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-rf")==0 && j+1 < argc  ) {
       strncpy(fnrst_,argv[j+1], FILE_NAME_MAX);
       bRestart_ = TRUE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-df")==0 && j+1 < argc  ) {
       strncpy(fndmp_,argv[j+1], FILE_NAME_MAX);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-dcd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_dmp_skip_ = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-lcd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_log_skip_ = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-otb")==0 && j+1 < argc  ) {
       //if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_out_beg_ = atof(argv[j+1]);
       iStopCond_ = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ote")==0 && j+1 < argc  ) {
       //if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_out_end_ = atof(argv[j+1]);
       time_max_     = time_out_end_;
       iStopCond_ = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-otd")==0 && j+1 < argc  ) {
       //if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_out_skip_ = atof(argv[j+1]);
       iStopCond_ = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ocb")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_out_beg_ = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-oce")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_out_end_ = atoi(argv[j+1]);
       icycle_max_     = icycle_out_end_;
       iStopCond_ = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ocd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_out_skip_ = MAX(atoi(argv[j+1]),1);
       iStopCond_ = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-acb")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_amr_beg_ = atoi(argv[j+1]);
       iAMRCond_       = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-acd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_amr_skip_ = MAX(atoi(argv[j+1]),1);
       iAMRCond_        = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ace")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_amr_last_ = MAX(atoi(argv[j+1]),1);
       iAMRCond_        = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-atb")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_amr_beg_ = atoi(argv[j+1]);
       iAMRCond_     = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-atd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_amr_skip_ = atoi(argv[j+1]);
       iAMRCond_      = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-ate")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_amr_last_ = atoi(argv[j+1]);
       iAMRCond_      = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-bcb")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_lbal_beg_ = atoi(argv[j+1]);
       iLBalCond_       = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-bcd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_lbal_skip_ = MAX(atoi(argv[j+1]),1);
       iLBalCond_       = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-bce")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       icycle_lbal_last_ = MAX(atoi(argv[j+1]),1);
       iLBalCond_       = BY_CYCLE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-btb")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_lbal_beg_ = atoi(argv[j+1]);
       iLBalCond_     = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-btd")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_lbal_skip_ = atoi(argv[j+1]);
       iLBalCond_      = BY_TIME;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-bte")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       time_lbal_last_ = atoi(argv[j+1]);
       iLBalCond_      = BY_TIME;
       j += 2 ;
    }

    else if ( strcmp(argv[j],"-m")==0 && j+1 < argc  ) {
       strncpy(fnmesh_,argv[j+1], FILE_NAME_MAX);
       bExtMesh = TRUE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-nu")==0 && j+1 < argc  ) {
       nu_[0] = nu_[1] = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-nu1")==0 && j+1 < argc  ) {
       nu_[0] = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-nu2")==0 && j+1 < argc  ) {
       nu_[1] = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-nfit")==0 && j+1 < argc  ) {
       nSpFit_ = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-rho")==0 && j+1 < argc  ) {
       rho_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-aux")==0 && j+1 < argc  ) {
       aux_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-aux0")==0 && j+1 < argc  ) {
       aux0_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-aux1")==0 && j+1 < argc  ) {
       aux1_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-c")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       Courant_ = atof(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-dd")==0 && j+1 < argc  ) {
       strncpy(sgdd_, argv[j+1], MAX_GDD_STRING);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-dt")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       dt_ = atof(argv[j+1]);
       bFixedTimeStep_ = TRUE;
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-bscaled_dt")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       bScaledTimeStep_ = (GBOOL)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-adv")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       iorderadv_ = atoi(argv[j+1]);
       j += 2; 
    }
    else if ( strcmp(argv[j],"-bdf")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       iorderBDF_ = atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-et")==0 && j+1 < argc  ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       iEvolType_ = (TIME_EVOLTYPE)atoi(argv[j+1]);
       j += 2 ;
    }
    else if ( strcmp(argv[j],"-upc")==0 ) {
       bUPC = TRUE;
       j += 1;
    }
    else if ( strcmp(argv[j],"-upctype")==0 ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       bUPC =  TRUE;
       upc_type_ = (GPC)atoi(argv[j+1]);
       j += 2;
    }
    else if ( strcmp(argv[j],"-stokestype")==0 ) {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       stokes_type_ = (STOKES_TYPE)atoi(argv[j+1]);
       j += 2;
    }
    else if ( strcmp(argv[j],"-noupc")==0 ) {
       bUPC = FALSE ;
       j += 1;
    }
    else if ( strcmp(argv[j],"-ppc")==0 ) {
       bPPC = TRUE;
       j += 1;
    }
    else if ( strcmp(argv[j],"-noppc")==0 ) {
       bPPC = FALSE;
       j += 1;
    }
    else if ( strcmp(argv[j],"-noadapt")==0 ) {
       bDoAdapt_ = FALSE;
       j += 1;
    }
    else if ( strcmp(argv[j],"-dealias")==0 ) {
       bDoDealiasing_ = (GBOOL)atoi(argv[j+1]);
       j += 2;
    }
    else if ( strcmp(argv[j],"-bdy")==0 ) {
       n = 2*nd_ - (nd_==1 ? 1 : 0);
       if ( j+n+1 >= argc ) return FALSE;
//     for ( i=1,isok=TRUE; i<=n+1; i++ ) {
//       isok = isok && isdigit(argv[j+i][0]) ;
//     }
//     if ( !isok ) return FALSE;
       if ( nBdyCond_ < 2*nd_ )
       {
         for ( i=1; i<=nd_; i++ ) B0_[nBdyCond_][i-1] = atof(argv[j+i]);
         if ( nd_ > 1 ) 
         {
           for ( i=nd_+1; i<=2*nd_; i++ ) B1_[nBdyCond_][i-nd_-1] = atof(argv[j+i]);
         }
         btype_[nBdyCond_] = (BDYTYPE)atoi(argv[j+n+1]); 
         nBdyCond_++;
       }
       j += (n+2);
    }
    else if ( strcmp(argv[j],"-h")==0  ) {
       GPrtHelp(argc, argv, cout);
       exit(0);
    }
    else {
       cout << "Incorrect option syntax: " << argv[j] << endl << endl;;
       GPrtHelp(argc, argv, cout);
       exit(1);
    }
  }

//nelems_ = Ne[0] * Ne[1];

  return TRUE;
} // end of function GCmdLineParse


//************************************************************************************
//************************************************************************************
// METHOD     : GPrtHelp
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void GPrtHelp(GSHORT argc, char **argv, ostream &pout)
{
  pout << "Enter:" <<endl << endl;
  pout << " " << argv[0]  
       <<        " [{-Nx}{-Ny} #Elems] [{-xNx}{-xNy} Exp_Order] [-P0 x0 {y0}] [-P1 x1 {y1}] [-(no)u1(2)pc]"     << endl;
  pout << "        [-(no)ppc] [-upctype Type] [-stokestype Type] [-spectralap YN] [-deriv1ap YN] [deriv2ap YN]" << endl;
  pout << "        [-aptol[1][2] Tol] [-apmult[1][2] Mult] [-p(u)iter #Iter] [-p(u)tol Tol]"                    << endl;
  pout << "        [-bdy Face# Type] [-c Courant#] [-dt TimeStep] [-dd Text] [-nu val] [-nu1 val] [-nu2 val]"   << endl;
  pout << "        [-rho val] [-time MaxTime] [-cycles MaxCycles] [-r] [-rg] [pgrid] [-rc YN] [-noadvect] [-linadvect]" << endl;
  pout << "        [-bal Y/N] [-filter Y/N] [-filter_strength alpha] [-filter_delta d] [-no_outongridchg]"      << endl; 
  pout << "        [-ocb OutCyc_Beg] [-oce OutCyc_End] [-ocd OutCyc_Skp] [-otb OutTim_Beg] [-ote OutTim_End]"   << endl;
  pout << "        [-otd OutTim_Skp] [-acd AMRCyc_Skp] [-atb AMRCyc_Beg] [-atd AMRTim_Skp] [-atb AMRTim_Beg]"   << endl;
  pout << "        [-lcd LogCyc_Skp] [-dcd DmpCyc_Skp] [-nfit NspFit] [-sigtol Tol] [-if Cmd_File]" << endl;
  pout << "        [-of Out_File] [-lf Log_File] [-uf User_File] [-rf Rst_File] [-df Dmp_File]" << endl;
  pout << "        [-m Mesh_File] [-ub BlkName] [-noadapt] [-dealias Y/N] [-nr nLevels] [-h]"              << endl;

       pout << endl << " Where: " << endl;
       pout << "      -Nx #Elems      : Number of elements in x-direction. Default is 1." << endl;
       pout << "      -Ny #Elems      : Number of elements in y-direction. Defaulst is 1." << endl;
       pout << "      -xNx Exp. Order : Expansion order in x-direction. Default is 4." << endl;
       pout << "      -xNy Exp. Order : Expansion order in y-direction. Default is 4." << endl;
       pout << "      -P0 x0 {y0}     : Bottom left corner. There must be dim of these. Default is (0,0,0)." << endl;
       pout << "      -P1 x0 {y0}     : Top right diagonal corner. There must be dim of these. Default is (1, 1,1)." << endl;
       pout << "      -(no)upc        : Use (don't use) preconditioner for u1 and u2." << endl;
       pout << "      -(no)ppc        : Use (don't use) preconditioner for p." << endl;
       pout << "      -upctype Type   : Use preconditioner type Type for u1 and u2. Type may be:" << endl; 
       pout << "                          0: GPC_BLOCKJAC_HELM" << endl;
       pout << "                          2: GPC_POINTJAC_HELM" << endl;
       pout << "                          3: GPC_NONE" << endl;
       pout << "                          Default is GPC_NONE." << endl;
       pout << "      -stokestype Type: Use Stokes solver type Type. Type may be:" << endl; 
       pout << "                          0: STOKES_SCHUR_DELP--Perot decomp. with pressure correction" << endl;
       pout << "                          1: STOKES_UZAWA     --Uzawa splitting " << endl;
       pout << "      -spectralap YN  : Use spectral a-posteriori error criterion (1); else don't (0). Default is 1." << endl;
       pout << "      -deriv1ap   YN  : Use 1-derivative a-posteriori error criterion (1); else don't (0). Default is 1." << endl;
       pout << "      -deriv2ap   YN  : Use 2-derivative a-posteriori error criterion (1); else don't (0). Default is 1." << endl;
       pout << "      -aptol     Tol  : Set spectral error refinement tolerance." << endl;
       pout << "      -apmult   Mult  : Set spectral coarsening scaling factor." << endl;
       pout << "      -aptol1    Tol  : Set 1-derivative refinemtn tolerance." << endl;
       pout << "      -apmult1  Mult  : Set 1-derivative coarsening scaling factor." << endl;
       pout << "      -aptol2    Tol  : Set 2-derivative refinemtn tolerance." << endl;
       pout << "      -apmult2  Mult  : Set 2-derivative coarsening scaling factor." << endl;
       pout << "      -p(u)iter Iter  : Max no. iterations for pressure (velocity) solve. " << endl;
       pout << "      -p(u)tol Tol    : Error tolerance for pressure (velocity) solve. " << endl;
       pout << "      -bdy Seg Type   : Set boundary condition Type on face (edge, point) given by Seg." << endl;
       pout << "                        Seg is given in 1d by X0; in 2d by segment endpoints X0 Y0 X1 Y1." << endl; 
       pout << "                        Type is one of: 0 (Dirichlet)," << endl;
       pout << "                        1 (Neumann), 2 (Periodic), 3 (No-slip), 4 (None). Default is 4. " << endl;
       pout << "      -c   Courant#   : Set Courant number to Courant. " << endl;
       pout << "      -dt  TimeStep   : Set time step to TimeStep. Courant number not used. " << endl;
       pout << "      -bscaled_dt y/n : If fixed time step, scale it due to refinement or not (0 or 1). Default is 1. " << endl;
       pout << "      -dd  Text       : Pass Text to grid partitioner" << endl;
       pout << "      -ab  N          : Set Adams-Bashforth order for adv. term (1, 2, 3, or 4)" << endl;
       pout << "      -ext N          : Set extrapolation order for adv. term (1, 2, or 3)" << endl;
       pout << "      -bdf N          : Set BDF order for derivative (1, 2, 3, or 4)" << endl;
       pout << "      -et  N          : Set time stepping method: 0=> OIFS; 1=>ABBDF; 2=>EXBDF." << endl;
       pout << "      -nu  val        : Set kinematic viscosity to val." << endl;
       pout << "      -nu1  val       : Set 1- kinematic viscosity to val." << endl;
       pout << "      -nu2  val       : Set 2- kinematic viscosity to val." << endl;
       pout << "      -rho val        : Set density to val. " << endl;
       pout << "      -time val       : Set max evolution time to val. " << endl;
       pout << "      -cycles val     : Set max evolution cycles to val. " << endl;
       pout << "      -r              : Do a restart with no regridding (default restart file is 'gaspar.dmp')." << endl;
       pout << "      -rg             : Do a restart with a regrid (default restart file is 'gaspar.dmp')." << endl;
       pout << "      -pgrid          : Construct a PERIODIC [0,1]^2  grid with Nx x Ny elements, each" << endl;
       pout << "                        of expansion order (xNx x xNy)." << endl;
       pout << "      -rc             : Read command file on startup? (0, 1; default is 1)." << endl;
       pout << "      -noadvect       : Do not perform advection." << endl;
       pout << "      -no_outongridchg: Do not do SDS output when grid changes." << endl;
       pout << "      -linadvect      : Do linear advection with specified advection velocity." << endl;
       pout << "      -bal Y/N        : Do load balancing (1/0). Default=0." << endl;
       pout << "      -filter Y/N     : Use filtering? (1/0). Default=0." << endl;
       pout << "      -filter_strength alpha" << endl;
       pout << "                      : Set filter strength = alpha. Default=0.0." << endl;
       pout << "      -filter_delta d : Set filter smoothing delta = d.Default=1." << endl;
       pout << "      -ocb OutCyc_Beg : Output cycle-begin." << endl;
       pout << "      -oce OutCyc_End : Output cycle-end." << endl;
       pout << "      -ocd OutCyc_Skp : Output cycle-delta." << endl;
       pout << "      -otb OutTim_Beg : Output begin time." << endl;
       pout << "      -ote OutTim_End : Output endtime." << endl;
       pout << "      -otd OutTim_Skp : Output time-delta." << endl;
       pout << "      -acd AMRCyc_Skp : AMR cycle-delta." << endl;
       pout << "      -acb AMRCyc_Beg : AMR cycle-begin." << endl;
       pout << "      -ace AMRCyc_End : AMR cycle-end." << endl;
       pout << "      -atd AMRTim_Skp : AMR time-delta." << endl;
       pout << "      -atb AMRTim_Beg : AMR time-begin." << endl;
       pout << "      -ate AMRTim_End : AMR time-end." << endl;
       pout << "      -bcd LBalCyc_Skp: Load balance cycle-delta." << endl;
       pout << "      -bcb LBalCyc_Beg: Load balance cycle-begin." << endl;
       pout << "      -bce LBalCyc_End: Load balance cycle-end." << endl;
       pout << "      -btd LBalTim_Skp: Load balance time-delta." << endl;
       pout << "      -btb LBalTim_Beg: Load balance time-begin." << endl;
       pout << "      -bte LBalTim_End: Load balance time-end." << endl;
       pout << "      -lcd LogCyc_Skp : Logging cycle-delta." << endl;
       pout << "      -dcd DmpCyc_Skp : Dump cycle-delta." << endl;
       pout << "      -nfit nSpFit    : Number of coefficients to use in spectral fit." << endl;
       pout << "      -sigtol Tol     : Spectral decay rate tolerance." << endl;
       pout << "      -if Cmd_File    : Get system options from Cmd_File, instead of from gaspar.dat (default)." << endl;
       pout << "      -of Outut_File  : Place output in Output_File. Default is results.######." << endl;
       pout << "      -lf Log_File    : Log to file Log_File. Default is gaspar.log." << endl;
       pout << "      -uf User_File   : Get user data from User_File. Default is gaspar.user." << endl;
       pout << "      -rf Restart_File: Use 'Restart_File' as restart file. Default is gaspar.dmp." << endl;
       pout << "      -df Dump_File   : Change default dump file name from 'gaspar.dmp' to Dump_File." << endl;
       pout << "      -m Mesh_File    : Use 'Mesh_File' to generate the mesh." << endl;
       pout << "      -ub BlkName     : Use BlkName as parameter block in the command file, instead of UserBlk (default)." << endl;
       pout << "      -noadapt        : Turn off grid adaption." << endl;
       pout << "      -dealias        : Set Dealias flag (0,1). Default is 0." << endl;
       pout << "      -nr nLevels     : Maximum number of refinement levels" << endl;
       pout << "      -h              : Print this help text." << endl;
       pout << endl << endl;

} // end of function PrtHelp


//************************************************************************************
//************************************************************************************
// METHOD     : GWrapup
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GWrapup()
{
  ofstream os;

  if ( rank == 0 ) {
    os.open(fnlog_,ios::app);
    if ( bConvError_ ) {
      os << "#gaspar: Convergence error: cycle:" << icycle_ << endl;
      os << "#       ***********> Convergence error"<< endl;
    }
    if ( bStepError_ ) {
      os << "#gaspar: Step failed: cycle:" << icycle_ << endl;
      os << "#       ***********> Stepper error" << endl;
    }
    if ( bAdaptError_ ) {
      os << "#gaspar: Adaption failed: cycle:" << icycle_ << endl;
      os << "#       ***********> Stepper error" << endl;
    }

    if ( bStepError_ || bConvError_ || bAdaptError_ )
      os   << "#" << endl << "#" << endl
           << "Run terminated after " << icycle_ << " cycles. " << endl
           << "Evolution time: " << time_ << endl;
    else
      os   << "#" << endl << "#" << endl
           << "#" << icycle_ << " cycles completed. " << endl
           << "#Evolution time: " << time_ << endl;
    os.close();
  }   
  GDeleteDynamic();

} // end of method GWrapup


//************************************************************************************
//************************************************************************************
// METHOD     : GTimeStep
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GTimeStep(GINT ilevel)
{ 
  GBOOL bRet;

  if ( bFixedTimeStep_ ) {
    if ( !bScaledTimeStep_ ) bRet = TRUE; 
    else                     bRet = GScaledTimeStep();
  }
  else {
    bRet = GCourantTimeStep(ilevel);
  }
  if ( !bRet ) {
    GDoLog(0,"#GTimeStep failure: error computing timestep");
    iGError_ = GERR_TIMESTEP;
    return FALSE;
  }
  
  if ( dt_ < dtMin_ ) {
    GDoLog(0,"#GTimeStep failure: time step too small");
    iGError_ = GERR_TIMESTEP;
    return FALSE;
  }


  return TRUE;

} // end of method GTimeStep


//************************************************************************************
//************************************************************************************
// METHOD     : GScaledTimeStep
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GScaledTimeStep()
{
  GDOUBLE    gmin=gMinElemLength_, grat;

  if ( bDoAdapt_ ) gmin = GUtils::ComputeMinLength(uelems);
  grat = gmin / gMinElemLength_;
  if ( grat == 1.0 ) return TRUE;
  gMinElemLength_ = gmin;

  dt_ *= grat;

  return TRUE;
} // end of GScaledTimeStep


//************************************************************************************
//************************************************************************************
// METHOD     : GCourantTimeStep
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GCourantTimeStep(GINT ilevel)
{
  GINT       i, j, k, n, NN1, P1, P2;
  GDOUBLE    dx, dy, dxm, aux, auy, 
             dtin=-SEHUGE, gdtin, vmax, dtinu, dtiad;
  GDOUBLE    numax;
  GVector    *x, *y, *vu=NULL; 
  Field2D    *field; 

  numax = max(nu_[0],nu_[1]);

  for ( k=0, dtin=0.0; k<ncourflds_; k++ ) {
    courflds_[k]->start(NULL);
    for ( n=0; n<nelems_; n++ ) {
      field = courflds_[k]->member();
      P1  = field->GetElement()->GetOrder(1);
      P2  = field->GetElement()->GetOrder(2);
//    vertices = field->GetElement()->GetSpVertices();
//    L1  = fabs((vertices+1)->x1 - (vertices)->x1);
//    L2  = fabs((vertices+3)->x2 - (vertices)->x2);
//    Lmin = MIN(L1,L2);
      x   = field->GetElement()->GetSpNodes(1);
      y   = field->GetElement()->GetSpNodes(2);
      vu  = field->GetExpCoeffs(ilevel);
      NN1 = P1 + 1;
//    NN2 = P2 + 1;
#if 1
      for ( j=0; j<P2; j++ ) {
        for ( i=0; i<P1; i++ ) {
          dx       = fabs((*x)(i+1+NN1*j  ) - (*x)(i+NN1*j)) + TINY;
          dy       = fabs((*y)(i+NN1*(j+1)) - (*y)(i+NN1*j)) + TINY;
          aux      = bDoAdvection_ ? 0.5*((*vu)(i+1+NN1*j) + (*vu)(i+NN1*j)) : 0.0;
          auy      = bDoAdvection_ ? 0.5*((*vu)(i+NN1*(j+1)) + (*vu)(i+NN1*j)) : 0.0;
          vmax     = MAX(fabs(aux),fabs(auy));
          dxm      = MIN(dx,dy);
//        dtinu    = bDoAdvection_ && numax < 0.1 ? 0.0 : fabs(4.0*numax/dxm)/dxm;
          dtinu    = fabs(4.0*numax/dxm)/dxm;
          dtiad    = bDoAdvection_ ? fabs(vmax/dxm) : 0.0;
          dtin     = MAX( dtin, dtiad+dtinu );
#if 0
          cout << "GCourantTimeStep: cycle=" << icycle_ << " vmax=" << vmax 
               << " dtinu=" << dtinu << " dtiad=" << dtiad << " dtin=" << dtin << endl;
#endif
        }
      }
#endif
//    dtin = max( dtin, P1*P2*P1*P2*numax/(Lmin*Lmin));
//    dtin = max( dtin, P1*P2*numax/Lmin);
      courflds_[k]->next();
    }
  }
  
  GComm::Allreduce(&dtin, &gdtin, 1, GC_GDOUBLE, G_OP_MAX);
  dt_ = Courant_/(gdtin + TINY);
//cout << "TimeStep: cycle=" << icycle_ << " gdtin=" << gdtin << " dt=" << dt_ << " Cour=" << Courant_ << endl;


  return TRUE;

} // end of method GTimeStep


//************************************************************************************
//************************************************************************************
// METHOD     : GOutCond
// DESCRIPTION: Provides boolean check of output condition
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GOutCond()
{   
  GBOOL bRet=FALSE;
   
  if ( bOutOnGridChange_ && bElemListChange_ ) {
    return TRUE;
  }
 
  switch (iStopCond_) { 
    case BY_CYCLE:
        if ( icycle_ >= icycle_out_beg_ && icycle_ <= icycle_out_end_ &&
             icycle_ % icycle_out_skip_ == 0 ) bRet = TRUE;
      break;
    case BY_TIME:
      if ( time_ >= time_out_beg_ && time_ <= time_out_end_&&
           ( ( time_ >  time_out_last_+time_out_skip_-dt_   &&
               time_ <= time_out_last_+time_out_skip_+dt_ ) ||
               time_ == time_out_beg_ || 
               time_ == time_out_end_) )  bRet = TRUE;
      break;     
    default:     
      bRet = FALSE;
  }

  return bRet;   
} // end of method GOutCond


//************************************************************************************
//************************************************************************************
// METHOD     : GLBalCond
// DESCRIPTION: Provides boolean check of load balancing condition
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GLBalCond()
{
  GBOOL bRet=FALSE;

  switch (iLBalCond_) {
    case BY_CYCLE:
        if ( icycle_ >= icycle_lbal_beg_ &&
             icycle_ % icycle_lbal_skip_ == 0 ) bRet = TRUE;
      break;
    case BY_TIME:
      if ( time_ >= time_lbal_beg_-time_lbal_skip_  &&
           time_ >= time_lbal_last_+time_lbal_skip_  )  bRet = TRUE;
      break;
    default:
      bRet = FALSE;
  }

  return bRet;
} // end of method GLBalCond


//************************************************************************************
//************************************************************************************
// METHOD     : GAMRCond
// DESCRIPTION: Provides boolean check of AMR condition
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GAMRCond()
{
  GBOOL bRet=FALSE;

  switch (iAMRCond_) {
    case BY_CYCLE:
        if ( icycle_ >= icycle_amr_beg_ && 
             icycle_ % icycle_amr_skip_ == 0 ) bRet = TRUE;
      break;
    case BY_TIME:
      if ( time_ >= time_amr_beg_-time_amr_skip_  && 
           time_ >= time_amr_last_+time_amr_skip_  )  bRet = TRUE;
      break;
    default:
      bRet = FALSE;
  }

  return bRet;
} // end of method GAMRCond


//************************************************************************************
//************************************************************************************
// METHOD     : GStopCond
// DESCRIPTION: Provides boolean check of stopping condition
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GStopCond()
{
  GBOOL bRet=FALSE;

  switch (iStopCond_) {
    case BY_CYCLE:
      if ( icycle_ >= icycle_max_ ) bRet = TRUE;
      break;
    case BY_TIME:
      if ( time_+dt_ >  time_max_ ) bRet = TRUE;
      break;
    default:
      bRet = FALSE;
  }

  return bRet;
} // end of method GStopCond


//************************************************************************************
//************************************************************************************
// METHOD     : GDoLog
// DESCRIPTION: Updates log file
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GDoLog(GINT num, char *msg, ...)
{
  va_list     ap;
  GINT        i=0, j, ns, ne;
  GBOOL       bPrintHeader=FALSE, bPrintDynamic=FALSE, bPrintStatic=FALSE, bPrintMsg=FALSE;
  GDOUBLE     T0, T1;
  char       *pa, stmp[FILE_NAME_MAX+1];
  ofstream    os;

  T0 = STK::CTimer(); 
  T1 = STK::Timer ();

  if ( msg != NULL && strlen(msg) > 0  ) bPrintMsg     = TRUE;

  va_start(ap, msg);
  while ( i < num ) {
    pa = va_arg(ap,char*);
    strncpy(stmp,pa,FILE_NAME_MAX);
    for ( j=0; j<strlen(stmp); j++ ) stmp[j]=tolower(stmp[j]);
    if ( strcmp("header" ,stmp) == 0 ) bPrintHeader  = TRUE;
    if ( strcmp("dynamic",stmp) == 0 ) bPrintDynamic = TRUE;
    if ( strcmp("static" ,stmp) == 0 ) bPrintStatic  = TRUE;
    i++;
  }
  va_end(ap);

//if ( icycle_ != 0 && icycle_log_last_ == icycle_  && !bRestart ) return TRUE;  // already logged.

  if ( !GUserLogUpdate() ) {
    cout << "GDoLog: Error in GUserLogUpdate" << endl;
    return FALSE;
  }

  for ( j=0; j<nEvolvedFields_; j++ ) {
    ncgit_[j] = stepper->GetSolver(j+1)->GetNumIterations();
    errcg_[j] = stepper->GetSolver(j+1)->GetError();
  }
#if defined(NAVIER_STOKES)
  divvmax_ = stepper->GetDivMax(0);
#endif
  ns = nEvolvedFields_;
  ne = nEvolvedFields_;

  GComm::Allreduce(errcg_   , gerrcg_  , ne, GC_GDOUBLE, G_OP_MAX);
  GComm::Allreduce(ncgit_   , gncgit_  , ns, GC_GINT , G_OP_MAX);

  if ( rank == 0 ) {
    os.open(fnlog_,ios::app);
    if ( bPrintMsg ) 
      os << msg << endl;
    if ( bPrintStatic )   
      glogger_.PutStaticData(os);
    if ( bPrintHeader )   
      glogger_.PutHeader(os);
    if ( bPrintDynamic )
      glogger_.PutDynamicData(os);
    os.close();
  }
  icycle_log_last_ = icycle_;
  GComm::Synch();

  TCLog_ += (STK::CTimer()-T0);         // already in sec
  TWLog_ += ((STK::Timer ()-T1)/1.0e6); // in sec

  return TRUE;
} // end of method GDoLog


//************************************************************************************
//************************************************************************************
// METHOD     : GGlobalAssembly
// DESCRIPTION: do global assembly of certain operators
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GGlobalAssembly()
{
  GUtils::ComputeMasks(uelems, glop, hDSOp, utmp);
  GUtils::ComputeGMass(uelems, glop, hDSOp, utmp);

  return TRUE;

} // end of method GGlobalAssembly


//************************************************************************************
//************************************************************************************
// METHOD     : GGlobalizeBdy
// DESCRIPTION: make certain that the global boundaries are
//              consistent with local bdy nodes. NOTE: u1 list
//              must exist, and the boundary quantities must have
//              been set prior to entry.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GGlobalizeBdy()
{
  GINT       i, j, m;
  GIBuffer   *bi;
  GBTBuffer  *bt;
  GVector   *vm;
  GVecList   diag;

  if ( glop == NULL ) {
    cout << "GlobalizeBdy: NULL comm operator" << endl;
    return FALSE;
  }
  
  // Do DIRICHLET and NOSLIP (a form of DIRICHLET) nodes
  // separately...

  // Find 'mask' for DIRICHLET nodes, contained in 'diag' list:
  bdyindices.start(NULL);
  bdytype   .start(NULL);
  u1        .start(NULL);  // needed for dimensions only
  for ( i=0, m=0; i<nelems_; i++ ) {
    diag.add(NULL,u1.member()->GetExpCoeffs(0)->dim(),TRUE);
    bi  = bdyindices.member();
    bt  = bdytype   .member();
    vm  = diag.member();
    *vm = 1.0;
    for ( j=0; j<bt->dim(); j++ ) {
      if ( (*bt)(j) == DIRICHLET ) (*vm)((*bi)(j)) = 0.0;
    }
    m += vm->dim();
    bdyindices.next();
    bdytype   .next();
    u1        .next();
  }
//cout << endl << endl;

  // Do gather/scatter multiply operation to find DIRICHLET nodes:
  if ( !glop->DSOp(diag, G_OP_PROD, hDSOp) ) {
    cout << "GlobalizeBdy: Unable to perform direct stiffness multiply (DIRICHLET nodes)" << endl;
    return FALSE;
  }

  bdyindices.start(NULL);
  bdytype   .start(NULL);
  diag      .start(NULL);
  for ( i=0,m=0; i<nelems_; i++ ) {
    bi = bdyindices.member();
    bt = bdytype   .member();
    vm = diag.member();
    for ( j=0; j<bt->dim(); j++ ) {
      if ( (*vm)((*bi)(j)) == 0.0 ) (*bt)(j) = DIRICHLET;
    }
    m += vm->dim();
    bdyindices.next();
    bdytype   .next();
    diag      .next();
  }

  // ...Now, same thing for NOSLIP nodes:

  // Find 'mask' for NOSLIP nodes, contained in 'diag' list:
  bdyindices.start(NULL);
  bdytype   .start(NULL);
  diag      .start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    bi  = bdyindices.member();
    bt  = bdytype   .member();
    vm  = diag.member();
    *vm = 1.0;
    for ( j=0; j<bt->dim(); j++ ) {
      if ( (*bt)(j) == NOSLIP ) (*vm)((*bi)(j)) = 0.0;
    }
    bdyindices.next();
    bdytype   .next();
    diag      .next();
  }

  // Do gather/scatter multiply operation to find NOSLIP nodes:
  if ( !glop->DSOp(diag, G_OP_PROD, hDSOp) ) {
    cout << "GlobalizeBdy: Unable to perform direct stiffness multiply (NOSLIP nodes)" << endl;
    return FALSE;
  }

  bdyindices.start(NULL);
  bdytype   .start(NULL);
  diag      .start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    bi = bdyindices.member();
    bt = bdytype   .member();
    vm = diag.member();
    for ( j=0; j<bt->dim(); j++ ) {
      if ( (*vm)((*bi)(j)) == 0.0 ) (*bt)(j) = NOSLIP;
    }
    bdyindices.next();
    bdytype   .next();
    diag      .next();
  }

  return TRUE;
} // end of method GGlobalizeBdy


//************************************************************************************
//************************************************************************************
// METHOD     : GResetExpandables
// DESCRIPTION: 
// ARGUMENTS  :  
// RETURNS    : TRUE if ok; else FALSE
//************************************************************************************
GBOOL GResetExpandables()
{ 
  GINT     i, j, NN;

//  cout << "GResetExpandables called..." << endl;
  u1bdyvals   .empty();
  u2bdyvals   .empty();
  utmp        .empty();
  ptmp        .empty();
  uptmp       .empty();
  poldElements.empty();
  if ( bFilter_ ) filter_->empty();
  for ( j=0; j<nUserFields_; j++ ) pufields_[j].empty();

  for ( i=0; i<uelems.size(); i++ ) {
    NN = uelems[i]->GetNumNodes();
    for ( j=0; j<nUserFields_; j++ ) pufields_[j].add(ntimelevels_,gusergrids_[j][i],ntmplevels_);

    u1bdyvals   .add(NULL, FALSE);  // will point to areas in corresp fields
    u2bdyvals   .add(NULL, FALSE);  // will point to areas in corresp fields
    utmp        .add(NULL, TRUE);   utmp        [i] = new GVector (NN);
    uptmp       .add(NULL, FALSE);  uptmp       [i] = NULL;
#if defined(NAVIER_STOKES)
    if ( pelems[i] != NULL ) {
      NN = pelems[i]->GetNumNodes();
      ptmp        .add(NULL, TRUE);   ptmp        [i] = new GVector (NN);
    }
#endif
    poldElements.add(uelems[i]); 
    u1bdyvals [i] = u1[i]->GetBdyValues();
    if ( nEvolvedFields_ > 1 ) 
    u2bdyvals [i] = u2[i]->GetBdyValues();
  }

  if ( bFilter_ ) {
    for ( i=0; i<uelems.size(); i++ ) {
      filter_->add(NULL,TRUE); 
      (*filter_)[i] = new GIFilter(uelems[i],filter_alpha_,filter_delta_);
    }
  }

  return TRUE;

} // end of method GResetExpandables


//************************************************************************************
//************************************************************************************
// METHOD     : GAdaptGrid
// DESCRIPTION: Performs grid adaption
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdaptGrid()
{
  char      *serr = "GASpAR::GAdaptGrid: ";
  GINT      i, k, n=0, nmax=1, nr, nc, irbuff[3], girbuff[3];
  GBOOL     bOK ;
  GIBuffer  irs, ird1, ird2, ics, icd1, icd2;
  
  bAdaptError_ = FALSE;
  bElemListChange_ = FALSE;
  if ( !bDoAdapt_ || !GAMRCond() ) {
    GInitNorms();
    return TRUE;
  }

  irs .Resize(nelems_); irs  = 0; ics .Resize(nelems_); 
  ird1.Resize(nelems_); ird1 = 0; icd1.Resize(nelems_); 
  ird2.Resize(nelems_); ird2 = 0; icd2.Resize(nelems_); 
  if ( !bSpectralAP_    ) ics  = 1; 
  else                    ics  = 0;
  if ( !b1DerivativeAP_ ) icd1 = 1;
  else                    icd1 = 0;
  if ( !b2DerivativeAP_ ) icd2 = 1;
  else                    icd2 = 0;

  if ( !bSpectralAP_ && !b1DerivativeAP_ && !b2DerivativeAP_ ) {
    cout << "GAdaptGrid: AMR requires a refinement criterion" << endl;
    exit(1);
  }

  icycle_amr_last_ = icycle_;
  time_amr_last_   = time_;

  if ( bDoLoadBalancing_ && GLBalCond() ) glop->DoLoadBalance(TRUE);

  bAdaptError_ = TRUE;
  while ( n < nmax ) {
      if ( bSpectralAP_ && !GTagSpectral(irs, ics) ) {
        GDoLog(0,"#GAdaptGrid failure: Spectral refinement tagging");
        iGError_ = GERR_ADAPT;
        return FALSE;
      } 
      
      if ( b1DerivativeAP_ &&  !GTag1Deriv(ird1, icd1) ) {
        GDoLog(0,"#GAdaptGrid failure: 1-Gr refinement tagging");
        iGError_ = GERR_ADAPT;
        return FALSE;
      } 

      if ( b2DerivativeAP_ &&  !GTag2Deriv(ird2, icd2) ) {
        GDoLog(0,"#GAdaptGrid failure: 2-Gr refinement tagging");
        iGError_ = GERR_ADAPT;
        return FALSE;
      } 

      // If either criterion makes a refinement request of 
      // an element, then refine it. Only if _both_ criteria
      // indicate that it's ok to coarsen, do we actually coarsen:
      for ( i=0, nr=0; i<nelems_; i++ ) nr += (irs[i]|ird1[i]|ird2[i]) ? 1 : 0;
      for ( i=0, nc=0; i<nelems_; i++ ) nc += (ics[i]&icd1[i]&icd2[i]) ? 1 : 0;
      hrefine_.Resize(nr); 
      hcoarse_.Resize(nc); 
      for ( i=0,k=0; i<nelems_; i++ ) if ( (irs[i]|ird1[i]|ird2[i]) ) hrefine_[k++] = i;
      for ( i=0,k=0; i<nelems_; i++ ) if ( (ics[i]&icd1[i]&icd2[i]) ) hcoarse_[k++] = i;
#if 0
      cout << "GAdaptGrid: irs  = " << irs  << endl;
      cout << "GAdaptGrid: ird1 = " << ird1 << endl;
      cout << "GAdaptGrid: ird2 = " << ird1 << endl;
      cout << "GAdaptGrid: ics  = " << ics  << endl;
      cout << "GAdaptGrid: icd1 = " << icd1 << endl;
      cout << "GAdaptGrid: icd2 = " << icd2 << endl;
      cout << "GAdaptGrid: hrefine= " << hrefine_<< endl;
      cout << "GAdaptGrid: hcoarse= " << hcoarse_<< endl;
#endif

#if 0
    if ( n > 0 ) {
      cout << "GAdaptGrid: refinement inadequate. Re-grid to higher P?" << endl;
      hrefine_.Resize(0); hcoarse_.Resize(0);
      return TRUE; //exit(1);
    }
#endif
//cout << serr << "Doing Adapt..." << endl;
    bOK = glop->Adapt(hrefine_,hcoarse_);
//cout << serr << "Adapt done." << endl;

    nelems_   = uelems.size();
    irbuff[0] = hrefine_.dim(); 
    irbuff[1] = hcoarse_.dim(); 
    irbuff[2] = nelems_;
    GComm::Allreduce(irbuff, girbuff, 3, GC_GINT , G_OP_SUM);
    ngelems_  = girbuff[2];
    if ( n == 0 ) { 
      bElemListChange_ = (girbuff[0] > 0 || girbuff[1] > 0); 
    }

    // Gather nelems from all procs here:
//cout << serr << "Doing gather..." << endl;
    GComm::Allgather(&nelems_, 1, GC_GINT, igNumElems_.Data(), 1, GC_GINT);
//cout << serr << "gather done." << endl;
//cout << serr << "Doing GComputeLoadBal..." << endl;
    GComputeLoadBal();
//cout << serr << "GComputeLoadBal done." << endl;

    if ( bElemListChange_ &&  !GResetExpandables() ) {
//    iGError_ = GERR_ADAPT;
//    GDoLog(0,"#GAdaptGrid failure: could not reset operators");
//    return FALSE;
      cout << "GAdaptGrid: unable to reset expandables" << endl;
      exit(1);
    }

    if ( !bOK ) {
      iGError_ = GERR_ADAPT;
      GDoLog(0,"#GAdaptGrid failure: glop->Adapt");
      icycle_++;
      GOutput();
      return FALSE;
    }
  
//cout << serr << "Doing global assembly..." << endl;

    if ( bElemListChange_ && !GGlobalAssembly() ) {
      iGError_ = GERR_ADAPT;
      GDoLog(0,"#GAdaptGrid failure: GlobalAssembly failure");
      return FALSE;
    }
//cout << serr << "Global assembly done." << endl;
//cout << serr << "Doing ComputeGlobalDOFs..." << endl;
    GUtils::ComputeGlobalDOFs(uelems, gndofs_, gMinElemLength_, gMaxElemLength_);
    for ( i=0; i<nEvolvedFields_ && stepper != NULL; i++ ) {
      stepper->GetSolver(i+1)->SetMaxIterations(MAX(gndofs_,u1iter));
    }
//cout << serr << "ComputeGlobalDOFs done." << endl;

    n++ ;
  }

  bAdaptError_ = FALSE;
  GInitNorms();
  if ( !GSetBdyCond() )  {
    cout << "GAdaptGrid: SetBdyCond failed" << endl;
    return FALSE;
  }

  return TRUE;
} // end of method GAdaptGrid


//************************************************************************************
//************************************************************************************
// METHOD     : GTagSpectral
// DESCRIPTION: Based on a-priori error estimates, tag elements for
//              refinement or coarsening. Accepts two arguments: one
//              buffer of length nelems that will be filled with 1's 
//              for the indices of the elements to be refined; and one
//              buffer that will be filled with 1's for the indices 
//              of the elements to be coarsened.
//             
// ARGUMENTS  : hr: refinement map list, of length nelems_  
//              nc: coarsen map list, of length nelems_
// RETURNS    : TRUE on sccess; else FALSE
//************************************************************************************
GBOOL GTagSpectral(GIBuffer &hr, GIBuffer &hc)
{
  GINT       i, m;
  GINT       imax1, imax2, rmax1, rmax2;
  GDOUBLE    rate , iuint, garea;
  GDOUBLE    err;
  GVector    *u;
//Elem2D     *elem;
  Field2D    *field;
  APostError aperr(APostError::EN_L2);
//APostError aperr(APostError::EN_H1_SEMI);
  
#if defined(MAIN_DEBUG_OUTPUT)
  cout << "GErrChk: cycle number: " << icycle_ << " ..." << endl;
#endif
  if ( hr.dim() < nelems_ || hc.dim() < nelems_ ) {
    cout << "GTagSpectral: refinement and coarsen maps improperly allocated" << endl;
    return FALSE;
  }
  aperr.SetNumFitPoints   (nSpFit_);
  aperr.SetMinNumFitPoints(nSpFit_);
  aperr.SetTiny(1.0e-10);
//garea  = GUtils::ComputeGlobalArea(uelems);
//iuint  = 1.0;

  for ( m=0; m<naPostFields_; m++ ) {
//  uint  = GUtils::ComputeGlobalIntegralF(*pafields_[m], uelems);
//  uint  = GUtils::ComputeInfNorm(*pafields_[m]);
//  iuint = garea / uint;
//  iuint = 1.0/uint;
    pafields_[m]->start(NULL);
    
    for ( i=0; i<nelems_; i++ ) {
      field = pafields_[m]->member();
//    elem  = field->GetElement();
      u     = field->GetExpCoeffs(0); 
      aperr.SetField(field);
      aperr.ComputeErrEst();
//    uint  = u->MaxA();
//    iuint = fabs(iuint) < TINY*10.0 ? 1.0 : 1.0/uint;
      err   = aperr.GetErrEst() * apunorm_[2*m];
//    cout << "element=" << i << " aperr=" << aperr.GetErrEst() << "apunorm=" << apunorm_[2*m] <<
//" err=" << err << endl;
      // Check for refinement:
      rate = aperr.GetMaxDecayRate();
#if defined(MAIN_DEBUG_OUTPUT)
      cout << "elem[" << i << "]: trunc_err=" << err << " max_sig=" << rate << endl; 
#endif
      hr[i] = (GINT)  ( err   >= ap_tol_ 
                   || ( rate  >= -sig_tol_  )
                      );

      // Check for coarsening:
      hc[i] = (GINT) ( bDoCoarse_ && err < ap_mult_*ap_tol_
#if 0
                                  && rmax <  -sig_tol_
#endif
                     );
      pafields_[m]->next();
    }
  }

  return TRUE;
} // end of method GTagSpectral


//************************************************************************************
//************************************************************************************
// METHOD     : GTag1Deriv
// DESCRIPTION: Based on 1-gradients in coord directions, tag elems for 
//              refinement or coarsening. Accepts two arguments: one
//              buffer of length nelems that will be filled with 1's 
//              for the indices of the elements to be refined; and one
//              buffer that will be filled with 1's for the indices 
//              of the elements to be coarsened.
//             
// ARGUMENTS  : hr: refinement map list, of length nelems_  
//              hc: coarsen map list, of length nelems_
// ARGUMENTS  :  
// RETURNS    : TRUE on sccess; else FALSE
//************************************************************************************
GBOOL GTag1Deriv(GIBuffer &hr, GIBuffer &hc)
{
  GINT       i, j, m;
  GDOUBLE    gL[GDIM], gLMax, del, d1crstol;
  GVector    *u, *dv1;
  GDBuffer   d1fmax ;
  Elem2D     *elem;
  Field2D    *field;

  if ( hr.dim() < nelems_ || hc.dim() < nelems_ ) {
    cout << "GTag1Deriv: refinement and coarsen maps improperly allocated" << endl;
    return FALSE;
  }
  d1fmax.Resize(nelems_);

  gL[0] = gd_[1][0] - gd_[0][0]; 
  gL[1] = gd_[3][1] - gd_[0][1]; 
  if ( GDIM == 3 ) gL[2] = gd_[4][0] - gd_[0][0];
  for ( j=0, gLMax=0.0; j<GDIM; j++ ) {
    gLMax = MAX(gLMax,gL[j]);
  }

  d1crstol = ap_tol1_ *ap_mult1_;
  d1fmax   = 0.0;

  // Do refinement tags:
  for ( m=0; m<naPostFields_; m++ ) {
    pafields_[m]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      field  = pafields_[m]->member();
      elem   = field->GetElement();
      u      = field->GetExpCoeffs(0);
      dv1    = field->GetTemp(0);      // locks memory
      del    = elem->GetMinEdgeLength();
      for ( j=0; j<GDIM; j++ ) {
        elem->Differentiate(dv1, u  , j+1); 
        d1fmax[i] = MAX(d1fmax[i],dv1->MaxA());
      }
      field->TempUnlock(dv1);
      // Check for refinement:
      d1fmax[i] *= dunorm_[2*m];
      if ( bScaleBySize_ ) 
      d1fmax[i] *= del;
      hr[i] = (GINT)( d1fmax[i] > ap_tol1_ );
      pafields_[m]->next();
    }
  }
 
  // Do coarsening tags:
  for ( i=0; i<nelems_ && bDoCoarse_; i++ ) { 
    // Check for coarsening:
      hc[i] = (GINT)( d1fmax[i] < d1crstol );
  }

  return TRUE;

} // end of method GTag1Deriv



//************************************************************************************
//************************************************************************************
// METHOD     : GTag2Deriv
// DESCRIPTION: Based on gradients in coord directions, tag elems for 
//              refinement or coarsening. Accepts two arguments: one
//              buffer of length nelems that will be filled with 1's 
//              for the indices of the elements to be refined; and one
//              buffer that will be filled with 1's for the indices 
//              of the elements to be coarsened.
//             
// ARGUMENTS  : hr: refinement map list, of length nelems_  
//              hc: coarsen map list, of length nelems_
// ARGUMENTS  :  
// RETURNS    : TRUE on sccess; else FALSE
//************************************************************************************
GBOOL GTag2Deriv(GIBuffer &hr, GIBuffer &hc)
{
  GINT       i, j, m;
  GDOUBLE    gL[GDIM], gLMax, del, d2crstol;
  GVector    *u, *dv1, *dv2;
  GDBuffer   d1fmax, d2fmax;
  Elem2D     *elem;
  Field2D    *field;

  if ( hr.dim() < nelems_ || hc.dim() < nelems_ ) {
    cout << "GTag2Deriv: refinement and coarsen maps improperly allocated" << endl;
    return FALSE;
  }

  d1fmax.Resize(nelems_);
  d2fmax.Resize(nelems_);

  gL[0] = gd_[1][0] - gd_[0][0]; 
  gL[1] = gd_[3][1] - gd_[0][1]; 
  if ( GDIM == 3 ) gL[2] = gd_[4][0] - gd_[0][0];
  for ( j=0, gLMax=0.0; j<GDIM; j++ ) {
    gLMax = MAX(gLMax,gL[j]);
  }

  d2crstol = ap_tol2_*ap_mult2_;

  d1fmax = 0.0;
  d2fmax = 0.0;

  // Do refinement tags:
  for ( m=0; m<naPostFields_; m++ ) {
    pafields_[m]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      field  = pafields_[m]->member();
      elem   = field->GetElement();
      u      = field->GetExpCoeffs(0);
      dv1    = field->GetTemp(0);      // locks memory
      dv2    = field->GetTemp(1);      
      del    = elem->GetMinEdgeLength();
      for ( j=0; j<GDIM; j++ ) {
        elem->Differentiate(dv1, u  , j+1); 
        d1fmax[i] = MAX(d1fmax[i],dv1->MaxA());
        elem->Differentiate(dv2, dv1, j+1); 
        d2fmax[i] = MAX(d2fmax[i],dv2->MaxA());
      }
      field->TempUnlock(dv1);
      field->TempUnlock(dv2);
      // Check for refinement:
      d1fmax[i] *= dunorm_[2*m];
      d2fmax[i] *= dunorm_[2*m+1];
      if ( bScaleBySize_ ) {
        d1fmax[i] *= del;
        d2fmax[i] *= del*del;
      }
      hr[i] = (GINT)( d2fmax[i] > ap_tol2_ );
      pafields_[m]->next();
    }
  }
 
  // Do coarsening tags:
  for ( i=0; i<nelems_ && bDoCoarse_; i++ ) { 
    // Check for coarsening:
      hc[i] = (GINT)( d2fmax[i] < d2crstol );
  }

  return TRUE;

} // end of method GTag2Deriv


//************************************************************************************
//************************************************************************************
// METHOD     : GDeleteDynamic
// DESCRIPTION: Delete dynamically-allocated quantities
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void GDeleteDynamic()
{
  GINT  i;

  if ( dX_        != NULL ) delete dX_;
  if ( glop       != NULL ) delete glop;
  if ( stepper    != NULL ) delete stepper;
  if ( morton_    != NULL ) delete morton_;
//if ( vnode_ids  != NULL ) delete vnode_ids;
  if ( velemtypes != NULL ) delete velemtypes;
  if ( gd_        != NULL ) delete [] gd_;
  if ( bLinAdvection_ ) {
    for ( i=0; i<3; i++ ) {
      if ( cadv_[i] ) delete cadv_[i];
    }
  }
  if ( pgfields_  != NULL ) delete [] pgfields_;
  if ( pafields_  != NULL ) delete [] pafields_;
  for ( i=0; i<nEvolvedFields_ && sDSLabel_ != NULL; i++ ) {
    if ( sDSLabel_[i] != NULL ) delete [] sDSLabel_[i] ;
  }
  if ( sDSLabel_ != NULL ) delete [] sDSLabel_;

} // end of method GDeleteDynamic

//************************************************************************************
//************************************************************************************
// METHOD     : GInitTimeLevels
// DESCRIPTION: Initializes all time levels by integrating from the initial
//              state a required number of timesteps s.t. the time levels are
//              filled. NOTE: solution must be initialized prior to entry. 
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE.
//************************************************************************************
GBOOL GInitTimeLevels()
{
  GINT    m;
  GDOUBLE dt0;
  GBOOL   bOK=TRUE;

  if ( bRestart_ ) return TRUE;

  // Start linked lists at beginning
#if 0
  for ( m=0; m<nEvolvedFields_; m++ ) pgfields_[m]->start(NULL);
  // Shift time levels once:
  for ( i=0; i<nelems_; i++ ) {
    for ( m=0; m<nEvolvedFields_; m++ ) {
      pgfields_[m]->member(i)->ShiftLevels();
    }
  }
#endif

//for ( m=0; m<nEvolvedFields_; m++ ) pgfields_[m]->start(NULL);
  m = 0;
  bOK = bOK && GTimeStep();
  GUpdateTimestepHistory(dt_);
  dt0 = dt_;
  dt_ = dt0;
  while ( m < ntimelevels_-1 && bOK ) {
//  bOK = bOK && GUserTimeDep(time_, dt_);
    stepper->SetBDFOrder(m+1);
    stepper->SetAdvOrder(m+1);
#if defined(BURGERS)
    stepper->SetEvolType((BurgersSolver::TIME_EVOLTYPE)TE_EXBDF);
#endif
#if defined(NAVIER_STOKES)
    stepper->SetEvolType((NavierStokes::TIME_EVOLTYPE)TE_EXBDF);
#endif
    GAdaptGrid();
    GUserTimeDep(time_, dt_);
    bOK = bOK && GStep();
    time_ += dt_;
    GTimeUpdate();
    m++;
  }

//for ( m=0; m<nEvolvedFields_; m++ ) pgfields_[m]->start(NULL);
  stepper->SetBDFOrder(iorderBDF_);
  stepper->SetAdvOrder(iorderadv_);
#if defined(BURGERS)
  stepper->SetEvolType((BurgersSolver::TIME_EVOLTYPE)iEvolType_);
#endif
#if defined(NAVIER_STOKES)
  stepper->SetEvolType((NavierStokes::TIME_EVOLTYPE)iEvolType_);
#endif
  dt_ = dt0;

#if 0
  GVector *ut;
  for ( i=0; i<nelems_; i++ ) {
    for ( m=0; m<ntimelevels_; m++ ) {
//    ut = pgfields_[0]->member(i)->GetExpCoeffs(m);
//    cout << "GInitTimeLevels: dt[timelev=" << m << "]=" << dthist_[m] << " u1[" << i << "][timelev=" << m << "]=" << *ut << endl;
      ut = cadv_[0]->member(i)->GetExpCoeffs(m);
      cout << "GInitTimeLevels: dt[timelev=" << m << "]=" << dthist_[m] << " u1[" << i << "][timelev=" << m << "]=" << *ut << endl;
    }
  }
#endif

  return bOK;
} // end of method GInitTimeLevels
 

//************************************************************************************
//************************************************************************************
// METHOD     : GRegisterUserFields
// DESCRIPTION: Registers user-specified fields, which can then be
//              used in system methods, e.g., output, automatically.
//              There is a corresponding grid set for each field specified, 
//              and this is the variable gusergrid_. This variable can be
//              changed in a call to GSetUserGrid method, and this should be
//              done prior to entry.
// ARGUMENTS  : numspecs: number of field list + descriptor pairs
//              f1      : pointer to field list 1
//              sf1     : character string describing f1
//              f2...
// RETURNS    : Number of fields registered, 0 if there is an error.
//************************************************************************************
GINT GRegisterUserFields(GINT numspecs, GFieldList *f1, char *sf1, ...)
{
  va_list     ap;
  GINT        i ;
  char        *ch=NULL;
  GFieldList  *fl;

  if ( f1 == NULL || sf1 == NULL ) {
    return 0;
  }
  
  // Fill system vars with the new user fields and
  // descriptors:
  gusergrids_ .add(usergrid_,FALSE); 
  pufields_   .add(f1,FALSE); 
  sUDSLabel_  .add(); 
  sUDSLabel_  .member()->assign(sf1,strlen(sf1));

  va_start(ap, sf1);  // last arg for which we _know_ type
  for ( i=1; i<numspecs; i++ ) {
    gusergrids_ .add(usergrid_,FALSE); // usergrid_ used unless changed in call to SetUserGrid
    fl          = va_arg(ap, GFieldList*);
    pufields_   .add(fl,FALSE); 
    sUDSLabel_  .add(); 
    ch                 = va_arg(ap, char*);
    sUDSLabel_ .member()->assign(ch,strlen(ch));
  }
  va_end(ap);

  nUserFields_  = pufields_.size();
  
  return TRUE;

} // end of method GRegisterUserFields

//************************************************************************************
//************************************************************************************
// METHOD     : GSetAPostFields
// DESCRIPTION: Sets the fields to use for checking a-posteriori errors for AMR.
//              Exiting list of such fields is not retained, but overwritten.
//              Only the first field is checked if NULL; all subsequent fields will
//              be used until the first NULL field is encountered. 
//
//              NOTE: User should check return type because if == 0, then _no_
//                    fields will be have their errors checked, as the number
//                    of such fields will be 0!
// ARGUMENTS  : numfields: number of fields
//              f1       : pointer to first field (must be non-NULL)
//              ...      : comma-separated list of field pointers, number of which
//                         must equal numfields.
// RETURNS    : Number of fields registered, 0 if there is an error.
//************************************************************************************
GINT GSetAPostFields(GINT numfields,  GFieldList *f1, ...)
{ 
  va_list     ap;
  GINT        i ;
  GBOOL       bOk;
    
  naPostFields_ = 0;
  if ( pafields_ != NULL ) delete [] pafields_; pafields_ = NULL;
  if ( f1 == NULL ) {
    return 0;
  }


  pafields_ = new GFieldList * [numfields];
  pafields_[0] = f1;

  va_start(ap, f1);  // last arg for which we _know_ type
  for ( i=1,bOk=TRUE; i<numfields && bOk; i++ ) {
    pafields_  [i] = va_arg(ap, GFieldList*);
    bOk = pafields_[i] != NULL;
  }
  va_end(ap);

  if ( bOk ) {
    naPostFields_ = numfields;
    return numfields;
  }
  
  return 0;

} // end of method GSetAPostFields
 

//************************************************************************************
//************************************************************************************
// METHOD     : GTimeUpdate
// DESCRIPTION: Performs those tasks associated with updating after a step has been
//              taken, and after the time quantity has been updated.
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE.
//************************************************************************************
GBOOL GTimeUpdate()
{
  GINT i, k;

  // Update field times:
  for ( k=0; k<nEvolvedFields_; k++ ) {
    pgfields_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      pgfields_[k]->member()->SetTime(0,time_);
      pgfields_[k]->next();
    }
  }
  
  // If doing linear advection, where the adv. velocity is
  // _not_ updated in the stepper, update time step on 
  // possibly adapted fields, and do a time shift, before 
  // computing a new one:
  if ( bLinAdvection_ ) {
    for ( k=0; k<GDIM; k++ ) {
      cadv_[k]->start(NULL);
      for ( i=0; i<nelems_; i++ ) {
        cadv_[k]->member()->SetTime(0,time_);
        cadv_[k]->member()->ShiftLevels();
        cadv_[k]->next();
      }
    }
  }
  GUpdateTimestepHistory(dt_);

  return TRUE;
} // end of method GTimeUpdate


//************************************************************************************
//************************************************************************************
// METHOD     : GUpdateTimestepHistory
// DESCRIPTION: Updates the timestep history with specified timestep
// ARGUMENTS  : GDOUBLE dt
// RETURNS    : none.
//************************************************************************************
void GUpdateTimestepHistory(GDOUBLE dtp)
{
  GINT k;

  // First, shift timestep buffer to more distant
  // times:
  for (k=dthist_.dim()-1; k>0; k-- ) {
    dthist_[k] = dthist_[k-1];  // shift from level i+1 to level i
  }

  // Now set most current level:
  dthist_[0] = dtp;

} // end of method GUpdateTimestepHistory
                             

//************************************************************************************
//************************************************************************************
// METHOD     : GInitLogger
// DESCRIPTION: Initializes logging object
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GInitLogger()
{
  GINT  i;
  char  stmp0[FILE_NAME_MAX], stmp1[FILE_NAME_MAX];
  GBOOL bOk=TRUE;

  bOk = glogger_.SetStaticParamDesc("%b %b %b %b %b %b %b %b %b %b %b %s %s %d %d %d %d %b"
                            , "#Is this a restart?"
                            , "#Is this run regridding?"
                            , "#Is this run reading cmd file?"
                            , "#Is there an external mesh?"
                            , "#Does this run use fix timestep?"
                            , "#Scale fixed timestep with AMR ?"
                            , "#Does this run do advection?"
                            , "#Does this run do linear advection?"
                            , "#Does this run do adaption?"
                            , "#Does this run do filtering?"
                            , "#Does this run do dealiasing?"
                            , "#Mesh file"
                            , "#Output file prefix"
                            , "#Initial number elements (global)"
                            , "#Time evolution type"
                            , "#Advection order"
                            , "#Time order"
                            , "#Does this run use a fixed timestep?"
                            );

  bOk = bOk & 
        glogger_.SetStaticData(18, &bRestart_    , &bRegrid_       , &bReadCmdFile_    
                                 , &bExtMesh     , &bFixedTimeStep_, &bScaledTimeStep_ 
                                 , &bDoAdvection_, &bLinAdvection_ , &bDoAdapt_         
                                 , &bFilter_     , &bDoDealiasing_  
                                 , fnmesh_       , fnout_          , &nelems_          , &iEvolType_   
                                 , &iorderadv_   , &iorderBDF_     , &bFixedTimeStep_
                               );

  bOk = bOk &
       glogger_.SetStaticParamDesc("%f %f %f %f %f %f"
                                 , "#nu1"
                                 , "#nu2"
                                 , "#rho"
                                 , "#time_"
                                 , "#dt_"
                                 , "#Courant_"
                                 );
  bOk = bOk & 
        glogger_.SetStaticData(6 , &nu_[0], &nu_[1] , &rho_, &time_
                                 , &dt_   , &Courant_ 
                             );
 
 
  if ( iStopCond_ == BY_CYCLE ) {
    bOk = bOk & 
          glogger_.SetStaticParamDesc("%d %d %d %d"
                                    , "#Max cycles"
                                    , "#Start out cycle"
                                    , "#End out cycle"
                                    , "#Out cycle delta"
                                    );
    bOk = bOk & 
          glogger_.SetStaticData(4 , &icycle_max_    , &icycle_out_beg_
                                   , &icycle_out_end_, &icycle_out_skip_
                               );
  }
  else if (  iStopCond_ == BY_TIME ) {
    bOk = bOk & 
          glogger_.SetStaticParamDesc("%f %f %f %f"
                                    , "#Max time"
                                    , "#Start out time"
                                    , "#End out time"
                                    , "#Out time delta"
                                    );
    bOk = bOk & 
          glogger_.SetStaticData(4 , &time_max_    , &time_out_beg_
                                   , &time_out_end_, &time_out_skip_
                               );
  }
 
  if ( bDoAdapt_ ) {
    bOk = bOk & 
          glogger_.SetStaticParamDesc("%b %b %b %b %f %f %f %f %f %f %d"
                                    , "#Do coarsening?"
                                    , "#Do spectral AP-estimator"
                                    , "#Do 1-derivative AP-estimator"
                                    , "#Do 2-derivative AP-estimator"
                                    , "#a-Posteriori tolerance"
                                    , "#a-Posteriori multiplier"
                                    , "#a-Posteriori d1 tolerance"
                                    , "#a-Posteriori d1 multiplier"
                                    , "#a-Posteriori d2 tolerance"
                                    , "#a-Posteriori d2 multiplier"
                                    , "#Max. number refine levels"
                                    );
    bOk = bOk & 
          glogger_.SetStaticData(11 , &bDoCoarse_, &bSpectralAP_, &b1DerivativeAP_, &b2DerivativeAP_
                                   , &ap_tol_   , &ap_mult_    , &ap_tol1_       , &ap_mult1_
                                   , &ap_tol2_  , &ap_mult2_   , &nrefinelevels_
                                );
    }


  // Set the dynamic (time-dependent) fields:

    bOk = bOk &
          glogger_.SetDynamicParamDesc("%d %f %f %d %d %f %f %f"
                                     , "#CYC"
                                     , "TIME"
                                     , "dt"
                                     , "NGLOB"
                                     , "NDOFS"
                                     , "LMIN"
                                     , "LMAX"
                                     , "LOAD_BAL"
                                    );
    bOk = bOk & 
          glogger_.SetDynamicData(8 , &icycle_ , &time_  , &dt_
                                    , &ngelems_, &gndofs_, &gMinElemLength_
                                    , &gMaxElemLength_   , &gLoadBalMeas_
                                 );

#if defined(BURGERS)
  for ( i=0; i<nEvolvedFields_; i++ ) {
    sprintf(stmp0, "U%d-CGERR", i+1);
    sprintf(stmp1, "U%d-NITER", i+1);
    bOk = bOk & glogger_.SetDynamicParamDesc("%f %d", stmp0, stmp1);
    bOk = bOk & glogger_.SetDynamicData(2 , &gerrcg_[i],  &gncgit_[i]);
  }
#elif defined(NAVIER_STOKES)
  for ( i=0; i<nEvolvedFields_; i++ ) {
    if ( i == nEvolvedFields_-1 ) {
      sprintf(stmp0, "P-CGERR");
      sprintf(stmp1, "P-NITER");
    }
    else {
      sprintf(stmp0, "U%d-CGERR", i+1);
      sprintf(stmp1, "U%d-NITER", i+1);
    }
    bOk = bOk & glogger_.SetDynamicParamDesc("%f %d", stmp0, stmp1);
    bOk = bOk & glogger_.SetDynamicData(2 , &gerrcg_[i],  &gncgit_[i]);
  }
  bOk = bOk & glogger_.SetDynamicParamDesc("%f", "DIV.V" );
  bOk = bOk & glogger_.SetDynamicData(1 , &divvmax_);
#endif
  bOk = bOk & GUserLogConfig();
  return bOk;

} // end of method GInitLogger
                             

//************************************************************************************
//************************************************************************************
// METHOD     : GSetUserGrid
// DESCRIPTION: Sets grid used in setting/computing on user-specified fields
// ARGUMENTS  : elems: user element list
// RETURNS    : none.
//************************************************************************************
void GSetUserGrid(GElemList *elems)
{
  usergrid_ = elems;
} // end of method GSetUserGrid


//************************************************************************************
//************************************************************************************
// METHOD     : GComputeLoadBal
// DESCRIPTION: Computes load balance measure. The variable igNumElems_ must be
//              set providing the number of elements for all procs.
// ARGUMENTS  : none.
// RETURNS    : none. But member data gLoadBalMeas_ is set.
//************************************************************************************
void GComputeLoadBal()
{
  GINT imax=0, imin=IHUGE;

  for ( GINT i=0; i<GComm::WorldSize(); i++ ) {
    imax = MAX(imax,igNumElems_[i]);
    imin = MIN(imin,igNumElems_[i]);
  }

  gLoadBalMeas_ = ( (GDOUBLE) imin ) / ( (GDOUBLE) imax );
 
} // end of method GComputeLoadBal


