//************************************************************************************//
// Module       : gasparFileReader.cpp
// Date         : 7/15/02 (RMP)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                Gaspar file input.
// Derived From : none.
// Modifications:
//************************************************************************************//


#include "gasparFileReader.hpp"

const char* GasparFileReader::THIS = "GasparFileReader";

GasparFileReader::GasparFileReader(){
  setOpenStatus(FALSE);
}


//************************************************************************************
//************************************************************************************
// METHOD     : GasparFileReader Destructor
// DESCRIPTION: Releases all resources associated with this reader. 
//************************************************************************************

GasparFileReader::~GasparFileReader(){
  if(isOpen()){
    closeFile();
  }

  if(getFile() != NULL){
    delete getFile();
    setFile(NULL);
  }
}
//end GasparFileReader Destructor method


//************************************************************************************
//************************************************************************************
// METHOD     : readSEVector()
// DESCRIPTION: 1) Reads the data array and its length, along with any needed metadata 
//              from an existing file storing an SEVector named in file by 'vectorName'.
//              2) Instantiates a new SEVector object initialized with the data read.
//                 and sets the argument 'vector' to point to it.
// ARGUMENTS  : vectorName  : The name of the SEVector in the file.
// RETURNS    : TRUE on success; otherwise FALSE if the SEVector cannot be read for
//              any reason.
// PRECOND    : The file to be read was opened with a successful call to openHDF()
//************************************************************************************

GBOOL GasparFileReader::read(SEVector*& vector, const char* vectorName){
  errorMsg = "::read(SEVector...) failed.";

  if(!isOpen()){
    cerr << THIS << errorMsg  << endl;
    delete[] errorMsg;
    return FALSE;
  }

  GDOUBLE* vData;
  ISUB vDim;

  if(!getFile()->readVector(vectorName, vData, vDim)){
    cerr << THIS << errorMsg << endl;
    delete[] errorMsg;
    return FALSE;
  }

  if(vector){
    delete vector;
    vector = NULL;
  }

  vector = new SEVector(vData, vDim);
  delete[] vData;
  vData = NULL;

  return TRUE;
}
//end of method readSEVector()



GBOOL GasparFileReader::read(SEMatrix* matrix, const char* matrixName){
  errorMsg = "::read(SEMatrix...) failed.";
  
  if(!isOpen()){
    cerr << THIS << errorMsg  << " The file is closed." << endl;
    return FALSE;
  }

  GDOUBLE* mData = NULL;
  ISUB dim1, dim2;

  if(!getFile()->readMatrix(matrixName, mData, dim1, dim2)){
    cerr << THIS << errorMsg << endl;
    return FALSE;
  }
  /*
  cout << "Matrix read from file:" << endl;
  for(int i=0; i<dim1*dim2; i++){
    cout << mData[i] << ", ";
  }
  cout << endl;
  */
  //*********************
  
  cout << "Matrix read from file:" << endl;

  for(ISUB i=0; i<dim1; i++){
    for(ISUB j=0; j<dim2; j++){
      cout << mData[i*dim2+j] << ", ";
    }
    cout << endl;
  }
  cout << endl;
  
  //*********************

  if(matrix){
    delete matrix;
    matrix = NULL;
  }

  //matrix = new SEMatrix(mData, dim1, dim2);
  free(mData);

  return TRUE;
}
//end of method read(SEMatrix...)
