//************************************************************************************//
// Module       : gbin_reader.hpp
// Date         : 10/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the object that reads a GASpAR binary data file
// Derived From : GBinStream.
// Modifications:
//************************************************************************************//
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#if defined(_LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include "gbin_reader.hpp"
#include "gdd_file.h"
#include "gcutils.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GBinReader::GBinReader(GBOOL isCollective, GBOOL isIndependent, GSHORT  ioTaskID)
: GBinStream(isCollective, isIndependent, ioTaskID),
ndatasets_    (0),
nd_           (NULL),
procid_       (NULL),
nvv_          (NULL),
nMeta_        (0),
nTag_         (NULL),
fMeta_        (NULL),
fTag_         (NULL),
dsoffsets_    (NULL),
dataoffsets_  (NULL),
coordoffsets_ (NULL),
elemtps_      (NULL),
idims_        (NULL),
cdims_        (NULL),
Vert_         (NULL),
sTag_         (NULL),
sMeta_        (NULL),
identifier_   (NULL)
{

} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Destructor
GBinReader::~GBinReader()
{
  DeleteDynamic();
}


//************************************************************************************
// METHOD     : CreeateDynamic
// DESCRIPTION:
// ARGUMENTS  : na: number of datasets required
// RETURNS    : TRUE on success; else FALSE. FALSE occurs with inability to
//              allocate data
//************************************************************************************
GBOOL GBinReader::CreateDynamic(GINT  na)
{
  GINT  i;
  GBOOL bRet;

  nd_           = new GINT     [na];
  procid_       = new GSHORT   [na];
  nvv_          = new GINT     [na];
  elemtps_      = new ELEMTYPE [na];
  dsoffsets_    = new GFPOS    [na];
  dataoffsets_  = new GFPOS    [na];
  coordoffsets_ = new GFPOS    [na];
  idims_        = new GINT    *[na];
  cdims_        = new GINT    *[na];
  Vert_         = new Point3D *[na];
  nTag_         = new GINT     [na];
  sTag_         = new char    *[na];
  fTag_         = new GDOUBLE   *[na];

  bRet = nd_          && procid_       && nvv_       && 
         elemtps_     && dsoffsets_    && cdims_     &&
         dataoffsets_ && coordoffsets_ && idims_     && 
         Vert_        && sTag_         && fTag_      ;

  for ( i=0; i<na & bRet; i++ ) {
    nd_          [i] = 0;
    procid_      [i] = 0;
    nvv_         [i] = 0;
    elemtps_     [i] = RECT_QUAD;
    dsoffsets_   [i] = 0;
    dataoffsets_ [i] = 0;
    coordoffsets_[i] = 0;
    nTag_        [i] = 0;
    idims_       [i] = NULL;
    cdims_       [i] = NULL;
    Vert_        [i] = NULL;
    sTag_        [i] = NULL;
    fTag_        [i] = NULL;
  }

  return bRet; 
} // end of CreateDynamic


//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void GBinReader::DeleteDynamic()
{
  GINT  i;


  if ( nd_        != NULL ) delete [] nd_       ; nd_        = NULL;
  if ( procid_    != NULL ) delete [] procid_   ; procid_    = NULL;
  if ( nvv_       != NULL ) delete [] nvv_      ; nvv_       = NULL;
  if ( elemtps_   != NULL ) delete [] elemtps_  ; elemtps_   = NULL; 
  if ( sMeta_     != NULL ) delete [] sMeta_    ; sMeta_     = NULL;
  if ( fMeta_     != NULL ) delete [] fMeta_    ; fMeta_     = NULL;
  if ( nTag_      != NULL ) delete [] nTag_     ; nTag_      = NULL;
  if ( identifier_!= NULL ) delete [] identifier_; identifier_= NULL;
  if ( ios_       != NULL ) delete ios_         ; ios_      = NULL;
  if ( elemtps_   != NULL ) delete [] elemtps_  ; elemtps_  = NULL;
  if ( dsoffsets_ != NULL ) delete [] dsoffsets_; dsoffsets_= NULL;
  if ( dataoffsets_  != NULL ) delete [] dataoffsets_ ; dataoffsets_ = NULL;
  if ( coordoffsets_ != NULL ) delete [] coordoffsets_; coordoffsets_= NULL;
  
  if ( fTag_ != NULL ) {
    for ( i=0; i<ndatasets_; i++ ) delete [] fTag_[i];
    delete [] fTag_; fTag_ = NULL;
  }
  if ( sTag_ != NULL ) {
    for ( i=0; i<ndatasets_; i++ ) delete [] sTag_[i]; 
    delete [] sTag_; sTag_ = NULL;
  } 
  if ( Vert_        != NULL ) {
    for ( i=0; i<ndatasets_; i++ ) delete [] Vert_[i];
    delete []  Vert_; Vert_ = NULL;
  }
  if ( idims_        != NULL ) {
    for ( i=0; i<ndatasets_; i++ ) delete [] idims_[i];
    delete []  idims_; idims_= NULL;
  }
  if ( cdims_        != NULL ) {
    for ( i=0; i<ndatasets_; i++ ) delete [] cdims_[i];
    delete []  cdims_; cdims_= NULL;
  }


} // end of method DeleteDynamic


//************************************************************************************
// METHOD     : Open
// DESCRIPTION: Opens file; checks header for validity
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL GBinReader::Open(const char *fn)
{


  // If datatype sizes are different than defaults, reset them:
  GBinStream::ResetSizes(TRUE);  
  if ( !GBinStream::Open(fn, gios::in) ) return FALSE;

  // Check datatype sizes:
  if ( szkey_ == NULL ) {
    cout << "GBinReader::Open: FATAL: NULL key data" << endl;
    exit(1);
  }

  if ( !GetFileInfo() ) { 
    isOpen_ = FALSE;
    return FALSE;
  }

#if 0
  // A dimension check:
  for ( i=0; i< ndatasets_; i++ )
  {
    if ( nvv_[i] != (GINT )(pow(2.0,(GSHORT )nd_[i])) )
    {
      isOpen_ = FALSE;
      ierror_ = ERRCORRUPT;
      return FALSE;
    }
  }
#endif
  return TRUE;
 
} // end of method Open


//************************************************************************************
// METHOD     : Close
// DESCRIPTION: Closes stream.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void GBinReader::Close()
{

  GBinStream::Close();
  isOpen_ = FALSE;
} // end of method Close


//************************************************************************************
// METHOD     : GetElemTypes
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
ELEMTYPE GBinReader::GetElemTypes(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return RECT_QUAD;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return RECT_QUAD;
  }


  return elemtps_[dsid];

} // end of method GetElemTypes


//************************************************************************************
// METHOD     : GetNumDataSets
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinReader::GetNumDataSets()
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }

  return ndatasets_;

} // end of method GetNumDataSets


//************************************************************************************
// METHOD     : GetMeta
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE *GBinReader::GetMeta()
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }

  return fMeta_;

} // end of method GetMeta

//************************************************************************************
// METHOD     : GetMetaDesc
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
char *GBinReader::GetMetaDesc()
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }

  return sMeta_;

} // end of method GetMetaDesc


//************************************************************************************
// METHOD     : GetNumMeta
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinReader::GetNumMeta()
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  return nMeta_;

} // end of method GetNumMeta


//************************************************************************************
// METHOD     : GetRank
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinReader::GetRank(GINT  dsid )
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }

  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return -1;
  }

  return nd_[dsid];

} // end of method GetRank


//************************************************************************************
// METHOD     : GetLabel
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
const char *GBinReader::GetLabel(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }

  return sTag_[dsid];

} // end of method GetLabel

//************************************************************************************
// METHOD     : GetNumTags
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinReader::GetNumTags(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }
  if ( dsid < 0 || dsid >= ndatasets_ )
  {
    ierror_ = ERRBADDATA;
    return 0;
  }

  return nTag_[dsid];

} // end of method GetNumTags


//************************************************************************************
// METHOD     : GetTags
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GDOUBLE *GBinReader::GetTags(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }
  if ( dsid < 0 || dsid >= ndatasets_ )
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }

  return fTag_[dsid];

} // end of method GetTags


//************************************************************************************
// METHOD     : GetOffsets
// DESCRIPTION: Get dataset offsets 
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GFPOS GBinReader::GetOffsets(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return -1;
  }

  return dsoffsets_[dsid];

} // end of method GetOffsets


//************************************************************************************
// METHOD     : GetDims (1)
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GINT  GBinReader::GetDims(GINT  dsid, GINT  idir)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }
  if ( dsid < 0 || dsid >= ndatasets_ || idir < 1 || idir > nd_[dsid] ) 
  {
    ierror_ = ERRBADDATA;
    return 0;
  }

  
  return idims_[dsid][idir-1];


} // end of method GetDims (1)


//************************************************************************************
// METHOD     : GetDims (2)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  *GBinReader::GetDims(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }
  if ( dsid < 0 || dsid >= ndatasets_ )
  {
    ierror_ = ERRBADDATA;
    return 0;
  }

 
  return idims_[dsid];


} // end of method GetDims (2)


//************************************************************************************
// METHOD     : GetCoordDims (1)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinReader::GetCoordDims(GINT  dsid, GINT  idir)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }
  if ( dsid < 0 || dsid >= ndatasets_ || idir < 1 || idir > nd_[dsid] )
  {
    ierror_ = ERRBADDATA;
    return 0;
  }


  return cdims_[dsid][idir-1];


} // end of method GetDims (1)


//************************************************************************************
// METHOD     : GetNumVert
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinReader::GetNumVert(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return 0;
  }


  return nvv_[dsid];

} // end of method GetNumVert


//************************************************************************************
// METHOD     : GetVert
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    :  array w/ vertices arranged:
//               x0      , x1                     in 1d
//               x0 y0   , x1 y1    ... x3 y3     in 2d
//               x0 y0 z0, x1 y1 z1 ... x7 y7 z7  in 3d
//               One vertex for each element.
//************************************************************************************
Point3D *GBinReader::GetVert(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }

  return Vert_[dsid];

} // end of method GetVert


//************************************************************************************
// METHOD     : GetGridData
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GDOUBLE *GBinReader::GetGridData(GINT  dsid, GINT  idir, GDOUBLE *x, GINT  n)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }
  if ( idir < 1 || idir > nd_[dsid] )
  {
    ierror_ = ERRDATANOTFND;
    return NULL;
  }

  GINT   i, minn, NN;
  GINT   nc, ncoord;
  GFPOS  icurr;


  icurr = tellg();
  for ( i=0, NN=1; i<nd_[dsid]; i++ ) NN*= idims_[dsid][i];
#if 0
  if ( n < NN )    // not enough space 
  {
    ierror_ = ERRMISC;
    return NULL;
  }
#endif

  // Offsets to coordinate segment should have been set in GetDSInfo: 
  seekg(coordoffsets_[dsid], ios::beg);
  if ( eof() )
  {
    ierror_ = ERREOF;
    return NULL;
  }

  read((GUCHAR*)&nc, isz);              // read number of coord arrays
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nc, isz);
  if ( idir > nc ) return NULL;

  for ( i=0; i<idir && i<nc; i++ ) {
    read((GUCHAR*)&ncoord, isz);        // read size of coord array
    if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ncoord, isz);
    minn = MIN(ncoord, n);
    if ( x != NULL && n < ncoord  ) {
      ierror_ = ERRMISC;
      return NULL;
    }
    if ( x != NULL ) {
      read((GUCHAR*)x, minn*fsz);                 // read coord array data
      if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)x, minn, fsz);
    }
    if ( fail() )
    {
      ierror_ = ERRCORRUPT;
      return NULL;
    }
  }
  seekg(icurr,ios::beg);
  return x;

} // end of method GetGridData



//************************************************************************************
// METHOD     : GetFieldData
// DESCRIPTION: Gets field data by dataset id
// ARGUMENTS  : dsid     : dataset id
//              data     : data storage array (must be non-NULL)
//              n        : size of data storage 
// RETURNS    : pointer to GDOUBLE *data on success; else NULL
//************************************************************************************
GDOUBLE *GBinReader::GetFieldData(GINT  dsid, GDOUBLE *data, GINT  n)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return NULL;
  }
  if ( data == NULL )
  {
    ierror_ = ERRMALLOC;
    return NULL;
  }
  if ( dsid < 0 || dsid >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }

  GINT   i, NN, ndata;
  GFPOS  icurr;

  icurr = tellg();
  for ( i=0, NN=1; i<nd_[dsid]; i++ ) NN*= idims_[dsid][i];
  if ( n < NN )    // not enough space
  {
    ierror_ = ERRMISC;
    return NULL;
  }

  // Offsets to data segment should have been set in GetDSInfo: 
  seekg(dataoffsets_[dsid], ios::beg);
  if ( eof() )
  {
    ierror_ = ERREOF;
    return NULL;
  }

  read((GUCHAR*)&ndata, isz);
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ndata, isz);
  if ( ndata != NN )
  {
    ierror_ = ERRCORRUPT;
    return NULL;
  }
  read((GUCHAR*)data, n*fsz);
  if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)data, n, fsz);
  if ( fail() )
  {
    ierror_ = ERRDATABLKR;
    return NULL;
  }
  seekg(icurr,ios::beg);

  return data;
} // end of GetFieldData


//************************************************************************************
// METHOD     : GetFieldData (2)
// DESCRIPTION: Gets field data by field label
// ARGUMENTS  : slabel   : dataset label 
//              ids_start: start looking at this dataset id 
//              data     : data storage array (must be non-NULL)
//              n        : size of data storage 
// RETURNS    : pointer to GDOUBLE *data on success; else NULL
//************************************************************************************
GDOUBLE *GBinReader::GetFieldData(const char *slabel, GINT  ids_start,  GDOUBLE *data, GINT  n)
{
  GINT  ids;
  if ( ids_start < 0 || ids_start >= ndatasets_ ) 
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }

  for ( ids=ids_start; ids<ndatasets_; ids++ ) {
    if ( sTag_[ids] &&  strcasecmp(slabel,sTag_[ids]) == 0 ) break;
  }
  if ( ids >= ndatasets_ )
  {
    ierror_ = ERRBADDATA;
    return NULL;
  }

  // if dataset found, use corresp. id to get field:
  return GetFieldData(ids, data, n);

} // end of method GetFieldData


//************************************************************************************
// METHOD     : GetFileInfo
// DESCRIPTION: Get file-level info
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinReader::GetFileInfo()
{
  char serr[] = "GBinReader::GetFileInfo: ";

  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }

  GINT      i , one;
  GINT      ilen, dlen;



  // Read file meta data:
  seekg(0,ios::beg);

  read((GUCHAR*)&one , isz);                    // endedness indicator
  for ( i=0; i<ndtypes_; i++ ) {               // key data type sizes
    read((GUCHAR*)(szkey_+i),i4sz);   
    if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&(szkey_[i]), i4sz);
  }

  read((GUCHAR*)&ilen, isz);                    // file identifier length
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ilen, isz);
  if ( identifier_ ) delete [] identifier_; identifier_ = NULL;
  identifier_ = new char [ilen+1]; identifier_[ilen] = '\0';
  if ( ilen > 0 )
  read((GUCHAR*)identifier_, ilen*sizeof(char));// file identifier
  
  read((GUCHAR*)&nMeta_, i4sz);                 // number of meta data elems:
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nMeta_, i4sz);
  if ( nMeta_ > 0 )  {
    if ( fMeta_      ) delete [] fMeta_ ; 
    fMeta_ = new GDOUBLE [nMeta_];
    read((GUCHAR*)fMeta_, nMeta_*fsz);          // meta data
    if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)fMeta_, nMeta_, fsz);
  }
  read((GUCHAR*)&dlen  , isz);                  // data file descriptor length
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&dlen, isz);
  if ( sMeta_      ) delete [] sMeta_     ; sMeta_      = NULL;
  sMeta_ = new char [dlen+1]; sMeta_[dlen] = '\0';
  if ( dlen > 0 ) 
  read((GUCHAR*)sMeta_  , dlen*sizeof(char));             // data file meta descriptor
  
  read((GUCHAR*)&ndatasets_,isz);               // number of datasets in file
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ndatasets_, isz);
  if ( ndatasets_ < 1 )
  {
    cout << serr << "no datasets " << endl;
    ierror_ = ERRNODATA;
    return FALSE;
  }
  if ( ndatasets_ > MAX_DATASETS )
  {
    cout << serr << "invalid number of data sets" << endl;
    ierror_ = ERRBADSNUM;
    return FALSE;
  }
 
  if ( !CreateDynamic(ndatasets_) )
  {
    cout << serr << "Unable to create data set structures" << endl;
    ierror_ = ERRMALLOC;
    return FALSE;
  }

  seekg(pOffsetBlk(),ios::beg);
  read((GUCHAR*)dsoffsets_,ndatasets_*gfsz);       // dataset offsets w/in file
  if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)dsoffsets_, ndatasets_, gfsz);
//seekg((MAX_DATASETS-ndatasets_)*isz,ios::cur); // skip to end of meta data section 
  if ( fail() ) 
  {
    ierror_ = ERRMETAR;
    return FALSE;
  }

  for ( i=0; i<ndatasets_; i++ )
  {
    if ( !GetDSInfo(i) ) // error set withing GetDSInfo 
    {
      cout << serr <<  "Data set info retrieval failed" << endl;
      return FALSE; 
    }
  }

  return TRUE;
} // end of GetFileInfo


//************************************************************************************
// METHOD     : GetDSInfo
// DESCRIPTION: Get dataset-level info
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinReader::GetDSInfo(GINT  dsid)
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }
  if ( dsid < 0 || dsid > MAX_DATASETS )
  {
    ierror_ = ERRDATABLKR;
    return FALSE;
  }

  GINT      i, j, ndata, NN;
  GINT      dlen, nv, nc;
  GDOUBLE   *avert=NULL;
  ELEMTYPE  etype;
  Point     *pvert;
  GFPOS     icb, icurr;

  seekg(dsoffsets_[dsid], ios::beg);
  if ( eof() || fail() )
  {
    ierror_ = ERRDATABLKR;
    return FALSE;
  }
  if ( sTag_    && sTag_ [dsid] ) delete [] sTag_ [dsid]; sTag_ [dsid] = NULL;      
  if ( fTag_    && fTag_ [dsid] ) delete [] fTag_ [dsid]; fTag_ [dsid] = NULL;
  if ( idims_   && idims_[dsid] ) delete [] idims_[dsid]; idims_[dsid] = NULL;
  if ( cdims_   && cdims_[dsid] ) delete [] cdims_[dsid]; cdims_[dsid] = NULL;
  if ( Vert_    && Vert_ [dsid] ) delete [] Vert_ [dsid]; Vert_   [dsid] = NULL;

  // Read data-set meta data:
  read((GUCHAR*)&icb, gfsz);                        // dataset's coord system block's offset  
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&icb, gfsz);

  read((GUCHAR*)(nTag_+dsid), isz);                 // number of tag elements
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nTag_[dsid], isz);
  if ( nTag_[dsid] > 0 ) {
    fTag_[dsid]  = new GDOUBLE [nTag_[dsid]];
    read((GUCHAR*)(fTag_[dsid]),nTag_[dsid]*fsz);   // data set tag elements
    if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)fTag_[dsid], nTag_[dsid], fsz);
  }
  read((GUCHAR*)&dlen, isz);                        // descriptor length
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&dlen, isz);
  sTag_[dsid] = new char [dlen+1];
  if ( dlen > 0 ) {
    sTag_[dsid][dlen]='\0';
    read((GUCHAR*)sTag_[dsid],dlen);                // dataset descriptor
  }
  dataoffsets_  [dsid] = tellg();
  read((GUCHAR*)&ndata ,isz);                       // data block size
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ndata, isz);

  
  icurr = tellg();
  seekg(icb,ios::beg);                              // move file pointer to associate coord data block
  if ( eof() || fail() )
  {
    ierror_ = ERRDATABLKR;
    return FALSE;
  }
  read((GUCHAR*)(nd_+dsid),isz);                    // dataset rank
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)(nd_+dsid), isz);
  idims_[dsid] = new GINT  [nd_[dsid]];
  read((GUCHAR*)(idims_[dsid]),nd_[dsid]*isz);      // dataset dimensions
  if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)idims_[dsid], (GINT )nd_[dsid], isz);

  for ( i=0, NN=1; i<nd_[dsid]; i++ ) NN *= idims_[dsid][i];
  if ( NN != ndata )
  {
    ierror_ = ERRBADDATA;
    return FALSE;
  }
  read((GUCHAR*)&etype ,esz);                       // elem type
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&etype, esz);
  elemtps_[dsid] = etype;

  read((GUCHAR*)&nvv_[dsid],isz);                   // number of vertices, each of length nd_
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nvv_[dsid], isz);
  nv             = nd_[dsid]*nvv_[dsid];
  if ( nvv_[dsid] ) {
  Vert_   [dsid] = new Point3D   [nvv_[dsid]];
  avert          = new GDOUBLE [nv];
  pvert          = Vert_[dsid];
  read((GUCHAR*)avert ,nv*fsz);                     // vertices, in array form
  if ( bEndianSwapped_ ) GCUtils::swapall((GBYTE*)avert, nv, fsz);
  for ( i=0; i<nvv_[dsid]; i++ )  {
    for ( j=0; j<nd_[dsid]; j++ ) pvert[i][j] = avert[i*nd_[dsid]+j];
  }
  }
  coordoffsets_[dsid] = tellg();             
  read((GUCHAR*)&nc,isz);                           // number of coord arrays:
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nc, isz);
  if ( nc > 0 ) 
  cdims_ [dsid] = new GINT  [nc]; 
  for ( i=0; i<nc; i++ )  {
    read((GUCHAR*)(cdims_[dsid]+i),isz);            // size of coord array data
    if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&cdims_[dsid][i], isz);
    seekg(cdims_[dsid][i]*fsz,ios::cur);           // skip over coord data
  }
  if ( avert ) delete [] avert; 
  seekg(icurr, ios::beg);
  if ( fail() )
  {
    ierror_= ERRCORRUPT;
    return FALSE;
  }
   
  return TRUE;
} // end of GetDSInfo


//************************************************************************************
// METHOD     : GetDatatypeSz
// DESCRIPTION: Get file's datatype sizes
// ARGUMENTS  : itype: one of the GBR_TypeSz enum datatypes
// RETURNS    : GINT  type size in bytes for specified datatype
//************************************************************************************
GINT  GBinReader::GetDatatypeSz(GBR_TypeSz itype)
{
  switch(itype) {
    case GBR_GINT :
      return (GINT )isz;
    case GBR_GSHORT :
      return (GINT )sisz;
    case GBR_GDOUBLE:
      return (GINT )fsz;
    case GBR_ELEM_SZ:
      return (GINT )esz;
    case GBR_GFPOS_SZ:
      return (GINT )gfsz;
    default:
      cout << "GBinReader::GetDatatypeSz: illegal BGinReader datatype size" << endl;
      exit(1);
  }
  return 0;
} // end of method GetDatatypeSz
