//************************************************************************************//
// Module       : gbin_stream.hpp
// Date         : 10/17/02 (DLR)
// Copyright    : 2002, 2003 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the object that associates a stream with a GASpAR binary data file
// Derived From : GStream.
// Modifications:
//************************************************************************************//
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#if defined(_LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include "gbin_stream.hpp"
#include "gdd_file.h"
#include "gcutils.hpp"

char *identifier_ = "GBIN_FILETYPE v0.1";



//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GBinStream::GBinStream(GBOOL isCollect, GBOOL isIndep, GSHORT  ioTaskID)
:GStream(isCollect, isIndep, ioTaskID),
bEndianSwapped_ (FALSE),
bResetSizes_    (FALSE),
gndatasets_     (0),
ndtypes_        (5),
szkey_          (NULL),
dsOffset_       (NULL)
{
  fsz  = sizeof(GDOUBLE);
  i4sz = sizeof(GINT4BYTE);
  isz  = sizeof(GINT );
  sisz = sizeof(GSHORT );
  esz  = sizeof(ELEMTYPE);
  gfsz = sizeof(GFPOS);

  dsOffset_ = new GFPOS[MAX_DATASETS];
  memset(dsOffset_, 0, MAX_DATASETS*gfsz);


} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Destructor
GBinStream::~GBinStream()
{
  if ( dsOffset_!= NULL ) delete [] dsOffset_; dsOffset_ = NULL; 
  if ( szkey_   != NULL ) delete [] szkey_   ; szkey_    = NULL;
}


//************************************************************************************
//************************************************************************************
// METHOD     : Open (1)
// DESCRIPTION: Opens file; checks header for validity
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL GBinStream::Open(const char *fn, GIOS_MODE iomode, GBOOL bDelete)
{
  char      *serr = "GBinStream::Open: ";
  GIOS_MODE access ;
  GFPOS     icurr;

  if ( fn == NULL ) 
  {
    ierror_ = ERRFILENAME;
    return FALSE;
  }
  strncpy(filename_,fn,FILE_NAME_MAX);
  
  access = iomode | gios::binary | gios::ate ;

//cout << serr << "GStream::Open ... " << endl;
  if ( !GStream::Open(fn, access, bDelete) ) return FALSE;
//cout << serr << "GStream::Open done. " << endl;

  // Do some checking:
//cout << serr << "Checking header... " << endl;
  if ( tellg() > 0 ) { // if file has contents
    if ( !Check() ) return FALSE;
    gndatasets_ = GBinStream::GetDSNum();
    icurr = tellg();
    seekg(pOffsetBlk(),ios::beg);
    read((GUCHAR*)dsOffset_,MAX_DATASETS*gfsz);
    seekg(icurr, ios::beg);
    // file may have only been initialized so that gndatasets_ = 0, so...
    if ( gndatasets_ <  0  || gndatasets_ > MAX_DATASETS ) { 
      ierror_ = ERRCORRUPT;
      return FALSE;
    }  
    if ( fail() ) {
      ierror_ = ERRCORRUPT;
      return FALSE;
    }
  }
//cout << serr << "Checking done. " << endl;
  isOpen_ = TRUE;
  return TRUE;
 
} // end of method Open (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Close
// DESCRIPTION: Closes stream.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void GBinStream::Close()
{
  GStream::Close();
  gndatasets_ = 0;
} // end of method Close


//************************************************************************************
//************************************************************************************
// METHOD     : pOffsetBlk
// DESCRIPTION: get file pointer offset of data set offset block
//              [NOTE: deep connxn to GBinWriter::WriteMeta]
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GFPOS GBinStream::pOffsetBlk()
{
  if ( !isFile_ )
  {
    ierror_ = ERRNOFILE;
    return 0;
  }

  GFPOS     iloc, icurr=tellg();

  icurr = tellg();
  seekg(pDSNum(),ios::beg);             // skip to location of dataset number
  seekg(i4sz,ios::cur);                 // skip number of datasets
  iloc = tellg();                       // start of offset blk
  seekg(icurr, ios::beg);               // reset get file pointer
  return iloc;

} // end of method pOffsetBlk


//************************************************************************************
//************************************************************************************
// METHOD     : pDSNum
// DESCRIPTION: get file pointer offset of dataset number
//              [NOTE: deep connxn to GBinWriter::WriteMeta;
//                     Also, this method will try to re-read 
//                     data, so if called when iomode = gios::out only,
//                     the results are unpredictable.]
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GFPOS GBinStream::pDSNum()
{
  if ( !isFile_ ) {
    ierror_ = ERRNOFILE;
    return 0;
  }

  GINT4BYTE nm,    ilen;
  GFPOS     iloc, icurr;

  if ( fail() ) {
    ierror_ = ERRNOFILE;
    cout << "GBinStream::pDSNum: General file failure" << endl;
    exit(1);
  }

  icurr = tellg();
  // Skip key section:
  seekg(0,ios::beg);
  seekg(i4sz,ios::cur);                             // skip endedness-indicator
  seekg(ndtypes_*i4sz,ios::cur);                    // skip data type sizes
  read((GUCHAR*)&ilen, i4sz);                       // file identifier length
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ilen, i4sz);
  seekg(ilen,ios::cur);                             // skip identifier
  read((GUCHAR*)&nm, i4sz);                         // num of meta tag elements
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nm, i4sz);
  seekg(nm*fsz,ios::cur);                           // skip meta tag elems
  read((GUCHAR*)&ilen, i4sz);                       // meta descriptor len
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ilen, i4sz);
  seekg(ilen,ios::cur);                             // skip meta descriptor

  iloc = tellg();                                   // start of dataset number 
  
  seekg(icurr, ios::beg);                           // reset get file pointer

  return iloc;

} // end of method pDSNum


//************************************************************************************
//************************************************************************************
// METHOD     : GetDSNum
// DESCRIPTION: gets number of dataset in file
//              [NOTE: This method will try to re-read 
//                     data, so if called when iomode = gios::out only,
//                     the results are unpredictable.]
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinStream::GetDSNum()
{
  if ( !isFile_ )
  {
    ierror_ = ERRNOFILE;
    return -1;
  }
  
  GFPOS icurr = tellg();
  GINT   nds;

  seekg(pDSNum(), ios::beg);
  read((GUCHAR*)&nds,isz);
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&nds, isz);
  seekg(icurr,ios::beg);

  return nds;

} // end of method GetDSNum

//************************************************************************************
//************************************************************************************
// METHOD     : Check
// DESCRIPTION: Verify file integrity
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinStream::Check()
{

  GINT4BYTE one, iOne=1;
  GINT     i ;
  GINT     ilen;
  GFPOS    icurr=tellg();
  char     *sid=NULL;

  sid = new char [strlen(identifier_)+2];

  if ( szkey_ != NULL ) delete [] szkey_; szkey_ = NULL;
  if ( ndtypes_ == 0 ) {
    ierror_ = ERRCORRUPT;
    cout << "GBinStream::Check: FATAL: inconsistent key length" << endl;
    exit(1);
  }
  szkey_ = new GINT4BYTE [ndtypes_];
  if ( szkey_ == NULL ) {
    cout << "GBinStream::Check: FATAL: NULL key data" << endl;
    exit(1);
  }


  // Get file identifier:
  seekg(0,ios::beg);
  read((GUCHAR*)&one, i4sz);                         // endedness indicator
  bEndianSwapped_ = FALSE;
  if ( one != iOne ) {
    GCUtils::swap((GBYTE*)&one, i4sz);
    if ( one != iOne ) {
      if ( sid ) delete [] sid;
      ierror_ = ERRCORRUPT;
      return FALSE;
    }
    bEndianSwapped_ = TRUE;
  }
  for ( i=0; i<ndtypes_; i++ ) {                     // key data type sizes
    read((GUCHAR*)(szkey_+i),i4sz);   
    if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&(szkey_[i]), i4sz);
  }
  if ( bResetSizes_ ) {
    if ( szkey_[0] != isz ) {
      isz = szkey_[0];
    }
    if ( szkey_[1] != sisz ) {
      sisz = szkey_[1];
    }
    if ( szkey_[2] != fsz ) {
      fsz = szkey_[2];
    }
    if ( szkey_[3] != esz ) {
      esz = szkey_[3];
    }
    if ( szkey_[4] != gfsz ) {
      gfsz = szkey_[4];
    }
  }
  read((GUCHAR*)&ilen,isz);                          // file identifier length
  if ( bEndianSwapped_ ) GCUtils::swap((GBYTE*)&ilen, isz);
  read((GUCHAR*)sid,strlen(identifier_)+2);          // file identifier
  sid[strlen(identifier_)] = '\0';
  if ( fail() || strcmp(identifier_,sid) )
  {
    if ( sid ) delete [] sid;
    ierror_ = ERRFILETYPE;
    return FALSE;
  }
  seekg(icurr,ios::beg);
  if ( sid ) delete [] sid;
  return TRUE;
} // end of Check


//************************************************************************************
//************************************************************************************
// METHOD     : EndianSwapped
// DESCRIPTION: Get endian swap flag
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinStream::EndianSwapped()
{
  return bEndianSwapped_;
} // end of method EndianSwapped


//************************************************************************************
//************************************************************************************
// METHOD     : ResetSizes
// DESCRIPTION: sets flag to reset default datatype sizes, in the event
//              that the default values don't agree with those in the file's key.
// ARGUMENTS  : bReset == TRUE to reset; else don't
// RETURNS    : none
//************************************************************************************
void GBinStream::ResetSizes(GBOOL bReset)
{
  bResetSizes_ = bReset;
} // end of method ResetSizes

