//************************************************************************************/t
// Module       : gbin_writer.hpp
// Date         : 10/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the object that writes a GASpAR binary data file
// Derived From : GBinStream.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include <stdio.h>
#if defined(_LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include <iostream.h>
#include "gbin_writer.hpp"
#include "gdd_file.h"
#if defined(DO_GBIN_TIMING)
#  include "timer.h"
#endif

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GBinWriter::GBinWriter(GBOOL isCollective, GBOOL isIndependent, GSHORT  ioTaskID)
: GBinStream(isCollective, isIndependent, ioTaskID),
rank_         (0),
idims_        (NULL),
nVert_        (0),
nVertMax_     (0),
nMeta_        (0),
nTags_        (0),
NN            (0),
nTotalWritten_(0),
ndsdelta_     (0),
gnds_         (0),
bRankSet_     (FALSE),
bIsNew_       (TRUE),
bNewCoord_    (FALSE),
etype_        (RECT_QUAD),
coordOffset_  (0),
datasegOffset_   (0),
fMeta_        (NULL),
Vert_         (NULL),
xCoord_       (NULL),
nCoord_       (NULL),
sMeta_        (NULL),
sCoord_       (NULL),
sData_        (NULL)
{
#if defined(DO_GBIN_TIMING)
  time_result_ = 0.0;
#endif
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Destructor
GBinWriter::~GBinWriter()
{
  DeleteDynamic();
}


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void GBinWriter::DeleteDynamic()
{
  GINT  i;

  if ( sCoord_ ) {
    for ( i=0; i<rank_; i++ ) 
      if ( sCoord_[i] ) delete [] sCoord_[i];
    delete [] sCoord_; sCoord_ = NULL;
  }

  if ( xCoord_!= NULL ) delete [] xCoord_ ; xCoord_ = NULL;
  if ( nCoord_!= NULL ) delete [] nCoord_ ; nCoord_ = NULL;
  if ( sMeta_ != NULL ) delete [] sMeta_; sMeta_ = NULL;
  if ( sData_ != NULL ) delete [] sData_; sData_ = NULL;

  if ( fMeta_   != NULL ) delete [] fMeta_   ; fMeta_    = NULL;
  if ( idims_   != NULL ) delete [] idims_   ; idims_    = NULL;
  if ( Vert_    != NULL ) delete [] Vert_    ; Vert_     = NULL;

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Open (1)
// DESCRIPTION: Opens file; checks header for validity; etc
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL GBinWriter::Open(const char *fn)
{
  char      *serr = "GBinWriter::Open(1): ";
  GIOS_MODE iomode;

//iomode = gios::inout | gios::create;
  iomode = gios::inout;
  if ( !GBinStream::Open(fn, iomode) ){
    cout << serr << "GStream::Open failed" << endl;
    return FALSE;
  }

  if ( tellp() == 0 )
    bIsNew_ = TRUE; 
  else
    bIsNew_ = FALSE;

  isOpen_ = TRUE;
  return TRUE;
 
} // end of method Open (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Open (2)
// DESCRIPTION: Opens file; checks header for validity, etc.. Caller may specify
//              iomode options here.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL GBinWriter::Open(const char *fn, GIOS_MODE iomode, GBOOL bDelete)
{ 
  char *serr = "GBinWriter::Open(2): ";
  if ( ! ( iomode && gios::out ) ) {
    cout << serr << "invalid GBinWriter io mode" << endl;
    return FALSE;
  }
  if ( !GBinStream::Open(fn, iomode, bDelete) ) {
    cout << serr << "GStream::Open failed" << endl;
    return FALSE;
  }
  
  if ( tellp() == 0 )
    bIsNew_ = TRUE; 
  else
    bIsNew_ = FALSE;

  isOpen_ = TRUE;
  return TRUE;
 
} // end of method Open (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Close
// DESCRIPTION: Closes stream.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void GBinWriter::Close()
{
  char *serr = "GBinWriter::Close: ";

//cout << serr << "closing file..." << endl;

  // First, update meta data due to writes:
  if ( !isOpen_ ) return;
  if ( !UpdateMeta() ) {
    cout << "GBinWriter::Close: Cannot update meta data" << endl;
    ierror_ = ERRDATABLKW;
    return;
  }
  GBinStream::Close();
  DeleteDynamic();
  isOpen_        = FALSE;
  bRankSet_      = FALSE;
  bIsNew_        = TRUE;
  bNewCoord_     = FALSE;
  coordOffset_   = 0;
  datasegOffset_ = 0;
  ndsdelta_      = 0;
//cout << serr << "file closed." << endl;

} // end of method Close


//************************************************************************************
//************************************************************************************
// METHOD     : SetDims
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GBOOL GBinWriter::SetDims(GINT  irank, GINT  *dims)
{
  GINT  i;

  if ( !isOpen_ ) 
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }
  if ( dims == NULL || irank <= 0 )
  {
    ierror_ = ERRDIMENSION;
    return FALSE;
  }

  if ( idims_ != NULL ) delete [] idims_; idims_ = NULL;
  if ( xCoord_ ) {
    delete [] xCoord_; xCoord_ = NULL;
  }
  if ( sCoord_ ) {
    for ( i=0; i<rank_; i++ ) if ( sCoord_[i] ) delete [] sCoord_[i];
    delete [] sCoord_; sCoord_ = NULL;
  }
  if ( nCoord_ ) delete [] nCoord_; nCoord_ = NULL;
  if ( Vert_ != NULL ) delete [] Vert_; Vert_ = NULL;
  nVert_ = 0; etype_ = UNDEFINED;

  rank_  = irank;
  idims_ = new GINT  [rank_];
  memcpy(idims_, dims, rank_*isz);

  NN = 1;
  sCoord_ = new char  * [rank_];
  xCoord_ = new GDOUBLE * [rank_];
  nCoord_ = new GINT    [rank_];
  for ( i=0; i<rank_; i++ ) {
    sCoord_[i] = NULL;
    xCoord_[i] = NULL;
    NN *= idims_[i];
  }
  nVertMax_ = (GINT )pow(2.0,rank_);
  bRankSet_ = TRUE;
  bNewCoord_ = TRUE;  // must be set so that dims will be written

  return TRUE;
} // end of method SetDims


//************************************************************************************
//************************************************************************************
// METHOD     : SetMeta
// DESCRIPTION: Sets meta data. 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
GBOOL GBinWriter::SetMeta(GINT  nm, GDOUBLE *meta_data, const char *descriptor )
{
  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }
  if ( nm > 0 && meta_data == NULL )
  {
    ierror_ = ERRMETAW;
    return FALSE;
  }
  if ( nm <= 0 || meta_data == NULL ) return TRUE;

  if ( fMeta_ ) delete [] fMeta_; fMeta_ = NULL;
  if ( sMeta_ ) delete [] sMeta_; sMeta_ = NULL;
  nMeta_ = nm;
  fMeta_ = new GDOUBLE [nMeta_];
  memcpy(fMeta_, meta_data, nMeta_*fsz);
  if ( descriptor != NULL ) {
    sMeta_ = new char [strlen(descriptor)+1];
    strcpy(sMeta_,descriptor);
  }

  return TRUE;

} // end of method SetMeta


//************************************************************************************
//************************************************************************************
// METHOD     : SetVertices
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinWriter::SetVertices(const GINT  num, Point *vertices, ELEMTYPE et)
{
  GINT  i, j, nv=rank_*num;

  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE; 
  }
  if ( !bRankSet_ ) 
  {
    ierror_ = ERRDIMENSION;
    return FALSE; 
  }
  if ( num != nVertMax_ )
  {
    ierror_ = ERRDIMENSION;
    return FALSE; 
  }

  etype_ = et;
  if ( Vert_ != NULL ) delete [] Vert_; Vert_ = NULL;
  Vert_ = new GDOUBLE [nv];
  if ( num <= 0 || vertices == NULL ) {
    memset(Vert_,'\0',nv);
  }

  nVert_ = num;
  for ( i=0; i<nVert_; i++ )  {
    for ( j=0; j<rank_; j++ ) Vert_[i*rank_+j] = vertices[i][j];
  }

  return TRUE;

} // end of method SetVertices


//************************************************************************************
//************************************************************************************
// METHOD     : SetCoord
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinWriter::SetCoord(GINT  idir, GINT  dim, GDOUBLE *coord , const char *descriptor)
{ 
  

  if ( !isOpen_ )
  { 
    ierror_ = ERRNOFILE;
    return FALSE; 
  }
  if ( !bRankSet_ ) 
  { 
    ierror_ = ERRDIMENSION;
    return FALSE; 
  }
  if ( idir <= 0 || idir > rank_ || coord == NULL )
  {
    ierror_ = ERRDIMENSION;
    return FALSE;
  }
  
  if ( sCoord_[idir-1] ) delete [] sCoord_[idir-1]; sCoord_[idir-1] = NULL;
  if ( descriptor  != NULL ) {
    sCoord_[idir-1] = new char [strlen(descriptor)+1];
    strcpy(sCoord_[idir-1],descriptor);
  }
  xCoord_[idir-1] = coord;
  nCoord_[idir-1] = dim;
  bNewCoord_ = TRUE;
 
  return TRUE;

} // end of method SetCoord


//************************************************************************************
//************************************************************************************
// METHOD     : WriteData (1)
// DESCRIPTION: Carries out actual putting of data set to file, after
//              the coordinate system and tags, etc. have been set. 
//              Successive calls will append already open file, and the
//              meta data will be automcatically updated when the file 
//              is closed.
// ARGUMENTS  : 
// RETURNS    : 
//************************************************************************************
GBOOL GBinWriter::WriteData(GINT  ir, GINT  *dims, GDOUBLE *data,  GINT  ntags, GDOUBLE *ftags, 
                            const char *descriptor, GFPOS *datablkoffret)
{
  char serr[]  = "GBinWriter::WriteData: ";
  GINT   i, lnds[2]={0}, gnds[2]={0,0};
  GSHORT  procid=GComm::WorldRank();
  GBOOL  bOK=( ir == rank_ && dims != NULL );
  GFPOS  datablkoffset;
  GFPOS  lsz[3]={0}, gsz[3]={0}, istart, disp;

  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }
  if ( !bRankSet_ )
  {
    ierror_ = ERRDIMENSION;
    return FALSE;
  }
  for ( i=0; i<rank_&&bOK; i++ ) bOK = bOK && (dims[i] == idims_[i]);
  if ( !bOK || data == NULL )
  {
    cout << "GBinWriter::WriteData: NULL data" << endl;
    ierror_ = ERRBADDATA;
    return FALSE;
  }
  nTags_ = ntags;
  if ( sData_ ) delete [] sData_; sData_ = NULL;
  if ( descriptor != NULL ) {
    sData_ = new char [strlen(descriptor)+1];
    strcpy(sData_, descriptor);
  }

  lsz[0] = MetaLen();
  lsz[1] = DataLen();
  lsz[2] = tellp();
  if ( bNewCoord_ ) lsz[1] += CoordLen();
  GComm::Allreduce(lsz, gsz, 3, GC_GFPOS, G_OP_MAX, &IOCommMPI_);
   
  istart = gsz[2];
  if ( bIsNew_ ) istart += gsz[0];  // add Meta segment to start
  disp = istart + procid * gsz[1];

  nTotalWritten_ = 0;
#if defined(DO_GBIN_TIMING)
  time_result_ = STK::Timer();
#endif

  if ( bIsNew_ && WriteMeta() != 0 )
  {
    cout << serr << "WriteMeta failed" << endl;
    ierror_ = ERRMETAW;
    return FALSE;
  }
  //SetFileView(disp);

  seekp(disp, ios::beg);

  GComm::Synch(&IOCommMPI_);
  if ( bNewCoord_ ) 
  {
    if ( (coordOffset_=WriteCoordBlock()) < 0 ) {
      cout << serr << "WriteCoordBlock failed" << endl;
      ierror_ = ERRCOORDBLKW;
      return FALSE;
    }
    bNewCoord_ = FALSE;
  }

  GComm::Synch(&IOCommMPI_);
  if ( (datasegOffset_=PutData(data,  ntags, ftags, descriptor, coordOffset_, datablkoffset)) < 0  ) 
  {
    cout << serr << "PutData failed" << endl;
    ierror_ = ERRDATABLKW;
    return FALSE;
  }
  if ( datablkoffret != NULL ) *datablkoffret = datablkoffset;
  bIsNew_ = FALSE;
  lnds[0] = ndsdelta_;    // pevious number
  lnds[1] = ++ndsdelta_;  // new number 
  GComm::Allreduce(&lnds, &gnds, 2, GC_GINT , G_OP_SUM, &IOCommMPI_);
  if ( gnds[1] > MAX_DATASETS ) {
    FileSynch();
    GComm::Synch(&IOCommMPI_);
    ierror_ = ERRMETAW;
    return FALSE;
  }
  gnds_  = gndatasets_ + gnds[1];         // new global number of datasets to be written to meta data
  
  GComm::Allgather(&datasegOffset_, 1, GC_GFPOS, dsOffset_+gndatasets_+gnds[0], 1, GC_GFPOS, &IOCommMPI_);

#if 0
// This isn't required anymore: we can to this once on close,
// by using UpdateMeta (2)
//FileSynch();
  if ( !UpdateMeta(gnds[1],gnds[0]+procid, datasegOffset_) )
  {
    cout << serr << "UpdateMeta failed" << endl;
    ierror_ = ERRDATABLKW;
    return FALSE;
  }
  FileSynch();
  GComm::Synch(&IOCommMPI_);
  FileSynch();
#endif

#if defined(DO_GBIN_TIMING)
  time_result_ = STK::Timer() - time_result_;
#endif
  
  return TRUE;

} // end of method WriteData (1)


//************************************************************************************
//************************************************************************************
// METHOD     : RewriteData
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GBinWriter::RewriteData(GFPOS datastart, GINT  n, GINT  *inewdata, GDOUBLE *newdata)
{
  GINT    i;
  GFPOS   icurr, ipos;
  GDOUBLE   *fnew;

  if ( !isOpen_  )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }
  if ( n <= 0 || newdata == NULL || inewdata == NULL ) return TRUE;

  icurr = tellp();
  for ( i=0; i<n; i++ ) {
    ipos = datastart+inewdata[i]*fsz;
    seekp(ipos,ios::beg);
    fnew = newdata + i;
    write((GUCHAR*)fnew, fsz);     
    if ( fail() ) {
      ierror_ = ERRDATABLKW;
      return FALSE;
    }
  }
  
  seekp(icurr,ios::beg);

  return TRUE;

} // end of method RewriteData


//************************************************************************************
//************************************************************************************
// METHOD     : InitData (1)
// DESCRIPTION: Performs a WriteData, essentially, of a dataset using
//              a default value of cdata. The coordinates, meta tags, etc.
//              must have been set prior to this call. 
//              Note:  a call to UpdateMeta() must be made by caller.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GFPOS GBinWriter::InitData(GINT  ir, GINT  *dims, GDOUBLE cdata,  GINT  ntags, GDOUBLE *ftags, const char *descriptor)
{
  char serr[]  = "GBinWriter::Initdata: ";
  GINT   i;
  GFPOS  idatablk;
  GDOUBLE  *data;

  data = new GDOUBLE [NN];
  for ( i=0; i<NN; i++ ) data[i] = cdata; 

  if ( !WriteData(ir, dims, data, ntags, ftags, descriptor, &idatablk) ) {
    delete [] data;
    return GFPOS_NULL;
  }
  delete [] data;

  return idatablk;

} // end of method InitData (1)


//************************************************************************************
//************************************************************************************
// METHOD     : WriteMeta
// DESCRIPTION: Write key and meta data.
//              Note: may want to embed endedness in the meta data, together
//                    with info on the word size of machine which produced
//                    the data.
// ARGUMENTS  :
// RETURNS    :  0 on success; else > 0 
//************************************************************************************
GFPOS GBinWriter::WriteMeta()
{
  char serr[]  = "GBinWriter::WriteMeta: ";
  GINT4BYTE   i, iOne=1;
  GINT4BYTE   slen=0, ilen=0, siZERO=0;
  GFPOS       ipZERO=0, ntot=0, ptot=0;

  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return GFPOS_NULL;
  }
#if 1
  if ( tellp() > 0 )
  {
    ierror_ = ERRFILEXISTS;
    return  GFPOS_NULL;
  }
#endif

  // Compute meta data section size, in order
  // to use just one write:

  if ( sMeta_      != NULL ) slen = (GINT4BYTE)strlen(sMeta_);
  if ( identifier_ != NULL ) ilen = (GINT4BYTE)strlen(identifier_);  

  ntot = MetaLen();
 
  if ( ntot == 0 ) { 
    cout << serr << "invalid meta-data size" << endl; 
    return GFPOS_NULL;
  }
  hMeta_.Resize(MAX(hMeta_.dim(),ntot));
  seekp(0,ios::beg);
  // Key info:
  memcpy(hMeta_.Data()+ptot,&iOne      ,  i4sz         ); ptot += i4sz;           // endedness indicator (the no. 1)
  memcpy(hMeta_.Data()+ptot,&isz       ,  i4sz         ); ptot += i4sz;           // GINT -byte length
  memcpy(hMeta_.Data()+ptot,&sisz      ,  i4sz         ); ptot += i4sz;           // GSHORT -byte length
  memcpy(hMeta_.Data()+ptot,&fsz       ,  i4sz         ); ptot += i4sz;           // GDOUBLE-byte length
  memcpy(hMeta_.Data()+ptot,&esz       ,  i4sz         ); ptot += i4sz;           // ELEMTYPE-byte length
  memcpy(hMeta_.Data()+ptot,&gfsz      ,  i4sz         ); ptot += i4sz;           // GFPOS-byte length
  memcpy(hMeta_.Data()+ptot,&ilen      ,  i4sz         ); ptot += i4sz;           // file identifier length
  if ( identifier_ != NULL )
  memcpy(hMeta_.Data()+ptot,identifier_,  ilen         ); ptot += ilen;           // file identifier 
  // Meta data info:
  memcpy(hMeta_.Data()+ptot,&nMeta_    ,  i4sz         ); ptot += i4sz;           // number of meta data elem
  if ( fMeta_ != NULL )
  memcpy(hMeta_.Data()+ptot,fMeta_     ,  (nMeta_*fsz) ); ptot += (nMeta_*fsz);   // meta data elem
  memcpy(hMeta_.Data()+ptot,&slen      ,  i4sz         ); ptot += i4sz;           // meta data descriptor length
  if ( sMeta_ != NULL )
  memcpy(hMeta_.Data()+ptot,sMeta_     ,  slen         ); ptot += slen;           // meta data descriptor 
//cout << "GBinWriter: WriteMeta: pDSNum=" << ptot << endl;
  memcpy(hMeta_.Data()+ptot,&siZERO    ,  i4sz         ); ptot += i4sz;           // current number of datasets
  for ( i=0; i<MAX_DATASETS; i++ ) {                                              // initialize dataset offsets block
    memcpy(hMeta_.Data()+ptot,&ipZERO  , gfsz          ); ptot += gfsz;          
  }
  if ( ntot != ptot ) {
    cout << serr << "FATAL: expected number of bytes not buffered" << endl;
    cout << serr << "       number expected=" << ntot << "; number received=" << ptot << endl;
    exit(1); 
  }

  write(hMeta_.Data(), ptot);                                                     // write meta data to file 

  if ( fail() ) { 
    cout << serr << "Meta write failed" << endl; 
    return GFPOS_NULL;
  }
  
  nTotalWritten_ += ptot;

  return (GFPOS)0; 
} // end of WriteMeta


//************************************************************************************
//************************************************************************************
// METHOD     : UpdateMeta (1)
// DESCRIPTION: update meta data
// ARGUMENTS  : nds    : new no., of datasets
//              ids    : index (0:nds-1) of dataset offset record to modify
//              nfo    : value to write in the ids-th record of dataset offset block 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GBinWriter::UpdateMeta(GINT  nds, GINT  ids, GFPOS nfo)
{ 
  char serr[]  = "GBinWriter::UpdateMeta: ";

  GFPOS     ntot=0, icurr, mfp;

  if ( !isOpen_ )
  { 
    ierror_ = ERRNOFILE;
    return FALSE;
  }

  icurr = tellp();
  seekp(pDSNum(),ios::beg);                  // move to file position  of dataset number
  write((GUCHAR*)&nds, isz);                 // update first meta data element == no. data sets

  mfp = pOffsetBlk()+ids*gfsz;
  seekp(mfp, ios::beg );                     // move to meta-location in file to change
  write((GUCHAR*)&dsOffset_, gnds_*gfsz);    // write new data set offset in meta data

  ntot = gfsz + isz;


#if 0
GINT  n;
GFPOS gpos;
seekg(mfp, ios::beg);
read((GUCHAR*)&gpos, gfsz);
seekg(pDSNum(), ios::beg);
read((GUCHAR*)&n, isz);
cout << serr << " check: nsds=" << nds << " read:" << n << endl;
cout << serr << " check: off=" << nfo<< " read:" << gpos << endl;
#endif
  seekp(icurr,ios::beg);

  nTotalWritten_ += ntot;
  return TRUE;
} // end of method UpdateMeta (1)


//************************************************************************************
//************************************************************************************
// METHOD     : UpdateMeta (2)
// DESCRIPTION: update meta data--enmasse
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GBinWriter::UpdateMeta()
{
  char serr[]  = "GBinWriter::UpdateMeta (2): ";

  GFPOS     ntot=0, icurr, mfp, idsn;

  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }

  icurr = tellp();
  mfp = pOffsetBlk();
  seekp(mfp, ios::beg );                             // move to meta-location in file to change
  write((GUCHAR*)dsOffset_, gnds_*gfsz);             // write new data set offset in meta data
  idsn=pDSNum();
  seekp(idsn,ios::beg);                              // move to file position  of dataset number
  write((GUCHAR*)&gnds_, isz);                       // update first meta data element == no. data sets
  ntot = isz + gnds_*gfsz;
  seekp(icurr,ios::beg);
  nTotalWritten_ += ntot;
  return TRUE;
} // end of method UpdateMeta (2)


//************************************************************************************
//************************************************************************************
// METHOD     : WriteCoordBlock
// DESCRIPTION: Write coordinate data
// ARGUMENTS  :
// RETURNS    : file pointer at _start of coordinate block
//************************************************************************************
GFPOS GBinWriter::WriteCoordBlock()
{ 
  char serr[]  = "GBinWriter::WriteCoordBlock: ";
  GINT   i, nv, ncoord;
  GINT   nc;
  GFPOS  ntot=0, ptot, istart=tellp();
  if ( fail() ) { cout << serr << "tellp failed" << endl; return GFPOS_NULL; }
  if ( !isOpen_ )
  { 
    ierror_ = ERRNOFILE;
    return GFPOS_NULL;
  }
  for ( i=0,nc=0; i<rank_; i++ ) nc += (xCoord_[i] != NULL ? 1 : 0);
  nv = rank_ * nVert_;

  // Get total amount of data to be put to file:
  ntot += isz                          ;                    // rank 
  ntot += (rank_*isz)                  ;                    // dims 
  ntot += esz                          ;                    // element type size
  ntot += isz                          ;                    // number of vertices
  ntot += (nv*fsz)                     ;                    // vertices
  ntot += isz                          ;                    // number of coord arrays
  for ( i=0; i<nc ; i++ ) { 
    ntot += isz                        ;                    // number of coord array elems
    if ( xCoord_[i] != NULL )  {
      ntot += (nCoord_[i]*fsz)         ;                    // coord array elems
    }
  }

  dput_.Resize(MAX(dput_.dim(),ntot));

#if 0
  if ( nVert_ > 0 ) {
     Vert_ = new GDOUBLE [nv];
     memset(Vert_, '\0', nv);
  }
#endif

  ptot = 0;
  memcpy(dput_.Data()+ptot, &rank_       , isz       );  ptot  += isz;                   // rank
  memcpy(dput_.Data()+ptot, idims_       , rank_*isz );  ptot  += (rank_*isz);           // dims
  memcpy(dput_.Data()+ptot, &etype_      , esz       );  ptot  += esz;                   // elemtype
  memcpy(dput_.Data()+ptot, &nVert_      , isz       );  ptot  += isz;                   // number of vertices
  if ( Vert_ && nv > 0 ) 
  memcpy(dput_.Data()+ptot, Vert_        , nv*fsz    );  ptot  += (nv*fsz);              // vertices
  memcpy(dput_.Data()+ptot, &nc          , isz       );  ptot  += isz;                   // number of coord array elems
  for ( i=0; i<nc; i++ ) { 
    ncoord = nCoord_[i]; 
    if ( xCoord_[i] == NULL ) ncoord=0;
    memcpy(dput_.Data()+ptot, &ncoord,      isz       );  ptot  += isz;                   // number of coord array elems
    if ( ncoord )  {
      memcpy(dput_.Data()+ptot,  xCoord_[i], nCoord_[i]*fsz);  ptot  += (nCoord_[i]*fsz);// coord array elems
    }
  }
  if ( ntot != ptot ) {
    cout << serr << "expected number of bytes not buffered" << endl;
    exit(1); 
  }
  write(dput_.Data(), ntot);                                                             // put coord data to file
  if ( fail() ) {
     cout << serr << "coordinate block write failed" << endl;
     return GFPOS_NULL; 
  }
  nTotalWritten_ += ntot; 
  return istart;
  
} // end of method WriteCoordBlock


//************************************************************************************
//************************************************************************************
// METHOD     : PutData
// DESCRIPTION: actually put the data segment to the file
// ARGUMENTS  : data      :  data set data
//              descriptor:  data descriptor
//              coord_off :  file offset of associated coordinate system
//              data_off  :  offset of the actual block of data 
//               
// RETURNS    : start of data segment (which includes coord offset, tags, etc.)
//************************************************************************************
GFPOS GBinWriter::PutData(GDOUBLE *data, GINT  ntags, GDOUBLE *ftags, const char *descriptor, 
                          GFPOS coord_off, GFPOS &data_off)
{ 
  char serr[]  = "GBinWriter::PutData: ";

  GINT    ilen=0;
  GFPOS   ntot=0, ptot, istart=tellp();

  if ( !isOpen_ )
  { 
    ierror_ = ERRNOFILE;
    return GFPOS_NULL;
  }
  if ( ntags > 0 && ftags == NULL )
  { 
    cout << "GBinWriter::PutData: tag data does not agree with number" << endl;
    ierror_ = ERRBADDATA;
    return GFPOS_NULL;
  }
  if ( NN > 0  && data == NULL ) {
    cout << "GBinWriter::PutData: invalid NULL data segment" << endl;
    ierror_ = ERRBADDATA;
    return GFPOS_NULL;
  }

  if ( descriptor ) ilen = strlen(descriptor);

  // Find total amount of data to be put to file:
  ntot    +=    gfsz         ;                  // file offset  of associated coord system
  ntot    +=    isz          ;                  // number of dataset tag elements
  ntot    +=    (ntags*fsz)  ;                  // dataset tag elements
  ntot    +=    isz          ;                  // dataset descriptor length
  ntot    +=    ilen         ;                  // dataset descriptor 
  ntot    +=    isz          ;                  // dataset size
  ntot    +=    (NN*fsz)     ;                  // dataset 

  dput_.Resize(MAX(dput_.dim(),ntot));
  ptot = 0;
//cout << "PutData: coord_off=" << coord_off << endl;
  memcpy(dput_.Data()+ptot, &coord_off, gfsz     );    ptot += gfsz;          // file offset of associated coord system
  memcpy(dput_.Data()+ptot, &ntags    , isz      );    ptot += isz;           // number of dataset tag data elems
  memcpy(dput_.Data()+ptot, ftags     , ntags*fsz);    ptot += (ntags*fsz);   // dataset tag data elems
  memcpy(dput_.Data()+ptot, &ilen     , isz      );    ptot += isz;           // dataset descriptor length
  memcpy(dput_.Data()+ptot, descriptor, ilen     );    ptot += ilen;          // dataset descriptor 
  memcpy(dput_.Data()+ptot, &NN       , isz      );    ptot += isz;           // dataset size
  data_off = istart + ptot;                                                   // dataset file ptr location
  if ( data != NULL ) 
  memcpy(dput_.Data()+ptot, data      , NN*fsz   );    ptot += (NN*fsz);      // dataset 
  if ( ntot != ptot ) {
    cout << serr << "expected number of bytes not buffered" << endl;
    exit(1); 
  }
  write(dput_.Data(), ptot);                                                  // put data segment to file
  if ( fail() ) {
    cout << serr << "data block write failed" << endl;
    return GFPOS_NULL; 
  }
  
  nTotalWritten_ += ntot;

  return istart;
} // end of method PutData

 
//************************************************************************************
//************************************************************************************
// METHOD     : MetaLen
// DESCRIPTION: gets GUCHAR size of Meta data segment
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GFPOS GBinWriter::MetaLen()
{ 
  GFPOS ntot=0;

  ntot += i4sz;                                           // ended-ness indicator
  ntot += ndtypes_*i4sz;                                  // basic data byte lengths
  ntot += i4sz;                                           // file identifier length
  if ( identifier_ != NULL ) ntot += strlen(identifier_); // file identifier
  ntot +=  i4sz;                                          // number of meta data elems
  ntot += (nMeta_*fsz );                                  // meta data elems
  ntot +=  i4sz;                                          // meta descriptor len
  if ( sMeta_ != NULL ) ntot += strlen(sMeta_);           // meta descriptor
  ntot +=  i4sz;                                          // number of datasets
  ntot += (MAX_DATASETS*gfsz );                           // dataset offsets

  return ntot;
} // end of MetaLen
  

//************************************************************************************
//************************************************************************************
// METHOD     : CoordLen
// DESCRIPTION: gets GUCHAR size of coordinate data segment
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GFPOS GBinWriter::CoordLen()
{ 
  GINT  i;
  GINT  nv = rank_ * nVertMax_;
  GFPOS ntot=0;

  ntot += isz                          ;                    // rank
  ntot += (rank_*isz)                  ;                    // dims
  ntot += esz                          ;                    // element type size
  ntot += isz                          ;                    // number of vertices
  ntot += (nv*fsz)                     ;                    // vertices
  ntot += isz                          ;                    // number of coord arrays
  for ( i=0; i<rank_; i++ ) {
    ntot += isz                        ;                    // number of coord array elems
    if ( xCoord_[i] != NULL )  {
      ntot += (nCoord_[i]*fsz)         ;                    // coord array elems
    }
  }

  return ntot;
  
} // end of CoordLen
  

//************************************************************************************
//************************************************************************************
// METHOD     : DataLen
// DESCRIPTION: gets GUCHAR size of current data segment
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GFPOS GBinWriter::DataLen()
{ 
  GINT  ilen=0;
  GFPOS ntot=0;

  if ( sData_ != NULL ) ilen=strlen(sData_);
  ntot    +=    gfsz         ;                  // file offset  of associated coord system
  ntot    +=    isz          ;                  // number of dataset tag elements
  ntot    +=    (nTags_*fsz) ;                  // dataset tag elements
  ntot    +=    isz          ;                  // dataset descriptor length
  ntot    +=    ilen         ;                  // dataset descriptor
  ntot    +=    isz          ;                  // dataset size
  ntot    +=    (NN*fsz)     ;                  // dataset

  return ntot;
  
} // end of DataLen
  

//************************************************************************************
//************************************************************************************
// METHOD     : GetTotalWritten
// DESCRIPTION: gets total bytes written
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GBinWriter::GetTotalWritten()
{
  return nTotalWritten_;
} // end of GetTotalWritten

//************************************************************************************
//************************************************************************************
//************************************************************************************
// METHOD     : EndLoop
// DESCRIPTION: creates new communicator for next loop if necessary. This
//              should be called at the end of a GBin dataset loop for all procs.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GBinWriter::EndLoop(GINT loopIndex)
{
  char *serr = "GBinWriter::EndLoop: ";

  if ( !GStream::SwitchingComms(loopIndex) ) return;

  
  Close();
  if ( ierror_ ) {
    cout << serr << "Re-open failed for file: " << filename_ << endl 
                 << "    Writer error: " << Error() << endl;
    exit(1);
  }
  GStream::EndLoop(loopIndex);
  if ( loopIndex < nLoopP_[GComm::WorldRank()]-1 ) {
    if ( !Open(filename_, gios::app | gios::inout ) ) {
      cout << serr << "Re-open failed for file: " << filename_ << endl 
                   << "    Writer error: " << Error() << endl;
      exit(1);
    }
  }
  
#if 0
  GStream::EndLoop(loopIndex);
  if ( GStream::NewComm() ) {
    if ( loopIndex < nLoopP_[GComm::WorldRank()]-1 ) {
//    GComm::Synch(&IOCommMPI_);
      if ( !Open(filename_, gios::ate | gios::app | gios::inout) ) {
        cout << serr << "Re-open failed for file: " << filename_ << endl 
                     << "    Writer error: " << Error() << endl;
        exit(1);
      }
    }
  }
#endif
    
} // end of method EndLoop


#if defined(DO_GBIN_TIMING)
void GBinWriter::ResetTime()
{
  time_result_ = 0.0;
}

GDOUBLE GBinWriter::GetTime()
{
  return time_result_;
}
#endif
