//************************************************************************************//
// Module       : gstream.hpp
// Date         : 12/10/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the GStream object.
// Derived From : fstream.
// Modifications:
//************************************************************************************//
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#if defined(_LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include <unistd.h>
#include "gstream.hpp"
#include "gdd_file.h"

char *streamerror_[] = {
                  "Normal",
                  "End of file",
                  "Invalid file name",
                  "Invalid file header",
                  "Invalid file type",
                  "Corrupt file/Bad data format",
                  "Bad data dimensionality",
                  "No data",
                  "No file",
                  "Miscellaneous error",
                  "Memory allocation error",
                  "Data not found",
                  "Invalid data item",
                  "Relevant data retrieved",
                  "Bad dataset number",
                  "Error writing meta block",
                  "Error writing coordinate block",
                  "Error writing data block",
                  "Error reading meta block",
                  "Error reading coordinate block",
                  "Error reading data block",
                  "File already exists",
                  "File does not exist for operation",
                  "MPI error"
                   };


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GStream::GStream(GBOOL isCollect, GBOOL isIndep, GSHORT  ioTaskID)
:
ierror_       (ERRNONE),
isOpen_       (FALSE),
isFile_       (FALSE),
iscollective_ (isCollect),
isindependent_(isIndep),
deletefirst_  (FALSE),
bNewComm_     (FALSE),
iotaskid_     (ioTaskID),
ios_          (NULL)
{
  GINT    i;
  GSHORT   *tmpids, np=GComm::WorldSize(), rank=GComm::WorldRank();

  filename_ = new char [FILE_NAME_MAX];
  if ( isindependent_ ) iotaskid_ = GComm::WorldRank();

  procExcluded_ .Resize(np);
  procExcluded_ =  0;
  pexcl_        .Resize(np);
  pexcl_        = -1;
  iRanks_       .Resize(np);
  for ( i=0; i<np; i++ ) iRanks_[i] = i;

#if defined(MPI_IO_DEFAULT)
  fhMPI_       = MPI_FILE_NULL;
  voffsetMPI_  = 0;
  etypeMPI_    = MPI_UNSIGNED_CHAR;
  filetypeMPI_ = MPI_UNSIGNED_CHAR;
  ierrorMPI_   = MPI_Errhandler_set(MPI_COMM_WORLD, MPI_ERRORS_RETURN);
  putposnMPI_  = 0;
  getposnMPI_  = 0;
  pviewMPI_    = 0;
  gviewMPI_    = 0;
  ierrorMPI_   = MPI_SUCCESS;
  infoMPI_     = MPI_INFO_NULL;
  MPI_Comm_dup  (MPI_COMM_WORLD,&IOCommMPI_);
  MPI_Comm_group(MPI_COMM_WORLD, &groupWorld_);
  MPI_Comm_group(MPI_COMM_WORLD, &currentGroup_);
  memset(datarepMPI_, '\0', MPI_MAX_DATAREP_STRING);

  iotasklist_ = NULL;
  sisterids_  = NULL;
  nLoopPMax_  = 0;
  bEvenDist_  = FALSE;
  if ( iscollective_ && !isindependent_ ) {
    iotasklist_  = new GSHORT  [np];
    ierrorMPI_   = GComm::Allgather(&iotaskid_, 1, GC_GSHORT , iotasklist_, 1, G_GSHORT );

    tmpids = new GSHORT  [np];
    for ( i=0, nrecvs_=0; i<np; i++ ) {
      if ( iotasklist_[i] == rank ) {
        tmpids[nrecvs_++] = i;
      }
    }   
    if ( nrecvs_ > 0 ) {
      sisterids_ = new GSHORT  [nrecvs_];
      for ( i=0; i<nrecvs_; i++ ) sisterids_[i] = tmpids[i];
    }
    delete [] tmpids;

    // find new MPI communicator:
    IOCommMPI_ = BuildComm();
  }
#elif defined(MPI_GENERIC_DEFAULT) 
  MPI_Comm_dup(MPI_COMM_WORLD,&IOCommMPI_);
#else
  IOCommMPI_ = 0;
#endif



} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Destructor
GStream::~GStream()
{
  char *serr = "GStream::~GStream: ";
#if defined(MPI_IO_DEFAULT)
  GINT isize, cres;

  if ( currentGroup_ != MPI_GROUP_NULL ) {
    ierrorMPI_=MPI_Group_free(&currentGroup_);
    if ( ierrorMPI_ != MPI_SUCCESS ) {
      MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
      cout << serr << "Group_free:MPI_Err: " << sErr_ << endl;
    }
  }
  if ( groupWorld_ != MPI_GROUP_NULL ) {
    ierrorMPI_=MPI_Group_free(&groupWorld_);
    if ( ierrorMPI_ != MPI_SUCCESS ) {
      MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
      cout << serr << "Group_free:MPI_Err: " << sErr_ << endl;
    }
  }


  if ( IOCommMPI_ != MPI_COMM_NULL ) {
//MPI_Comm_size(IOCommMPI_,&isize);
//cout << serr << " comm_size=" << isize << endl;
  ierrorMPI_=MPI_Comm_free(&IOCommMPI_);
  if ( ierrorMPI_ != MPI_SUCCESS ) {
    MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
    cout << serr << "Comm_free:MPI_Err: " << sErr_ << endl;
  }
  }

  if ( iotasklist_ != NULL ) delete [] iotasklist_;  iotasklist_ = NULL;
  if ( sisterids_  != NULL ) delete [] sisterids_;   sisterids_  = NULL;
#endif
  DeleteDynamic();

}

//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void GStream::DeleteDynamic()
{

  if ( filename_!= NULL ) delete [] filename_; filename_ = NULL;
  if ( ios_     != NULL ) delete ios_        ;  ios_     = NULL;

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Error
// DESCRIPTION: Gets error string
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
const char *GStream::Error()
{
  if ( ierror_ < 0 || ierror_ >= ERRMAX ) return " ";
  return streamerror_[ierror_];
 
} // end of method Error


//************************************************************************************
//************************************************************************************
// METHOD     : ErrorID
// DESCRIPTION: Gets error ID
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
GINT  GStream::ErrorID()
{
#if defined(MPI_IO_DEFAULT)
  if ( ierrorMPI_ != MPI_SUCCESS ) {
    ierror_ = ERRMPI;
  }
#endif

  return ierror_;
} // end of method ErrorID


//************************************************************************************
//************************************************************************************
// METHOD     : Open (1) 
// DESCRIPTION: Opens file; useful for general opens.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else, FALSE
//************************************************************************************
GBOOL GStream::Open(const char *fn, GIOS_MODE iomode, GBOOL bDelete)
{
  char   *serr = "GStream::Open: ";
  GINT   maccess;
  GBOOL  chkAcc;
  

  if ( fn == NULL )
  {
    ierror_ = ERRFILENAME;
    return FALSE;
  }
  strncpy(filename_,fn,FILE_NAME_MAX);
  deletefirst_ = bDelete; //(GBOOL)(iomode & gios::trunc);
  chkAcc =  (iomode & gios::in )  &&
         !( (iomode & gios::out) || (iomode & gios::inout) );
  if ( chkAcc && !Access(filename_ ) ) {
     ierror_ = ERRFILENOTEXIST;
     return FALSE;
  }

#if defined(MPI_IO_DEFAULT)
  if ( !(iomode && gios::binary) ) {
    etypeMPI_    = MPI_CHAR;
    filetypeMPI_ = MPI_CHAR;
  }
  maccess = iomode;
  if ( iscollective_ ) maccess = iomode | MPI_MODE_UNIQUE_OPEN;
//if ( deletefirst_ ) cout << serr << "deleting file..." << endl;
  if ( deletefirst_ ) MPI_File_delete(filename_,infoMPI_);
//cout << serr << "MPI_File_open... " << endl;
  ierrorMPI_ = MPI_File_open(IOCommMPI_, filename_, maccess, infoMPI_, &fhMPI_);
//cout << serr << "MPI_File_open done. " << endl;
  if ( ierrorMPI_ != MPI_SUCCESS ) {
    MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
    cout << serr << "MPI_Err: " << sErr_ << endl;
    ierror_ = ERRMPI;
    return FALSE;
  }
  isFile_ = TRUE;
#else
  if ( ios_ != NULL ) delete ios_; ios_ = NULL;

  maccess = (GINT )iomode;
  if ( deletefirst_ ) maccess |= ios::trunc;
  ios_ = new fstream(filename_,(GINT)maccess);
  if ( ios_ == NULL || fail() )
  {
    ierror_ = ERRFILENAME;
    return FALSE;
  }
  isFile_ = TRUE;

#endif

  isOpen_ = TRUE;
  return TRUE;

} // end of method Open(1)


//************************************************************************************
//************************************************************************************
// METHOD     : Close
// DESCRIPTION: Closes stream.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void GStream::Close()
{
#if defined(MPI_IO_DEFAULT)
  if ( isOpen_ ) ierrorMPI_ = MPI_File_close(&fhMPI_);
#else
  if ( ios_ != NULL )
  {
    if ( isOpen_ ) ios_->close();
  }
#endif
  isOpen_ = FALSE;
  isFile_ = FALSE;
} // end of method Close


//************************************************************************************
//************************************************************************************
// METHOD     : SetFileView
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GStream::SetFileView(GFPOS disp)
{
#if defined(MPI_IO_DEFAULT)
  ierrorMPI_ = MPI_File_set_view(fhMPI_, disp, etypeMPI_, filetypeMPI_, datarepMPI_, infoMPI_);
  return (ierrorMPI_ == MPI_SUCCESS ) ;
#endif
  return TRUE;
} // end of method SetFileView


//************************************************************************************
//************************************************************************************
// METHOD     : FileSynch
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GStream::FileSynch()
{
#if defined(MPI_IO_DEFAULT)
  MPI_File_sync(fhMPI_);
//ierrorMPI_ = MPI_File_sync(fhMPI_);
//return (ierrorMPI_ == MPI_SUCCESS ) ;
#endif
} // end of method FileSynch





//************************************************************************************
//************************************************************************************
// METHOD     : SwitchingComms
// DESCRIPTION: returns whether a new communicator will be used in the next loop
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GStream::SwitchingComms(GINT n)
{
#if defined(MPI_IO_DEFAULT)
  GBOOL bSwitch = FALSE;
  for ( GINT i=0; i<GComm::WorldSize() && !bSwitch; i++ ) {
//  bSwitch = (n >= nLoopP_[i]-1 && procExcluded_[i] == 0) ;
    bSwitch = bSwitch || (n >= nLoopP_[i]-1 && procExcluded_[i] == 0);
  }
  return bSwitch;
#else
  return FALSE;
#endif
} // end of method SwitchingComms

//************************************************************************************
//************************************************************************************
//************************************************************************************
// METHOD     : SwitchedComms
// DESCRIPTION: returns whether a different communicator was used in the last loop
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GStream::SwitchedComms(GINT n)
{

#if defined(MPI_IO_DEFAULT)
  if (n==0)
    return true;
  else {
    GBOOL notFound = true;
    for ( GINT i=0; i<GComm::WorldSize() && notFound; i++ ) {
      if ( nLoopP_[i] == n )
        notFound = false;
    }
    return !notFound;
  }
#else
  return FALSE;
#endif
} // end of method SwitchedComms


//************************************************************************************
//************************************************************************************
//************************************************************************************
// METHOD     : EndLoop
// DESCRIPTION: creates new communicator for next loop if necessary
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GStream::EndLoop(GINT loopIndex)
{
  char *serr = "GStream::EndLoop: ";
#if defined(MPI_IO_DEFAULT)

  //Make new communicator, if necessary
  GINT     worldSize = GComm::WorldSize();
  GINT     i, index, iref=-1;
  GINT     nPx = 0 ;
  GINT     nAllReadyExcluded = 0;
  GINT     isize, rank;
  MPI_Group grp;
  MPI_Comm  cpy;

  bNewComm_ = FALSE;

#if 0
  cout << serr << " Initial procExcluded_=" << procExcluded_ << endl;
#endif
  for ( i=0, nPx=0; i<worldSize; i++ ) {
    if ( loopIndex >= nLoopP_[i]-1 && procExcluded_[i]==0 ) {
      pexcl_        [nPx] = i;
      procExcluded_ [i] = 1;
      nPx++;
    } 
  }
#if 0
  bEvenDist_ = TRUE;
  i=0;
  while ( loopIndex >= nLoopP_[i]-1 && i < worldSize ) i++;
  if ( i < worldSize ) iref = nLoopP_[i];
  for ( i=0; i<worldSize; i++ ) {
    if ( loopIndex >= nLoopP_[i]-1 ) continue;
    bEvenDist_ = bEvenDist_ && ( nLoopP_[i] == iref ); 
  }
  cout << serr << "iref=" << iref << " bEvenDist_=" << bEvenDist_ << endl;
#endif

  // Get ranks in the current group relative to original process group.
  // These are the ranks that should be used in changing the group:
  ierrorMPI_ = MPI_Group_translate_ranks(groupWorld_, nPx, pexcl_.Data(), currentGroup_, iRanks_.Data());
  if ( ierrorMPI_ != MPI_SUCCESS ) {
    MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
    cout << serr << "MPI_Group_translate_ranks error: " << sErr_ << endl;
    exit(1);
  }

//GComm::Synch(&IOCommMPI_);
  if ( nPx && !bEvenDist_ ) {
#if 0
  cout << serr << "nloopP=" << nLoopP_ << endl;
  cout << serr << "loopIndex=" << loopIndex << " nPx=" << nPx << endl;
  cout << serr << "New procExcluded=" << procExcluded_ << endl;
  cout << serr << "pexcl_ = " << pexcl_ << endl;
  cout << serr << "iRanks = " << iRanks_ << endl;
#endif
#if 0
    MPI_Group_size(currentGroup_,&isize);
    cout << serr << "creating new comm; old size=" << isize <<  endl;
#endif
    ierrorMPI_= MPI_Group_excl(currentGroup_, nPx, iRanks_.Data(), &currentGroup_);
    if ( ierrorMPI_ != MPI_SUCCESS ) {
      MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
      cout << serr << "MPI_Group_excl error: " << sErr_ << endl;
      exit(1);
    }
    
#if 0
    MPI_Group_size(currentGroup_,&isize);
    cout << serr << "group created; new size=" << isize << endl;
#endif
    ierrorMPI_ = MPI_Comm_create(IOCommMPI_, currentGroup_, &IOCommMPI_);
    if ( ierrorMPI_ != MPI_SUCCESS ) {
      MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
      cout << serr << "Comm_create: MPI_Err: " << sErr_ << endl;
    }
    bNewComm_ = TRUE;
#if 0
    cout << serr << "comm created." << endl;
#endif
  }
#if 0
  if ( loopIndex >= nLoopP_[iotaskid_]-1 ) bNewComm_ == FALSE;
  if ( loopIndex >= nLoopP_[iotaskid_]-1 && IOCommMPI_ != MPI_COMM_NULL ) {
    ierrorMPI_ = MPI_Comm_free(&IOCommMPI_);
    if ( ierrorMPI_ != MPI_SUCCESS ) {
      MPI_Error_string(ierrorMPI_, sErr_, &iLen_);
      cout << serr << "Comm_free: MPI_Err: " << sErr_ << endl;
      exit(1);
    }
    if ( procExcluded_.contains(iotaskid_,index) ) {
      MPI_Group_free(&currentGroup_);
    }
  }
#endif
#endif
} // end of method EndLoop


//************************************************************************************
//************************************************************************************
//************************************************************************************
// METHOD     : NewComm
// DESCRIPTION: Was new comm created in last EndLoop call?
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GStream::NewComm()
{
  return bNewComm_;
}

//************************************************************************************
//************************************************************************************
//************************************************************************************
// METHOD     : InitCommList
// DESCRIPTION: creates an array of communicators
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GStream::InitCommList(GINT nLoop)
{
#if defined(MPI_IO_DEFAULT)
  bEvenDist_ = TRUE;
  nLoopP_.Resize(GComm::WorldSize());
  GComm::Allgather(&nLoop, 1, GC_GINT, nLoopP_.Data(), 1, GC_GINT);
  for ( GINT i=0,nLoopPMax_=0; i<nLoopP_.dim(); i++ )  {
    nLoopPMax_ = MAX(nLoopPMax_,nLoopP_[i]);
    bEvenDist_ = bEvenDist_ && nLoopP_[i] == nLoopP_[0];
  }
//EndLoop(-1);
#endif
} // end of method InitCommList


//************************************************************************************
//************************************************************************************
//************************************************************************************
// METHOD     : FinishedLoop
// DESCRIPTION: restores IOCommMPI_ to be MPI_COMM_WORLD
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************v
void GStream::FinishedLoop()
{
#if defined(MPI_IO_DEFAULT)
  MPI_Comm_free(&IOCommMPI_);
  MPI_Comm_dup(MPI_COMM_WORLD, &IOCommMPI_);
#else
  IOCommMPI_ = 0;
#endif
} // end of method Finished


//************************************************************************************
//************************************************************************************
// METHOD     : read
// DESCRIPTION: binary read
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void GStream::read(GUCHAR *ibuffer, GINT  nbuffer)
{
  char serrmsg[] = "GStream::read: ";

  GINT    i, maxbuffer, nrecv;
  GUCHAR  *cbuffer=NULL, *abuffer=NULL;
  GBOOL   bDelete=FALSE, bRet;
  CHandle rhandle;

#if defined(MPI_IO_DEFAULT)
  MPI_Status status;
  if ( !iscollective_ ) {
    ierrorMPI_ = MPI_File_read(fhMPI_, ibuffer, nbuffer, etypeMPI_, &status);
    return;
  }

  // Read is collective, so find common max and resize data:
  GComm::Allreduce(&nbuffer, &maxbuffer, 1, GC_GINT , G_OP_MAX, &IOCommMPI_);
  if ( nbuffer < maxbuffer ) {
    cbuffer = new GUCHAR [maxbuffer];
    memset(cbuffer, '\0', maxbuffer);
    bDelete = TRUE;
  }
  else cbuffer = ibuffer;

  // Collective read:
  if ( isindependent_ ) {
    ierrorMPI_ = MPI_File_read_all(fhMPI_, cbuffer, maxbuffer, etypeMPI_, &status);
    if ( ibuffer != cbuffer )
    memcpy(ibuffer, cbuffer, nbuffer);
    if ( bDelete && cbuffer ) delete [] cbuffer; 
    return;
  }
 
  // ... in non-independent collective-read, IO task must send data
  // to all its sister tasks after reading collectively:
  bRet = TRUE;
  if ( iotaskid_ != GComm::WorldRank() ) { // recv data from IO task
     GComm::ARecv((void**)&cbuffer, 1, &maxbuffer, NULL, GC_GUCHAR, &iotaskid_, rhandle);
     bRet = bRet && GComm::AWaitOnRecv(rhandle);
     memcpy(ibuffer, cbuffer, nbuffer);
  }
  else {                                   // Send read data to sister tasks for which this task serves as IO task
    abuffer = new GUCHAR [nrecvs_*maxbuffer];
    ierrorMPI_ = MPI_File_read_all(fhMPI_, abuffer, nrecvs_*maxbuffer, etypeMPI_, &status);
    for ( i=0, bRet=TRUE; i<nrecvs_; i++ ) {
      bRet = bRet && GComm::BSend((void**)(abuffer+i*maxbuffer), maxbuffer, 0, NULL, GC_GUCHAR, &sisterids_[i]);
    }
  }
  if ( abuffer ) delete [] abuffer;
  if ( bDelete && cbuffer ) delete [] cbuffer;
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  ios_->read((char*)ibuffer, nbuffer);
  return;
#endif


} // end of method read


//************************************************************************************
//************************************************************************************
// METHOD     : get
// DESCRIPTION: get single character from associated stream, store in
//              specified argument.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GStream::get(char &ch)
{
  char serrmsg[] = "GStream::get: ";
#if defined(MPI_IO_DEFAULT)
  read((GUCHAR*) &ch, 1);
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  ios_->get(ch);
#endif
} // end of method get


//************************************************************************************
//************************************************************************************
// METHOD     : write
// DESCRIPTION: binary write
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void GStream::write(GUCHAR *ibuffer, GINT  nbuffer)
{

  char serrmsg[] = "GStream::write: ";

  GINT   i, maxbuffer, nrecv;
  GUCHAR *cbuffer=NULL, **abuffer=NULL;
  GBOOL  bDelete=FALSE, bRet;

#if defined(MPI_IO_DEFAULT)
  MPI_Status status;
  if ( !iscollective_ ) {
    ierrorMPI_ = MPI_File_write(fhMPI_, ibuffer, nbuffer, etypeMPI_, &status);
    return;
  }

  // Write is collective, so find common max and resize data:
  GComm::Allreduce(&nbuffer, &maxbuffer, 1, GC_GINT , G_OP_MAX, &IOCommMPI_);
  if ( nbuffer < maxbuffer ) {
    cbuffer = new GUCHAR [maxbuffer];
    memcpy(cbuffer, ibuffer, nbuffer);
    memset(cbuffer+nbuffer, '\0', maxbuffer-nbuffer);
    bDelete = TRUE;
  }
  else cbuffer = ibuffer;

  // Collective write:
  if ( isindependent_ ) {  
    ierrorMPI_ = MPI_File_write_all(fhMPI_, cbuffer, maxbuffer, etypeMPI_, &status);
    if ( bDelete && cbuffer ) delete [] cbuffer; 
    return;
  }
  
  // ... in non-independent collective-write, IO task must receive data
  // from all its sister tasks before writing collectively:
  bRet = TRUE;
  if ( iotaskid_ != GComm::WorldRank() ) { // send data to IO task 
    bRet = GComm::BSend((void**)&cbuffer, maxbuffer, 0, NULL, GC_GUCHAR, &iotaskid_);
  }
  else {                                   // receive all IO data from sisters, include own; write collectively:
    nrecvs_ = PostDataRecvs(abuffer, maxbuffer); // abuffer allocated in call
    ierrorMPI_ = MPI_File_write_all(fhMPI_, cbuffer, maxbuffer, etypeMPI_, &status);
    for ( i=0; i<nrecvs_; i++ ) {
      ierrorMPI_ = MPI_File_write_all(fhMPI_, abuffer[i], maxbuffer, etypeMPI_, &status);
      delete [] abuffer[i];
    }
  }
  if ( abuffer ) delete [] abuffer;
  if ( bDelete && cbuffer ) delete [] cbuffer;
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  ios_->write((char*)ibuffer, nbuffer);
  return;
#endif

} // end of method write


//************************************************************************************
//************************************************************************************
// METHOD     : seekg
// DESCRIPTION: set file position for a read
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void GStream::seekg(GFPOS offset, seek_dir origin)
{
  char serrmsg[] = "GStream::seekg: ";
#if defined(MPI_IO_DEFAULT)
  if      ( origin == ios::beg )
    ierrorMPI_ = MPI_File_seek(fhMPI_, offset, MPI_SEEK_SET); 
  else if ( origin == ios::cur ) 
    ierrorMPI_ = MPI_File_seek(fhMPI_, offset, MPI_SEEK_CUR); 
  else if ( origin == ios::end ) 
    ierrorMPI_ = MPI_File_seek(fhMPI_, offset, MPI_SEEK_END); 
  else
   ierrorMPI_ = MPI_ERR_ACCESS;
  
  return;
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  ios_->seekg(offset, origin);
  return;
#endif
} // end of method seekg


//************************************************************************************
//************************************************************************************
// METHOD     : seekp
// DESCRIPTION: set file position for a write
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void GStream::seekp(GFPOS offset, seek_dir origin)
{
  char serrmsg[] = "GStream::seekp: ";
#if defined(MPI_IO_DEFAULT)
  if      ( origin == ios::beg )
    ierrorMPI_ = MPI_File_seek(fhMPI_, offset, MPI_SEEK_SET); 
  else if ( origin == ios::cur ) 
    ierrorMPI_ = MPI_File_seek(fhMPI_, offset, MPI_SEEK_CUR); 
  else if ( origin == ios::end ) 
    ierrorMPI_ = MPI_File_seek(fhMPI_, offset, MPI_SEEK_END); 
  else
   ierrorMPI_ = MPI_ERR_ACCESS;
  
  return;
#if 0
  ierrorMPI_ = MPI_File_get_byte_offset(fhMPI_, &voffsetMPI_, &getposnMPI_);
  if ( ierrorMPI_ != MPI_SUCCESS ) return ;
  ierrorMPI_ = MPI_File_set_view(fhMPI_, &offset, &etypeMPI_, &filetypeMPI_, datarepMPI_);
#endif
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  ios_->seekp(offset, origin);
  return;
#endif
} // end of method seekp


//************************************************************************************
//************************************************************************************
// METHOD     : peek
// DESCRIPTION: do a file peek
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
char GStream::peek()
{
  char serrmsg[] = "GStream::peek: ";
  char cRet;

#if defined(MPI_IO_DEFAULT)
  read((GUCHAR *)&cRet,1);
  seekg(-1,ios::cur);
  return cRet;
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  return ios_->peek();
#endif
} // end of method peek


//************************************************************************************
//************************************************************************************
// METHOD     : tellg
// DESCRIPTION: get current file position for a read
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GFPOS GStream::tellg()
{
  char serrmsg[] = "GStream::tellg: ";
#if defined(MPI_IO_DEFAULT)
  ierrorMPI_ = MPI_File_get_position(fhMPI_, &voffsetMPI_);
  if ( ierrorMPI_ != MPI_SUCCESS ) { ierror_ = ERRMPI; return GFPOS_NULL;}
  ierrorMPI_ = MPI_File_get_byte_offset(fhMPI_, voffsetMPI_, &getposnMPI_);
  if ( ierrorMPI_ != MPI_SUCCESS ) { ierror_ = ERRMPI; return GFPOS_NULL;}
  return getposnMPI_;  
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  return ios_->tellg();
#endif
} // end of method tellg


//************************************************************************************
//************************************************************************************
// METHOD     : tellp
// DESCRIPTION: get current file position for a write
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GFPOS GStream::tellp()
{
  char serrmsg[] = "GStream::tellp: ";
#if defined(MPI_IO_DEFAULT)
  ierrorMPI_ = MPI_File_get_position(fhMPI_, &voffsetMPI_);
  if ( ierrorMPI_ != MPI_SUCCESS ) { ierror_ = ERRMPI; return GFPOS_NULL;}
  ierrorMPI_ = MPI_File_get_byte_offset(fhMPI_, voffsetMPI_, &putposnMPI_);
  if ( ierrorMPI_ != MPI_SUCCESS ) {ierror_ = ERRMPI; return GFPOS_NULL;}
  return putposnMPI_;  
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  return ios_->tellp();
#endif
} // end of method tellp


//************************************************************************************
//************************************************************************************
// METHOD     : eof
// DESCRIPTION: determine whether file pointer  is at EOF
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GBOOL GStream::eof()
{
  char serrmsg[] = "GStream::eof: ";
#if defined(MPI_IO_DEFAULT)
  GINT  beof=0, gbeof;
  GFPOS isz, idisp ;
  ierrorMPI_ = MPI_File_get_size(fhMPI_, &isz);
  if ( ierrorMPI_ != MPI_SUCCESS ) beof = 1;;
  ierrorMPI_ = MPI_File_get_byte_offset(fhMPI_, tellg(), &idisp);
  if ( ierrorMPI_ != MPI_SUCCESS ) beof = 1;
  if ( idisp >= isz ) beof = 1;;
  gbeof = beof;
  if ( iscollective_ || !isindependent_ ) {
    GComm::Allreduce(&beof, &gbeof, 1, GC_GINT , G_OP_MAX);
  }
  return ( gbeof > 0 );
#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  return ios_->eof();
#endif
} // end of method eof


//************************************************************************************
//************************************************************************************
// METHOD     : fail
// DESCRIPTION: determine whether next operation is likely to fail (e.g., in
//              POSIX stream, whether the fail bit has been set)
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL GStream::fail()
{
  char serrmsg[] = "GStream::fail: ";
#if defined(MPI_IO_DEFAULT)
  GINT  bfail=0, gbfail;
  gbfail = bfail = (ierrorMPI_ != MPI_SUCCESS);
  if ( iscollective_ || !isindependent_ ) {
    GComm::Allreduce(&bfail, &gbfail, 1, GC_GINT , G_OP_MAX, &IOCommMPI_);
  }
  return ( gbfail > 0 );

#else
  if ( ios_ == NULL ) {
    cout << serrmsg << "NULL stream handle" << endl;
    exit(1);
  }
  return ios_->fail();
#endif
} // end of method fail

//************************************************************************************
//************************************************************************************
// METHOD     : Access
// DESCRIPTION: Establishes existence of file
// ARGUMENTS  :
// RETURNS    : TRUE if file exists; else FALSE
//************************************************************************************
GBOOL GStream::Access(char *fn)
{
#if defined(MPI_IO_DEFAULT)
  GSHORT    ierr;
  MPI_File fhmpi;
  MPI_Info infompi=MPI_INFO_NULL;
  ierr = MPI_File_open(MPI_COMM_WORLD, fn, MPI_MODE_RDONLY, infompi, &fhmpi);
  if ( ierr == MPI_ERR_NO_SUCH_FILE ||
       ierr != MPI_SUCCESS           ) return FALSE;
  MPI_File_close(&fhmpi);
  return TRUE;
#else
  if ( access(fn,F_OK) ) return FALSE;
  return TRUE;
#endif

} // end of method Access



#if defined(MPI_IO_DEFAULT)
//************************************************************************************
//************************************************************************************
// METHOD     : PostDataRecvs
// DESCRIPTION: Recvs data from other processors, as IO task
//              recvbuffs is allocated fully here; caller responsible for clean-up
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  GStream::PostDataRecvs(GUCHAR **&recvbuffs, GINT  buffsize )
{
  char serrmsg[] = "GStream::PostDataRecvs(): " ;
  if ( GComm::WorldSize() == 1 ) return 0;
  if ( sisterids_         == NULL ) return -1;
  if ( iotasklist_        == NULL ) return -1;
  
  GINT    i, nrecved, *abuffsize;
  GBOOL   bRet;
  CHandle *rhandle;

  rhandle   = new CHandle [nrecvs_];
  recvbuffs = new GUCHAR * [nrecvs_];
  abuffsize = new GINT    [nrecvs_];
  for ( i=0; i<nrecvs_; i++ ) {
    recvbuffs[i] = new GUCHAR [buffsize];
    abuffsize[i] = buffsize;
  }
  GComm::ARecv((void**)recvbuffs, nrecvs_, abuffsize, NULL, GC_GUCHAR, &sisterids_[i], rhandle[i]);
  for ( i=0, bRet=TRUE; i<nrecvs_; i++ ) {
    bRet &= GComm::AWaitOnRecv(rhandle[i]);
  }
  delete [] rhandle;
  delete [] abuffsize;
  if ( !bRet ) return -1;
  return nrecvs_;
} // end of method PostDataRecvs


//************************************************************************************
//************************************************************************************
// METHOD     : BuildComm
// DESCRIPTION: Builds MPI communicator for collective, non-independent writes
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
MPI_Comm GStream::BuildComm()
{
  char serrmsg[] = "GStream::BuildComm(): " ;

  if ( GComm::WorldSize() == 1 ) return 0;
  if ( sisterids_         == NULL ) return -1;
  if ( iotasklist_        == NULL ) return -1;

  GINT   i, nd;
  GSHORT  np=GComm::WorldSize(), rank=GComm::WorldRank();
  GINT   ind;
  GIBuffer  *proclist;
  MPI_Group  group_world, new_group;
  MPI_Comm   new_comm;

  // Find distinct procs that will serve as
  // IO tasks. These will be the same for all
  // procs:
  proclist = new GIBuffer  (np);
  proclist->Set(-1);
  for ( i=0, nd=0; i<np; i++ ) {
    if ( iotasklist_[i] == rank && !proclist->contains(i,ind) ) {
      (*proclist)(nd++) = i; 
    }
  }

  // Create new MPI Comm handle for these IO tasks:
  ierrorMPI_ = MPI_Comm_group(MPI_COMM_WORLD, &group_world);
  if ( ierrorMPI_ == MPI_SUCCESS ) 
    ierrorMPI_ = MPI_Group_incl(group_world, nd, proclist->Data(), &new_group);
  if ( ierrorMPI_ == MPI_SUCCESS ) 
    ierrorMPI_ = MPI_Comm_create(MPI_COMM_WORLD, new_group, &new_comm);

  if ( ierrorMPI_ != MPI_SUCCESS ) { ierror_= ERRMPI; new_comm = MPI_COMM_NULL;}
  delete proclist;

  return new_comm;

} // end of method BuildComm


//************************************************************************************
//************************************************************************************
// METHOD     : in_list
// DESCRIPTION: Deterimines if imember is among those in the list, ilist.
//              argument, index, provies the corresponding member index,
//              if found. Method returns TRUE if found; else FALSE.
// ARGUMENTS  : ilist  : input list to search
//              imember: member we are searching for
//              index  : index of imember in ilist; if imember not found,
//                       this is set to -1.
// RETURNS    : TRUE if imember found; else FALSE
//************************************************************************************
GBOOL GStream::in_list(GSBuffer  *ilist, GINT  imember, GINT  &index)
{
  GINT  i=0;

  index = -1;
  while ( i<ilist->dim() && (*ilist)(i)>-1 && (*ilist)(i)!=imember ) i++;

  if ( i >= ilist->dim() ) return FALSE;

  if ( (*ilist)(i) < 0 ) return FALSE;

  index = i;

  return TRUE;
} // end of  method in_list
#endif


