//************************************************************************************//
// Module       : hdfFile.hpp
// Date         : 8/04/02 (RMP)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Derived class encapsulating the methods and data associated with
//                an HDF-format Gaspar file.
// Derived From : GasparFile (gasparFile.hpp)
// Modifications:
//************************************************************************************//

#if !defined(HDFFILE_HPP)
#define HDFFILE_HPP

#include "mpi.h"
#include "gasparFile.hpp"
#include "hdf5.h"
#include "gtypes.h"
#include <iostream.h>
#include <string.h>
#include <stdlib.h>

class HDFFile : public GasparFile{

public:

  //public methods:
  HDFFile(const char* fileNameIn, IO_TYPE fType = COLLECTIVE_IO, IO_TYPE dType = INDEPENDENT_IO, MPI_Comm commIn = MPI_COMM_WORLD);
  HDFFile(const char* fileNameIn, ISUB** procGrid, ISUB nLists, ISUB* listSizes, IO_TYPE dType = INDEPENDENT_IO, MPI_Comm commIn = MPI_COMM_WORLD);
  ~HDFFile();
  GBOOL close();
  GBOOL open(FILE_ACCESS accessIn, ISUB fileNestDepth = FILE_DEPTH_DEFAULT);
  GBOOL openCreate(GBOOL overWriteIn, ISUB fileNestDepth = FILE_DEPTH_DEFAULT);
  GBOOL openGroup(const char* groupName);
  GBOOL createGroup(const char* groupName, GUSHORT  hint = GROUP_HINT);
  GBOOL closeGroup();
  GBOOL up();
  GBOOL top();
  GBOOL writeMeta(const char* metaName, HDFID location, GDOUBLE meta);
  GBOOL writeMeta(const char* metaName, HDFID location, GDOUBLE *meta, GSHORT  len);
  GBOOL writeMeta(const char* metaName, HDFID location, ISUB meta);
  GBOOL readMeta(const char* metaName, HDFID location, GDOUBLE& meta);
  GBOOL readMeta(const char* metaName, HDFID location, ISUB& meta);
  GBOOL writeVector(const char* dataName, const GDOUBLE* data, ISUB dataSize){ return writeTVector(dataName, data, dataSize); }
  GBOOL writeVector(const char* dataName, const ISUB* data, ISUB dataSize){ return writeTVector(dataName, data, dataSize); }
  GBOOL readVector(const char* dataName, GDOUBLE*& data, ISUB& dataSize){ return readTVector(dataName, data, dataSize); }
  GBOOL readVector(const char* dataName, ISUB*& data, ISUB& dataSize){ return readTVector(dataName, data, dataSize); }
  GBOOL writeMatrix(const char* dataName, GDOUBLE* data, ISUB n1, ISUB n2);
  GBOOL readMatrix(const char* dataName, GDOUBLE*& data, ISUB& n1, ISUB& n2);
  GBOOL checkHDF(HDFID statusIn, char* message = NULL) const;
  HDFID getCurrentHDFObj();

private:

  //private methods:
  GBOOL setFilePlist();
  GBOOL setDataPlist(IO_TYPE ioType, DATA_OPERATION op);
  void cleanPath();
  GBOOL gatherNames(const char* name, char*& nameBuffer, ISUB& offset);
  template<class T> T max(T* array, ISUB length);
  template<class T> GBOOL writeTVector(const char* dataName, const T* data, ISUB dataSize);
  template<class T> GBOOL readTVector(const char* dataName, T*& data, ISUB& dataSize);
  friend HDFERR stripLinkLayer(HDFID location, const char* memName, void* nada);
  friend HDFERR makeLink(HDFID location, const char* memName, void* parentName);

  //private data:
  enum DEGREE {ONED=1, TWOD=2, THREED=3};   //Identifies the dimensionality of data set data
  HDFID  filePlistID;                       //HDF File property list identifier (access template) 
  HDFID  dataPlistID;                       //HDF DataSet property list identifier (access template)
  HDFID** path;                             //Array of HDFIDs used to navigate collective-access HDF file hierarchies
  HDFID fileID;                             //Handle for the currently-open file.
  ISUB cursorD;                             //Offset into path[Depth][] used to record position in the file
  HDFERR status;                            //return status flag for HDF5 library function calls
  GBOOL pass;                               //general success flag
  IO_TYPE dataIO;                           //The data transfer I/O type for this file (independent or collective)
  ISUB nProcs, rank, depth, breadth;        //The number of processes, and pid of current process for this file
  char* nameBuffer;                         //Buffer used to collect dataset and group names across processes
  ISUB nbOffset;                            //Offset size for a single name entry in the buffer
  char* errorMsg;                           //The message used to report errors to the user
  static const ISUB TOP;                    //The cursorD start position indicating the top of the file
  static const char* THIS;                  //The name of this class; used in error reporting
};
#endif









