//************************************************************************************//
// Module       : param_reader.cpp
// Date         : 12/5/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the ParamReader object.
// Derived From : GStream.
// Modifications:
//************************************************************************************//
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#if defined(_LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include "param_reader.hpp"
#include "gdd_file.h"
#include "gcutils.hpp"
#include "gtlist.hpp"

char *serror_paramreader_[] = {
                  "Normal",
                  "End of file",
                  "Invalid file name",
                  "Miscellaneous error",
                  "Memory allocation error",
                  "Parameter not found",
                  "Block not found",
                  "Invalid parameter",
                  "No file",
                  "Bad format specifier",
                  "Bad file format"
                   };



//************************************************************************************
//************************************************************************************
// Constructor Method (1)
ParamReader::ParamReader(GBOOL isCollective, GBOOL isIndependent, GSHORT  ioTaskID)
: GStream(isCollective, isIndependent, ioTaskID),
buffsize_      (2048),
maxParams_     (0),
buff_          (NULL)
{
  buff_     = new char [buffsize_];
  strcpy(delimblk_[0], "{");
  strcpy(delimblk_[1], "}");
  strcpy(delimeol_    , ";");
  strcpy(delimlist_   , ":");
  strcpy(comment_, "//");

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
ParamReader::~ParamReader()
{
  if ( buff_        != NULL ) delete [] buff_       ;  buff_        = NULL;
  Reset();
}

//************************************************************************************
// METHOD     : Reset
// DESCRIPTION: Resets resettables
//
//
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void ParamReader::Reset()
{
  GINT  i;

  sParamDesc_.empty();
  sFormat_.empty();
} // end of method Reset


//************************************************************************************
// METHOD     : SetBuffSize
// DESCRIPTION: sets max buffer size
//
//
// ARGUMENTS  :  n: size
// RETURNS    : none
//************************************************************************************
void ParamReader::SetBuffSize(GINT  n)
{
  if ( buff_ ) delete [] buff_; buff_ = NULL;
  buff_ = new char [n+1];
  buffsize_ = n;
} // end of method SetBuffSize


//************************************************************************************
// METHOD     : GetParams
// DESCRIPTION: Gets parameters from file
// ARGUMENTS  : filename: ASCII file from which to read parameter data
//              blk_name: block w/in file in which data is contained.
//              ...     : _pointers_ to data items whose types are specified in 
//                        format string. These are filled only if the value
//                        correspoding to the parameter description is found
//                        in the file.
// RETURNS    : if <=0, there is an error; else, it is the total number
//              of parameters from the set in SetParams, for which
//              block contains data.
//************************************************************************************
GINT  ParamReader::GetParams(char *filename, char *blk_name, ...)
{

  char       serrmsg_[] = "ParamReader::GetParams: " ;
  GSHORT    *sisz=NULL;
  GINT       i, ia, ireplace=0, nget, *isz=NULL;
  GDOUBLE   *fsz=NULL;
  GBOOL     *bsz=NULL, bOk=TRUE, bFound, bEOB=FALSE;
  char      *bend, c, *pdesc, *peob, *sval, *pval, *ssz=NULL;
  va_list   ap;

  if ( sFormat_.size() == 0 || sParamDesc_.size() == 0 || maxParams_ == 0 ) {
    cout << serrmsg_ << "ParamReader not initialized by call to SetParams" << endl;
    exit(1);
  }

  if ( !isOpen_ ) {
    if ( !GStream::Open(filename,gios::in) ) {
      return 0;
    }
  }
  isOpen_ = TRUE;

  if ( !LocateBlock(blk_name) ) {
    ierror_ = ERRDATANOTFND;
    return 0;
  }

  while ( bOk && (nget=getline(buff_, buffsize_,delimeol_)) > 0 || !eof() ) { 
//  FileSynch();
//  GComm::Synch();
#if 0
    if ( (peob=strtok(buff_,delimblk_[1])) != NULL || strcmp(buff_,delimblk_[1])==0 ) {
      bEOB = TRUE;
    }
#endif
    if ( (pdesc=strtok(buff_,delimlist_)) == NULL ) {
      bOk = FALSE;
      ierror_ = ERRCORRUPT;
      break;
    }
    sval = buff_ + strlen(pdesc) + 1;
    ia = 0;
    bFound = FALSE;
    while ( bOk && !bEOB && !bFound && ia < maxParams_ ) {

      // Compare parameter descriptor with ia-th descr. from call to 
      // SetParams. If ==, then set the pointer value to the parameter
      // value:
//cout << "GetParams: sParamDesc_[" << ia << "]=" << sParamDesc_[ia] << endl;
      if ( strcasecmp(rmwhite(pdesc),sParamDesc_[ia].c_str()) == 0 ) {  // a match...
        pval = rmwhite(sval);
        if ( pval == NULL ) {
          bOk = FALSE;
          ierror_ = ERRCORRUPT;
          break;
        }

        va_start(ap,blk_name);
        for ( i=0; i<=ia && bOk; i++ ) {
          c = sFormat_[i][sFormat_[i].length()-1];
          switch ( c ) {                // set appropriate input argument to file parameter pointer
            case 'b':
              bsz = va_arg(ap, GBOOL *);
              if ( bsz == NULL ) {
                bOk = FALSE;
                break;
              }
              if ( i == ia ) {
                *bsz = (GBOOL)atoi(pval);
              }
              break;
            case 'd':
//            sisz = va_arg(ap, GSHORT  *);
//            if ( sisz == NULL ) {
//              bOk = FALSE;
//              break;
//            }
//            if ( i == ia ) {
//              *sisz = (GSHORT )atoi(pval);
//            }
//            break;
            case 'i':
            case 'l':
              isz = va_arg(ap, GINT  *);
              if ( isz == NULL ) {
                bOk = FALSE;
                break;
              }
              if ( i == ia ) {
                *isz = (GINT )atoi(pval);
              }
              break;
            case 'f':
              fsz = va_arg(ap, GDOUBLE *);
              if ( fsz == NULL ) {
                bOk = FALSE;
                break;
              }
              if ( i == ia ) {
                *fsz = (GDOUBLE)atof(pval);
               }
              break;
            case 's':
              ssz = va_arg(ap, char *);
              if ( ssz == NULL ) {
                bOk = FALSE;
                break;
              }
              if ( i == ia ) {
                strcpy(ssz, pval);
              }
              break;
            default:
              bOk = FALSE;
              break;
          } // end of switch
        } // end of va_arg parameter-get loop
        va_end(ap);
        bFound = TRUE;
        if ( !bOk ) break;
        ireplace++;
      } // end, arg,. match 
      ia++;
    }                                         // end of argument match while loop check
    if ( !bFound && !bEOB ) {
      bOk = FALSE;
      ierror_ = ERRCORRUPT;
//    cout << "ParamReader::GetParams: type: " << pdesc << "; val: " << pval << endl; 
      cout << "ParamReader::GetParams: type: " << pdesc <<  endl; 
      break; 
    }
  } // end of file block read
  GStream::Close();

  bend = rmwhite(buff_);
  if ( nget < 0 || !bOk ) {
    if ( (bend == NULL || !strchr(delimblk_[1],bend[0]) )) {
      cout << serrmsg_ << "Invalid parameter item: " << buff_ << endl;
      exit(1);
    }
  }

  return ireplace;

} // end of method GetParams


//************************************************************************************
// METHOD     : SetParams
// DESCRIPTION: Sets parameter descriptions and types
//
//
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void ParamReader::SetParams(char *format, ...)
{
  va_list ap;
  char    serrmsg_[] = "ParamReader::SetParams: ";

  if ( format == NULL ) {
    cout << serrmsg_ << "NULL format string" << endl;
    exit(1);
  }
  
  if ( isOpen_ ) Close();
  Reset();

  // Sort fomat string: find out how many numeric parameters, and
  // how many string parameters:
  if ( !GCUtils::CheckFormat(&sFormat_, format) ) {
    cout << serrmsg_ << "Invalid format string" << endl;
    exit(1);
  }
  maxParams_ = sFormat_.size();
  va_start(ap, format);
  if ( !GCUtils::SetParams(maxParams_, &sParamDesc_, ap) ) {
    cout << serrmsg_ << "Bad descriptor buffer specification" << endl;
    exit(1);
  }
  va_end(ap);

  if ( sParamDesc_.size() != sFormat_.size() ) {
    cout << serrmsg_ << "Number of descriptors does not match number of format specs" << endl;
    exit(1);
  }

} // end of method SetParams


//************************************************************************************
// METHOD     : LocateBlock
// DESCRIPTION: Locates start of block with name, name
//
//
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE. ErrorID will be set.
//************************************************************************************
GBOOL ParamReader::LocateBlock(const char *name)
{
  GSHORT    i;
  GBOOL  bFound=FALSE;

  if ( !isOpen_ )
  {
    ierror_ = ERRNOFILE;
    return FALSE;
  }

  while ( !bFound )
  {
//  if ( getline(buff_, buffsize_, delimlist_) < 0 ) return FALSE;
    if ( getline(buff_, buffsize_, delimblk_[0]) <= 0 ) return FALSE;
    if ( strcmp(rmwhite(buff_), name) == 0 ) {
       bFound = TRUE;
    }
    else {
      ignore(delimblk_[1]);
//    if ( !skipblock(delimblk_[0],delimblk_[1]) ) return FALSE;
    }
  }

  return TRUE;

} // end of method LocateBlock

//************************************************************************************
// METHOD     : rmwhite
// DESCRIPTION: returns a string with the white space removed.
//              Whitespace here includes spaces, newline, linefeed and other
//              control characters.
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
char *ParamReader::rmwhite(char *str)
{
  GINT  len, ib, ie;

  if ( str == NULL ) return NULL;
  len = strlen(str);
  ib = 0;
  ie = len-1;
  while ( ib < len && (isspace(str[ib]) || iscntrl(str[ib])) ) ib++;
  while ( ie > 0   && (isspace(str[ie]) || iscntrl(str[ie]))  ) ie--;
  if ( (ie-ib+1) <= 0 ) return NULL;
  str[ie+1] = '\0';
  return (str+ib);
} // end of method rmwhite


//************************************************************************************
// METHOD     : getdata_t
// DESCRIPTION: Gets ndata elements of data of type dtype from input string.
//              Method skips over all initial white space.
// ARGUMENTS  :
// RETURNS    : number of data items stored in data array. -1 on failure.
//************************************************************************************
GSHORT  ParamReader::getdata_t(char *sdata, G_DATATYPE dtype, const char *datadelim,
                             const char *termdelim,  void *data, const GSHORT  ndata)
{
  GSHORT   n=0;
  GINT     i, tsize=G_TYPESZ[dtype];
  GDOUBLE  f;
  GQUAD    q;
  GDOUBLE  sf;
  char   *p, *beg, delim[2*DELIM_MAX+1];

//if ( dtype == G_GDOUBLE ) bf = TRUE;
//if ( dtype == G_GQUAD   ) bq = TRUE;
//if ( dtype == G_GFLOAT  ) bsf = TRUE;


  if ( data  == NULL ) return -1;
  if ( sdata == NULL ) return  0;
  if ( datadelim == NULL ) return  0;
  if ( termdelim != NULL )
    sprintf(delim,"%s%s", datadelim, termdelim);
  else
    strcpy(delim, datadelim);


  // skip all non-numeric data:
  beg = sdata;
  i=0;
  while ( i<strlen(sdata) && !(isdigit(*beg) || *beg=='-') ) { i++; beg++; }

  p = strtok(beg,delim);
  while ( p != NULL && n < ndata )
  {
    if ( !isdigit(*p) && *p!='-' )
    {
      n = -1;
      break;
    }
    switch ( dtype )
    {
      case G_GDOUBLE:
        f = atof(p);
        memmove((GBYTE*)data+n*tsize, (GBYTE*) &f, tsize);
        break;
      case G_GQUAD:
        q = atof(p);
        memmove((GBYTE*)data+n*tsize, (GBYTE*) &q, tsize);
        break;
      case G_GFLOAT:
        sf = atof(p);
        memmove((GBYTE*)data+n*tsize, (GBYTE*) &sf, tsize);
        break;
      default:
        i = atoi(p);
        memmove((GBYTE*)data+n*tsize, (GBYTE*) &i, tsize);
    }

    n++;
    p = strtok(NULL,delim);
  }

  return n;

} // end of method getdata_t


//************************************************************************************
// METHOD     : getline
// DESCRIPTION: Gets valid line. Reads stream, filling sbuff, until either
//              num characters have been read, or until a delimiter (in delim)
//              has been reached. If a comment is reached, this is the extent of
//              the line.
// ARGUMENTS  :
// RETURNS    : number of characters filling sbuff, if any. Returns -1 on error.
//************************************************************************************
GSHORT  ParamReader::getline(char *sbuff, const GINT  num, const char *idelim)
{
  GFPOS    ic0;
  GSHORT    i, n=0;
  GBOOL    bComment;
  char     delim[DELIM_MAX+1], c='0';

  if ( !isOpen_ ) return -1;

  if ( sbuff  == NULL ) return  0;
  if ( idelim == NULL ){
     strcpy(delim,"\n\0");
  }
  else
  {
     if ( strlen(idelim) > DELIM_MAX )
       cout << "ParamReader::getline: Warning: input delimiter size too large" << endl;
     strncpy(delim,idelim,DELIM_MAX);
  }

  if ( fail() ) return 0;

//ndelim = strlen(delim);
  get(c);
  // skip over delimters:
  while ( !eof() && strchr(delim,c) )
  {
    get(c);
  }

  // skip over white space:
  while ( !eof() && (isspace(c) || iscntrl(c)) )
  {
    get(c);
  }

  // skip over comments:
  ic0 = tellg();
  while ( c == comment_[0]  && !eof() ) {
    for ( i=1, bComment=TRUE; i<strlen(comment_) && bComment; i++ ) {
      get(c); 
      bComment = bComment && (c == comment_[i]);
    }
    if ( bComment ) {

      get(c);
      while ( !eof() && c != '\n' ) get(c);
    }
    else seekg(ic0,ios::beg);
    get(c);
  }

  // find string before next delimeter:
  n = 0;
  while ( !eof() && n < num-1 && !strchr(delim,c) )
  {
    sbuff[n] = c;
    n++;
    get(c);
  }

  if ( n < num-1 && !strchr(delim,c) )  // delimter not found
  {
    sbuff[n] = '\0';
    return ( n = -1);
  }

  sbuff[n] = '\0';

  return n;

} // end of method getline


//************************************************************************************
// METHOD     : skipbock
// DESCRIPTION: Skips a data block whose delimeters are d0 and d1. Next
//              getline will start with line/data following block.
//              Data block need not begin with *d0.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL ParamReader::skipblock(const char *d0, const char *d1)
{

  if ( !isOpen_ )
  {
     ierror_ = ERRNOFILE;
     return FALSE;
  }

  if ( getline(buff_, buffsize_, d0) == -1 ||
       getline(buff_, buffsize_, d1) == -1 )
  {
     ierror_ = ERREOF;
     return FALSE;
  }

  return TRUE;

} // end of method skipblock

//************************************************************************************
// METHOD     : ignore
// DESCRIPTION: Ignores everything from current position to
//              appearance of first char among delim. If delim
//              is not found, FALSE is returned, and file pointer
//              is unchanged. If delim is found, then file pointer
//              is set to first position beyond the delimiter.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL ParamReader::ignore(const char *delim)
{
  GFPOS p0, p1;

  if ( !isOpen_ )
  {
     ierror_ = ERRNOFILE;
     return FALSE;
  }

  p1 = p0 = tellg();


  while ( !strchr(delim,peek()) && !eof() )
     seekg(++p1,ios::beg);

  if ( eof() )
  {
    seekg(p0,ios::beg);
    return FALSE ;
  }

  while ( strchr(delim,peek()) && !eof() )
     seekg(++p1,ios::beg);

  if ( eof() )
  {
    seekg(p0,ios::beg);
    return FALSE ;
  }

  return TRUE;

} // end of method ignore


