function [u,p,d]=BurgersH(nu,t,x,n,u00,k,wavevec,ostr);
%
% [u p d]=BurgersH(nu,t,x,N,u00,k,[m n]);
%       z=BurgersH(nu,t,x,N,u00,k,[m n],'w');
% (All arguments are optional.)
%
% Compute solution u of Burgers equation
%
% u  + u*u  = nu*u  ,  |x|<=1, u(0,x)=u00-sin(pi*x)+k*sin(2*pi*x),
%  t      x       xx
%
% and optionally the associated heat solution p (-u/2*nu is the
% x-derivative of log p) and x-derivative d of u, for
% viscosity nu (default 1e-2) and as a function of
% time row vector t (default (1:3)/pi) and
% space column vector x (default (-511.5:511.5)'/512).
% This differs from Burgers0.m by using Gauss-Hermite quadrature
% of degree N (default 50).
% The background flow u(0,0)=u00 and fine-flow factor k are 0 by default;
% use k=.5 to get a translating front.
% If the argument [m n] is present then return
% u(m x(:,:,1)+n x(:,:,2),(m^2+n^2) t) with dimensions
% u(1-size,2-size,t-size); [m n] u solves the 2D vector Burgers eq.
% If 'w' (default 'u') is one of 'p' or 'd' then z=p or d.
%
% With z=exp(-1/(2 pi nu)) and r =x-sqrt(4 nu t) x  the 1D formula
%                               j                 j
%
%                 1      N      u00 pi r + cos pi r -.5 k cos 2 pi r
% for p is p = -------- sum w  z        j          j                j.
%              sqrt(pi) j=1  j
%
% Aime' Fournier 2004/5 and 2005/1
%
if nargin<1|isempty(nu)
   nu=1e-2;			% viscosity
end
if nargin<2|isempty(t)
   t=(1:3)/pi;			% times
end
N=length(t);			% number of times
if nargin<3|isempty(x)
   x=(-511.5:511.5)'/512;	% 1D nodes
end
M=length(x);			% number of 1D nodes
if nargin<4|isempty(n)
   n=50;			% quadrature degree
end
if nargin<5|isempty(u00)
   u00=0;			% background flow u00=u(0,0)
end
if nargin<6|isempty(k)
   k=0;				% finer flow k*sin 2*pi*x
end
if nargin>=7&~isempty(wavevec)	% then map 2D to 1D as follows:
   L=size(x(:,:,1),1);		% number of 1st dim nodes
   x=wavevec(1)*x(:,:,1)+wavevec(2)*x(:,:,2);
   x=x(:);			% stack 2D in a column
   M=length(x);			% number of 2D nodes
   t=(wavevec(1)^2+wavevec(2)^2)*t;
   f2D=true;			% 2D flag
else
   f2D=false;
end
[xi w]=Gauss(n-1,'Hermite');	% quadrature nodes & weights
[t x]=meshgrid(pi*sqrt(4*nu*t),pi*x);
if any(t(:)==0)
%  warning('BurgersH:t==0 branch','Not explicitly periodized')
   a=0;
   b=0;
else
   a=exp(-(2*pi./t).^2);	% exp(-1./nu t)
   b=4*pi./t;			% 2./sqrt(nu*t)
end
y=1/(2*nu);
z=exp(-y/pi);
[u p d]=deal(zeros(M,N));	% allocate solutions
[q r s]=deal(p);		% allocate workspace
for j=1:n			% loop over nodes:
   r=x-t*xi(j);			% r=pi*(x-sqrt(4*nu*t)*xi)
   s=cos(r)-.5*k*cos(2*r);
   v=1;%periofac(a,b*xi(j));	% 2-periodization factor
   q=w(j)*z.^(u00*r+s).*v;	% p increment
   p=p+q;			% update p
   s=s-1.5*k*cos(2*r);
   r=sin(r)-k*sin(2*r)-u00;	% r*q = 2*nu*q_x
   u=u-r.*q;			% update u*p = -2*nu*p_x
   d=d+(pi*s+y*r.*r).*q;	% update 2*nu*p_xx
end
u=u./p;				% u=(u*p)/p
d=y*u.*u-d./p;			% u_x = u^2/2*nu-2*nu*p_xx/p
p=p/sqrt(pi);
if f2D				% then map 1D to 2D as follows:
   M=M/L;			% number of 2nd dim nodes
   for v='udp'
      eval([v '=reshape(' v ',[L M N]);'])
   end
end
if nargin>7&nargout==1
   u=eval(ostr);		% evaluate output string
end
%
function v=periofac(a,b)	% 2-periodization factor
e=Inf;
v=1;				% i=0 term
f=1;				% f=a.^(i^2) for i=0
g=1;				% g=a.^(2*i) for i=0
A=cosh(b);
B=sinh(b);
C=1;				% C=cosh(b*xi(j)*i) for i=0
S=0;				% S=sinh(b*xi(j)*i) for i=0
while e/max(abs(v(:)))>eps	% 2-periodization (i) loop
   f=f.*g.*a;			% f=a.^(i^2)
   g=g.*a.*a;			% g=a.^(2*i)
   C=C.*A+S.*B;			% C=cosh(b*xi(j)*i)
   S=S.*A+C.*B;			% S=sinh(b*xi(j)*i)
   h=2*f.*C;			% h=2*a.^(i^2)*.cosh(b*i*xi(j))
   v=v+h;			% v=sum_{i=1}^infty h
   e=max(abs(h(:)));		% max norm of last term
end
