function [z u v p]=GaPoisum(sig,x,nu,t)
%
% [z u v p]=GaPoisum([r s],{x y},nu,t);
% U=GaPoisum([r s a b c],elems,nu,t);
%
% 2D Gaussian periodized using the Poissum sum formula:
%
%      oo  oo           2  2        2  2
% z = sum sum exp(-(x-m) /R  - (y-n) /S )
%     -oo -oo
%
%                    oo        2 2 2
%   = pi S R [1 + 2 sum exp(-pi R m )cos(2 pi m x)]*[1 + ...],
%                   m=1
%
% where R^2 = r^2+4*nu*t, S^2 = s^2+4*nu*t (default R=r, S=s).
% Also div grad p = z, so if z is the vorticity then p is the
% streamfunction, and the velocity (u,v) is given by (v,-u) = grad p.
% The 2nd calling syntax returns [U.Zeta U.V1 U.V2 U.Psi] (and their
% coordinate indexes) multiplied by c and with
% {x y}={elems.x{i}(:,:,1)-a elems.x{i}(:,:,2)-b}
% for i=elems.ilist (default 1:elems.n).
%
if isstruct(x)
   flag=true;
   h=sig(3:5);
   sig=sig(1:2);
   if ~isfield(x,'ilist')
      x.ilist=1:x.n;
   end
   k=1;
else
   flag=false;
   x=struct('ilist',1,'n',1,'x',{{cat(3,x{1},x{2})}});
   h=[0 0];
end
if nargin<3
   nu=0;
end
if nargin<4
   t=0;
end
sig=pi*sig;
t=4*nu*t*pi^2;
b=exp(-sig.^2-t);
c=b.^2;
for i=x.ilist		% element loop
   y={2*pi*(x.x{i}(:,:,1)-h(1)) 2*pi*(x.x{i}(:,:,2)-h(2))};
   f={cos(y{1}) cos(y{2})};
   g={sin(y{1}) sin(y{2})};
   o=ones(size(y{1}));	% assume correct y size
   z=o;
   for l=1:2		% loop over spatial directions:
      r=b(l);		% exp(-(sig(l)^2+t)*n^2)
      s=c(l);		% exp(-(sig(l)^2+t)*2*n)
      w=o;
      a=Inf;		% accuracy tolerance
      n=1;		% wavenumber
      q={f{l} g{l}};	% {cos,sin}(n*y{l})
      while a>eps*max(abs(w(:)))
         d=2*r*q{1};
         w=w+d;
         a=max(abs(d(:)./w(:)));
         q={q{1}.*f{l}-q{2}.*g{l} q{2}.*f{l}+q{1}.*g{l}};
         r=r*s*b(l);	% exp(-(sig(l)^2+t)*n^2)
         s=s*c(l);	% exp(-(sig(l)^2+t)*2*n)
         n=n+1;		% wavenumber
      end
      z=sig(l)*z.*w;
%     nit(l)=n-1;	% no. iterations
   end
   z=z/pi;
   if flag
      U.Zeta{k}=h(3)*z;
      U.Zetaic(k)=i;
      z=U;
      k=k+1;
   end
end
if nargout>1|exist('U')	% compute p, u, v:
%  disp('GaPoisum: compute p, u, v')
   for l=1:2		% loop over spatial directions:
      r=b(l);   	% exp(-(sig(l)^2+t)*n^2)
      s=c(l);		% exp(-(sig(l)^2+t)*2*n)
      p=0;
      a=Inf;		% accuracy tolerance
      n=1;		% wavenumber
      while a>eps	% determine max wavenumber M(l):
         d=r/n;		% exp(-(sig(l)^2+t)*n^2)/n
         p=p+d;
         a=d/p;		% accuracy tolerance
%        disp([l n a])
         r=r*s*b(l);	% exp(-(sig(l)^2+t)*n^2)
         s=s*c(l);	% exp(-(sig(l)^2+t)*2*n)
         n=n+1;		% wavenumber
      end
      M(l)=n-1;	% max wavenumber
   end
   disp(sprintf('GaPoisum: max wavenumbers: (%d,%d)',M))
   k=1;
   for i=x.ilist	% element loop
      y={2*pi*(x.x{i}(:,:,1)-h(1)) 2*pi*(x.x{i}(:,:,2)-h(2))};
      f={cos(y{1}) cos(y{2})};
      g={sin(y{1}) sin(y{2})};
      o=ones(size(y{1}));
      p=0;
      u={0 0};
      for l=1:2		% loop over spatial directions:
         j=3-l;		% orthogonal-direction index
         qm={f{l} g{l}};% {cos,sin}(m*y{l})
         rm=b(l);	% exp(-(sig(l)^2+t)*m^2)
         sm=c(l);	% exp(-(sig(l)^2+t)*2*m)
         for m=1:M(l)
            qn={f{j} g{j}};
            rn=b(j);	% exp(-(sig(j)^2+t)*n^2)
            sn=c(j);	% exp(-(sig(j)^2+t)*2*n)
            w=0;
            for n=1:floor(sqrt(1-(m/M(l))^2)*M(j))
               d=rn*qn{1}/(m^2+n^2);
               w=w+d;
               qn={qn{1}.*f{j}-qn{2}.*g{j} qn{2}.*f{j}+qn{1}.*g{j}};
               rn=rn*sn*b(j);
               sn=sn*c(j);
            end
            p=p+rm*qm{1}.*(1/m^2+w);
            u{j}=u{j}+rm*qm{2}.*(1/m+2*m*w);
            qm={qm{1}.*f{l}-qm{2}.*g{l} qm{2}.*f{l}+qm{1}.*g{l}};
            rm=rm*sm*b(l);
            sm=sm*c(l);	% exp(-(sig(l)^2+t)*2*n)
         end
      end
      p=(-.5*sig(1)*sig(2)/pi^3)*p;
      v=(sig(1)*sig(2)/pi^2)*u{2};
      u=(-sig(1)*sig(2)/pi^2)*u{1};
      if flag
         z.Psi{k}=h(3)*p;
         z.V1{k}=h(3)*u;
         z.V2{k}=h(3)*v;
         [z.Psiic(k) z.V1ic(k) z.V2ic(k)]=deal(i);
         k=k+1;
      end
   end
end
