function [xi,rho]=GausLoba(n)
%
% xi=GausLoba(n)
% [xi rho]=GausLoba(n)
%
% n+1 Gauss-Lobatto-Legendre nodes xi (and weights rho), using
% eigenvalues of the companion matrix of the monomial
% expansion, Funaro (A.1.2) <=> Courant & Hilbert p. 83.
%
% Example:
% 
% for n=1:8			% loop over degrees
%    [xi rho]=GausLoba(n);
%    p=1:2*n;			% compute integrals exactly
%    [P Xi]=meshgrid(p,xi);	% ... for p < 2*n :
%    disp([n (1-(-1).^(1+p))./(1+p)-rho*Xi.^P])
% end
%
% Aime' Fournier 2000/5/5 and 2004/3/31
%
if n>45
   error(sprintf('may find complex xi for n=%d>45 !!',n))
end
c=zeros(1,n);
if n<=0
   error('GausLoba needs n>0')
elseif mod(n,2)==0
   m=n/2;
   c(n-1)=-factorial(n+2)/factorial(m+1)/factorial(m-1)*(-1)^m/2^n;
   for j=1:m-1
      c(n-2*j-1)=(j-m)*(2*j+n+1)/(2*j+1)/j*c(n-2*j+1);
   end
else
   m=(n-1)/2;
   c(n  )= factorial(n+1)/factorial(m+1)/factorial(m  )*(-1)^m/2^n;
   for j=1:m
      c(n-2*j  )=(j-m-1)*(2*j+n)/(2*j-1)/j*c(n-2*j+2);
   end
end
xi=[-1 sort(roots(c))' 1];
xi=.5*(xi-xi(n+1:-1:1));	% remove eps from symmetry
if nargout>1			% compute weights
   rho=legendre(n,xi);
   rho=1./(n*(n+1)*rho(1,:).^2);
   rho=rho+rho(n+1:-1:1);
end
return
%___test lines:
[xi rho]=GausLoba(5);
norm(xi'-[-1,0,0;0,-1,0;0,0,-1;0,0,1;0,1,0;1,0,0]*[1;sqrt((7+2*sqrt(7))/21);sqrt((7-2*sqrt(7))/21)])
norm(rho'-[1,0,0;0,1,0;0,0,1;0,0,1;0,1,0;1,0,0]*[1/15;(14-sqrt(7))/30;(14+sqrt(7))/30])
for n=1:45
   [x w]=GausLoba(n);
   p=2*n+1;
   t=(1+(-1)^p)/(1+p);			% true monomial integral
   disp([n t-w*x'.^p])
end
