function U=dLt(u,idim)
%
% U=dLt(u);
% U=dLt(u,idim);
%
% Given values u(xi(j)) of u(x) on nn Gauss-Lobatto nodes xi(j),
% return nn discrete Legendre transform coefficients U,
% along dimension idim (default 1).  Normalize so that
%
%  nn                             nn
% sum rho(j) u(xi(j)) v(xi(j)) = sum U(k) V(k),
% j=1                            k=1
%
% that is, U(k) is given by the r.h.s. of Deville et al. (2002)
% Eq. (B.3.13) _multiplied by_ sqrt(gamma(k)).
%
persistent L nnold
if nargin<2
   idim=1;
end
d=size(u);
if length(d)~=2
   error(sprintf('need length-2 u-size vector: it''s [%s]',sprintd('%d',d)))
end 
nn=d(idim);			% no. nodes
if isempty(nnold)		% if 1st call to dLt:
   nnold=-Inf;			% assign impossible nn value
end
if nn~=nnold			% then make L:
   nnold=nn;			% new old nn value
   p=nn-1;			% polynomial degree
   js=0:p;			% nn indexes
   oogamma=sqrt([2*js(1:p)'+1
                 p]);		% 1/norm of j'th function
   [xi rho]=GausLoba(p);	% nodes xi & weights on [-1,1]
   L=zeros(nn);			% allocate transform matrix
   for j=js			% loop over modes:
      y=legendre(j,xi);		% associated Legendre polynomials
      L(j+1,:)=y(1,:).*rho;	% j'th row of L
   end
   L=.5*diag(oogamma)*L;	% .5* since rho is on [-1,1]
end
if idim==1
   U=L*u;
else
   U=u*L';
end
