//************************************************************************************//
// Module       : dd_rect.hpp
// Date         : 7/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the object that decomposes a general 2d (3d) rectangular
//                (cubic) domain into a specified number of elements in each
//                coordinate direction.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <iostream.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "dd_rect.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
DD_Rect::DD_Rect()
:
bPartitioned(FALSE),
nd              (2),
nv              (0),
node_dyn_range  (-1),
neighbors    (NULL),
nne          (NULL),
nno          (NULL),
bnno         (NULL),
bdy_ids      (NULL),
ncorner_ids  (4),
lcorner_ids  (NULL),
gcorner_ids  (NULL),
ivert        (NULL),
node_ids     (NULL),
bdy_node_indices(NULL),
graph_coords (NULL),
dual_coords1 (NULL),
dual_coords2 (NULL),
dual_coords3 (NULL)
{
  GINT    i;
  for ( i=0; i<3; i++)
  {
    bPeriodic[i] = FALSE;
    NE       [i] = 0;
    NN       [i] = 0;
    P0       [i] = 0.0;
    P1       [i] = 0.0;
  }
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
DD_Rect::DD_Rect(GFLOAT *X0, GFLOAT *X1, GINT    n)
:
bPartitioned(FALSE),
nd              (n),
nv              (0),
node_dyn_range  (-1),
neighbors    (NULL),
nne          (NULL),
nno          (NULL),
bnno         (NULL),
bdy_ids      (NULL),
lcorner_ids  (NULL),
gcorner_ids  (NULL),
ivert        (NULL),
node_ids     (NULL),
bdy_node_indices(NULL),
graph_coords (NULL),
dual_coords1 (NULL),
dual_coords2 (NULL),
dual_coords3 (NULL)
{
  GINT    i;
  for ( i=0; i<3; i++)
  {
    bPeriodic[i] = FALSE;
    NE       [i] = 0;
    NN       [i] = 0;
    P0       [i] = 0.0;
    P1       [i] = 0.0;
  }
  if ( nd < 1 || nd > 3 )
  {
    cout << "DD_Rect::DD_Rect: invalid dimension" << endl;
    exit(1);
  }

  for ( i=0; i<nd; i++)
  {
    P0[i] = X0[i]; 
    P1[i] = X1[i]; 
  }
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
DD_Rect::~DD_Rect()
{
  DeleteDynamic();
}

//************************************************************************************
// METHOD     : SetDomain
// DESCRIPTION: Sets global domain
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
//************************************************************************************
void DD_Rect::SetDomain(GFLOAT *X0, GFLOAT *X1, GINT    n)
{
  GINT    i;

  if ( n < 1 || n > 3 )
  {
    cout << "DD_Rect::SetDomain: invalid dimension" << endl;
    exit(1);
  }
  nd = n;
  for ( i=0; i<nd; i++)
  {
    P0[i] = X0[i]; 
    P1[i] = X1[i]; 
  }
  bPartitioned = FALSE;
  
} // end of method SetDomain


//************************************************************************************
// METHOD     : SetPartSize
// DESCRIPTION: Sets partition size: number of elements in each direction
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
//************************************************************************************
void DD_Rect::SetPartSize(GINT    Nx, GINT    Ny, GINT    Nz)
{
  
  NE[0] = Nx; NE[1] = Ny; NE[2] = Nz;
  bPartitioned = FALSE;
  
} // end of method SetPartSize


//************************************************************************************
// METHOD     : SetElemDescretization
// DESCRIPTION: Sets descretization of each element. This is the _total_ number
//              of grid points in each coordinate direction (i.e., _not_
//              the number of zones for example).
//              NOTE: may want to generalize this s.t. the descretization
//                    changes for each element (graph vertex).
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
//************************************************************************************
void DD_Rect::SetElemDescretization(GLONG Nx, GLONG Ny, GLONG Nz)
{
  
  NN[0] = Nx; NN[1] = Ny; NN[2] = Nz;
  bPartitioned = FALSE;
  
} // end of method SetElemDescretization


//************************************************************************************
// METHOD     : SetBoundaryCond
// DESCRIPTION: Sets bcs: NOTE: should be last 'Set' method called.
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::SetBoundaryCond(GFLOAT *B0, GFLOAT *B1, GINT    n,  BDYTYPE btype)
{

  if ( n < 1 || n > 3  || n != nd ) 
  {
    cout << "DD_Rect::SetBoundaryCond: invalid dimension" << endl;
    exit(1);
  }

  if ( !bPartitioned && !DoPart() ) return FALSE;

  if ( nd == 1 )  
    return Set1dBdyCond(B0, B1, n, btype);
  else if ( nd == 2 )
    return Set2dBdyCond(B0, B1, n, btype);
  else
    return Set3dBdyCond(B0, B1, n, btype);
    
  return FALSE;  

} // end of method SetBoundaryCond


//************************************************************************************
// METHOD     : Set1dBdyCond
// DESCRIPTION: Sets bcs for a 1d problem
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::Set1dBdyCond(GFLOAT *B0, GFLOAT *B1, GINT    n,  BDYTYPE btype)
{   

  bPeriodic[0] = FALSE;
  if ( btype == PERIODIC )   // if periodic, then entire grid is
  {
    bPeriodic[0] = TRUE;
    node_ids[NE[0]-1][NN[0]-1] = node_ids[0][0];
    return TRUE;
  }

  if ( B0[0] < P0[0] || B0[0] > P1[0] ) return FALSE;
  if ( NE[0] > 1 )
  {
    if ( B0[0] == P0[0] ) bcs[0]      [0] = btype;
    if ( B0[0] == P1[0] ) bcs[NE[0]-1][0] = btype;
  }
  else
  {
    if ( B0[0] == P0[0] ) bcs[0][0] = btype;
    if ( B0[0] == P1[0] ) bcs[0][1] = btype;
  }

  return TRUE;

} // end of method Set1dBdyCond
  
//************************************************************************************
// METHOD     : Set2dBdyCond
// DESCRIPTION: Sets bcs for a 2d problem
//              NOTE: SetCorners should be called prior to
//                    entry.
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::Set2dBdyCond(GFLOAT *B0, GFLOAT *B1, GINT    n,  BDYTYPE btype)
{
  GINT    i, j, iv1, iv2, iedge, m, n0, n1, nc=4;
  GBOOL  bSet;

  iv1 = -1;
  iv2 = -1;

  // Find edge corresponding to input segment
  for ( i=0; i<nc; i++ )
  {
    if ( B0[0] == pV[i].x1 && B0[1] == pV[i].x2 ) 
    {
      iv1 = i;
      break;
    }
  }
  for ( i=0; i<nc; i++ )
  {
    if ( B1[0] == pV[i].x1 && B1[1] == pV[i].x2 ) 
    {
      iv2 = i;
      break;
    }
  }
  if ( iv1 < 0 || iv2 < 0 || 
       (abs(iv1-iv2) != 1 && abs(iv1-iv2) != 3) ) return FALSE;
  iedge = MIN(iv1, iv2);  // edge in range 0-3 (b, r, t, l)
  if ( abs(iv1-iv2) == 3 ) iedge = 3;

  // For all nodes on this edge, set the bc to the
  // input type, except if it is a global corner:
  for ( i=0; i<nv; i++ )
  { 
    for ( j=0; j<bnno[i]; j++ ) {
#if 0
      for ( m=0, bSet=FALSE; m<ncorner_ids && !bSet; m++ ) 
        bSet = node_ids[i][bdy_node_indices[i][j]] == gcorner_ids[m];
      if ( !bSet )  bcs[i][j] = btype;
#endif
      if ( bdy_ids[i][j] == iedge ) bcs[i][j] = btype;
    }
  }

  if ( btype == PERIODIC ) 
  {
    if ( iedge%2 == 0 ) DoPeriodic(2,TRUE);
    else                DoPeriodic(1,TRUE);
  }

  return TRUE;

} // end of method Set2dBdyCond


//************************************************************************************
// METHOD     : ResetPeriodic
// DESCRIPTION: Resets or relabels individual nodes corresp. All Set2dBdyCond
//              calls should be completed before entry, as should all calls to
//              SetCorners method.
// ARGUMENTS  :
// RETURNS    : none  
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::ResetPeriodic()
{
  GINT    i, ii, j, jj, m, n0, n1, iv;
  GBOOL  bc;


  // Check for periodicity condition; 
  // reassign nodes accordingly, if they are
  // not corner nodes. If corner nodes, then
  // reassign only if bc is PERIODIC:  
  if ( bPeriodic[0] )  // X-periodic
  {
    for ( j=0; j<NE[1]; j++ )
    {
      n0 = j*NE[0]; 
      n1 = j*NE[0] + NE[0] - 1;
      for ( jj=0, m=0; jj<NN[1]; jj++ ) {  
        if ( ((bc=bCornerNode( node_ids[n1][(jj+1)*NN[0]-1], iv))
        &&    bcs[ivert[iv]][lcorner_ids[iv]] == PERIODIC)   
        ||   !bc )  
          node_ids[n1][(jj+1)*NN[0]-1] = node_ids[n0][jj*NN[0]];
      }
    }
  }

  if ( bPeriodic[1] )  // Y-periodic
  {
    for ( i=0; i<NE[0]; i++ )
    {
      n0 =  i; 
      n1 = (NE[1]-1)*NE[0] + i;
      for ( ii=0; ii<NN[0]; ii++ ) {
        if ( ((bc=bCornerNode(node_ids[n1][(NN[1]-1)*NN[0]+ii], iv))
        &&    bcs[ivert[iv]][lcorner_ids[iv]] == PERIODIC)    
        ||   !bc )
          node_ids[n1][(NN[1]-1)*NN[0]+ii] = node_ids[n0][ii];
      }
    }
  }
  return TRUE;

} // end of method ResetPeriodic


//************************************************************************************
// METHOD     : Set3dBdyCond
// DESCRIPTION: Sets bcs for a 3d problem
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::Set3dBdyCond(GFLOAT *B0, GFLOAT *B1, GINT    n,  BDYTYPE btype)
{
  cout << "DD_Rect::SetBoundaryCond: 3D not yet working" << endl;
  return FALSE;

} // end of method Set3dBdyCond



//************************************************************************************
// METHOD     : DoPeriodic
// DESCRIPTION: Sets periodicity flag in idir direction, to bval
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
//************************************************************************************
void DD_Rect::DoPeriodic(GINT    idir, GBOOL bval)
{

  if ( idir < 1 || idir > 3 ) return;

  bPeriodic[idir-1] = bval;

} // end of method DoPeriodic


//************************************************************************************
// METHOD     : NumVertices
// DESCRIPTION: Gets number of graph vertices
// ARGUMENTS  :
// RETURNS    : number of vertices (elements)
//************************************************************************************
//************************************************************************************
GINT    DD_Rect::NumVertices()
{

  if ( !bPartitioned && !DoPart() ) return 0;

  return nv;

} // end of method GetNeighborList


//************************************************************************************
// METHOD     : GetNeighborList
// DESCRIPTION: Gets neighbor list for each graph vertex (element)
// ARGUMENTS  : 
// RETURNS    : GINT    ** list:  for each vertex (element), there is a
//              variable number of neightbor indices, arranged from low to high.
//              NULL is returned on failure.
//************************************************************************************
//************************************************************************************
GINT    **DD_Rect::GetNeighborList()
{
 
  if ( !bPartitioned && !DoPart() ) return NULL;

  return neighbors;
  
} // end of method GetNeighborList


//************************************************************************************
// METHOD     : GetNeighborNum
// DESCRIPTION: Gets neighbor number for each vertex
// ARGUMENTS  :
// RETURNS    : GINT    * array:  for each vertex gives number of neighbors
//************************************************************************************
//************************************************************************************
GINT    *DD_Rect::GetNeighborNum()
{

  if ( !bPartitioned && !DoPart() ) return NULL;

  return nne;
 
} // end of method GetNeighborNum


//************************************************************************************
// METHOD     : GetNodeIDs
// DESCRIPTION: Gets global ids for fully descretized domain
// ARGUMENTS  : 
// RETURNS    : GINT    **list:  1 list for each graph vertex (element), 
//              containing the global ids for that element.
//              NULL returned on failure.
//************************************************************************************
//************************************************************************************
GLONG **DD_Rect::GetNodeIDs()
{
  if ( !bPartitioned && !DoPart() ) return NULL;
 
  return node_ids;
 
} // end of method GetNodeIDs


//************************************************************************************
// METHOD     : GetNodeDynRange
// DESCRIPTION: Gets dynamic range for node ids
// ARGUMENTS  :
// RETURNS    : GINT    range number on success; else -1
//************************************************************************************
//************************************************************************************
GLONG DD_Rect::GetNodeDynRange()
{
  if ( !bPartitioned && !DoPart() ) return -1;

  return node_dyn_range;

} // end of method GetNodeDynRange


//************************************************************************************
// METHOD     : GetNodeNum
// DESCRIPTION: Gets number of nodes for each vertex
// ARGUMENTS  :
// RETURNS    : GINT    * array:  for each vertex gives number of nodes
//************************************************************************************
//************************************************************************************
GLONG   *DD_Rect::GetNodeNum()
{

  if ( !bPartitioned && !DoPart() ) return NULL;

  return nno;
 
} // end of method GetNodeNum


//************************************************************************************
// METHOD     : GetBdyNodeIndices
// DESCRIPTION: Gets index lists for boundary nodes. These indices 
//              point to elements in node_ids arrays, and there is
//              one list for each graph vertex (element). If there are
//              no boundary nodes in an element, then the list is NULL. 
// ARGUMENTS  :
// RETURNS    : GINT    **list:  1 list for each graph vertex (element),
//              containing the indices to the global ids for that element's 
//              boundary nodes.
//              NULL returned on failure.
//************************************************************************************
//************************************************************************************
GLONG   **DD_Rect::GetBdyNodeIndices()
{
  if ( !bPartitioned && !DoPart() ) return NULL;

  return bdy_node_indices;

} // end of method GetBdyNodeIndices 


//************************************************************************************
// METHOD     : GetBdyNodeNum
// DESCRIPTION: Gets no. of bdy nodes for each vertex
// ARGUMENTS  :
// RETURNS    : GINT    *bnno:  each graph vertex (element),
//              containing the no of bdy nodes in that element 
//              NULL returned on failure.
//************************************************************************************
//************************************************************************************
GLONG *DD_Rect::GetBdyNodeNum()
{
  if ( !bPartitioned && !DoPart() ) return NULL;

  return bnno;

} // end of method GetBdyNodeNum

//************************************************************************************
// METHOD     : GetBdyCond
// DESCRIPTION: Gets bdy conditions for all bdy nodes  for each vertex
// ARGUMENTS  :
// RETURNS    : **bcs 
//************************************************************************************
//************************************************************************************
BDYTYPE **DD_Rect::GetBdyCond()
{
  if ( !bPartitioned && !DoPart() ) return NULL;

  return bcs;

} // end of method GetBdyCond


//************************************************************************************
// METHOD     : GetbBdy
// DESCRIPTION: Gets flags as to whether or not edge is a global bdy
// ARGUMENTS  :
// RETURNS    : **bdyn
//************************************************************************************
//************************************************************************************
GINT    **DD_Rect::GetbBdy()
{
  if ( !bPartitioned && !DoPart() ) return NULL;

  return bdyn;

} // end of method GetbBdy


//************************************************************************************
// METHOD     : GetGraphVertexCoords
// DESCRIPTION: Gets graph vertex coordinates.
// ARGUMENTS  : idir:  coordinate direction whose coords are desired
// RETURNS    : GINT    *array :  array containing nv coords for input coord direction
//              NULL is returned on failure due to bad direction.
//************************************************************************************
//************************************************************************************
GFLOAT *DD_Rect::GetGraphVertexCoords(GINT    idir)
{
  if ( idir < 1  || idir > nd ) return NULL;  
  if ( !bPartitioned && !DoPart() ) return NULL;
 
  return graph_coords[idir-1];
 
} // end of method GetGraphVertexCoords


//************************************************************************************
// METHOD     : GetDualVertexCoords
// DESCRIPTION: Gets dual graph vertex coordinates.
// ARGUMENTS  : 
// RETURNS    : GINT    **list:  list of length nv containing 2^nd points for 
//              each graph vertex.
//              In 1d: the data type for each vertex is an array GFLOAT  [2]
//              In 2d: the data type for each vertex is an array Point3D [4]
//              In 3d: the data type for each vertex is an array Point3D [8]
//              NULL is returned on failure due to incorrect dimensionality. 
//************************************************************************************
//************************************************************************************
void **DD_Rect::GetDualVertexCoords()
{
  if ( !bPartitioned && !DoPart() ) return NULL;
 
  if ( nd == 1 )
    return (void**)dual_coords1;
  if ( nd == 2 )
    return (void**)dual_coords2;
  else
    return (void**)dual_coords3;
 
} // end of method GetDualVertexCoords


//************************************************************************************
// METHOD     : GetDomainVertices
// DESCRIPTION: Gets vertices for global domain
// ARGUMENTS  :
// RETURNS    : Point3D array of length 2^nd 
//************************************************************************************
//************************************************************************************
Point3D *DD_Rect::GetDomainVertices()
{
  if ( !bPartitioned && !DoPart() ) return NULL;

  return pV;

} // end of method GetDomainVertices


//************************************************************************************
// METHOD     : GetNumDomainVertices
// DESCRIPTION: Gets munber of vertices for global domain
// ARGUMENTS  :
// RETURNS    : GINT  number of verts.
//************************************************************************************
//************************************************************************************
GINT  DD_Rect::GetNumDomainVertices()
{
  if ( !bPartitioned && !DoPart() ) {
    cout << "DD_Rect::GetNumDomainVertices: no paritioning!" << endl;
    exit(1);
  }

  return (GINT   )pow(2.0,nd);

} // end of method GetDomainVertices



//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
//************************************************************************************
void DD_Rect::DeleteDynamic()
{
   
  GINT    i;

  if ( neighbors != NULL )
  {
    for ( i=0; i<nv; i++ ) if ( neighbors[i] != NULL ) delete neighbors[i];
    delete [] neighbors;
  }
  if ( nne != NULL )
    delete [] nne;

  if ( nno != NULL )
    delete [] nno;

  if ( bnno != NULL )
    delete [] bnno;

  if ( node_ids != NULL )
  {
    for ( i=0; i<nv; i++ ) if ( node_ids[i] != NULL ) delete node_ids[i];
    delete [] node_ids;
  }
  
  if ( bdy_node_indices != NULL )
  {
    for ( i=0; i<nv; i++ ) if ( bdy_node_indices[i] != NULL ) delete bdy_node_indices[i];
    delete [] bdy_node_indices;
  }

  if ( lcorner_ids != NULL )
  {
    delete [] lcorner_ids;
  }

  if ( gcorner_ids != NULL )
  {
    delete [] gcorner_ids;
  }

  if ( ivert != NULL )
  {
    delete [] ivert ;
  }

  if ( bdy_ids != NULL )
  {
    for ( i=0; i<nv; i++ ) if ( bdy_ids[i] != NULL ) delete bdy_ids[i];
    delete [] bdy_ids;
  }

  if ( bcs != NULL )
  {
    for ( i=0; i<nv; i++ ) if ( bcs[i] != NULL ) delete bcs[i];
    delete [] bcs;
  }
  if ( bdyn != NULL )
  {
    for ( i=0; i<nv; i++ ) if ( bdyn[i] != NULL ) delete bdyn[i];
    delete [] bdyn;
  }
  
  if ( dual_coords1 != NULL )
  {
      for ( i=0; i<nv; i++ ) if ( dual_coords1[i] != NULL ) delete [] dual_coords1[i];
      delete []  dual_coords1;
  }
  if ( dual_coords2 != NULL )
  {
      for ( i=0; i<nv; i++ ) if ( dual_coords2[i] != NULL ) delete dual_coords2[i];
      delete [] dual_coords2;
  }
  if ( dual_coords3 != NULL )
  {
      for ( i=0; i<nv; i++ ) if ( dual_coords3[i] != NULL ) delete dual_coords3[i];
      delete [] dual_coords3;
  }
    

  if ( graph_coords != NULL )
  {
    for ( i=0; i<nd; i++ ) if ( graph_coords[i] != NULL ) delete graph_coords[i];
    delete [] graph_coords;
  }

  if ( pV != NULL )
  {
    delete [] pV;
  }

  neighbors    = NULL;
  nne          = NULL;
  nno          = NULL;
  bnno         = NULL;
  bdy_ids    = NULL;
  node_ids     = NULL;
  bdy_node_indices = NULL;
  bcs          = NULL;
  bdyn         = NULL;
  dual_coords1 = NULL;
  dual_coords2 = NULL;
  dual_coords3 = NULL;
  graph_coords = NULL;
  pV           = NULL;

 
} // end of method DeleteDynamic


//************************************************************************************
// METHOD     : CreateDynamic
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::CreateDynamic()
{
  GINT    i, j, m, mm, n;

  ncorner_ids = (GINT   )pow(2.0,nd);
  if      ( nd == 1 )
  {
    nv = NE[0];
    mm = NN[0];
    dual_coords1 = new GFLOAT  * [nv];
    bnno         = new GLONG      [nv];
    bdy_ids      = new GLONG    * [nv];
    lcorner_ids  = new GLONG      [ncorner_ids];
    gcorner_ids  = new GLONG      [ncorner_ids];
    ivert        = new GINT       [ncorner_ids];
    bcs          = new BDYTYPE * [nv];
    bdyn         = new GINT     * [nv];
    if ( dual_coords1 == NULL ) return FALSE;
    if ( bnno         == NULL ) return FALSE;
    if ( bdy_ids      == NULL ) return FALSE;
    if ( bcs          == NULL ) return FALSE;
    if ( bdyn         == NULL ) return FALSE;
    for ( i=0; i<nv; i++ )
    {
      dual_coords1[i] = new GFLOAT [2];
      bnno        [i] = 0;
      bdy_ids     [i] = NULL;
      bcs         [i] = NULL;
      bdyn        [i] = NULL;
      if ( dual_coords1[i] == NULL ) return FALSE;
    }
  }
  else if ( nd == 2 )
  {
    nv = NE[0]*NE[1];
    mm = NN[0]*NN[1];
    dual_coords2 = new Point3D * [nv];
    bnno         = new GLONG      [nv];
    bdy_ids      = new GLONG    * [nv];
    lcorner_ids  = new GLONG      [ncorner_ids];
    gcorner_ids  = new GLONG      [ncorner_ids];
    ivert        = new GINT       [ncorner_ids];
    bcs          = new BDYTYPE * [nv];
    bdyn         = new GINT     * [nv];
    if ( dual_coords2 == NULL ) return FALSE;
    if ( bnno         == NULL ) return FALSE;
    if ( bdy_ids      == NULL ) return FALSE;
    if ( bcs          == NULL ) return FALSE;
    if ( bdyn         == NULL ) return FALSE;
    for ( i=0; i<nv; i++ )
    {
      dual_coords2[i] = new Point3D [4];
      bnno        [i] = 0;
      bdy_ids     [i] = NULL;
      bcs         [i] = NULL;
      bdyn        [i] = NULL;
      if ( dual_coords2[i] == NULL ) return FALSE;
    }
  }
  else
  {
    nv = NE[0]*NE[1]*NE[2];
    mm = NN[0]*NN[1]*NN[3];
    dual_coords3 = new Point3D * [nv];
    bnno         = new GLONG      [nv];
    bdy_ids      = new GLONG    * [nv];
    lcorner_ids  = new GLONG      [ncorner_ids];
    gcorner_ids  = new GLONG      [ncorner_ids];
    ivert        = new GINT       [ncorner_ids];
    bcs          = new BDYTYPE * [nv];
    bdyn         = new GINT     * [nv];
    if ( dual_coords3 == NULL ) return FALSE;
    if ( bnno         == NULL ) return FALSE;
    if ( bdy_ids      == NULL ) return FALSE;
    if ( bcs          == NULL ) return FALSE;
    if ( bdyn         == NULL ) return FALSE;
    for ( i=0; i<nv; i++ )
    {
      dual_coords3[i] = new Point3D [8];
      bnno        [i] = 0;
      bdy_ids     [i] = NULL;
      bcs         [i] = NULL;
      bdyn        [i] = NULL;
      if ( dual_coords3[i] == NULL ) return FALSE;
    }
  }

  neighbors    = new GINT    * [nv];
  if ( neighbors == NULL ) return FALSE;

  nne  = new GINT    [nv];
  if ( nne == NULL ) return FALSE;

  nno  = new GLONG   [nv];
  if ( nno == NULL ) return FALSE;

  bnno  = new GLONG   [nv];
  if ( bnno == NULL ) return FALSE;

  node_ids     = new GLONG   * [nv];
  if ( node_ids == NULL ) return FALSE;

  bdy_node_indices = new GLONG    * [nv];
  if ( bdy_node_indices == NULL ) return FALSE;

  for ( i=0; i<nv; i++ )
  {
    node_ids        [i] = new GLONG   [mm];
    bcs             [i] = NULL;
    bdyn            [i] = NULL;
    bdy_node_indices[i] = NULL;
    if ( node_ids[i] == NULL ) return FALSE;
  }

  graph_coords = new GFLOAT * [nd];
  if ( graph_coords == NULL ) return FALSE;
  for ( i=0; i<nd; i++ )
  {
    graph_coords[i] = new GFLOAT [nv];
    if ( graph_coords[i] == NULL ) return FALSE;
  }

  pV = new Point3D [(GINT   )pow(2.0,nd)];

  return TRUE;

} // end of method CreateDynamic


//************************************************************************************
// METHOD     : DoPart
// DESCRIPTION: Carries out the partitioning in 2- or 3D
// ARGUMENTS  :
// RETURNS    : TRUE or FALSE on success or failure.
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::DoPart()
{
  DeleteDynamic();
  bPartitioned = FALSE;
  if ( !CreateDynamic() ) return FALSE;

  if      ( nd == 1 ) return DoPart1D();
  else if ( nd == 2 ) return DoPart2D();
  else if ( nd == 3 ) return DoPart3D();

  return FALSE;
} // end of method DoPart


//************************************************************************************
// METHOD     : DoPart1D
// DESCRIPTION: Carries out the partitioning in 1D
// ARGUMENTS  :
// RETURNS    : TRUE or FALSE on success or failure.
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::DoPart1D()
{
  GLONG     i, ii, is, m, n;
  GFLOAT    x0,  Lx;
        
  // At this point, everything has been 
  // fully allocated, except the neighbor lists, 
  // whose dimensions we don't yet know, and the
  // boundary node quantities...
  if ( neighbors    == NULL || nne          == NULL ||
       node_ids     == NULL || nno          == NULL ||
       graph_coords == NULL || dual_coords1 == NULL ) return FALSE;
  

  Lx = fabs( P1[0] - P0[0] )/NE[0];

  // Do coords
  for ( i=0,n=0; i<NE[0]; i++, n++ )
  {
    graph_coords[0][n] = P0[0] + (i + 0.5)*Lx; 
    x0 = P0[0] + i*Lx;
    
    dual_coords1[n][0] = x0;
    dual_coords1[n][1] = x0 + Lx;
    nne[n] = 0;
  }

  // Do node_ids. This is valid only for fixed
  // descretizations (those that don't change with
  // element).
  node_dyn_range = -1;
  for ( i=0,n=0; i<NE[0]; i++, n++ )
  {
    is = i*(NN[0] - 1 ) + 1;
    for ( ii=0,m=0; ii<NN[0]; ii++, m++ ) 
    {
      node_ids[n][m] = is + ii;
      node_dyn_range = MAX(node_dyn_range,node_ids[n][m]);
    }
    nno[n] = NN[0];
    nne[i] = 0;
  } 
  if ( bPeriodic[0] )
    node_ids[NE[0]-1][NN[0]-1] = node_ids[0][0];

  // For each graph vertex, find number of
  // neighbors, no. bdy nodes 
  for ( i=1,n=0; i<NE[0]-1; i++, n++ )  // interior
  {
    nne [n] = 2; 
    bnno[n] = 0;
  }
  if ( NE[0] > 1 )                      // boundaries
  {
     nne       [0] = 1;
     nne [NE[0]-1] = 1;
  }

  // Find the neighbor lists:
  for ( i=1; i<NE[0]+1; i++, n++ )
  {
    neighbors[n] = new GINT    [nne[n]]; 
    neighbors[n][0] = i-1;
    neighbors[n][1] = i+1;
  }
  neighbors      [0] = NULL;
  neighbors[NE[0]-1] = NULL;
  if ( NE[0] > 1 )
  {
    neighbors      [0]    = new GINT    [1]; 
    neighbors      [0][0] = 1;
    neighbors[NE[0]-1]    = new GINT    [1];
    neighbors[NE[0]-1][0] = NE[0]-2;
  }

  // For boundary elements, find indices to 
  // boundary nodes
  if ( NE[0] == 1 )
  {
    bnno[0] = 2;
    bcs [0] = new BDYTYPE [2];
    bdyn[0] = new GINT    [2];
    bcs[0][0] = NONE;
    bcs[0][1] = NONE;
    bdy_node_indices[0] = new GLONG   [2];
    bdy_node_indices[0][0] = 0;
    bdy_node_indices[0][1] = NN[0] - 1;
  }
  else 
  {
    bnno      [0] = 1;
    bnno[NE[0]-1] = 1;
    bcs         [0] = new BDYTYPE [1];
    bcs   [NE[0]-1] = new BDYTYPE [1];
    bcs      [0][0] = NONE;
    bcs[NE[0]-1][0] = NONE;
    bdy_node_indices       [0] = new GLONG   [1];
    bdy_node_indices [NE[0]-1] = new GLONG   [1];
    bdy_node_indices         [0] = 0;
    bdy_node_indices[NE[0]-1][0] = NN[0] - 1;
  }
  
  bPartitioned = TRUE;
  return TRUE;
 
} // end of method DoPart1D


//************************************************************************************
// METHOD     : DoPart2D
// DESCRIPTION: Carries out the partitioning in 2D, determining all 
//              relevant quantities
// ARGUMENTS  : 
// RETURNS    : TRUE or FALSE on success or failure.
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::DoPart2D()
{
  GLONG     i, j, ii, jj, is, js, k, m, n, n0, n1, nrow;
  GINT      **checker, ib[4], P[2];
  GFLOAT    x0, y0;
  GFLOAT    Lx, Ly;
        
  // At this point, everything has been 
  // fully allocated, except the neighbor lists, 
  // whose dimensions we don't yet know...
  if ( neighbors    == NULL || nne          == NULL ||
       node_ids     == NULL || nno          == NULL ||
       graph_coords == NULL || dual_coords2 == NULL ) return FALSE;
  
   for ( i=0; i<2; i++ ) P[i] = NN[i] - 1;

  // checker is a 'colored' graph of the domain, 
  // including boundaries. The interior represents the
  // valid nonzero elements, while the zero-elements
  // represent 'ghost zones' used for determining
  // neighboring elements easily on the domain
  // boundary.
  checker = new GINT    * [NE[1]+2];
  for ( j=0; j<NE[1]+2; j++ )
    checker[j]  = new GINT    [NE[0]+2];

  Lx = fabs( P1[0] - P0[0] ) / NE[0];
  Ly = fabs( P1[1] - P0[1] ) / NE[1];

  // Do coords: 2d, fill in valid elements of checker board:
  for ( j=0, n=0; j<NE[1]; j++ )
  {
    for ( i=0; i<NE[0]; i++, n++ )
    {
      graph_coords[0][n] = P0[0] + (i + 0.5)*Lx; 
      graph_coords[1][n] = P0[1] + (j + 0.5)*Ly;
      x0 = P0[0] + i*Lx;
      y0 = P0[1] + j*Ly;
      
      dual_coords2[n][0].x1 = x0;
      dual_coords2[n][0].x2 = y0;
      dual_coords2[n][1].x1 = x0 + Lx;
      dual_coords2[n][1].x2 = y0;
      dual_coords2[n][2].x1 = x0 + Lx;
      dual_coords2[n][2].x2 = y0 + Ly;
      dual_coords2[n][3].x1 = x0;
      dual_coords2[n][3].x2 = y0 + Ly;
      bnno        [n]       = 0;
      checker[j+1][i+1] = 1;
    }
  }

  // Fill ghost zones of checker board:
  for ( i=0; i<NE[0]+2; i++ )
  {
    checker      [0][i] = 0;
    checker[NE[1]+1][i] = 0;
  }
  for ( j=0; j<NE[1]+2; j++ )
  {
    checker[j]      [0] = 0;
    checker[j][NE[0]+1] = 0;
  }

  // Find node ids. This is valid only for fixed
  // descretizations (those that don't change with
  // element).
  node_dyn_range = -1;
  nrow = NE[0] * NN[0] - NE[0] + 1;
  for ( j=0, n=0; j<NE[1]; j++ )
  {
    js = j>0 ? j*(NN[1]-1) :  j*NN[1]; 
    for ( i=0; i<NE[0]; i++, n++ )
    {
      for ( jj=0, m=0; jj<NN[1]; jj++ ) 
      {
        is = (jj+js)*nrow + i*(NN[0] - 1 );
        for ( ii=0; ii<NN[0]; ii++, m++ ) 
        {
          node_ids[n][m] = is + ii;
          node_dyn_range = MAX(node_dyn_range,node_ids[n][m]);
        }
        nno[n] = m;
      }
    } 
  }

  // Check for periodicity condition; 
  // find id reassignment lists: 
  if ( bPeriodic[0] )  // X-periodic
  {
    for ( j=0; j<NE[1]; j++ )
    {
      n0 = j*NE[0]; 
      n1 = j*NE[0] + NE[0] - 1;
      for ( jj=0, m=0; jj<NN[1]; jj++ ) 
        node_ids[n1][(jj+1)*NN[0]-1] = node_ids[n0][jj*NN[0]];
    }
  }

  if ( bPeriodic[1] )  // Y-periodic
  {
    for ( i=0; i<NE[0]; i++ )
    {
      n0 =  i; 
      n1 = (NE[1]-1)*NE[0] + i;
      for ( ii=0; ii<NN[0]; ii++ ) 
        node_ids[n1][(NN[1]-1)*NN[0]+ii] = node_ids[n0][ii];
    }
  }

  // For each graph vertex, find number of
  // neighbors by adding all checker board
  // neighbor colors:
  for ( j=0, n=0; j<NE[1]; j++ )
  {
    for ( i=0; i<NE[0]; i++, n++ )
    {
      for ( jj=j-1, nne[n]=0; jj<=j+1; jj++ )
      {
        for ( ii=i-1; ii<=i+1; ii++ )
          if ( ii != i || jj != j ) nne[n] += checker[jj+1][ii+1];
      }
    }
  }

  // Find the neighbor lists:
  for ( j=0, n=0; j<NE[1]; j++ )
  {
    for ( i=0; i<NE[0]; i++, n++ )
    {
      neighbors[n] = new GINT    [nne[n]]; 
      for ( jj=j-1, m=0; jj<=j+1; jj++ )
      {
        for ( ii=i-1; ii<=i+1; ii++ )
        {
          if ( checker[jj+1][ii+1] && (ii!=i || jj!=j) )
          {
            neighbors[n][m] = ii + (jj)*NE[0];
            m++;
          }
        }
      }
    }
  }

  // Find boundary nodes: include edges, subtract common
  // corners (MESSY!!):
  for ( j=1, n=0; j<NE[1]+1; j++ )
  {
    for ( i=1; i<NE[0]+1; i++, n++ )
    {
      bdyn[n] = new GINT    [4];                  // for quads only
      memset(bdyn[n], 0, 4*sizeof(GINT   ));
      memset(ib     , 0, 4*sizeof(GINT   ));
      if ( checker[j-1][i  ] == 0 ) bdyn[n][0] = 1;
      if ( checker[j  ][i+1] == 0 ) bdyn[n][1] = 1;
      if ( checker[j+1][i  ] == 0 ) bdyn[n][2] = 1;
      if ( checker[j  ][i-1] == 0 ) bdyn[n][3] = 1;

      if ( bdyn[n][0]  ) bnno[n] += NN[0];      // bottom of elem
      if ( bdyn[n][1]  ) bnno[n] += NN[1];      // right of elem
      if ( bdyn[n][2]  ) bnno[n] += NN[0];      // top of elem
      if ( bdyn[n][3]  ) bnno[n] += NN[1];      // left of elem


      if ( bdyn[n][0] && bdyn[n][1] ) { bnno[n]--; ib[1] = 1; }   // bottom right corner in common
      if ( bdyn[n][1] && bdyn[n][2] ) { bnno[n]--; ib[2] = 1; }   // top right corner in common
      if ( bdyn[n][2] && bdyn[n][3] ) { bnno[n]--; ib[3] = 1; }   // top left corner in common
      if ( bdyn[n][3] && bdyn[n][0] ) { bnno[n]--; ib[0] = 1; }   // bottom left corner in common

      // global ids for global corners:
      if ( i==1     && j==1       ) {                // bottom left corner
        gcorner_ids[0] = 0                            ; ivert[0] = n;
      }
      if ( i==NE[0] && j==1       ) {                // bottom right corner
        gcorner_ids[1] = NE[0]*P[0]                   ; ivert[1] = n;
      }
      if ( i==NE[0] && j==NE[1]   ) {                // top right corner
        gcorner_ids[2] = (NE[0]*P[0]+1)*(NE[1]*P[1]+1) 
                                                  - 1; ivert[2] = n;
      }
      if ( i==1     && j==NE[1]   ) {                // top left corner
        gcorner_ids[3] = (NE[0]*P[0]+1)*(NE[1]*P[1]+1) 
                                      -NE[0]*P[0] - 1; ivert[3] = n;
      }

      if ( bnno[n] > 0 ) 
      {
         bdy_node_indices[n] = new GLONG   [bnno[n]];
         bdy_ids         [n] = new GLONG   [bnno[n]];
         bcs             [n] = new BDYTYPE[bnno[n]];
         for ( k=0; k<bnno[n]; k++ ) bcs[n][k] = NONE;
      }
      
      // Indices must follow manner in which they were
      // labelled above in computing node_ids:

      m = 0;
      if ( bdyn[n][0] ) { 
        for ( ii=0; ii<NN[0]      ; ii++,m++ ) { 
          bdy_node_indices[n][m] = ii;
          bdy_ids[n][m] = 0;
        }
      }
      if ( bdyn[n][1] ) {
        for ( jj=ib[1]; jj<NN[1]      ; jj++,m++ ) {
          bdy_node_indices[n][m] = NN[0]-1 + jj       *NN[0];
          bdy_ids[n][m] = 1;
        }
      }
      if ( bdyn[n][2] )  {
        for ( ii=0    ; ii<NN[0]-ib[2]; ii++,m++ ) {
          bdy_node_indices[n][m] = ii      + (NN[1]-1)*NN[0];
          bdy_ids[n][m] = 2;
        }
      }
      if ( bdyn[n][3] ) {
        for ( jj=ib[0]; jj<NN[1]-ib[3]; jj++,m++ ) {
          bdy_node_indices[n][m] =           jj       *NN[0];
          bdy_ids[n][m] = 3;
        }
      }
    }
  }

  // convert corner bdy global ids to (local) indices in 'bcs' table:
  for ( i=0; i<ncorner_ids; i++ )
  {
    n = ivert[i];
    for ( j=0; j<bnno[n]; j++ ) 
    {
      if ( node_ids[n][bdy_node_indices[n][j]] == gcorner_ids[i] ) 
      {
        lcorner_ids[i] = j;
  cout << "gcorner_ids[" << i << "]=" << gcorner_ids[i] << "; node_id =" << 
          node_ids[n][bdy_node_indices[n][j]] 
       << " lcorner_ids[" << i << "]=" <<  lcorner_ids[i] << " bdy_node_indices[" << n << "][" << j << "]=" << bdy_node_indices[n][j] << endl;
        break;
      }
    }
  }

  // Set domain vertices:
  pV[0].x1 = P0[0]; pV[0].x2 = P0[1]; pV[0].x3 = 0.0;
  pV[1].x1 = P1[0]; pV[1].x2 = P0[1]; pV[1].x3 = 0.0;
  pV[2].x1 = P1[0]; pV[2].x2 = P1[1]; pV[2].x3 = 0.0;
  pV[3].x1 = P0[0]; pV[3].x2 = P1[1]; pV[3].x3 = 0.0;


  for ( j=0; j<NE[1]; j++ )
    delete [] checker[j];
  delete [] checker;

  bPartitioned = TRUE;
  return TRUE;
 
} // end of method DoPart2D


//************************************************************************************
// METHOD     : DoPart3D
// DESCRIPTION: Carries out the partitioning in 3D
// ARGUMENTS  :
// RETURNS    : TRUE or FALSE on success or failure.
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::DoPart3D()
{
  cout << "DD_Rect::DoPart3D: 3D partition not yet available" << endl;
  return FALSE;
} // end of method DoPart3D


//************************************************************************************
// METHOD     : SetCorners
// DESCRIPTION: Sets bc for specified common node (only for 2d right now)
//                n=0 ==> bottom left corner
//                n=1 ==> bottom right corner
//                n=2 ==> top right corner
//                n=3 ==> top left corner
// RETURNS    : TRUE or FALSE on success or failure.
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::SetCorners(GINT n, BDYTYPE btype)
{
  if  ( n  < 0 || n >= ncorner_ids ) {
    cout << "DD_Rect::SetCorners: invalid node specification" << endl;
    return FALSE;
  }
  if ( !bPartitioned && !DoPart() ) return FALSE;
  
  bcs[ivert[n]][lcorner_ids[n]] = btype;

  return TRUE;
} // end of method SetCorners


//************************************************************************************
// METHOD     : bCornerNode
// DESCRIPTION: Checks if node is a corner node, and returns corresp. vertex id if so
//              node   : node id to check
//              ivertex: index of dual vertex of corner node if TRUE
// RETURNS    : TRUE (FALSE) if it is (is not) a corner node
//************************************************************************************
//************************************************************************************
GBOOL DD_Rect::bCornerNode(GINT node, GINT    &ivertex)
{
  GINT    i=0;

  ivertex = -1;
  while ( gcorner_ids[i] != node && i<ncorner_ids ) i++;
  if ( i>=ncorner_ids ) return FALSE;

  ivertex = ivert[i];
  return TRUE ;
} // end of method bCornerNode
