//======================================================================================
// Name         : gdd.cpp
// Date         : 9/06/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Driver for grid partitioning preprocessor. 
//                We will want to provide an interface directly
//                to this main, so that we can use it in
//                direct calls, if required.
// Modifications:
//======================================================================================
#include "gtypes.h"
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include "dd_rect.hpp"
#include "gdd.h"
#include "gdd_file.h"
#include "gexlib.hpp"


int main(int argc, char* argv[])
{
  GINT    i, j;

  // Parse command line. In the future, this may be 
  // replaced by or enhanced by using file input:
  if ( !Parse(argc, argv) )
  {
    cout << argv[0] << "::main: incorrect argument specification" << endl;
    exit(1);
  }
  if ( !bQuiet_ ) Echo(cout);

  // Create domain decomposition object. For now, the
  // only one available is for a rectangular
  // domain. In the future, this portion will be replaced
  // by a GUI which will provide the information for 
  // general 2- and 3-d domains:
  dd = new DD_Rect(P0_, P1_, nd_);

  dd->SetPartSize(nelem_[0],nelem_[1] ,nelem_[2]);
  dd->SetElemDescretization(nnodes_[0],nnodes_[1],nnodes_[2]);
  dd->DoPart();

  for ( i=0; i<nBdyCond_; i++ )
  {
    if ( !dd->SetBoundaryCond(B0_[i], B1_[i], nd_,  btype_[i]) )
    {
      cout << argv[0] << ":: main: boundary condition set failed" << endl;
      exit(1);
    }
  }
  for ( i=0; i<ncorner_set_; i++ ) 
  {
    dd->SetCorners(icorner_set_[i], corner_bc_[i]);
  }

  // Re-label nodes that correspond to PERODIC bcs:
#if 0
  if ( !dd->ResetPeriodic() ) 
  {
    cout << argv[0] << ":: node reset failed" << endl;
    exit(1);
  }
#endif

  // By now, all relevant data should be set...
  if ( !bPartLib_ ) // do a simple cyclic assignment of elements:
  {
    if ( !DoCyclicDist() )
    {
      cout << argv[0] << ":: main: cyclic distribution failed." << endl;
      exit(1);
    }
  }
  else if ( !DoSODist() )
  {
      cout << argv[0] << ":: main: distribution using dynamic lib. " << partlib_ << " failed." << endl;
      exit(1);
  }

  exit(0);
} // end of main


//************************************************************************************
// METHOD     : DoCyclicDist
// DESCRIPTION:
// ARGUMENTS  : none
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
//************************************************************************************
GBOOL DoCyclicDist()
{
  if ( filetype_ == 0 )
    return WriteByProc();
  else if ( filetype_ == 1 )
    return WriteByElem();

  return FALSE;

} // end of method DoCyclicDist


//************************************************************************************
// METHOD     : WriteByProc
// DESCRIPTION: 
// ARGUMENTS  : none
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
//************************************************************************************
GBOOL WriteByProc()
{
  GLONG    i, j, k, m;
  GINT     ne, K2P, nvert, ngvert, nface, nrem, nrange;
  GLONG    *nno, **nodeids, *bnno, **bindices;
  GINT     **bBdy;
  BDYTYPE  **bcs;
  Point    *pDomain;
  void     **vertices;
  ofstream os;

  if ( dd == NULL ) 
  {
    cout << "WriteByProc: NULL decomposition object" << endl;
    return FALSE;
  }

  os.open(fname_);  //,ios::app);

  K2P      = dd->NumVertices() / nproc_;
  nno      = dd->GetNodeNum();
  nodeids  = dd->GetNodeIDs();
  bnno     = dd->GetBdyNodeNum();
  bindices = dd->GetBdyNodeIndices();
  bcs      = dd->GetBdyCond();
  bBdy     = dd->GetbBdy();
  nrange   = dd->GetNodeDynRange();
  nrem     = dd->NumVertices() % nproc_;
  vertices = dd->GetDualVertexCoords();
  pDomain  = dd->GetDomainVertices();
  ngvert   = dd->GetNumDomainVertices();
  nvert    = (GINT )pow(2.0,nd_);
  nface    = 2*nd_;

  // Write header: 
  os << FT_BYPROC << " " << bDoNodeIDs_ << " " << nd_ << " " << nproc_ << " " << totalelem_ << " " << nrange << tokeol << endl ;
  os << nvert << toklist << endl ;
  for ( i=0; i<ngvert; i++ ) {
    for ( j=0; j<3; j++ ) os << pDomain[i][j] << " " ;
  }
  os << tokeol << endl ;
  for ( i=0; i<nproc_; i++ )
  {
    ne = K2P + ( i<nrem ? 1 : 0);
    os << ne << " " ;
  }
  os << tokeol << endl;

  // Write element info for each proc:
  for ( i=0; i<nproc_; i++ )
  {
    ne = K2P + ( i<nrem ? 1 : 0);
    os << i << toklist << endl << toklev1_2[0] << endl;
    for ( j=0; j<ne; j++ )
    {
      m = j * nproc_ + (K2P*nproc_ < dd->NumVertices() ? 0 : i);
      os << j << " " << RECT_QUAD << toklist << endl << toklev2_2[0] << endl;

      // Write element type,  vertices:
      if ( nd_ == 1 )
      {
        for ( k=0; k<nvert; k++ )
          os << *((GFLOAT*)vertices[m]+k) << "  ";
      }
      else if ( nd_ == 2 )
      {
        for ( k=0; k<nvert; k++ )
          os << ((Point3D*)vertices[m]+k)->x1 << " " <<  ((Point3D*)vertices[m]+k)->x2 << "  ";
      }
      else if ( nd_ == 3 )
      {
        for ( k=0; k<nvert; k++ )
          os << ((Point3D*)vertices[m]+k)->x1 << " " << ((Point3D*)vertices[m]+k)->x2
                                              << " " << ((Point3D*)vertices[m]+k)->x3 << "  " ;
      }
      os << tokeol  << endl;

      // ... next, the bdy indicators:
      for ( k=0; k<nface; k++ )
        os <<  bBdy[m][k] << " "; 
      os << tokeol << endl;

      // Write node information: exp. order, node ids, boundary types:
      //  ...exp. order in each direction:
      for ( k=0; k<nd_; k++ )
        os <<  nnodes_[k] << " "; 
      os << toklist << endl;
  
      if ( bDoNodeIDs_ )
      {
        // ... node ids:
        for ( k=0; k<nno[m]; k++ )
          os <<  nodeids[m][k] << " " ; 
        os << tokeol <<  endl;
      }
  
      // ... Boudary condition information
      // ... first, the indices of nodeids that are boundary nodes:
      os << bnno[m]  << toklist << endl;
      for ( k=0; k<bnno[m]; k++ )
        os <<  bindices[m][k] << " "; 
      os << tokeol << endl;
  
      // ... next the boundary conditions, of the boundary nodes:
      os << bnno[m]  << toklist << endl;
      for ( k=0; k<bnno[m]; k++ )
        os <<  bcs[m][k] << " "; 
      os << tokeol << endl;

      os << toklev2_2[1] << endl;
    } // end of element loop
    os << toklev1_2[1] << endl;
  }   // end of proc loop
  
  os.close();

  return TRUE;
} // end method WriteByProc


//************************************************************************************
// METHOD     : WriteByElem
// DESCRIPTION: 
// ARGUMENTS  : none
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
//************************************************************************************
GBOOL WriteByElem()
{
  GLONG    i, j, k, m, iproc;
  GINT     ne, K2P, nvert, ngvert, nface, nrem, nrange;
  GLONG    *nno, **nodeids, *bnno, **bindices;
  GINT     **bBdy;
  BDYTYPE  **bcs;
  Point    *pDomain=NULL;
  void     **vertices;
  ofstream os;

  if ( dd == NULL ) 
  {
    cout << "WriteByElem: NULL decomposition object" << endl;
    return FALSE;
  }

  os.open(fname_);  //,ios::app);

  K2P      = dd->NumVertices() / nproc_;
  nno      = dd->GetNodeNum();
  nodeids  = dd->GetNodeIDs();
  bnno     = dd->GetBdyNodeNum();
  bindices = dd->GetBdyNodeIndices();
  bcs      = dd->GetBdyCond();
  bBdy     = dd->GetbBdy();
  nrem     = dd->NumVertices() % nproc_;
  nrange   = dd->GetNodeDynRange();
  vertices = dd->GetDualVertexCoords();
  nvert    = (GINT ) pow(2.0,nd_);
  nface    = 2 * nd_;
  ngvert   = dd->GetNumDomainVertices();
  pDomain  = dd->GetDomainVertices();

  // Write header:
  os << FT_BYELEM << " " << bDoNodeIDs_ << " " << nd_ << " " << dd->NumVertices() << " " << totalelem_ << " " << nrange << tokeol << endl; 
  os << ngvert << toklist << endl ;
  for ( i=0; i<ngvert; i++ ) {
    for ( j=0; j<3; j++ ) os << pDomain[i][j] << " " ;
  }
  os << tokeol << endl ;
  for ( i=0; i<nproc_; i++ )
  {
    ne = K2P + ( i<nrem ? 1 : 0);
    os << ne << " " ;
  }
  os << tokeol << endl;

  // Element-by-element data:
  for ( i=0; i<dd->NumVertices(); i++ )
  {
      iproc = i % nproc_;
      os << i << " " << RECT_QUAD << " " << iproc << toklist << endl << toklev2_2[0] << endl;

      // Write vertices:
      if ( nd_ == 1 )
      {
        for ( k=0; k<nvert; k++ )
          os << *((GFLOAT*)vertices[i]+k) << "  ";
      }
      else if ( nd_ == 2 )
      {
        for ( k=0; k<nvert; k++ )
          os << ((Point3D*)vertices[i]+k)->x1 << " " <<  ((Point3D*)vertices[i]+k)->x2 << "  ";
      }
      else if ( nd_ == 3 )
      {
        for ( k=0; k<nvert; k++ )
          os << ((Point3D*)vertices[i]+k)->x1 << " " << ((Point3D*)vertices[i]+k)->x2
                                              << " " << ((Point3D*)vertices[i]+k)->x3 << "  " ;
      }
      os << tokeol << endl;

      // ... next, the bdy indicators:
      for ( k=0; k<nface; k++ )
        os <<  bBdy[m][k] << " "; 
      os << tokeol << endl;
    
      // Write node information: exp. order, node ids, boundary types:
      //  ...exp. order in each direction:
      for ( k=0; k<nd_; k++ )
        os <<  nnodes_[k] << " "; 
      os << toklist << endl;
  
      if ( bDoNodeIDs_ ) 
      {
        // ...node ids:
        for ( k=0; k<nno[i]; k++ )
          os <<  nodeids[i][k] << " "; 
        os << toklist << endl;
      }
  
      // ... Boudary condition information
      // ... first, the indices of nodeids that are boundary nodes:
      os << bnno[i]  << toklist << endl;
      for ( k=0; k<bnno[i]; k++ )
        os <<  bindices[i][k] << " ";
      os << tokeol << endl;

      // ... next the boundary conditions, of the boundary nodes:
      os << bnno[i]  << toklist << endl;
      for ( k=0; k<bnno[i]; k++ )
        os <<  bcs[i][k] << " ";
      os << tokeol << endl;

      os <<  toklev2_2[1] << endl;
  } // end of element loop
  
  os.close();

  return TRUE;
} // end method WriteByElem


//************************************************************************************
// METHOD     : DoSODist
// DESCRIPTION: 
// ARGUMENTS  : none
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
//************************************************************************************
GBOOL DoSODist()
{
  return FALSE;
} // end method DoSODist


//************************************************************************************
// METHOD     : Parse
// DESCRIPTION: Parses command line args.
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
//************************************************************************************
GBOOL Parse(GINT argc, char *argv[])
{
  GINT    i, j=0, itmp, n=0;
  GBOOL    isok=FALSE, bP0=FALSE, bP1=FALSE;

  // find dimensionality; first one found is 
  // selected:
  while ( j<argc && strcmp(argv[j],"-d")!=0 ) j++;

  if ( j+1 < argc ) 
  {
    if ( !isdigit(*argv[j+1]) ) return FALSE;  
    nd_ = atoi(argv[j+1]); 
    if ( nd_ < 1 || nd_ > 3 ) return FALSE;
  }
  else if ( j >= argc )
    nd_ = 2;
  else
    return FALSE;
 
  nBdyCond_ = 0; 
  j = 1;
  while ( j<argc )
  {
    if ( strcmp(argv[j],"-p")==0 && j+1 < argc  )
    { 
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       if      ( atoi(argv[j+1]) == 1 ) bPeriodic_[0] = TRUE;
       else if ( atoi(argv[j+1]) == 2 ) bPeriodic_[1] = TRUE;
       else if ( atoi(argv[j+1]) == 3 ) bPeriodic_[2] = TRUE;
       j += 2;
    }
    else if ( strcmp(argv[j],"-Nx")==0 && j+1 < argc  ) 
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nelem_[0] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-Ny")==0 && j+1 < argc  ) 
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nelem_[1] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-Nz")==0 && j+1 < argc  ) 
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nelem_[2] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-xNx")==0 && j+1 < argc  )
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nnodes_[0] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-xNy")==0 && j+1 < argc  )
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nnodes_[1] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-xNz")==0 && j+1 < argc  )
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nnodes_[2] = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-Np")==0 && j+1 < argc  ) 
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       nproc_ = atoi(argv[j+1]);;
       j += 2;
    }
    else if ( strcmp(argv[j],"-cb")==0 && j+2 < argc  ) 
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       if ( !isdigit(*argv[j+2]) ) return FALSE;
       icorner_set_[ncorner_set_] = atoi(argv[j+1]);
       corner_bc_  [ncorner_set_] = (BDYTYPE)atoi(argv[j+2]);
       ncorner_set_++;
       j += 3;
    }
    else if ( strcmp(argv[j],"-P0")==0 ) 
    {
       if ( j+nd_ >= argc ) return FALSE;
//     for ( i=1; i<=nd_; i++ ) isok = isok && isdigit(*argv[j+i]);
//     if ( !isok ) return FALSE;
       for ( i=1; i<=nd_; i++ ) P0_[i-1] = atof(argv[j+i]);
       bP0 = TRUE;
       j += (nd_+1);
    }
    else if ( strcmp(argv[j],"-P1")==0 && j+1 < argc  ) 
    {
       if ( j+nd_ >= argc ) return FALSE;
//     for ( i=1; i<=nd_; i++ ) isok = isok && isdigit(*argv[j+i]);
//     if ( !isok ) return FALSE;
       for ( i=1; i<=nd_; i++ ) P1_[i-1] = atof(argv[j+i]);
       bP1 = TRUE;
       j += (nd_+1);
    }
    else if ( strcmp(argv[j],"-b")==0 )
    {
       n = 2*nd_ - (nd_==1 ? 1 : 0);
       if ( j+n+1 >= argc ) return FALSE;
//     for ( i=1,isok=TRUE; i<=n+1; i++ ) {
//       cout << " i=" << i << " isok=" << isok << " argv=" << argv[j+i][0] << endl;
//       isok = isok && isdigit(argv[j+i][0]) ;
//     }
//     if ( !isok ) return FALSE;
       if ( nBdyCond_ < 2*nd_ )
       {
         for ( i=1; i<=nd_; i++ ) B0_[nBdyCond_][i-1] = atof(argv[j+i]);
         if ( nd_ > 1 ) 
         {
           for ( i=nd_+1; i<=2*nd_; i++ ) B1_[nBdyCond_][i-nd_-1] = atof(argv[j+i]);
         }
         btype_[nBdyCond_] = (BDYTYPE)atoi(argv[j+n+1]); 
         nBdyCond_++;
       }
       j += (n+2);
    }
    else if ( strcmp(argv[j],"-o")== 0  && j+1 < argc )
    {
      if ( !isalpha(*argv[j+1]) ) return FALSE;
      if      ( strcmp(argv[j+1],sfiletypes_[FT_BYPROC]) == 0 )
      {
         filetype_ = FT_BYPROC;
      }
      else if ( strcmp(argv[j+1],sfiletypes_[FT_BYELEM]) == 0 )
      {
         filetype_ = FT_BYELEM;
      }
      else
      {
        cout << "Parse: Bad file type specification." << endl;
        return FALSE;
      }
      j += 2;
    }
    else if ( strcmp(argv[j],"-l")== 0  && j+1 < argc )
    {
      strncpy(partlib_,argv[j+1],FILE_NAME_MAX);
      bPartLib_ = TRUE;
      j += 2;
    }
    else if ( strcmp(argv[j],"-f")== 0  && j+1 < argc )
    {
      strncpy(fname_,argv[j+1],FILE_NAME_MAX);
      j += 2;
    }
    else if ( strcmp(argv[j],"-quiet")==0  ) 
    {
      bQuiet_ = TRUE;
      j += 1;
    }
    else if ( strcmp(argv[j],"-d")==0  ) 
    {
       if ( !isdigit(*argv[j+1]) ) return FALSE;
       j += 2;
    }
    else if ( strcmp(argv[j],"-h")==0  ) 
    {
       PrtHelp(argc, argv, cout);
       exit(0);
    }
    else if ( strcmp(argv[j],"-nodes")==0  ) 
    {
       bDoNodeIDs_ = TRUE;
       j++;
    }
    else if ( strcmp(argv[j],"-nonodes")==0  ) 
    {
       bDoNodeIDs_ = FALSE;
       j++;
    }
    else 
    {
       cout << "Incorrect option syntax" << endl << endl;;
       PrtHelp(argc, argv, cout);
       exit(1);
    }
  }
  totalelem_ = 1;
  for ( i=0; i<nd_; i++ )  totalelem_ *= nelem_[i];
  if ( nproc_ > totalelem_ )
  {
    cout << argv[0] << "::Parse: Error: No. processors > No. elements!" << endl;
    return FALSE;
  }

  return TRUE;
} // end of function Parse

//************************************************************************************
// METHOD     : Echo
// DESCRIPTION: Prints argument summary
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
//************************************************************************************
void Echo(ostream &pout)
{
  GINT i;

  pout << endl << "          GASpAR Mesh Generation Utility v." << cversiono_ << endl << endl;
  pout << endl << "GMG Summary:" << endl << endl;
  pout <<  " Dimensionality      : " << nd_ << endl;
  pout <<  " No. Processors      : " << nproc_ << endl;
  pout <<  " No. Elements        : ("; 
       for (i=0; i<nd_; i++ ) {  pout << nelem_[i]; if ( i<nd_-1 ) pout <<  "," ; } 
                                 pout << ")" << endl;
  pout <<  " Exp. order          : (";
       for (i=0; i<nd_; i++ ) {  pout << nnodes_[i]; if ( i<nd_-1 ) pout << "," ; } 
                                 pout << ")" << endl;
  pout <<  " Boundary Periodicity: ("; 
       for (i=0; i<nd_; i++ ) {  pout << bPeriodic_[i]; if ( i<nd_-1 ) pout << "," ; } 
                                 pout << ")" << endl;
  pout <<  " Domain Extent       : ("; 
       for (i=0; i<nd_; i++ ) {  pout << P0_[i]; if ( i<nd_-1 ) pout << ",";  } 
                                 pout << "); (" ;
       for (i=0; i<nd_; i++ ) {  pout << P1_[i]; if ( i<nd_-1 ) pout << ","; } 
                                 pout << ")" << endl;
  if ( bPartLib_ ) 
  pout <<  " Partition library   : " << partlib_ << endl;
  pout <<  " Output file         : " << fname_ << endl;
  pout <<  " Output file type    : " << sfiletypes_[filetype_] << endl;
  pout <<  " Nodal output        : " << bDoNodeIDs_ << endl;
  pout << "***************************************************************************" << endl; 
  pout << "***************************************************************************" << endl << endl;

} // end of method Echo


//************************************************************************************
// METHOD     : PrtHelp
// DESCRIPTION: Prints help page to specified stream
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
//************************************************************************************
void PrtHelp(GINT argc, char **argv, ostream &pout)
{
       pout << "Enter:" <<endl;
       pout << argv[0] << " [-d dim] [{-Nx}{-Ny}{-Nz} #Elems] [{-xNx}{-xNy}{-xNz} Exp. Order] [-P0 x0 {y0} {z0}] [-P1 x1 {y1} {z1}] [-b Face# Type] [-Np #Procs] [-cb N Type] [-o Output_format] [-l filename] [-f filename] [-quiet] [-h]" << endl;
       pout << endl << " Where: " << endl;
       pout         << "      -d dim          : problem dimensionality (1, 2, or 3). Default is 2." << endl;
       pout         << "      -Nx #Elems      : Number of elements in x-direction. Default is 1." << endl;
       pout         << "      -Ny #Elems      : Number of elements in y-direction. Defaulst is 1." << endl;
       pout         << "      -Nz #Elems      : Number of elements in z-direction. Default is 1." << endl;
       pout         << "      -xNx Exp. Order : Expansion orderin x-direction. Default is 4." << endl;
       pout         << "      -xNy Exp. Order : Expansion orderin y-direction. Default is 4." << endl;
       pout         << "      -xNz Exp. Order : Expansion orderin z-direction. Default is 4." << endl;
       pout         << "      -P0 x0 {y0 {z0}}: Bottom left corner. There must be dim of these. Default is (0,0,0)." << endl;
       pout         << "      -P1 x0 {y0 {z0}}: Top right diagonal corner. There must be dim of these. Default is (1,1,1)" << endl;
       pout         << "      -b   Seg Type   : Set boundary condition Type on face (edge, point) given by Seg." << endl;
       pout         << "                        Seg is given in 1d by X0; in 2d by segment endpoints X0 Y0 X1 Y1;" << endl;
       pout         << "                        in 3d by diag corners X0 Y0 Z0 X1 Y1 Z1. Type is one of: 0 (Dirichlet)," << endl;
       pout         << "                        1 (Nuemann), 2 (Periodic), 3 (No-slip), 4 (None). Default is 4. " << endl;
       pout         << "      -Np  #Procs     : Number of processors. Default is 1." << endl;
       pout         << "      -cb  N Type     : Set common bdy node, N, to type, where Type is one of: 0 (Dirichlet)," << endl;
       pout         << "                        1 (Nuemann), 2 (Periodic), 3 (No-slip), 4 (None). No default." << endl;
       pout         << "                        Corners, N, are s.t. 0=>bottom left; 1=>bottom right; 2=>top right; 3=>top left in 2d." << endl;
 
       pout         << "      -o  Output form.: Output format: By_Proc==>sort by proc id; By_Elem==>sort by element id. Default is By_Proc." << endl;
       pout         << "      -l filename     : Partition library file name. Default is NULL." << endl;
       pout         << "      -f filename     : Output file name. Default is mesh.dat." << endl;
       pout         << "      -quiet          : run without echoing parameters. Default is full echo." << endl;
       pout         << "      -(no)nodes      : (do not) print nodal information to file. Default is -nodes." << endl;
       pout         << "      -h              : Print this help list." << endl;

       pout         << endl << endl;
       pout << "  Currently, only rectangular domains are supported. In the" << endl;
       pout << "future, a GUI will be added to decompose grids and provide" << endl;
       pout << "mesh information for general domains. " << endl;
       pout << "  Output data is placed in the 'filname' file, or in 'mesh.dat'" << endl;
       pout << "if no filename is provided. All data for constructing the finite" << endl;
       pout << "element-based mesh on a multi-processor system is supplied. By default, " << endl;
       pout << "a simple cyclic distribution of the elements among the processors is" << endl;
       pout << "performed. If a shared library (SO)is specified, then the processor" << endl;
       pout << "distribution is performed by that library. The SO must contain the" << endl;
       pout << "entry point 'GDDInterface', whose format is given in gdd.h" << endl;
} // end of method prtHelp


