//************************************************************************************//
// Module       : ab.cpp
// Date         : 11/8/04 (DLR)
// Copyright    : 2004-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                specifying and computing coefficients in an
//                Adams-Bashforth scheme with variable timestep.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "ab.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
AB::AB(GSHORT iorder)
:
iorder_               (iorder),
maxorder_             (3),
dthist_               (NULL)
{
  SetOrder(iorder);
} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Destructor
AB::~AB()
{
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
AB::AB(const AB &a)
{
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : SetOrder 
// DESCRIPTION: Sets AB order.
// ARGUMENTS  : iorder : order.
// RETURNS    : none.
//************************************************************************************
void AB::SetOrder(GSHORT iorder)
{
  if ( iorder < 1 || iorder > maxorder_ ) {
    cout << "AB::SetOrder: invalid specification";
    exit(1);
  }
  iorder_ = iorder;
  coeffs_.Resize(iorder);
  coeffs_ = 0.0;

} // end of method SetOrder
 

//************************************************************************************
//************************************************************************************
// METHOD     : SetTimestepHistory
// DESCRIPTION: Sets timestep history. This list must contain iorder timesteps at least,
//              with the ordering: dt[0] = dt^n, dt[1]=dt^n-1, dt[2]=dt^n-2...
// ARGUMENTS  : timestep buffer.
// RETURNS    : none
//************************************************************************************
void AB::SetTimestepHistory(GDBuffer *dthist)
{
  dthist_ = dthist;

} // end of method SetTimestepHistory

 
//************************************************************************************
//************************************************************************************
// METHOD     : GetTimestepHistory
// DESCRIPTION: Gets timestep history. This list must contain iorder timesteps at least,
//              with the ordering: dt[0] = dt^n, dt[1]=dt^n-1, dt[2]=dt^n-2...
// ARGUMENTS  : none.
// RETURNS    : GDBuffer * to dthist_ pointer.
//************************************************************************************
GDBuffer *AB::GetTimestepHistory()
{
  return dthist_;

} // end of method GetTimestepHistory

 
//************************************************************************************
//************************************************************************************
// METHOD     : GetCoeffs
// DESCRIPTION: Gets AB coefficients. There will be iorder coeffs 
//              with the time ordering:  n, n-1, n-2 ....
// ARGUMENTS  : none.
// RETURNS    : GDBuffer * to coeffs_ pointer.
//************************************************************************************
GDBuffer *AB::GetCoeffs()
{
  return &coeffs_;

} // end of method GetCoeffs

 
//************************************************************************************
//************************************************************************************
// METHOD     : ComputeCoeffs
// DESCRIPTION: Computes AB coefficients with variable timestep history.
//              NOTE: dthist_ pointer to timestep history buffer must be 
//                    set properly prior to entry, or 'exit' will be called.
//                    'exit' will be called if the timestep hist. buffer == NULL or
//                    if the number of buffer elements is too small as required
//                    by iorder_ variable.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void AB::ComputeCoeffs()
{
  GDOUBLE r1, r2, r3;

  if ( dthist_ == NULL ) {
    cout << "AB::ComputeCoeffs: NULL timestep history buffer" << endl;
    exit(1);
  }

  if ( dthist_->dim() < iorder_ ) {
    cout << "AB::ComputeCoeffs: insufficient timestep history" << endl;
    exit(1);
  }

  if      ( iorder_ == 1 ) {
    coeffs_[0] = 1.0;
  }
  else if ( iorder_ == 2 ) {
    r1         = (*dthist_)[0] / (*dthist_)[1];
    coeffs_[0] = 1.0 + 0.5*r1;
    coeffs_[1] = 0.5*r1;
  }
  else if ( iorder_ == 3 ) {
    r1         = (*dthist_)[0] / (*dthist_)[1];
    r2         = (*dthist_)[1] / (*dthist_)[2];
    r3         = (*dthist_)[0] / (*dthist_)[2];
    
    coeffs_[1] = r1* ( r3/3.0 + 0.5*r2 + 0.5);
    coeffs_[2] = -(0.5*r3 + coeffs_[1]*r2) / ( 1.0 + r2);
    coeffs_[0] = 1.0 - coeffs_[1] - coeffs_[2];
  }

} // end of method ComputeCoeffs

//************************************************************************************
//************************************************************************************
// METHOD     : ()
// DESCRIPTION: Gets AB coefficients, by index. There will be iorder coeffs 
//              with the time ordering:  n, n-1, n-2 ....
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE &AB::operator()(const GSHORT i)
{  
  return coeffs_(i);
} // end of operator () 

 
//************************************************************************************
//************************************************************************************
// METHOD     : []
// DESCRIPTION: Gets AB coefficients, by index. There will be iorder coeffs 
//              with the time ordering:  n, n-1, n-2 ....
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE &AB::operator[](const GSHORT i)
{  
  return coeffs_[i];
} // end of operator  []

 
