//************************************************************************************//
// Module       : burgers.cpp
// Date         : 5/4/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the solver for the multi-d Burgers equation. The problem solved is
//                du/dt + V.Del u   
//                - nu_x  Del_x^2 u - nu_y Del_y^2 u - nu_z Del_z^2 u  = f(x,y,z;t)
//
//                for specifiable components, u, V. The advection velocity, V, *is*
//                also solved for (is = u) by default, but can be held fixed depending on
//                how the object is constructed, in order to perform linear advection.
//                
// Derived From : none.
// Modifications: 7/21/03: converted to linked-list-based data structures
//************************************************************************************//
#include "burgers.hpp"
#include "pcpointjac_helm.hpp"
#include "pcblockjac_helm.hpp"
#include "pcblockjac_lap.hpp"
#include "mtk.hpp"
#include "gutils.hpp"


//************************************************************************************
//************************************************************************************
// METHOD     : Constructor Method (1)
// DESCRIPTION: This version specified the advection velocity, and how many
//              evolved components to expect. The evolved fields are set in a
//              call to ::SetFields.
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
BurgersSolver::BurgersSolver(GElemList *elems)
:
hDSOp            (NULL_HANDLE),
itime_type_      (TE_ABBDF),
upc_type_        (NULL),
bConstAdvVel_    (FALSE),
bPrecond_        (FALSE),
bDoAdvection_    (TRUE),
bInitReqd_       (TRUE),
bDoDiffusion_    (TRUE),
nfields_         (GDIM),
nadv_            (GDIM),
iadv_            (NULL),
nelems_          (0),
ntimelevels_     (3),
iorderam_        (3),
icycle_          (0),
maxCFL_          (0.5),
dtlast_          (0.0),
time_            (0.0),
nu_              (NULL),
Mcoeff_          (1.0),
mu_              (NULL),
gamma0_          (1.0),
u_list_          (NULL),
ubdyvals_        (NULL),
upc_             (NULL),
up_              (NULL),
up_rhs_          (NULL),
Nj_              (NULL),
gelems_          (elems),
advect_          (NULL),
filter_          (NULL),
gsop             (NULL)
{
  GINT  i;


  nu_       = new GDOUBLE         [GDIM];
  for ( i=0; i<GDIM; i++ ) nu_[i] = 1.0;
  if ( nadv_ > 0 ) {
    iadv_ = new GINT  [nadv_];
    for ( i=0; i<nadv_; i++ ) {
      iadv_[i] = i;
    }
  }
  for ( i=0; i<GDIM; i++ ) Lcoeff_[i] = 1.0;
  BuildCoeffs();
  ResetErrors();
  SetAdvOrder(2);
  SetAMOrder (2);
  SetBDFOrder(2);
  SetEvolType(itime_type_);


} // end of constructor (1)



//************************************************************************************
//************************************************************************************
// METHOD     : Constructor Method (2)
// DESCRIPTION: 
// ARGUMENTS  : elems  : element list
//              uf     : array of pointers to velocity component lists; at least
//                       one array value must be non-NULL.
//              nfields: number of array elements in uf.
//              cadv   : specified advection velocity components' vector lists. If
//                       NULL, or unspecified, then advection velocities are evolved;
//                       else, they are not, and the equation is pure advection-diffusion.
//                       If non-NULL, there *must* be nadv components specified.
//                       If nadv > GDIM, this is an error.
//              iadv   : array of indices, used if cadv==NULL. These arrays tell
//                       which components of the adv. velocity the fields, uf,
//                       represent. In most cases, they will simply be 0, 1, [2]
//                       in 2[3]d, but if we're advecting the 2-velocity in the 
//                       2-direction, for example, we would have uf[0] = u2, and
//                       iadv[0] = 1, with nadv = 1. All indices start at 0, which 
//                       is in contrast with how components are normally specified in
//                       GASpAR access methods.
//              nadv   : number of advection components. If > GDIM, this is an error.
// RETURNS    :

//************************************************************************************
BurgersSolver::BurgersSolver(GElemList *elems, GFieldList *uf[], GINT  nfields, 
                             GFieldList **cadv, GINT  *iadv, GINT  nadv)
:
hDSOp            (NULL_HANDLE),
itime_type_      (TE_ABBDF),
upc_type_        (NULL),
bConstAdvVel_    (FALSE),
bPrecond_        (FALSE),
bDoAdvection_    (TRUE),
bInitReqd_       (TRUE),
bDoDiffusion_    (TRUE),
nfields_         (nfields),
nadv_            (0),
iadv_            (NULL),
nelems_          (0),
ntimelevels_     (3),
iorderam_        (3),
icycle_          (0),
maxCFL_          (0.5),
dtlast_          (0.0),
time_            (0.0),
nu_              (NULL),
Mcoeff_          (1.0),
mu_              (NULL),
gamma0_          (1.0),
u_list_          (NULL),
ubdyvals_        (NULL),
upc_             (NULL),
up_              (NULL),
up_rhs_          (NULL),
Nj_              (NULL),
gelems_          (elems),
filter_          (NULL),
gsop             (NULL)
{
  GINT     i;
 

  nu_       = new GDOUBLE         [GDIM];
  for ( i=0; i<GDIM; i++ ) nu_[i] = 0.0;


  u_list_ = new GFieldList * [nfields_];
  for ( i=0; i<nfields_; i++ ) u_list_[i] = NULL;

  if ( !SetFields(uf, nfields, cadv, 3, iadv, nadv) ) {
    cout << "BurgersSolver::BurgersSolver(2): invalid number of evolved components" << endl;
    exit(1);
  }
  for ( i=0; i<GDIM; i++ ) Lcoeff_[i] = 1.0;

  ResetErrors();
  SetAdvOrder(2);
  SetAMOrder (2);
  SetBDFOrder(2);
  SetEvolType(itime_type_);
  if ( bConstAdvVel_ )
  advect_->SetAdvVel(cadv_, nadv_);

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
BurgersSolver::~BurgersSolver()
{
  DeleteDynamic    ();
}

//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: Delete quantities that depend on the number of components, n 
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void  BurgersSolver::DeleteDynamic()
{
  
  GINT  i;

  for ( i=0; i<nfields_; i++ ) {
    if ( cgu_     [i] ) delete cgu_     [i];
  }
  if ( nu_        ) delete [] nu_        ; 
  if ( upc_type_  ) delete [] upc_type_  ;
  if ( up_        ) delete [] up_        ; 
  if ( up_rhs_    ) delete [] up_rhs_    ; 
  if ( Nj_        ) delete [] Nj_        ; 
  if ( upc_       ) delete [] upc_       ; 
  if ( u_list_    ) delete [] u_list_ ;

  if ( ubdyvals_  ) delete [] ubdyvals_  ;
  if ( cgu_       ) delete [] cgu_       ; 
  if ( mu_        ) delete [] mu_        ;
  if ( iadv_      ) delete [] iadv_      ;
  if ( niter_     ) delete [] niter_     ;
  if ( ierror_    ) delete [] ierror_    ;
  if ( error_     ) delete [] error_     ;
  if ( min_error_ ) delete [] min_error_ ;
  if ( max_error_ ) delete [] max_error_ ;
  if ( advect_    ) delete    advect_    ;

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : CreateCompDynamic
// DESCRIPTION: Create quantities that depend on the number of components evolved, n .
//              Member data, nfields_, representing the class's number of
//              vector components evolved, *is* set here.
// ARGUMENTS  : n: new number of components evolved
// RETURNS    :
//************************************************************************************
GBOOL BurgersSolver::CreateCompDynamic(GINT  n)
{
  GINT  i;

  upc_type_ = new GPC           [n];
  up_       = new GVecList      [n];
  up_rhs_   = new GVecList      [n];
  Nj_       = new GVecList      [n];
  upc_      = new GLinOpList    [n];
  ubdyvals_ = new GVecList   *  [n];
  niter_    = new GINT          [n];
  ierror_   = new GINT          [n];
  error_    = new GDOUBLE       [n];
  min_error_= new GDOUBLE       [n];
  max_error_= new GDOUBLE       [n];
  cgu_      = new CG         *  [n];  
  if ( !( upc_type_ && up_    && upc_    && Nj_    && up_rhs_    &&
          ubdyvals_ && niter_ && ierror_ && error_ && min_error_ && max_error_  ) ) return FALSE;
  
  for ( i=0; i<n; i++ ) {
    upc_type_[i] = GPC_NONE;
    cgu_     [i] = new CG(CG_STANDARD, FALSE);
  }
  nfields_ = n;

  return TRUE;
 
} // end of method CreateCompDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : ResetExpandables
// DESCRIPTION: Set quantities that depend on the number and size of elements in 
//              field lists. The member data, nelems_, repres. the current
//              number of elements *is* set here.
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GBOOL BurgersSolver::ResetExpandables(GFieldList *uf[], GINT  nf)
{
  GINT     i, j, k, num, ncurr, NN;
  GVector *cu[3]={NULL,NULL,NULL};
  Elem2D   *e;

  if ( uf[0] == NULL ) return FALSE;

  num   = gelems_->size();
  vecop_.empty();
  H_    .empty();
  M_    .empty();
  Hdiag_.empty();
  ncurr = H_.size();
  for ( k=0; k<nfields_; k++ ) {
    up_     [k].empty();
    up_rhs_ [k].empty();
    Nj_     [k].empty();
    upc_    [k].empty();
  }
#if 0
  for ( j=0; j<nadv_; j++ ) {
    uadv_[iadv_[j]].empty();
  }
#endif
#if 0
  // Resize the member data for each existing element:
  for ( i=0; i<num; i++ ) {
    for ( j=0; j<3; j++ ) 
    cu[j] = uf[j] && uf[j]->member(i) ? uf[j]->member(i)->GetExpCoeffs(0) : NULL;
    NN    = uf[j]->member(i)->GetExpCoeffs(0)->dim();

    // Operators:
    vecop_[i]->SetElem(uf[j]->member(i)->GetElement());
    vecop_[i]->SetVec(cu[0], cu[1], cu[2]);
    ((HelmholtzOp*)H_    [i])->SetElem(uf[0]->member(i)->GetElement());

    // Member data:
    Hdiag_[i]->Resize(NN);
  }
#endif

  // Add elements as required:
  for ( i=ncurr; i<num; i++ ) {
//  n = uf[0]->member(i)->GetExpCoeffs(0)->dim();
    for ( j=0; j<nfields_; j++ ) 
    cu[j] = uf[j] && uf[j]->member(i) ? uf[j]->member(i)->GetExpCoeffs(0) : NULL;
    NN    = uf[0]->member(i)->GetExpCoeffs(0)->dim();
    e     = uf[0]->member(i)->GetElement();

    // Operators:
    vecop_ .add(NULL,TRUE); vecop_[i] = new VectorOp(e, cu[0], cu[1], cu[2]);
    H_     .add(NULL,TRUE); H_    [i] = new HelmholtzOp(e) ;
#if 0
    if ( itime_type_ == TE_LEAPFROG || itime_type_ == TE_RKK ) { 
      M_     .add(NULL,TRUE); M_    [i] = new MassOp     (e) ;
    } 
#endif

    // Pointers
    for ( k=0; k<nfields_; k++ ) {
      up_     [k].add(NULL,FALSE);
      up_rhs_ [k].add(NULL,FALSE);
      Nj_     [k].add(NULL,FALSE);
      upc_    [k].add(NULL, TRUE); // TRUE means list deletes it (this will be created later)
    }

    // Fully-instantiated data:
    Hdiag_.add(NULL,FALSE); // Hdiag_[i] = new GVector(NN);
#if 0
    for ( j=0; j<nadv_; j++ ) {
      uadv_[iadv_[j]].add(NULL,FALSE); 
    }
#endif
  }

  // Delete unnecessary elements from lists:
  for ( i=num; i<ncurr; i++ ) {
    vecop_ .del(i);
    H_     .del(i);
    M_     .del(i);
    Hdiag_ .del(i);
    for ( k=0; k<nfields_; k++ ) {
      up_     [k].del(i);
      up_rhs_ [k].del(i);
      Nj_     [k].del(i);
      upc_    [k].del(i);
    }
#if 0
    for ( j=0; j<nadv_; j++ ) {
      uadv_[iadv_[j]].del(i); 
    }
#endif
  }

  // Must now renumber/re-order the reference ids for lists:
  vecop_ .renumber();
  H_     .renumber();
  M_     .renumber();
  Hdiag_ .renumber();
  for ( k=0; k<nfields_; k++ ) {
    up_     [k].renumber();
    up_rhs_ [k].renumber();
    Nj_     [k].renumber();
    upc_    [k].renumber();
  }
#if 0
  for ( j=0; j<nadv_; j++ ) {
    uadv_[iadv_[j]].renumber(); 
  }
#endif

  nelems_ = num;
  dtlast_ = 0.0;

  if  ( !InitPrecond() ) {
    cout << "BurgersSolver::ResetExpandables: InitPrecond failed" << endl;
    exit(1);
  }
  bInitReqd_ = TRUE;
  return TRUE;

} // end of method ResetExpandables


//************************************************************************************
//************************************************************************************
// METHOD     : SetElemList
// DESCRIPTION: sets element list
//
// ARGUMENTS  : GElemList *
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetElemList(GElemList *elems)
{
  if ( elems == NULL ) {
    cout << "BurgersSolver::SetElemList: NULL element list" << endl;
    exit(1);
  }
  gelems_ = elems;
  bInitReqd_ = TRUE;
} // end of method SetElemList


//************************************************************************************
//************************************************************************************
// METHOD     : SetTime
// DESCRIPTION: sets time stamp (time at start of integration)
//
// ARGUMENTS  : GDOUBLE time
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetTime(GDOUBLE time)
{
  time_ = time;
} // end of method SetTime


//************************************************************************************
//************************************************************************************
// METHOD     : SetTimestepHistory
// DESCRIPTION: sets timestep history buffer
//
// ARGUMENTS  : GDBuffer *
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetTimestepHistory(GDBuffer *dthist)
{ 
  advect_->SetTimestepHistory(dthist);
} // end of method SetTimestepHistory


//************************************************************************************
//************************************************************************************
// METHOD     : SetFilter
// DESCRIPTION: sets filter
//
// ARGUMENTS  : GLinOpList *
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetFilter(GLinOpList *filter)
{
  filter_ = filter;
} // end of method SetFilter


//************************************************************************************
//************************************************************************************
// METHOD     : SetEvolType
// DESCRIPTION: Sets time evolution method type by setting appropriate 
//              coefficient. BDF, Ext AB orders must be set prior to entry.
//
//              Caller must set fields with appropriate
//              number of time levels set.
// ARGUMENTS  : 
// RETURNS    :  
//************************************************************************************
void  BurgersSolver::SetEvolType(TIME_EVOLTYPE i)
{
  char  *serr = "BurgersSolver::SetEvolType: ";
  GINT  j;

  itime_type_ = i;

  switch (itime_type_) {
    case TE_OIFS:
      advect_->SetEvolType(GAdvect::TE_OIFS);
    case TE_ABBDF: 
      advect_->SetEvolType(GAdvect::TE_ABBDF);
    case TE_EXBDF:
      advect_->SetEvolType(GAdvect::TE_EXBDF);
      break;
  case TE_EXBDF_Weak:
      advect_->SetEvolType(GAdvect::TE_EXBDF_Weak);
      break;
    default:
      cout << serr << "invalid NavierStokes time stepping method specified" << endl;
      exit(1);
  }
  ResetExpandables(u_list_,nfields_);

  ntimelevels_ = iorderbdf_;
  if ( i == TE_ABBDF || i == TE_EXBDF || i == TE_OIFS  || i == TE_EXBDF_Weak ) {
    ntimelevels_ = MAX(iorderadv_, iorderbdf_);
    gamma0_ = advect_->GetBDFSum();
  }
#if 0
  else if ( i == TE_AMFE ) {
    if ( alpha_ != NULL ) delete [] alpha_; alpha_  = NULL;
    if ( beta_  != NULL ) delete [] beta_ ; beta_   = NULL;
    if ( mu_    != NULL ) delete [] mu_   ; mu_     = NULL;
    alpha_ = new GDOUBLE [iorderadv_];
    ntimelevels_ = MAX(iorderadv_, MAX(iorderbdf_,iorderam_));
    memcpy(alpha_, c_ab [iorderab_ -1], iorderab_ *sizeof(GDOUBLE));
    iorderadv__ = iorderab_;
    beta_ = new GDOUBLE [iorderbdf_];
    memcpy(beta_,c_bdf[iorderbdf_-1],iorderbdf_*sizeof(GDOUBLE));
    for ( j=0,gamma0_=0.0; j<iorderbdf_; j++ ) {
      gamma0_ += beta_[j];
    }
    mu_ = new GDOUBLE [iorderam_];
    memcpy(mu_,c_am[iorderam_-1],iorderam_*sizeof(GDOUBLE));
  }
#endif

} // end of method SetEvolType


//************************************************************************************
//************************************************************************************
// METHOD     : SetFields
// DESCRIPTION: Sets fields to be evolved
// ARGUMENTS  : 
// RETURNS    :  
//************************************************************************************
GBOOL BurgersSolver::SetFields(GFieldList *uf[], GINT  nfields, GFieldList **cadv, GINT  ncadv, 
                               GINT  *iadv, GINT  nadv)
{
  GINT  i, j, k, nf;

  if ( u_list_ != NULL ) delete [] u_list_ ;
  u_list_ = new GFieldList * [nfields];
  for ( k=0; k<nfields; k++ ) u_list_[k] = NULL; 
  for ( k=0, nf=0; k<nfields; k++ ) {
    if ( uf[k] != NULL && uf[k]->size() > 0 ) u_list_[nf++] = uf[k];
  }
  nfields_ = nf;
  
  if ( nfields_ <= 0 ) {
    cout << "BurgersSolver::SetFields: NULL input field" << endl;
    return FALSE;
  }

  nf = u_list_[0]->size();
  for ( k=1; k<nfields_; k++ ) {
    if ( u_list_[k]->size() != nf ) {
      cout << "BurgersSolver::SetFields: incompatible input fields" << endl;
      return FALSE;
    }
  }

  // If there is a constant advect velocity, store it:
  bConstAdvVel_ = cadv != NULL ;
  if ( iadv_ != NULL ) delete [] iadv_;
  if ( iadv  != NULL && nadv > 0 ) {
    nadv_ = nadv;
    if ( nadv_ <=0 || nadv_ > GDIM ) {
      cout << "BurgersSolver::Burgers (2): invalid number of advection velocity components" << endl;
      exit(1);
    }
    iadv_ = new GINT  [nadv_];
    for ( k=0; k<nadv_; k++ ) iadv_[k] = -1;
    for ( k=0; k<nadv_; k++ ) iadv_[k] = iadv[k];
  }
  else {
    if ( bConstAdvVel_ ) {
      for ( k=0, nadv_=0; k<ncadv; k++ ) {
        if ( cadv[k] != NULL && cadv[k]->size() > 0 ) nadv_++;
      }
      if ( nadv_ <=0 || nadv_ > GDIM ) {
        cout << "BurgersSolver::Burgers (2): invalid number of advection velocity components" << endl;
        exit(1);
      }
      iadv_ = new GINT  [nadv_];
      for ( k=0; k<nadv_; k++ ) iadv_[k] = -1;
      for ( k=0, j=0; k<nadv_; k++ ) {
        if ( cadv[k] != NULL  && cadv[k]->size() > 0 ) iadv_[j++] = k;
      }
      for ( k=0; k<nadv_; k++ ) {
        cadv_[iadv_[k]] = cadv[iadv_[k]];
      }
    }
    else {
      nadv_ = nfields_;
      iadv_ = new GINT  [nadv_];
      for ( k=0; k<nadv_; k++ ) iadv_[k] = -1;
      for ( k=0, j=0; k<nadv_; k++ ) {
        if ( u_list_[k] != NULL && u_list_[k]->size() > 0 ) iadv_[j++] = k;
      }
    }
  }
  if ( bConstAdvVel_ ) {
    for ( k=0; k<nadv_; k++ ) {
      cadv_[k] = cadv[k];
    }
  }
  for ( k=0; k<nadv_; k++ ) {
    if ( iadv_[k] < 0 || iadv_[k] > GDIM ) {
      cout << "BurgersSolver::SetFields: invalid advection velocity component specification" << endl;
      exit(1);
    }
  }

  if ( !CreateCompDynamic(nfields_) ) {
    cout << "BurgersSolver::SetFields: unable to create dynamic quantities" << endl;
    exit(1);
  }

  if ( ElemListChange() ) {
    if ( !ResetExpandables(u_list_,nfields_) ) {
      cout << "BurgersSolver::SetFields: allocation error" << endl;
      return FALSE;
    }
  }
  // Set the fields in CG so that the bdy's have meaning:
  for ( k=0; k<nfields_; k++ ) {
    for ( i=0; i<u_list_[k]->size(); i++ ) {
      up_[k][i]  = u_list_[k]->member()->GetExpCoeffs(0);    // where to store solution
    }
//  if ( itime_type_ == TE_LEAPFROG ) 
//    (*cgu_[k])(gelems_, &M_, NULL, &up_[k], NULL, NULL);
//  else
    (*cgu_[k])(gelems_, &H_, NULL, &up_[k], NULL, NULL);
  }

#if 0
  // Set the advection velocity to constant value, if required:
  // NOTE: the advection velocity pointers must not change from those
  //       given in the constructor!!
  if ( bConstAdvVel_ ) {
    for ( i=0; i<nadv_; i++ ) {
      for ( j=0; j<cadv_[i]->size(); j++ ) {
        uadv_[i][j] = (*cadv_[i])[j]; 
      }
    }
  }
#endif
  advect_ = new GAdvect(u_list_, nfields_, gelems_);
  bInitReqd_ = TRUE;
  return TRUE;

} // end of method SetFields


//************************************************************************************
//************************************************************************************
// METHOD     : SetComm
// DESCRIPTION: Sets global gather scatter operator
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
void  BurgersSolver::SetComm(GS *c)
{
  GINT  k;

  gsop = c;
  if ( gsop != NULL ) {
    for ( k=0; k<nfields_; k++ ) {
      if ( cgu_[k] == NULL ) {
         cout << "BurgersSolver::SetComm: NULL solver, component " << k+1 << endl;
         exit(1);
      }
      cgu_[k]->SetComm(gsop);
    }
  }

} // end of method SetComm


//************************************************************************************
//************************************************************************************
// METHOD     : SetCommHandle
// DESCRIPTION: Sets handle to active gather/scatter Init call 
// ARGUMENTS  : hIn :  existing handle
// RETURNS    : GCHandle
//************************************************************************************
GCHandle BurgersSolver::SetCommHandle(GCHandle hIn)
{
  GINT  k;

  if ( !gsop ) return NULL_HANDLE;

  if ( hDSOp != NULL_HANDLE ) return hDSOp;

  hDSOp = hIn;
  for ( k=0; k<nfields_; k++ ) {
    if ( cgu_[k] == NULL ) {
       cout << "BurgersSolver::SetCommHandle: NULL solver, component " << k+1 << endl;
       exit(1);
    }
    cgu_[k]->SetCommHandle(hDSOp);
  }

  return hDSOp;

} // end of method SetCommHandle


//************************************************************************************
//************************************************************************************
// METHOD     : SetVisc
// DESCRIPTION: Sets nu_
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void BurgersSolver::SetVisc(const GDOUBLE f, const GINT  idir)
{
  GINT  i;

  if ( idir < 0 || idir > GDIM ) {
    cout << "BurgersSolver::SetVisc: invalid coordinate direction" << endl;
    exit(1);
  }

  if ( idir == 0 ) {
    for ( i=0; i<GDIM; i++ ) nu_[i] = f;
    return;
  }
  nu_[idir-1] = f;
  for ( i=0,bDoDiffusion_=FALSE; i<GDIM; i++ ) bDoDiffusion_ = bDoDiffusion_ || nu_[i] != 0.0;
} // end of method SetVisc


//************************************************************************************
//************************************************************************************
// METHOD     : SetAdvOrder
// DESCRIPTION: Sets order of advection term approximation on the n+1 time level.
//              If this method is called, then SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here and in SetBDFOrder, or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3, ...
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetAdvOrder(GINT  iorder)
{

  iorderadv_   = iorder;
  advect_->SetAdvOrder(iorder);

} // end of method SetAdvOrder


//************************************************************************************
//************************************************************************************
// METHOD     : SetAMOrder
// DESCRIPTION: Sets order of Adams-Moulton method used to approximate the 
//              treatment of the diffusion operator. 
//
//              If this method is called, then ::SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here and in SetBDFOrder, or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3, or 4
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetAMOrder(GINT  iorder)
{

  if ( iorder < 0 || iorder > 4 ) {
    cout << "BurgersSolver::SetAMOrder: invalid approximation order: " << iorder << endl;
    exit(1);
  }
  iorderam_   = iorder;

} // end of method SetAMOrder


//************************************************************************************
//************************************************************************************
// METHOD     : SetBDFOrder
// DESCRIPTION: Sets order of BDF (backward time diff.) used to approximate the
//              explicit treatment of the time derivative.
//
//              If this method is called, then ::SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here, and in SetAdvOrder or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3, or 4
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetBDFOrder(GINT  iorder)
{
  iorderbdf_ = iorder;
  advect_->SetBDFOrder(iorder);

} // end of method SetBDFOrder


//************************************************************************************
//************************************************************************************
// METHOD     : Step
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    :
//************************************************************************************
GBOOL BurgersSolver::Step(GDOUBLE dt)
{
  
  char  *serr = "BurgersSolver::Step ";
  GBOOL bRet=FALSE;

  // If element list has changed, must resize member data:
  if ( ElemListChange() || bInitReqd_ ) {
    if ( !ResetExpandables(u_list_, nfields_) ) {
      cout << serr << "allocation error" << endl;
      exit(1);
    }
    UpdateSolver();
    bInitReqd_ = FALSE;
  }


  // Choose which time stepping scheme to use, and take a step:
  switch (itime_type_) {
    case TE_OIFS:
    case TE_ABBDF:    
    case TE_EXBDF: 
      bRet = DoNormalStep(dt);
      break;
    case TE_EXBDF_Weak: 
      bRet = DoNormalStep(dt);
      break;
#if 0
    case TE_AMFE:
      bRet = StepABBDFAM(dt);
      break;
    case TE_RKK: 
      bRet = StepTimeIndepRKK(dt);
      break;
    case TE_LEAPFROG: 
      bRet = StepLeapfrog(dt);
      break;
#endif
    default:
      cout << serr << "invalid time stepping method specified" << endl;
      bRet = FALSE;
      break;
  }

  if ( bRet ) {
    time_ += dt;
    icycle_ ++;
    DoFilter();
  }

  return bRet;

} //end of operator Step


//************************************************************************************
//************************************************************************************
// METHOD     : Advect
// DESCRIPTION: Performs advection partial update using explicit Adams-Bashforth 
//              or extrapolation time splitting of advection operators. Also, 
//             the explicit portion
//              of the BDF time derivative is added to the advection term on the
//              RHS, so that the system is ready for an update of the implicit
//              terms.  The result is placed in the iLevelTemp. Fields must have 
//              appropriate number of time levels set on instantiation. 
//              NOTE: Typically, the result of this method (in iLevelTemp field) is to
//              serve as a RHS for the update of the implicit Burgers update. Hence,
//              this RHS is already multiplied by the mass matrix as required for 
//              the weak form. 
//              
//              The partial update, f^n,  involved is:
//
//              f_i^n = Sum_m=(1,l) a_m v_i^(n-m+1) - Sum_j=(1,k) b_j N_i(v_i^(n-j+1))  
//
//              where Sum_m represents the explicit portion of the BDF contribution
//              to the time derivative, and Sum_j=(1,k) represents the explicit
//              representation of the Adams-Bashforth approximation to the 
//              advection term. The orders l, and k are the orders of the
//              BDF and AB expansions, respectively. The nonlinear terms, 
//
//              N_i(v_i) = M (v_1 D_1 + v_2 D_2 + v_3 D_3), 
//
//              where M is the mass matrix and v_i are the expansion coeffs for component
//              i in natural ordering, and the D_1 are the tensor product forms 
//              for the derivatives (D_1 = 1 X 1 X D; D_2= 2 X D X 1...). These local
//              operations must then be made global by performing a direct
//              stiffness summation operation. 
//
//              NOTE: InitComm must have been called prior to entry, in order to
//                    perform DSS on the advective quantities. 
//
//                    Also, this method requires that the field elements within the 
//                    field_lists contain one temporary storage level in addition to the 
//                    iLevelTemp level on instantiation.
//
//              NOTE: Timestep history buffer must be set prior to entry. This is
//                    set in call to SetTimestepHistory, and it points
//                    to a buffer managed outside this class. The history must
//                    contain timesteps for each time level, n, n-1, n-2, n-iorderab(ext)
//                    where iorderab(ext) are the AB (EXT) method orders. The timestep
//                    history buffer must be ordered s.t. dthist[0] = dt^n, dthist[1]=dt^n-1,...
//              
// ARGUMENTS  : iLevelTemp : temp place to put result within the Field temp manager.
//              dt         : time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::Advect(GDOUBLE dt, GINT  &iLevelTemp)
{
  char    *serr = "BurgersSolver::Advect: ";
  GINT     i, j, k;
  GBOOL    bRet;
  GDOUBLE  dti;
  
  iLevelTemp = u_list_[0]->member(0)->GetTempMgr()->GetFirstAvail();  // start at the lowest
  dti        = dt == 0.0 ? 0.0 : 1.0/dt;

  // Need the following for the Helmholtz solve:
  Mcoeff_    = gamma0_*dti;
  for ( j=0; j<GDIM; j++ ) Lcoeff_[j] = nu_[j];

  // Do non-linear advection term:
  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      up_ [k][i] = u_list_[k]->member()->GetTemp(iLevelTemp);    // temp space to hold intermediate sums
      Nj_ [k][i] = u_list_[k]->member()->GetTemp();              // temp space to hold advect update
     *Nj_ [k][i] = 0.0;
//cout << serr << "u[k=" << k << "][i=" << i << "]=" << *((*u_list_[k])[i]->GetExpCoeffs(0)) << endl;
//cout << serr << "u[k=" << k << "][i=" << i << "]=" << *((*u_list_[k])[i]->GetExpCoeffs(1)) << endl;
      u_list_[k]->next();
    }
  }

  // Use GAdvect operator: first arg is a pointer to the
  // explicit portion of the terms -M(dudt +u.Grad u) : the
  // '-' sign indicates that this term is on the RHS of
  // an invertible equation.
  bRet = advect_->Advect(up_, Nj_, time_, dt);
   

#if 0
    // Must now do a direct stiffness summation (global gather/scatter) on 
    // advection partial update:
    ((NTreeAdapt*)gsop)->DoProjection(TRUE);
    for ( k=0; k<nfields_; k++ ) {
      if ( !gsop->DSOp(Nj_[k], G_OP_SUM, hDSOp) ) {
        cout << serr << "Unable to perform DSS on advection operator" << endl;
        return FALSE;
      }
    }
    ((NTreeAdapt*)gsop)->DoProjection(FALSE);
#endif

  // Unlock temp memory so that we can re-use it later:
  for ( k=0; k<nfields_; k++ ) {
    for ( i=0; i<nelems_; i++ ) {
      u_list_[k]->member(i)->TempUnlock(up_[k][i]);
      u_list_[k]->member(i)->TempUnlock(Nj_[k][i]);
    }
  }

  return TRUE;

} // end of operator Advect


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : AdvectABFE
// DESCRIPTION: Performs advection partial update using explicit Adams-Bashforth
//              treatment for the advective term, and an Adams-Moulton treatment
//              for the diffusive term. The explicit portions of the diffusion
//              approximation are added to the advection approximation to provide 
//              the RHS for the implicit solve. The time derivative is approximated
//              using BDF (typically 1-order, which is foward-Euler (hence, FE)). 
//              The result is placed in the iLevelTemp. Fields must have 
//              appropriate number of time levels set on instantiation. 
//              NOTE: Typically, the result of this method (in iLevelTemp field) is to
//              serve as a RHS for the update of the implicit Burgers update. Hence,
//              this RHS is already multiplied by the mass matrix as required for 
//              the weak form. 
//              
//              The partial update, f^n,  involved is:
//
//              f_i^n = Sum_m=(1,Nb) a_m v_i^(n-m+1) - Sum_j=(1,Na) b_j N_i(v_i^(n-j+1))  
//                    - nu * Sum_k=(1,Nm) g_m L u^(n-m+1) 
//
//              where Sum_m represents the explicit portion of the BDF contribution
//              to the time derivative, and Sum_j=(1,k) represents the explicit
//              representation of the Adams-Bashforth approximation to the 
//              advection term, and Sum_k represents the explicit portion
//              for the diffusion approximation. The orders Na, and Nb, and Nm
//              are the orders of the BDF and AB and AM expansions, respectively.
//              The nonlinear terms, 
//
//              N_i(v_i) = M (v_1 D_1 + v_2 D_2 + v_3 D_3), 
//
//              where M is the mass matrix and v_i are the expansion coeffs for component
//              i in natural ordering, and the D_1 are the tensor product forms 
//              for the derivatives (D_1 = 1 X 1 X D; D_2= 2 X D X 1...). These local
//              operations must then be made global by performing a direct
//              stiffness summation operation. 
//
//              NOTE: InitComm must have been called prior to entry, in order to
//                    perform DSS on the advective quantities. 
//
//                    Also, this method requires that the field elements within the 
//                    field_lists contain one temporary storage level in addition to the 
//                    iLevelTemp level on instantiation.
//              
// ARGUMENTS  : iLevelTemp : place to put result within the Field.
//              dt         : time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::AdvectAMFE(GDOUBLE dt, GINT  &iLevelTemp)
{
  GINT     i, j, k;
  GDOUBLE  dti;
  GVector  *u;
  GVector  *tmp ;
  Field2D  *field;

  dti = dt == 0.0 ? 0.0 : 1.0/dt;

  if ( !AdvectABBDF(dt, iLevelTemp) ) {
    cout << "BurgersSolver::AdvectAMFE: error in approximating time derivative and advective term" << endl;
    return FALSE;
  }

  // Overwrite the coeffs in the implicit portion (linear operator to be inverted):
  Mcoeff_ = gamma0_*dti;
  for ( j=0; j<GDIM; j++ ) Lcoeff_[j] = mu_[0]*nu_[j];

  for ( i=0; i<nelems_; i++ ) {
    ((HelmholtzOp*)H_ [i])->SetConst(Lcoeff_[0], Lcoeff_[1], 0.0);
  } 
  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      field       = u_list_[k]->member();
      up_[k][i]  = field->GetTemp(iLevelTemp);  // t-deriv and convect stored here already
      Nj_[k][i]  = field->GetTemp(iLevelTemp+1);
//    cout << "AdvectAMFE: rhs[" << i << "]=" << *Nj_[k][i] << endl;
      if ( up_[k][i] == NULL || Nj_[k][i] == NULL ) {
        cout << "BurgersSolver::AdvectAMFE: one or more NULL temp vectors" << endl;
        exit(1);
      }
      u_list_[k]->next();
    }
  } 

  // Now, add in the AM approximation for the diffusion term (only the
  // the explicit portion):
  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
  }
  for ( i=0; i<nelems_; i++ ) {
    for ( k=0; k<nfields_; k++ ) {
      field      =  u_list_[k]->member();
      *Nj_[k][i] = 0.0;
      tmp        = field->GetTemp(iLevelTemp+2);
      for ( j=1; j<iorderam_; j++ )  {    // Do diffusionAM(explicit) contribution to RHS:
        u    = field->GetExpCoeffs(j);
        MTK::fvec_const_prod_sum_rep(*Nj_[k][i], *u, 1.0, -mu_[j]);
      }  // end of AM(explicit) loop
      ((HelmholtzOp*)H_[i])->OpVec_prod(*Nj_[k][i], *tmp);  // L * u^(n-j+1), includes nu
      MTK::fvec_const_prod_sum_rep(*up_[k][i], *tmp, 1.0, 1.0);
    } // end of field loop
    for ( k=0; k<nfields_; k++ ) {
      u_list_[k]->next();
    }
  }  // end of elem loop

  // Unlock temp memory so that we can re-use it later:
  for ( k=0; k<nfields_; k++ ) {
    for ( i=0; i<nelems_; i++ ) {
      u_list_[k]->member(i)->TempUnlock(up_[k][i]);
      u_list_[k]->member(i)->TempUnlock(Nj_[k][i]);
    }
  }

  return TRUE;

} // end of operator AdvectAMFE
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : DoNormalStep
// DESCRIPTION: Time step using GAdvect operator
// ARGUMENTS  : GDOUBLE dt: time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::DoNormalStep(GDOUBLE dt)
{
  GINT     iTempLevel;
  GBOOL    bRet;
 

  if ( !Advect(dt, iTempLevel) ) return FALSE;
  if ( !DoForcing  (dt, iTempLevel) ) return FALSE;
  bRet = DoDiffusion(dt, iTempLevel);

  return bRet;

} // end of operator DoNormalStep


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : StepABBDFAM 
// DESCRIPTION: Time step using Adams-Bashforth Backward time 
//              Differencing Formulae + Adams-Moulton for the diffusion
//              term
// ARGUMENTS  : GDOUBLE dt: time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::StepABBDFAM(GDOUBLE dt)
{
  GINT     iTempLevel;
  GBOOL    bRet;
 

  if ( !AdvectAMFE(dt, iTempLevel) ) return FALSE;
  if ( !DoForcing  (dt, iTempLevel) ) return FALSE;
  bRet = DoDiffusion(dt, iTempLevel);

  return bRet;

} // end of operator StepABBDFAM
#endif



//************************************************************************************
//************************************************************************************
// METHOD     : DoDiffusion
// DESCRIPTION: Computes contribution due to diffusion 
// ARGUMENTS  : dt        : time step
//              iTempLevel: temp storage level where advection contribution is stored.
//                          Note: one add'l temp level (iTempLevel+1) is 
//                          also required here.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::DoDiffusion(GDOUBLE dt, const GINT  iTempLevel)
{
  char     *serr = "BurgersSolver::DoDiffusion: ";
  GINT     i, k ;
  GBOOL    bRet;
  Field2D  *field;
  
  // Reset errors:
  ResetErrors();

  // Reset time in required operators:
  for ( i=0; i<nelems_; i++ ) {
    ((HelmholtzOp*)H_ [i])->SetConst(Lcoeff_[0], Lcoeff_[1], Mcoeff_);
  } 
  UpdatePrecond(dt);
  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      up_rhs_ [k][i] = u_list_[k]->member()->GetTemp(iTempLevel);    // temp space holding explicit data
      u_list_[k]->next();
    }
  }

  // Get RHS for Helmholtz solve (assume that
  // product of M_ * RHS is computed, so that solver doesn't
  // have to do it): 
  for ( k=0, bRet=TRUE; k<nfields_&& bRet; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      field       = u_list_[k]->member();
      if ( up_rhs_[k][i] == NULL ) {
        cout << "BurgersSolver::DoDiff: one or more NULL temp vectors" << endl;
        exit(1);
      }
      up_[k][i] = u_list_[k]->member()->GetExpCoeffs(0);  // where to store solution
      field->ShiftLevels();                               // shift time levels, since solve _is_ solution
      u_list_[k]->next();
    }
    
//  (*cgu_[k])(gelems_, &H_, NULL, &up_[k], &up_rhs_, NULL);
//  cgu_[k]->SetBdyValues(ubdyvals_[k]);
    bRet          = cgu_[k]->SolveCG();
    niter_    [k] = cgu_[k]->GetNumIterations();
    ierror_   [k] = cgu_[k]->ErrNo();
    error_    [k] = cgu_[k]->GetError();
    min_error_[k] = cgu_[k]->GetMinError();
    max_error_[k] = cgu_[k]->GetMaxError();
    if ( !bRet ) {
      cout << "BurgersSolver::DoDiff : Helmholtz solve [" << k << "] failed" << endl;
    }
  } 

  // Unlock temp memory so that we can re-use it later:
  for ( k=0; k<nfields_; k++ ) {
    for ( i=0; i<nelems_; i++ ) {
      u_list_[k]->member(i)->TempUnlock(up_rhs_[k][i]);
    }
  }

  return bRet;

} // end of method DoDiff


//************************************************************************************
//************************************************************************************
// METHOD     : SetVBdyData
// DESCRIPTION: Set boundary conditions
// ARGUEMENTS : idir       : field component whose bdy is to be set
//              bdy_vals   : if present, gives the boundary values; each
//                           member of list must be of the same dimensionality
//                           as bdy_indices, even though a given bdy node value
//                           is set only if the btype for this node is DIRICHLET.
//                           NOTE: Once set, the bdy_vals pointers MUST NOT change!
// RETURNS    :
//************************************************************************************
void BurgersSolver::SetVBdyData(GINT  idir, GVecList *bdy_vals)
{
  if ( idir < 1 || idir > nfields_) {
    cout << "BurgersSolver::SetVBdyData: invalid component direction" << endl;
    exit(1);
  }

#if 0
  if ( ne != nelems_ ) {
    cout << "BurgersSolver::SetVBdyData: boundary lists incompatible with element number" << endl;
    exit(1);
  }
#endif

  ubdyvals_[idir-1] = bdy_vals;
  bInitReqd_ = TRUE;


} // end of method SetVBdyData


//************************************************************************************
//************************************************************************************
// METHOD     : GetIntegTime
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE BurgersSolver::GetIntegTime()
{
  return time_;
} // end of method GetIntegTime


//************************************************************************************
//************************************************************************************
// METHOD     : GetUSolver
// DESCRIPTION: Gets pointer to velocity solver
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
CG *BurgersSolver::GetSolver(GINT  idir)
{
  if      ( idir < 1 || idir > nfields_ ) return NULL;
  return cgu_[idir-1];
} // end of method GetUSolver


//************************************************************************************
//************************************************************************************
// METHOD     : ResetErrors
// DESCRIPTION: Resets error quantities
//
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void BurgersSolver::ResetErrors()
{
  
  GINT  j;

  for ( j=0; j<nfields_; j++ ) {
    niter_    [j]      = -1;
    ierror_   [j]      = -1;
    error_    [j]      = 0.0;
    min_error_[j]      = 0.0;
    max_error_[j]      = 0.0;
  }

} // end of method ResetErrors


//************************************************************************************
//************************************************************************************
// METHOD     : SetPreconditioner
// DESCRIPTION: Sets preconditioners for all fields
// ARGUMENTS  : 
//              itype : preconditioner type
// RETURNS    :
//************************************************************************************
void BurgersSolver::SetPreconditioner(GPC itype)
{
  GINT  k;
  
  bPrecond_  =  FALSE;

  for ( k=0; k<nfields_; k++ ) {
    if ( cgu_[k] == NULL ) {
      cout << "BurgersSolver::SetPreconditioner: invalid solver" << endl;
      exit(1);
    }
    upc_type_[k] = itype;
  }

  if ( !InitPrecond() ) {
    cout << "BurgersSolver::SetPreconditioner: could not initialize preconditioner(s)" << endl;
    exit(1);
  }
  bPrecond_  =  TRUE;

  for ( k=0; k<nfields_; k++ ) {
    cgu_[k]->SetPreconditioner(&upc_[k]);
  }
} // end of method SetPreconditioner


//************************************************************************************
//************************************************************************************
// METHOD     : GetPreconditioner
// DESCRIPTION: Gets preconditioners
// ARGUMENTS  : idir: velocity component (1 or 2, (or 3))
//              eid : element number
// RETURNS    :
//************************************************************************************
LinOp *BurgersSolver::GetPreconditioner(GINT  idir,GINT  eid)
{
  if ( idir < 1 || idir > nfields_ ) {
    cout << "BurgersSolver::GetPreconditioner: invalid component" << endl;
    return NULL;
  }
  return (LinOp*)(upc_[idir-1][eid]);
} // end of method GetPreconditioner


//************************************************************************************
//************************************************************************************
// METHOD     : InitPrecond
// DESCRIPTION: Initializes preconditioners
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL BurgersSolver::InitPrecond()
{
  GINT   i, k;
  Elem2D *ev;
 
  if ( !bPrecond_ ) return TRUE;

  for ( k=0; k<nfields_; k++ ) {
    switch ( upc_type_[k] ) {
      case GPC_BLOCKJAC_HELM:
        u_list_[k]->start(NULL);
        for ( i=0; i<nelems_; i++ ) {
          if ( upc_[k][i] != NULL ) { return FALSE; }
          if ( (ev=u_list_[k]->member()->GetElement()) == NULL ) return FALSE;
          upc_[k][i] = new PCBlockJac_Helm(ev);
          if ( upc_[k][i] == NULL ) return FALSE;
          u_list_[k]->next();
        }
        break;
      case GPC_POINTJAC_HELM:
        u_list_[k]->start(NULL);
        for ( i=0; i<nelems_; i++ ) {
          if ( upc_[k][i] != NULL ) { return FALSE ;}
          if ( (ev=u_list_[k]->member()->GetElement()) == NULL ) return FALSE;
          upc_[k][i] = new PCPointJac_Helm(ev);
          if ( upc_[k][i] == NULL ) return FALSE;
          u_list_[k]->next();
        }
        break;
       case GPC_NONE:
         break;
    default:
        return FALSE;
    }
  }

  return TRUE;

} // end of method InitPrecond


//************************************************************************************
//************************************************************************************
// METHOD     : UpdatePrecond
// DESCRIPTION: Updates preconditioners, as required, for time step change
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void BurgersSolver::UpdatePrecond(GDOUBLE ddt)
{
  GINT  i, k;
  GDOUBLE fdt;

  if ( !bPrecond_ )     return;  // don't update if no precond. set
  if ( ddt == dtlast_ ) return;  // don't update if we don't need to

  dtlast_ = ddt;

  fdt  =  gamma0_/ddt;

  for ( k=0; k<nfields_; k++ ) {
    switch ( upc_type_[k] ) {
    case GPC_BLOCKJAC_HELM:
      if (  upc_[k].size() == 0 ) break;
      for ( i=0; i<nelems_; i++ ) {
        if ( upc_[k][i] == NULL )  continue;
        ((PCBlockJac_Helm*)upc_[k][i])->SetConst(nu_[0], nu_[1], fdt);
      }
      break;
    case GPC_POINTJAC_HELM:
      if (  upc_[k].size() == 0 ) break;
      // Note: update here if nu, dt don't change is dangerous! Thus,
      // the check with dtlast_ above....
      for ( i=0; i<nelems_; i++ ) {
        if ( upc_[k][i] == NULL )  continue;
        Hdiag_[i] = ((PCPointJac_Helm*)upc_[k][i])->SetConst(nu_[0], nu_[1], fdt);
      }

      // Must now do a direct stiffness summation (global gather/scatter) on
      // unassembled H-diagonal:
//    if ( !GUtils::ComputeGDiag(Hdiag_, *gelems_, (NTreeAdapt *)gsop, hDSOp, Nj_) ) {
      if ( !gsop->DSOp(Hdiag_, G_OP_SUM, hDSOp) ) {
        cout << "BurgersSolver::UpdatePrecond: Unable to assemble operator" << endl;
        exit(1);
      }

      for ( i=0; i<nelems_; i++ ) {
        if ( ((PCPointJac_Helm*)upc_[k][i])->Invert() == NULL ) {
          cout << "BurgersSolver::UpdatePrecond: Unable to form preconditioner" << endl;
          exit(1);
        }
      }
      break;
     case GPC_NONE:
       break;
    default:
      cout << "BurgersSolver::UpdatePrecond: invalid preconditioner type" << endl ;
      exit(1);
    }
  }

} // end of method UpdatePrecond


//************************************************************************************
//************************************************************************************
// METHOD     : DoForcing
// DESCRIPTION: Performs update due to state-independent forcing term(s).
//              Note: in addition to the iLevelTemp temporary field in the
//                    field list member, one additional temp. field must
//                    exist.
//
// ARGUMENTS  :
//             dt         : time step over which to perform update
//             iLevelTemp : temp field to update
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::DoForcing(GDOUBLE dt, GSHORT  iLevelTemp)
{
  GINT          i, k;
  GVector      *up, *accel;
  SIForceList  *force_list;
  SIForce      *force;

  return TRUE;


  if ( !u_list_ ) return FALSE;

  for ( k=0; k<nfields_; k++ ) {
    if ( !u_list_[k] )  return FALSE;
    u_list_[k]->start(NULL);
    force_list = u_list_[k]->member()->GetSIForceList();
    if ( force_list == NULL ) continue;
    force_list->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      while ( (force=force_list->member()) ) {
        up   = u_list_[k]->member()->GetTemp(iLevelTemp);
        accel = u_list_[k]->member()->GetTemp(iLevelTemp+1);
        if ( !up || !accel ) {
          cout << "BurgersSolver::DoForcing: one or more NULL storage-level fields" << endl;
          return FALSE;
        }
        force->SetTimeStep(dt);
        if ( !force->Accel(time_, k+1, *accel) ) return FALSE;
//      *up += ((*accel) * dt);
        MTK::fvec_const_prod_sum_rep(*up, *accel, 1.0, dt);
        u_list_[k]->member(i)->TempUnlock(up);
        force_list->next();
      }
    }
    u_list_[k]->next();
  }

  return TRUE;

} // end of method DoForcing


//************************************************************************************
//************************************************************************************
// METHOD     : BuildCoeffs
// DESCRIPTION: creates the 'database' of approximation constant
//              expansion coefficients.
//
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void BurgersSolver::BuildCoeffs()
{
  memset(c_am , 0, 16*sizeof(GDOUBLE));

  c_am [0][0] =  1.0;
  c_am [1][0] =  0.5;
  c_am [1][1] =  0.5;
  c_am [2][0] =  5.0/12.0;
  c_am [2][1] =  4.0/3.0;
  c_am [2][2] = -0.5;
  c_am [3][0] =  3.0/8.0;
  c_am [3][1] =  19.0/24.0;
  c_am [3][2] = -5.0/24.0;
  c_am [3][3] =  1.0/24.0;

} // end of method BuildCoeffs


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumIterations
// DESCRIPTION: 
//
// ARGUMENTS  : none
// RETURNS    : num iterations
//************************************************************************************
GINT  BurgersSolver::GetNumIterations(GINT  idir)
{
  if ( idir < 1 || idir > nfields_ ) {
    cout << "BurgersSolver::GetNumIterations: invalid coordinate direction" << endl;
    exit(1);
  }
  return niter_[idir-1];
} // end of method GetNumIterations


//************************************************************************************
//************************************************************************************
// METHOD     : GetErrorType
// DESCRIPTION: 
//
// ARGUMENTS  : none
// RETURNS    : error condition type
//************************************************************************************
GINT  BurgersSolver::GetErrorType(GINT  idir)
{
  if ( idir < 1 || idir > nfields_ ) {
    cout << "BurgersSolver::GetErrorType: invalid coordinate direction" << endl;
    exit(1);
  }
  return ierror_[idir-1];
} // end of method GetErrorType


//************************************************************************************
//************************************************************************************
// METHOD     : GetError
// DESCRIPTION: 
//
// ARGUMENTS  : none
// RETURNS    : error value
//************************************************************************************
GDOUBLE BurgersSolver::GetError(GINT  idir)
{
  if ( idir < 1 || idir > nfields_ ) {
    cout << "BurgersSolver::GetError: invalid coordinate direction" << endl;
    exit(1);
  }
  return error_[idir-1];
} // end of method GetError


//************************************************************************************
//************************************************************************************
// METHOD     : GetMaxError
// DESCRIPTION: 
//
// ARGUMENTS  : none
// RETURNS    : max iterative error value
//************************************************************************************
GDOUBLE BurgersSolver::GetMaxError(GINT  idir)
{
  if ( idir < 1 || idir > nfields_ ) {
    cout << "BurgersSolver::GetMaxError: invalid coordinate direction" << endl;
    exit(1);
  }
  return max_error_[idir-1];
} // end of method GetMaxError


//************************************************************************************
//************************************************************************************
// METHOD     : GetMinError
// DESCRIPTION: 
//
// ARGUMENTS  : none
// RETURNS    : min iterative error value
//************************************************************************************
GDOUBLE BurgersSolver::GetMinError(GINT  idir)
{
  if ( idir < 1 || idir > nfields_ ) {
    cout << "BurgersSolver::GetMinError: invalid coordinate direction" << endl;
    exit(1);
  }
  return min_error_[idir-1];
} // end of method GetMinError


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoAdvection
// DESCRIPTION: sets advection flag
//
// ARGUMENTS  : bDoAdv: flag value
// RETURNS    : none
//************************************************************************************
void BurgersSolver::SetDoAdvection(GBOOL bDoAdv)
{
  bDoAdvection_ = bDoAdv;
} // end of method SetDoAdvection


//************************************************************************************
//************************************************************************************
// METHOD     : ElemListChange
// DESCRIPTION: Checks element list against local pointer copy to
//              determine if element list has been changed.
// ARGUMENTS  : none
// RETURNS    : TRUE if list has changed; else FALSE
//************************************************************************************
GBOOL BurgersSolver::ElemListChange()
{ 
  GINT  i=0, esz=gelems_->size();
  if ( esz != H_.size() || esz != nelems_ ) return TRUE;

  while ( i<esz && (*gelems_)[i] == ((HelmholtzOp*)H_[i])->GetElem() ) i++;
 
 
  return ( i < esz );
} // end of method ElemListChange


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : StepLeapfrog
// DESCRIPTION: Takes  a step using a (explicit) leapfrog method:
// 
//                M u^n+1 = M u^n-1 - 2dt[M C(u^n)  + nu L u^n-1]
//
//              where u^k is the solution on each time level, M, the
//              mass matrix, dt the time step, C(u) the advection operator,
//              L the weak Laplacian operator and nu, the viscosity.
//
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::StepLeapfrog(GDOUBLE dt)
{
  GINT     i, j, k, m;
  GBOOL    bRet=TRUE;
  GVector  *uj[3]={NULL,NULL,NULL}, *M, *tmp1, *tmp2, *un, *unm1; 
  Elem2D   *elem;
  Field2D  *field;

  // Reset solver errors:
  ResetErrors();
  for ( j=0; j<GDIM; j++ ) Lcoeff_[j] = nu_[j];
  for ( i=0; i<nelems_; i++ ) {
    ((HelmholtzOp*)H_ [i])->SetConst(Lcoeff_[0], Lcoeff_[1], 0.0);
  } 

  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      field        = u_list_[k]->member();
      elem         = field->GetElement();
      M            = elem->GetMassMatrix();
      un           = field->GetTemp(0);
      unm1         = field->GetTemp(1);
      up_rhs_[k][i]= field->GetTemp(2);
      if ( tmp1 == NULL || tmp2 == NULL || up_rhs_[k][i] == NULL ) {
        cout << "BurgersSolver::StepLeapfrog: one or more NULL temp vectors" << endl;
        exit(1);
      }
      up_ [k][i]  = u_list_[k]->member()->GetExpCoeffs(0);  // where to store solution
      *unm1       = *(u_list_[k]->member()->GetExpCoeffs(1));
      *un         = *up_[k][i];
      field->ShiftLevels();                                 // shift time levels, since solve _is_ solution
      if ( bConstAdvVel_ ) {
        for ( m=0; m<nadv_; m++ )
          uj[iadv_[m]] = cadv_[m]->member(i)->GetExpCoeffs(0);
      }
      else { 
        for ( m=0; m<nfields_; m++ )
         uj[iadv_[m]] = u_list_[m]->member(i)->GetExpCoeffs(0);
      }

      ((VectorOp*)vecop_[i])->SetVec(uj[0], uj[1], uj[2]);
      ((VectorOp*)vecop_[i])->Advect(*up_[k][i], *tmp1);    
      MTK::fvec_point_prod_rep(*tmp1,*M);                        // M*Advection          
//    cout << "Burgers::StepLeapfrog: adv [" << i << "]=" << *tmp1 << endl;
      ((HelmholtzOp*)H_[i])->OpVec_prod(*unm1, *tmp2);           // L * u^n-1, includes nu
//    cout << "Burgers::StepLeapfrog: diff[" << i << "]=" << *tmp2 << endl;
      MTK::fvec_const_prod_sum_rep(*tmp2, *tmp1, 1.0, 1.0); 
      MTK::fvec_const_prod(*tmp2, -2.0*dt, *up_rhs_[k][i]);      // RHS vector            
      *tmp1 = *unm1;
      MTK::fvec_point_prod_rep(*tmp1,*M);                        // M*u^n-1 + C(c)*u^n-1 + L*u^n-1
      MTK::fvec_const_prod_sum_rep(*up_rhs_[k][i], *tmp1, 1.0, 1.0); 
//    cout << "Burgers::StepLeapfrog: rhs[" << i << "]=" << *up_rhs_[k][i] << endl;
      field->TempUnlock(tmp1);
      field->TempUnlock(tmp2);
      field->TempUnlock(up_rhs_[k][i]);
      u_list_[k]->next();
    }
  }
#if 0
  for ( k=0, bRet=TRUE; k<nfields_ && bRet; k++ ) {
    bRet = cgu_[k]->SolveCG();
    niter_    [k] = cgu_[k]->GetNumIterations();
    ierror_   [k] = cgu_[k]->ErrNo();
    error_    [k] = cgu_[k]->GetError();
    min_error_[k] = cgu_[k]->GetMinError();
    max_error_[k] = cgu_[k]->GetMaxError();
    if ( !bRet ) {
      cout << "BurgersSolver::StepLeapfrog: Solve [" << k << "] failed" << endl;
    }
  }
#endif
#if 1
  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
//  cout << "Burgers::StepLeapfrog: u^n+1[" << i << "]=" << *up_[k][i] << endl;
      u_list_[k]->next();
    }
  }
#endif
  return bRet;
} // end of method StepLeapfrog
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : UpdateSolver
// DESCRIPTION: updates solver due to grid changes
//
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void BurgersSolver::UpdateSolver() 
{
  GINT     k;

  for ( k=0; k<nfields_; k++ ) {
    if ( bPrecond_ )
      (*cgu_[k]).SetPreconditioner(&upc_[k]);
//  if ( itime_type_ == TE_LEAPFROG ) 
//    (*cgu_[k])(gelems_, &M_, NULL, &up_[k], &up_rhs_[k], NULL);
//  else
    (*cgu_[k])(gelems_, &H_, NULL, &up_[k], &up_rhs_[k], NULL);
    cgu_[k]->SetBdyValues(ubdyvals_[k]);
  }
} // end of UpdateSolver


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : StepTimeIndepRKK
// DESCRIPTION: Takes  a step using (explicit) RKk method, which is 
//              good as long as there is no explicit time dependence in any
//              of the forcing terms (see Canutto, etal: Spectral Methods in 
//              Fluid Dynamics, pp.107-109, 1988 Springer-Verlag).
//              Equation solved for explicitly is:
// 
//              du/dt = -M C(u) - nu L u.
//
//              where M, is the mass matrix, dt the time step, C(u) the advection 
//              operator, L the weak Laplacian operator and nu, the viscosity.
//              This method will provide a second-order in time solution.
//
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::StepTimeIndepRKK(GDOUBLE dt)
{
  GINT     i, j, k, m, s;
  GBOOL    bRet=TRUE;
  GDOUBLE  dtfact;
  GVector  *uj[3]={NULL,NULL,NULL}, *M, *iM, *tmp1, *tmp2; 
  Elem2D   *elem;
  Field2D  *field;

  // Reset solver errors:
  ResetErrors();
  for ( j=0; j<GDIM; j++ ) Lcoeff_[j] = nu_[j];
  for ( i=0; i<nelems_; i++ ) {
    ((HelmholtzOp*)H_ [i])->SetConst(Lcoeff_[0], Lcoeff_[1], 0.0);
  } 

  for ( k=0; k<nfields_; k++ ) u_list_[k]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( k=0; k<nfields_; k++ ) {
      if ( bConstAdvVel_ ) {
        for ( m=0; m<nadv_; m++ )
          uj[iadv_[m]] = cadv_[m]->member(i)->GetExpCoeffs(0);
      }
      else { 
        for ( m=0; m<nfields_; m++ )
         uj[iadv_[m]] = u_list_[m]->member(i)->GetExpCoeffs(0);
      }
    }
    ((VectorOp*)vecop_[i])->SetVec(uj[0], uj[1], uj[2]);
    for ( k=0; k<nfields_; k++ ) {
      field          = u_list_[k]->member();
      elem           = field->GetElement();
      M              = elem->GetMassMatrix();
      iM             = elem->GetiMass();
      up_     [k][i] = u_list_[k]->member()->GetExpCoeffs(0);           // where to store solution
      up_rhs_ [k][i] = field->GetTemp(0);
      *up_rhs_[k][i] = *up_[k][i];                                      // u = u^n
      tmp1           = field->GetTemp(1);
      tmp2           = field->GetTemp(2);
      if ( up_rhs_[k][i] == NULL || tmp1 == NULL || tmp2 == NULL ) {
        cout << "BurgersSolver::StepTimeIndepRKK: one or more NULL temp vectors" << endl;
        exit(1);
      }
      for ( s=iorderRKK_; s>0; s-- ) {
        dtfact = dt / s;                                               
        ((VectorOp*)vecop_[i])->Advect(*up_rhs_[k][i], *tmp1);    
        MTK::fvec_point_prod_rep(*tmp1,*M);                            // M*C(u)u
        ((HelmholtzOp*)H_[i])->OpVec_prod(*up_rhs_[k][i], *tmp2);      // L * u^n, includes nu
        MTK::fvec_const_prod_sum_rep(*tmp1, *tmp2,  -1.0, -1.0);       // F(u) = -M*C(u)u - nu L u
        MTK::fvec_point_prod_rep(*tmp1,*iM);                           // M^-1 * F(u)
        MTK::fvec_const_prod_rep(*tmp1, dtfact);                       // dt/s * M^-1 F(u)
        MTK::fvec_add(*up_[k][i], *tmp1, *up_rhs_[k][i]);              // u=u^n+dt/s * M^-1 F(u)
      } // end of RK -sum loop
      field->TempUnlock(tmp1);
      field->TempUnlock(tmp2);
    } // end of field loop
    for ( k=0; k<nfields_; k++ ) u_list_[k]->next();
  } // end of element loop

#if 1
  for ( k=0; k<nfields_; k++ ) u_list_[k]->start(NULL);
  for ( k=0; k<nfields_; k++ ) {
    u_list_[k]->start(NULL);
    for ( i=0; i<nelems_; i++ ) {
      *up_[k][i] = *up_rhs_[k][i]; 
    }
    u_list_[k]->next();
  }
#endif
#if 1
  // Finally, do an 'H1-projection' to make sure that solution is in H1:
  for ( k=0, bRet=TRUE; k<nfields_ && bRet; k++ ) {
    bRet = GUtils::H1_Project(up_[k], *gelems_, (NTreeAdapt*)gsop, hDSOp); 
  }
#endif

  // Unlock temp memory so that we can re-use it later:
  for ( k=0; k<nfields_; k++ ) {
    for ( i=0; i<nelems_; i++ ) {
      u_list_[k]->member(i)->TempUnlock(up_rhs_[k][i]);
    }
  }

  return bRet;
} // end of method StepTimeIndepRKK
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : SetNSubcycles
// DESCRIPTION: sets max number of OIFS subcycles
//
// ARGUMENTS  : GINT nsub
// RETURNS    : none.
//************************************************************************************
void BurgersSolver::SetNSubcycles(GINT nsub)
{
  advect_->SetNSubcycles(nsub);
} // end of method SetNSubcycles



//************************************************************************************
//************************************************************************************
// METHOD     : DoFilter
// DESCRIPTION: Carries out filter operation
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BurgersSolver::DoFilter()
{
  GINT    i, j;
  GVector *vtmp;

  if ( filter_ == NULL ) return TRUE;
  
  for ( j=0; j<nfields_; j++ ) u_list_[j]->start(NULL); gelems_->start(NULL); filter_->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    vtmp = gelems_->member()->GetTemp();
    *vtmp = *(u_list_[j]->member()->GetExpCoeffs(0));
    filter_->member()->OpVec_prod(*vtmp,*u_list_[j]->member()->GetExpCoeffs(0));
    gelems_->member()->TempUnlock(vtmp);
    for ( j=0; j<nfields_; j++ ) u_list_[j]->next(); gelems_->next(); filter_->next();
  }
  
  return TRUE;
} // end of method DoFilter



