//************************************************************************************//
// Module       : burgers.hpp
// Date         : 8/4/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the solver for the multi-d Burgers equation. The problem solved is
//                du/dt + u.Del u + -  nu_x  Del_x^2 u -nu_y Del_y^2 u = f(x,y;t)
// Derived From : none.
// Modifications: 7/21/03: converted to linked-list-based data structures
//************************************************************************************//
#if !defined(BURGERS_SOLVER_HPP)
#define BURGERS_SOLVER_HPP 

#include "gstepper.hpp"
#include "cg.hpp"
#include "gtlist.hpp"
#include "gttlist.hpp"
#include "glinoplist.hpp"
#include "gobjlist.hpp"
#include "gelemlist.hpp"
#include "gfieldlist.hpp"
#include "helmholtzop.hpp"
#include "gadvect.hpp"
#include "massop.hpp"
#include "vectorop.hpp"
#include "ntree_adapt.hpp"
#include "ab.hpp"
#include "ext.hpp"
#include "evforce.hpp"


class BurgersSolver : public GStepper
{
public:
  enum TIME_EVOLTYPE {TE_OIFS=0, TE_ABBDF, TE_EXBDF, TE_EXBDF_Weak};


                           BurgersSolver(GElemList *elems);
                           BurgersSolver(GElemList *elems, GFieldList *ulist[], GINT  nlist, 
                                         GFieldList **cadv, GINT  *iadv, GINT  nadv);
                          ~BurgersSolver();

         GBOOL             Step(GDOUBLE dt);                                                // Take one time step
         GDOUBLE           GetIntegTime();                                                // Get total integration time
         CG                *GetSolver(GINT  idir);                                        // Get pointer to iterative solver
         LinOp             *GetPreconditioner(GINT  idir, GINT  eid);                     // Get preconditioner for component idir, element eid
         GINT              GetNumIterations(GINT  idir);                                  // Get no. solver iterations
         GINT              GetErrorType(GINT  idir);                                      // Get solver error type
         GDOUBLE           GetError(GINT  idir);                                          // Get solver error
         GDOUBLE           GetMinError(GINT  idir);                                       // Get solver min error
         GDOUBLE           GetMaxError(GINT  idir);                                       // Get solver max error


         void              SetFilter(GLinOpList *filter);                                 // Set filter
         void              SetElemList(GElemList *elems);                                 // Set element list
         void              SetEvolType(TIME_EVOLTYPE itype);                              // Set time stepping scheme
         void              SetTime(GDOUBLE time);                                         // Set integration start time
         void              SetNSubcycles(GINT nsub);                                      // Set number of OIFS subcycles

         GBOOL             SetFields(GFieldList *ulist[], GINT  nfields, 
                                     GFieldList **cadv, GINT  ncadv, 
                                     GINT  *iadv, GINT  nadv);                            // Set field list
         void              SetPreconditioner(GPC itype);                                  // Set preconditioners for all fields
         void              SetAdvectDim(GINT  nto);                                       // Set advection direction(s)
         void              SetComm(GS *ggs);                                              // Set global gather scatter operator
         GCHandle          SetCommHandle(GCHandle hIn) ;                                  // Initializes comm with existing handle 
         void              SetVisc(const GDOUBLE nu, const GINT  idir=0);                 // Set kinematic viscosity, nu_idir
         void              SetAdvOrder(GINT  iorder);                                     // Sets order of adv. term
         void              SetAMOrder(GINT  iorder_ab);                                   // Sets order of Adams-Moulton diff. approx. 
         void              SetBDFOrder(GINT  iorder);                                     // Sets order of BDF time approx.
         void              SetVBdyData (GINT  idir, GVecList *bdy_vals);                  // Set bcs for component idir
         void              SetDoAdvection(GBOOL bDoAdv);                                  // Set advection flag
         void              SetTimestepHistory(GDBuffer *dthist);                          // Set timestep hist. buffer

private:
         // Methods:
         GBOOL             DoNormalStep(GDOUBLE dt);                                      // Step with GAdvect oprator
//       GBOOL             StepABBDFAM(GDOUBLE dt);                                       // Adams-Bashforth, backward time diff, + Adams-Moulton form
         GBOOL             DoDiffusion(GDOUBLE dt, const GINT  tmpLevel);                 // Do diffusion terms
//       GBOOL             StepTimeIndepRKK(GDOUBLE dt);                                  // Use RK2 to step advection part
//       GBOOL             StepLeapfrog(GDOUBLE dt);                                      // Use leapfrog to step 
         GBOOL             ResetExpandables(GFieldList *u1[], GINT  nf);                  // Reset lists
         GBOOL             CreateCompDynamic(GINT  ncomps);                               // Create dynamically-allocated list-dependent quantities
         GBOOL             CreateAdvDynamic(GINT  nadv);                                  // Create dynamically-allocated list-dependent quantities
         void              DeleteDynamic();
         GBOOL             Advect(GDOUBLE dt, GINT  &iLevel);                             // Do advection term using AB/BDF; store in temp level iLevel
//       GBOOL             AdvectAMFE(GDOUBLE dt, GINT  &iLevel);                         // Do advection term using AM+Forward Euler ; store in temp level iLevel

         GBOOL             DoForcing  (GDOUBLE dt, GSHORT   iLevel);                      // Do state-indep. forcing: update tmp iLevel field

         GBOOL             InitPrecond();                                                 // Create/initialize preconditioners 
         void              UpdatePrecond(GDOUBLE dt);                                     // Update preconditioners for time step
         void              UpdateSolver();                                                // Update solver for time step
         void              ResetErrors();                                                 // Resets solver errors
         void              BuildCoeffs();                                                 // Builds coefficients
         GBOOL             ElemListChange();                                              // Checks if elemlist has changed
         GBOOL             DoFilter();                                                    // Carries out filter operation, if any

         // Field and other data:
         GCHandle          hDSOp;
         TIME_EVOLTYPE     itime_type_;
         SI_FORCE_TYPE     isiforce_type_;
         GPC               *upc_type_;
         GBOOL             bConstAdvVel_;
         GBOOL             bPrecond_;
         GBOOL             bDoAdvection_;
         GBOOL             bInitReqd_;
         GBOOL             bDoDiffusion_;
         GINT              nfields_;
         GINT              nadv_;
         GINT              *iadv_;
         GINT              nelems_;
         GINT              iorderadv_;
         GINT              iorderbdf_;
         GINT              ntimelevels_;
         GINT              iorderam_;
         GINT              icycle_;
         GDOUBLE           maxCFL_;
         GDOUBLE           dtlast_;
         GDOUBLE           time_;
         GDOUBLE           *nu_ ;
         GDOUBLE           Lcoeff_[GDIM];
         GDOUBLE           Mcoeff_;
         GDOUBLE           *mu_;
         GDOUBLE           gamma0_;
         GDOUBLE           c_am [4][4];
         GFieldList        **u_list_;    // entry pointers to fields
         GVecList          **ubdyvals_;
         GVecList          *up_;           // GVector          ***up_;
         GVecList          *up_rhs_;       // GVector          **up_rhs_;
         GVecList          *Nj_;           // GVector          ***Nj_;
         GVecList          Hdiag_;         // GVector          **Hdiag_;
         GVecList          uadv_[3];
         GFieldList        *cadv_[3];
         GElemList         *gelems_;
         GAdvect           *advect_;

         // Solver return vals:
         GINT              *niter_    ;
         GINT              *ierror_   ;
         GDOUBLE           *error_    ;
         GDOUBLE           *min_error_;
         GDOUBLE           *max_error_;

         // Linear solvers:
         CG                **cgu_;

         // Math operators
         GLinOpList        M_;             // HelmholtzOp       **H_;
         GLinOpList        H_;             // HelmholtzOp       **H_;
         GLinOpList        *upc_;          // GVector         ***upc_;
         GVecOpList        vecop_;         // VectorOp         **vecop_;    // Vector collocation operator -- gives advection
         GLinOpList        *filter_;

         // Communication:
         GS               *gsop;       // communication (gather-scatter) operator
};

#endif
