//************************************************************************************//
// Module       : defquad2d.hpp
// Date         : 9/14/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a single deformed quadrilateral Cartesian 2D spectral 
//                element comprised only of straight-line segments
//                The following is the ordering for the vertices (Vi)
//                and segment midpoints (Mi):
//
//          V7 ________________ V6
//            /|     M6       /|
//       M7  / |             / |
//          /  |        M5  /  |
//         /M11|  M4       /   |
//     V4 /____|___ ______/    | M10
//        |    |         | V5  |
//        |    |     M2  |     |
//        | V3 |_________|_____| V2
//    M8  |    /         |    /
//        |   /       M9 |   /
//        |  / M3        |  / M1
//        | /            | /
//        |______________|/
//       V0       M0     V1
//
// [Note that in 2d, we use just the bottom plane.]

// Derived From : Elem2D.
// Modifications:
//************************************************************************************//
#include "defquad2d.hpp"
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <stdio.h>

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
//************************************************************************************
DefQuad2D::DefQuad2D(GSHORT ntmp)
: Elem2D(ntmp),
xiNodes1              (NULL),
xiNodes2              (NULL),
Weights1              (NULL),
Weights2              (NULL),
Weights2D             (NULL),
MassMatrix            (NULL),
iMassMatrix           (NULL),
gMassMatrix           (NULL),
StiffMat1             (NULL),
StiffMat2             (NULL),
StiffMat1T            (NULL),
StiffMat2T            (NULL),
D1                    (NULL),
D2                    (NULL),
D1T                   (NULL),
D2T                   (NULL),
opInterp1             (NULL),
opInterp2             (NULL),
dInterp1              (NULL),
dInterp2              (NULL),
Jac                   (NULL),
d11                   (NULL),
d12                   (NULL),
d22                   (NULL),
g11                   (NULL),
g12                   (NULL),
g22                   (NULL),
gWJ11                 (NULL),
gWJ12                 (NULL),
gWJ22                 (NULL)
{
  GINT  n;

  elemtype_   = DEFORMED_QUAD;
  nVertices_  = (GINT )pow(2.0,GDIM);
  nEdges_     =  2*(GDIM-1)*GDIM;
#if defined(IS3D)
  nFaces_     =     2*GDIM ;
#endif

  bInitialized  = Initialize(Np1, Np2);

  for ( n=0; n<GDIM*nVertices_; n++ )  spvVertices [n] = 0.0;

}


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
//************************************************************************************
DefQuad2D::DefQuad2D(GINT  inOrder1, GINT  inOrder2, GSHORT ntmp)
: Elem2D(ntmp),
xiNodes1              (NULL),
xiNodes2              (NULL),
Weights1              (NULL),
Weights2              (NULL),
Weights2D             (NULL),
MassMatrix            (NULL),
iMassMatrix           (NULL),
gMassMatrix           (NULL),
StiffMat1             (NULL),
StiffMat2             (NULL),
StiffMat1T            (NULL),
StiffMat2T            (NULL),
D1                    (NULL),
D2                    (NULL),
D1T                   (NULL),
D2T                   (NULL),
opInterp1             (NULL),
opInterp2             (NULL),
dInterp1              (NULL),
dInterp2              (NULL),
Jac                   (NULL),
d11                   (NULL),
d12                   (NULL),
d22                   (NULL),
g11                   (NULL),
g12                   (NULL),
g22                   (NULL),
gWJ11                 (NULL),
gWJ12                 (NULL),
gWJ22                 (NULL)
{
  GINT  n;

  elemtype_  = DEFORMED_QUAD;
  nVertices_   = (GINT )pow(2.0,GDIM);
  nEdges_     =  2*(GDIM-1)*GDIM;
  vertex_neighbor_ = new GNeighborList [nVertices_];
  edge_neighbor_   = new GNeighborList [nEdges_];
#if defined(IS3D)
  nFaces_     =     2*GDIM ;
  face_neighbor_   = new GNeighborList [nFaces_];
#endif

//  basis1  = new GNBasis(Np1) ;
//  basis2  = new GNBasis(Np2) ;
  
//  if ( basis1 != NULL && basis2 != NULL ) bInitialized = TRUE;

  bInitialized  = Initialize(inOrder1, inOrder2);

  spVertices[0].x1 = spVertices[0].x2 = 0.0;
  spVertices[1].x1 = spVertices[1].x2 = 0.0;
  spVertices[2].x1 = spVertices[2].x2 = 0.0;
  spVertices[3].x1 = spVertices[3].x2 = 0.0;
  for ( n=0; n<GDIM*nVertices_; n++ )  spvVertices [n] = 0.0;

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
//************************************************************************************
DefQuad2D::DefQuad2D(GNBasis *inBasis1, GNBasis *inBasis2, GSHORT ntmp)
: Elem2D(ntmp),
xiNodes1              (NULL),
xiNodes2              (NULL),
Weights1              (NULL),
Weights2              (NULL),
Weights2D             (NULL),
MassMatrix            (NULL),
iMassMatrix           (NULL),
gMassMatrix           (NULL),
StiffMat1             (NULL),
StiffMat2             (NULL),
StiffMat1T            (NULL),
StiffMat2T            (NULL),
D1                    (NULL),
D2                    (NULL),
D1T                   (NULL),
D2T                   (NULL),
opInterp1             (NULL),
opInterp2             (NULL),
dInterp1              (NULL),
dInterp2              (NULL),
Jac                   (NULL),
d11                   (NULL),
d12                   (NULL),
d22                   (NULL),
g11                   (NULL),
g12                   (NULL),
g22                   (NULL),
gWJ11                 (NULL),
gWJ12                 (NULL),
gWJ22                 (NULL)
{
  GINT  n;

  elemtype_   = DEFORMED_QUAD;
  nVertices_   = (GINT )pow(2.0,GDIM);
  nEdges_     =  2*(GDIM-1)*GDIM;
  vertex_neighbor_ = new GNeighborList [nVertices_];
  edge_neighbor_   = new GNeighborList [nEdges_];
#if defined(IS3D)
  nFaces_     =     2*GDIM ;
  face_neighbor_   = new GNeighborList [nFaces_];
#endif


  basis1 =  inBasis1;
  basis2 =  inBasis2;

  Np1 = inBasis1->GetOrder();
  Np2 = inBasis2->GetOrder();

  if ( basis1 != NULL && basis2 != NULL ) bInitialized = TRUE;
  bInitialized &= Initialize(Np1, Np2);

  spVertices[0].x1 = basis1->GetXimin(); spVertices[0].x2 = basis2->GetXimin();
  spVertices[1].x1 = basis1->GetXimax(); spVertices[1].x2 = basis2->GetXimin();
  spVertices[2].x1 = basis1->GetXimax(); spVertices[2].x2 = basis2->GetXimax();
  spVertices[3].x1 = basis1->GetXimin(); spVertices[3].x2 = basis2->GetXimax();
  for ( n=0; n<GDIM*nVertices_; n+=2 ) 
  {
    spvVertices  [n] = spVertices[n/2].x1;
    spvVertices[n+1] = spVertices[n/2].x2;
  }

} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
DefQuad2D::~DefQuad2D()
{
   DeleteDynamic();
}

//************************************************************************************
//************************************************************************************
// Assignment operator method (1)
//************************************************************************************
void DefQuad2D::operator=(const DefQuad2D &elem)
{
  if ( &elem != this ) 
  {
   // copy data:
    Np1      = elem.Np1;
    bSolved  = elem.bSolved;

    //  copy basis, matrices, and node data:
    basis1 = elem.basis1;
    basis2 = elem.basis2;


#if 0
    if ( basis1 != NULL ) delete basis1;
    basis1 = NULL;
    if ( elem.basis1  != NULL )
    {
      basis1  = new GNBasis(Np1);
     *basis1   = *(elem.basis1) ;
    }

    if ( basis2 != NULL ) delete basis2;
    basis2 = NULL;
    if ( elem.basis2  != NULL )
    {
      basis2  = new GNBasis(Np2);
     *basis2   = *(elem.basis2) ;
    }
#endif

    if ( Weights1 != NULL ) delete Weights1 ;
    Weights1 = NULL;
    if ( elem.Weights1 != NULL ) 
    {
      Weights1 = new GVector(Np1+1);
     *Weights1 = *(elem.Weights1);
    }   

    if ( Weights2 != NULL ) delete Weights2 ;
    Weights2 = NULL;
    if ( elem.Weights2 != NULL ) 
    {
      Weights2 = new GVector(Np2+1);
     *Weights2 = *(elem.Weights2);
    }   


    if ( xiNodes1 != NULL ) delete xiNodes1;
    xiNodes1 = NULL;
    if ( elem.xiNodes1 != NULL ) 
    {
      xiNodes1 = new GVector(Np1+1);
     *xiNodes1 = *(elem.xiNodes1);
    }   

    if ( xiNodes2 != NULL ) delete xiNodes2;
    xiNodes2 = NULL;
    if ( elem.xiNodes2 != NULL ) 
    {
      xiNodes2 = new GVector(Np2+1);
     *xiNodes2 = *(elem.xiNodes2);
    }   

    if ( Jac != NULL ) delete Jac;
    Jac = NULL;
    if ( elem.Jac != NULL)
    {
      Jac = new GVector((Np1+1)*(Np2+1));
      Jac = elem.Jac;
    }

    if ( d11 != NULL ) delete d11;
    d11 = NULL;
    if ( elem.d11 != NULL)
    {
      d11 = new GVector((Np1+1)*(Np2+1));
      d11 = elem.d11;
    }

    if ( d12 != NULL ) delete d12;
    d12 = NULL;
    if ( elem.d12 != NULL)
    {
      d12 = new GVector((Np1+1)*(Np2+1));
      d12 = elem.d12;
    }

    if ( d22 != NULL ) delete d22;
    d22 = NULL;
    if ( elem.d22 != NULL)
    {
      d22 = new GVector((Np1+1)*(Np2+1));
      d22 = elem.d22;
    }

    if ( D1  != NULL ) delete D1;
    if ( D1T != NULL ) delete D1T;
    D1  = NULL;
    D1T = NULL;
    if ( elem.D1 != NULL)
    {
      D1 = new GMatrix(Np1+1,Np1+1);
      D1 = elem.D1;
    }
    if ( elem.D1T != NULL)
    {
      D1T = new GMatrix(Np1+1,Np1+1);
      D1T = elem.D1T;
    }

    if ( D2  != NULL ) delete D2;
    if ( D2T != NULL ) delete D2T;
    D2  = NULL;
    D2T = NULL;
    if ( elem.D2 != NULL)
    {
      D2 = new GMatrix(Np2+1,Np2+1);
      D2 = elem.D2;
    }
    if ( elem.D2T != NULL)
    {
      D2T = new GMatrix(Np2+1,Np2+1);
      D2T = elem.D2T;
    }
  }

} // end of = operator


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetOrder
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  GINT  element expansion order
//************************************************************************************
GINT  DefQuad2D::GetOrder(const GINT  iDir)
{
  if ( iDir == 1 )
    return Np1;
  else if ( iDir == 2 ) 
    return Np2;
  else 
    return 0;
} // end of method GetOrder
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeSpNodes
// DESCRIPTION: Computes real space locations of parent space nodes.
//              Two matrices are created, spNodes1(2), of the form
//                x_i(0,0), x_i(1,0), ..., x_i(Np1,0),
//                x_i(0,1), ...
//                .
//                .
//                .
//                x_i(0,Np2), x_i(1,Np1),..., x_i(Np1,Np2)
//
//               where x_i(l,m) is x1(2) evaluated at parent node
//               xi1_l, xi2_m, and xi are increasing with index in
//               each direction. 
//            
// ARGUMENTS  :
// RETURNS    :  TRUE on success; else FALSE. Failure occurs if
//               nodes and weights cannot be computed, or if memory
//               cannot be allocated.
//************************************************************************************
GBOOL DefQuad2D::ComputeSpNodes()
{

  // NOTE: Geometry quantities assumed to be computed in SolveFE

  if ( spNodes1 != NULL ) delete spNodes1;
  spNodes1 = NULL;
  spNodes1 = new GVector ((Np1+1)*(Np2+1));
  if ( spNodes2 != NULL ) delete spNodes2;
  spNodes2 = NULL;
  spNodes2 = new GVector ((Np1+1)*(Np2+1));

  if ( spNodes1 == NULL || spNodes2 == NULL ) 
  {
    cout << "DefQuad2D::ComputeSpNodes: memory allocation failed." << endl;
    exit(1);
  }

  if ( !bSolved )
    if ( SolveFE() <= 0 ) return FALSE; 

  GINT  i, j, k;

  
  for ( i=0; i<Np1+1; i++)
  {
     for ( j=0; j<Np2+1; j++ )
    {
        k = i + j*(Np1+1);
        (*spNodes1)(k) =  0.25* ( spVertices[0].x1*xi1m[i]*xi2m[j] + spVertices[1].x1*xi1p[i]*xi2m[j]
                       +          spVertices[3].x1*xi1m[i]*xi2p[j] + spVertices[2].x1*xi1p[i]*xi2p[j] );
        (*spNodes2)(k) =  0.25* ( spVertices[0].x2*xi1m[i]*xi2m[j] + spVertices[1].x2*xi1p[i]*xi2m[j]
                       +          spVertices[3].x2*xi1m[i]*xi2p[j] + spVertices[2].x2*xi1p[i]*xi2p[j] );
    } 
  }
  
  return TRUE;
} // end of method ComputeSpNodes


//************************************************************************************
//************************************************************************************
// METHOD     : GetSpNodes 
// DESCRIPTION: Returns pointer to spatial nodes (not std element nodes).
//              The real space nodes are not formatted in any way
//              in this routine, and take the 'native' format described
//              in the ComputeSpNodes method.
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::GetSpNodes(GINT  idir)
{
  if      ( idir == 1 ) return spNodes1;
  else if ( idir == 2 ) return spNodes2;
  else                  return NULL;

} // end of method GetSpNodes 


//************************************************************************************
//************************************************************************************
// METHOD     : Get2DWeights
// DESCRIPTION: Returns pointer to quadtraure weights corresponding to
//              1- and 2-direction nodes. These are bilinear in w1 and w2.
//
//              This quantity is the diagonal matrix.
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::Get2DWeights()
{
  return Weights2D;
} // end of method Get2DWeights


//************************************************************************************
//************************************************************************************
// METHOD     : GetJacobian 
// DESCRIPTION: Returns vector pointer  to Jacobian (dX/dXi),
//              in native format
//
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::GetJacobian()
{
  return Jac;
} // end of method GetJacobian

#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : D1
// DESCRIPTION: Returns value of tensor product 1-derivative at indices i, j
//
// ARGUMENTS  :
// RETURNS    :  GDOUBLE
//************************************************************************************
GDOUBLE &DefQuad2D::TensorD1(GINT  i, GINT  j)
{
  if ( i<0 || i>=(Np1+1)*(Np2+1) || j<0 || j>=(Np1+1)*(Np2+1) ) {
    cout << "DefQuad2D::D1: 1-index out of range" << endl;
    exit(1);
  }
  fRet = 0.0;
  if ( abs(i-j) < (Np1+1) )  fRet = (*D1)(i%N1,j%N1);
  return fret;
} // end of method TensorD1


//************************************************************************************
//************************************************************************************
// METHOD     : TensorD2
// DESCRIPTION: Returns value of tensor product 2-derivative at indices i, j
//
// ARGUMENTS  :
// RETURNS    :  GDOUBLE
//************************************************************************************
GDOUBLE &DefQuad2D::TensorD2(GINT  i, GINT  j)
{
  if ( i<0 || i>=(Np1+1)*(Np2+1) || j<0 || j>=(Np1+1)*(Np2+1) ) {
    cout << "DefQuad2D::D2: index out of range" << endl;
    exit(1);
  }
  fRet = 0.0;
  GINT  ib=i/(Np1+1), jb=j/(Np1+1);
  GINT  ir=i%(Np1+1), jr=j%(Np1+1);

  if ( i%(Np1+1) == j%(Np1+1)  )  fRet = (*D2)(i%N1,j%N1);
  return fret;
} // end of method TensorD2
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : GetdXidX (1)
// DESCRIPTION: Returns matrices to all the quantities dXi_i/dX_j,
//              used in computing the derivatives wrt X_j. These
//              matrices have the form established in the
//              ComputeGeo method, where these derivatives are
//              computed (i.e., 'native' ordering):
//
// ARGUMENTS  :  Array of GMatrix *'s, of length num. These
//               are the individual derivatives, s.t.
//               dxidx[0] = d11
//               dxidx[1] = d12
//               dxidx[2] = d21
//               dxidx[3] = d22
// RETURNS    :  GMatrix * of the first matrix, NULL on failure.
//               Failure occurs if the return matrices are NULL, or
//               if there are too few. Method will also fail if method
//               SolveFE is required in order to compute the derivatives,
//               and this method fails.
//************************************************************************************
GVector *DefQuad2D::GetdXidX(GVector *dxidx[], GINT  num)
{
  if ( dxidx == NULL || num < 4 ) return NULL;

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  dxidx[0] = d11;
  dxidx[1] = d12;
  dxidx[2] = d12;
  dxidx[3] = d22;

  return dxidx[0];
} // end of method GetdXidX (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetdXidX (2)
// DESCRIPTION: Returns pointers to the quantities dXi_i/dX_j,
//              used in computing the derivatives wrt X_j. These
//              matrices have the form established in the 
//              ComputeGeo method, where these derivatives are
//              computed (i.e., 'native' ordering): 
// ARGUMENTS  :  i, j
// RETURNS    :  GMatrix *: non-null on success; else NULL 
//************************************************************************************
GVector *DefQuad2D::GetdXidX(const GINT  i, const GINT  j)
{
  if ( i != 1 && i != 2 ) return NULL;
  if ( j != 1 && j != 2 ) return NULL;

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  if      ( i == 1 && j == 1 ) return d11;
  else if ( i == 1 && j == 2 ) return d12;
  else if ( i == 2 && j == 1 ) return d21;
  else if ( i == 2 && j == 2 ) return d22;
  else                         return NULL;

} // end of method GetdXidX (2) 


//************************************************************************************
//************************************************************************************
// METHOD     : GetMetric
// DESCRIPTION: Returns pointers to matric quantities 
//             Sum_k (  dXi_i/dX_k * dXi_j/dX_k
//              (in 'native' ordering):
// ARGUMENTS  :  indices i, j
// RETURNS    :  GVector *: non-null on success; else NULL
//************************************************************************************
GVector  *DefQuad2D::GetMetric(const GINT  i, const GINT  j)
{
  if ( i != 1 && i != 2 ) return NULL;
  if ( j != 1 && j != 2 ) return NULL;

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  if      ( i == 1 && j == 1 ) return g11;
  else if ( i == 1 && j == 2 ) return g12;
  else if ( i == 2 && j == 1 ) return g12;
  else if ( i == 2 && j == 2 ) return g22;
  else                         return NULL;

} // end of method GetMetric


//************************************************************************************
//************************************************************************************
// METHOD     : GetWJMetric
// DESCRIPTION: Returns pointers to matric quantities 
//             Sum_k (  dXi_i/dX_k * dXi_j/dX_k ) * W(X) * J(X)
//              (in 'native' ordering):
// ARGUMENTS  :  indices i, j
// RETURNS    :  GMatrix *: non-null on success; else NULL
//************************************************************************************
GVector *DefQuad2D::GetWJMetric(const GINT  i, const GINT  j)
{
  if ( i != 1 && i != 2 ) return NULL;
  if ( j != 1 && j != 2 ) return NULL;

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  if      ( i == 1 && j == 1 ) return gWJ11;
  else if ( i == 1 && j == 2 ) return gWJ12;
  else if ( i == 2 && j == 1 ) return gWJ12;
  else if ( i == 2 && j == 2 ) return gWJ22;
  else                         return NULL;

} // end of method GetWJMetric


//************************************************************************************
//************************************************************************************
// METHOD     : GetMassMatrix 
// DESCRIPTION: returns pointer to 2D mass matrix
//              Note: The Jacobian factor _is_ included in this
//              mass matrix. The matrix is diagonal....
// ARGUMENTS  :  
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::GetMassMatrix()
{
   return MassMatrix;

} // end of method GetMassMatrix

//************************************************************************************
//************************************************************************************
// METHOD     : GetiMass
// DESCRIPTION: returns pointer to inverse of mass matrix
//              Note: this is, by definition, a fully assemebled matrix
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::GetiMass()
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
   return iMassMatrix;

} // end of method GetiMass


//************************************************************************************
//************************************************************************************
// METHOD     : GetgMass
// DESCRIPTION: returns pointer to global mass matrix
//              Note: this is, to be assembled, so that inverse
//                    may be taken
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::GetgMass()
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
   return gMassMatrix;

} // end of method GetgMass


//************************************************************************************
//************************************************************************************
// METHOD     : Assemble
// DESCRIPTION: assembles quantities that can be assembled
//              Note: Assembly is done on quantities from the outside. 
//                    Here, it means to use the assembled quantities to
//                    compute, e.g, local inverses which are of use to the
//                    element.
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
GBOOL DefQuad2D::Assemble()
{
  GINT  i;

  if ( !bSolved )
    if ( !SolveFE() ) return FALSE;

  // Comnpute assemble inverse mass matrix. The 
  // global mass matrix, gMassMatrix, should have 
  // been assembled by caller prior to entry.
  for ( i=0; i<gMassMatrix->dim(); i++ ) {
    (*iMassMatrix)(i) = 1.0/((*gMassMatrix)(i)+TINY);
  }

  return TRUE;

} // end of method Assemble


//************************************************************************************
//************************************************************************************
// METHOD     : Get1DStiffMatrix 
// DESCRIPTION: returns pointer to appropriate 1D stiffness matrix
//              Note: The Jacobian factor is _not_ included in this
//              mass matrix, but the appropriate 1D weight is. 
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GMatrix  *DefQuad2D::Get1DStiffMatrix(GINT  idir, GBOOL bTranspose)
{
  GMatrix *ret = NULL;

  if ( idir == 1 )
  {
    if ( !bTranspose )
      ret = StiffMat1;
    else
      ret = StiffMat1T;
  }
  else if ( idir == 2 )
  {
    if ( !bTranspose )
      ret = StiffMat2;
    else
      ret = StiffMat2T;
  }

  return ret;

} // end of method Get1DStiffMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : Get1DDerivMatrix 
// DESCRIPTION: returns pointer to 1d deriv. matrix, D/DXi1(2),
//              given the coord. direction, iDir. The derivative
//              is that computed in GNBasis: the 1d deriv of 
//              iDir-basis function wrt Xi-iDir, evaluated at all
//              iDir-quadrature points. That is,
//                DXi(i,j) = dh_j/dXi (Xi_i)  
//              Note: The Jacobian factor is _not_ included in this
//              derivative. The derivative is stored in native format.
// ARGUMENTS  :
// RETURNS    :  GMatrix *
//************************************************************************************
GMatrix *DefQuad2D::Get1DDerivMatrix(GINT  iDir, GBOOL bTranspose)
{
  if ( basis1 == NULL || basis2 == NULL ) return NULL;

  if ( iDir == 1 )
  {
    if ( !bTranspose ) return D1;
    else               return D1T;
  }
  else if ( iDir == 2 )
  {
    if ( !bTranspose ) return D2;
    else               return D2T;
  }
  else
    return NULL;
   
} // end of method Get1DDerivMatrix

//************************************************************************************
//************************************************************************************
/**
 * METHOD     : Get1DDerivMatrixWithMass 
 * DESCRIPTION: Computes M D_i and its transpose
 *              Note: The Jacobian factor is _not_ included in this
 *              derivative. The derivative is stored in native format.
 * ARGUMENTS  :
 * RETURNS    :  GMatrix *
 */
//************************************************************************************
GMatrix *DefQuad2D::Get1DDerivMatrixWithMass(GINT  iDir, GBOOL bTranspose){
 if ( basis1 == NULL || basis2 == NULL ) return NULL;

  if ( iDir == 1 )
  {
    if ( !bTranspose ) return D1;
    else               return D1T;
  }
  else if ( iDir == 2 )
  {
    if ( !bTranspose ) return D2;
    else               return D2T;
  }
  else
    return NULL;
} // end of method Get1DDerivMatrixWithMass
  
//************************************************************************************
//************************************************************************************
/**
 * METHOD     : Get1DDerivMatrixWeak 
 * DESCRIPTION: Computes D^T_i M and its transpose
 *              Note: The Jacobian factor is _not_ included in this
 *              derivative. The derivative is stored in native format.
 * ARGUMENTS  :
 * RETURNS    :  GMatrix *
 */
//************************************************************************************
GMatrix *DefQuad2D::Get1DDerivMatrixWeak(GINT  iDir, GBOOL bTranspose){
 if ( basis1 == NULL || basis2 == NULL ) return NULL;

  if ( iDir == 1 )
  {
    if ( !bTranspose ) return D1;
    else               return D1T;
  }
  else if ( iDir == 2 )
  {
    if ( !bTranspose ) return D2;
    else               return D2T;
  }
  else
    return NULL;
} // end of method Get1DDerivMatrixWeak

//************************************************************************************
//************************************************************************************
// METHOD     : GetXiNodes
// DESCRIPTION: Returns array to std element nodes.
//              Nodes are arranged in the order
//              xi1(0),...,xi1(N1); or i2(0),...,xi2(N2)
//              each group being arranged from smallest to
//              largest.
// ARGUMENTS  : GINT  idir gives coordinate direction whose Xi-coords are desired.
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::GetXiNodes(GINT  idir)
{

  GVector *ret = NULL;

  if ( idir == 1 )
    ret = xiNodes1;
  else if ( idir == 2 )
    ret = xiNodes2;

  return ret;
} // end of method GetXiNodes


//************************************************************************************
//************************************************************************************
// METHOD     : GetWeights
// DESCRIPTION: Returns array quadrature weights in each parent domain 
//              coord. direction.
//              Weights correspond to nodes, which are ordered from smallest
//              to largest.
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *DefQuad2D::Get1DWeights(GINT  idir)
{
  GVector *ret = NULL;

  if ( idir == 1 )
    ret = Weights1;
  else if ( idir == 2 )
    ret = Weights2;

  return ret;
} // end of method GetWeights



//************************************************************************************
//************************************************************************************
// METHOD     : GetBasisObj
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  pointer to basis object
//************************************************************************************
GNBasis *DefQuad2D::GetBasisObj(GINT  iDir)
{
   if ( iDir == 1 ) 
     return basis1;
   else if ( iDir == 2 )
     return basis2;
   else
     return NULL;

} // end of method GetBasisObj


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetBasisAtXi
// DESCRIPTION: Compute expansion basis at the parent domanin (xi) points.
//              This quantity really only has meaning for modal basis,
//              since for nodal bases, the basis is 1 at nodal points
//              (and 0 when not at nodal points). Provision is made for
//              this quantity, but only for specified 1- and 2-indices.
//              
// ARGUMENTS  : GINT  i, j, representing the 1- and 2-indices for labeling
//              the basis function, phi(i,j). The matrix ret is of type GMatrix,
//              and contains the basis fcn evaulated at the quadrature points, and
//              arranged s.t.
//
//              B(0,0) B(0,1)    ...   B(0,Np2)
//              B(1,0)           ...
//              .
//              .
//              B(Np1,0)         ...   B(Np1,Np2)
//
//              where the value of the quadrature 1- &  2- coord. increases with index.
// RETURNS    : GMatrix pointer of return matrix on success; else NULL.
//************************************************************************************
GMatrix *DefQuad2D::GetBasisAtXi(GINT  i, GINT  j, GMatrix *B)
{
  if ( B == NULL ) return NULL;
  if ( B->dim(1) != (Np1+1)  || B->dim(2) != (Np2+1) ) return NULL;

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  GINT  l, m;
  GDOUBLE *b_data, *b1_data, *b2_data;
  GMatrix B1(Np1+1,Np1+1), B2(Np2+1,Np2+1);

  // Get 1- and 2- bases evaluated at the respective 1- and 2- quadrature points:
  if ( basis1->GetBasisAtNodes(&B1) == NULL ) return NULL;
  if ( basis2->GetBasisAtNodes(&B2) == NULL ) return NULL;
  
  // Note that each Ba from GetBasis is s.t.
  //   Ba(i,j) = Phi_i(xi_j)

  // When doing the multiplication here, it would be best not to make
  // the method calls with each (*,*) invocation, but instead work
  // direcly on the data.... This is done here in order to be explicit.
  b_data  = B->Data();
  b1_data = B1.Data();
  b2_data = B2.Data();
  for ( l=0; l<Np1+1; l++ )
  {
    for ( m=0; m<Np2+1; m++ )
    {
//    (*B)(l,m) = B1(i,l)*B2(j,m);
      *(b_data+i*(B->dim(2))+j) = (*(b1_data+i*(Np1+1)+l)) * (*(b2_data+j*(Np2+1)+m));
    }
  }
 
  return B;

} // end of method GetBasisAtXi
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : SetVertices
// DESCRIPTION: Set elements real-space vertices. These should be
//              set in counter clockwise order.
// ARGUMENTS  : Array of Point3D points representing vertex points.
//              There must be at least 4 of these, and only the
//              first 4 are used. Not very robust. Should be in
//              counter clockwise order.
// RETURNS    :  none
//************************************************************************************
GBOOL DefQuad2D::SetVertices(Point3D p[], GINT  num)
{
  if ( num < nVertices_ ) return FALSE;

  GINT  j, m, n;

  for ( n=0; n<nVertices_; n++ )
  {
    spVertices[n].x1 = p[n].x1;
    spVertices[n].x2 = p[n].x2;
  }
  for ( n=0; n<GDIM*nVertices_; n+=2 )
  {
    spvVertices  [n] = spVertices[n/2].x1;
    spvVertices[n+1] = spVertices[n/2].x2;
  }

  // Compute the edge segment midpoints:
  for ( n=0; n<4; n++ ) {   // bottom plane
    m = (n+1) % 4;
    for ( j=0; j<GDIM; j++ ) spMidpoints[n][j]= 0.5*(spVertices[m][j]+spVertices[n][j]);
  }

  // Compute element center:
  (*elemCenter)[0] = spMidpoints[0][0];
  (*elemCenter)[1] = 0.5*(spMidpoints[0][1]+spMidpoints[2][1]);

#if defined(IS3D) 
  for ( n=4; n<8; n++ ) {   // top plane, if in 3D
    m = (n+1) % 8;
    for ( j=0; j<GDIM; j++ ) spMidpoints[n][j]= 0.5*(spVertices[m][j]+spVertices[n][j]);
  }
  for ( n=8,k=0; n<12; n++,k++ ) { // vertical edges, if in 3D
    m = 0.5*(n+k);   
    for ( j=0; j<GDIM; j++ ) spMidpoints [n][j]= 0.5*(spVertices[m][j]+spVertices[m-4][j]);
  }
#endif


  return TRUE;

} // end of method SetVertices

#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis (1)
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void DefQuad2D::SetBasis(GNBasis *semb1, GNBasis *semb2)
{
   basis1 = semb1;
   basis2 = semb2;

   bSolved = FALSE;
   
   if ( basis1 == NULL || basis2 == NULL ) return;

   Resize(basis1->GetOrder(), basis2->GetOrder());

} // end of method SetBasis (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis (2)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void DefQuad2D::SetBasis(GNBasis *b, GINT  idir)
{
  if ( idir < 1 || idir > GDIM) {
    cout << "DefQuad2D::SetBasis(2): invalid coordinate direction" << endl;
    exit(1);
  }
  if      ( idir == 1 ) basis1 = b;
  else if ( idir == 2 ) basis2 = b;

  bSolved = FALSE;

  Resize(basis1->GetOrder(), basis2->GetOrder());

} // end of method SetBasis (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetOrder
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void DefQuad2D::SetOrder(GINT  iorder1, GINT  iorder2)
{

   if ( iorder1 == Np1 && iorder2 == Np2  ) return;

   bInitialized = FALSE;
 
   Resize(iorder1, iorder2);

} // end of method SetOrder
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void DefQuad2D::DeleteDynamic()
{

  if ( Weights1    != NULL ) delete Weights1;
  if ( Weights2    != NULL ) delete Weights2;
  if ( Weights2D   != NULL ) delete Weights2D;
  if ( MassMatrix  != NULL ) delete MassMatrix;
  if ( gMassMatrix != NULL ) delete gMassMatrix;
  if ( iMassMatrix != NULL ) delete iMassMatrix;
  if ( StiffMat1   != NULL ) delete StiffMat1;
  if ( StiffMat2   != NULL ) delete StiffMat2;
  if ( StiffMat1T  != NULL ) delete StiffMat1T;
  if ( StiffMat2T  != NULL ) delete StiffMat2T;
  if ( xiNodes1    != NULL ) delete xiNodes1;
  if ( xiNodes2    != NULL ) delete xiNodes2;
  if ( spNodes1    != NULL ) delete spNodes1;
  if ( spNodes2    != NULL ) delete spNodes2;
  if ( D1          != NULL ) delete D1;
  if ( D2          != NULL ) delete D2;
  if ( D1T         != NULL ) delete D1T;
  if ( D2T         != NULL ) delete D2T;
  if ( Jac    != NULL ) delete Jac;

  if ( d11     != NULL ) delete d11;
  if ( d12     != NULL ) delete d12;
  if ( d21     != NULL ) delete d21;
  if ( d22     != NULL ) delete d22;

  if ( g11     != NULL ) delete g11;
  if ( g12     != NULL ) delete g12;
  if ( g22     != NULL ) delete g22;

  if ( gWJ11   != NULL ) delete gWJ11;
  if ( gWJ12   != NULL ) delete gWJ12;
  if ( gWJ22   != NULL ) delete gWJ22;

  if ( xi1p    != NULL ) delete [] xi1p;
  if ( xi1m    != NULL ) delete [] xi1m;
  if ( xi2p    != NULL ) delete [] xi2p;
  if ( xi2m    != NULL ) delete [] xi2m;

  Weights1    = NULL;
  Weights2    = NULL;
  Weights2D   = NULL;
  MassMatrix  = NULL;
  gMassMatrix = NULL;
  iMassMatrix = NULL;
  StiffMat1   = NULL;
  StiffMat2   = NULL;
  StiffMat1T  = NULL;
  StiffMat2T  = NULL;
  xiNodes1    = NULL;
  xiNodes2    = NULL;
  spNodes1    = NULL;
  spNodes2    = NULL;
  Jac         = NULL;
  D1          = NULL;
  D2          = NULL;
  D1T         = NULL;
  D2T         = NULL;

  d11     = NULL;
  d12     = NULL;
  d21     = NULL;
  d22     = NULL;

  g11     = NULL;
  g12     = NULL;
  g22     = NULL;

  gWJ11   = NULL;
  gWJ12   = NULL;
  gWJ22   = NULL;

  xi1p    = NULL;
  xi1m    = NULL;
  xi2p    = NULL;
  xi2m    = NULL;


} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// ARGUMENTS  :
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL DefQuad2D::Resize(GINT  newOrder1, GINT  newOrder2)
{
    GINT  N1, N2, NN;

    // No resizing necessary if alreadyless than
    // previously allocated quantities:
    if ( newOrder1 == Np1 && newOrder2 == Np2 ) return TRUE;

    if ( basis1 == NULL | basis2 == NULL ) return FALSE;


    bInitialized = FALSE;

    //  Resize bases:
    basis1->SetOrder(newOrder1);
    basis2->SetOrder(newOrder2);

    if ( !basis1->Solve() ) return FALSE;
    if ( !basis2->Solve() ) return FALSE;

    Np1 = newOrder1;
    Np2 = newOrder2;

    N1  = Np1 + 1;
    N2  = Np2 + 1;
    NN  = N1 * N2;

    //  Resize spNodes:
    spNodes1     ->Resize(NN);
    spNodes2     ->Resize(NN);

    //  Resize xiNodes:
    xiNodes1     ->Resize(N1);
    xiNodes2     ->Resize(N2);

    //  Resize Weights, MassMatrix StiffMats:
    Weights1     ->Resize(N1);
    Weights2     ->Resize(N2);
    Weights2D    ->Resize(NN);
    MassMatrix   ->Resize(NN);
    gMassMatrix  ->Resize(NN);
    iMassMatrix  ->Resize(NN);
    StiffMat1 ->Resize(N1,N1);
    StiffMat2 ->Resize(N2,N2);
    StiffMat1T->Resize(N1,N1);
    StiffMat2T->Resize(N2,N2);

    //  Resize Derivative matrices:
    D1        ->Resize(N1,N1);
    D1T       ->Resize(N1,N1);

    D2        ->Resize(N2,N2);
    D2T       ->Resize(N2,N2);

    nodal_multiplicity_->Resize(NN);
    *nodal_multiplicity_ = 1.0;

    mask_->Resize(NN);
    *mask_ = 1.0;

    bmask_->Resize(NN);
    *bmask_ = 1.0;

    SetIndices();

    // Resize misc objects:
    delete [] xi1p;
    delete [] xi1m;
    delete [] xi2p;
    delete [] xi2m;
    xi1p = new GDOUBLE [N1];
    xi1m = new GDOUBLE [N1];
    xi2p = new GDOUBLE [N2];
    xi2m = new GDOUBLE [N2];
    if ( xi1p == NULL || xi1m == NULL ||
         xi2p == NULL || xi2m == NULL ) return FALSE;

    bInitialized = TRUE;

    return bInitialized;

}  

//************************************************************************************
//************************************************************************************
// METHOD     : Initialize
// DESCRIPTION: Initialize dynamically allocated quantities
//              Basis is not re-allocated here.
// ARGUMENTS  :
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL DefQuad2D::Initialize(GINT  newOrder1, GINT  newOrder2)
{

    GINT  N1, N2, NN;

    Np1 = newOrder1;
    Np2 = newOrder2;
    
    N1 = Np1;
    N2 = Np2;
    NN = (Np1+1)*(Np2+1);

    //  Resize spNodes:
    spNodes1 = new GVector(NN);
    if ( spNodes1 == NULL ) return FALSE;
    spNodes2 = new GVector(NN);
    if ( spNodes2 == NULL ) return FALSE;

    //  Resize xiNodes:
    xiNodes1 = new GVector(N1);
    if ( xiNodes1 == NULL ) return FALSE;
    xiNodes2 = new GVector(N2);
    if ( xiNodes2 == NULL ) return FALSE;

    //  Resize Weights, MassMatrix StiffMats:
    Weights1 = new GVector(N1);
    if ( Weights1 == NULL ) return FALSE;
    Weights2 = new GVector(N2);
    if ( Weights2 == NULL ) return FALSE;
    Weights2D = new GVector (NN);
    if ( Weights2D == NULL ) return FALSE;
    MassMatrix = new GVector(NN);
    if ( MassMatrix == NULL ) return FALSE;
    gMassMatrix = new GVector(NN);
    if ( gMassMatrix == NULL ) return FALSE;
    iMassMatrix = new GVector(NN);
    if ( iMassMatrix == NULL ) return FALSE;
    StiffMat1  = new GMatrix(N1,N1);
    if ( StiffMat1 == NULL ) return FALSE;
    StiffMat2  = new GMatrix(N2,N2);
    if ( StiffMat2 == NULL ) return FALSE;
    StiffMat1T = new GMatrix(N1,N1);
    if ( StiffMat1T == NULL ) return FALSE;
    StiffMat2T  = new GMatrix(N2,N2);
    if ( StiffMat2T == NULL ) return FALSE;

    //  Resize Derivative matrices:
    D1  = new GMatrix(N1,N1);
    if ( D1== NULL ) return FALSE;
    D1T = new GMatrix(N1,N1);
    if ( D1T== NULL ) return FALSE;

    D2  = new GMatrix(N2,N2);
    if ( D2== NULL ) return FALSE;
    D2T = new GMatrix(N2,Np2);
    if ( D2T== NULL ) return FALSE;

    //  Resize Jacobian matrix:
    Jac = new GVector(NN);
    if ( Jac == NULL ) return FALSE;

    //  Resize dXi_ij/dX quantities:
    d11 = new GVector(NN);
    if ( d11 == NULL ) return FALSE;

    d12 = new GVector(NN);
    if ( d12 == NULL ) return FALSE;

    d21 = new GVector(NN);
    if ( d21 == NULL ) return FALSE;

    d22 = new GVector(NN);
    if ( d22 == NULL ) return FALSE;


    //  Resize metric quantities:
    g11   = new GVector(NN);
    gWJ11 = new GVector(NN);
    if ( g11   == NULL ) return FALSE;
    if ( gWJ11 == NULL ) return FALSE;

    g12   = new GVector(NN);
    gWJ12 = new GVector(NN);
    if ( g12   == NULL ) return FALSE;
    if ( gWJ12 == NULL ) return FALSE;

    g22   = new GVector(NN);
    gWJ22 = new GVector(NN);
    if ( g22   == NULL ) return FALSE;
    if ( gWJ22 == NULL ) return FALSE;

    nodal_multiplicity_ = new GVector (NN);
    if ( nodal_multiplicity_== NULL ) return FALSE;
    *nodal_multiplicity_ = 1.0;

    if ( mask_ == NULL ) return FALSE;
    mask_->Resize(NN);
    *mask_ = 1.0;

    if ( bmask_ == NULL ) return FALSE;
    bmask_->Resize(NN);
    *bmask_ = 1.0;

    SetIndices();

    xi1p = new GDOUBLE [Np1+1];
    xi1m = new GDOUBLE [Np1+1];
    xi2p = new GDOUBLE [Np2+1];
    xi2m = new GDOUBLE [Np2+1];
    if ( xi1p == NULL || xi1m == NULL ||
         xi2p == NULL || xi2m == NULL ) return FALSE;

    return TRUE;
} // end of method Initialize


//************************************************************************************
//************************************************************************************
// METHOD     : XToXi() 
// DESCRIPTION: converts the real space values into parent-domain values
// ARGUMENTS  : Point *x = pointer to real space points;
//              Point *xi= pointer to parent domain points
//              GINT  num= number of points to invert
//
//              NOTE: there will be problems if the number of
//              points in each array is not the same.
// RETURNS    : TRUE on success; FALSE on failure if the 
//              solution is imaginary, or undefined, or if input
//              point lies outside of element.
//************************************************************************************
GBOOL DefQuad2D::XToXi(Point3D x[], Point3D xi[], const GINT  num)
{
  // Check that point is within element boundaries:
  if ( !Point_in_poly(spVertices, 4, x, num) ) 
  {
    cout << " DefQuad2D::XToXi: one or more points lies outside element!" << endl;
    return FALSE;
  }

  if ( !bSolved )
    if ( !SolveFE() ) return FALSE;
  
  GINT  i;
  GDOUBLE a[2], b[2], c[2], d[2], A, B, C, ch;
  GDOUBLE x1, x2, del0, del1, ximin, ximax, xip, xim, xi1tmp, xi2tmp;

  ximin = basis1->GetXimin()<basis2->GetXimin()?basis1->GetXimin():basis2->GetXimin();
  ximax = basis1->GetXimax()>basis2->GetXimax()?basis1->GetXimax():basis2->GetXimax();


  a[0] = spVertices[0].x1 - spVertices[1].x1 + spVertices[2].x1 - spVertices[3].x1; 
  a[1] = spVertices[0].x2 - spVertices[1].x2 + spVertices[2].x2 - spVertices[3].x2; 
  
  b[0] =-spVertices[0].x1 + spVertices[1].x1 + spVertices[2].x1 - spVertices[3].x1; 
  b[1] =-spVertices[0].x2 + spVertices[1].x2 + spVertices[2].x2 - spVertices[3].x2; 

  c[0] =-spVertices[0].x1 - spVertices[1].x1 + spVertices[2].x1 + spVertices[3].x1; 
  c[1] =-spVertices[0].x2 - spVertices[1].x2 + spVertices[2].x2 + spVertices[3].x2; 

  d[0] = spVertices[0].x1 + spVertices[1].x1 + spVertices[2].x1 + spVertices[3].x1; 
  d[1] = spVertices[0].x2 + spVertices[1].x2 + spVertices[2].x2 + spVertices[3].x2; 

  for ( i=0; i<num; i++ )
  {
    x1   = x[i].x1;
    x2   = x[i].x2;
    A    = a[0]*c[1] - a[1]*c[0];
    del0 = 4.0*x1-d[0];
    del1 = 4.0*x2-d[1];
    B    = a[1]*del0 - a[0]*del1 + a[0]*d[1]-a[1]*d[0] + b[0]*c[1]-b[1]*c[0];
    C    =-b[0]*del1 + b[1]*del0;

   // xi_2 is solution to quadratic equation:
   //  A xi^2 + B xi + C = 0:

    if ( A != 0.0 ) // quadratic solution 
    {
      ch   = B*B - 4.0*A*C;

      if ( ch < 0 )
      {
        cout << "DefQuad2D::XToXi(1): xi_2 not real!" << endl;
        return FALSE;
      }

      xip   = ( -B + sqrt(ch) )  / ( 2.0*A);
      xim   = ( -B - sqrt(ch) )  / ( 2.0*A);

      if ( xip > ximax || xip < ximin ) 
        xi2tmp = xim;
      else if ( xim > ximax || xim < ximin ) 
        xi2tmp = xip;
      else 
        xi2tmp = xip;

      if ( xi2tmp > ximax || xi2tmp < ximin ) 
      {
        cout << "DefQuad2D::XToXi(1): xi_2 out of range! xi_2=" << xi2tmp << endl;
        return FALSE;
      }
      xi[i].x2 = xi2tmp;
      xi1tmp = ( 4.0*x1 - c[0]*xi[i].x2 - d[0] ) / ( a[0]*(xi[i].x2) + b[0] );
      if ( xi1tmp > ximax || xi1tmp < ximin ) 
      {
        cout << "DefQuad2D::XToXi(1): xi_1 out of range! (xi_1,xi_2)= (" << 
        xi1tmp << "," << xi2tmp << ")"  << endl;
        return FALSE;
      }
      xi[i].x1 = xi1tmp;
    }
    else  // linear solution (A=0)
    {
      if ( B == 0.0 )
      {
        cout << "DefQuad2D::XToXi(1): xi_2 singular!" << endl;
        return FALSE;
      }

      xi[i].x2 = -C/B;
      xi[i].x1 = ( 4.0*x1 - c[0]*xi[i].x2 - d[0] ) / ( a[0]*(xi[i].x2) + b[0] );
    }
  }
  
  return TRUE;
} // end of method XToXi 


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeGeo
// DESCRIPTION: compputes the Jacobian, and the vector dXi_i/dx_j evaluated at the
//              quadrature points
// ARGUMENTS  :
// RETURNS    : TRUE if Jac >0 at all nodal points, and all else ok; else FALSE
//************************************************************************************
GBOOL DefQuad2D::ComputeGeo()
{
  
  GINT     i, j, k, N1, N2, NN;
  GDOUBLE    zChk   ;
  //GDOUBLE    di11[Np2+1][Np1+1], di12[Np2+1][Np1+1], di21[Np2+1][Np1+1], di22[Np2+1][Np1+1];
  GDOUBLE    **di11, **di12, **di21, **di22;

   N1 = Np1 + 1;
   N2 = Np2 + 1;
   NN = N1 * N2;

   di11 = new GDOUBLE * [N2];
   di12 = new GDOUBLE * [N2];
   di21 = new GDOUBLE * [N2];
   di22 = new GDOUBLE * [N2];
   for ( i=0; i<N2; i++ )
   {
     di11[i] = new GDOUBLE [N1];
     di12[i] = new GDOUBLE [N1];
     di21[i] = new GDOUBLE [N1];
     di22[i] = new GDOUBLE [N1];
   }

  // Compute Jacobian:
  zChk = 1.0;
  for ( i=0; i<N1; i++ )
  {
    xi1p[i] = 1.0 + (*xiNodes1)(i);
    xi1m[i] = 1.0 - (*xiNodes1)(i);
    for ( j=0; j<N2; j++ )
    {
       k = i + j*N1;
       xi2p[j] = 1.0 + (*xiNodes2)(j);
       xi2m[j] = 1.0 - (*xiNodes2)(j);

       di11[j][i] = 0.25*( -spVertices[0].x1 * xi2m[j] + spVertices[1].x1 * xi2m[j]
                    - spVertices[3].x1 * xi2p[j] + spVertices[2].x1 * xi2p[j] );
       di12[j][i] = 0.25*( -spVertices[0].x1 * xi1m[i] - spVertices[1].x1 * xi1p[i]
                    + spVertices[3].x1 * xi1m[i] + spVertices[2].x1 * xi1p[i] );

       di21[j][i] = 0.25*( -spVertices[0].x2 * xi2m[j] + spVertices[1].x2 * xi2m[j]
                    - spVertices[3].x2 * xi2p[j] + spVertices[2].x2 * xi2p[j] );
       di22[j][i] = 0.25*( -spVertices[0].x2 * xi1m[i] - spVertices[1].x2 * xi1p[i]
                    + spVertices[3].x2 * xi1m[i] + spVertices[2].x2 * xi1p[i] );
          
       (*Jac)(k) = di11[j][i]*di22[j][i] - di12[j][i]*di12[j][i];

       zChk *= (*Jac)(k);
      
    }
  }

  // Test Jacobian for being non-zero:
  if ( zChk == 0.0 ) return FALSE;

  j = 1;
  zChk = *(Jac->Data());
  while ((zChk *= (*(Jac->Data()+j)) ) > 0 && j<NN ) j++;

  // Test Jacobian for sign:
  if ( zChk < 0.0 ) return FALSE;

  // Now compute remaining geometric factors, used for
  // computing real space gradients as a function of parent domain
  // (std element) spatial coordinates:
  //... dxi_i/dx_j factors:
  for ( j=0; j<N2; j++ )
  {
    for ( i=0; i<N1; i++ )
    {
       k = i + j*N1;
       (*d11)(k) = di22[j][i]/(*Jac)(k);
       (*d12)(k) =-di12[j][i]/(*Jac)(k);
       (*d12)(k) =-di12[j][i]/(*Jac)(k);
       (*d22)(k) = di11[j][i]/(*Jac)(k);
    }
  }

  // ...metric components: SUM(k) dXi_i/dx_k dXi_j/da_kx * J
  for ( j=0; j<N2; j++ )
  {
    for ( i=0; i<N1; i++ )
    {
       k = i + j*N1;
       (*g11)  (k) = (di11[j][i]*di11[j][i] + di12[j][i]*di12[j][i] ) ;
       (*g12)  (k) = (di11[j][i]*di21[j][i] + di12[j][i]*di22[j][i] ) ;
       (*g22)  (k) = (di21[j][i]*di21[j][i] + di22[j][i]*di22[j][i] ) ;
       (*gWJ11)(k) = (di11[j][i]*di11[j][i] + di12[j][i]*di12[j][i] ) * (*Jac)(k) * (*Weights2D)(k);
       (*gWJ12)(k) = (di11[j][i]*di21[j][i] + di12[j][i]*di22[j][i] ) * (*Jac)(k) * (*Weights2D)(k);
       (*gWJ22)(k) = (di21[j][i]*di21[j][i] + di22[j][i]*di22[j][i] ) * (*Jac)(k) * (*Weights2D)(k);
    }
  }
   for ( i=0; i<N2; i++ )
   {
     delete [] di11[i];
     delete [] di12[i];
     delete [] di21[i];
     delete [] di22[i];
   }
   delete [] di11;
   delete [] di12;
   delete [] di21;
   delete [] di22;

  return TRUE;

} // end of method ComputeGeo

//************************************************************************************
//************************************************************************************
// METHOD     : SolveFE
// DESCRIPTION: computes quadrature points, weights, deriv. matrices 
//              for element. These are computed for the 1d component bases, 
//              and scatter arrays are set up that can determine the
//              V, E, and I parent nodal locations from the components.
// ARGUMENTS  :
// RETURNS    : integer status flag
//************************************************************************************
GINT  DefQuad2D::SolveFE()
{
  GINT  i, j, k;
 
  bSolved = FALSE;

  if ( !bInitialized || basis1 == NULL || basis2 == NULL ) return 0;
   

 
  // Compute quadrature points and corresponding weights for each coordinate
  // basis:
  //if ( !basis1->ComputeNodes () ) return 0;
  //if ( !basis2->ComputeNodes () ) return 0;


  if ( basis1->GetXiNodes(xiNodes1) == NULL ) return 0;
  if ( basis1->GetWeights(Weights1) == NULL ) return 0;
  if ( basis2->GetXiNodes(xiNodes2) == NULL ) return 0;
  if ( basis2->GetWeights(Weights2) == NULL ) return 0;
  if ( basis1->GetDerivMatrix(D1)   == NULL ) return 0;
  if ( basis2->GetDerivMatrix(D2)   == NULL ) return 0;
  D1->Transpose(*D1T);
  D2->Transpose(*D2T);
  if ( basis1->GetStiffMatrix(StiffMat1) == NULL ) return 0;
  if ( basis2->GetStiffMatrix(StiffMat2) == NULL ) return 0;
  StiffMat1->Transpose(*StiffMat1T);
  StiffMat2->Transpose(*StiffMat2T);

  if ( !ComputeGeo() ) return 0; 

  for ( j=0; j<Np2+1; j++ )
  {
    for ( i=0; i<Np1+1; i++ )
    {
      k = i + j*(Np1+1);
      (*MassMatrix)(k) = (*Weights1)(i) * (*Weights2)(j) * (*Jac)(k);
      (*Weights2D) (k) = (*Weights1)(i) * (*Weights2)(j);
    }
  }
  *gMassMatrix = *MassMatrix;

  bSolved = ComputeSpNodes();

  InitMortars();

  return bSolved;

} // end of method SolveFE

//************************************************************************************
//************************************************************************************
// METHOD     : SetInterpBasis
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void DefQuad2D::SetInterpBasis(GNBasis *b1, GNBasis *b2)
{

  if ( b1 == NULL    || b2 == NULL    ) return;  // nothing to do...
  if ( b1 == gbasis1 && b2 == gbasis2 ) return;  // continue if either basis is different
  if ( !bSolved )
    if ( !SolveFE() ) return;                    // ensure that all is ok with this->basis

  gbasis1 = b1;
  gbasis2 = b2;

  // If basis to which to interpolate required 
  // quantities is valid, allocate and compute the 
  // quantities:

  GINT      i, j;
  GBOOL     bValid = TRUE;
  GVector *pxi1, *pxi2, *gW1, *gW2;

  if ( opInterp1  ) delete opInterp1 ;  opInterp1  = NULL;
  if ( opInterp2  ) delete opInterp2 ;  opInterp2  = NULL;
  if ( opInterpT1 ) delete opInterpT1;  opInterpT1 = NULL;
  if ( opInterpT2 ) delete opInterpT2;  opInterpT2 = NULL;
  if ( dInterp1   ) delete dInterp1  ;  dInterp1   = NULL;
  if ( dInterp2   ) delete dInterp2  ;  dInterp2   = NULL;

  // Instantiate new interp. basis quantities:
  opInterp1  = new GMatrix(gbasis1->GetOrder()+1,this->GetOrder(1)+1);
  opInterp2  = new GMatrix(gbasis2->GetOrder()+1,this->GetOrder(2)+1);
  opInterpT1 = new GMatrix(this->GetOrder(1)+1,gbasis1->GetOrder()+1);
  opInterpT2 = new GMatrix(this->GetOrder(2)+1,gbasis2->GetOrder()+1);
  dInterp1   = new GMatrix(gbasis1->GetOrder()+1,this->GetOrder(1)+1);
  dInterp2   = new GMatrix(gbasis2->GetOrder()+1,this->GetOrder(2)+1);
  dInterpT1  = new GMatrix(this->GetOrder(1)+1,gbasis1->GetOrder()+1);
  dInterpT2  = new GMatrix(this->GetOrder(2)+1,gbasis2->GetOrder()+1);
  pxi1 = new GVector(gbasis1->GetOrder()+1);
  pxi2 = new GVector(gbasis2->GetOrder()+1);
  gW1  = new GVector(gbasis1->GetOrder()+1);
  gW2  = new GVector(gbasis2->GetOrder()+1);
  gbasis1->GetXiNodes(pxi1);
  gbasis2->GetXiNodes(pxi2);
  gbasis1->GetWeights(gW1);
  gbasis2->GetWeights(gW2);
 
  if ( basis1->EvalBasis (pxi1,opInterp1) == NULL ) bValid = FALSE;
  if ( basis2->EvalBasis (pxi2,opInterp2) == NULL ) bValid = FALSE;
  if ( basis1->EvalDBasis(pxi1,dInterp1)  == NULL ) bValid = FALSE;
  if ( basis2->EvalDBasis(pxi2,dInterp2)  == NULL ) bValid = FALSE;
  

  if ( bValid )
  {
  // Interp and derivative operators include the weights:
    for ( j=0; j<opInterp1->dim(2); j++ )
    {
      for ( i=0; i<opInterp1->dim(1); i++ )
      {
        (*opInterp1)(i,j) *= (*gW1)(i);
        (*dInterp1) (i,j) *= (*gW1)(i);
      }
    }
    for ( j=0; j<opInterp2->dim(2); j++ )
    {
      for ( i=0; i<opInterp2->dim(1); i++ )
      {
        (*opInterp2)(i,j) *= (*gW2)(i);
        (*dInterp2) (i,j) *= (*gW2)(i);
      }
    }
  }

  if ( !opInterp1->Transpose(*opInterpT1) 
    || !opInterp2->Transpose(*opInterpT2) 
    || !dInterp1 ->Transpose (*dInterpT1) 
    || !dInterp2 ->Transpose (*dInterpT2) ) bValid == FALSE;

  // if any operation fails, destroy all interp. quantities:
  delete pxi1;
  delete pxi2;
  delete gW1;
  delete gW2;
  if ( !bValid )
  {
    if ( opInterp1  != NULL ) delete opInterp1;  opInterp1  = NULL;
    if ( opInterp2  != NULL ) delete opInterp2;  opInterp2  = NULL;
    if ( opInterpT1 != NULL ) delete opInterpT1; opInterpT1 = NULL;
    if ( opInterpT2 != NULL ) delete opInterpT2; opInterpT2 = NULL;
    if ( dInterp1   != NULL ) delete dInterp1;   dInterp1   = NULL;
    if ( dInterp2   != NULL ) delete dInterp2;   dInterp2   = NULL;
    if ( dInterpT1  != NULL ) delete dInterpT1;  dInterpT1  = NULL;
    if ( dInterpT2  != NULL ) delete dInterpT2;  dInterpT2  = NULL;
    return;
  }
  
} // end of method SetInterpBasis


//************************************************************************************
//************************************************************************************
// METHOD     : GetInterpOp
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GMatrix *DefQuad2D::GetInterpOp(GINT  idir, GBOOL bTrans)
{
  
  if      ( idir == 1 )
  {
    if ( !bTrans ) return opInterp1;
    else           return opInterpT1; 
  }
  else if ( idir == 2 )
  {
    if ( !bTrans ) return opInterp2;
    else           return opInterpT2;
  }
  else
    return NULL;

} // end of method GetInterpOp


//************************************************************************************
//************************************************************************************
// METHOD     : GetInterpDeriv
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GMatrix *DefQuad2D::GetInterpDeriv(GINT  idir, GBOOL bTrans)
{

  if      ( idir == 1 )
  {
    if ( !bTrans ) return dInterp1;
    else           return dInterpT1;
  }
  else if ( idir == 2 )
  {
    if ( !bTrans ) return dInterp2;
    else           return dInterpT2;
  }
  else
    return NULL;

} // end of method GetInterpDeriv


//************************************************************************************
//************************************************************************************
// METHOD     : Integrate
// DESCRIPTION: Computes integral over element of input vector.
//              s.t.
//                int  = Int ( v^2) dx dy
//              Note: this could be placed in the base class, if MassMatrix is
//                    also included.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE DefQuad2D::Integrate(GVector *v, GDOUBLE *imultiplicity)
{

  GINT  j;
  GDOUBLE sum;

  if ( !bSolved )
    if ( !SolveFE() ) return -1.0;

  if ( v == NULL ) return -1.0;

  if ( imultiplicity == NULL ) {
    sum = MTK::fvec_dot(*v, *MassMatrix);
  }
  else {
    for ( j=0, sum=0.0; j<v->dim(); j++ ) {
      sum += (*v)(j)*(*MassMatrix)(j)*imultiplicity[j];
    }
  }

  return sum;

} // end of method Integrate

//************************************************************************************
//************************************************************************************
// METHOD     : PIntegrate
// DESCRIPTION: Computes integral of input vector over 'parent' element subdomain
//              s.t.
//                int  = Int_{-1}^{1}   v  dxi_1 dxi_2
//              Note: this could be placed in the base class, if Weights2D is
//                    also included.
// ARGUMENTS  : v            : integrand as a GVector
//              imultiplicity: node multiplicity; used if non-NULL. Must be of the same
//                            length as v.
// RETURNS    :
//************************************************************************************
GDOUBLE DefQuad2D::PIntegrate(GVector *v, GDOUBLE *imultiplicity)
{

  GINT  j;
  GDOUBLE sum;

  if ( !bSolved &&  !SolveFE() ) {
    cout << "DefQuad2D::PIntegrate: Element solve failed" << endl;
    exit(1);
  }

  if ( v == NULL ) {
    cout << "DefQuad2D::PIntegrate: NULL integrand" << endl;
    exit(1);
  }

  if ( imultiplicity == NULL ) {
    sum = MTK::fvec_dot(*v, *Weights2D);
  }
  else {
    for ( j=0, sum=0.0; j<v->dim(); j++ ) {
      sum += (*v)(j)*(*Weights2D)(j)*imultiplicity[j];
    }
  }
 
  return sum;

} // end of method PIntegrate


//************************************************************************************
//************************************************************************************
// METHOD     : Differentiate
// DESCRIPTION: Computes derivative of input vector in direction idir,
//              overwriting input vector with the result
// ARGUMENTS  : v   : argument as a GVector
//              idir: coordinate direction
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL DefQuad2D::Differentiate(GVector *dv, GVector *v, GINT  idir)
{

  // For now,
  cout << "DefQuad2D::Differentiate: method out of service!" << endl;
  exit(1);

#if 0
  GINT    NN, N1, N2;
  GDOUBLE   L1, L2, iL1, iL2;
  GVector vtmp;
  GMatrix *K1, *K2T;

  K1   = StiffMat1; 
  K2T  = StiffMat2T;
    
  L1   = PDISTANCE(spVertices[1],spVertices[0]);
  L2   = PDISTANCE(spVertices[2],spVertices[1]);
  if ( L1 == 0 || L2 == 0 ) return FALSE;
    
  iL1 = 1.0/L1;
  iL2 = 1.0/L2;

  N1 = Np1 + 1;
  N2 = Np2 + 1;
  NN = N1* N2;
  if ( v->dim() != NN ) return FALSE;
  vtmp.Resize(NN);
  vtmp = *v; 
    
  if ( idir == 1 ) {
    // Do I2 X K1 term: 
    *v = 0.0;
    MTK::I2_X_D1(*K1, vtmp, N1, N2, *v);
    if ( iL1 != 1.0 ) MTK::fvec_const_prod_rep(*v, iL1);
  }
  else if ( idir == 2 ) {
    // Do K2 X I1 term: 
    MTK::D2_X_I1(*K2T, vtmp, N1, N2, *v);
    if ( iL2 != 1.0 ) MTK::fvec_const_prod_rep(*v, iL2);
  }
#endif

  return TRUE;

} // end of method Differentiate

// To be implemented later
GBOOL  DefQuad2D::DifferentiateWithMass(GVector *dv, GVector *v, GVector* tmp, GINT  idir)
{return FALSE;};
GBOOL  DefQuad2D::DifferentiateWeak(GVector *dv, GVector *v, GVector* tmp, GINT  idir)
{return FALSE;};

