//************************************************************************************//
// Module       : diagop.hpp
// Date         : 6/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the Diag operator
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "diagop.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
//************************************************************************************
DiagOp::DiagOp()
: LinOp(),
bInverse    (FALSE),
n           (0),
diag        (NULL)
{
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
//************************************************************************************
DiagOp::DiagOp(GINT  len)
: LinOp(),
bInverse    (FALSE),
n           (0),
diag        (NULL)
{
  n = len;
  diag  = new GVector(n);

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
//************************************************************************************
DiagOp::DiagOp(GDBuffer  *d)
: LinOp(),
bInverse    (FALSE),
n           (0),
diag        (NULL)
{
  GINT   i;
  GDOUBLE *px;
  if ( d == NULL )
  {
    cout << " DiagOp::DiagOp(3): NULL input " << endl;
    exit(1);
  }
  n = d->dim();
  diag  = new GVector(d->Data(),n);
  if ( bInverse )
  {
     px  = diag ->Data();
     for ( i=0; i<n; i++ ) {
       px [i] = 1.0/px [i];
     }
  }

} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Constructor Method (4)
//************************************************************************************
DiagOp::DiagOp(GVector  *d)
: LinOp(),
bInverse    (FALSE),
n           (0),
diag        (NULL)
{
  
  GINT    i;
  GDOUBLE *px;

  if ( d == NULL )
  {
    cout << " DiagOp::DiagOp(4): NULL input " << endl;
    exit(1);
  }
  n = d->dim();
  diag  = new GVector(d->Data(),n);
  if ( bInverse )
  {
     px  = diag ->Data();
     for ( i=0; i<n; i++ ) {
       px [i] = 1.0/px [i];
     }
  }

} // end of constructor method (4)


//************************************************************************************
//************************************************************************************
// Constructor Method (5)
//************************************************************************************
DiagOp::DiagOp(GDOUBLE *d, GINT  nd)
: LinOp(),  
bInverse    (FALSE),
n           (nd),
diag        (NULL)
{
  GINT    i;
  GDOUBLE *px;
  
  if ( d == NULL || nd <= 0  )
  {
    cout << " DiagOp::DiagOp(5): NULL input " << endl;
    exit(1);
  }
  diag  = new GVector(d,n);
  if ( bInverse )
  {
     px  = diag ->Data();
     for ( i=0; i<n; i++ ) {
       px [i] = 1.0/px [i];
     }
  }

} // end of constructor method (5)


//************************************************************************************
//************************************************************************************
// Constructor Method (6)
//************************************************************************************
DiagOp::DiagOp(Elem2D *e)
: LinOp(),
bInverse    (FALSE),
n           (0),
diag        (NULL)
{
  if ( e == NULL )
  {
    cout << " DiagOp::DiagOp(6): NULL input " << endl;
    exit(1);
  }
  n = (e->GetOrder(1)+1) * ( e->GetOrder(2) + 1);
  diag  = new GVector(n);

} // end of constructor method (6)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
DiagOp::~DiagOp()
{
  if ( diag  != NULL ) delete diag;
  diag  = NULL;
}


//************************************************************************************
//************************************************************************************
// METHOD     : Inverse
// DESCRIPTION: Toggeles whether inverse of operator should be applied;
//              NOTE: no 0-checking is done here
// RETURNS    :  

//************************************************************************************
void  DiagOp::Inverse()
{
  GINT  i;
  GDOUBLE *px;

  if ( diag == NULL ) {
    cout << "DiagOp::Inverse: NULL diag data invalid" << endl;
    exit(1);
  }
  bInverse = !bInverse;
  if ( bInverse )
  {
     px = diag->Data();
     for ( i=0; i<n; i++ )
       px[i] = 1.0/(px[i]+TINY);      
  }
} // end of method Inverse


//************************************************************************************
//************************************************************************************
// METHOD     : SetDiag 
// DESCRIPTION: Sets diagonal operator elements
// RETURNS    :  
//************************************************************************************
void  DiagOp::SetDiag(GDOUBLE  *d, GINT  nn)
{
  GINT    i;
  GDOUBLE *px;

  if ( nn != n )
  {
    n     = nn;
    if ( diag  != NULL ) delete diag;
    diag  = new GVector (d, n);
  }
  if ( diag != NULL && d != NULL )
  {
    memcpy(diag ->Data(), d, n*sizeof(GDOUBLE));
  }
  if ( bInverse )
  {
     px = diag->Data();
     for ( i=0; i<n; i++ )
       px[i] = 1.0/px[i];      
  }
  if ( mask ) {
    MTK::fvec_point_prod_rep(*diag, *mask);
  }

} // end of method SetDiag


//************************************************************************************
//************************************************************************************
// METHOD     : SetMask
// DESCRIPTION: Sets mask
// RETURNS    :  
//************************************************************************************
void  DiagOp::SetMask(GVector *m)
{
  mask = m;
  if ( diag && mask ) {
    MTK::fvec_point_prod_rep(*diag, *mask);
  } 

} // end of method SetMask
  

//************************************************************************************
//************************************************************************************
// METHOD     : GetDiag
// DESCRIPTION: Gets vector diagonal
// RETURNS    :
//************************************************************************************
GVector *DiagOp::GetDiag()
{
  return diag;
} // end of method GetDiag


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
//              NOTE: No 0-checking is performed for the inverse operation
// RETURNS    :
//************************************************************************************
GVector  DiagOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector ret(sei);

  OpVec_prod(q, ret);
  return ret;

} //end of operator \*


//************************************************************************************
//************************************************************************************
// METHOD     : OpVec_prod
// DESCRIPTION: Multiplies operator by right hand vector, x, returning
//              result in ret.
// ARGUMENTS  : x  : RHS vector
//              y  : return vector
// RETURNS    :  none
//************************************************************************************
void DiagOp::OpVec_prod(GVector &x, GVector &y)  
{
  if ( diag != NULL && n != x.dim() )
  {
    cout << "DiagOp::OpVec_prod: incompatible vector" << endl;
    exit(1);
  }

  if ( diag == NULL )
  {
    y = x;
    return;
  }

  MTK::fvec_point_prod(x, *diag, y);
} // end of method oerator OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: Resizes member data
// ARGUMENTS  : 
// RETURNS    :  none
//************************************************************************************
void DiagOp::Resize(GINT  NN)  
{
  if ( diag  != NULL ) diag ->Resize(NN); 
} // end of method oerator Resize

