//************************************************************************************//
// Module       : elem2d.hpp
// Date         : 9/14/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a single 2D spectral element
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "elem2d.hpp"
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <stdio.h>


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
Elem2D::Elem2D(GSHORT ntmp)
:
Np1              (0),
Np2              (0),
bInitialized     (FALSE),
bSolved          (FALSE),
elemtype_        (RECT_QUAD),
elemid_          (0),
rootid_          (0),
irootid_         (0.0),
nVertices_        ((GINT )pow(2.0,GDIM)),
nEdges_          (2*(GDIM-1)*GDIM),
nFaces_          (0),
spVertices       (NULL),
elemCenter       (NULL),
spMidpoints      (NULL),
spvVertices      (NULL),
spNodes1         (NULL),
spNodes2         (NULL),
basis1           (NULL),
basis2           (NULL),
gbasis1          (NULL),
gbasis2          (NULL),
iep2v_           (NULL),
edge_mortars_    (NULL),
nodal_multiplicity_(NULL),
bdy_indices_     (NULL),
bdy_types_       (NULL),
vert_types_      (NULL),
vert_indices_    (NULL),
edge_types_      (NULL),
face_types_      (NULL),
edge_indices_    (NULL),
edge_indicesn_   (NULL),
inter_indices_   (NULL),
bGlobal_edge_    (NULL),
#if defined(IS3D)
face_mortar_     (NULL),
face_neighbor_   (NULL),
face_types_      (NULL),
#endif
vertex_neighbor_ (NULL),
vvertex_neighbor_(NULL),
edge_neighbor_   (NULL),
bmask_           (NULL),
mask_            (NULL),
area_            (0.0) ,
vtmpmgr_         (NULL)
{
  GINT  i;

  for ( i=0; i<GDIM; i++ ) dealias_basis_[i] = NULL;

  iep2v_ = new GINT  * [4];
  for ( i=0; i<4; i++ ) iep2v_[i] = new GINT  [2];
  iep2v_[0][0] = 0;  iep2v_[0][1] = 1;
  iep2v_[1][0] = 1;  iep2v_[1][1] = 2;
  iep2v_[2][0] = 3;  iep2v_[2][1] = 2;
  iep2v_[3][0] = 0;  iep2v_[3][1] = 3;
  vtmpmgr_ = new GMemMgr (ntmp);
 
} // end of constructor method


//************************************************************************************
//************************************************************************************
// Desstructor Method (1)
Elem2D::~Elem2D()
{
  GINT  i;

  if ( spVertices  != NULL ) delete [] spVertices;
  if ( spvVertices != NULL ) delete [] spvVertices;
  if ( spMidpoints != NULL ) delete [] spMidpoints;
  if ( elemCenter  != NULL ) delete    elemCenter;

  if ( edge_mortars_    != NULL ) delete [] edge_mortars_;
  if ( edge_neighbor_   != NULL ) delete [] edge_neighbor_;
  if ( vertex_neighbor_ != NULL ) delete [] vertex_neighbor_;
  if ( vvertex_neighbor_!= NULL ) delete [] vvertex_neighbor_;
  if ( vert_indices_    != NULL ) delete [] vert_indices_;
  if ( edge_indices_    != NULL ) delete [] edge_indices_;
  if ( edge_indicesn_   != NULL ) delete [] edge_indicesn_;
  if ( inter_indices_   != NULL ) delete    inter_indices_;
#if defined(IS3D)
  if ( face_mortar_   != NULL ) delete [] face_mortar_;
  if ( face_neighbor_ != NULL ) delete [] face_neighbor_;
  if ( face_indices_  != NULL ) delete [] face_indices_;
  if ( face_types_    != NULL ) delete face_types_;
#endif
  if ( nodal_multiplicity_ != NULL ) delete nodal_multiplicity_;
  if ( bdy_indices_        != NULL ) delete bdy_indices_;
  if ( bdy_types_          != NULL ) delete bdy_types_;
  if ( vert_types_         != NULL ) delete vert_types_;
  if ( edge_types_         != NULL ) delete edge_types_;
  if ( face_types_         != NULL ) delete face_types_;
  if ( bGlobal_edge_       != NULL ) delete bGlobal_edge_;
  if ( mask_               != NULL ) delete mask_;
  if ( bmask_              != NULL ) delete bmask_;
  if ( vtmpmgr_            != NULL ) delete vtmpmgr_; 

  for ( i=0; i<4; i++ ) delete [] iep2v_[i];
  delete [] iep2v_;

} // end of destructor method


//************************************************************************************
//************************************************************************************
// METHOD     : GetSpVertices 
// DESCRIPTION: Gets element's vertices
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
Point3D *Elem2D::GetSpVertices(GINT  iv)
{
  if ( iv < 0 ) return spVertices;
  if ( iv >= nVertices_ ) return NULL;

  return &spVertices[iv];
} // end of method GetSpVertices (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetSpMidpoints
// DESCRIPTION: Gets element's edge midpoints in Point3D format 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
Point3D *Elem2D::GetSpMidpoints(GINT  ie)
{
  if ( ie < 0 ) return spMidpoints;
  if ( ie >= nEdges_ ) return NULL;
  return  &spMidpoints[ie];
} // end of method GetSpMidpoints


//************************************************************************************
//************************************************************************************
// METHOD     : GetElemCenter
// DESCRIPTION: Gets element's center point
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
Point3D *Elem2D::GetElemCenter()
{
  return elemCenter;
} // end of method GetElemCenter


//************************************************************************************
//************************************************************************************
// METHOD     : GetSpvVertices 
// DESCRIPTION: Gets element's vertices in GDOUBLE vector format
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE *Elem2D::GetSpvVertices()
{
  return spvVertices;
} // end of method GetSpvVertices


//************************************************************************************
//************************************************************************************
// METHOD     : GetID
// DESCRIPTION: Gets element's id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GKEY Elem2D::GetID()
{
  return elemid_;
} // end of method GetID


//************************************************************************************
//************************************************************************************
// METHOD     : GetPArentID
// DESCRIPTION: Gets element's parent's  id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GKEY Elem2D::GetParentID()
{
  return parentid_;
} // end of method GetParentID


//************************************************************************************
//************************************************************************************
// METHOD     : GetRootID
// DESCRIPTION: Gets element's root id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GKEY Elem2D::GetRootID()
{
  return rootid_;
} // end of method GetRootID


//************************************************************************************
//************************************************************************************
// METHOD     : GetiRootID
// DESCRIPTION: Gets inverse of element's root id 
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GDOUBLE Elem2D::GetiRootID()
{
  return irootid_;
} // end of method GetRootiID


//************************************************************************************
//************************************************************************************
// METHOD     : SetID
// DESCRIPTION: Sets element's id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::SetID(GKEY id)
{
  elemid_  = id;
} // end of method SetID


//************************************************************************************
//************************************************************************************
// METHOD     : SetPArentID
// DESCRIPTION: Sets element's parent's id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::SetParentID(GKEY id)
{
  parentid_  = id;
} // end of method SetParentID


//************************************************************************************
//************************************************************************************
// METHOD     : SetRootID
// DESCRIPTION: Sets element's root id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::SetRootID(GKEY id)
{
  rootid_  = id;
  irootid_ = id == 0 ? 0 : 1.0 / ( (GDOUBLE)id );
} // end of method SetRootID


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumVertices
// DESCRIPTION: Gets number of element vertices
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  Elem2D::GetNumVertices()
{
  return nVertices_;
} // end of method GetNumVertices

//************************************************************************************
//************************************************************************************
// METHOD     : GetVertexIndices
// DESCRIPTION: Gets vertex indices corresp. to specified vertex index
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GIBuffer  *Elem2D::GetVertexIndices(GINT  ivert)
{
  if ( ivert >= nVertices_ || vert_indices_ == NULL ) return NULL;
  if ( ivert < 0 ) return vert_indices_;

  return &vert_indices_[ivert];
} // end of method GetVertexIndices


//************************************************************************************
//************************************************************************************
// METHOD     : Edge2Vertex
// DESCRIPTION: converts edge id to vertex ids
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::Edge2Vertex(GINT  iv[], const GINT  ie)
{

  if ( ie < 0 || ie >= nEdges_  ) {
    cout << "Elem2D::Edge2Vertex: invalid index: iedge="
         << ie << endl;
    exit(1);
  }
  iv[0] = iep2v_[ie][0];
  iv[1] = iep2v_[ie][1];

} // end of method Edge2Vertex


//************************************************************************************
//************************************************************************************
// METHOD     : EdgePoint2Vertex
// DESCRIPTION: converts edge (mortar)+ point ids to vertex id
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  Elem2D::EdgePoint2Vertex(const GINT  ie, const GINT  ip)
{

  if ( ie < 0 || ie >= nEdges_  || ip < 0 || ip > 1 ) {
    cout << "Elem2D::EdgePoint2Vertex: invalid index: iedge="
         << ie << " ipoint=" << ip << endl;
    exit(1);
  }

  return ( iep2v_[ie][ip] );

} // end of method EdgePoint2Vertex


//************************************************************************************
//************************************************************************************
// METHOD     : Vertex2EdgePoint
// DESCRIPTION: converts vertex id to edge (mortar)+ point ids
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::Vertex2EdgePoint(GINT  iedge[], GINT  ipoint[], const GINT  ivert)
{
  GINT  i, j, n;

  if ( ivert < 0 || ivert >= nVertices_ ) {
    cout << "Elem2D::Vertex2EdgePoint: invalid index: ivert=" << ivert << endl;
    exit(1);
  }
 
  for ( i=0,n=0; i<4 && n<=2; i++ ) {
    for ( j=0; j<2 && n<=2; j++ ) { 
      if ( iep2v_[i][j] == ivert ) {
        iedge [n] = i;
        ipoint[n] = j;
        n++;
      }
    }
  }

} // end of method Vertex2EdgePoint


//************************************************************************************
//************************************************************************************
// METHOD     : isVirtualVertex
// DESCRIPTION: establishes if vertex specified by edge(mortar) id and endpoint index
//              is virtual (meaning that the mortar doesn't line up with the edge
//              node-by-node).
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL Elem2D::isVirtualVertex(GINT  ie, GINT  ip)
{
  GINT    nm;
  Point   *mendpt, *vert=spVertices;

  if ( ie < 0 || ie >= nEdges_  || ip < 0 || ip > 1 ) {
    cout << "Elem2D::isVirtualVertex: invalid index: iedge=" 
         << ie << " ipoint=" << ip << endl;
  }

  mendpt  = edge_mortars_[ie].GetBdyPoints(nm);

  // Find if mortar endpoint is virtual.
  // Note that iep2v_ is a mapping from edge id and endpoint to 
  // vertex id; i.e., iep2v_[iedge][iendpoint] = corresponding real 
  // vertex id
#if 0
  if ( !edge_mortars_[ie].isConforming() ) {
    cout << "Elem2D::isVirtualVertex: bdy_pnt[" << ie << "][" << ip << "]=" << mendpt[ip] << " vert[" << 
    iep2v_[ie][ip] << "]=" << vert[iep2v_[ie][ip]] << " isVirtual=" << (mendpt[ip] != vert[iep2v_[ie][ip]]) << endl;
  }
#endif
  return ( !edge_mortars_[ie].isConforming() && mendpt[ip] != vert[iep2v_[ie][ip]] );

} // end of method isVirtualVertex


//************************************************************************************
//************************************************************************************
// METHOD     : GetCommonVertex
// DESCRIPTION: Gets edge id and vertex id on the edge that have a vertex in 
//              common with specified edge/vertex pair
// ARGUMENTS  : ice    : contains common edge id (returned)
//              icp    : contains endpoint id on ice (returned)
//              ie     : input edge
//              ip     : input endpoint id on ie edge
// RETURNS    :
//************************************************************************************
GBOOL Elem2D::GetCommonVertex(GINT  &ice, GINT  &icp, const GINT  ie, const GINT  ip)
{ 
  GINT    icev[4][2]={{3,1},{0,2},{3,1},{0,2}};  
  GINT    icpv[4][2]={{0,0},{1,1},{1,1},{0,0}};

  if ( ie < 0 || ie >= nEdges_  || ip < 0 || ip > 1 ) {
    cout << "Elem2D::GetCommonVertex: invalid index: iedge="
         << ie << " ipoint=" << ip << endl;
  }

  // Given ie, ip, icev[ie][ip] gives the common edge, and
  //               icpv[ie][ip] gives the vertex id on that edge 
  ice = icev[ie][ip];  icp = icpv[ie][ip];

  return TRUE;

} // end of method GetCommonVertex


//************************************************************************************
//************************************************************************************
// METHOD     : GetVertexCommonEdges
// DESCRIPTION: Gets edge id's and vertex id's for the specified real vertex
// ARGUMENTS  : ice    : contains common edge id (returned, of length 2)
//              icp    : contains endpoint id on ice (returned, of length 2)
//              ie     : input vertex
// RETURNS    :
//************************************************************************************
void Elem2D::GetVertexCommonEdges(GINT  ice[], GINT  icp[], const GINT  iv)
{
  GINT    ip, jp, ie=iv, je;

  if ( iv < 0 || iv >= nVertices_  ) {
    cout << "Elem2D::GetVertexCommonEdges: invalid index: vert=" << iv << endl;
    exit(1);
  }
  je = (ie + nVertices_ - 1 ) % nVertices_;
  // Note that iep2v_ is a mapping from edge id and endpoint to 
  // vertex id; i.e., iv[iedge][iendpoint] = corresponding real 
  // vertex id
  ip = jp = 0;
  while ( iv != iep2v_[ie][ip] && ip < 2 ) ip++;
  while ( iv != iep2v_[je][jp] && jp < 2 ) jp++;
  if ( ip > 1 || jp > 1 ) {
    cout << "Elem2D::GetVertexCommonEdges: invalid vertex id" << endl;
    exit(1);
  }
  ice[0] = ie;  icp[0] = ip;
  ice[1] = je;  icp[1] = jp;

} // end of method GetVertexCommonEdges


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumEdges
// DESCRIPTION: Gets number of element edges
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  Elem2D::GetNumEdges()
{
  return nEdges_;
} // end of method GetNumEdges 


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumFaces
// DESCRIPTION: Gets number of element faces
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  Elem2D::GetNumFaces()
{
  return nFaces_;
} // end of method GetNumFaces


#if defined(IS3D)
//************************************************************************************
//************************************************************************************
// METHOD     : GetNumFaces
// DESCRIPTION: Gets number of element faces
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GINT  Elem2D::GetNumFaces()
{
  return nFaces_;
} // end of method GetNumFaces


//************************************************************************************
//************************************************************************************
// METHOD     : GetFaceMortar
// DESCRIPTION: Gets face mortar corresp. to specified face index
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GMortar1D *Elem2D::GetFaceMortar(GINT  iface)
{
  if ( iedge >= nFaces_ || face_mortar_ == NULL ) return NULL;
  if ( iface < 0 ) return face_mortar_;

  return &face_mortar_[iface];
} // end of method GetFaceMortar


//************************************************************************************
//************************************************************************************
// METHOD     : GetFaceIndices
// DESCRIPTION: Gets face indices
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GIBuffer  *Elem2D::GetFaceIndices(GINT  iface)
{
  if ( iedge >= nFaces_ || face_indices_ == NULL ) return NULL;
  if ( iface < 0 ) return face_indices_;

  return &face_indices_[iface];
} // end of method GetFaceIndices


//************************************************************************************
//************************************************************************************
// METHOD     : GetFaceType
// DESCRIPTION: Gets reference to face bdy type
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
BDYTYPE &Elem2D::GetFaceType(GINT  iface)
{
  return face_types_[iface];
} // end of method GetFaceType
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : GetEdgeMortar
// DESCRIPTION: Gets edge mortar corresp. to specified edge index
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GMortar1D *Elem2D::GetEdgeMortar(GINT  iedge)
{
  if ( iedge >= nEdges_ || edge_mortars_ == NULL ) return NULL;
  if ( iedge < 0 ) return edge_mortars_;

  return &edge_mortars_[iedge];
} // end of method GetEdgeMortar


//************************************************************************************
//************************************************************************************
// METHOD     : GetEdgeIndices
// DESCRIPTION: Gets edge indices corresp. to specified edge index
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GIBuffer  *Elem2D::GetEdgeIndices(GINT  iedge)
{
  if ( iedge >= nEdges_ || edge_indices_ == NULL ) return NULL;
  if ( iedge < 0 ) return edge_indices_;
  return &edge_indices_[iedge];
} // end of method GetEdgeIndices


//************************************************************************************
//************************************************************************************
// METHOD     : GetEdgeIndicesN
// DESCRIPTION: Gets edge indices corresp. to specified edge index, but
//              not including endpoints
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GIBuffer  *Elem2D::GetEdgeIndicesN(GINT  iedge)
{
  if ( iedge >= nEdges_ || edge_indicesn_ == NULL ) return NULL;
  if ( iedge < 0 ) return edge_indicesn_;
  return &edge_indicesn_[iedge];
} // end of method GetEdgeIndicesN


//************************************************************************************
//************************************************************************************
// METHOD     : GetInteriorIndices
// DESCRIPTION: Gets interior indices for element
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GIBuffer  *Elem2D::GetInteriorIndices()
{
  return inter_indices_;
} // end of method GetInteriorIndices


//************************************************************************************
//************************************************************************************
// METHOD     : GetVNeighbor
// DESCRIPTION: Gets vertex neighbor list
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GNeighborList *Elem2D::GetVNeighbor(GINT  ivertex)
{
  if ( ivertex >= nVertices_ || vertex_neighbor_ == NULL ) return NULL;
  if ( ivertex < 0 ) return vertex_neighbor_;

  return &vertex_neighbor_[ivertex];
} // end of method GetVNeighbor


//************************************************************************************
//************************************************************************************
// METHOD     : GetVVNeighbor
// DESCRIPTION: Gets virtual vertex neighbor list
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GNeighborList *Elem2D::GetVVNeighbor(GINT  ivertex)
{
  if ( ivertex >= nVertices_ || vvertex_neighbor_ == NULL ) return NULL;
  if ( ivertex < 0 ) return vvertex_neighbor_;

  return &vvertex_neighbor_[ivertex];
} // end of method GetVVNeighbor


//************************************************************************************
//************************************************************************************
// METHOD     : GetENeighbor
// DESCRIPTION: Gets edge neighbor list
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GNeighborList *Elem2D::GetENeighbor(GINT  iedge)
{
  if ( iedge >= nEdges_ || edge_neighbor_ == NULL ) return NULL;
  if ( iedge < 0 ) return edge_neighbor_;

  return &edge_neighbor_[iedge];
} // end of method GetENeighbor


//************************************************************************************
//************************************************************************************
// METHOD     : GetBdyIndices
// DESCRIPTION: Gets bdy indices corresponding to specified edge. BCs are
//              specified none-by-node, and a given edge may have no BCs,
//              so this is differentiated from the edge_indices_ quantitiy, 
//              which always exist. If no argument is specified, the whole array
//              is given.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GIBuffer  *Elem2D::GetBdyIndices()
{
  return bdy_indices_;
} // end of method GetBdyIndices


//************************************************************************************
//************************************************************************************
// METHOD     : GetBdyTypes
// DESCRIPTION: Gets bdy type array--giving bdy types for each of the indices in the
//              bdy_indices_ buffer.
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBTBuffer  *Elem2D::GetBdyTypes()
{
  return bdy_types_;
} // end of method GetBdyTypes


//************************************************************************************
//************************************************************************************
// METHOD     : GetVertType
// DESCRIPTION: Gets bdy type reference for the specified vertex
// ARGUMENTS  : vertex index
// RETURNS    :
//************************************************************************************
BDYTYPE &Elem2D::GetVertType(GINT  i)
{
  return *(vert_types_->Data()+i);
} // end of method GetVertType


//************************************************************************************
//************************************************************************************
// METHOD     : GetEdgeType
// DESCRIPTION: Gets bdy type reference for the specified edge
// ARGUMENTS  : edge index
// RETURNS    :
//************************************************************************************
BDYTYPE &Elem2D::GetEdgeType(GINT  i)
{
  return *(edge_types_->Data()+i);
} // end of method GetEdgeType


//************************************************************************************
//************************************************************************************
// METHOD     : GetFaceType
// DESCRIPTION: Gets bdy type reference for the specified face
// ARGUMENTS  : face index
// RETURNS    :
//************************************************************************************
BDYTYPE &Elem2D::GetFaceType(GINT  i)
{
  return *(face_types_->Data()+i);
} // end of method GetFaceType


//************************************************************************************
//************************************************************************************
// METHOD     : bGlobalBdyEdge
// DESCRIPTION: Gets flag indicating whether edge is a global bdy
// ARGUMENTS  : edge index
// RETURNS    :
//************************************************************************************
GINT  &Elem2D::bGlobalBdyEdge(GINT  i)
{
  return (*(bGlobal_edge_->Data()+i));
} // end of method bGlobalBdyEdge


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeBdyInfo
// DESCRIPTION: Computes bdy data from 'Set' data
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void Elem2D::ComputeBdyInfo()
{
  GINT       j, m, nb, ind;
  GIBuffer   bi;
  GBTBuffer  bt;
  
  for ( j=0,nb=0; j<nEdges_; j++ ) {
//  nb = (*bGlobal_edge_)(j) ? nb+GetOrder(j%GDIM+1)+1 : nb;
    nb = (*bGlobal_edge_)(j) ? nb+edge_indices_[j].dim() : nb;
  }
  bi.Resize(nb);
  bt.Resize(nb);
  bi = -1;
  for ( j=0,nb=0; j<nEdges_; j++ ) {
    if ( (*bGlobal_edge_)(j) ) {
      for ( m=0; m<edge_indicesn_[j].dim(); m++ ) {
        if ( !bi.contains(edge_indicesn_[j][m],ind) ) {
          bi[nb] =   edge_indicesn_[j][m]; 
          bt[nb] = (*edge_types_ )[j]; 
          nb++;
        }
      }
#if 0
cout << " after edge set: bi=" << bi << endl;
cout << " after edge set: bt=" << bt << endl;
#endif
      if ( !bi.contains(vert_indices_[j][0],ind) ) {
        bi[nb] =   vert_indices_               [j][0]; 
        bt[nb] = (*vert_types_)                [j]   ;  nb++; 
      }
#if 0
cout << "       vert 1 ind: "  << vert_indices_[j][0] << endl;
cout << "       vert 1 typ: "  << (*vert_types_)[j] << endl;
cout << " after vert 1 set: bi=" << bi << endl;
cout << " after vert 1 set: bt=" << bt << endl;
#endif
      if ( !bi.contains(vert_indices_[(j+1)%nVertices_][0],ind) ) {
        bi[nb] =   vert_indices_[(j+1)%nVertices_][0]; 
        bt[nb] = (*vert_types_ )[(j+1)%nVertices_]   ;  nb++; 
      }
#if 0
cout << "       vert 2 ind: " << vert_indices_[(j+1)%nVertices_][0] << endl;
cout << "       vert 2 typ: "  << (*vert_types_)[(j+1)%nVertices_] << endl;
cout << " after vert 2 set: bi=" << bi << endl;
cout << " after vert 2 set: bt=" << bt << endl;
#endif
    } // end, global bdy check
  } // end, face loop

  bdy_indices_->Resize(nb);
  bdy_types_  ->Resize(nb);
  for ( j=0; j<nb; j++ ) {
    (*bdy_indices_)[j] = bi[j];
    (*bdy_types_  )[j] = bt[j];
  }
  
} // end of method ComputeBdyInfo


//************************************************************************************
//************************************************************************************
// METHOD     : GetNodalMultiplicity
// DESCRIPTION: Gets buffer containing (inverse of)global multiplicity of nodes
// ARGUMENTS  :
// RETURNS    : GIBuffer  & to nultiplicity
//************************************************************************************
GVector *Elem2D::GetNodalMultiplicity()
{
  return nodal_multiplicity_; 
} // end of method GetNodalMultiplicity


//************************************************************************************
//************************************************************************************
// METHOD     : GetMask
// DESCRIPTION: Gets mask (if non-zero, point represents a non-bdy global DOF)
// ARGUMENTS  :
// RETURNS    : GVector * to mask 
//************************************************************************************
GVector *Elem2D::GetMask()
{
  return mask_;
} // end of method GetMask


//************************************************************************************
//************************************************************************************
// METHOD     : GetBdyMask
// DESCRIPTION: Gets dependency mask (if non-zero, point represents a global DOF)
// ARGUMENTS  :
// RETURNS    : GVector * to mask
//************************************************************************************
GVector *Elem2D::GetBdyMask()
{
  return bmask_;
} // end of method GetBdyMask


//************************************************************************************
//************************************************************************************
// METHOD     : SetIndices
// DESCRIPTION: Resizes buffer for and sets element's component (local) indices.
//              Note that basis order should be set prior to entry. Also, note that
//              these indices are meaningless if the basis is not GLL-type.
// ARGUMENTS  : none.
// RETURNS    : none
//************************************************************************************
void Elem2D::SetIndices()
{
  GINT  i, j, m;
  GINT  n[] = {GetOrder(1)+1,GetOrder(2)+1};

  if ( basis1 == NULL || basis2 == NULL ) return;  // can't set if there is no valid basis

  // Resize buffers:
  for ( i=0; i<nVertices_; i++ ) vert_indices_[i].Resize(1);
  for ( i=0; i<nEdges_   ; i++ ) {
    edge_indices_  [i].Resize(n[i%2]);
    edge_indicesn_ [i].Resize(n[i%2]-2);  // don't include endpoints
  }
  inter_indices_->Resize(n[0]*n[1]-2*(n[0]+n[1])+4);
  
  // Set the indices for vertices:
  (vert_indices_[0])(0) = 0;
  (vert_indices_[1])(0) = n[0] - 1;
  (vert_indices_[2])(0) = n[0]*n[1] - 1;
  (vert_indices_[3])(0) = n[0]*(n[1]-1);

  // Set the indices for edges:
  for ( i=0; i<n[0]; i++ ) {                
    (edge_indices_[0])(i) = i;                // include endpoints
    (edge_indices_[2])(i) = n[0]*(n[1]-1) + i;
    if ( i>0 && i<(n[0]-1) ) {
      (edge_indicesn_[0])(i-1) = i;             // don't include endpoints
      (edge_indicesn_[2])(i-1) = n[0]*(n[1]-1) + i;
    }
  }

  for ( j=0; j<n[1]; j++ ) {               
    (edge_indices_[1])(j) = (j+1)*n[0] - 1; // include endpoints
    (edge_indices_[3])(j) = j*n[0];
    if ( j>0 && j<(n[1]-1) ) {            // don't include endpoints
      (edge_indicesn_[1])(j-1) = (j+1)*n[0] - 1;
      (edge_indicesn_[3])(j-1) = j*n[0];
    }
  }

  // Set the indices for interior nodes:
  for ( j=1,m=0; j<n[1]-1; j++ ) { 
    for ( i=1; i<n[0]-1; i++,m++ ) {
      (*inter_indices_)[m] = i + n[0]*j;
    }
  }


} // end of method SetIndices


//************************************************************************************
//************************************************************************************
// METHOD     : Point_in_poly (1)
// DESCRIPTION: Determines if all input points lie within
//              the element whose vertices are specified.
// ARGUMENTS  :
// RETURNS    : TRUE if all points are in the polygon; else FALSE
//************************************************************************************
GBOOL Elem2D::Point_in_poly(Point3D *Vertices, GINT  nVertices_, Point3D P[], GINT  num)
{
  GINT  n, np1, k;
  GDOUBLE dotProd;
  Point3D Norm, dR;

  for ( k=0; k<num; k++ )
  {
    for ( n=0; n<nVertices_; n++ )
    {
      np1 = (n + 1) % nVertices_;

      // Assume vertices in counterclockwise order, and
      // compute normal to edge segment n,np1. This vector
      // points 'inward' in a r.h. sense:
      Norm.x1 = -(Vertices[np1].x2 - Vertices[n].x2);
      Norm.x2 =  (Vertices[np1].x1 - Vertices[n].x1);

      // Compute vector dR =  R_test - R_VertexPoint:
      dR.x1   = P[k].x1 - Vertices[n].x1;
      dR.x2   = P[k].x2 - Vertices[n].x2;

      // If dR dot Norm < 0, then point is outside of polygon:
      dotProd = dR.x1*Norm.x1 + dR.x2*Norm.x2;
      if ( fabs(dotProd) > P[k].GetBracket() && dotProd < 0 ) return FALSE;
    }
  }

  return TRUE;
} // end of method Point_in_poly (1)



//************************************************************************************
//************************************************************************************
// METHOD     : Point_in_poly (2) 
// DESCRIPTION: Determines if all input points lie within
//              the element.
// ARGUMENTS  : 
// RETURNS    : TRUE if all points are in the polygon; else FALSE
//************************************************************************************
GBOOL Elem2D::Point_in_poly(Point3D P[], GINT  num)
{ 
  GINT    n, np1, k;
  GDOUBLE dotProd;
  Point3D Norm, dR;
  
  for ( k=0; k<num; k++ )
  { 
    for ( n=0; n<nVertices_; n++ )
    {
      np1 = (n + 1) % nVertices_;

      // Assume vertices in counterclockwise order, and
      // compute normal to edge segment n,np1. This vector
      // points 'inward' in a r.h. sense:
      Norm.x1 = -(spVertices[np1].x2 - spVertices[n].x2);
      Norm.x2 =  (spVertices[np1].x1 - spVertices[n].x1);

      // Compute vector dR =  R_test - R_VertexPoint:
      dR.x1   = P[k].x1 - spVertices[n].x1;
      dR.x2   = P[k].x2 - spVertices[n].x2;

      // If dR dot Norm < 0, then point is outside of polygon:
      dotProd = dR.x1*Norm.x1 + dR.x2*Norm.x2;
      if ( fabs(dotProd) > P[k].GetBracket() && dotProd < 0 ) return FALSE;
    }
  }
  
  return TRUE;
} // end of method Point_in_poly (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Point_in_poly (3) 
// DESCRIPTION: Determines if input point lies within
//              the element.
// ARGUMENTS  : 
// RETURNS    : TRUE if all points are in the polygon; else FALSE
//************************************************************************************
GBOOL Elem2D::Point_in_poly(GDOUBLE x, GDOUBLE y)
{ 
  Point p;

  p.x1 = x;
  p.x2 = y;
  p.x3 = 0.0;

  return Point_in_poly(&p, 1);
  
} // end of method Point_in_poly (3)


//************************************************************************************
//************************************************************************************
// METHOD     : CreateElemDynamic
// DESCRIPTION: Create dynamic quantities that depend on the characteristics
//              of the element
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::CreateElemDynamic()
{ 
  spVertices       = new Point3D            [nVertices_];
  spvVertices      = new GDOUBLE       [GDIM*nVertices_];
  spMidpoints      = new Point3D               [nEdges_];
  elemCenter       = new Point3D                        ;
  vertex_neighbor_ = new GNeighborList      [nVertices_];
  vvertex_neighbor_= new GNeighborList         [nEdges_];
  vert_indices_    = new GIBuffer           [nVertices_];
  edge_neighbor_   = new GNeighborList         [nEdges_];
  edge_mortars_    = new GMortar1D             [nEdges_];
  edge_indices_    = new GIBuffer              [nEdges_];
  edge_indicesn_   = new GIBuffer              [nEdges_];
  inter_indices_   = new GIBuffer                       ;
  bdy_indices_     = new GIBuffer                       ;
  bdy_types_       = new GBTBuffer                      ;
  vert_types_      = new GBTBuffer          (nVertices_);
  edge_types_      = new GBTBuffer             (nEdges_);
  face_types_      = new GBTBuffer             (nFaces_);
  bGlobal_edge_    = new GIBuffer              (nEdges_);            
  *vert_types_      = NONE;
  *edge_types_      = NONE;
  *face_types_      = NONE;
  *bGlobal_edge_    = FALSE;

#if defined(IS3D)
  face_neighbor_   = new GNeighborList*        [nFaces_];
  face_mortar_     = new GMortar2D    *        [nFaces_];
  face_indices_    = new GIBuffer              [nFaces_];
  face_types_      = new GBTBuffer             (nFaces_);
  face_types_      = NONE;
#endif

  nodal_multiplicity_ = new GVector();
  mask_               = new GVector();
  bmask_              = new GVector();

} // end of method CreateElemDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : InitMortars
// DESCRIPTION: Initializes mortars
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void Elem2D::InitMortars()
{ 
  GINT    i, n;
  Point3D *v[2];
  GVector *x[GDIM]={spNodes1,spNodes2};
  GNBasis *b;

  if ( edge_mortars_ == NULL ) {
    cout << "Elem2D::InitMortars: NULL mortar data structure" << endl;
    exit(1);
  }
  if ( edge_indices_ == NULL ) {
    cout << "Elem2D::InitMortars: NULL index data structure" << endl;
    exit(1);
  }

#if defined(IS3D)
  cout << "Elem2D::InitMortars: 3D currently unsupported" << endl;
  exit(1);
#endif

  for ( i=0; i<nEdges_; i++ ) {
    b = i%2 == 0 ? basis1 : basis2;
    switch (i) {
      case 0: v[0] = &spVertices[0]; v[1] = &spVertices[1]; break;
      case 1: v[0] = &spVertices[1]; v[1] = &spVertices[2]; break;
      case 2: v[0] = &spVertices[3]; v[1] = &spVertices[2]; break;
      case 3: v[0] = &spVertices[0]; v[1] = &spVertices[3]; break;
    }
    edge_mortars_[i].SetBasis(b);
    edge_mortars_[i].SetBdyPoints(*v[0],*v[1]);
    edge_mortars_[i].SetHostGrid(x, GDIM, edge_indices_[i].Data(), edge_indices_[i].dim());
    n = b->GetOrder()+1;
  }

} // end of method InitMortars


//************************************************************************************
//************************************************************************************
// METHOD     :  << operator method (1)
// DESCRIPTION: output stream operator
// ARGUMENTS  :
//
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, Elem2D &e)
{
  GINT  i;
  str << endl << "ElemID: " << e.elemid_ << " {" << endl;
  str << " ROOT ID: " << e.rootid_;
  str << " elemtype: " << e.elemtype_;
  str << " nVertices: " << e.nVertices_;
  str << " nEdges: " << e.nEdges_;
  str << endl << " Vertices: " ;
  for ( i=0; i<e.nVertices_; i++ ) str << (e.spVertices[i]) << " "; 
  str << endl << " Edge_indices: " ;
  for ( i=0; i<e.nEdges_; i++ ) str << endl << "    edge[" << i << "]: " << (e.edge_indices_[i]); 
  str << endl << " Vertex_indices: " << endl;
  for ( i=0; i<e.nVertices_; i++ ) str << "    vertex[" << i << "]: " << (e.vert_indices_[i]) << " "; 
  str << endl;
  str << " bGlobalEdge: " << *(e.bGlobal_edge_) << endl;
  str << " bdy_indices: " << *(e.bdy_indices_) << endl;
  str << " bdy_types  : " << *(e.bdy_types_) << endl;
  str << " vert_indices: " << *(e.vert_indices_) << endl;
  str << " vert_types  : " << *(e.vert_types_) << endl;
  str << " bSolved: " << e.bSolved << endl;
  str << " basis1: " << e.basis1 << endl;
  str << " basis2: " << e.basis2 << endl;
  str << " Np1: " << e.Np1;
  str << " Np2: " << e.Np2;
  str << endl << "}";

  return str;
} // end of operator <<


//************************************************************************************
//************************************************************************************
// METHOD     : Map2NewCoords
// DESCRIPTION: Maps field on current element to new specified coords
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GBOOL Elem2D::Map2NewCoords(GVector *oldU, GVector *newX[], GINT  nc, GINT  *inew, GINT  ni,
                            GVector *newU, GIBuffer  *&iremap)
{
  GINT      i, j;
  GINT      newTot, n2Interp ;
  GBOOL     bRet ;
  Point3D   *newX2Interp, *newXiInterp, ptmp ;
  GIBuffer  *i2Interp;
  

  if ( !bInitialized ) return FALSE;
  
  if ( oldU == NULL || newU == NULL ) return FALSE;
  for ( i=0; i<nc; i++ ) {
    if ( newX[i] == NULL ) return FALSE;
  }
  
  newTot      = inew == NULL ? newU->dim() : ni;
  if ( newTot <= 0 ) return TRUE;

  newX2Interp = new Point3D   [newTot] ;
  newXiInterp = new Point3D   [newTot] ;
  i2Interp    = new GIBuffer  (newTot) ;
  bRet        = newX2Interp != NULL && i2Interp != NULL;
 
  if ( bRet ) {
    n2Interp = 0;
    if ( inew == NULL ) {
      for ( i=0; i<newTot; i++ ) {
        for ( j=0; j<nc; j++ ) ptmp(j) = (*newX[j])[i];
        if ( Point_in_poly(&ptmp,1) ) {
          newX2Interp[n2Interp] = ptmp;
          (*i2Interp)[n2Interp] = i;
          n2Interp++;
        }
      }
    }
    else {
      for ( i=0; i<newTot; i++ ) {
        for ( j=0; j<nc; j++ ) ptmp(j) = (*newX[j])[inew[i]];
        if ( Point_in_poly(&ptmp,1) ) {
          newX2Interp[n2Interp] = ptmp;
          (*i2Interp)[n2Interp] = inew[i];
          n2Interp++;
        }
      }
    }
    if ( n2Interp <= 0 ) bRet = FALSE;
  }  
  
  // Do interpolation:
  if ( bRet ) { 
    bRet &= Interp(oldU, newX2Interp, i2Interp->Data(), n2Interp, newU);
  }

  // Copy remapped points to return array, if required:
  if ( bRet && iremap != NULL ) { 
    iremap->Resize(n2Interp);
    for ( i=0; i<n2Interp; i++ )
      (*iremap)(i) = (*i2Interp)(i);
  }
 
  // Delete temporary vars:
  if ( i2Interp    ) delete i2Interp;
  if ( newX2Interp ) delete [] newX2Interp;
  if ( newXiInterp ) delete [] newXiInterp;
  
  return bRet;

} // end of method Map2NewCoords (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Interp (1)
// DESCRIPTION: Interpolate quantity Ufrom to vector of points, xto;
//              return interpolants in array Uto. The quantity ufrom
//              must reside in the grid established by *this element
// ARGUMENTS  : Ufrom: field that we are interpolating from; must cover
//                     the entire element 
//              xto  : array of spatial points at which we want to interpolate
//                     values of Ufrom
//              ito  : array of indices in Uto where the xto point
//              nto  : number of interpolation points
//              Uto  : where to store the interpolations. If ito != NULL, then
//                     they are stored at the ito[nto] positios in the Uto vector;
//                     else they are stored in order starting at 0.
// RETURNS    : TRUE on success; else, FALSE. Failure occurs if
//              any of the points to which field is to be interpolated
//              lies outside of its associated element, or if the transformation
//              from real space to parent domain cannot be done. Failure also
//              occurs if the UFrom factor has dimension less than that
//              implied by element expansion orders.
//************************************************************************************
GBOOL Elem2D::Interp(GVector *Ufrom, Point3D *xto, GINT  *ito, GINT  nto, GVector *Uto)
{
  char    *serr = "Elem2D::Interp: ";
  GINT    i, j, m, n;
  GINT    NN1=Np1+1, NN2=Np2+1;
  GBOOL   bRet = TRUE;
  GDOUBLE b2;
  Point3D *xi;
  GVector Utmp;
  GMatrix Ito;

  if ( !bInitialized ) {
    cout << "Elem2D::Interp: object not initialized" << endl;
    return FALSE;
  }
  if ( !Point_in_poly(xto,nto) ) {
    cout << "Elem2D::Interp: list of interpolation points not contained within element" << endl;
    for ( i=0; i<nto; i++ ) cout << "p_interp[" << i << "]=" << xto[i] << "; " ;
    cout << endl << " element vertices:" << endl;
    for ( i=0; i<nVertices_; i++ ) cout << "vertex[" << i << "]=" << spVertices[i] << "; " ;
    cout << endl;

    
    return FALSE;
  }
  if ( Ufrom->dim() < NN1*NN2 ) {
    cout << "Elem2D::Interp: Interpolant dimension incorrect" << endl;
    return FALSE;
  }

  if ( basis1 == NULL || basis2 == NULL ) return FALSE;

  xi = new Point3D [nto];

  if ( !XToXi(xto, xi, nto) ){
    cout << "Elem2D::Interp: XToXi failed" << endl;
    return FALSE;
  }
  
  Ito.Resize(nto,NN1*NN2);

  // Compute interpolation matrices for unordered input points,
  // and then carry out a matrix-vector product for the interpolated quantities:
  for ( m=0; m<nto; m++ ) {  
    for ( j=0,n=0; j<NN2; j++ ) {  
      b2 =  basis2->EvalBasis(j,xi[m].x2);
      for ( i=0; i<NN1; i++,n++ ) {  
        Ito(m,n) = b2 * basis1->EvalBasis(i,xi[m].x1);
      }
    }
  }

  if ( ito == NULL ) {
    MTK::fmatvec_prod(Ito, *Ufrom, *Uto);
  }
  else {
    Utmp.Resize(nto);
    MTK::fmatvec_prod(Ito, *Ufrom, Utmp);
    for ( m=0; m<nto; m++ ) (*Uto)(ito[m]) = Utmp(m);
  }

//cout << serr << "Uto  =" << *Uto   << endl;

#if 0
  // Loop over the parent domain points:
  if ( ito == NULL ) {
    for ( m=0; m<nto;  m++ ) {
      // Loop over nodes, i, j:
      for ( j=0, n=0, sum=0.0; j<NN2; j++ ) {
        b2 = basis2->EvalBasis(j,xi[m].x2);
        for ( i=0; i<NN1; i++, n++ ) {
          phi = (basis1->EvalBasis(i, xi[m].x1)) * b2;
          sum += (*(u_data+n)) * phi;
        }
      }
      (*Uto)(m) = sum;
    }
    bRet = TRUE;
  }
  else {
    for ( m=0; m<nto;  m++ ) {
      // Loop over nodes, i, j:
      for ( j=0, sum=0.0, n=0; j<NN2; j++ ) {
        b2 = basis2->EvalBasis(j,xi[m].x2);
        for ( i=0; i<NN1; i++, n++ ) {
          phi = (basis1->EvalBasis(i, xi[m].x1)) * b2;
          sum += (*(u_data+n)) * phi;
        }
      }
      (*Uto)(ito[m]) = sum;
    }
    bRet = TRUE;
  }
#endif

  delete [] xi;

  return bRet;
} // end of method Interp (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetArea
// DESCRIPTION: Get area of element
// ARGUMENTS  : none
// RETURNS    : area
//************************************************************************************
GDOUBLE Elem2D::GetArea()
{   
  return area_;
} // end of method GetArea

#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetTmp
// DESCRIPTION: Gets temp space vectors
// ARGUMENTS  : iwhich: which one
// RETURNS    : pointer to temp space on success; else NULL
//************************************************************************************
GVector *Elem2D::GetTmp(GINT  iwhich)
{
  if ( iwhich < 0 || iwhich >= G_NUM_ELEM_TMP ) return NULL;
  return &vtmp[iwhich];
} // end of method GetTmp


//************************************************************************************
//************************************************************************************
// METHOD     : GetNTmp
// DESCRIPTION: Gets number of temp space vectors
// ARGUMENTS  : none
// RETURNS    : number of vectors available
//************************************************************************************
GINT  Elem2D::GetNTmp()
{
  return G_NUM_ELEM_TMP;
} // end of method GetNTmp
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis (1)
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void Elem2D::SetBasis(GNBasis *semb1, GNBasis *semb2)
{
  GINT  i, NN, Np[2];

  basis1 = semb1;
  basis2 = semb2;

  bSolved = FALSE;

  if ( basis1 == NULL || basis2 == NULL ) return;

  Np[0] = basis1 == NULL ? 0 : basis1->GetOrder();
  Np[1] = basis2 == NULL ? 0 : basis2->GetOrder();
  Resize(Np[0], Np[1]);
  NN = (Np[0]+1) * (Np[1]+1);
  vtmpmgr_->SetSize(NN);
  for ( i=0; i<G_NUM_ELEM_TMP; i++ ) {
    vtmp[i].Resize(NN);
  }

} // end of method SetBasis (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis (2)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void Elem2D::SetBasis(GNBasis *b, GINT  idir)
{
  GINT  i, NN , Np[GDIM];

  if ( idir < 1 || idir > GDIM) {
    cout << "Elem2D::SetBasis(2): invalid coordinate direction" << endl;
    exit(1);
  }
  if      ( idir == 1 ) basis1 = b;
  else if ( idir == 2 ) basis2 = b;

  bSolved = FALSE;

  Np[0] = basis1 == NULL ? 0 : basis1->GetOrder();
  Np[1] = basis2 == NULL ? 0 : basis2->GetOrder();
  Resize(Np[0], Np[1]);
  NN = (Np[0]+1) * (Np[1]+1);
  vtmpmgr_->SetSize(NN);
  vtmpmgr_->SetSize(NN);
  for ( i=0; i<G_NUM_ELEM_TMP; i++ ) {
    vtmp[i].Resize(NN);
  }

} // end of method SetBasis (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetDBasis
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void Elem2D::SetDBasis(GNBasis *b, GINT  idir)
{
  GINT  i, NN , Np[GDIM];

  if ( idir < 1 || idir > GDIM) {
    cout << "Elem2D::SetDBasis: invalid coordinate direction" << endl;
    exit(1);
  }

  dealias_basis_[idir-1] = b;

} // end of method SetDBasis 


//************************************************************************************
//************************************************************************************
// METHOD     : GetDMass
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
GVector *Elem2D::GetDMass()
{
  return &DMassMatrix_;

} // end of method GetDMass



//************************************************************************************
//************************************************************************************
// METHOD     : GetDJ
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
GMatrix *Elem2D::GetDJ(GINT idir)
{
  if ( idir < 1 || idir > GDIM) {
    cout << "Elem2D::GetDJ: invalid coordinate direction" << endl;
    exit(1);
  }
  return &JD_[idir-1];

} // end of method GetDJ


//************************************************************************************
//************************************************************************************
// METHOD     : GetDJT
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
GMatrix *Elem2D::GetDJT(GINT idir)
{
  if ( idir < 1 || idir > GDIM) {
    cout << "Elem2D::GetDJT: invalid coordinate direction" << endl;
    exit(1);
  }
  return &JDT_[idir-1];

} // end of method GetDJT




//************************************************************************************
//************************************************************************************
// METHOD     : SetOrder
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void Elem2D::SetOrder(GINT  iorder1, GINT  iorder2)
{
   if ( iorder1 == Np1 && iorder2 == Np2  ) return;

   bInitialized = FALSE;

   Resize(iorder1, iorder2);

} // end of method SetOrder


//************************************************************************************
//************************************************************************************
// METHOD     : GetOrder
// DESCRIPTION:   
// ARGUMENTS  :   
// RETURNS    :  GINT  element expansion order
//************************************************************************************
GINT  Elem2D::GetOrder(const GINT  iDir)
{ 
  GINT  Np[GDIM];
  if ( iDir < 1 || iDir > GDIM ) return 0;

  Np[0] = basis1 == NULL ? 0 : basis1->GetOrder();
  Np[1] = basis2 == NULL ? 0 : basis2->GetOrder();

  return Np[iDir-1];
} // end of method GetOrder


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumNodes
// DESCRIPTION:   
// ARGUMENTS  :   
// RETURNS    :  GINT  total number of nodes in element
//************************************************************************************
GINT  Elem2D::GetNumNodes()
{ 
  return (basis1->GetOrder()+1) * (basis2->GetOrder()+1);

} // end of method GetNumNodes


//************************************************************************************
//************************************************************************************
// METHOD     : GetMinEdgeLength
// DESCRIPTION:   
// ARGUMENTS  :   
// RETURNS    :  GDOUBLE min edge length
//************************************************************************************
GDOUBLE Elem2D::GetMinEdgeLength()
{
  GINT  j;
  GDOUBLE lmin=SEHUGE;

  for ( j=0; j<nVertices_-1; j++ ) { 
    lmin = MIN( lmin, PDISTANCE(spVertices[(j+1)%nVertices_],spVertices[j]) );
  }

  return lmin;
} // end of method GetMinEdgeLength


//************************************************************************************
//************************************************************************************
// METHOD     : GetMaxEdgeLength
// DESCRIPTION:   
// ARGUMENTS  :   
// RETURNS    :  GDOUBLE max edge length
//************************************************************************************
GDOUBLE Elem2D::GetMaxEdgeLength()
{
  GINT  j;
  GDOUBLE lmax=0.0;

  for ( j=0; j<nVertices_-1; j++ ) {
    lmax = MAX( lmax, PDISTANCE(spVertices[(j+1)%nVertices_],spVertices[j]) );
  }

  return lmax;
} // end of method GetMaxEdgeLength


//************************************************************************************
//************************************************************************************
// METHOD     : GetTemp (1)
// DESCRIPTION: Get temp space (GVector)  
// ARGUMENTS  : none
// RETURNS    : GVector pointer or NULL if not successful
//************************************************************************************
GVector *Elem2D::GetTemp()
{
  GVector *vec;
  if ( !vtmpmgr_->GetGVec(vec) ) {
    cout << "Elem2D::GetTemp(1): GetGVec failed" << endl;
    exit(1);
  }
  return vec;

} // end of method GetTemp (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetTemp (2)
// DESCRIPTION: Get temp space (GVector)  
// ARGUMENTS  : i: index specifying temp vector
// RETURNS    : GVector *
//************************************************************************************
GVector *Elem2D::GetTemp(GSHORT i)
{
  return vtmpmgr_->GetGVec(i);
} // end of method GetTemp (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetTempMgr
// DESCRIPTION: Get temp mem mgr
// ARGUMENTS  : none.
// RETURNS    : GMemMgr *
//************************************************************************************
GMemMgr *Elem2D::GetTempMgr()
{
  return vtmpmgr_;
} // end of method GetTempMgr


//************************************************************************************
//************************************************************************************
// METHOD     : TempLock
// DESCRIPTION: lock temp vector
// ARGUMENTS  : vec: pointer from memory manager to vector to be locked 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL Elem2D::TempLock(GVector *vec)
{
  return vtmpmgr_->Lock(vec);
} // end of method TempLock


//************************************************************************************
//************************************************************************************
// METHOD     : TempUnlock
// DESCRIPTION: unlock lock temp vector
// ARGUMENTS  : vec: pointer from memory manager to vector to be unlocked 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL Elem2D::TempUnlock(GVector *vec)
{
  return vtmpmgr_->Unlock(vec);
} // end of method TempUnlock


//************************************************************************************
//************************************************************************************
// METHOD     : GetDBasisObj
// DESCRIPTION: gets dealias bases
// ARGUMENTS  : idir: coord direction
// RETURNS    : GNBasis * pointer to object
//************************************************************************************
GNBasis *Elem2D::GetDBasisObj(GINT idir)
{
  if ( idir < 1 || idir > GDIM) {
    cout << "Elem2D::GetDBasisObj: invalid coordinate direction" << endl;
    exit(1);
  }

  return dealias_basis_[idir-1];
} // end of method GetDBasisObj


//************************************************************************************
//************************************************************************************
// METHOD     : DealiasingEnabled
// DESCRIPTION: tells if dealiasing is enabled or not.
// ARGUMENTS  : 
// RETURNS    : TRUE or FALSE depending on whether approprioate dealias_basis_ 
//              quantities have been set.
//************************************************************************************
GBOOL Elem2D::DealiasingEnabled()
{
  GINT  j;
  GBOOL bflag;

  for ( j=0,bflag=TRUE; j<GDIM; j++ ) bflag = bflag && dealias_basis_[j] != NULL;

  return bflag;
} // end of method DealiasingEnabled


