//************************************************************************************//
// Module       : ext.hpp
// Date         : 11/8/04 (DLR)
// Copyright    : 2004-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                specifying and computing coefficients in an
//                extrapoloation-based scheme with variable timestep.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "ext.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
EXT::EXT(GSHORT iorder)
:
iorder_               (iorder),
maxorder_             (3),
dthist_               (NULL)
{
  SetOrder(iorder);
} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Destructor
EXT::~EXT()
{
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
EXT::EXT(const EXT &a)
{
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : SetOrder 
// DESCRIPTION: Sets EXT order.
// ARGUMENTS  : iorder : order.
// RETURNS    : none.
//************************************************************************************
void EXT::SetOrder(GSHORT iorder)
{
  if ( iorder < 1 || iorder > maxorder_ ) {
    cout << "EXT::SetOrder: invalid specification";
    exit(1);
  }
  iorder_ = iorder;
  coeffs_.Resize(iorder);
  coeffs_ = 0.0;

} // end of method SetOrder
 

//************************************************************************************
//************************************************************************************
// METHOD     : SetTimestepHistory
// DESCRIPTION: Sets timestep history. This list must contain iorder timesteps at least,
//              with the ordering: dt[0] = dt^n, dt[1]=dt^n-1, dt[2]=dt^n-2...
// ARGUMENTS  : timestep buffer.
// RETURNS    : none
//************************************************************************************
void EXT::SetTimestepHistory(GDBuffer *dthist)
{
  dthist_ = dthist;

} // end of method SetTimestepHistory

 
//************************************************************************************
//************************************************************************************
// METHOD     : GetTimestepHistory
// DESCRIPTION: Gets timestep history. This list must contain iorder timesteps at least,
//              with the ordering: dt[0] = dt^n, dt[1]=dt^n-1, dt[2]=dt^n-2...
// ARGUMENTS  : none.
// RETURNS    : GDBuffer * to dthist_ pointer.
//************************************************************************************
GDBuffer *EXT::GetTimestepHistory()
{
  return dthist_;

} // end of method GetTimestepHistory

 
//************************************************************************************
//************************************************************************************
// METHOD     : GetCoeffs
// DESCRIPTION: Gets EXT coefficients. There will be iorder coeffs 
//              with the time ordering:  n, n-1, n-2 ....
// ARGUMENTS  : none.
// RETURNS    : GDBuffer * to coeffs_ pointer.
//************************************************************************************
GDBuffer *EXT::GetCoeffs()
{
  return &coeffs_;

} // end of method GetCoeffs

 
//************************************************************************************
//************************************************************************************
// METHOD     : ComputeCoeffs
// DESCRIPTION: Computes EXT coefficients with variable timestep history.
//              NOTE: dthist_ pointer to timestep history buffer must be 
//                    set properly prior to entry, or 'exit' will be called.
//                    'exit' will be called if the timestep hist. buffer == NULL or
//                    if the number of buffer elements is too small as required
//                    by iorder_ variable.
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void EXT::ComputeCoeffs()
{
  GDOUBLE r1, r2, r3, r4, r5;

  if ( dthist_ == NULL ) {
    cout << "EXT::ComputeCoeffs: NULL timestep history buffer" << endl;
    exit(1);
  }

  if ( dthist_->dim() < iorder_ ) {
    cout << "EXT::ComputeCoeffs: insufficient timestep history" << endl;
    exit(1);
  }

  if      ( iorder_ == 1 ) {
    coeffs_[0] = 1.0;
  }
  else if ( iorder_ == 2 ) {
    r1         = (*dthist_)[0] / (*dthist_)[1];
    coeffs_[0] = 1.0 + r1;
    coeffs_[1] = -r1;
  }
  else if ( iorder_ == 3 ) {
    r1         = (*dthist_)[0] / (*dthist_)[1];
    r2         = (*dthist_)[1] / (*dthist_)[2];
    r3         = (*dthist_)[0] / (*dthist_)[2];
    r4         = (r2  + r3) / (1.0 + r2);
    r5         = (1.0 + r1) * (1.0 + r2);

    coeffs_[0] = 1.0 - r1*r1 - r3*r4 + r1*r5;;
    coeffs_[1] = r1*r1 - r1*r5;
    coeffs_[2] = r3* r4;
  }

} // end of method ComputeCoeffs

//************************************************************************************
//************************************************************************************
// METHOD     : ()
// DESCRIPTION: Gets EXT coefficients, by index. There will be iorder coeffs 
//              with the time ordering:  n, n-1, n-2 ....
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE &EXT::operator()(const GSHORT i)
{
  return coeffs_(i);
} // end of operator () 


//************************************************************************************
//************************************************************************************
// METHOD     : []
// DESCRIPTION: Gets EXT coefficients, by index. There will be iorder coeffs 
//              with the time ordering:  n, n-1, n-2 ....
// ARGUEMENTS :
// RETURNS    :
//************************************************************************************
GDOUBLE &EXT::operator[](const GSHORT i)
{ 
  return coeffs_[i];
} // end of operator  []
  
 
