//************************************************************************************//
// Module       : field2d.cpp
// Date         : 9/14/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a single 2D spectral element field
// Derived From :
// Modifications:
//************************************************************************************//
#include "field2d.hpp"
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include "defquad2d.hpp"
#include "rectquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
Field2D::Field2D(GINT  NLevels, Elem2D *e, GSHORT ntmp)
:
Np1             (0),
Np2             (0),
NN1             (0),
NN2             (0),
iTot            (0),
nTimeLevels     (NLevels),
iTimeLevel      (0),
bInitialized    (FALSE),
time            (NULL),
basis1          (NULL),
basis2          (NULL),
elem            (e),
flist_          (NULL),
Un              (NULL),
u_bdy_          (NULL),
vtmpmgr_        (NULL)
{
  GSHORT  i;

  if ( elem == NULL ) {
    cout << "Field2D::Field2D: NULL element not allowed" << endl;
    exit(1);
  }
  basis1 = elem->GetBasisObj(1);;
  basis2 = elem->GetBasisObj(2);
  if ( basis1 == NULL || basis2 == NULL ) {
    cout << "Field2D::Field2D: One or more NULL bases" << endl;
    exit(1);
  }
  if ( nTimeLevels <= 0 ) {
    cout << "Field2D::Field2D: invalid number of time levels or temp levels" << endl;
    exit(1);
  }

  Np1   = basis1->GetOrder();
  Np2   = basis2->GetOrder();
  NN1   = Np1 + 1;
  NN2   = Np2 + 1;
  iTot  = NN1*NN2;

  Un        = new GVector * [nTimeLevels] ;
  if ( Un == NULL ) {
    cout << "Field2D::Field2D: allocation error" << endl;
    exit(1);
  }
  u_bdy_ = new GVector ();

  time      = new GDOUBLE [nTimeLevels];
  memset(time, '\0', nTimeLevels*sizeof(GDOUBLE));

  for ( i=0; i<nTimeLevels; i++ ) {
    Un  [i] = new GVector(iTot);
    if ( Un[i] == NULL ) {
      cout << "Field2D::Field2D: allocation error" << endl;
      exit(1);
    }
  }
  vtmpmgr_ = new GMemMgr (ntmp);
  vtmpmgr_->SetSize(iTot);

  flist_ = new SIForceList();
  bInitialized = TRUE;
  
} // end of constructor method


//************************************************************************************
//************************************************************************************
// Destructor
Field2D::~Field2D()
{
   DeleteDynamic();
}

// Assignment operator method (1)
void Field2D::operator=(Field2D &elem)
{
  if ( iTimeLevel < 0 || iTimeLevel >= nTimeLevels ) {
    cout << "Field2D::operator=: invalid time level" << endl;
    exit(1);
  }
  *Un[iTimeLevel] = *(elem.GetExpCoeffs(0));
}

// Assignment operator method (2)
void Field2D::operator=(GVector &vec)
{
  if ( iTimeLevel < 0 || iTimeLevel >= nTimeLevels ) {
    cout << "Field2D::operator=: invalid time level" << endl;
    exit(1);
  }
  *Un[iTimeLevel] = vec;
}

// Assignment operator method (3)
void Field2D::operator=(GDOUBLE a)
{
  *Un[iTimeLevel] = a;
}


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : SetElement
// DESCRIPTION: Sets or resets the 2d geometry element associated with this
//              field. If element replaces an existing one, then the 
//              exp. coefficients are mapped to the new element. 
//
// ARGUMENTS  : Elem2D *inElem           : new element to which to assign field
//              GIBuffer  iRemappedPoints: buffer containing the points in the old 
//                                         element that have been remapped to the new.
//                                         This is allocated, and caller is responsible
//                                         for cleanup. 
// RETURNS    : TRUE on success, if all initializes properly; else FALSE
//************************************************************************************
GBOOL Field2D::SetElement(Elem2D *inElem, GIBuffer  *&iRemappedPoints)
{
 
  GINT  i, newNp1, newNp2, nTot; 
  GBOOL bRet = TRUE;

  if ( inElem == NULL ) return FALSE;

  newNp1 = inElem->GetOrder(1);
  newNp2 = inElem->GetOrder(2); 

  nTot = (newNp1+1)*(newNp2+1);
 
  // If field was already associated with an element, remap it
  // to the new element:
  if ( bInitialized )
  {
    GVector *newU[nTimeLevels];
    for ( i=0; i<nTimeLevels; i++ )
      newU[i] = new GVector(nTot);
    
    // First, get the new expansion coefficients:
    if ( !Map2NewElem(inElem, newU, nTimeLevels, iRemappedPoints) ) bRet = FALSE;

    // Now, resize the current field, and reset the member data:
    if ( bRet && !Resize(newNp1, newNp2) ) bRet = FALSE;

    for ( i=0; i<nTimeLevels && bRet; i++ )
      *Un[i] = *newU[i];
    
    for ( i=0; i<nTimeLevels; i++ ) delete newU[i];
    delete [] newU;
  }
   
  if ( bRet )
  {
    elem = inElem;
    Np1     = elem->GetOrder(1); 
    Np2     = elem->GetOrder(2); 
    NN1     = Np1 + 1;
    NN2     = Np2 + 1;
    iTot    = nTot;
    bInitialized = TRUE;
  }

  return bRet;
} // end of method SetElement


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// ARGUMENTS  : Elem2D *
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL Field2D::Resize(GINT  newOrder1, GINT  newOrder2)
{

    GINT  i, nTot=(newOrder1+1)*(newOrder2+1);

    if ( Un != NULL ) return FALSE;

    //  Resize element and bases:
    elem->SetOrder(newOrder1, newOrder2) ;
    
    // Size of Un should not have changed (it's set constructor via
    // nTimeLevels), so only need to resize vectors:
    
    for ( i=0; i<nTimeLevels && Un[i]; i++ )
    {
      Un[i]->Resize(nTot);
    }
    

    return TRUE;

} // end of method Resize
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : GetSIForceList
// DESCRIPTION: Gets the state-independent force list for field.
//
// ARGUMENTS  : none
// RETURNS    : SIForceList *
//************************************************************************************
SIForceList *Field2D::GetSIForceList()
{
  return flist_;
} // end of method GetSIForceList


//************************************************************************************
//************************************************************************************
// METHOD     : ResetOrder
// DESCRIPTION: Resets exp. order of field. All time levels are interpolated 
//              to modified bases.
// ARGUMENTS  : GSHORT  n1, n2: modified 1- and 2- expansion orders
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL Field2D::ResetOrder(const GSHORT  n1, const GSHORT  n2)
{
  GSHORT     i, nTot;
  GBOOL     bRet;
  Elem2D    *enew;
  GVector  **Unew;
  GIBuffer  *iremapped=NULL;

  if ( elem == NULL || basis1 == NULL || basis2 == NULL ) return FALSE;

  if      ( elem->ElemType() == DEFORMED_QUAD )
    enew = new DefQuad2D(basis1, basis2);
  else if ( elem->ElemType() == RECT_QUAD )
    enew = new RectQuad2D(basis1, basis2);
  else if ( elem->ElemType() == TRIANGULAR )
  {
    cout << "Field2D::Field2D: TRIANGULAR element currently unsupported" << endl;
    exit(1);
  }

  *enew = *elem;

  if ( !elem->Resize(n1, n2) )
  {
    delete enew;
    return FALSE;
  }
  nTot = (n1+1)*(n2+1);

  // Interpolate time field data:
  Unew = new GVector * [nTimeLevels];
  for ( i=0; i<nTimeLevels; i++ )
    Unew[i] = new GVector(nTot); 
  
  bRet = Map2NewElem(enew, Unew, nTimeLevels, iremapped);
  
  if ( bRet ) 
  {

    for ( i=0; i<nTimeLevels && bRet && Un[i]; i++ )
    {
      bRet &= Un[i]->Resize(nTot);
      *Un[i] = *Unew[i];
    }
    *elem = *enew;
    Np1 = n1;
    Np2 = n2;
    NN1 = Np1 + 1;
    NN2 = Np2 + 1;
    iTot = nTot;
  }
 
  delete enew;
  for ( i=0; i<nTimeLevels; i++ )
    if ( Unew[i] ) delete Unew[i] ;
  delete [] Unew;

  return bRet;
} // end of method ResetOrder


//************************************************************************************
//************************************************************************************
// METHOD     : GetElement
// DESCRIPTION: Retrieve pointer to the finaite element object 
// ARGUMENTS  : none
// RETURNS    : pointer to Elemt2D data member
//************************************************************************************
Elem2D *Field2D::GetElement()
{
  return elem;
} // end of method GetElement


//************************************************************************************
//************************************************************************************
// METHOD     : GetExpCoeffs (1)
// DESCRIPTION: Retrieve the expansion coefficients for level iLevel
// ARGUMENTS  : GINT  nLevel: time level whose coeffs. are sought
// RETURNS    : pointer to (non-NULL) return Un array if successful; else NULL
//              NULL will result if !(0 <= nLevel < nTimeLevels)
//************************************************************************************
GVector *Field2D::GetExpCoeffs(const GINT  iLevel)
{
  if (  iLevel < 0 || iLevel >= (nTimeLevels) ) {
    cout << "Field2D::GetExpCoeffs iLevel requested: " << iLevel << " Un=" << 0x0 << endl;
    return NULL;
  }
//cout << "Field2D::GetExpCoeffs iLevel requested: " << iLevel << " Un=" << Un[iLevel] << endl;
  return Un[iLevel];
} // end of method GetExpCoeffs (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetExpCoeffs (2)
// DESCRIPTION: Retrieve expansion coefficients for all time levels
// ARGUMENTS  : none
// RETURNS    : pointer to (non-NULL) return *Un[] array
//************************************************************************************
GVector **Field2D::GetExpCoeffs()
{
  return Un;
} // end of method GetExpCoeffs (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumTimeLevels
// DESCRIPTION: Retrieve the number of time levels begin stored
// ARGUMENTS  : 
// RETURNS    : nTimeLevels data 
//************************************************************************************
GINT  Field2D::GetNumTimeLevels()
{   
  return nTimeLevels;
} // end of method GetNumTimeLevels


//************************************************************************************
//************************************************************************************
// METHOD     : GetTime
// DESCRIPTION: Retrieve the time corresp. to specified time level
// ARGUMENTS  :
// RETURNS    : evolution time 
//
//************************************************************************************
GDOUBLE Field2D::GetTime(const GINT  ilevel)
{
  if ( ilevel < 0 || ilevel >= nTimeLevels ) {
    cout << "Field2D::GeTime: invalid time level" << endl;
    exit(1);
  }
  return time[ilevel];
} // end of method GetTime


//************************************************************************************
//************************************************************************************
// METHOD     : SetTime
// DESCRIPTION: sets the time corresp. to specified time level
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void Field2D::SetTime(const GINT  ilevel, GDOUBLE t)
{
  if ( ilevel < 0 || ilevel >= nTimeLevels ) {
    cout << "Field2D::SeTime: invalid time level" << endl;
    exit(1);
  }
  time[ilevel] = t;
} // end of method SetTime


//************************************************************************************
//************************************************************************************
// METHOD     : SetExpCoeffs (1) 
// DESCRIPTION: Set basis coeffs at time level iLevel
// ARGUMENTS  : GINT  iLevel: provides the time level whose coeffs are being set 
//              GVector   *a: provides the coeffs 
// RETURNS    : TRUE on success; else FALSE. Failure results if dimension
//              of input vector != that of iLevel internal vector.
//************************************************************************************
GBOOL Field2D::SetExpCoeffs(const GINT  iLevel, GVector *a)
{
  if ( Un   == NULL || iLevel < 0 || iLevel >= nTimeLevels ) return FALSE;

  if ( a->dim() != iTot ) return FALSE;
  
  *Un[iLevel] = *a;
  
  return TRUE;
} // end of method SetExpCoeffs (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetExpCoeffs (2)
// DESCRIPTION: Set basis coeffs at current time level, and rotates the others
// ARGUMENTS  : 
//              GVector   *a: provides the coeffs. If NULL, then time levels
//                             are shifted, but the 0-th level is left alone.
// RETURNS    : TRUE on success; else FALSE. Failure results if dimension
//              of input vector != that of internal vector.
//************************************************************************************
GBOOL Field2D::SetExpCoeffs(GVector *a)
{
  GSHORT  i;

  if ( a!= NULL && a->dim() != iTot ) return FALSE;
 
  ShiftLevels();

  if ( a!= NULL ) *Un[0] = *a;
 
  return TRUE;
} // end of method SetExpCoeffs


//************************************************************************************
//************************************************************************************
// METHOD     : SetTimeLevel
// DESCRIPTION: Set time level for methods that require it
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void Field2D::SetTimeLevel(GINT iLevel)
{
  if ( iLevel < 0 || iLevel >= nTimeLevels ) {
    cout << "Field2D::SetTimeLevel: invalid time level" << endl;
    exit(1);
  }
  iTimeLevel = iLevel;
} // end of method SetTimeLevel


//************************************************************************************
//************************************************************************************
// METHOD     : ShiftLevels
// DESCRIPTION: Rotate time levels of exp. coefficients
// ARGUMENTS  :
//           
// RETURNS    : 
//************************************************************************************
void Field2D::ShiftLevels()
{ 
  GSHORT  i;
  
  for ( i=nTimeLevels-1; i>0; i-- ) {
    *Un  [i] = *Un [i-1];
     time[i] = time[i-1];
  }
  
} // end of method ShiftLevels


//************************************************************************************
//************************************************************************************
// METHOD     : operator()
// DESCRIPTION: Member field access method
// ARGUMENTS  :
//           
// RETURNS    : 
//************************************************************************************
GDOUBLE &Field2D::operator()(const GSHORT  nLevel, const GLONG i)
{ 
 
  if ( nLevel < 0 || nLevel >= nTimeLevels || !Un )
  {
    cout << "Field2D::operator(): Time level " << nLevel << " index out of range" << endl;
    exit(1);
  } 
  if ( i < 0 || i >= Un[nLevel]->dim() )
  {
    cout << "Field2D::operator(): index " << i << " out of range for level " << nLevel  << endl;
    exit(1);
  } 
  return (*Un[nLevel])(i);

} // end of method operator ()
  

//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void Field2D::DeleteDynamic()
{
  GINT  i;

  if ( Un          != NULL ) {
    for ( i=0; i<nTimeLevels; i++ )
      if ( Un[i] != NULL ) delete Un[i];
    delete [] Un ;
    Un = NULL;
  }
  if ( u_bdy_   != NULL ) delete u_bdy_  ; u_bdy_  = NULL;
  if ( time     != NULL ) delete [] time ; time    = NULL; 
  if ( flist_   != NULL ) delete flist_  ; flist_  = NULL;
  if ( vtmpmgr_ != NULL ) delete vtmpmgr_; vtmpmgr_= NULL;

} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Map2NewCoords (1)
// DESCRIPTION: Maps current field from old element to the specified new
//              coords. The interpolation of the exp. coefficients from
//              the old to the new element is stored in the array of GVector's.
// ARGUMENTS  : newX     : new X
//              newY     : new Y
//              inew     : which indices of newX and new Y to use. May be NULL, in which
//                         case all indices of newX and newY will be used.
//              newU     : coefficients, number of which is the same as the number of
//                         newX and newY, if inew==NULL else must be the same dimension
//                         as inew.
//              iLevel   : time level of data within Field to interpolate to new coords.
//              iremapped: Points in new element that were successfully remapped (i.e., did not lie
//                         outside of *this element). Is allocated here, so caller is 
//                         responsible for deletion.
// RETURNS    : TRUE on success, else FALSE
//************************************************************************************
GBOOL Field2D::Map2NewCoords(GVector *newX, GVector *newY, GIBuffer  *inew, GVector *newU,  
                            const GINT  iLevel, GIBuffer  *iremapped)
{
  GBOOL   bRet;
  GVector *x_new[]={newX,newY};

  if ( !bInitialized ) return FALSE;
  if ( elem  == NULL ) return FALSE;
  if ( newX == NULL || newY == NULL || newU == NULL ) return FALSE;
  if ( inew == NULL ) {
    if ( newX->dim() != newY->dim() || newX->dim() != newU->dim() ) return FALSE;
  } else {
    if ( newX->dim() != newY->dim() || inew->dim() != newU->dim() ) return FALSE;
  }

  bRet = elem->Map2NewCoords(Un[iLevel], x_new, 2, inew->Data(), inew->dim(), newU, iremapped) ;

  return bRet;

} // end of method Map2NewCoords (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Map2NewElem
// DESCRIPTION: Maps current field from old element to the specified new
//              element. The interpolation of the exp. coefficients from 
//              the old to the new element is stored in the array of GVector's.
// ARGUMENTS  : Elem2D *newElem: new element to which to interpolate old element's coefficients
//              GVector *            U[]  : array of coefficients, each of which is a quantity of type
//                                          GVector, associated with newElem. These represent the new
//                                          (interpolated) exp. coeffs.
//              GINT                  nU  : number of coefficient arrays. There may be one for each of
//                                          the nTimeLevel sets of coefficients maintained by Field2D; but 
//                                          1 <= nU <= nTimeLevels.
//              GIBuffer  *iRemappedPoints: Points in new element that were successfully remapped (i.e., did not lie
//                                          outside of old element).
// RETURNS    : TRUE on success, else FALSE
//************************************************************************************
GBOOL Field2D::Map2NewElem(Elem2D *newElem, GVector *U[], const GINT  nU, GIBuffer  *iRemappedPoints)
{
  if ( !bInitialized ) return FALSE;
  if ( newElem == NULL ) return FALSE;

  GINT     newTot, i, n2Interp, nTimes, nV=4;
  GBOOL    bRet=TRUE, bEqualVertices=TRUE;
  Point3D  *oldVertices, *newVertices, newp;
  Point3D  *newX2Interp ;
  GIBuffer  *i2Interp;
  GVector *newX, *newY;
  
  oldVertices = elem->GetSpVertices();
  newTot      = (newElem->GetOrder(1) + 1)*(newElem->GetOrder(2) + 1);
  newVertices = newElem->GetSpVertices();
  
  // If new elem is, in fact, old element, geometrically, don't remap it:
  for ( i=0; i<nV; i++ ) bEqualVertices &= elem->AreEqual(oldVertices[i],newVertices[i]);
  if ( newElem              == elem
   &&  newElem->GetOrder(1) == elem->GetOrder(1)
   &&  newElem->GetOrder(2) == elem->GetOrder(2) 
   &&  bEqualVertices                              ) return TRUE;

  newX2Interp = new Point3D   [newTot] ;
  i2Interp    = new GIBuffer  (newTot) ;

  // Get spatial points corresponding to nodes of new element:
  if (  (newX=newElem->GetSpNodes   (1))  == NULL ) bRet=FALSE;
  if (  (newY=newElem->GetSpNodes   (2))  == NULL ) bRet=FALSE;

  *i2Interp = -1;
  if ( bRet ) {
    n2Interp = 0;
    for ( i=0; i<newTot; i++ ) {
      if ( elem->Point_in_poly(*(newX->Data()+i), *(newY->Data()+i)) ) {
        newX2Interp[n2Interp].x1 = (*newX)(i);
        newX2Interp[n2Interp].x2 = (*newY)(i);
        (*i2Interp)(n2Interp) = i;
        n2Interp++;
      }
    }
    if ( n2Interp <= 0 ) bRet = FALSE;
  }

//cout << "Field2D::Map2NewElem: iInterp=" <<  *i2Interp << endl;



  // Do interpolation to new spatial grid points:
  if ( bRet ) {
    nTimes = MIN(nU, nTimeLevels );
    for ( i=0; i<nTimes && bRet; i++ ) { // for each time slice, do interpolation:
      // Convert new interpolatable spatial points to parent domain points:
      bRet &= elem->Interp(Un[i], newX2Interp, i2Interp->Data(), n2Interp, U[i] );
#if 0
      cout << "Field2D::Map2NewElem: Un[" << i << "]=" << *Un[i] << endl <<  " newU[" << i << "]=" << *U[i] << endl;
#endif
    }
  }

  // Copy remapped points to return array if provided:
  if ( bRet && iRemappedPoints != NULL ) {
    iRemappedPoints->Resize(n2Interp);
    for ( i=0; i<n2Interp; i++ )
      (*iRemappedPoints)(i) = (*i2Interp)(i);
  }
  

  // Delete temporary vars:
  delete i2Interp;
  delete [] newX2Interp;

  return bRet;
  
} // end of method Map2NewElem



#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : ComputeUL2
// DESCRIPTION: Compute L-2 norm for u^2
// ARGUMENTS  : Elem2D *
// RETURNS    : return norm; returns FBAD on failure
//************************************************************************************
GDOUBLE Field2D::ComputeUL2(GINT  iTimeLevel)
{
  if ( !bInitialized ) return FBAD;
  if ( iTimeLevel < 0 || iTimeLevel >= nTimeLevels ) return FBAD;

  GINT  i;
  GDOUBLE sum=0.0, *u_data, *w_data;
  GVector W(iTot);

  if ( elem->Get2DWeights(&W) == NULL ) return FBAD;

   
  // Loop over nodes, i:
  u_data = Un[iTimeLevel]->Data();
  w_data = W.Data();
  for ( i=0; i<iTot; i++ )
  {
//  sum += W(i)*(*Un[iTimeLevel])(i)*(*Un[iTimeLevel])(i);
    sum += (*(w_data+i)) * (*(u_data+i)) * (*(u_data+i));
  }

  return sqrt(sum);
} // end of method ComputeUL2


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeDUL2
// DESCRIPTION: Compute L-2 norm of DU/Dx
// ARGUMENTS  : Elem2D *
// RETURNS    : return norm; returns FBAD on failure
//************************************************************************************
GDOUBLE Field2D::ComputeDUL2(GINT  iTimeLevel)
{
  if ( !bInitialized ) return FBAD;
  if ( iTimeLevel < 0 || iTimeLevel >= nTimeLevels ) return FBAD;


  GINT  i ;
  GDOUBLE sum=0.0, *w_data, *dudx_data, *dudy_data;
  GVector W(iTot), dUdx(iTot), dUdy(iTot);

  if ( elem->Get2DWeights(&W)            == NULL ) return FBAD;
  if ( CollocatedDDx(1, iTimeLevel, &dUdx)  == NULL ) return FBAD;
  if ( CollocatedDDx(2, iTimeLevel, &dUdy)  == NULL ) return FBAD;

  // Loop over nodes, i:
  dudx_data = dUdx.Data();
  dudy_data = dUdy.Data();
  w_data = W.Data();
  for ( i=0; i<iTot; i++ )
  {
//  sum += W(i) * ( dUdx(i)*dUdx(i) +  dUdy(i)*dUdy(i) ); 
    sum += (*(w_data+i)) * ( (*(dudx_data+i))*(*(dudx_data+i))  + (*(dudy_data+i))*(*(dudy_data+i)) ); 
  }

  return sqrt(sum);
} // end of method ComputeDUL2
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : GetBdyValues
// DESCRIPTION: Get bdy vals from specified bdy; else provide entire array
// ARGUMENTS  : iedge: specified edge
// RETURNS    : GVector * bdy_vals
//************************************************************************************
GVector *Field2D::GetBdyValues()
{
  return u_bdy_;
} // end of method GetBdyValues

//************************************************************************************
//************************************************************************************
// METHOD     :  << operator method (1)
// DESCRIPTION: output stream operator
// ARGUMENTS  :
//
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, Field2D &f)
{
  GINT  i;
  str << "{" << endl;
  for ( i=0; i<f.nTimeLevels; i++ ) {
    cout << "Data[" << i << "]=" << *(f.Un[i]) << endl;
  }
    cout << "bdy_data=" << *(f.u_bdy_) << endl;
  str << "}" << endl;
  return str;
} // end of operator <<


//************************************************************************************
//************************************************************************************
// METHOD     : Interp (1)
// DESCRIPTION: Interpolate the most time level iLevel field to point, x
// ARGUMENTS  : x, y of point at which to interpolate
// RETURNS    : return interpolant
//************************************************************************************
GDOUBLE Field2D::Interp(const GSHORT  iLevel, const GDOUBLE x, const GDOUBLE y)
{
  if ( !bInitialized ) return FBAD;
  if ( elem  == NULL ) return FBAD;
  if ( iLevel < 0 || iLevel >= nTimeLevels ) return FBAD;

  if ( !elem->Point_in_poly(x,y) ) return FBAD;


  GINT     i, j, n;
  GDOUBLE    sum, phi, *u_data;
  GNBasis *basis1, *basis2;
  Point3D  *xi, *P;

  basis1 = elem->GetBasisObj(1);
  basis2 = elem->GetBasisObj(2);

  if ( basis1 == NULL || basis2 == NULL ) return FBAD;

  xi = new Point3D;
  P  = new Point3D;

  sum = FBAD;
  P->x1 = x;
  P->x2 = y;
  if ( elem->XToXi(P, xi, 1) )
  {
    sum = 0.0;
  // Loop over nodes, i, j:
    u_data = Un[iLevel]->Data();
    for ( j=0, n=0; j<NN2; j++ )
    {
      for ( i=0; i<NN1; i++, n++ )
      {
        phi = (basis1->EvalBasis(i, xi->x1)) * (basis2->EvalBasis(j,xi->x2));
        sum += (*(u_data+n)) * phi;
      }
    }

  }

  delete xi;
  delete P;

  return sum;
} // end of method Interp (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetTemp (1)
// DESCRIPTION: Get temp space (GVector)  
// ARGUMENTS  : none.
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GVector *Field2D::GetTemp()
{
  GVector *vec;

  if ( !vtmpmgr_->GetGVec(vec) ) {
    cout << "Field2D::GetTemp(1): GetGVec failed" << endl;
    exit(1);
  }
#if 0
  elem->GetTemp(vec);
  if ( vec == NULL ) {
    cout << "Field2D::GetTemp(1): GetGVec failed" << endl;
    exit(1);
  }
#endif
  return vec;
} // end of method GetTemp


//************************************************************************************
//************************************************************************************
// METHOD     : GetTemp (2)
// DESCRIPTION: Get temp space (GVector)  
// ARGUMENTS  : i: vector index (id)
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GVector *Field2D::GetTemp(GSHORT i)
{
  GVector *vec;

  if ( (vec=vtmpmgr_->GetGVec(i)) == NULL ) {
    cout << "Field2D::GetTemp(2): GetGVec failed" << endl;
    exit(1);
  }
  return vec;
} // end of method GetTemp (2)


//************************************************************************************
//************************************************************************************
// METHOD     : TempLock
// DESCRIPTION: lock temp vector
// ARGUMENTS  : vec: pointer from memory manager to vector to be locked 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL Field2D::TempLock(GVector *vec)
{
  return vtmpmgr_->Lock(vec);
} // end of method TempLock


//************************************************************************************
//************************************************************************************
// METHOD     : TempUnlock
// DESCRIPTION: unlock lock temp vector
// ARGUMENTS  : vec: pointer from memory manager to vector to be unlocked 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL Field2D::TempUnlock(GVector *vec)
{
  return vtmpmgr_->Unlock(vec);
} // end of method TempUnlock


//************************************************************************************
//************************************************************************************
// METHOD     : GetTempMgr
// DESCRIPTION: Gets pointer to temp memory mgr
// ARGUMENTS  : none
// RETURNS    : GMemMgr *
//************************************************************************************
GMemMgr *Field2D::GetTempMgr()
{
  return vtmpmgr_;
} // end of method TempMgr


