//************************************************************************************//
// Module       : gadvect.hpp
// Date         : 8/9/05 (DLR)
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the access methods and data associated with
//                an advection computation class.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "gadvect.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GAdvect::GAdvect(GFieldList *ulist[], GINT nf, GElemList *uelems)
:
itime_type_      (TE_EXBDF),
bTimeDeriv_      (TRUE),
bRotationalForm_ (FALSE),
bDoAdvect_       (TRUE),
bConstAdvVel_    (FALSE),
bDoDealiasing_   (FALSE),
bDealiasingEnabled_(FALSE),
irank_           (GDIM),
nfields_         (nf),
ntimelevels_     (3),
nelems_          (0),
iorderbdf_       (2),
iorderOIFS_RKK_  (4),
iorderadv_       (2),
nsubcycoifs_     (2),
nsubcycles_      (2),
gamma0_          (0.0),
alpha_           (NULL),
beta_            (NULL),
dthist_          (NULL),
uelems_          (uelems)
{
  char  *serr = "GAdvect::GAdvect (1) : ";
  GINT  j;

  if ( uelems_ == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }

  if ( nfields_ <= 0 ) {
    cout << serr << "insufficient number of fields computed" << endl;
    exit(1);
  }

  for ( j=0; j<GDIM; j++ ) var_list_[j] = NULL;

  for ( j=0; j<nfields_; j++ ) {
    if ( ulist[j] == NULL ) {
      cout << serr << "NULL input field" << endl;
      exit(1);
    }
    var_list_[j] = ulist[j];
  }
  nelems_  = uelems_->size();
  for ( j=0; j<irank_; j++ ) {
    cadv_[j] = NULL;
  }

  BuildCoeffs();
  SetAdvOrder(iorderadv_);
  SetBDFOrder(iorderbdf_);
  SetEvolType(itime_type_);

} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Destructor
GAdvect::~GAdvect()
{
  if ( alpha_ != NULL ) delete [] alpha_;
  if ( beta_  != NULL ) delete [] beta_;
}

//************************************************************************************
//************************************************************************************
// METHOD     : Advect
// DESCRIPTION: Performs advection partial update.
//              
// ARGUMENTS  : pu         : GVecList array containing result of the advect term + 
//                           time deritvative, if done. There must be at least nfields_
//                           members in this array.
//              tmpary     : Temp GVecList _array_ of length >= nfields_ for
//                           each component in pu array
//              dt         : time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdvect::Advect(GVecList *pu, GVecList *tmpary, GDOUBLE time, GDOUBLE dt)
{
  char  *serr = "GAdvect::Advect: ";
  GBOOL bRet;


  if ( ElemListChange() && !ResetExpandables() ) {
    cout << serr << "allocation error" << endl;
    exit(1);
  }

  if ( bDoDealiasing_ && !bDealiasingEnabled_ ) {
    cout << serr << "dealiasing requested but not enabled" << endl;
    exit(1);
  }

  switch (itime_type_) {
  case TE_OIFS:
    bRet = AdvectRKBDF(pu, tmpary, time, dt);
    break;
  case TE_ABBDF:
  case TE_EXBDF:
    if ( bDoDealiasing_ )
      bRet = AdvectABBDF_D(pu, tmpary, time, dt);
    else
      bRet = AdvectABBDF  (pu, tmpary, time, dt);
    break;
  case TE_EXBDF_Weak:
    bRet = AdvectABBDF_Weak(pu, tmpary, time, dt);
    break;
  default:
    cout << serr << "invalid NavierStokes time stepping method specified" << endl;
    bRet = FALSE;
    break;
  }

  return bRet;

} // end of method Advect


//************************************************************************************
//************************************************************************************
// METHOD     : SetAdvOrder
// DESCRIPTION: Sets order of advection approximation on n+1 timestep
//  
//              If this method is called, then SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here and in SetBDFOrder, or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3,...
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void GAdvect::SetAdvOrder(GINT  iorder)
{ 
  iorderadv_ = iorder;
  
} // end of method SetAdvOrder

  
//************************************************************************************
//************************************************************************************
// METHOD     : SetBDFOrder
// DESCRIPTION: Sets order of BDF (backward time diff.) used to approximate the
//              explicit treatment of the time derivative.
//
//              If this method is called, then SetEvolType must be called
//              to select coefficient set. If SetEvolType is not called, then
//              this call will have no effect.
//
//              NOTE: caller must be certain that field lists are set
//                    with an appropriate number of time levels for the
//                    order chosen here, and in SetABOrder or solve will fail.
// ARGUMENTS  : GINT  iorder: 1, 2, 3, ...
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
void GAdvect::SetBDFOrder(GINT  iorder)
{ 
  char *serr = "GAdvect::SetBDFOrder: ";
  if ( iorder < 1 || iorder > 4 ) {
    cout << serr << "invalid approximation order" << endl;
    exit(1);
  }
  iorderbdf_ = iorder;

} // end of method SetBDFOrder


//************************************************************************************
//************************************************************************************
// METHOD     : SetEvolType
// DESCRIPTION: Sets time evolution method type by setting appropriate 
//              coefficient. Must call this _after_ calling SetXXXOrder methods.
//
//              Caller must set fields with appropriate
//              number of time levels set.
// RETURNS    :  
//************************************************************************************
void  GAdvect::SetEvolType(TIME_EVOLTYPE i)
{
  GINT  j;

  itime_type_ = i;

  if ( itime_type_ == TE_ABBDF || itime_type_ == TE_EXBDF || itime_type_ == TE_OIFS || itime_type_ == TE_EXBDF_Weak) {
    if ( alpha_ != NULL ) delete [] alpha_; alpha_ = NULL;
    if ( beta_  != NULL ) delete [] beta_ ; beta_  = NULL;
    alpha_ = new GDOUBLE [iorderadv_];
    for ( j=0; j<iorderadv_; j++ ) alpha_[j] = 0.0;
    ntimelevels_ = MAX(iorderadv_, iorderbdf_);
    for ( j=0; j<iorderadv_; j++ ) alpha_[j] = 0.0;
    beta_ = new GDOUBLE [iorderbdf_];
    memcpy(beta_,c_bdf_[iorderbdf_-1],iorderbdf_*sizeof(GDOUBLE));
    for ( j=0,gamma0_=0.0; j<iorderbdf_; j++ ) {
      gamma0_ += beta_[j];
    }
  }
  switch ( itime_type_ ) {
    case TE_ABBDF:
      ab_.SetOrder(iorderadv_);
    case TE_EXBDF:
      ext_.SetOrder(iorderadv_);
    case TE_OIFS:
      iorderOIFS_RKK_ = iorderadv_;
    case TE_EXBDF_Weak:
      ext_.SetOrder(iorderadv_);
  }

  ResetExpandables();

} // end of method SetEvolType


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeTimeDepCoeffs
// DESCRIPTION: computes time or time-step dependent coefficients
//
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
void GAdvect::ComputeTimeDepCoeffs()
{ 
  char *serr = " GAdvect::ComputeTimeDepCoeffs: ";
  GINT j;   
  
  if ( dthist_ == NULL ) {
    cout << serr << "NULL timestep history buffer" << endl;
    exit(1);
  }
    
  if      ( itime_type_ == TE_ABBDF ) {
    ab_.ComputeCoeffs();
    for ( j=0; j<iorderadv_; j++ ) alpha_[j] = -ab_[j];
  }
  else if ( itime_type_ == TE_EXBDF ) {
    ext_.ComputeCoeffs();
    for ( j=0; j<iorderadv_; j++ ) alpha_[j] = -ext_[j];
  }
  else if ( itime_type_ == TE_EXBDF_Weak ) {
    ext_.ComputeCoeffs();
    for ( j=0; j<iorderadv_; j++ ) alpha_[j] = -ext_[j];
  }
} // end of method ComputeTimeDepCoeffs
 

//************************************************************************************
//************************************************************************************
// METHOD     : SetTimestepHistory
// DESCRIPTION: sets timestep history buffer
//
// ARGUMENTS  : GDBuffer *
// RETURNS    : none.
//************************************************************************************
void GAdvect::SetTimestepHistory(GDBuffer *dthist)
{
  dthist_ = dthist;
  ab_ .SetTimestepHistory(dthist_);
  ext_.SetTimestepHistory(dthist_);
} // end of method SetTimestepHistory
 

//************************************************************************************
//************************************************************************************
// METHOD     : BuildCoeffs
// DESCRIPTION: creates the 'database' of approximation constant
//              expansion coefficients.
//
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void GAdvect::BuildCoeffs()
{   
  memset(c_bdf_, 0, 16*sizeof(GDOUBLE));
    
  c_bdf_[0][0] =  1.0;
  c_bdf_[1][0] =  2.0;
  c_bdf_[1][1] = -0.5;
  c_bdf_[2][0] =  3.0;
  c_bdf_[2][1] = -1.5;
  c_bdf_[2][2] =  1.0/3.0;
  c_bdf_[3][0] =  4.0;
  c_bdf_[3][1] = -3.0;
  c_bdf_[3][2] =  4.0/3.0;
  c_bdf_[3][3] = -0.25;
  
    
} // end of method BuildCoeffs


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoAdvection
// DESCRIPTION: 
// ARGUMENTS  : bAdvect: TRUE or FALSE
// RETURNS    : none.
//************************************************************************************
void GAdvect::SetDoAdvection(GBOOL bAdvect)
{
  bDoAdvect_ = bAdvect;
} // end of method SetDoAdvection
    

//************************************************************************************
//************************************************************************************
// METHOD     : AdvectABBDF
// DESCRIPTION: Performs advection partial update using explicit Adams-Bashforth 
//              or extrapolation time splitting of advection operators. Also, 
//             the explicit portion
//              of the BDF time derivative is added to the advection term on the
//              RHS, so that the system is ready for an update of the implicit
//              terms.  The result is placed in the iLevelTemp. Fields must have 
//              appropriate number of time levels set on instantiation. 
//              NOTE: Typically, the result of this method (in iLevelTemp field) is to
//              serve as a RHS for the update of the implicit Stokes update. Hence,
//              this RHS is already multiplied by the mass matrix as required for 
//              the weak form. 
//              
//              The partial update, f^n,  involved is:
//
//              f_i^n = Sum_m=(1,l) a_m v_i^(n-m+1) - Sum_j=(1,k) b_j N_i(v_i^(n-j+1))  
//
//              where Sum_m represents the explicit portion of the BDF contribution
//              to the time derivative, and Sum_j=(1,k) represents the explicit
//              representation of the Adams-Bashforth approximation to the 
//              advection term. The orders l, and k are the orders of the
//              BDF and AB expansions, respectively. The nonlinear terms, 
//
//              N_i(v_i) = M (v_1 D_1 + v_2 D_2 + v_3 D_3), 
//
//              where M is the mass matrix and v_i are the expansion coeffs for component
//              i in natural ordering, and the D_1 are the tensor product forms 
//              for the derivatives (D_1 = 1 X 1 X D; D_2= 2 X D X 1...). These local
//              operations must then be made global by performing a direct
//              stiffness summation operation. 
//
//              NOTE: InitComm must have been called prior to entry, in order to
//                    perform DSS on the advective quantities. 
//
//                    Also, this method requires that the field elements within the 
//                    field_lists contain one temporary storage level in addition to the 
//                    iLevelTemp level on instantiation.
//              
// ARGUMENTS  : pu_        : GVecList array containing result of the advect term + 
//                           time deritvative, if done. There must be at least nfields_ 
//                           (GDIM) members in this array.
//              Nj_        : Temp GVecList _array_ of length >= nfields_ (GDIM) for
//                           each component in pu_ array
//              time       : current time
//              dt         : time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdvect::AdvectABBDF(GVecList *pu_, GVecList *Nj_, GDOUBLE time, GDOUBLE dt)
{
  char      *serr = "GAdvect::AdvectABBDF: ";
  GINT      i, j, k;
  GDOUBLE   coefbdf, dti;
  GVector   *ua, *cj[3]={NULL,NULL,NULL}, *M, *NNj[3]={NULL,NULL,NULL};


  if ( uelems_ == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }
  ComputeTimeDepCoeffs();

  dti        = dt == 0.0 ? 0.0 : 1.0/dt;

  // Do non-linear advection term:
  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<nfields_; j++ ) {
      *Nj_[j][i] = 0.0; // initialize nonlin temp var
    }
  }
  if ( bDoAdvect_ ) {
    for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL); uelems_->start(NULL);
    if ( bRotationalForm_ ) {      // do rotational form
      for ( i=0; i<nelems_; i++ ) {
        M = uelems_->member()->GetMassMatrix();
        for ( k=0; k<iorderadv_; k++ ) {  // Do Adams-Bashforth approx. of advection contrib: 
          for ( j=0; j<irank_; j++ ) {
            cj [j] = var_list_[j]->member()->GetExpCoeffs(k);
            NNj[j] = pu_[j].member();
          }
          ((VectorOp*)vecop_[i])->AdvectR(cj, NNj);
          for ( j=0; j<nfields_; j++ ) {
            MTK::fvec_const_prod_sum_rep(*Nj_[j][i], *NNj[j], 1.0, alpha_[k]); // alpha contains - sign already!
          }
        } // end of AB/EXT sum of non-linear terms
        for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
        uelems_->next();
      } // end, element loop
    }
    else {                          // do convective form:
      for ( i=0; i<nelems_; i++ ) {
        M = uelems_->member()->GetMassMatrix();
        for ( k=0; k<iorderadv_; k++ ) {  // Do Adams-Bashforth approx. of advection contrib: 
        for ( j=0; j<irank_; j++ ) {    // Set advection velocity components
          if ( var_list_[j] != NULL )
          cj[j] = bConstAdvVel_ ?  cadv_[j]->member(i)->GetExpCoeffs(k) : var_list_[j]->member()->GetExpCoeffs(k);
         }
          ((VectorOp*)vecop_[i])->SetVec(cj[0], cj[1], cj[2]);
          for ( j=0; j<nfields_; j++ ) {
            ua = var_list_[j]->member()->GetExpCoeffs(k);
//if ( k==0 )
//cout << serr << " u_i[i=" << i << "]=" << *pu_[j][i] << endl;
            ((VectorOp*)vecop_[i])->Advect(*ua, *pu_[j][i]);
//if ( k==0 )
//cout << serr << " u_f[i=" << i << "]=" << *pu_[j][i] << endl;
            MTK::fvec_const_prod_sum_rep(*Nj_[j][i], *pu_[j][i], 1.0, alpha_[k]); // alpha contains - sign already!
          } // end of field loop 
//cout << serr << " N[0][k=" << k << "][i=" << i << "]=" << *Nj_[0][i] << endl;
        } // end of AB/EXT sum of non-linear terms
        for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
        uelems_->next();

      } // end, element loop
    }
  }

  if ( !bTimeDeriv_ ) {
    for ( i=0; i<nelems_; i++ ) {
      for ( j=0; j<nfields_; j++ ) {
        *pu_[j][i] = *Nj_[j][i];
      }
    }  
    return TRUE;
  }

  // Now, add in the BDF approximation for the time derivative:
  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  uelems_->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<nfields_; j++ ) *pu_[j][i] = 0.0;
    M = uelems_->member()->GetMassMatrix();
    for ( k=0; k<iorderbdf_; k++ )  {           // Do BDF contribution to RHS:
      coefbdf = beta_[k] * dti  ;
      for ( j=0; j<nfields_; j++ ) {
        ua      = var_list_[j]->member()->GetExpCoeffs(k);
        MTK::fvec_const_prod_sum_rep(*pu_[j][i], *ua, 1.0, coefbdf);
      }  // end of i-field loop
     } // end of BDF loop
    for ( j=0; j<nfields_; j++ ) {
      MTK::fvec_add_rep(*pu_[j][i], *Nj_[j][i]);  // add BDF time deriv. and nonlin. term to advect update
      MTK::fvec_point_prod_rep(*pu_[j][i],*M);    // include mass operator in time deriv + nonlin. term
    }  // end of i-field loop 
    for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
    uelems_->next();
  }  // end of elem loop

  return TRUE;

} // end of operator AdvectABBDF



//************************************************************************************
//************************************************************************************
/**
 * METHOD     : AdvectABBDF_Weak
 * DESCRIPTION: Performs advection partial update using weak divergence.
 *              Fields must have 
 *              appropriate number of time levels set on instantiation. 
 *              NOTE: Typically, the result of this method (in iLevelTemp field) is to
 *              serve as a RHS for the update of the implicit Stokes update. Hence,
 *              this RHS is already multiplied by the mass matrix as required for 
 *              the weak form. 
 *              
 *              The partial update, f^n,  involved is:
 *
 *              f_i^n = Sum_m=(1,l) a_m v_i^(n-m+1) - Sum_j=(1,k) b_j N_i(v_i^(n-j+1))  
 *
 *              where Sum_m represents the explicit portion of the BDF contribution
 *              to the time derivative, and Sum_j=(1,k) represents the explicit
 *              representation of the Adams-Bashforth approximation to the 
 *              advection term. The orders l, and k are the orders of the
 *              BDF and AB expansions, respectively. The nonlinear terms, 
 *
 *              N_i(v_i) = M (v_1 D_1 + v_2 D_2 + v_3 D_3), 
 *
 *              where M is the mass matrix and v_i are the expansion coeffs for component
 *              i in natural ordering, and the D_1 are the tensor product forms 
 *              for the derivatives (D_1 = 1 X 1 X D; D_2= 2 X D X 1...). These local
 *              operations must then be made global by performing a direct
 *              stiffness summation operation. 
 *
 *              NOTE: InitComm must have been called prior to entry, in order to
 *                    perform DSS on the advective quantities. 
 *
 *                    Also, this method requires that the field elements within the 
 *                    field_lists contain one temporary storage level in addition to the 
 *                    iLevelTemp level on instantiation.
 *              
 * ARGUMENTS  : pu_        : GVecList array containing result of the advect term + 
 *                           time deritvative, if done. There must be at least nfields_ 
 *                           (GDIM) members in this array.
 *              Nj_        : Temp GVecList _array_ of length >= nfields_ (GDIM) for
 *                           each component in pu_ array
 *              time       : current time
 *              dt         : time step
 * RETURNS    : TRUE on success; else FALSE
 */
//************************************************************************************
GBOOL GAdvect::AdvectABBDF_Weak(GVecList *pu_, GVecList *Nj_, GDOUBLE time, GDOUBLE dt)
{
  char      *serr = "GAdvect::AdvectABBDF_Weak: ";
  GINT      i, j, k;
  GDOUBLE   coefbdf, dti;
  GVector   *ua, *cj[3]={NULL,NULL,NULL}, *M, *NNj[3]={NULL,NULL,NULL};

  cout << "Weak advection part " << endl;

  if ( uelems_ == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }
  ComputeTimeDepCoeffs();

  dti        = dt == 0.0 ? 0.0 : 1.0/dt;

  // Do non-linear advection term:
  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<nfields_; j++ ) {
      *Nj_[j][i] = 0.0; // initialize nonlin temp var
    }
  }
  if ( bDoAdvect_ ) {

    for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL); uelems_->start(NULL);
    cout << __LINE__ << ":" << __FILE__ << endl;
    if ( bRotationalForm_ ) {  
      cout << serr << "Rotational form of advection not supported for Discontinuous Galerkin" << endl;
      exit(1);      
    }
    else {                          // do convective form:
      for ( i=0; i<nelems_; i++ ) {
        M = uelems_->member()->GetMassMatrix();
        for ( k=0; k<iorderadv_; k++ ) {  // Do Adams-Bashforth approx. of advection contrib: 
	  for ( j=0; j<irank_; j++ ) {    // Set advection velocity components
	    if ( var_list_[j] != NULL )
	      cj[j] = bConstAdvVel_ ?  cadv_[j]->member(i)->GetExpCoeffs(k) : var_list_[j]->member()->GetExpCoeffs(k);
	  }
          ((VectorOp*)vecop_[i])->SetVec(cj[0], cj[1], cj[2]);
          for ( j=0; j<nfields_; j++ ) {
            ua = var_list_[j]->member()->GetExpCoeffs(k);
            ((VectorOp*)vecop_[i])->Advect_Weak(*ua, *pu_[j][i]);
            MTK::fvec_const_prod_sum_rep(*Nj_[j][i], *pu_[j][i], 1.0, alpha_[k]); // alpha contains - sign already!
          } // end of field loop 
        } // end of AB/EXT sum of non-linear terms
        for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
        uelems_->next();
      } // end, element loop
    }
  }
  
  if ( !bTimeDeriv_ ) {
    for ( i=0; i<nelems_; i++ ) {
      for ( j=0; j<nfields_; j++ ) {
        *pu_[j][i] = *Nj_[j][i];
      }
    }  
    return TRUE;
  }
#if 1
  // Now, add in the BDF approximation for the time derivative:
  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  uelems_->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<nfields_; j++ ) *pu_[j][i] = 0.0;
    M = uelems_->member()->GetMassMatrix();
    for ( k=0; k<iorderbdf_; k++ )  {           // Do BDF contribution to RHS:
      coefbdf = beta_[k] * dti  ;
      for ( j=0; j<nfields_; j++ ) {
        ua      = var_list_[j]->member()->GetExpCoeffs(k);
        MTK::fvec_const_prod_sum_rep(*pu_[j][i], *ua, 1.0, coefbdf);
      }  // end of i-field loop
     } // end of BDF loop
    for ( j=0; j<nfields_; j++ ) {
      // In the weak case, Nj already contains a mass matrix 
      // first multiply pu with M then add N to pu. 
      MTK::fvec_point_prod_rep(*pu_[j][i],*M);    // include mass operator in time deriv + nonlin. term
      MTK::fvec_add_rep(*pu_[j][i], *Nj_[j][i]);  // add BDF time deriv. and nonlin. term to advect update
    }  // end of i-field loop 
    for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
    uelems_->next();
  }  // end of elem loop
#endif
  return TRUE;

} // end of operator AdvectABBDF_Weak



//************************************************************************************
//************************************************************************************
// METHOD     : AdvectABBDF_D
// DESCRIPTION: Performs advection partial update using explicit Adams-Bashforth 
//              or extrapolation time splitting of advection operators. Also, 
//              the explicit portion of the BDF time derivative is added to the advection 
//              term on the RHS, so that the system is ready for an update of the implicit
//              terms.  
//              
//              The partial update, f^n,  involved is:
//
//              f_i^n = Sum_m=(1,l) a_m v_i^(n-m+1) - Sum_j=(1,k) b_j N_i(v_i^(n-j+1))  
//
//              where Sum_m represents the explicit portion of the BDF contribution
//              to the time derivative, and Sum_j=(1,k) represents the explicit
//              representation of the Adams-Bashforth approximation to the 
//              advection term. The orders l, and k are the orders of the
//              BDF and AB expansions, respectively. The nonlinear terms, 
//
//              N_i(v_i) = M (v_1 D_1 + v_2 D_2 + v_3 D_3), 
//
//              where M is the mass matrix and v_i are the expansion coeffs for component
//              i in natural ordering, and the D_1 are the tensor product forms 
//              for the derivatives (D_1 = 1 X 1 X D; D_2= 2 X D X 1...). These local
//              operations must then be made global by performing a direct
//              stiffness summation operation. 
//
//              Here, the dealiased form is computed:
//              (Jv)^T Ju . BB J Grad u, 
//              where J interpolates to the nodal grid of degree M from the grid
//              of degree N, and BB is the mass matrix on the M-grid.
//              
// ARGUMENTS  : pu_        : GVecList array containing result of the advect term + 
//                           time deritvative, if done. There must be at least nfields_ 
//                           (GDIM) members in this array.
//              Nj_        : Temp GVecList _array_ of length >= nfields_ (GDIM) for
//                           each component in pu_ array
//              time       : current time
//              dt         : time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdvect::AdvectABBDF_D(GVecList *pu_, GVecList *Nj_, GDOUBLE time, GDOUBLE dt)
{
  char      *serr = "GAdvect::AdvectABBDF_D: ";
  GINT      i, j, k, m, NN[GDIM], ND[GDIM], NT0[GDIM], NT1[GDIM];
  GDOUBLE   coefbdf, dti;
  GVector   *BD, *du, *ua, *cj[3]={NULL,NULL,NULL}, *M, tD0, tD1, ttD0, ttD1, NjD[GDIM];
  GMatrix   *J[GDIM], *JT[GDIM];


  if ( uelems_ == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }
  ComputeTimeDepCoeffs();

  dti        = dt == 0.0 ? 0.0 : 1.0/dt;

  // Do non-linear advection term:
  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<nfields_; j++ ) {
      *Nj_[j][i] = 0.0; // initialize nonlinear temp var
    }
  }
  if ( bDoAdvect_ ) {
    for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL); uelems_->start(NULL);
    for ( j=0; j<nfields_&&bConstAdvVel_; j++ ) cadv_[j]->start(NULL); 
    for ( i=0; i<nelems_; i++ ) {
      M   = uelems_->member()->GetMassMatrix();
      BD  = uelems_->member()->GetDMass();
      tD0 .Resize(BD->dim());
      tD1 .Resize(BD->dim());
      for ( j=0; j<GDIM; j++ ) {
        J  [j] = uelems_->member()->GetDJ(j+1);
        JT [j] = uelems_->member()->GetDJT(j+1);
        NN [j] = uelems_->member()->GetOrder(j+1) + 1;
	ND [j] = uelems_->member()->GetDBasisObj(j+1)->GetOrder() + 1;
	NjD[j] . Resize(BD->dim());
      }
      NT0[0] = J[0]->dim(1);
      NT0[1] = NN[1];
      NT1[0] = JT[0]->dim(1);
      NT1[1] = ND[1];
      ttD0 .Resize(NT0[0]*NT0[1]);
      ttD1 .Resize(NT1[0]*NT1[1]);
      du     = uelems_->member()->GetTemp();
      for ( k=0; k<iorderadv_; k++ ) {  // Do Adams-Bashforth approx. of advection contrib: 
        for ( j=0; j<irank_; j++ )      // Set advection velocity components
          cj[j] = bConstAdvVel_ ?  cadv_    [j]->member()->GetExpCoeffs(k) :
                                   var_list_[j]->member()->GetExpCoeffs(k);
        for ( j=0; j<nfields_; j++ ) {  // Do advection for each field quantity, j
          NjD[j] = 0.0;
          ua = var_list_[j]->member()->GetExpCoeffs(k);
          for ( m=0; m<irank_; m++ ) {  // loop over adv-velocity components:
            uelems_->member()->Differentiate(du, ua, m+1);                                         // Grad_m u_j
            MTK::D2_X_D1(*J[0],*JT[1],*du   ,NN[0],NN[1],ttD0,NT0[0],NT0[1],tD0,ND[0],ND[1]);      // J Grad_m u_j
            MTK::D2_X_D1(*J[0],*JT[1],*cj[m],NN[0],NN[1],ttD0,NT0[0],NT0[1],tD1,ND[0],ND[1]);      // J c_m
            MTK::fvec_point_prod_rep(tD1,tD0);                                                     // Jc_m JGrad_m u_j
            MTK::fvec_add_rep(NjD[j],tD1);                                                         // Nj += Jc_m JGrad_m u_j 
          } // end, adv-velocity component loop
          MTK::fvec_point_prod_rep(NjD[j],*BD);                                                    // Nj = B  Nj
          MTK::D2_X_D1(*JT[0],*J[1],NjD[j],ND[0],ND[1],ttD1,NT1[0],NT1[1],*pu_[j][i],NN[0],NN[1]); // J^T B * Nj --kth AB-iterate
          MTK::fvec_const_prod_sum_rep(*Nj_[j][i], *pu_[j][i], 1.0, alpha_[k]); // alpha contains - sign already!
        } // end, field loop 
      } // end, AB/EXT sum of non-linear terms
      for ( j=0; j<nfields_; j++ ) var_list_[j]->next(); uelems_->next();
      for ( j=0; j<nfields_&&bConstAdvVel_; j++ ) cadv_[j]->next(); 
    } // end, element loop
  }

  if ( !bTimeDeriv_ ) {
    for ( i=0; i<nelems_; i++ ) {
      for ( j=0; j<nfields_; j++ ) {
        *pu_[j][i] = *Nj_[j][i];
      }
    }  
    return TRUE;
  }

  // Now, add in the BDF approximation for the time derivative:
  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  uelems_->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<nfields_; j++ ) *pu_[j][i] = 0.0;
    M = uelems_->member()->GetMassMatrix();
    for ( k=0; k<iorderbdf_; k++ )  {           // Do BDF contribution to RHS:
      coefbdf = beta_[k] * dti  ;
      for ( j=0; j<nfields_; j++ ) {
        ua      = var_list_[j]->member()->GetExpCoeffs(k);
        MTK::fvec_const_prod_sum_rep(*pu_[j][i], *ua, 1.0, coefbdf);
      }  // end of i-field loop
    } // end of BDF loop
    for ( j=0; j<nfields_; j++ ) {
      // Note: mass matrix already included in the nonlinear term update... 
      MTK::fvec_point_prod_rep(*pu_[j][i],*M);    // include mass operator in time deriv term
      MTK::fvec_add_rep(*pu_[j][i], *Nj_[j][i]);  // add BDF time deriv. and nonlin. term to advect update
    }  // end of i-field loop 
    for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
    uelems_->next();
  }  // end of elem loop

  return TRUE;

} // end of operator AdvectABBDF_D


//************************************************************************************
//************************************************************************************
// METHOD     : AdvectRKBDF
// DESCRIPTION: Performs advection partial update using RK integration of the advection
//              terms in the context of the OIFS splitting of the NS equations. The
//              RK-integrated solutions are then combined to determine the solution of
//              the advection partial update using a BDF scheme.
//
// ARGUMENTS  : pu_        : GVecList array containing result of the advect term + 
//                           time deritvative, if done. There must be at least nfields_ 
//                           members in this array.
//              Nj_        : Temp GVecList _array_ of length >= nfields_ for
//                           each component in pu_ array
//              time       : current time
//              dt         : time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdvect::AdvectRKBDF(GVecList *pu_, GVecList *Nj_, GDOUBLE time, GDOUBLE dt)
{
  char      *serr = "GAdvect::AdvectRKBDF: ";
  GINT      i, j, k;
  GDOUBLE   coefbdf, dti, delt;
  GVector   *cj[3]={NULL,NULL,NULL}, *Nj[3]={NULL,NULL,NULL};
  GVector   *M;
  Field2D   *field;

  dti          = dt == 0.0 ? 0.0 : 1.0/dt;

  // Integrate M dv_j^n+1/dt = - Sum_k v_k del_k v_j 
  // at each of iorderbdf_ time levels, from time level n-j+1 to time
  // level n+1, then use BDF to find advection partial update 
  // at n+1 time level:
  for ( k=0; k<nfields_; k++ ) {
    for ( i=0; i<nelems_; i++ ) {
      *pu_[k][i] = 0.0; // initialize var
    }
  }
  if ( !bDoAdvect_ ) return TRUE;
  
  // Do BDF determination of advect term at n+1:
  for ( i=0; i<nelems_; i++ ) {
    for ( k=0; k<iorderbdf_; k++ ) {  
      nsubcycles_ = (k+1)*nsubcycoifs_;
      coefbdf     =  beta_[k] * dti ;
      for ( j=0; j<irank_; j++ ) {
        cj[j] = var_list_[j]->member(i)->GetExpCoeffs(j);
      }
      field       = (*var_list_[0])[i];
      delt         = time + dt - field->GetTime(k);
      AdvectInt(Nj, cj, irank_, k, delt, i);
      for ( j=0; j<nfields_; j++ ) {
        MTK::fvec_const_prod_rep(*Nj[j], coefbdf);
        MTK::fvec_add_rep(*pu_[j][i], *Nj[j]);
        var_list_[j]->member(i)->TempUnlock(Nj[j]);
      }
    } // end of BDF loop   
    for ( j=0; j<nfields_; j++ ) {
      M           = (*var_list_[j])[i]->GetElement()->GetMassMatrix();
      MTK::fvec_point_prod_rep(*pu_[j][i], *M);
    }
  }  // end of elem-loop

  return TRUE;

} // end of operator AdvectRKBDF


//************************************************************************************
//************************************************************************************
// METHOD     : AdvectInt
// DESCRIPTION: Integrates the equation:
//              M dv_j/dt = v.del v . 
//              The equation is integrated over the 
//              interval, dt, using fixed time steps, h, which are a fraction
//              of this interval, s.t., dt = nsubcycles_ * h. Variable
//              nsubcycles_ must be computed/specified prior to entry, and
//              must be nonzero.
//
//
// ARGUMENTS  : 
//              uf         : final integrated result (there must be 3), overwritten
//              ui         : initial velocity components (may be NULL, but there
//              nf         : num. field elems in uf, ui
//              ilevel     : time level (not used)
//              dt         : time step
//              ie         : element id (index)
//                            must be 3), not overwritten
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdvect::AdvectInt(GVector *uf[], GVector *ui[], GINT nf, GINT ilevel,
                              GDOUBLE dt, GINT  ie)
{
  GINT     j, n;
  GBOOL    bOk=TRUE;
  GDOUBLE  h;
  GVector  *ua[3]={NULL,NULL,NULL};

  h = dt / nsubcycles_;
  for ( j=0; j<nf; j++ ) {
    *yt1[j][ie]  = *ui[j];   // make deep copy of initial state
  }

  // Note: yt1 is temp space, that must _not_ be overwritten in 
  //       calls to FixedStepAdvRKK method
  for ( n=0, bOk; n<nsubcycles_-1 && bOk; n++ ) {
    for ( j=0; j<nf; j++ ) ua[j] = yt1[j][ie];
    bOk = FixedStepAdvRKK(uf, ua, nf, ilevel, h, ie);
    for ( j=0; j<nf; j++ ) *yt1[j][ie] = *uf[j];
  }
  for ( j=0; j<nf; j++ ) ua[j] = yt1[j][ie];
  bOk = bOk && FixedStepAdvRKK(uf, ua, nf, ilevel, h, ie);

  return bOk;

} // end of method AdvectInt


//************************************************************************************
//************************************************************************************
// METHOD     : FixedStepAdvRKK
// DESCRIPTION: Takes one fixed stepsize time step using RKK
//              to solve equation:
//                dv_i/dt = -M^-1 (v.del)v_i
//              on element, ie. Since RHS is indep. of time, we can
//              employ the RK-K algorithm in Canuto, etal (1988), Springer.
//
//              Note: advection operator must be available for each
//                    element, via VectorOp class.
//
// ARGUMENTS  : 
// ARGUMENTS  : 
//              uf         : array of final integrated result, of size nf
//              ui         : array of initial values, of size nf
//              nf         : size of uf, ui arrays
//              ilevel     : starting time level of state, ui
//              h          : time step
//              ie         : element id (index)
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GAdvect::FixedStepAdvRKK(GVector *uf[], GVector *ui[], GINT nf, GINT ilevel,
                                     GDOUBLE h, GINT  ie)
{
  GINT     j, k, s;
  GDOUBLE  dtfact;
  GVector  *ua[3]={NULL,NULL,NULL};

  // K1 and K2 are temp space. It may be faster to use only a single
  // element's worth for each component (obviate need for ie index). This
  // might be more efficient for cases where exp. order is the same
  // for all elements. But a check will be done to insure this, and, if the
  // exp. orders aren't the same, reallocation will occur. In this latter case,
  // the continual deletion/reallocation of temp space will become expensive....
  for ( j=0; j<nf; j++ ) *K1[j][ie] = *ui[j];
  if ( bConstAdvVel_ ) {
    for ( k=0; k<irank_; k++ ) ua[k] = (*cadv_[k])[ie]->GetExpCoeffs(ilevel);
    ((VectorOp*)vecop_[ie])->SetVec(ua[0],ua[1],ua[2]);
  }
  for ( s=iorderOIFS_RKK_; s>0; s-- ) { // RK due to Canuto, etal book 1988, Springer
    dtfact = h/((GDOUBLE)s);
    if ( !bConstAdvVel_ ) {
      for ( j=0; j<nf; j++ ) ua[j] = K1[j][ie];
      ((VectorOp*)vecop_[ie])->SetVec(ua[0],ua[1],ua[2]);
    }
    for ( j=0; j<nf; j++ ) {
      ((VectorOp*)vecop_[ie])->Advect(*K1[j][ie], *K2[j][ie]);          // F(u) = C(u) u
      MTK::fvec_const_prod_sum_rep(*K2[j][ie], *ui[j], -dtfact, 1.0);   // u^n - dt/j F(u)
      *K1[j][ie] = *K2[j][ie];                                          // u = u^n - dt/j F(u)
    }
  }
  for ( j=0; j<nf; j++ ) *uf[j] = *K1[j][ie];

  return TRUE;

} // end of method FixedStepAdvRKK


//************************************************************************************
//************************************************************************************
// METHOD     : SetTemp
// DESCRIPTION: Sets up the temporary fields 
//
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
GBOOL GAdvect::ResetExpandables()
{
  char     *serr = "GAdvect::ResetExpandables: ";
  GINT     i, j, num, ncurr, NN;
  GVector *cu[3]={NULL,NULL,NULL};
  Elem2D   *ue; 


  if ( uelems_ == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }

  num   = uelems_->size();
  vecop_.empty();
  for ( j=0; j<nfields_; j++ ) {
    yt1     [j].empty();
    K1      [j].empty();
    K2      [j].empty();
  }
  ncurr = vecop_.size();

  for ( j=0; j<nfields_; j++ ) var_list_[j]->start(NULL);
  for ( i=ncurr; i<num; i++ ) {
    NN = var_list_[0]->member(i)->GetExpCoeffs(0)->dim();
    ue = (*uelems_)[i];
    for ( j=0; j<irank_; j++ ) 
//  cu[j] = var_list_[j] && var_list_[j]->member(i) ? var_list_[j]->member(i)->GetExpCoeffs(0) : NULL;
    cu[j] = var_list_[j] != NULL ? var_list_[j]->member()->GetExpCoeffs(0) : NULL;

    // Operators:
    vecop_ .add(NULL,TRUE); vecop_[i] = new VectorOp(ue, cu[0], cu[1], cu[2]);

    // Fully-instantiated data:
    for ( j=0; j<nfields_&& (itime_type_ == TE_OIFS); j++ ) {
      yt1     [j].add(NULL,TRUE); yt1[j][i] = new GVector(NN);
      K1      [j].add(NULL,TRUE); K1 [j][i] = new GVector(NN);
      K2      [j].add(NULL,TRUE); K2 [j][i] = new GVector(NN);
    }
    for ( j=0; j<nfields_; j++ ) var_list_[j]->next();
  }

  // Delete unnecessary elements from lists:
  for ( i=num; i<ncurr; i++ ) {
    vecop_ .del(i);
    for ( j=0; j<nfields_&& (itime_type_ == TE_OIFS) ; j++ ) {
      yt1     [j].del(i);
      K1      [j].del(i);
      K2      [j].del(i);
    }
  }

  // Must now renumber/re-order the reference ids for lists:
  vecop_ .renumber();
  for ( j=0; j<nfields_&& (itime_type_ == TE_OIFS); j++ ) {
    yt1     [j].renumber();
    K1      [j].renumber();
    K2      [j].renumber();
  }

  nelems_ = num;
  uelems_->start(NULL);
  for ( i=0,bDealiasingEnabled_=TRUE; i<uelems_->size(); i++ ) {
    bDealiasingEnabled_ = bDealiasingEnabled_ && uelems_->member()->DealiasingEnabled();
    uelems_->next();
  }

  return TRUE;


} // end of method ResetExpandables


//************************************************************************************
//************************************************************************************
// METHOD     : GetBDFSum
// DESCRIPTION: Gets sum of BDF coefficients
//
// ARGUMENTS  : none.
// RETURNS    : none.
//************************************************************************************
GDOUBLE GAdvect::GetBDFSum()
{
  return gamma0_;
  
} // end of method GetBDFSum


//************************************************************************************
//************************************************************************************
// METHOD     : GetVecOp
// DESCRIPTION: Gets vector operator list
//
// ARGUMENTS  : none.
// RETURNS    : GVecOpList *
//************************************************************************************
GVecOpList *GAdvect::GetVecOp()
{
  return &vecop_;

} // end of method GetVecOp



//************************************************************************************
//************************************************************************************
// METHOD     : SetNSubcycles
// DESCRIPTION: sets max number of OIFS subcycles
//
// ARGUMENTS  : GINT nsub
// RETURNS    : none.
//************************************************************************************
void GAdvect::SetNSubcycles(GINT nsub)
{
  nsubcycoifs_ = nsub;

} // end of method SetNSubcycles


//************************************************************************************
//************************************************************************************
// METHOD     : SetAdvVel
// DESCRIPTION: sets predetermined advection velocity components
//
// ARGUMENTS  : *cadv[] : GFieldList * components representing adv. vel components
//              ncadv   : number of adv. vel. compoents in array cadv
// RETURNS    : none.
//************************************************************************************
void GAdvect::SetAdvVel(GFieldList *cadv[], GINT ncadv)
{
  char   *serr = "GAdvect::SetAdvVel: ";
  GINT   k;

  if ( ncadv != irank_ ) {
    cout << serr << "insufficient number of advection velocity components" << endl;
  }
  for ( k=0; k<ncadv; k++ ) {
    cadv_[k] = cadv[k];
  }
  bConstAdvVel_    = TRUE;
  bRotationalForm_ = FALSE;


} // end of method SetAdvVel


//************************************************************************************
//************************************************************************************
// METHOD     : SetDoDealiasing
// DESCRIPTION: sets do dealiasing flag
//
// ARGUMENTS  : bflag: flag to set to
// RETURNS    : none.
//************************************************************************************
void GAdvect::SetDoDealiasing(GBOOL bflag)
{
  bDoDealiasing_ = bflag;
} // end of method SetDoDealiasing


//************************************************************************************
//************************************************************************************
// METHOD     : ElemListChange
// DESCRIPTION: Checks element list against local pointer copy to
//              determine if element list has been changed.
// ARGUMENTS  : none
// RETURNS    : TRUE if list has changed; else FALSE
//************************************************************************************
GBOOL GAdvect::ElemListChange()
{
  char *serr = "GAdvect::ElemListChange: ";
  GINT  i=0, esz;

  if ( uelems_ == NULL ) {
    cout << serr << "NULL element list" << endl;
    exit(1);
  }
  esz = uelems_->size();

  if ( esz != nelems_ ) return TRUE;

  vecop_.start(NULL); uelems_->start(NULL);
  while ( i<esz && uelems_->member() == ((VectorOp*)vecop_.member())->GetElem() ) {
     i++;
     vecop_.next(); uelems_->next();
  }

  return ( i < esz );
} // end of method ElemListChange

