//************************************************************************************//
// Module       : gifilter.hpp
// Date         : 8/5/05 (DLR)
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                an interpolation-based filter (Fischer & Mullen C.R. Acad. Sci.
//                Paris, 332(1), 265 (2001).
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "gifilter.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GIFilter::GIFilter()
:
ismooth_delta_      (1),
alpha_              (0.05),               // a minumum
elem                (NULL)
{
  GINT j;
 
  for ( j=0; j<GDIM; j++ ) {
    deletehere_[j] = TRUE;
    basisGLL_  [j] = NULL;
    Fn_        [j] = NULL;
  }
  
} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
GIFilter::GIFilter(Elem2D *e, GDOUBLE alpha, GINT ismooth_delta_modes)
:
ismooth_delta_      (ismooth_delta_modes),
alpha_              (alpha),                 // turn it off
elem                (e)
{
  GINT j;

  for ( j=0; j<GDIM; j++ ) {
    deletehere_[j] = TRUE;
    basisGLL_  [j] = NULL;
    Fn_        [j] = NULL;
  }
  SetElem(elem);
} // end of constructor (2) method



//************************************************************************************
//************************************************************************************
// Destructor
GIFilter::~GIFilter()
{
  GINT j;

  for ( j=0; j<GDIM; j++ ) {
    if ( basisGLL_[j] != NULL && deletehere_[j] ) delete basisGLL_[j];
    if ( Fn_      [j] != NULL && deletehere_[j] ) delete Fn_      [j];
  }
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetStrength
// DESCRIPTION: Sets filter strength
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void GIFilter::SetStrength(GDOUBLE alpha)
{
  alpha_ = alpha;
} // end of method SetStrength


//************************************************************************************
//************************************************************************************
// METHOD     : SetDelta
// DESCRIPTION: Sets filter mode range to smooth
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void GIFilter::SetDelta(GINT idelta)
{
  ismooth_delta_ = idelta;
} // end of method SetDelta


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element, and with element information computes
//              the 1-d filters for GLL-based vectors. The total filter is computed
//              only when the OpVec_prod method (or operator*) is calledu, where
//              a tensor product effect of the 1d filters is computed acting
//              directly on the input vectors. 
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void GIFilter::SetElem(Elem2D *e)
{

  char    *serr = "GIFilter::SetElem: ";
  GINT    j, k, kequal, l, N, M;
  GDOUBLE f;
  GBOOL   bequal;
  GVector xid, *xiu;
  GMatrix Ftmp, Id, Iu;

  if ( e == NULL ) {
    cout << serr << "NULL element" << endl;
    exit(1);
  }

  elem = e;

  // Build interpolation operators, basis operators. Don't 
  // create new operators if we don't need them:
  deletehere_[0] = TRUE;
  N = elem->GetOrder(1);
  M = N - ismooth_delta_;
  basisGLL_[0] = new GLLBasis (M);                    // N-ismooth_delta_ GLL basis
  Fn_[0] = new GMatrix(N+1,N+1);                      // 1-d filter
  Id.Resize(M+1,N+1);                                 // interp to N-ismooth_delta_ basis
  Iu.Resize(N+1,M+1);                                 // interp from N-ismooth_delta_ basis
  xid.Resize(M+1);
  basisGLL_[0]->GetXiNodes(&xid);                     // get M-basis nodes
  elem->GetBasisObj(1)->EvalBasis(&xid,&Id);          // find interp op from basis PN->PM 
  xiu = elem->GetXiNodes(1);
  basisGLL_[0]->EvalBasis(xiu,&Iu);                   // find interp op from basis PM->PN

  *Fn_[0] = Iu * Id;                                  // PP = I_NM * I_MN , pseudo projector     
  for ( l=0; l<Fn_[0]->dim(2); l++ ) {
    for ( k=0; k<Fn_[0]->dim(1); k++ ) {              // Fn = alpha*PP + (1-alpha)*I
      f = (*Fn_[0])(k,l);
      (*Fn_[0])(k,l) = (k==l) ? alpha_*(f-1.0)+1.0 : alpha_*f;
    }
  }
//cout << serr << "Fn[0]=" << *Fn_[0] << endl;
  
  for ( j=1; j<GDIM; j++ ) {
    N = elem->GetOrder(j+1);
    M = N - ismooth_delta_;
    for ( k=0,bequal=FALSE; k<j && !bequal; k++ ) {
      bequal = ( elem->GetOrder(j+1) == elem->GetOrder(k+1) );
      kequal = k;
    }
   if ( bequal ) {
      deletehere_[j] = FALSE;
      basisGLL_  [j] = basisGLL_[kequal]; 
      Fn_        [j] = Fn_      [kequal];
      continue;
    }
    deletehere_[j] = TRUE;
    basisGLL_  [j] = new GLLBasis     (M);           // N-ismooth_delta_ GLL basis    
    Fn_        [j] = new GMatrix(N+1,N+1);           // 1-d filter
    Id.Resize(M+1,N+1);                              // interp to N-ismooth_delta_ basis
    Iu.Resize(M+1,N+1);                              // interp from N-ismooth_delta_ basis
    xid.Resize(M+1);
    basisGLL_[j]->GetXiNodes(&xid);                  // get M-basis nodes
    elem->GetBasisObj(j+1)->EvalBasis(&xid,&Id);     // find interp op from basis PN->PM 
    xiu = elem->GetXiNodes(j+1);
    basisGLL_[j]->EvalBasis(xiu,&Iu);                // find interp op from basis PM->PN
    *Fn_[j] = Iu * Id;                                // PP = I_NM * I_MN , pseudo projector
    for ( l=0; l<Fn_[0]->dim(2); l++ ) {
      for ( k=0; k<Fn_[0]->dim(1); k++ ) {           // Fn = alpha*PP + (1-alpha)*I
        f = (*Fn_[j])(k,l);
        (*Fn_[j])(k,l) = (k==l) ? alpha_*(f-1.0)+1.0 : alpha_*f;
      }
    }
    if ( j==1) { // take transpose for tensor product
      Ftmp.Resize(N+1,N+1); Ftmp = *Fn_[j];
      Ftmp.Transpose(*Fn_[j]);
    }
  }

  
} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
GVector  GIFilter::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);
  
  OpVec_prod(q, newq);
  
  return newq;

} // end of \* operator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : void
//************************************************************************************
void GIFilter::OpVec_prod(GVector &q, GVector &ret)  
{
  char *serr = "GIFilter::OpVec_prod: ";
  GBOOL    bRet = TRUE;

  if ( elem == NULL ) {
    cout << serr << "NULL element" << endl;
    exit(1);
  }

  switch ( elem->ElemType() ) {
    case DEFORMED_QUAD:
      bRet = Filter(q,ret);
      break;
    case RECT_QUAD:
      bRet = Filter(q,ret);
      break;
    case TRIANGULAR:
      bRet = Filter(q,ret);
      break;
    default:
      cout << serr << "Invalid element type" << endl;
      exit(1);
  }
  if ( !bRet ) {
    cout << serr << "Operator product failed" << endl;
    exit(1);
  }

} // end of method oerator OpVec_prod 


//************************************************************************************
//************************************************************************************
// METHOD     : Filter
// DESCRIPTION: 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GIFilter::Filter(GVector &v, GVector &vn)
{
  char    *serr = "GIFilter::Filter: ";
  GINT    j, n[GDIM];
  GVector *vtmp, vm;


  if ( alpha_ == 0.0 ) return TRUE;  // no filtering required.
  if ( elem == NULL ) return FALSE;

  vtmp = elem->GetTemp();
  for ( j=0; j<GDIM; j++ ) n[j] = elem->GetOrder(j+1)+1;

  if ( GDIM == 2 ) {
//  cout << serr << "v=" << v << endl;
//  cout << serr << "Fn[0]=" << *Fn_[0] << endl;
//  cout << serr << "Fn[1]=" << *Fn_[1] << endl;
//  vm.Resize(v.dim()); for ( j=0; j<v.dim(); j++ ) vm[j]=v[j];
    MTK::D2_X_D1(*Fn_[0], *Fn_[1], v, n[0], n[1], *vtmp, n[0], n[1], vn, n[0], n[1]);
//  cout << serr << "vn=" << vn << endl;
  }
  else if ( GDIM == 3 ) {
    cout << serr << "currently unsupported" << endl;
    exit(1);
  }
  elem->TempUnlock(vtmp);

  return TRUE;
} // end of method Filter


//************************************************************************************
//************************************************************************************
// METHOD     : GetElem
// DESCRIPTION: Gets member element pointer
// ARGUMENTS  : none.
// RETURNS    : member data pointer
//************************************************************************************
Elem2D *GIFilter::GetElem()
{
  return elem;
} // end of method GetElem
