//************************************************************************************//
// Module       : gllbasis.cpp
// Date         : 7/9/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a spectral element nodal basis
//                Note: This basis is the Gauss-Lobatto-Legendre
//                basis, defined as 
//                  h(xi)_j = -1/(N(N+1)*L_N(xi_j))  * (1-xi^2)*dL_N(xi)/dxi / (xi-xi_j)
//                where N is the order of the expansion, and L_N is the Legendre
//                polynomial of order N, and the xi_j are the nodal points. L_N
//                is obtained from the generalized Jacobi polynomial computed in
//                ComputeJacobi.
// Derived From :
// Modifications:
//************************************************************************************//
#include "gllbasis.hpp"
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <stdio.h>


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GLLBasis::GLLBasis(GINT  inOrder, GINT  MaxOrder)
:
Np                    (MIN(inOrder,MaxOrder)),
NpMax                 (MAX(inOrder,MaxOrder)),
kstop                 (128),
bNeedNodes            (TRUE),
bNeedWeights          (TRUE),
bNeedDerivMatrix      (TRUE),
bNeedBasis            (TRUE),
bNeedDBasis           (TRUE),
bNeedLegMat           (TRUE),
alpha                 (0.0),
beta                  (0.0),
ximin                 (-1.0),
ximax                 (1.0),
eps                   (1.0e-16)
{
  if ( Np < 1 )
  {
    cout << "GLLBasis::GLLBasis: invalid expansion order" << endl;
    exit(1);
  }
  Resize(NpMax);
} // end of constructor method


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
GLLBasis::GLLBasis()
:
Np                    (4),
NpMax                 (32),
kstop                 (128),
bNeedNodes            (TRUE),
bNeedWeights          (TRUE),
bNeedDerivMatrix      (TRUE),
bNeedBasis            (TRUE),
bNeedDBasis           (TRUE),
bNeedLegMat           (TRUE),
alpha                 (0.0),
beta                  (0.0),
ximin                 (-1.0),
ximax                 (1.0),
eps                   (1.0e-16)
{
  if ( Np < 1 )
  {
    cout << "GLLBasis::GLLBasis: invalid expansion order" << endl;
    exit(1);
  }
  Resize(NpMax);
} // end of constructor method


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
GLLBasis::GLLBasis(GINT  Order)
:
Np                    (Order),
NpMax                 (Order),
kstop                 (128),
bNeedNodes            (TRUE),
bNeedWeights          (TRUE),
bNeedDerivMatrix      (TRUE),
bNeedBasis            (TRUE),
bNeedDBasis           (TRUE),
bNeedLegMat           (TRUE),
alpha                 (0.0),
beta                  (0.0),
ximin                 (-1.0),
ximax                 (1.0),
eps                   (1.0e-16)
{
  if ( Np < 1 )
  {
    cout << "GLLBasis::GLLBasis: invalid expansion order" << endl;
    exit(1);
  }
  Resize(NpMax);

} // end of constructor method


// Copy constructor method
GLLBasis::GLLBasis(const GLLBasis &b)
{
   // copy data:
    alpha    = b.alpha;
    beta     = b.beta;
    ximin    = b.ximin;
    ximax    = b.ximax;
    eps      = b.eps;
    Np       = b.Np;
    NpMax    = b.NpMax;
    kstop    = b.kstop;
    bNeedNodes = b.bNeedNodes;
    bNeedWeights = b.bNeedWeights;
    bNeedDerivMatrix = b.bNeedDerivMatrix;
    bNeedBasis   = b.bNeedBasis; 
    bNeedDBasis  = b.bNeedDBasis;
    bNeedLegMat  = b.bNeedLegMat;


    //  matrices, and node data:
    xiNodes     = b.xiNodes;
    Weights     = b.Weights;
    Pn          = b.Pn;
    dPn         = b.dPn;
    Phi         = b.Phi;
    dPhi        = b.dPhi;
    MassMatrix  = b.MassMatrix;
    StiffMatrix = b.StiffMatrix;
    LegMatrix   = b.LegMatrix;

}

//************************************************************************************
//************************************************************************************
// Destructor
GLLBasis::~GLLBasis()
{
}

//************************************************************************************
//************************************************************************************
// Assignment operator method (1)
void GLLBasis::operator=(const GLLBasis &b)
{
  if ( &b != this ) 
  {
   // copy data:
    ximin        = b.ximin;
    ximax        = b.ximax;
    eps          = b.eps;
    Np           = b.Np;
    NpMax        = b.NpMax;
    kstop        = b.kstop;
    bNeedNodes   = b.bNeedNodes;
    bNeedWeights = b.bNeedWeights;
    bNeedDerivMatrix = b.bNeedDerivMatrix;
    bNeedBasis   = b.bNeedBasis; 
    bNeedDBasis  = b.bNeedDBasis;
    bNeedLegMat  = b.bNeedLegMat;


    //  matrices, and node data:
    xiNodes     = b.xiNodes;
    Weights     = b.Weights;
    Pn          = b.Pn;
    dPn         = b.dPn;
    Phi         = b.Phi;
    dPhi        = b.dPhi;
    MassMatrix  = b.MassMatrix;
    StiffMatrix = b.StiffMatrix;
    LegMatrix   = b.LegMatrix;
  }

}

//************************************************************************************
//************************************************************************************
// METHOD     : GetXimin
// DESCRIPTION: 
// RETURNS    :  GQUAD element left boundary
//************************************************************************************
GDOUBLE GLLBasis::GetXimin()
{
  return (GDOUBLE) ximin;
} // end of method GetXimin


//************************************************************************************
//************************************************************************************
// METHOD     : GetXimax
// DESCRIPTION: 
// RETURNS    :  GQUAD element right boundary
//************************************************************************************
GDOUBLE GLLBasis::GetXimax()
{
  return (GDOUBLE) ximax;
} // end of method GetXimax



//************************************************************************************
//************************************************************************************
// METHOD     : GetOrder
// DESCRIPTION: 
// RETURNS    :  GINT  element expansion order
//************************************************************************************
GINT  GLLBasis::GetOrder()
{
  return Np;
} // end of method GetOrder


//************************************************************************************
//************************************************************************************
// METHOD     : GetXiNodes
// DESCRIPTION: 
// RETURNS    :  GVector*
//************************************************************************************
GVector *GLLBasis::GetXiNodes(GVector *ret)
{
  if ( ret == NULL  ) return NULL;
  if ( ret->dim() < xiNodes.dim() ) return NULL;
  if ( bNeedNodes ||  bNeedWeights )
    if ( !ComputeNodes() ) return NULL;

  GINT  i;
  GDOUBLE *fptr=ret->Data();
  GQUAD *qptr=xiNodes.Data();

  for ( i=0; i<Weights.dim(); i++ )
    *(fptr+i) = (GDOUBLE) (*(qptr+i));


  return ret;
} // end of method GetXiNodes


//************************************************************************************
//************************************************************************************
// METHOD     : GetXiNodes (2)
// DESCRIPTION:
// RETURNS    :  GQUAD *
//************************************************************************************
GDOUBLE *GLLBasis::GetXiNodes(GDOUBLE *ret, GINT  num)
{

  if ( ret == NULL || num < xiNodes.dim() ) return NULL;
  if ( bNeedNodes || bNeedWeights )
    if ( !ComputeNodes() ) return NULL;
  GINT  i;
  GQUAD *qptr=xiNodes.Data();


  for ( i=0; i<xiNodes.dim(); i++ )
    ret[i] = (GDOUBLE)(*(qptr+i));

  return ret;
} // end of method GetXiNodes (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetXiNodes (3)
// DESCRIPTION:
// RETURNS    :  GQUAD *
//************************************************************************************
GTVector<GQUAD> *GLLBasis::GetXiNodes()
{

  return &xiNodes;
} // end of method GetXiNodes (3)



//************************************************************************************
//************************************************************************************
// METHOD     : GetWeights
// DESCRIPTION: 
// RETURNS    :  GVector *
//************************************************************************************
GVector *GLLBasis::GetWeights(GVector *ret)
{
  if ( ret == NULL  ) return NULL;
  if ( ret->dim() < Weights.dim() ) return NULL;
  if ( bNeedNodes || bNeedWeights )
    if ( !ComputeNodes() ) return NULL;

  GINT  i;
  GDOUBLE *fptr=ret->Data();
  GQUAD *qptr=Weights.Data();

  for ( i=0; i<Weights.dim(); i++ )
    *(fptr+i) = (GDOUBLE) (*(qptr+i));


  return ret;
} // end of method GetWeights 


//************************************************************************************
//************************************************************************************
// METHOD     : GetWeights (2)
// DESCRIPTION:
// RETURNS    :  GQUAD *
//************************************************************************************
GDOUBLE *GLLBasis::GetWeights(GDOUBLE *ret, GINT  num)
{

  if ( ret == NULL || num < Weights.dim() ) return NULL;
  if ( bNeedNodes || bNeedWeights )
    if ( !ComputeNodes() ) return NULL;

  GINT  i;
  GQUAD *qptr=Weights.Data();

  for ( i=0; i<Weights.dim(); i++ )
    *(ret+i) = (GDOUBLE) (*(qptr+i));

  return ret;
} // end of method GetWeights (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetWeights (3)
// DESCRIPTION:
// RETURNS    :  GQUAD *
//************************************************************************************
GTVector<GQUAD> *GLLBasis::GetWeights()
{

  return &Weights;
} // end of method GetWeights (3)


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetMassMatrix
// DESCRIPTION: 
// RETURNS    :  GMatrix *
//************************************************************************************
GMatrix *GLLBasis::GetMassMatrix(GMatrix *ret)
{
  if ( ret == NULL) return NULL;
  if ( ret->dim(1) < MassMatrix.dim(1) || ret->dim(2) < MassMatrix.dim(2) ) return NULL;

  GINT  i, j;
  GDOUBLE *fptr;
  GQUAD *qptr;
  GBasisVector  **v=MassMatrix.Data();

  for ( i=0; i<MassMatrix.dim(1); i++ )
  {
    fptr = (*ret+i)->Data();
    qptr = MassMatrix.Data() ;
    for ( j=0; j<MassMatrix.dim(2); j++ )
      *(fptr+j) = (GDOUBLE)((*v+i)->Data()+j);
  }

  return ret;
} // end of method GetMassMatrix
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : GetStiffMatrix
// DESCRIPTION: Note:  
// RETURNS    :  GBasisMatrix *
//************************************************************************************
GMatrix *GLLBasis::GetStiffMatrix(GMatrix *ret)
{
  if ( ret == NULL) return NULL;
  if ( ret->dim(1) < StiffMatrix.dim(1) || ret->dim(2) < StiffMatrix.dim(2) ) return NULL;

  GINT  i, j; 

  for ( i=0; i<StiffMatrix.dim(1); i++ ) 
    for ( j=0; j<StiffMatrix.dim(2); j++ ) 
      (*ret)(i,j) = (GDOUBLE)StiffMatrix(i,j);

  return ret;
} // end of method GetStiffMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : GetLegMatrix
// DESCRIPTION: Gets/fills Legendre poly. matrix: Lp = Lp(nLegOrder,iNodeIndex)
// RETURNS    :  GBasisMatrix *
//************************************************************************************
GMatrix *GLLBasis::GetLegMatrix(GMatrix *ret)
{
  if ( ret == NULL) return NULL;
  if ( ret->dim(1) < LegMatrix.dim(1) || ret->dim(2) < LegMatrix.dim(2) ) return NULL;
  if ( bNeedLegMat && !ComputeLegendreMatrix() ) return NULL;

  GINT  i, j;

  for ( i=0; i<LegMatrix.dim(1); i++ )
    for ( j=0; j<LegMatrix.dim(2); j++ )
      (*ret)(i,j) = (GDOUBLE)LegMatrix(i,j);

  return ret;
} // end of method GetLegMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : GetDerivMatrix
// DESCRIPTION: 
// RETURNS    :  GBasisMatrix *
//************************************************************************************
GMatrix *GLLBasis::GetDerivMatrix(GMatrix *ret)
{
  if ( ret == NULL) return NULL;
  if ( ret->dim(1) < dPhi.dim(1) || ret->dim(2) < dPhi.dim(2) ) return NULL;

  if ( bNeedNodes )
    if ( !ComputeNodes() ) return NULL;
  if ( bNeedDerivMatrix )
    if ( !ComputeDerivMatrix() ) return NULL;

  GINT  i, j; 

  for ( i=0; i<dPhi.dim(1); i++ ) 
    for ( j=0; j<dPhi.dim(2); j++ ) 
      (*ret)(i,j) = (GDOUBLE)dPhi(i,j);
  
  return ret;
} // end of method GetDerivMatrix


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetDerivMatrix (2)
// DESCRIPTION:
// RETURNS    :  GQUAD *
//************************************************************************************
GDOUBLE *GLLBasis::GetDerivMatrix(GDOUBLE *ret, GINT  num)
{
  GINT  i,j, n=0;

  if ( ret == NULL || num < dPhi.dim(1)*dPhi.dim(2) ) return NULL;
  if ( bNeedNodes )
    if ( !ComputeNodes() ) return NULL;
  if ( bNeedDerivMatrix )
    if ( !ComputeDerivMatrix() ) return NULL;

  for ( i=0; i< dPhi.dim(1); i++ )
  {
    for ( j=0; j< dPhi.dim(2); j++ )
    {
      ret[n] = (GDOUBLE) dPhi(i,j);
      n++;
    }
  }
  return ret;
} // end of method GetDerivMatrix (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetBasisAtNodes
// DESCRIPTION: Computes Gauss-Lobatto Legendre basis
//              as a function of  nodes in the parent domain. 
//              The returned quantity is a pointer to a matrix
//              Phi_i(xi_j)==Phi(i,j),
//              where xi_j is the jth node element, and
//              Phi_i is given by (see Canuto, et. al.):
//
//              Phi_j (x) = (N*(N+1)Pn(xi_j))^-1  * (1-x^2)*dPn/dx(x)/(x - xi_j)
//              
// RETURNS    :  non NULL &Phi; else NULL 
//************************************************************************************
GBasisMatrix *GLLBasis::GetBasisAtNodes(GBasisMatrix *ret)
{
  if ( bNeedNodes || bNeedWeights )
    if ( !ComputeNodes() ) return NULL;
 
  if ( bNeedBasis       && !ComputeBasisAtNodes() ) return NULL;


  if ( ret == NULL) return NULL;
  if ( ret->dim(1) < Phi.dim(1) || ret->dim(2) < Phi.dim(2) ) return NULL;

  GINT  i;
  GDOUBLE *fptr=ret->Data();
  GQUAD *qptr=Phi.Data();
   
  for ( i=0; i<Phi.dim(1) * Phi.dim(2); i++ )
    *(fptr+i) = (GDOUBLE)(*(qprt+i));

  return ret;
} // end of method GetBasisAtNodes


//************************************************************************************
//************************************************************************************
// METHOD     : SetXiDomain
// DESCRIPTION: 
// RETURNS    :  none
//************************************************************************************
void GLLBasis::SetXiDomain(GQUAD min, GQUAD max)
{
  ximin = MIN(min,max);
  ximax = MAX(min,max);

} // end of method SetXiDomain
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : SetOrder
// DESCRIPTION: 
// RETURNS    :  none
//************************************************************************************
void GLLBasis::SetOrder(GINT  order)
{
  if ( order != Np )
  {
     Resize(order);
     bNeedNodes  = TRUE;
     bNeedWeights = TRUE;
     bNeedBasis  = TRUE;
     bNeedDBasis = TRUE;
     bNeedDerivMatrix = TRUE;
  }
  Np = order;
} // end of method SetOrder


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
//************************************************************************************
void GLLBasis::DeleteDynamic()
{
/*
  if ( xiNodes     != NULL ) delete xiNodes;
  if ( Weights     != NULL ) delete Weights;
  if ( Phi         != NULL ) delete Phi;
  if ( dPhi        != NULL ) delete dPhi;
  if ( MassMatrix  != NULL ) delete MassMatrix;
  if ( StiffMatrix != NULL ) delete StiffMatrix;
  xiNodes     = NULL;
  Weights     = NULL;
  Phi         = NULL;
  dPhi        = NULL;
  MassMatrix  = NULL;
  StiffMatrix = NULL;
*/
} // end of method DeleteDynamic
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::Resize(GINT  newOrder)
{

  // No resizing necessary if already less than
  // previously allocated quantities:

  NpMax = Np = newOrder;

  //  Resize xiNodes:
  xiNodes.Resize(Np+1);

  //  Resize Weights:
  Weights.Resize(Np+1);

  //  Resize Pn:
  Pn.Resize(Np+1);

  //  Resize dPn:
  dPn.Resize(Np+1);

  //  Resize basis Phi:
  Phi.Resize(Np+1,Np+1);

  //  Resize basis dPhi:
  dPhi.Resize(Np+1,Np+1);

  //  Resize MassMatrix:
  MassMatrix.Resize(Np+1,Np+1);

  //  Resize StiffMatrix:
  StiffMatrix.Resize(Np+1,Np+1);

  //  Resize LegMatrix:
  LegMatrix.Resize(Np+1,Np+1);

  return TRUE;

} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeNodes 
// DESCRIPTION: Computes nodes and weights based on a Gauss-Lobatto Legendre integration scheme
//              Note: nodes are computed from smalles to largest xi
//              Note: this method was taken largely from Canuto et al, 1987, Appendix C.
//              Note: this method will be generalized to accept this as a base class,
//                    and derive other basis types from it....
//              Note: Legendre polynomials of order Np, Np-1 and Np-2 and their
//                    derivatives are computed here for use here and in 
//                    computing integrals. These polynomials are used in
//                    forming basis functions, Phi, and their derivatives,
//                    dPhi, evaluated at each node.
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::ComputeNodes()
{
  if ( !bNeedNodes && !bNeedWeights ) return TRUE;

  if ( Np < 1 ) return FALSE;
  if ( Np == 1 ) {
    xiNodes(0) = ximin;
    xiNodes(1) = ximax;
    bNeedNodes = FALSE;
    return(ComputeWeights());
  }
 
  GINT  i, j, k, nh, np1;
//GQUAD rv;
  GQUAD det, pnp, pnp1p, pnp1m, rp, rm, dth, cd, sd, cs, ss, 
        pn , pnp1, pdnp1, pdn, pnm1, pdnm1, a, b, poly, pder,
        pdnp1p, pnm1p,  pdnp1m, pnm, pdnm, pnm1m, pdnp,
        recsum, x, delx, cssave, error;
 
  alpha = 0.0;
  beta  = 0.0;  //alpha=beta=0 ==> Legendre fcn basis 

//rv    = 1.0 + alpha;
  np1   = Np + 1;

  ComputeJacobi(np1,alpha, beta, pnp1p, pdnp1p, pnp, pdnp, pnm1p, pdnm1, ximax);
  ComputeJacobi(np1,alpha, beta, pnp1m, pdnp1m, pnm, pdnm, pnm1m, pdnm1, ximin);
  det  = pnp*pnm1m - pnm*pnm1p;
  rp   = -pnp1p;
  rm   = -pnp1m;
  a    = ( rp*pnm1m - rm*pnm1p ) / det;
  b    = ( rm*pnp   - rp*pnm   ) / det;

  // order nodes from largest to smallest:
  xiNodes (0) = ximax;
  xiNodes(Np) = ximin;
  nh = ( Np + 1 ) / 2;

  // set up recursion relation for the initial guesses for roots:
  dth = 3.1415926535898/( 2.0*Np + 1.0 );
  cd  = cos(2.0*dth);
  sd  = sin(2.0*dth);
  cs  = cos    (dth);
  ss  = sin    (dth);

  // compute the first half of the roots by polynomial deflation:
  for ( j=1; j<nh+1; j++ ) 
  {
    x = cs;
    for ( k=0; k<kstop; k++ )
    {
      ComputeJacobi(np1, alpha, beta, pnp1, pdnp1, pn, pdn, pnm1, pdnm1, x);
      poly = pnp1 + a*pn + b*pnm1;
      pder = pdnp1 + a*pdn + b*pdnm1;
      recsum = 0.0;
      for ( i=0; i<j; i++ )
      {
        recsum += (  1.0/(x - xiNodes(i)) );
      }
      delx = -poly/(pder - recsum*poly);
      x += delx;
      error = fabs(delx)/ fabs(x);    // MIN(fabs(ximin),fabs(ximax));
      if ( error < eps ) break;
    }
    xiNodes(j) = x;
    cssave     = cs*cd - ss*sd;
    ss         = cs*sd + ss*cd;
    cs         = cssave;
  }

  
  // Symmetrize the nodes:  NOTE: this is correct only for the cases
  // where (xmin,xmax) = (-y, y); ie, is symmetric interval.
  for ( i=0; i<nh; i++ )
  {
    xiNodes(np1-i-1) = -xiNodes(i);
  }

  bNeedNodes   = FALSE;
  bNeedWeights = TRUE;

  if ( Np == ( 2*(Np/2) ) ) 
  {
    xiNodes(nh) = 0.0;
  }


  // re-order from smallest to largest:
  GQUAD txi;
  for ( i=0; i<nh; i++ )
  {
    txi           = xiNodes(Np-i);
    xiNodes(Np-i) = xiNodes(i);
    xiNodes   (i) = txi;
  }

  return(ComputeWeights());

} // end of method ComputeNodes


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeWeights
// DESCRIPTION: 
//            : NOTE: method not really intended to be called publicly; 
//              it should be called only when ComputeNodes is called. 
//              For other bases, this method will change, however.
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::ComputeWeights()
{
  if ( Np < 1 ) return FALSE;
#if 0
  if ( Np == 1 ) {
    Weights(0) = 0.5;
    Weights(1) = 0.5;
    bNeedWeights = FALSE;
    return TRUE;
  }
#endif
 
  GINT  i;
  GQUAD fact = 2.0/(Np*(Np + 1.0));
  GQUAD ppn, pder, pm1, pdm1, pm2, pdm2;

  for ( i=0; i<Np+1; i++ ) 
  {
    ComputeJacobi(Np, alpha, beta, ppn, pder,pm1, pdm1, pm2, pdm2, xiNodes(i));
    Pn (i) = ppn;    
    dPn(i) = pder;    
    Weights(i) = (Pn(i)==0.0)?0.0:fact/(Pn(i)*Pn(i)); // Note: Pn computed in ComputNodes
  }
  bNeedWeights = FALSE;
  return TRUE;
} // end of method ComputeWeights


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : ComputeBasisAtNodes
// DESCRIPTION:
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::ComputeBasisAtNodes()
{
  if ( bNeedNodes || bNeedWeights )
    if ( !ComputeNodes() ) return NULL;


  GINT  i, j;
  GQUAD ppn_i, pder_i, pm1, pdm1, pm2, pdm2, ppn_j, pder_j;
  GQUAD fact=1.0/(Np*(Np+1.0)), gfact;

  Phi = 0.0;

  for ( i=0; i< Np+1; i++ )
  {
    ComputeJacobi(Np, alpha, beta, ppn_i, pder_i,pm1, pdm1, pm2, pdm2, xiNodes(i));
    for ( j=0; j< Np+1; j++ )
    {
      Phi(i,j) = 1.0;
      if ( i != j )
      {
        ComputeJacobi(Np, alpha, beta, ppn_j, pder_j,pm1, pdm1, pm2, pdm2, xiNodes(j));
        gfact = (1-xiNodes(j)*xiNodes(j))/(xiNodes(j)-xiNodes(i));
        Phi(i,j) = fact/ppn_i * gfact * pder_j;
      }
    }
  }
  bNeedBasis = FALSE;

} // end of method ComputeBasisAtNodes
#endif

//************************************************************************************
//************************************************************************************
// METHOD     : ComputeDerivMatrix
// DESCRIPTION:
//            NOTE: ComputeWeights (ComputeNodes) must have been called
//                  prior to entering this method, s.t. the Pn_i have been
//                  calculated. 
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::ComputeDerivMatrix()
{
  if ( bNeedWeights ) return FALSE;

  GINT  l, j;
  GQUAD delxi;

  dPhi = 0.0;
  // Note: index j sweeps over basis number; l sweeps over node number

  for ( j=0; j<Np+1; j++ )
  {
    for ( l=0; l<j; l++ )
    {
      delxi      = xiNodes(l) - xiNodes(j);
      dPhi (l,j) =  Pn(l)/(Pn(j)*delxi+TINY);
    }
    for ( l=j+1; l<Np+1; l++ )
    {
      delxi       = xiNodes(l) - xiNodes(j);
      dPhi  (l,j) =  Pn(l)/(Pn(j)*delxi+TINY);
    }
  }
  dPhi (0 ,0 ) = -0.25*Np*(Np + 1.0);
  dPhi (Np,Np) = -dPhi(0,0);


  bNeedDerivMatrix = FALSE;
  return TRUE;

} // end of method ComputeDerivMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeJacobi
// DESCRIPTION: 
// RETURNS    :  
//************************************************************************************
void GLLBasis::ComputeJacobi(GINT  &N, GQUAD  alpha , GQUAD  beta  , 
                                       GQUAD &poly  , GQUAD &pder  , GQUAD &polym1, 
                                       GQUAD &pderm1, GQUAD &polym2, GQUAD &pderm2, GQUAD &x)
{

  GINT  k; 
  GQUAD apb  , polylist, pderlist, rv                ;
  GQUAD polyn, pdern   , psave   , pdsave  , fk      ;
  GQUAD a1   , a2      , a3      , a4      , b3      ;


  apb = alpha + beta;
  rv = 1.0 + alpha;

  poly = 1.0;
  pder = 0.0;

  if ( N == 0 ) return;

  polylist = poly;
  pderlist = pder;
  poly     = rv * x;
  pder     = rv;
  
  if ( N == 1 ) return;

  for ( k=2; k<=N; k++ ) 
  {
    fk = double(k);
    a1 = 2.0*fk*(fk+apb)*(2.0*fk+apb-2.0);
    a2 = (2.0*fk+apb-1.0)*(alpha*alpha -beta*beta);
    b3 = 2.0*fk+apb-2.0;
    a3 = b3*(b3+1.0)*(b3+2.0);
    a4 = 2.0*(fk+alpha-1.0)*(fk+beta-1.0)*(2.0*fk+apb);
    polyn    = ((a2+a3*x)*poly - a4*polylist)/a1;
    pdern    = ((a2+a3*x)*pder - a4*pderlist+a3*poly)/a1;
    psave    = polylist;
    pdsave   = pderlist;
    polylist = poly;
    poly     = polyn;
    pderlist = pder;
    pder     = pdern;
  }

  polym1 = polylist;
  pderm1 = pderlist;
  polym2 = psave;
  pderm2 = pdsave; 

} // end of method ComputeJacobi


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : ComputeMassMatrix
// DESCRIPTION: Computes mass matrix for this basis.
//              Method uses Gaussian integration, so the
//              nodes and the weights for the current Np order
//              must be calculated.
//
//              The mass matrix is defined as:
//              
//              M_(i,j )= Integral(ximin,ximax) { Phi_i(xi) Phi_j(xi) dxi }
//                      ~ Sum { w_k * Phi_i (xi_k) Phi_j(xi_k) }   (Gaussian quadrature)
//              where w_k are the weights for the basis associated with each node, xi_k.
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::ComputeMassMatrix()
{
  if ( bNeedNodes || bNeedWeights )
  {
    if ( !ComputeNodes() ) return FALSE;
  }
  if ( bNeedBasis && GetBasisAtNodes(&Phi)==NULL ) return FALSE;
 
  GINT  i;

  MassMatrix = 0.0;
 
  for ( i=0; i<Np+1; i++ ) 
    MassMatrix(i,i) = Weights(i);

  return TRUE;
} // end of method ComputeMassMatrix
#endif
 

//************************************************************************************
//************************************************************************************
// METHOD     : ComputeStiffMatrix
// DESCRIPTION: Computes stiffness matrix for this basis.
//              Method uses Gaussian integration, so the
//              nodes and the weights for the current Np order
//              must be calculated.
//
//              The stiffness matrix is defined as:
//              
//              A_(i,j )= Integral(ximin,ximax) { dPhi_i(xi)/dxi dPhi_j(xi)/dxi dxi }
//                      ~ Sum { w_k * dPhi_i(xi_k)/dxi dPhi_j(xi_k)/dxi }   (Gaussian quadrature)
//              where w_k are the weights for the basis associated with each node, xi_k.
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::ComputeStiffMatrix()
{
  if ( bNeedNodes || bNeedWeights )
  {
    if ( !ComputeNodes() ) return FALSE;
  }
  if ( bNeedDerivMatrix )
    if ( !ComputeDerivMatrix() ) return FALSE;

  GINT  i, j, k;

  StiffMatrix = 0.0;

  // Could use matrix algebra here....
  for ( i=0; i<Np+1; i++ ) 
  {
    for ( j=0; j<Np+1; j++ ) 
    {
       for ( k=0; k<Np+1; k++ )
       {
         StiffMatrix(i,j) += Weights(k)*dPhi(k,i)*dPhi(k,j) ;  
       }
    } 
  } 
  return TRUE;
} // end of method ComputeStiffMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : Solve
// DESCRIPTION: Computes all quantities, nodes, weights, mass and
//              stiffness matrices, and Phi, and dPhi.
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL GLLBasis::Solve()
{
  if ( bNeedNodes || bNeedWeights )
  {
    if ( !ComputeNodes() ) return FALSE;
  }

  if ( bNeedDerivMatrix )
    if ( !ComputeDerivMatrix() ) return FALSE;

  if ( !ComputeStiffMatrix() ) return FALSE; // stiffness matrix computed; dPhi also computed.

  return TRUE;

} // end of method Solve



//************************************************************************************
//************************************************************************************
// METHOD     : EvalBasis (1)
// DESCRIPTION: 
//                h(xi)_i = 1/(N(N+1)*L_N(xi_i))  * (1-xi^2)*dL_N(xi)/dxi / (xi-xi_i)
// RETURNS    :  
//************************************************************************************
GDOUBLE GLLBasis::EvalBasis (GINT  i, GDOUBLE eta)
{
  GQUAD ppn_i, pm1, pdm1, pm2, pdm2, ppn_xi, pder_xi, del;
  GQUAD fact=-1.0/(Np*(Np+1.0)), gfact, fRet, xi=(GQUAD)eta;

  if ( (bNeedNodes || bNeedWeights) &&  !ComputeNodes() ) {
    cout << "GLLBasis::EvalBasis (1): basis data incomplete" << endl;
    exit(1);
  }

  fRet = 1.0;
  del  = xi-xiNodes(i);
  if ( (GDOUBLE)xi < (GDOUBLE)ximin || (GDOUBLE)xi > (GDOUBLE)ximax ) fRet = 0.0;
  else if ( fabs(del) > 10.0*TINY )
  {
    ppn_i = Pn(i);
    ComputeJacobi(Np, alpha, beta, ppn_xi, pder_xi,pm1, pdm1, pm2, pdm2, xi);
    gfact = (1.0-xi*xi)/del; 
    fRet  = fact * gfact * pder_xi/ppn_i;
  }
  return (GDOUBLE)fRet;

} // end of method EvalBasis (1)


//************************************************************************************
//************************************************************************************
// METHOD     : EvalBasis (2)
// DESCRIPTION: 
//                h(xi)_j = 1/(N(N+1)*L_N(xi_j))  * (1-xi^2)*dL_N(xi)/dxi / (xi-xi_j)
// RETURNS    :  
//************************************************************************************
GVector *GLLBasis::EvalBasis (GINT  i, GVector *eta, GVector *vret)
{
  GINT  j;
  GQUAD ppn_i, pm1, pdm1, pm2, pdm2, ppn_xi, pder_xi;
  GQUAD fact=-1.0/(Np*(Np+1.0)), gfact, fRet, xi;

  if ( eta == NULL || vret == NULL ) return NULL;
  if ( (bNeedNodes || bNeedWeights) &&  !ComputeNodes() ) {
    cout << "GLLBasis::EvalBasis (2): basis data incomplete" << endl;
    exit(1);
  }

  for ( j=0; j<eta->dim(); j++ )
  {
    xi = (*eta)(j);
    fRet = 1.0;
    if ( (GDOUBLE)xi < (GDOUBLE)ximin || (GDOUBLE)xi > (GDOUBLE)ximax ) fRet = 0.0;
    else if ( fabs(xi-xiNodes(i)) > 100.0*TINY )
    {
      ppn_i = Pn(i);
      ComputeJacobi(Np, alpha, beta, ppn_xi, pder_xi,pm1, pdm1, pm2, pdm2, xi);
      gfact = (1.0-xi*xi)/(xi-xiNodes(i)+TINY); 
      fRet  = fact * gfact * pder_xi/(ppn_i+TINY);
    }
    (*vret)(j) = (GDOUBLE)fRet;
  }
  return vret;

} // end of method EvalBasis (2)


//************************************************************************************
//************************************************************************************
// METHOD     : EvalBasis (3)
// DESCRIPTION: Evaluates basis at input parent domain points , eta_i
//              For Gauss-Lobatto, the basis is:
//              h_j(eta) = -1/(Np*(Np+1)) * (1-eta**2) dL_Np (eta)dxi / (L_Np(xi_j) (eta-xi_j))
// RETURNS    : GMatrix, M_ij = h_j(eta_i)
//              NOTE: Prob better to have eta as an GBuffer...
//************************************************************************************
GMatrix *GLLBasis::EvalBasis (GVector *eta, GMatrix *mret)
{
  GINT  i, j;
  GQUAD ppn_j, pm1, pdm1, pm2, pdm2, ppn_xi, pder_xi;
  GQUAD fact=-1.0/(Np*(Np+1.0)), gfact, fRet, xi;

  if ( eta == NULL || mret == NULL ) return NULL;

  if ( (bNeedNodes || bNeedWeights) &&  !ComputeNodes() ) {
    cout << "GLLBasis::EvalBasis (3): basis data incomplete" << endl;
    exit(1);
  }

//nn = MIN(eta->dim(),mret->dim(1));
//mm = MIN(Np+1,mret->dim(2)); 
  for ( i=0; i<eta->dim(); i++ ) 
  {
    for ( j=0; j<Np+1; j++ )  
    {
      fRet = 1.0;
      xi    = (GQUAD) (*eta)(i);
      if ( (GDOUBLE)xi < (GDOUBLE)ximin || (GDOUBLE)xi > (GDOUBLE)ximax ) fRet = 0.0;
      else if ( fabs(xi-xiNodes(j)) > 100.0*TINY )
      {
        ppn_j = Pn(j);
        ComputeJacobi(Np, alpha, beta, ppn_xi, pder_xi,pm1, pdm1, pm2, pdm2, xi);
        gfact = (1.0-xi*xi)/(xi-xiNodes(j)); 
        fRet  = fact * gfact * pder_xi/ppn_j;
      }
      (*mret)(i,j) = fRet;
    }
  }
  return mret;

} // end of method EvalBasis (3)


//************************************************************************************
//************************************************************************************
// METHOD     : EvalDBasis
// DESCRIPTION: Evaluates basis derivative at input parent domain points , eta_i
//              Deriv. is derived from :
//              h_j(eta) =  -1/(Np*(Np-1)) * (1-eta**2) dL_Np (eta)dxi / (L_Np(xi_j) (eta-xi_j))
// RETURNS    : GMatrix, M_ij = dh_j(eta_i)/dxi
//************************************************************************************
GMatrix *GLLBasis::EvalDBasis (GVector *eta, GMatrix *mret)
{
  if ( eta == NULL || mret == NULL ) return NULL;

  GINT  i, j, mm, nn;
  GQUAD ppn_j, pm1, pdm1, pm2, pdm2, ppn_xi, pder_xi, pdd;
  GQUAD fact=-1.0/(Np*(Np+1.0)), gfact, g1, g1i, g2, fRet, xi ;
  
  if ( eta == NULL || mret == NULL ) return NULL;

  if ( bNeedNodes || bNeedWeights &&  !ComputeNodes() ) {
    cout << "GLLBasis::EvalDBasis: basis data incomplete" << endl;
    exit(1);
  }

  nn = MIN(eta->dim(),mret->dim(1));
  mm = MIN(Np+1,mret->dim(2)); 
  for ( i=0; i<nn; i++)
  {
    for ( j=0; j<mm;  j++)
    {
      fRet = 0.0;
      xi     = (GQUAD) (*eta)(i);
      g1     = xi - xiNodes(j);
      if ( (GDOUBLE)xi < (GDOUBLE)ximin || (GDOUBLE)xi > (GDOUBLE)ximax ) fRet = 0.0;
      else if ( fabs(g1) > 100.0*TINY ) 
      {
//      ComputeJacobi(Np, alpha, beta, ppn_j , pder_j ,pm1, pdm1, pm2, pdm2, xiNodes(j));
        ppn_j = Pn(j);
        ComputeJacobi(Np, alpha, beta, ppn_xi, pder_xi,pm1, pdm1, pm2, pdm2, xi);
        gfact = fact / ppn_j;
        g1i   = 1.0/g1;
        g2    = (1.0 - xi*xi)*g1i;
        pdd   = 2.*xi*pder_xi - Np*(Np + 1.)*ppn_xi; 
        fRet  = gfact * g1i * ( pdd - (2.0*xi + g2 )*pder_xi);
      } 
      (*mret)(i,j) = fRet;
    }
  }
  return mret;
} // end of method EvalDBasis


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeLegendreMatrix
// DESCRIPTION: Computes matrix M_ij = P_i (xi_j),
//              where P_i is the Legendre polynomial of order i, and
//              xi_j is the j-th nodal point
// RETURNS    :
//************************************************************************************
GBOOL GLLBasis::ComputeLegendreMatrix()
{

  if ( bNeedNodes || bNeedWeights &&  !ComputeNodes() ) return FALSE;

  GINT  i, j, p;
  GQUAD ppn_i, pder_i, pm1, pdm1, pm2, pdm2;

  for ( i=0; i<Np+1; i++ )
  {
    for ( j=0; j<Np+1; j++ )
    {
      p = i;
      ComputeJacobi(p, 0.0, 0.0, ppn_i , pder_i ,pm1, pdm1, pm2, pdm2, xiNodes(j));
//    LegMatrix(i,j) = ppn_i * Weights(j);
      LegMatrix(i,j) = ppn_i ;
    }
  }

  return TRUE;

} // end of method ComputeLegendreMatrix


