//************************************************************************************//
// Module       : gradop.cpp
// Date         : 6/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                defining a (collocated) gradient operator object
// Derived From : none
// Modifications:
//************************************************************************************//
#include "gradop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GradOp::GradOp()
:
var         (NULL),
elem        (NULL)
{
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
GradOp::GradOp(Elem2D *e, GVector  *v)
:
var         (v),
elem        (e)
{

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
GradOp::~GradOp()
{
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetVar
// DESCRIPTION: Sets variable 
// RETURNS    :  

void  GradOp::SetVar(GVector *u, GBOOL bResize)
{
  var = u;
} // end of method SetVar


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets element
// RETURNS    :  

void  GradOp::SetElem(Elem2D *e)
{
  elem = e;
} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : Grad (1)
// DESCRIPTION: Computes Grad operator
// RETURNS    :  

GVector GradOp::Grad(const GINT  idir)
{
  GVector vt(var->GetIndex());
  Grad(idir,vt);

  return  vt;

} // end of method Grad (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Grad (2)
// DESCRIPTION: Computes Grad operator
// RETURNS    :  

void GradOp::Grad(const GINT  idir, GVector &vtmp)
{
#if defined(GRAD_LEVEL3_TIMING)
    GDOUBLE  tstart = STK::Timer();
#endif

  if ( elem == NULL | var == NULL ) {
    cout << "GradOp::Grad: NULL argument or element" << endl;
    exit(1);
  }
  if ( var->dim() != (elem->GetOrder(1)+1)*(elem->GetOrder(2)+1) ) {
    cout << "GradOp::Grad: Invalid argument" << endl;
    exit(1);
  }
  if ( idir != 1 && idir != 2 ) {
    cout << "GradOp::Grad: Invalid differention direction" << endl;
    exit(1);
  }


  GBOOL    bRet = TRUE;

  switch ( elem->ElemType() ) {
    case DEFORMED_QUAD:
      bRet = DefmQuadOp(&vtmp, idir); 
      break;
    case RECT_QUAD:
      bRet = RectQuadOp(&vtmp, idir);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&vtmp, idir);
      break;
    default:
      cout << "StiffOp::operator*: Invalid element type" << endl;
      exit(1);
  }

  if ( !bRet ) {
    cout << "GradOp::Grad: Operator failed" << endl;
    exit(1);
  }


#if defined(GRAD_LEVEL3_TIMING)
  time_result = STK::Timer()-tstart;
#endif

  
} // end of method Grad


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadOp
// DESCRIPTION: Computes operator*vector for a deformed quad element 
// ARGUMENTS  :
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
GBOOL GradOp::DefmQuadOp(GVector *vn, const GINT  idir)
{ 
  if ( var  == NULL || vn  == NULL || elem == NULL ) return FALSE;

  GVector *g1=NULL, *g2=NULL;

  if ( idir == 1 ) {
    g1 = ((DefQuad2D*)elem)->GetdXidX(1,1);
    g2 = ((DefQuad2D*)elem)->GetdXidX(2,1);
  }
  else {
    g1 = ((DefQuad2D*)elem)->GetdXidX(1,2);
    g2 = ((DefQuad2D*)elem)->GetdXidX(2,2);
  }
  DiagOp  dg1(g1), dg2(g2);    

  *vn = ( dg1 * pGrad(1)  ) + ( dg2 * pGrad(2)  );

  return TRUE;
  
} // end of method DefmQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadOp
// DESCRIPTION: Computes gradient of vector for a rectangular element 
// ARGUMENTS  : 
//              vn  : resulting gradient
// RETURNS    : TRUE on success; else FALSE
  
GBOOL GradOp::RectQuadOp(GVector *vn, const GINT  idir)
{ 
#if defined(GRAD_LEVEL2_TIMING)
    GDOUBLE  tstart = STK::Timer();
#endif


  if ( var  == NULL || vn  == NULL || elem == NULL ) return FALSE;

  GDOUBLE    L=0.0;
  Point3D  *vert;
  
  vert = elem->GetSpVertices();
  
  // Find dxi/dx--fine where mapping is on
  // parent interval of (-1,1) (the case for 
  // all nodal bases):
  if ( idir == 1 ) {
    L   = fabs((vert+1)->x1 - (vert)->x1);
    if (  L != 0.0 ) L = 2.0/L;
    pGrad(1,*vn);
    if ( L != 1.0 ) 
     MTK::fvec_const_prod_rep(*vn,L);
  }
  else if ( idir == 2 ) {
    if ( elem->Dim() >  1 ) L   = fabs((vert+3)->x2 - (vert)->x2);
    if (  L != 0.0 ) L = 2.0/L;
    pGrad(2,*vn);
    if ( L != 1.0 ) 
      MTK::fvec_const_prod_rep(*vn,L);
  }
  else
    return FALSE;
#if defined(GRAD_LEVEL2_TIMING)
  time_result = STK::Timer() - tstart;;
#endif
  
  return TRUE;

} // end of method RectQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  :
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL GradOp::TriangleOp(GVector *vn, const GINT  idir)
{
  return FALSE;
  if ( var  == NULL || vn  == NULL || elem == NULL ) return FALSE;
} // end of method TriangleOp


//************************************************************************************
//************************************************************************************
// METHOD     : pGrad (1)
// DESCRIPTION: Computes Grad operator in parent domain
// RETURNS    :

GVector GradOp::pGrad(const GINT  idir)
{
  GVector vt(var->GetIndex());
  pGrad(idir,vt);
  return vt;
} // end of method pGrad (1)


//************************************************************************************
//************************************************************************************
// METHOD     : pGrad (2)
// DESCRIPTION: Computes Grad operator in parent domain
// RETURNS    :  

void GradOp::pGrad(const GINT  idir, GVector &vtmp)
{

#if defined(GRAD_LEVEL1_TIMING)
    GDOUBLE  tstart = STK::Timer();
#endif
#if defined(GRAD_LEVEL0_TIMING)
    time_result = 0.0;
#endif

  if ( elem == NULL | var == NULL ) {
    cout << "GradOp::Grad: NULL argument or element" << endl;
    exit(1);
  }
  if ( var->dim() != (elem->GetOrder(1)+1)*(elem->GetOrder(2)+1) ) {
    cout << "GradOp::pGrad: Invalid argument" << endl;
    exit(1);
  }
  if ( idir != 1 && idir != 2  && idir != 3) {
    cout << "GradOp::pGrad: Invalid differention direction" << endl;
    exit(1);
  }

  GINT       N1=0, N2=0;
  GMatrix    *D;

  if ( elem->Dim() == 1 ) N1 = elem->GetOrder(1)+1;
  if ( elem->Dim()  > 1 ) {
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
  }

  if ( idir == 1 ) {       // Compute tensor product: I2 X D1 * var
    D = elem->Get1DDerivMatrix(1,FALSE);
    MTK::I2_X_D1(*D, *var, N1, N2,  vtmp);
  }
  else {                  // Compute tensor product: D2 X I1 * var
    D = elem->Get1DDerivMatrix(2,TRUE);  // get transpose, in order to reduce stride in tensor product...
    MTK::D2_X_I1(*D, *var, N1, N2,  vtmp);
  }


#if defined(GRAD_LEVEL1_TIMING)
  time_result = STK::Timer() - tstart;;
#endif

} // end of method pGrad (2)

