//************************************************************************************//
// Module       : helmholtzop.cpp
// Date         : 6/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the (weak) Helmholtz operator class.
//                  H = hc L + mc M
//                where hc and mc are settable constants, and
//                L is the Laplacian operator, and M is the mass operator
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "helmholtzop.hpp"
#include "laplacianop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "rectquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
HelmholtzOp::HelmholtzOp()
: LinOp(),
lc1       (1.0),
lc2       (1.0),
mc        (1.0),
Laplacian (NULL),
Mass      (NULL),
elem      (NULL) 
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
HelmholtzOp::HelmholtzOp(Elem2D *e)
: LinOp( ), 
lc1       (1.0),
lc2       (1.0),
mc        (1.0),
Laplacian (NULL),
Mass      (NULL),
elem      (e) 
{

  Laplacian = new LaplacianOp (elem);
  Mass      = new MassOp (elem);

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
HelmholtzOp::~HelmholtzOp()
{
  if ( Laplacian != NULL ) delete Laplacian;
  if ( Mass      != NULL ) delete Mass;

  Laplacian  = NULL;
  Mass       = NULL;

}

//************************************************************************************
//************************************************************************************
// METHOD     : SetConst (1)
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void HelmholtzOp::SetConst(GDOUBLE clc, GDOUBLE cmc)
{
  if ( Laplacian == NULL || Mass == NULL )
  {
    cout << " HelmholtzOp::SetConst: one or more NULL operators" << endl;
    exit(1);
  }
  lc1 = lc2 = clc;
  mc  = cmc;
  Laplacian->SetConst(lc1,lc2);
  Mass->SetConst(mc);
  
} // end of method SetConst (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst (2)
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void HelmholtzOp::SetConst(GDOUBLE clc1, GDOUBLE clc2, GDOUBLE cmc)
{
  if ( Laplacian == NULL || Mass == NULL )
  {
    cout << " HelmholtzOp::SetConst: one or more NULL operators" << endl;
    exit(1);
  }
  lc1 = clc1;
  lc2 = clc2;
  mc  = cmc ;
  Laplacian->SetConst(lc1,lc2);
  Mass->SetConst(mc);

} // end of method SetConst (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void HelmholtzOp::SetElem(Elem2D *e)
{

  elem = e;
  if ( Laplacian != NULL )
    Laplacian->SetElem(elem);
  if ( Mass != NULL )
    Mass->SetElem(elem);
  
} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
GVector  HelmholtzOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);
  
  OpVec_prod(q, newq);
  
  return newq;

} // end of \* operator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : void
//************************************************************************************
void HelmholtzOp::OpVec_prod(GVector &q, GVector &ret)  
{
  if ( elem == NULL )
  {
    cout << "HelmholtzOp::operator*: NULL element" << endl;
    exit(1);
  }

  GBOOL    bRet = TRUE;


  switch (elem->ElemType())
  {
    case DEFORMED_QUAD:
      bRet = DefmQuadOp(&q,&ret);
      break;
    case RECT_QUAD:
      bRet = RectQuadOp(&q,&ret);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&q,&ret);
      break;
    default:
      cout << "HelmholtzOp::operator*: Invalid element type" << endl;
      exit(1);
  }
  if ( mask ) MTK::fvec_point_prod_rep(ret,*mask);
  if ( !bRet )
  {
    cout << "HelmholtzOp::operator*: Operator product failed" << endl;
    exit(1);
  }

} // end of method oerator * (vector)


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadOp
// DESCRIPTION: Compute ret = (hConst * L  + mConst * M ) v
//              where L is the weak Laplacian, and M is 
//              the mass matrix. h(m)Const are constants, which
//              are set implicitly w/in Lap and Mass operators via
//              SetConst.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL HelmholtzOp::DefmQuadOp(GVector *v, GVector *vn)
{
  GVector *vtmp;

  if ( v  == NULL || vn  == NULL ) return FALSE;
  if ( Laplacian == NULL || Mass == NULL ) return FALSE;

  vtmp = elem->GetTemp();
  Laplacian->OpVec_prod(*v, *vn);
  Mass     ->OpVec_prod(*v, *vtmp);
  MTK::fvec_add_rep(*vn, *vtmp);
  elem->TempUnlock(vtmp);

  
  return TRUE;
} // end of method DefmQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadOp
// DESCRIPTION: Compute ret = (hConst * L  + mConst * M ) v
//              where L is the weak Laplacian, and M is   
//              the mass matrix. h(m)Const are constants, which
//              are set implicitly w/in Lap and Mass operators via
//              SetConst.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL HelmholtzOp::RectQuadOp(GVector *v, GVector *vn)
{
  GVector *vtmp;

  if ( v  == NULL || vn  == NULL ) return FALSE;
  if ( Laplacian == NULL || Mass == NULL ) return FALSE;
  
  vtmp = elem->GetTemp();
  Laplacian->OpVec_prod(*v, *vn);
  Mass     ->OpVec_prod(*v, *vtmp);
  MTK::fvec_add_rep(*vn, *vtmp);
  elem->TempUnlock(vtmp);

  return TRUE;
} // end of method RectQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL HelmholtzOp::TriangleOp(GVector *v, GVector *vn)
{
  return FALSE;
} // end of method TriangleOp


//************************************************************************************
//************************************************************************************
// METHOD     : GetElem
// DESCRIPTION: Gets member element pointer
// ARGUMENTS  : none.
// RETURNS    : member data pointer
//************************************************************************************
Elem2D *HelmholtzOp::GetElem()
{
  return elem;
} // end of method GetElem
