//************************************************************************************//
// Module       : ihelmholtzop.hpp
// Date         : 7/17/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the inverse to the local helmholtz operator
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include "ihelmholtzop.hpp"
#include "mtk.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
iHelmholtzOp::iHelmholtzOp()
:        LinOp(),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
mc       (1.0),
elem     (NULL),
H        (NULL),
iH       (NULL)
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
iHelmholtzOp::iHelmholtzOp(LinOp *AA)
:        LinOp(AA),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
mc       (1.0),
elem     (NULL),
H        (NULL),
iH       (NULL)
{
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
iHelmholtzOp::iHelmholtzOp(Elem2D *e)
:        LinOp(),
N1       (0),
N2       (0),
NN       (0),
lc1      (1.0),
lc2      (1.0),
mc       (1.0),
elem     (NULL),
H        (NULL),
iH       (NULL)
{
  SetElem(e);

} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
iHelmholtzOp::~iHelmholtzOp()
{
  if ( H  != NULL ) delete H ; H  = NULL;
  if ( iH != NULL ) delete iH; iH = NULL;
}


//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void iHelmholtzOp::SetElem(Elem2D *e)
{
  GBOOL bRet ;

  if ( e == NULL )
  {
    cout << "iHelmholtzOp::SetElem: NULL element" << endl;
    exit(1);
  }

  if ( e != elem || N1 != (e->GetOrder(1)+1) ||
                    N2 != (e->GetOrder(2)+1) )
  {
    elem = e;
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( H  != NULL ) delete H ; H  = NULL;
    if ( iH != NULL ) delete iH; iH = NULL;
    if ( !ComputeOp() )
    {
      cout << "iHelmholtzOp::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
    return;
  }

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void iHelmholtzOp::SetConst(GDOUBLE clc, GDOUBLE cmc)
{


  if ( lc1 != clc || lc2 != clc || mc != cmc )
  {
    if ( elem == NULL )
    {
      cout << "iHelmholtzOp::SetConst: NULL element" << endl;
      exit(1);
    }
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( H  != NULL ) delete H ; H  = NULL;
    if ( iH != NULL ) delete iH; iH = NULL;
    lc1 = lc2 = clc;
    mc = cmc;
    if ( !ComputeOp() )
    {
      cout << "iHelmholtzOp::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void iHelmholtzOp::SetConst(GDOUBLE clc1, GDOUBLE clc2, GDOUBLE cmc)
{


  if ( lc1 != clc1 || lc2 != clc2 || mc != cmc )
  {
    if ( elem == NULL )
    {
      cout << "iHelmholtzOp::SetConst: NULL element" << endl;
      exit(1);
    }
    N1 = elem->GetOrder(1)+1;
    N2 = elem->GetOrder(2)+1;
    NN = N1 * N2;
    if ( H  != NULL ) delete H ; H  = NULL;
    if ( iH != NULL ) delete iH; iH = NULL;
    lc1 = clc1;
    lc2 = clc2;
    mc = cmc;
    if ( !ComputeOp() )
    {
      cout << "iHelmholtzOp::SetElem: ComputeOp failed" << endl;
      exit(1);
    }
  }
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
GVector  iHelmholtzOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of method \* oeprator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
void iHelmholtzOp::OpVec_prod(GVector &q, GVector &ret)
{

  if ( elem == NULL || iH == NULL )
  {
    cout << "iHelmholtzOp::operator*: NULL element or operator" << endl;
    exit(1);
  }

  if ( q.dim() != NN  || ret.dim() != NN ) 
  {
    cout << "iHelmholtzOp::operator*: incompatible vector(s)" << endl;
    exit(1);
  }

//ret = (*iH) * q;
  MTK::fmatvec_prod(*iH, q, ret);
  if ( mask )  MTK::fvec_point_prod_rep(ret, *mask);

} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeOp
// DESCRIPTION: Computes inverse Helmholtz operator
// ARGUMENTS  : 
//             
// RETURNS    : TRUE on success; else FALSE
GBOOL iHelmholtzOp::ComputeOp()
{

  GBOOL bRet ;

  if ( elem == NULL ) return FALSE;
  if ( H != NULL || iH != NULL ) return FALSE;

  if ( N1 <= 0 || N2 <= 0 || NN <= 0 ) return FALSE;

  H    = new GMatrix(NN,NN);
  iH   = new GMatrix(NN,NN);

  switch (elem->ElemType())
  {
    case DEFORMED_QUAD:
      bRet = DefmQuadH();
      break;
    case RECT_QUAD:
      bRet = RectQuadH();
      break;
    case TRIANGULAR:
      bRet = TriangleH();
      break;
    default:
      cout << "LaplacianOp::Constructor(3): Invalid element type" << endl;
      exit(1);
  }
#if 0
  GMatrix C(NN,NN);
  C = (*H) * (*iH);
  cout.setf(ios::fixed);
  cout << "iHelmholtzOp::ComputeOp: H * iH = " << C << endl;
#endif


  // done with H, so delete it:
  if ( H != NULL )
  {
    delete H;
    H = NULL;
  }

  return bRet;
} // end of method ComputeOp



//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadH
// DESCRIPTION: Computes Helmholtz operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = Sum_k[D1_ki ^2  G11_pj  + D2_pj^2 G22_ip]
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
GBOOL iHelmholtzOp::DefmQuadH()
{ 

  GINT      i, j, k, n, m;
  GBOOL     bRet = TRUE;
  GVector  *G11, *G12, *G21, *G22, *M, *GA[2];
  GMatrix  *D1 , *D2;

  M   = elem->GetMassMatrix();  // already includes Jacobian
  D1  = elem->Get1DDerivMatrix(1,FALSE);
  D2  = elem->Get1DDerivMatrix(2,FALSE);
  G11 = ((DefQuad2D*)elem)->GetWJMetric(1,1);
  G12 = ((DefQuad2D*)elem)->GetWJMetric(1,2);
  G21 = G12;
  G22 = ((DefQuad2D*)elem)->GetWJMetric(2,2);

  if ( D1  == NULL || D2  == NULL ||
       G11 == NULL || G12 == NULL ||
       G21 == NULL || G22 == NULL ) return FALSE;

 
  GMatrix *d1v, *d2v;

  d1v = new GMatrix(NN,NN);
  d2v = new GMatrix(NN,NN);
  H   = new GMatrix(NN,NN);

  // Do I2 X D1 term:
  for ( k=0; k<N2; k++ )
  {
    for ( i=0; i<N1; i++ )
    {
      for ( j=0; j<N1; j++ )
      {
        n = k * N1 + i;
        m = k * N1 + j;
        (*d1v)(n,m) = (*D1)(i,j);
      }
    }
  }

  // Do D2 X I1 term:
  for ( i=0; i<N2; i++ )
  {
    for ( j=0; j<N2; j++ )
    {
      for ( k=0; k<N1; k++ )
      {
        n = i * N1 + k;
        m = j * N1 + k;
        (*d2v)(n,m) = (*D2)(i,j);
      }
    }
  }

  // Incorporate the metric terms:
  GA[0] = G11;
  GA[1] = G12;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN; i++ )
      for ( j=0; j<NN; j++ )
        (*d1v)(i,j) = (*d1v)(i,j) * (*GA[k])(j);

  GA[0] = G21;
  GA[1] = G22;
  for ( k=0; k<2; k++ )
    for ( i=0; i<NN; i++ )
      for ( j=0; j<NN; j++ )
        (*d2v)(i,j) = (*d2v)(i,j) * (*GA[k])(j);
  // Multiply by the appropriate transposes, and add up:
  *H   = ( d1v->Transpose() * ( *d1v + *d2v ) )
       + ( d2v->Transpose() * ( *d1v + *d2v ) );

  // Add in mass matrix:
  for ( k=0; k<NN; k++ )
  {
    (*H)(k,k) += ( mc*(*M)(k));
  }
  delete d1v;
  delete d2v;

  // Now, invert:
  return H->Inverse(*iH);

} // end of method DefmQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadH
// DESCRIPTION: Computes Helmholtz operator*vector  for a deformed quad element.
//              Only diagonal^1 elements of operator are considered.
//              Diag elements are:
//                K_ij = 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL iHelmholtzOp::RectQuadH()
{ 
#if 0
  GINT      i, j, k;
  GDOUBLE     L1, L2, R1, R2, RM;
  Point2D  *vert;
  GVector  *M  , *M1, *M2;  
  GMatrix  *K1 , *K2 ;

  M   = elem->GetMassMatrix();  // already include Jacobian
  M1  = elem->Get1DWeights(1);
  M2  = elem->Get1DWeights(2);
  K1   = elem->Get1DStiffMatrix(1,FALSE);
  K2   = elem->Get1DStiffMatrix(2,FALSE);
  vert = elem->GetSpVertices();
  L1   = fabs((vert+1)->x1 - (vert)->x1);
  L2   = fabs((vert+3)->x2 - (vert)->x2);

  if ( L1 == 0 || L2 == 0 ) return FALSE;
  
  R1  = lc1 * L2 / L1;
  R2  = lc2 * L1 / L2;
  RM  = 0.25 * mc * L1 * L2;

 
  if ( K1  == NULL || K2  == NULL ||
       M1  == NULL || M2  == NULL ||
       M   == NULL                 ) return FALSE;
  
  *H = 0.0;
  // Do M2 X K1 term:
  for ( k=0; k<N2; k++ )
  {
    for ( i=0; i<N1; i++ )
    {
      for ( j=0; j<N1; j++ )
      {
        n = k * N1 + i;
        m = k * N1 + j;
        (*Lap)(n,m) = R1*(*M2)(k) * (*K1)(i,j);
      }
    }
  }

  // Do K2 X M1 term:
  for ( i=0; i<N2; i++ )
  {
    for ( j=0; j<N2; j++ )
    {
      for ( k=0; k<N1; k++ )
      {
        n = i * N1 + k;
        m = j * N1 + k;
        (*Lap)(n,m) += ( R2*(*M1)(k) * (*K2)(i,j);
      }
    }
  }

  // Add in mass matrix:
  for ( k=0; k<NN; k++ )
  {
    (*Lap)(k,k) += ( mc*(*MassMatrix)(k));
  }
#endif

  GSHORT       i, j;
  GVector    *icol, *rcol;
  HelmholtzOp *hop;


  icol = new GVector(NN);
  rcol = new GVector(NN);
  hop  = new HelmholtzOp(elem);

  hop->SetConst(lc1, lc2,mc);
  for ( j=0; j<NN; j++ )
  {
    for ( i=0; i<NN; i++ )
    {
      (*icol)(i) = 0.0;
      if ( i == j ) (*icol)(i) = 1.0;
    }
    hop->OpVec_prod(*icol, *rcol);
    for ( i=0; i<NN; i++ )
      (*H)(i,j) = (*rcol)(i);
  }

  delete hop;
  delete icol;
  delete rcol;

  
  // Now, invert :
  return H->Inverse(*iH);

} // end of method RectQuadH


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleH
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL iHelmholtzOp::TriangleH()
{
  return FALSE;
} // end of method TriangleH


