//************************************************************************************//
// Module       : laplacianop.cpp
// Date         : 6/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the stiffness operator (weak Laplacian) class
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "laplacianop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "rectquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
LaplacianOp::LaplacianOp()
: LinOp(),
hC1       (1.0),
hC2       (1.0),
elem      (NULL) 
{
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
LaplacianOp::LaplacianOp(Elem2D *e)
: LinOp( ), 
hC1       (1.0),
hC2       (1.0),
elem      (e) 
{
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
LaplacianOp::~LaplacianOp()
{
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void LaplacianOp::SetElem(Elem2D *e)
{
  GINT  NN;

  elem = e;

  NN        = (elem->GetOrder(1)+1) * (elem->GetOrder(2)+1);

} // end of method SetElem


//************************************************************************************
// METHOD     : SetConst (1)
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  : GDOUBLE 
// RETURNS    : none
//************************************************************************************
void LaplacianOp::SetConst(GDOUBLE c)
{
  hC1 = hC2 = c;
} // end of method SetConst


//************************************************************************************
// METHOD     : SetConst (2)
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  : GDOUBLE c1 - 1-derivative const
//              GDOUBLE c2 = 2-derivative const
// RETURNS    : none
//************************************************************************************
void LaplacianOp::SetConst(GDOUBLE c1, GDOUBLE c2)
{
  hC1 = c1;
  hC2 = c2;
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
GVector  LaplacianOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);
  
  OpVec_prod(q, newq);

  return newq;
} // end of operator \*

//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
void LaplacianOp::OpVec_prod(GVector &q, GVector &ret)  
{
  if ( elem == NULL )
  {
    cout << "LaplacianOp::operator*: NULL element" << endl;
    exit(1);
  }

  GBOOL    bRet = TRUE;


  switch (elem->ElemType())
  {
    case DEFORMED_QUAD:
      bRet = DefmQuadOp(&q,&ret);
      break;
    case RECT_QUAD:
      bRet = RectQuadOp(&q,&ret);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&q,&ret); 
      break;
    default:
      cout << "LaplacianOp::operator*: Invalid element type" << endl;
      exit(1);
  }
  if ( mask ) MTK::fvec_point_prod_rep(ret, *mask);
  if ( !bRet )
  {
    cout << "LaplacianOp::operator*: Operator product failed" << endl;
    exit(1);
  }


} // end of method oerator * (vector)


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadOp
// DESCRIPTION: Computes operator*vector  for a deformed quad element.
//              Stiffness operator for deformed quadrilateral element is:
//                (D1T G11 D1) + (D1T G12 D2) + (D2T G12 D1) + (D2T G22 D2)
//              where D1 = I X D1_  & D2 = D2_ X I, s.t. D1_ and D2_ are,
//              respectively, the 1d derivative operators in the 1 and 2
//              directions, and 'X' is a tensor product. The GIJ are the 
//              metric quantities (I,J in (1,2), each evaluated at all
//              2d grid points. 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
GBOOL LaplacianOp::DefmQuadOp(GVector *v, GVector *vn)
{
  if ( v  == NULL || vn  == NULL ) return FALSE;

  GINT      N1, N2, NN;
  GVector  *G11, *G12, *G21, *G22, *vt0;
  GMatrix  *D1 , *D2 , *D1T, *D2T; 

  D1  = elem->Get1DDerivMatrix(1,FALSE);
  D2  = elem->Get1DDerivMatrix(2,FALSE);
  D1T = elem->Get1DDerivMatrix(1,TRUE);
  D2T = elem->Get1DDerivMatrix(2,TRUE);
  G11 = ((DefQuad2D*) elem)->GetWJMetric(1,1);
  G12 = ((DefQuad2D*) elem)->GetWJMetric(1,2);
  G21 = G12;
  G22 = ((DefQuad2D*) elem)->GetWJMetric(2,2);

  if ( D1  == NULL || D2  == NULL ||
       D1T == NULL || D2T == NULL ||
       G11 == NULL || G12 == NULL ||
       G21 == NULL || G22 == NULL ) return FALSE;

  vt0 = elem->GetTemp();

  N1 = elem->GetOrder(1)+1;
  N2 = elem->GetOrder(2)+1;
  NN = N1 * N2;
  if ( v->dim() != NN  || vn->dim() != NN ) return FALSE;      

  // NOTE: move these temporary vectors to the constructor/destructor:
  GVector d1v(NN), d2v(NN), gd1v(NN), gd2v(NN);

  // Compute tensor product derivative operators:
  // ... (I2 X D1_)  v:
  MTK::I2_X_D1(*D1, *v, N1, N2, d1v);

  // ... (D2_ X I1)  v:
  MTK::D2_X_I1(*D2T, *v, N1, N2, d2v);
  

  // Multiply tensor product by geometry factors s.t.
  // gd_i_u = Sum_i G_i,j du_j:
//gd1v = ( diag11 * d1v )  + ( diag12 * d2v );
//gd2v = ( diag12 * d1v )  + ( diag22 * d2v );
  MTK::fvec_point_prod(d1v, *G11, gd1v);
  MTK::fvec_point_prod(d2v, *G12, *vt0);
  MTK::fvec_add_rep(gd1v,*vt0);
  MTK::fvec_point_prod(d1v, *G21, gd2v);
  MTK::fvec_point_prod(d2v, *G22, *vt0);
  MTK::fvec_add_rep(gd2v,*vt0);

  // Now sum across transpose of derivative operators s.t.
  // K u = Sum_i D_i^T gdu_i = D1T gd1v + D2T gd2v = I2_X_D1 gd1v + D2_X_I1 gd2v;
  //... 
  MTK::I2_X_D1(*D1T,gd1v, N1, N2, *vn);
  MTK::D2_X_I1(*D2 ,gd2v, N1, N2, *vt0);
  MTK::fvec_add_rep(*vn, *vt0);

  elem->TempUnlock(vt0);
  
  return TRUE;
} // end of method DefmQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadOp
// DESCRIPTION: Computes operator*vector  for a regular rect. quad element.
//              Stiffness operator for rectangular element is:
//                L2/L1 * (M2 X K1)  + L1/L2 * (K2 X M1),
//              where 'X' is tensor product. 
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL LaplacianOp::RectQuadOp(GVector *v, GVector *vn)
{
  if ( v  == NULL || vn  == NULL ) return FALSE;

  GINT        N1, N2, NN;
  GDOUBLE     L1, L2, R1, R2;
  Point3D    *vert;
  GVector    *M1, *M2, *vt0, *vt1;
  GMatrix    *K1, *K2T;

  M1   = elem->Get1DWeights(1);
  M2   = elem->Get1DWeights(2);
  K1   = elem->Get1DStiffMatrix(1,FALSE);
  K2T  = elem->Get1DStiffMatrix(2,TRUE);
  vert = elem->GetSpVertices();
  L1   = fabs((vert+1)->x1 - (vert)->x1);
  L2   = fabs((vert+3)->x2 - (vert)->x2);

  vt0 = elem->GetTemp();
  vt1 = elem->GetTemp();
  if ( L1 == 0 || L2 == 0 ) return FALSE;

  // These Jacobian factors are accurate when parent interval is
  // on (-1,1), and the mapping is s.t. x-x0 = L/2 + xi(L/2)
  R1  = hC1 * L2 / L1;
  R2  = hC2 * L1 / L2;

#if 0
  cout << "LapOp: L1= " << L1 << " L2= " << L2 << endl;
  cout << "LapOp: R1=L2/L1 = " << R1 << " R2=L1/L2 = " << R2 << endl;
#endif

  if ( M1  == NULL || M2  == NULL ||
       K1  == NULL || K2T == NULL ) return FALSE;

  N1 = elem->GetOrder(1)+1;
  N2 = elem->GetOrder(2)+1;
  NN = N1 * N2;
  if ( v  ->dim() != NN  || vn ->dim() != NN || 
       vt0->dim() != NN  || vt1->dim() != NN ) return FALSE;      

  // Do M2 X K1 term: 
  *vn = 0.0;
  if ( R1 != 0.0 ) {
    MTK::Dg2_X_D1(*K1, *M2, *v, *vt0, *vn);
    if ( R1 != 1.0 ) MTK::fvec_const_prod_rep(*vn, R1);  
  }
  // Do K2 X M1 term: 
//DiagOp dM1(M1);
  if ( R2 != 0.0 ) {
    MTK::D2_X_Dg1(*K2T, *M1, *v, *vt1, *vt0);
    if ( R2 != 1.0 ) MTK::fvec_const_prod_rep(*vt0, R2);  
    MTK::fvec_add_rep(*vn,*vt0);
  }

  elem->TempUnlock(vt0);
  elem->TempUnlock(vt1);

  return TRUE;
} // end of method RectQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE

GBOOL LaplacianOp::TriangleOp(GVector *v, GVector *vn)
{
  return FALSE;
} // end of method TriangleOp

//************************************************************************************
//************************************************************************************
// METHOD     : GetElem
// DESCRIPTION: Gets member element pointer
// ARGUMENTS  : none.
// RETURNS    : member data pointer
//************************************************************************************
Elem2D *LaplacianOp::GetElem()
{
  return elem;
} // end of method GetElem

