//************************************************************************************//
// Module       : rectquad2d.hpp
// Date         : 9/14/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a single regular rectangular 2D spectral element
//                The following is the ordering for the vertices (Vi)
//                and segment midpoints (Mi):
//
//          V7 ________________ V6
//            /|     M6       /|
//       M7  / |             / |
//          /  |        M5  /  |
//         /M11|  M4       /   |
//     V4 /____|___ ______/    | M10
//        |    |         | V5  |
//        |    |     M2  |     |
//        | V3 |_________|_____| V2
//    M8  |    /         |    /
//        |   /       M9 |   /
//        |  / M3        |  / M1
//        | /            | /
//        |______________|/
//       V0       M0     V1
//
// [Note that in 2d, we use just the bottom plane.]

// Derived From : Elem2D.
// Modifications:
//************************************************************************************//
#include "rectquad2d.hpp"
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include <stdio.h>

//************************************************************************************
//************************************************************************************
// constructor (1)
//************************************************************************************
RectQuad2D::RectQuad2D(GSHORT ntmp)
: Elem2D(ntmp),
xiNodes1              (NULL),
xiNodes2              (NULL),
Weights1              (NULL),
Weights2              (NULL),
Weights2D             (NULL),
MassMatrix            (NULL),
gMassMatrix           (NULL),
iMassMatrix           (NULL),
StiffMat1             (NULL),
StiffMat2             (NULL),
StiffMat1T            (NULL),
StiffMat2T            (NULL),
D1                    (NULL),
D2                    (NULL),
D1T                   (NULL),
D2T                   (NULL),
opInterp1             (NULL),
opInterp2             (NULL),
opInterpT1            (NULL),
opInterpT2            (NULL),
dInterp1              (NULL),
dInterp2              (NULL),
dInterpT1             (NULL),
dInterpT2             (NULL)
{
  GINT  n;

  elemtype_   = RECT_QUAD;
  nVertices_   = (GINT )pow(2.0,GDIM);
  nEdges_     =  2*(GDIM-1)*GDIM;
#if defined(IS3D)
  nFaces_     =     2*GDIM ;
#endif
  CreateElemDynamic();


  bInitialized &= Initialize(Np1, Np2);

  for ( n=0; n<GDIM*nVertices_; n++ ) spvVertices[n] = 0.0;

} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
//************************************************************************************
RectQuad2D::RectQuad2D(GINT  inOrder1, GINT  inOrder2, GSHORT ntmp)
:Elem2D(ntmp),
xiNodes1              (NULL),
xiNodes2              (NULL),
Weights1              (NULL),
Weights2              (NULL),
Weights2D             (NULL),
MassMatrix            (NULL),
gMassMatrix           (NULL),
iMassMatrix           (NULL),
StiffMat1             (NULL),
StiffMat2             (NULL),
StiffMat1T            (NULL),
StiffMat2T            (NULL),
D1                    (NULL),
D2                    (NULL),
D1T                   (NULL),
D2T                   (NULL),
MD1                   (NULL),
MD2                   (NULL),
D1TM                  (NULL),
D2TM                  (NULL),
opInterp1             (NULL),
opInterp2             (NULL),
opInterpT1            (NULL),
opInterpT2            (NULL),
dInterp1              (NULL),
dInterp2              (NULL),
dInterpT1             (NULL),
dInterpT2             (NULL)
{
  GINT  n;

  SetOrder(inOrder1, inOrder2);

  elemtype_   = RECT_QUAD;
  nVertices_   = (GINT )pow(2.0,GDIM);
  nEdges_     =  2*(GDIM-1)*GDIM;
#if defined(IS3D)
  nFaces_     =     2*GDIM ;
#endif
  CreateElemDynamic();


  bInitialized  = Initialize(inOrder1, inOrder2);

  for ( n=0; n<GDIM*nVertices_; n++ ) spvVertices[n] = 0.0;

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
//************************************************************************************
RectQuad2D::RectQuad2D(GNBasis *inBasis1, GNBasis *inBasis2, GSHORT ntmp)
: Elem2D(ntmp),
xiNodes1              (NULL),
xiNodes2              (NULL),
Weights1              (NULL),
Weights2              (NULL),
Weights2D             (NULL),
MassMatrix            (NULL),
gMassMatrix           (NULL),
iMassMatrix           (NULL),
StiffMat1             (NULL),
StiffMat2             (NULL),
StiffMat1T            (NULL),
StiffMat2T            (NULL),
D1                    (NULL),
D2                    (NULL),
D1T                   (NULL),
D2T                   (NULL),
opInterp1             (NULL),
opInterp2             (NULL),
opInterpT1            (NULL),
opInterpT2            (NULL),
dInterp1              (NULL),
dInterp2              (NULL),
dInterpT1             (NULL),
dInterpT2             (NULL)
{
  GINT  n;

  elemtype_   = RECT_QUAD;
  nVertices_  = (GINT )pow(2.0,GDIM);
  nEdges_     =  2*(GDIM-1)*GDIM;
#if defined(IS3D)
  nFaces_     =  2*GDIM ;
#endif
  CreateElemDynamic();


  basis1 =  inBasis1;
  basis2 =  inBasis2;

  Np1 = inBasis1->GetOrder();
  Np2 = inBasis2->GetOrder();

  if ( basis1 != NULL && basis2 != NULL ) bInitialized = TRUE;
  bInitialized &= Initialize(Np1, Np2);

  spVertices[0].x1 = basis1->GetXimin(); spVertices[0].x2 = basis2->GetXimin();
  spVertices[1].x1 = basis1->GetXimax(); spVertices[1].x2 = basis2->GetXimin();
  spVertices[2].x1 = basis1->GetXimax(); spVertices[2].x2 = basis2->GetXimax();
  spVertices[3].x1 = basis1->GetXimin(); spVertices[3].x2 = basis2->GetXimax();
  for ( n=0; n<GDIM*nVertices_; n+=2 ) 
  {
    spvVertices  [n] = spVertices[n/2].x1; 
    spvVertices[n+1] = spVertices[n/2].x2;
  }

} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
RectQuad2D::~RectQuad2D()
{
   DeleteDynamic();
}

//************************************************************************************
//************************************************************************************
// Assignment operator method (1)
//************************************************************************************
void RectQuad2D::operator=(const RectQuad2D &elem)
{
  GINT  N1, N2;

  N1 = Np1+1;
  N2 = Np2+1;

  if ( &elem != this ) 
  {
   // copy data:
    Np1      = elem.Np1;
    bSolved  = elem.bSolved;

    //  copy basis, matrices, and node data:
    basis1 = elem.basis1;
    basis2 = elem.basis2;


    if ( Weights1 != NULL ) delete Weights1 ;
    Weights1 = NULL;
    if ( elem.Weights1 != NULL ) 
    {
      Weights1 = new GVector(N1);
     *Weights1 = *(elem.Weights1);
    }   

    if ( Weights2 != NULL ) delete Weights2 ;
    Weights2 = NULL;
    if ( elem.Weights2 != NULL ) 
    {
      Weights2 = new GVector(N2);
     *Weights2 = *(elem.Weights2);
    }   


    if ( xiNodes1 != NULL ) delete xiNodes1;
    xiNodes1 = NULL;
    if ( elem.xiNodes1 != NULL ) 
    {
      xiNodes1 = new GVector(N1);
     *xiNodes1 = *(elem.xiNodes1);
    }   

    if ( xiNodes2 != NULL ) delete xiNodes2;
    xiNodes2 = NULL;
    if ( elem.xiNodes2 != NULL ) 
    {
      xiNodes2 = new GVector(N2);
     *xiNodes2 = *(elem.xiNodes2);
    }   

    if ( D1  != NULL ) delete D1;
    if ( D1T != NULL ) delete D1T;
    D1  = NULL;
    D1T = NULL;
    if ( elem.D1 != NULL)
    {
      D1 = new GMatrix(N1,N1);
      D1 = elem.D1;
    }
    if ( elem.D1T != NULL)
    {
      D1T = new GMatrix(N1,N1);
      D1T = elem.D1T;
    }

    if ( D2  != NULL ) delete D2;
    if ( D2T != NULL ) delete D2T;
    D2  = NULL;
    D2T = NULL;
    if ( elem.D2 != NULL)
    {
      D2 = new GMatrix(N2,N2);
      D2 = elem.D2;
    }
    if ( elem.D2T != NULL)
    {
      D2T = new GMatrix(N2,N2);
      D2T = elem.D2T;
    }
  }

} // end of = operator


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetOrder
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  GINT  element expansion order
//************************************************************************************
GINT  RectQuad2D::GetOrder(const GINT  iDir)
{
  if ( iDir == 1 )
    return Np1;
  else if ( iDir == 2 ) 
    return Np2;
  else 
    return 0;
} // end of method GetOrder
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeSpNodes
// DESCRIPTION: Computes real space locations of parent space nodes.
//              Two matrices are created, spNodes1(2), of the form
//                x_i(0,0), x_i(1,0), ..., x_i(Np1,0),
//                x_i(0,1), ...
//                .
//                .
//                .
//                x_i(0,Np2), x_i(1,Np1),..., x_i(Np1,Np2)
//
//               where x_i(l,m) is x1(2) evaluated at parent node
//               xi1_l, xi2_m, and xi are increasing with index in
//               each direction. 
//            
// ARGUMENTS  :
// RETURNS    :  TRUE on success; else FALSE. Failure occurs if
//               nodes and weights cannot be computed, or if memory
//               cannot be allocated.
//************************************************************************************
GBOOL RectQuad2D::ComputeSpNodes()
{

  GINT  i, j, k;

  // NOTE: Geometry quantities assumed to be computed in SolveFE

  if ( spNodes1 != NULL ) spNodes1->Resize((Np1+1)*(Np2+1));
  else                    spNodes1 = new GVector((Np1+1)*(Np2+1));
  if ( spNodes2 != NULL ) spNodes2->Resize((Np1+1)*(Np2+1));
  else                    spNodes2 = new GVector((Np1+1)*(Np2+1));

  if ( spNodes1 == NULL || spNodes2 == NULL ) 
  {
    cout << "RectQuad2D::ComputeSpNodes: memory allocation failed." << endl;
    exit(1);
  }

  if ( !bSolved )
    if ( SolveFE() <= 0 ) return FALSE; 


  
  for ( j=0; j<Np2+1; j++ )
  {
    xi2p[j] = 1.0 + (*xiNodes2)(j);
    xi2m[j] = 1.0 - (*xiNodes2)(j);
    for ( i=0; i<Np1+1; i++ )
    {
      xi1p[i] = 1.0 + (*xiNodes1)(i);
      xi1m[i] = 1.0 - (*xiNodes1)(i);
        k = i + j*(Np1+1);
        (*spNodes1)(k) = 0.25* ( spVertices[0].x1*xi1m[i]*xi2m[j] + spVertices[1].x1*xi1p[i]*xi2m[j]
                               + spVertices[3].x1*xi1m[i]*xi2p[j] + spVertices[2].x1*xi1p[i]*xi2p[j] );
        (*spNodes2)(k) = 0.25* ( spVertices[0].x2*xi1m[i]*xi2m[j] + spVertices[1].x2*xi1p[i]*xi2m[j]
                               + spVertices[3].x2*xi1m[i]*xi2p[j] + spVertices[2].x2*xi1p[i]*xi2p[j] );
    } 
  }
  return TRUE;
} // end of method ComputeSpNodes


//************************************************************************************
//************************************************************************************
// METHOD     : GetSpNodes (1)
// DESCRIPTION: Returns pointer to spatial nodes (not std element nodes).
//              The real space nodes are not formatted in any way
//              in this routine, and take the 'native' format. 
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::GetSpNodes(GINT  idir)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  if      ( idir == 1 ) return spNodes1;
  else if ( idir == 2 ) return spNodes2;
  else                  return NULL;

} // end of method GetSpNodes (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Get2DWeights
// DESCRIPTION: Returns pointer to quadtraure weights corresponding to
//              1- and 2-direction nodes. These are bilinear in w1 and w2;
//              the  Native format is
//
//              w2(0)w1(0)  w2(0)w1(1) ... w2(0)w1(Np2)
//              w2(1)w1(0)  ...
//              .
//              .
//              w2(Np1)w1(0) ...           w2(Np1)w1(Np2)
//              This quantity is the diagonal matrix.
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::Get2DWeights()
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  return Weights2D;
} // end of method Get2DWeights


//************************************************************************************
//************************************************************************************
// METHOD     : GetMassMatrix 
// DESCRIPTION: returns pointer to 2D mass matrix
//              Note: The Jacobian factor is included in this
//              mass matrix. The matrix is diagonal....
// ARGUMENTS  :  
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::GetMassMatrix()
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
   return MassMatrix;

} // end of method GetMassMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : GetiMass
// DESCRIPTION: returns pointer to inverse of mass matrix
//              Note: this is, by definition, a fully assemebled matrix
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::GetiMass()
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
   return iMassMatrix;

} // end of method GetiMass


//************************************************************************************
//************************************************************************************
// METHOD     : GetgMass
// DESCRIPTION: returns pointer to global mass matrix
//              Note: this is, to be assembled, so that inverse
//                    may be taken
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::GetgMass()
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
   return gMassMatrix;

} // end of method GetgMass


//************************************************************************************
//************************************************************************************
// METHOD     : Assemble
// DESCRIPTION: assembles quantities that can be assembled
//              Note: Assembly is done on quantities from the outside. 
//                    Here, it means to use the assembled quantities to
//                    compute, e.g, local inverses which are of use to the
//                    element.
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
GBOOL RectQuad2D::Assemble()
{
  GINT  i;

  if ( !bSolved )
    if ( !SolveFE() ) return FALSE;

  // Compute assemble inverse mass matrix. The 
  // global mass matrix, gMassMatrix, should have 
  // been assembled by caller prior to entry.
  for ( i=0; i<gMassMatrix->dim(); i++ ) {
    (*iMassMatrix)(i) = 1.0/((*gMassMatrix)(i));
  }

  return TRUE;

} // end of method Assemble


//************************************************************************************
//************************************************************************************
// METHOD     : Get1DStiffMatrix
// DESCRIPTION: returns pointer to appropriate 1D stiffness matrix
//              Note: The Jacobian factor is _not_ included in this
//              mass matrix, but the appropriate 1D weight is.
// ARGUMENTS  :
// RETURNS    :  GMatrix *
//************************************************************************************
GMatrix  *RectQuad2D::Get1DStiffMatrix(GINT  idir, GBOOL bTranspose)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  GMatrix *ret = NULL;

  if ( idir == 1 )
  {
    if ( !bTranspose )
      ret = StiffMat1;
    else
      ret = StiffMat1T;

  }
  else if ( idir == 2 )
  {
    if ( !bTranspose )
      ret = StiffMat2;
    else
      ret = StiffMat2T;
  }

  return ret;

} // end of method Get1DStiffMatrix



//************************************************************************************
//************************************************************************************
// METHOD     : Get1DDerivMatrix 
// DESCRIPTION: returns pointer to 1d deriv. matrix, D/DXi1(2),
//              given the coord. direction, iDir. The derivative
//              is that computed in GNBasis: the 1d deriv of 
//              iDir-basis function wrt Xi-iDir, evaluated at all
//              iDir-quadrature points. That is,
//                DXi(i,j) = dh_j/dXi (Xi_i)  
//              Note: The Jacobian factor is _not_ included in this
//              derivative. The derivative is stored in native format.
// ARGUMENTS  :
// RETURNS    :  GMatrix *
//************************************************************************************
GMatrix *RectQuad2D::Get1DDerivMatrix(GINT  iDir, GBOOL bTranspose)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  if ( basis1 == NULL || basis2 == NULL ) return NULL;

  if ( iDir == 1 ) 
  {
    if ( !bTranspose ) return D1;
    else               return D1T;
  }
  else if ( iDir == 2 )
  {
    if ( !bTranspose ) return D2;
    else               return D2T;
  }
  else
    return NULL;
    
} // end of method Get1DDerivMatrix 



//************************************************************************************
//************************************************************************************
/**
 * METHOD     : Get1DDerivMatrixWeak 
 * DESCRIPTION: D^T M or its transpose.
 *              Note: The Jacobian factor is _not_ included in this
 *              derivative. The derivative is stored in native format.
 * ARGUMENTS  :
 * RETURNS    :  GMatrix *
 */
//************************************************************************************
GMatrix *RectQuad2D::Get1DDerivMatrixWeak(GINT  iDir, GBOOL bTranspose)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  if ( basis1 == NULL || basis2 == NULL ) return NULL;

  if ( iDir == 1 ) 
  {
    if ( !bTranspose ) return D1TM;
    else               return MD1;
  }
  else if ( iDir == 2 )
  {
    if ( !bTranspose ) return D2TM;
    else               return MD2;
  }
  else
    return NULL;
    
} // end of method Get1DDerivMatrixWeak 

//************************************************************************************
//************************************************************************************
/**
 * METHOD     : Get1DDerivMatrixWithMass
 * DESCRIPTION: M D or its transpose,
 *              Note: The Jacobian factor is _not_ included in this
 *              derivative. The derivative is stored in native format.
 * ARGUMENTS  :
 * RETURNS    :  GMatrix *
 */
//************************************************************************************
GMatrix *RectQuad2D::Get1DDerivMatrixWithMass(GINT  iDir, GBOOL bTranspose)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  if ( basis1 == NULL || basis2 == NULL ) return NULL;

  if ( iDir == 1 ) 
  {
    if ( !bTranspose ) return MD1;
    else               return D1TM;
  }
  else if ( iDir == 2 )
  {
    if ( !bTranspose ) return MD2;
    else               return D2TM;
  }
  else
    return NULL;
    
} // end of method Get1DDerivMatrixWithMass 


//************************************************************************************
//************************************************************************************
// METHOD     : GetXiNodes
// DESCRIPTION: Returns array to std element nodes.
//              Nodes are arranged in the order
//              xi1(0),...,xi1(N1); or i2(0),...,xi2(N2)
//              each group being arranged from smallest to
//              largest.
// ARGUMENTS  : GINT  idir gives coordinate direction whose Xi-coords are desired.
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::GetXiNodes(GINT  idir)
{

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  GVector *ret = NULL;

  if ( idir == 1 )
    ret = xiNodes1;
  else if ( idir == 2 )
    ret = xiNodes2;

  return ret;
} // end of method GetXiNodes


//************************************************************************************
//************************************************************************************
// METHOD     : GetWeights
// DESCRIPTION: Returns array quadrature weights in each parent domain 
//              coord. direction.
//              Weights correspond to nodes, which are ordered from smallest
//              to largest.
// ARGUMENTS  :
// RETURNS    :  GVector *
//************************************************************************************
GVector *RectQuad2D::Get1DWeights(GINT  idir)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  GVector *ret = NULL;

  if ( idir == 1 )
    ret = Weights1;
  else if ( idir == 2 )
    ret = Weights2;

  return ret;
} // end of method GetWeights



//************************************************************************************
//************************************************************************************
// METHOD     : GetBasisObj
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  pointer to basis object
//************************************************************************************
GNBasis *RectQuad2D::GetBasisObj(GINT  iDir)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
   if ( iDir == 1 ) 
     return basis1;
   else if ( iDir == 2 )
     return basis2;
   else
     return NULL;

} // end of method GetBasisObj



#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetBasisAtXi
// DESCRIPTION: Compute expansion basis at the parent domanin (xi) points.
//              This quantity really only has meaning for modal basis,
//              since for nodal bases, the basis is 1 at nodal points
//              (and 0 when not at nodal points). Provision is made for
//              this quantity, but only for specified 1- and 2-indices.
//              
// ARGUMENTS  : GINT  i, j, representing the 1- and 2-indices for labeling
//              the basis function, phi(i,j). The matrix ret is of type GMatrix,
//              and contains the basis fcn evaulated at the quadrature points, and
//              arranged s.t.
//
//              B(0,0) B(0,1)    ...   B(0,Np2)
//              B(1,0)           ...
//              .
//              .
//              B(Np1,0)         ...   B(Np1,Np2)
//
//              where the value of the quadrature 1- &  2- coord. increases with index.
// RETURNS    : GMatrix pointer of return matrix on success; else NULL.
//************************************************************************************
GMatrix *RectQuad2D::GetBasisAtXi(GINT  i, GINT  j, GMatrix *B)
{
  if ( B == NULL ) return NULL;
  if ( B->dim(1) != (Np1+1)  || B->dim(2) != (Np2+1) ) return NULL;

  if ( !bSolved )
    if ( !SolveFE() ) return NULL;

  GINT  l, m;
  GDOUBLE *b_data, *b1_data, *b2_data;
  GMatrix B1(Np1+1,Np1+1), B2(Np2+1,Np2+1);

  // Get 1- and 2- bases evaluated at the respective 1- and 2- quadrature points:
  if ( basis1->GetBasisAtNodes(&B1) == NULL ) return NULL;
  if ( basis2->GetBasisAtNodes(&B2) == NULL ) return NULL;
  
  // Note that each Ba from GetBasis is s.t.
  //   Ba(i,j) = Phi_i(xi_j)

  // When doing the multiplication here, it would be best not to make
  // the method calls with each (*,*) invocation, but instead work
  // direcly on the data.... This is done here in order to be explicit.
  b_data  = B->Data();
  b1_data = B1.Data();
  b2_data = B2.Data();
  for ( l=0; l<Np1+1; l++ )
  {
    for ( m=0; m<Np2+1; m++ )
    {
//    (*B)(l,m) = B1(i,l)*B2(j,m);
      *(b_data+i*(B->dim(2))+j) = (*(b1_data+i*(Np1+1)+l)) * (*(b2_data+j*(Np2+1)+m));
    }
  }
 
  return B;

} // end of method GetBasisAtXi
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : SetVertices
// DESCRIPTION: Set elements real-space vertices. These should be
//              set in counter clockwise order.
// ARGUMENTS  : Array of Point3D points representing vertex points.
//              There must be at least 4 of these, and only the
//              first 4 are used. Not very robust. Should be in 
//              counter clockwise order.
// RETURNS    :  none
//************************************************************************************
GBOOL RectQuad2D::SetVertices(Point3D p[], GINT  num)
{
  if ( num < nVertices_ ) return FALSE;

  GINT  j, m, n;
  
  for ( n=0; n<nVertices_; n++ )
  {
    spVertices[n] = p[n]; 
  }
  for ( n=0; n<GDIM*nVertices_; n+=2 ) 
  {
    spvVertices  [n] = spVertices[n/2].x1;
    spvVertices[n+1] = spVertices[n/2].x2;
  }

  // Compute the edge segment midpoints:
  for ( n=0; n<4; n++ ) {   // bottom plane
    m = (n+1) % 4;
    for ( j=0; j<GDIM; j++) {
      spMidpoints[n][j]= 0.5*(spVertices[m][j]+spVertices[n][j]);
    }
  }

  // Compute element center:
  (*elemCenter)[0] = spMidpoints[0][0];
  (*elemCenter)[1] = 0.5*(spMidpoints[0][1]+spMidpoints[2][1]);

#if defined(IS3D)
  for ( n=4; n<8; n++ ) {   // top plane, if in 3D
    m = (n+1) % 8;
    for ( j=0; j<GDIM; j++ ) { 
      spMidpoints[n][j]= 0.5*(spVertices[m][j]+spVertices[n][j]);
    }
  }
  for ( n=8,k=0; n<12; n++,k++ ) { // vertical edges, if in 3D
    m = 0.5*(n+k);   
    for ( j=0; j<GDIM; j++ ) {
      spMidpoints [n][j]= 0.5*(spVertices[m][j]+spVertices[m-4][j]);
    }
  }
#endif
  

  return TRUE;

} // end of method SetVertices 

#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis (1)
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void RectQuad2D::SetBasis(GNBasis *semb1, GNBasis *semb2)
{
  basis1 = semb1;
  basis2 = semb2;

  bSolved = FALSE;
  
  if ( basis1 == NULL || basis2 == NULL ) return;

  Resize(basis1->GetOrder(), basis2->GetOrder());

} // end of method SetBasis (1)


//************************************************************************************
//************************************************************************************
// METHOD     : SetBasis (2)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void RectQuad2D::SetBasis(GNBasis *b, GINT  idir)
{
  if ( idir < 1 || idir > GDIM) {
    cout << "RectQuad2D::SetBasis(2): invalid coordinate direction" << endl;
    exit(1);
  }
  if      ( idir == 1 ) basis1 = b;
  else if ( idir == 2 ) basis2 = b;

  bSolved = FALSE;
 
  if ( basis1 == NULL || basis2 == NULL ) return;
  Resize(basis1->GetOrder(), basis2->GetOrder());

} // end of method SetBasis (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetOrder
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void RectQuad2D::SetOrder(GINT  iorder1, GINT  iorder2)
{

   if ( iorder1 == Np1 && iorder2 == Np2  ) return;

   bInitialized = FALSE;
 
   Resize(iorder1, iorder2);

} // end of method SetOrder
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// ARGUMENTS  :
// RETURNS    :  none
//************************************************************************************
void RectQuad2D::DeleteDynamic()
{

  if ( Weights1    != NULL ) delete Weights1;
  if ( Weights2    != NULL ) delete Weights2;
  if ( Weights2D   != NULL ) delete Weights2D;
  if ( MassMatrix  != NULL ) delete MassMatrix;
  if ( gMassMatrix != NULL ) delete gMassMatrix;
  if ( iMassMatrix != NULL ) delete iMassMatrix;
  if ( StiffMat1   != NULL ) delete StiffMat1;
  if ( StiffMat2   != NULL ) delete StiffMat2;
  if ( StiffMat1T  != NULL ) delete StiffMat1T;
  if ( StiffMat2T  != NULL ) delete StiffMat2T;
  if ( xiNodes1    != NULL ) delete xiNodes1;
  if ( xiNodes2    != NULL ) delete xiNodes2;
  if ( spNodes1    != NULL ) delete spNodes1;
  if ( spNodes2    != NULL ) delete spNodes2;
  if ( D1          != NULL ) delete D1;
  if ( D2          != NULL ) delete D2;
  if ( D1T         != NULL ) delete D1T;
  if ( D2T         != NULL ) delete D2T;
  if ( MD1         != NULL ) delete MD1;
  if ( MD2         != NULL ) delete MD2;
  if ( D1TM        != NULL ) delete D1TM;
  if ( D2TM        != NULL ) delete D2TM;
  if ( opInterp1   != NULL ) delete opInterp1;
  if ( opInterp2   != NULL ) delete opInterp2;
  if ( opInterpT1  != NULL ) delete opInterpT1;
  if ( opInterpT2  != NULL ) delete opInterpT2;
  if ( dInterp1    != NULL ) delete dInterp1;
  if ( dInterp2    != NULL ) delete dInterp2;
  if ( dInterpT1   != NULL ) delete dInterpT1;
  if ( dInterpT2   != NULL ) delete dInterpT2;
  if ( xi1p        != NULL ) delete [] xi1p;
  if ( xi2p        != NULL ) delete [] xi2p;
  if ( xi1m        != NULL ) delete [] xi1m;
  if ( xi2m        != NULL ) delete [] xi2m;

  Weights1    = NULL;
  Weights2    = NULL;
  Weights2D   = NULL;
  MassMatrix  = NULL;
  gMassMatrix = NULL;
  iMassMatrix = NULL;
  StiffMat1   = NULL;
  StiffMat2   = NULL;
  StiffMat1T  = NULL;
  StiffMat2T  = NULL;
  xiNodes1    = NULL;
  xiNodes2    = NULL;
  spNodes1    = NULL;
  spNodes2    = NULL;
  D1          = NULL;
  D2          = NULL;
  D1T         = NULL;
  D2T         = NULL;
  MD1         = NULL;
  MD2         = NULL;
  D1TM        = NULL;
  D2TM        = NULL;
  opInterp1   = NULL;
  opInterp2   = NULL;
  opInterpT1  = NULL;
  opInterpT2  = NULL;
  dInterp1    = NULL;
  dInterp2    = NULL;
  dInterpT1   = NULL;
  dInterpT2   = NULL;


} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// ARGUMENTS  :
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL RectQuad2D::Resize(GINT  newOrder1, GINT  newOrder2)
{
    GINT  N1, N2, NN;

    // No resizing necessary if already the same size as
    // previously allocated quantities:
    if ( newOrder1 == Np1 && newOrder2 == Np2 ) return TRUE;

    if ( basis1 == NULL | basis2 == NULL ) return FALSE;

    bInitialized = FALSE;

    //  Resize bases:
    basis1->SetOrder(newOrder1);
    basis2->SetOrder(newOrder2);
    if ( !basis1->Solve() ) return FALSE;
    if ( !basis2->Solve() ) return FALSE;

    Np1 = newOrder1;
    Np2 = newOrder2;

    N1  = Np1 + 1;
    N2  = Np2 + 1;
    NN  = N1 * N2;

    //  Resize spNodes:
    spNodes1     ->Resize(NN);
    spNodes2     ->Resize(NN);

    //  Resize xiNodes:
    xiNodes1     ->Resize(N1);
    xiNodes2     ->Resize(N2);

    //  Resize Weights, MassMatrix StiffMats:
    Weights1     ->Resize(N1);
    Weights2     ->Resize(N2);
    Weights2D    ->Resize(NN);
    MassMatrix   ->Resize(NN);
    gMassMatrix  ->Resize(NN);
    iMassMatrix  ->Resize(NN);
    StiffMat1 ->Resize(N1,N1);
    StiffMat2 ->Resize(N2,N2);
    StiffMat1T->Resize(N1,N1);
    StiffMat2T->Resize(N2,N2);

    //  Resize Derivative matrices:
    D1        ->Resize(N1,N1);
    D1T       ->Resize(N1,N1);

    D2        ->Resize(N2,N2);
    D2T       ->Resize(N2,N2);

    MD1       ->Resize(N1,N1);
    D1TM      ->Resize(N1,N1);

    MD2       ->Resize(N2,N2);
    D2TM      ->Resize(N2,N2);

    nodal_multiplicity_->Resize(NN);
    *nodal_multiplicity_ = 1.0;

    mask_->Resize(NN);
    *mask_ = 1.0;
    bmask_->Resize(NN);
    *bmask_ = 1.0;

    SetIndices();

    // Resize misc objects:
    delete [] xi1p;
    delete [] xi1m;
    delete [] xi2p;
    delete [] xi2m;
    xi1p = new GDOUBLE [N1];
    xi1m = new GDOUBLE [N1];
    xi2p = new GDOUBLE [N2];
    xi2m = new GDOUBLE [N2];
    if ( xi1p == NULL || xi1m == NULL ||
         xi2p == NULL || xi2m == NULL ) return FALSE;

    bInitialized = TRUE;

    return bInitialized;

}  // end of method Resize

//************************************************************************************
//************************************************************************************
// METHOD     : Initialize
// DESCRIPTION: Initialize dynamically allocated quantities
//              Basis is not re-allocated here.
// ARGUMENTS  :
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
GBOOL RectQuad2D::Initialize(GINT  newOrder1, GINT  newOrder2)
{

    GINT  N1, N2, NN;


    Np1 = newOrder1;
    Np2 = newOrder2;

    N1  = Np1 + 1;
    N2  = Np2 + 1;
    NN  = N1 * N2;

    //  Resize spNodes:
    spNodes1 = new GVector(NN);
    if ( spNodes1 == NULL ) return FALSE;
    spNodes2 = new GVector(NN);
    if ( spNodes2 == NULL ) return FALSE;

    //  Resize xiNodes:
    xiNodes1 = new GVector(N1);
    if ( xiNodes1 == NULL ) return FALSE;
    xiNodes2 = new GVector(N2);
    if ( xiNodes2 == NULL ) return FALSE;

    //  Resize Weights, MassMatrix StiffMats:
    Weights1 = new GVector(N1);
    if ( Weights1 == NULL ) return FALSE;
    Weights2 = new GVector(N2);
    if ( Weights2 == NULL ) return FALSE;
    Weights2D = new GVector(NN);
    if ( Weights2D == NULL ) return FALSE;
    MassMatrix = new GVector(NN);
    if ( MassMatrix == NULL ) return FALSE;
    gMassMatrix = new GVector(NN);
    if ( gMassMatrix == NULL ) return FALSE;
    iMassMatrix = new GVector(NN);
    if ( iMassMatrix == NULL ) return FALSE;
    StiffMat1  = new GMatrix(N1,N1);
    if ( StiffMat1 == NULL ) return FALSE;
    StiffMat2  = new GMatrix(N2,N2);
    if ( StiffMat2 == NULL ) return FALSE;
    StiffMat1T  = new GMatrix(N1,N1);
    if ( StiffMat1T == NULL ) return FALSE;
    StiffMat2T  = new GMatrix(N2,N2);
    if ( StiffMat2T == NULL ) return FALSE;

    //  Resize Derivative matrices:
    D1  = new GMatrix(N1,N1);
    if ( D1== NULL ) return FALSE;
    D1T = new GMatrix(N1,N1);
    if ( D1T== NULL ) return FALSE;

    D2  = new GMatrix(N2,N2);
    if ( D2== NULL ) return FALSE;
    D2T = new GMatrix(N2,N2);
    if ( D2T== NULL ) return FALSE;

    MD1  = new GMatrix(N1,N1);
    if ( MD1== NULL ) return FALSE;
    D1TM = new GMatrix(N1,N1);
    if ( D1TM== NULL ) return FALSE;

    MD2  = new GMatrix(N2,N2);
    if ( MD2== NULL ) return FALSE;
    D2TM = new GMatrix(N2,N2);
    if ( D2TM== NULL ) return FALSE;

    if ( nodal_multiplicity_== NULL ) return FALSE;
    nodal_multiplicity_->Resize(NN);
    *nodal_multiplicity_ = 1.0;
  
    if ( mask_ == NULL ) return FALSE;
    mask_->Resize(NN);
    *mask_ = 1.0;

    if ( bmask_ == NULL ) return FALSE;
    bmask_->Resize(NN);
    *bmask_ = 1.0;

    SetIndices();

    xi1p = new GDOUBLE [Np1+1];
    xi1m = new GDOUBLE [Np1+1];
    xi2p = new GDOUBLE [Np2+1];
    xi2m = new GDOUBLE [Np2+1];
    if ( xi1p == NULL || xi1m == NULL ||
         xi2p == NULL || xi2m == NULL ) return FALSE;

    return TRUE;
} // end of method Initialize


//************************************************************************************
//************************************************************************************
// METHOD     : XToXi() 
// DESCRIPTION: converts the real space values into parent-domain values
// ARGUMENTS  : Point *x = pointer to real space points;
//              Point *xi= pointer to parent domain points
//              GINT  num= number of points to invert
//
//              NOTE: there will be problems if the number of
//              points in each array is not the same.
// RETURNS    : TRUE on success; FALSE on failure if the 
//              solution is imaginary, or undefined, or if input
//              point lies outside of element.
//************************************************************************************
GBOOL RectQuad2D::XToXi(Point3D x[], Point3D xi[], const GINT num)
{
  GDOUBLE eps;

  // Check that point is within element boundaries:
  if ( !Point_in_poly(spVertices, nVertices_, x, num) ) 
  {
    cout << " RectQuad2D::XToXi: one or more points lies outside element!" << endl;
    return FALSE;
  }

  if ( !bSolved )
    if ( !SolveFE() ) return FALSE;
  
  GINT     i, j;
  GDOUBLE  L[GDIM], iL[GDIM], x0[GDIM], ximin, ximax;
#if 0
  GDOUBLE a[2], b[2], c[2], d[2], A, B, C, ch;
  GDOUBLE x1, x2, L[GDIM], del0, del1, ximin, ximax, xip, xim, xi1tmp, xi2tmp;
#endif

  ximin = basis1->GetXimin()<basis2->GetXimin()?basis1->GetXimin():basis2->GetXimin();
  ximax = basis1->GetXimax()>basis2->GetXimax()?basis1->GetXimax():basis2->GetXimax();
  L [0] = spVertices[1].x1 - spVertices[0].x1;
  L [1] = spVertices[3].x2 - spVertices[0].x2;
  x0[0] = spVertices[0].x1;
  x0[1] = spVertices[0].x2;
  for ( j=0; j<GDIM; j++ ) iL[j] = 1.0 / L[j];

  eps = 10.0 * TINY;
  for ( i=0; i<num; i++ )
  {
//  xi[i].Bracket(SFPTINY);
    for ( j=0; j<GDIM; j++ ) {
      xi[i][j]   = 2.0*( x[i][j] - x0[j] ) * iL[j] - 1.0;
      if ( xi[i][j] <  1+eps  && xi[i][j] >  1.0-eps ) xi[i][j] = 1.0;
      if ( xi[i][j] < -1+eps  && xi[i][j] > -1.0-eps ) xi[i][j] = -1.0;
    }
//  xi[i].Truncate(); // remove all data outside xi' bracket
  }

#if 0
  a[0] = spVertices[0].x1 - spVertices[1].x1 + spVertices[2].x1 - spVertices[3].x1; 
  a[1] = spVertices[0].x2 - spVertices[1].x2 + spVertices[2].x2 - spVertices[3].x2; 
  
  b[0] =-spVertices[0].x1 + spVertices[1].x1 + spVertices[2].x1 - spVertices[3].x1; 
  b[1] =-spVertices[0].x2 + spVertices[1].x2 + spVertices[2].x2 - spVertices[3].x2; 

  c[0] =-spVertices[0].x1 - spVertices[1].x1 + spVertices[2].x1 + spVertices[3].x1; 
  c[1] =-spVertices[0].x2 - spVertices[1].x2 + spVertices[2].x2 + spVertices[3].x2; 

  d[0] = spVertices[0].x1 + spVertices[1].x1 + spVertices[2].x1 + spVertices[3].x1; 
  d[1] = spVertices[0].x2 + spVertices[1].x2 + spVertices[2].x2 + spVertices[3].x2; 

  for ( i=0; i<num; i++ )
  {
//  xi[i].Bracket(SFPTINY);
    x1   = x[i].x1;
    x2   = x[i].x2;
//  A    = a[0]*c[1] - a[1]*c[0];
    del0 = 4.0*x1-d[0];
    del1 = 4.0*x2-d[1];
    B    = a[1]*del0 - a[0]*del1 + a[0]*d[1]-a[1]*d[0] + b[0]*c[1]-b[1]*c[0];
    C    =-b[0]*del1 + b[1]*del0;

   // xi_2 is solution to quadratic equation:
   //  A xi^2 + B xi + C = 0:

    if ( fabs(A) > TINY ) // quadratic solution 
    {
      ch   = B*B - 4.0*A*C;

      if ( ch < 0 )
      {
        cout << "RectQuad2D::XToXi(1): xi_2 not real!" << endl;
        return FALSE;
      }

      xip   = ( -B + sqrt(ch) )  / ( 2.0*A);
      xim   = ( -B - sqrt(ch) )  / ( 2.0*A);

      if ( xip > ximax || xip < ximin ) 
        xi2tmp = xim;
      else if ( xim > ximax || xim < ximin ) 
        xi2tmp = xip;
      else 
        xi2tmp = xip;

#if 0
      if ( xi2tmp > ximax || xi2tmp < ximin ) 
      {
        cout << "RectQuad2D::XToXi(1): xi_2 out of range! xi_2=" << xi2tmp << endl;
        return FALSE;
      }
#endif
      xi2tmp = MAX(MIN(xi2tmp,ximax),ximin);
      xi[i].x2 = xi2tmp;
      xi1tmp = ( 4.0*x1 - c[0]*xi[i].x2 - d[0] ) / ( a[0]*(xi[i].x2) + b[0] );
#if 0
      if ( xi1tmp > ximax || xi1tmp < ximin ) 
      {
        cout << "RectQuad2D::XToXi(1): xi_1 out of range! (xi_1,xi_2)= (" << 
        xi1tmp << "," << xi2tmp << ")"  << endl;
        return FALSE;
      }
#endif
      xi1tmp = MAX(MIN(xi1tmp,ximax),ximin);
      xi[i].x1 = xi1tmp;
    }
    else  // linear solution (A=0)
    {
      if ( B == 0.0 )
      {
        cout << "RectQuad2D::XToXi(1): xi_2 singular!" << endl;
        return FALSE;
      }

      xi[i].x2 = -C/B;
      xi[i].x1 = ( 4.0*x1 - c[0]*xi[i].x2 - d[0] ) / ( a[0]*(xi[i].x2) + b[0] );
#if 0
    }
#endif
#if 0
    if ( x1 == spVertices[0].x1 ) xi[i].x1 = ximin;
    if ( x1 == spVertices[1].x1 ) xi[i].x1 = ximax;
    if ( x2 == spVertices[0].x2 ) xi[i].x2 = ximin;
    if ( x2 == spVertices[3].x2 ) xi[i].x2 = ximax;
#endif

//  xi[i].Truncate(); // remove all data outside xi' bracket
  }
#endif
  
  return TRUE;
} // end of method XToXi 



//************************************************************************************
//************************************************************************************
// METHOD     : SolveFE
// DESCRIPTION: computes quadrature points, weights, deriv. matrices 
//              for element. These are computed for the 1d component bases, 
//              and scatter arrays are set up that can determine the
//              V, E, and I parent nodal locations from the components.
// ARGUMENTS  :
// RETURNS    : integer status flag
//************************************************************************************
GINT  RectQuad2D::SolveFE()
{
  GINT     i, j, k;
  GDOUBLE  L1, L2, Jac;
 
  bSolved = TRUE;

  if ( !bInitialized || basis1 == NULL || basis2 == NULL ) return 0;
   
 
  // Compute quadrature points and corresponding weights for each coordinate
  // basis:

  if ( basis1->GetXiNodes(xiNodes1) == NULL ) return 0;
  if ( basis1->GetWeights(Weights1) == NULL ) return 0;
  if ( basis2->GetXiNodes(xiNodes2) == NULL ) return 0;
  if ( basis2->GetWeights(Weights2) == NULL ) return 0;
  if ( basis1->GetDerivMatrix(D1)   == NULL ) return 0;
  if ( basis2->GetDerivMatrix(D2)   == NULL ) return 0;
  D1->Transpose(*D1T);
  D2->Transpose(*D2T);
  if ( basis1->GetStiffMatrix(StiffMat1) == NULL ) return 0;
  if ( basis2->GetStiffMatrix(StiffMat2) == NULL ) return 0;
  StiffMat1->Transpose(*StiffMat1T);
  StiffMat2->Transpose(*StiffMat2T);
  L1   = fabs((spVertices+1)->x1 - (spVertices)->x1);
  L2   = fabs((spVertices+3)->x2 - (spVertices)->x2);
  if ( L1 == 0 || L2 == 0 ) return 0;
  Jac  = 0.25 * L2 * L1 ;
  area_ = L1 * L2;

  for ( j=0; j<Np2+1; j++ )
  {
    for ( i=0; i<Np1+1; i++ )
    {
      k = i + j*(Np1+1);
      (*MassMatrix)(k) = (*Weights1)(i) * (*Weights2)(j)*Jac; //ASC This should stay 2D
      (*Weights2D) (k) = (*Weights1)(i) * (*Weights2)(j);//ASC this should stay 2D
    }
  }
  
  // Compute weak and derivative with mass matrix

  for ( j=0; j<Np1+1; j++ ){
    for ( i=0; i<Np1+1; i++ )
      {
	(*MD1)(i,j) = (*Weights1)(i)*(*D1)(i,j);
	(*D1TM)(i,j) = (*D1T)(i,j)*(*Weights1)(j);
      }
  }
  
  for ( j=0; j<Np2+1; j++ ){
    for ( i=0; i<Np2+1; i++ )
      {
	(*MD2)(i,j) = (*Weights2)(i)*(*D2)(i,j);
	(*D2TM)(i,j) = (*D2T)(i,j)*(*Weights2)(j);
      }
  }
  

  *gMassMatrix = *MassMatrix;  // make deep copy

  bSolved = ComputeSpNodes();
  InitMortars();

  // Compute dealiasing qantities:
  return (GINT)ComputeDealias();
  
} // end of method SolveFE



//************************************************************************************
//************************************************************************************
// METHOD     : SetInterpBasis
// DESCRIPTION: 
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void RectQuad2D::SetInterpBasis(GNBasis *b1, GNBasis *b2)
{

  if ( b1 == NULL    || b2 == NULL    ) return;  // nothing to do...
  if ( b1 == gbasis1 && b2 == gbasis2 ) return;  // continue if either basis is different
  if ( !bSolved )
    if ( !SolveFE() ) return;                    // ensure that all is ok with this->basis

  gbasis1 = b1;
  gbasis2 = b2;

  // If basis to which to interpolate required 
  // quantities is valid, allocate and compute the 
  // quantities:

  GINT      i, j;
  GBOOL     bValid=TRUE;
  GVector *pxi1, *pxi2, *gW1, *gW2;
  
  if ( opInterp1  ) delete opInterp1 ;  opInterp1  = NULL;
  if ( opInterp2  ) delete opInterp2 ;  opInterp2  = NULL;
  if ( opInterpT1 ) delete opInterpT1;  opInterpT1 = NULL;
  if ( opInterpT2 ) delete opInterpT2;  opInterpT2 = NULL;
  if ( dInterp1   ) delete dInterp1  ;  dInterp1   = NULL;
  if ( dInterp2   ) delete dInterp2  ;  dInterp2   = NULL;

  // Instantiate new interp. basis quantities:
  opInterp1  = new GMatrix(gbasis1->GetOrder()+1,this->GetOrder(1)+1);
  opInterp2  = new GMatrix(gbasis2->GetOrder()+1,this->GetOrder(2)+1);
  opInterpT1 = new GMatrix(this->GetOrder(1)+1,gbasis1->GetOrder()+1);
  opInterpT2 = new GMatrix(this->GetOrder(2)+1,gbasis2->GetOrder()+1);
  dInterp1   = new GMatrix(gbasis1->GetOrder()+1,this->GetOrder(1)+1);
  dInterp2   = new GMatrix(gbasis2->GetOrder()+1,this->GetOrder(2)+1);
  dInterpT1  = new GMatrix(this->GetOrder(1)+1,gbasis1->GetOrder()+1);
  dInterpT2  = new GMatrix(this->GetOrder(2)+1,gbasis2->GetOrder()+1);
  pxi1       = new GVector(gbasis1->GetOrder()+1);
  pxi2       = new GVector(gbasis2->GetOrder()+1);
  gW1        = new GVector(gbasis1->GetOrder()+1);
  gW2        = new GVector(gbasis2->GetOrder()+1);
  gbasis1->GetXiNodes(pxi1);
  gbasis2->GetXiNodes(pxi2);
  gbasis1->GetWeights(gW1);
  gbasis2->GetWeights(gW2);
  
  if ( basis1->EvalBasis (pxi1,opInterp1) == NULL ) bValid = FALSE;
  if ( basis2->EvalBasis (pxi2,opInterp2) == NULL ) bValid = FALSE;
  if ( basis1->EvalDBasis(pxi1,dInterp1)  == NULL ) bValid = FALSE;
  if ( basis2->EvalDBasis(pxi2,dInterp2)  == NULL ) bValid = FALSE;
 
  if ( bValid )
  {
  // Interp and derivative operators include the weights:
    for ( j=0; j<opInterp1->dim(2); j++ )
    {
      for ( i=0; i<opInterp1->dim(1); i++ )
      {
        (*opInterp1)(i,j) *= (*gW1)(i);
        (*dInterp1) (i,j) *= (*gW1)(i);
      }
    }
    for ( j=0; j<opInterp2->dim(2); j++ )
    {
      for ( i=0; i<opInterp2->dim(1); i++ )
      {
        (*opInterp2)(i,j) *= (*gW2)(i);
        (*dInterp2) (i,j) *= (*gW2)(i);
      }
    }
  }
  if ( !opInterp1->Transpose(*opInterpT1) 
    || !opInterp2->Transpose(*opInterpT2) 
    || !dInterp1 ->Transpose (*dInterpT1) 
    || !dInterp2 ->Transpose (*dInterpT2) ) bValid == FALSE;

  delete pxi1;
  delete pxi2;
  delete gW1;
  delete gW2;

  // if any operation fails, destroy all interp. quantities:
  if ( !bValid )
  {
    if ( opInterp1  != NULL ) delete opInterp1;  opInterp1  = NULL;
    if ( opInterp2  != NULL ) delete opInterp2;  opInterp2  = NULL;
    if ( opInterpT1 != NULL ) delete opInterpT1; opInterpT1 = NULL;
    if ( opInterpT2 != NULL ) delete opInterpT2; opInterpT2 = NULL;
    if ( dInterp1   != NULL ) delete dInterp1;   dInterp1   = NULL;
    if ( dInterp2   != NULL ) delete dInterp2;   dInterp2   = NULL;
    if ( dInterpT1  != NULL ) delete dInterpT1;  dInterpT1  = NULL;
    if ( dInterpT2  != NULL ) delete dInterpT2;  dInterpT2  = NULL;
    cout << "RectQuad2D::SetInterpBasis: interpolated quantities NULL" << endl;
    exit(1);
    return;
  }
   
  
} // end of method SetInterpBasis



//************************************************************************************
//************************************************************************************
// METHOD     : GetInterpOp
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GMatrix *RectQuad2D::GetInterpOp(GINT  idir, GBOOL bTrans)
{
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
 
  if      ( idir == 1 )
  {
    if ( !bTrans ) return opInterp1;
    else           return opInterpT1;
  }
  else if ( idir == 2 )
  {
    if ( !bTrans ) return opInterp2;
    else           return opInterpT2;
  }
  else
    return NULL;

} // end of method GetInterpOp


//************************************************************************************
//************************************************************************************
// METHOD     : GetInterpDeriv
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
GMatrix *RectQuad2D::GetInterpDeriv(GINT  idir, GBOOL bTrans)
{
  
  if ( !bSolved )
    if ( !SolveFE() ) return NULL;
  if      ( idir == 1 )
  {
    if ( !bTrans ) return dInterp1;
    else           return dInterpT1;
  }
  else if ( idir == 2 )
  {
    if ( !bTrans ) return dInterp2;
    else           return dInterpT2;
  }
  else
    return NULL;

} // end of method GetInterpDeriv


//************************************************************************************
//************************************************************************************
// METHOD     : Integrate
// DESCRIPTION: Computes integral of input vector over full geometric element.
//              s.t.
//                int  = Int   v  dx dy 
//              Note: this could be placed in the base class, if MassMatrix is
//                    also included.
// ARGUMENTS  : v            : integrand as a GVector
//              imultiplicity: inverse of node multiplicity; used if non-NULL. 
//                             Must be of the same length as v.
// RETURNS    :
//************************************************************************************
GDOUBLE RectQuad2D::Integrate(GVector *v, GDOUBLE *imultiplicity)
{

  GINT  j;
  GDOUBLE sum;

  if ( !bSolved &&  !SolveFE() ) {
    cout << "RectQuad2D::Integrate: Element solve failed" << endl;
    exit(1);
  }

  if ( v == NULL ) {
    cout << "RectQuad2D::Integrate: NULL integrand" << endl;
    exit(1);
  }

  if ( imultiplicity == NULL ) {
    sum = MTK::fvec_dot(*v, *MassMatrix);
  }
  else {
    for ( j=0, sum=0.0; j<v->dim(); j++ ) {
      sum += (*v)(j)*(*MassMatrix)(j)*imultiplicity[j];
    }
  }
  
  return sum;

} // end of method Integrate


//************************************************************************************
//************************************************************************************
// METHOD     : PIntegrate
// DESCRIPTION: Computes integral of input vector over 'parent' element subdomain
//              s.t.
//                int  = Int_{-1}^{1}   v  dxi_1 dxi_2 
//              Note: this could be placed in the base class, if Weights2D is
//                    also included.
// ARGUMENTS  : v            : integrand as a GVector
//              imultiplicity: node multiplicity inverse; used if non-NULL. Must be of the same
//                            length as v.
// RETURNS    :
//************************************************************************************
GDOUBLE RectQuad2D::PIntegrate(GVector *v, GDOUBLE *imultiplicity)
{

  GINT  j;
  GDOUBLE sum;

  if ( !bSolved &&  !SolveFE() ) {
    cout << "RectQuad2D::PIntegrate: Element solve failed" << endl;
    exit(1);
  }

  if ( v == NULL ) {
    cout << "RectQuad2D::PIntegrate: NULL integrand" << endl;
    exit(1);
  }

  if ( imultiplicity == NULL ) {
    sum = MTK::fvec_dot(*v, *Weights2D);
  }
  else {
    for ( j=0, sum=0.0; j<v->dim(); j++ ) {
      sum += (*v)(j)*(*Weights2D)(j)*imultiplicity[j];
    }
  }
  
  return sum;

} // end of method PIntegrate


//************************************************************************************
//************************************************************************************
// METHOD     : Differentiate
// DESCRIPTION: Computes derivative of input vector in direction idir
// ARGUMENTS  : dv  : output arguement: derivative in idir direction of v
//              v   : input argument as a GVector
//              idir: coordinate direction
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL RectQuad2D::Differentiate(GVector *dv, GVector *v, GINT  idir)
{
  GINT    NN, N1, N2;
  GDOUBLE   L1, L2, iL1, iL2;


  if ( v == NULL || dv == NULL ) return FALSE;

  L1   = PDISTANCE(spVertices[1],spVertices[0]);
  L2   = PDISTANCE(spVertices[2],spVertices[1]);
  if ( L1 == 0 || L2 == 0 ) return FALSE;

  iL1 = 2.0/L1;
  iL2 = 2.0/L2;

  N1 = Np1 + 1;
  N2 = Np2 + 1;
  NN = N1  * N2;
  if ( v->dim() != NN || dv->dim() != NN ) return FALSE;

  if ( idir == 1 ) {
    // Do I2 X D1 term: 
    MTK::I2_X_D1(*D1, *v, N1, N2, *dv);
    if ( iL1 != 1.0 ) MTK::fvec_const_prod_rep(*dv, iL1);
  }
  else if ( idir == 2 ) {
    // Do D2 X I1 term: 
    MTK::D2_X_I1(*D2T, *v, N1, N2, *dv);
    if ( iL2 != 1.0 ) MTK::fvec_const_prod_rep(*dv, iL2);
  }

  return TRUE;

} // end of method Differentiate


//************************************************************************************
//************************************************************************************
/**
 * METHOD     : DifferentiateWithMass
 * DESCRIPTION: Computes derivative of input vector in direction idir with mass matrix
 * ARGUMENTS  : dv  : output arguement: derivative in idir direction of v
 *              v   : input argument as a GVector
 *              idir: coordinate direction
 * RETURNS    : TRUE on success; else FALSE
 */
//************************************************************************************
GBOOL RectQuad2D::DifferentiateWithMass(GVector *dv, GVector *v, GVector* tmp, GINT  idir)
{
  GINT    NN, N1, N2;
  GDOUBLE   L1, L2, iL1, iL2;

  if ( v == NULL || dv == NULL ) return FALSE;

  L1   = PDISTANCE(spVertices[1],spVertices[0]);
  L2   = PDISTANCE(spVertices[2],spVertices[1]);
  if ( L1 == 0 || L2 == 0 ) return FALSE;

  iL1 = L2/2.0;
  iL2 = L1/2.0;

  N1 = Np1 + 1;
  N2 = Np2 + 1;
  NN = N1  * N2;
  if ( v->dim() != NN || dv->dim() != NN ) return FALSE;

  if ( idir == 1 ) {
    // Do Dg2 X D1 term: 
    MTK::Dg2_X_D1(*MD1,*Weights2, *v, *tmp, *dv);
    if ( iL1 != 1.0 ) MTK::fvec_const_prod_rep(*dv, iL1);
  }
  else if ( idir == 2 ) {
    // Do D2 X D1 term: 
    MTK::D2_X_Dg1(*D2TM,*Weights1, *v, *tmp, *dv);
    if ( iL2 != 1.0 ) MTK::fvec_const_prod_rep(*dv, iL2);
  }

  return TRUE;

} // end of method DifferentiateWithMass


//************************************************************************************
//************************************************************************************
/**
 * METHOD     : DifferentiateWeak
 * DESCRIPTION: Computes weak derivative of input vector in direction idir with mass matrix
 * ARGUMENTS  : dv  : output arguement: derivative in idir direction of v
 *              v   : input argument as a GVector
 *              idir: coordinate direction
 * RETURNS    : TRUE on success; else FALSE
 */
//************************************************************************************
GBOOL RectQuad2D::DifferentiateWeak(GVector *dv, GVector *v, GVector *tmp,GINT  idir)
{
  GINT    NN, N1, N2;
  GDOUBLE   L1, L2, iL1, iL2;

  if ( v == NULL || dv == NULL ) return FALSE;

  L1   = PDISTANCE(spVertices[1],spVertices[0]);
  L2   = PDISTANCE(spVertices[2],spVertices[1]);
  if ( L1 == 0 || L2 == 0 ) return FALSE;

  iL1 = L2/2.0;
  iL2 = L1/2.0;

  N1 = Np1 + 1;
  N2 = Np2 + 1;
  NN = N1  * N2;
  if ( v->dim() != NN || dv->dim() != NN ) return FALSE;

  if ( idir == 1 ) {
    // Do Dg2 X D1 term: 
    MTK::Dg2_X_D1(*D1TM,*Weights2, *v, *tmp, *dv);
    if ( iL1 != 1.0 ) MTK::fvec_const_prod_rep(*dv, -iL1);// Notice minus sign here
  }
  else if ( idir == 2 ) {
    // Do D2 X D1 term: 
    MTK::D2_X_Dg1(*MD2,*Weights1, *v, *tmp, *dv);
    if ( iL2 != 1.0 ) MTK::fvec_const_prod_rep(*dv, -iL2);//Notice minus sign here
  }
  return TRUE;

} // end of method DifferentiateWeak



//************************************************************************************
//************************************************************************************
// METHOD     : ComputeDealias
// DESCRIPTION: Computes quantities used in performing dealiasing.
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL RectQuad2D::ComputeDealias()
{
  GINT     i, j, k, DNp[GDIM];
  GBOOL    bDealiased;
  GDOUBLE  L1, L2, Jac;
  GVector  dweights[GDIM], pxi, dpxi;
  GNBasis  *b[GDIM];

  for ( j=0, bDealiased=TRUE; j<GDIM; j++ )
    bDealiased = bDealiased && dealias_basis_[j] != NULL;

  if ( !bDealiased ) return TRUE;

  b[0] = basis1;
  b[1] = basis2;
  L1   = fabs((spVertices+1)->x1 - (spVertices)->x1);
  L2   = fabs((spVertices+3)->x2 - (spVertices)->x2);
  if ( L1 == 0 || L2 == 0 ) return FALSE;
  Jac  = 0.25 * L2 * L1 ;

  for ( j=0; j<GDIM; j++ ) {
    DNp     [j] = dealias_basis_[j]->GetOrder();
    dweights[j].Resize(DNp[j]+1);
    dealias_basis_[j]->GetWeights(&dweights[j]);
  } 
  DMassMatrix_.Resize((DNp[0]+1)*(DNp[1]+1));
  for ( j=0; j<DNp[1]+1; j++ ) {
    for ( i=0; i<DNp[0]+1; i++ ) {
      k = i + j*(DNp[0]+1);
      DMassMatrix_[k] = dweights[0][i] * dweights[1][j] * Jac;
    }
  }

  // Compute JD, JDT:
  for ( j=0; j<GDIM; j++ ) {
    JD_            [j].Resize(DNp[j]+1,this->GetOrder(j+1)+1);
    JDT_           [j].Resize(this->GetOrder(j+1)+1,DNp[j]+1);
//  pxi               .Resize(b[j]->GetOrder()+1);
    dpxi              .Resize(dealias_basis_[j]->GetOrder()+1);
//  b              [j]->GetXiNodes(&pxi);
    dealias_basis_ [j]->GetXiNodes(&dpxi);
    b              [j]->EvalBasis(&dpxi,&JD_[j]);
    JD_            [j].Transpose(JDT_[j]);
//  dealias_basis_ [j]->EvalBasis(&pxi,&JDT_[j]);
  }

  return TRUE;
} // end of method ComputeDealias
