//************************************************************************************//
// Module       : schurlapop.cpp
// Date         : 7/16/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the Laplacian operator derived from a Schur
//                complement decompoosition of the 
//                discrete form  of the Stokes equations
//                where the pdV work term has a pressure residing 
//                on a different grid than the velocity, within
//                the same element. It is intended to act on the
//                p-quantity in the p Div v term. Its form is
//                 E =  D1 M^-1 D1^T  + D2 M^-1 D2^T
//                where M is the mass matrix, and Di are the 
//                weak pdV derivative operators defined in SchurLapOp.
//                velem and pelem are the elements where the 
//                v-quantity  and p-quantity reside, respectively.
//
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "schurlapop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "rectquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
SchurLapOp::SchurLapOp()
: LinOp(),
bCentralOp_(TRUE),
mConst     (1.0),
velem      (NULL),
pelem      (NULL),
vtmpp0     (NULL),
vtmpv0     (NULL),
vtmpv1     (NULL),
v1bdy      (NULL),
v2bdy      (NULL),
M          (NULL),
iM         (NULL),
pD1        (NULL),
pD2        (NULL),
pDT1       (NULL),
pDT2       (NULL)
{
   vtmpp0 = new GVector();
   vtmpv0 = new GVector();
   vtmpv1 = new GVector();
   M      = new GVector();
   pD1    = new StokesOp();
   pD2    = new StokesOp();
   pDT1   = new StokesOp();
   pDT2   = new StokesOp();
   pDT1->Transpose();
   pDT1->SetDir(1);
   pD1 ->SetDir(1);
   pDT2->Transpose();
   pDT2->SetDir(2);
   pD2 ->SetDir(2);

} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
SchurLapOp::SchurLapOp(Elem2D *ve, Elem2D *pe)
: LinOp( ), 
bCentralOp_(TRUE),
mConst     (1.0),
velem      (ve),
pelem      (pe),
vtmpp0     (NULL),
vtmpv0     (NULL),
vtmpv1     (NULL),
v1bdy      (NULL),
v2bdy      (NULL),
M          (NULL),
iM         (NULL),
pD1        (NULL),
pD2        (NULL),
pDT1       (NULL),
pDT2       (NULL)
{
   vtmpp0 = new GVector();
   vtmpv0 = new GVector();
   vtmpv1 = new GVector();
   M      = new GVector();
   pD1    = new StokesOp();
   pD2    = new StokesOp();
   pDT1   = new StokesOp();
   pDT2   = new StokesOp();
   pDT1->Transpose();
   pDT1->SetDir(1);
   pD1 ->SetDir(1);
   pDT2->Transpose();
   pDT2->SetDir(2);
   pD2 ->SetDir(2);

   Init (ve, pe); 

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
SchurLapOp::~SchurLapOp()
{
  DeleteDynamic();
}

//************************************************************************************
//************************************************************************************
// METHOD     : Init
// DESCRIPTION: Initializes operators and data
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void SchurLapOp::Init(Elem2D *ve, Elem2D *pe)
{
  GINT  NNv, NNp;

  velem = ve;
  pelem = pe;

  // Initialize the interpolated quantities:
  velem->SetInterpBasis(pelem->GetBasisObj(1), pelem->GetBasisObj(2));
 
  // Instantiate the required operators:
  pD1->SetElem(velem,pelem);
  pD2->SetElem(velem,pelem);

  pDT1->SetElem(velem,pelem);
  pDT2->SetElem(velem,pelem);

  NNv = (velem->GetOrder(1)+1)*(velem->GetOrder(2)+1);
  NNp = (pelem->GetOrder(1)+1)*(pelem->GetOrder(2)+1);
  vtmpp0->Resize(NNp); 
  vtmpv0->Resize(NNv);
  vtmpv1->Resize(NNv);
  M     ->Resize(NNv);
  *M = *(velem->GetMassMatrix()); // make deep copy from element; will be gather/scattered
} // end of method Init


//************************************************************************************
//************************************************************************************
// METHOD     : SetVBdyData
// DESCRIPTION: Sets v-bdy conditions
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void SchurLapOp::SetVBdyData(GINT idir, GVector *vbdy) 
{
  if ( idir < 1 || idir > GDIM ) {
    cout << "SchurLapOp::SetVBdyData: invalid component" << endl;
    exit(1);
  }
  if ( idir == 1 ) v1bdy = vbdy;
  if ( idir == 2 ) v2bdy = vbdy;

} // end of method SetVBdyData


//************************************************************************************
//************************************************************************************
// METHOD     : GetMassMatrix
// DESCRIPTION: gets pointer to local mass matrix
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
GVector *SchurLapOp::GetMassMatrix()
{
  return M;
} // end of method GetMassMatrix


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constant
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void SchurLapOp::SetConst(GDOUBLE c)
{
  mConst = c;
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void SchurLapOp::SetElem(Elem2D *ve, Elem2D *pe)
{
   if ( ve != velem || pe != pelem )
   {
     Reset();
     Init (ve, pe); 
   }

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : DoCentralOp
// DESCRIPTION: Carry out central operator multiplication (here M^-1)?
// ARGUMENTS  : GBOOL bop
// RETURNS    : none
//************************************************************************************
void SchurLapOp::DoCentralOp(GBOOL bop)
{   
  bCentralOp_ = bop;
} // end of method DoCentralOp


//************************************************************************************
//************************************************************************************
// METHOD     : GetiMass
// DESCRIPTION: Gets inverse of mass matrix
// ARGUMENTS  : none
// RETURNS    : GVector *--> to inverse of mass matrix vector
//************************************************************************************
GVector *SchurLapOp::GetiMass()
{
  return iM;
} // end of method GetiMass



//************************************************************************************
//************************************************************************************
// METHOD     : GetElem
// DESCRIPTION: Gets associated v-space element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
Elem2D *SchurLapOp::GetElem()
{
  return velem;
} // end of method GetElem


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : GetPElem
// DESCRIPTION: Gets associated p-space element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
Elem2D *SchurLapOp::GetPElem()
{
  return pelem;
} // end of method GetPElem
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : Reset
// DESCRIPTION:
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void SchurLapOp::Reset()
{
  vtmpp0->Resize(0);
  vtmpv0->Resize(0);
  vtmpv1->Resize(0);
} // end of method Reset


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void SchurLapOp::DeleteDynamic()
{
  if ( pD1    != NULL ) delete pD1;    pD1   = NULL;
  if ( pD2    != NULL ) delete pD2;    pD2   = NULL;
  if ( pDT1   != NULL ) delete pDT1;   pDT1  = NULL;
  if ( pDT2   != NULL ) delete pDT2;   pDT2  = NULL;
  if ( vtmpp0 != NULL ) delete vtmpp0;  vtmpp0 = NULL;
  if ( vtmpv0 != NULL ) delete vtmpv0;  vtmpv0 = NULL;
  if ( vtmpv1 != NULL ) delete vtmpv1;  vtmpv1 = NULL;
  if ( M      != NULL ) delete M     ;  M      = NULL;
} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
GVector  SchurLapOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of operator \* 


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation (1)
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector  args.
// RETURNS    : none
//************************************************************************************
void SchurLapOp::OpVec_prod(GVector &q, GVector &ret)  
{
  if ( velem == NULL || pelem == NULL ) 
  {
    cout << "SchurLapOp::OpVec_prod: Invalid elements or vector components" << endl;
    exit(1);
  }

  if ( velem->ElemType() != pelem->ElemType() )
  {
    cout << "SchurLapOp::OpVec_prod: Scalar and vector have imcompatible elements" << endl;
    exit(1);
  }
  iM = velem->GetiMass();
  if ( iM == NULL || 
       pD1  == NULL || pD2  == NULL || 
       pDT1 == NULL || pDT2 == NULL  )  
  {
    cout << "SchurLapOp::OpVec_prod: Derivative operators improperly instantiated" << endl;
    exit(1);
  }

  GBOOL    bRet = TRUE;

  switch (velem->ElemType())
  {
    case DEFORMED_QUAD:
    case RECT_QUAD:
      bRet = QuadOp(&q,&ret);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&q,&ret);
      break;
    default:
      cout << "SchurLapOp::OpVec_prod: Invalid element type" << endl;
      exit(1);
  }
  if ( !bRet )
  {
    cout << "SchurLapOp::OpVec_prod: Operator product failed" << endl;
    exit(1);
  }

} // end of method OpVec_prod (1)


//************************************************************************************
//************************************************************************************
// METHOD     : QuadOp
// DESCRIPTION: Computes operator*vector  for a quad element.
//              Intermediate quantities, iM should have
//              been computed by velem, after,
//              perhaps, a DSSum.
// ARGUMENTS  : p   : vector argument (in the pressure space)
//              pn  : resulting product (in the pressure space)
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL SchurLapOp::QuadOp(GVector *p, GVector *pn)
{

  if ( p  == NULL || pn  == NULL ) return FALSE;

//*pn =    ( (*pD1) * ( (*iM) * ((*pDT1)*(*p)) ) );
//*pn +=   ( (*pD2) * ( (*iM) * ((*pDT2)*(*p)) ) );

  // Component 1: pD1: vel->press; pD1T: press->vel : D1 M^-1 D1^T p:
//cout << "SchurLapOp: p    =" << *p << endl;
  pDT1->OpVec_prod(*p     , *vtmpv0);
  if ( mask        ) MTK::fvec_point_prod_rep(*vtmpv0, *mask);
  if ( v1bdy       ) MTK::fvec_add_rep       (*vtmpv0, *v1bdy);
//cout << "SchurLapOp: D1T p=" << *vtmpv0 << endl;
  if ( bCentralOp_ ) MTK::fvec_point_prod_rep(*vtmpv0, *iM);
  if ( mask        ) MTK::fvec_point_prod_rep(*vtmpv0, *mask);
  pD1 ->OpVec_prod(*vtmpv0, *pn );
//cout << "SchurLapOp: D1 iM D1T p=" << *pn << endl;

  // Component 2: D2 M^-1 D2^T p :
  pDT2->OpVec_prod(*p     , *vtmpv0);
  if ( mask        ) MTK::fvec_point_prod_rep(*vtmpv0, *mask);
  if ( v2bdy       ) MTK::fvec_add_rep       (*vtmpv0, *v2bdy);
  if ( bCentralOp_ ) MTK::fvec_point_prod_rep(*vtmpv0, *iM);
  if ( mask        ) MTK::fvec_point_prod_rep(*vtmpv0, *mask);
  pD2 ->OpVec_prod(*vtmpv0, *vtmpp0);

  
  MTK::fvec_add_rep(*pn, *vtmpp0);
  
  if ( mConst != 1.0 )
     MTK::fvec_const_prod_rep(*pn, mConst);

//cout << "SchurLapOp: Sum Di iM DiT p=" << *pn << endl;
  return TRUE;
} // end of method QuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp (1)
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : p   : vector argument   (in pressure space)
//              pn  : resulting product (in pressure space)
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL SchurLapOp::TriangleOp(GVector *p, GVector *pn)
{
  return FALSE;
} // end of method TriangleOp (1)


