//************************************************************************************//
// Module       : stokesop.cpp
// Date         : 7/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the pdV work operator. This weak operator is associated with
//                the following:
//                   (p,Div _v_)
//                The action of the operator is to multiply the
//                interpolated derivative (I2 X D1 or D2 X I1), by the
//                input vector. I1(2) is the interpolation operator,
//                I_ij = h_j(eta_i), where h_j is the basis fcn for
//                v in the 1(2) direction, and eta_i is the 1(2)
//                parent node of p. D1(2)_ij = dh_j(eta_i)/dxi.
//
//                Thus, the operator is actually a weak (integrated)
//                derivative  operator acting on the vector space of
//                the parent domain (the velocity-space), to produce
//                a vector in the subspace (the pressure-space).

//                This (Stokes) operator is somewhat different
//                from the others, in that it is assumed that p and v
//                reside on different grid nodal points within the same
//                element, thus requiring that a complete geometric
//                description of both field quantities be provided.
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "stokesop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "rectquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
StokesOp::StokesOp()
: LinOp(),
icomp      (1)  ,
Nv1        (0),
Np1        (0),
Nv2        (0),
Np2        (0),
bTranspose (FALSE),
pConst     (1.0),
velem      (NULL),
pelem      (NULL)
{
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
StokesOp::StokesOp(Elem2D *ve, Elem2D *pe, GINT  idir)
: LinOp( ),
icomp      (idir)  ,
Nv1        (0),
Np1        (0),
Nv2        (0),
Np2        (0),
bTranspose (FALSE),
pConst     (1.0),
velem      (ve),
pelem      (pe)
{

  if ( velem == NULL || pelem == NULL ) {
    cout << "StokesOp::StokesOp (2): one or more NULL elements" << endl;
    exit(1);
  }
  if ( idir != 1 && idir != 2 ) {
    cout << "StokesOp::StokesOp (2): invalid component direction" << endl;
    exit(1);
  }

  // Initialize the interpolated quantities:
  velem->SetInterpBasis(pelem->GetBasisObj(1), pelem->GetBasisObj(2));
  // Deriv ops, D1 and D2 are s.t. D1 = D1(N
  Np1 = pelem->GetOrder(1)+1;
  Nv1 = velem->GetOrder(1)+1;
  Np2 = pelem->GetOrder(2)+1;
  Nv2 = velem->GetOrder(2)+1;
    
  vtmp.Resize(Np2*Nv2);

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
StokesOp::~StokesOp()
{
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constant
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void StokesOp::SetConst(GDOUBLE pc)
{
  pConst = pc;
} // end of method SetConst

//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void StokesOp::SetElem(Elem2D *ve, Elem2D *pe)
{
  velem = ve;
  pelem = pe;

  if ( velem == NULL || pelem == NULL ) {
    cout << "StokesOp::StokesOp (2): NULL elements" << endl;
    exit(1);
  }
  velem->SetInterpBasis(pelem->GetBasisObj(1), pelem->GetBasisObj(2));
  Np1 = pelem->GetOrder(1)+1;
  Nv1 = velem->GetOrder(1)+1;
  Np2 = pelem->GetOrder(2)+1;
  Nv2 = velem->GetOrder(2)+1;
  vtmp.Resize(Np2*Nv2);


} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetDir
// DESCRIPTION: Sets desired vector component
// ARGUMENTS  : GINT  
// RETURNS    : none
//************************************************************************************
void StokesOp::SetDir(GINT  idir)
{
  icomp = idir;
} // end of method SetDir


//************************************************************************************
//************************************************************************************
// METHOD     : Transpose
// DESCRIPTION: Toggles condition that the operator should be
//              transposed
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void StokesOp::Transpose()
{
  bTranspose = !bTranspose;
} // end of method Transpose


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful) 
//************************************************************************************
GVector  StokesOp::operator*(GVector q)
{
  GVector *newq;

  if ( bTranspose )         // returns v-space vector
    newq = new GVector(Nv1*Nv2);
  else                      // returns p-space vector
    newq = new GVector(Np1*Np2);

  OpVec_prod(q, *newq);
 
  return *newq;

}  // end of operator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector args.
// RETURNS    : none
//************************************************************************************
void StokesOp::OpVec_prod(GVector &q, GVector &ret)  
{
  if ( velem == NULL || pelem == NULL || icomp < 1 || icomp > 2) {
    cout << "StokesOp::operator*: Invalid elements or vector components" << endl;
    exit(1);
  }

  if ( velem->ElemType() != pelem->ElemType() ) {
    cout << "StokesOp::operator*: Scalar and vector have imcompatible elements" << endl;
    exit(1);
  }

  GBOOL    bRet = TRUE;

  switch (velem->ElemType()) {
    case DEFORMED_QUAD:
      bRet = DefmQuadOp(&q,&ret);
      break;
    case RECT_QUAD:
      bRet = RectQuadOp(&q,&ret);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&q,&ret);
      break;
    default:
      cout << "StokesOp::operator*: Invalid element type" << endl;
      exit(1);
  }
  if ( !bRet ) {
    cout << "StokesOp::operator*: Operator product failed" << endl;
    exit(1);
  }

} // end of method oerator OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadOp
// DESCRIPTION: Computes operator*vector  for a deformed quad element.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL StokesOp::DefmQuadOp(GVector *v, GVector *vn)
{
  if ( v  == NULL || vn  == NULL ) return FALSE;

  GINT      j, k, N1= pelem->GetOrder(1)+1, 
            N2= pelem->GetOrder(2)+1, NN=N1*N2;
  GVector  *g1, *g2;
  GVector  d1v  (NN), d2v  (NN);
  GMatrix  *I1, *I2, *D1, *D2; 
  DiagOp   *diag1, *diag2;

  I1   = velem->GetInterpOp   (1, bTranspose); // already includes weights
  I2   = velem->GetInterpOp   (2,!bTranspose); // already includes weights
  D1   = velem->GetInterpDeriv(1, bTranspose); // already includes weights
  D2   = velem->GetInterpDeriv(2,!bTranspose); // already includes weights
//vert = velem->GetSpVertices();
  
  if ( icomp == 1 ) {
    g1  = ((DefQuad2D*)velem)->GetdXidX(1,1);
    g2  = ((DefQuad2D*)velem)->GetdXidX(2,1);
  }
  else {
    g1  = ((DefQuad2D*)velem)->GetdXidX(1,2);
    g2  = ((DefQuad2D*)velem)->GetdXidX(2,2);
  }

  if ( I1  == NULL || I2  == NULL ||
       D1  == NULL || D2  == NULL ||
       g1  == NULL || g2  == NULL ) return FALSE;


  if ( v->dim() != NN  || vn->dim() != NN ) return FALSE;      


  // Compute tensor product derivative operators:
  // ... (I2 X D1_)  v:
  for ( j=0; j<N2; j++ ) {
    d1v(0,N1-1,1,j*N1);  // Change computational slice 
    for ( k=0; k<N2; k++ ) {
      (*v) (0,N1-1,1,k*N1);  // Change computational slice
      d1v += ( (*D1) * (*v) ) * (*I2)(j,k);
    }
  }
  d1v(0,vn->dim()-1,1,0); // Reset slice under consideration to full vector

  // ... (D2_ X I1)  v:
  for ( j=0; j<N2; j++ ) {
    d2v(0,N1-1,1,j*N1);  // Change computational slice
    for ( k=0; k<N2; k++ ) {
      (*v)(0,N1-1,1,k*N1);  // Change computational slice
      d2v +=  ( (*v) * (*D2)(j,k) ) * (*I1)(j,k);
    }
  }
  d2v   (0,NN-1,1,0); // Reset slice under consideration to full vector
  (*v)  (0,NN-1,1,0); // Reset slice under consideration to full vector


  // Multiply tensor product by geometry factors s.t.
  // gd_i_u = Sum_i G_i,j du_j:
  diag1 = new DiagOp(g1);
  diag2 = new DiagOp(g2);
  *vn = ( (*diag1) * d1v )  + ( (*diag2) * d2v );
  delete diag1;
  delete diag2;

  if ( pConst != 1.0 ) 
     MTK::fvec_const_prod_rep(*vn, pConst); 
 
  return TRUE;
} // end of method DefmQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadOp
// DESCRIPTION: Computes operator*vector  for a regular rect. quad element.
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL StokesOp::RectQuadOp(GVector *v, GVector *vn)
{
  GINT      nv1, nv2, nvn1, nvn2, nt1, nt2;
  GDOUBLE   L1, L2, g  ;
  Point3D   *vert;
  GMatrix   *I, *D;

  if ( v  == NULL || vn  == NULL ) {
    return FALSE;
  }

  // Transpose of operator is (I X D)^T = (D^T X I^T)
  vert = velem->GetSpVertices();
  L1   = fabs((vert+1)->x1 - (vert)->x1);
  L2   = fabs((vert+3)->x2 - (vert)->x2);
  
  if ( L1 == 0 || L2 == 0 ) {
     return FALSE;
  }

  if ( !bTranspose ) {
    nv1  = velem->GetOrder(1)+1;
    nv2  = velem->GetOrder(2)+1;
    nt1  = pelem->GetOrder(2)+1;
    nt2  = velem->GetOrder(2)+1;
    nvn1 = pelem->GetOrder(1)+1;
    nvn2 = pelem->GetOrder(2)+1;
  }
  else {
    nv1  = pelem->GetOrder(1)+1;
    nv2  = pelem->GetOrder(2)+1;
    nt1  = velem->GetOrder(2)+1;
    nt2  = pelem->GetOrder(2)+1;
    nvn1 = velem->GetOrder(1)+1;
    nvn2 = velem->GetOrder(2)+1;
  }

//*vn = 0.0;
  if ( icomp == 1 ) {
    // Do I2 X D1 term (1-derivative): 
    I    = velem->GetInterpOp   (2,!bTranspose); // already includes weights
    D    = velem->GetInterpDeriv(1, bTranspose); // already includes weights
    g  = 0.5 * L2 * pConst;
    MTK::D2_X_D1(*D, *I, *v, nv1, nv2, vtmp, nt1, nt2, *vn, nvn1, nvn2);
#if 0
cout << "StokesOp: bTrans=" << bTranspose << endl;
cout << "StokesOp: I =" << *I << endl;
cout << "StokesOp: D =" << *D << endl;
cout << "StokesOp: v =" << *v << endl;
cout << "StokesOp: vn=" << *vn << endl;
exit(1);
#endif
  }
  else {
    // Do D2 X I1 term (2-derivative): 
    I    = velem->GetInterpOp   (1, bTranspose); // already includes weights
    D    = velem->GetInterpDeriv(2,!bTranspose); // already includes weights
    g = 0.5 * L1 * pConst;
    MTK::D2_X_D1(*I, *D, *v, nv1, nv2, vtmp, nt1, nt2, *vn, nvn1, nvn2);
//  MTK::D2_X_D1(*I, *D, *v, vtmp, *vn);
  }

  if ( g != 1.0 ) MTK::fvec_const_prod_rep(*vn,g);


  return TRUE;
} // end of method RectQuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL StokesOp::TriangleOp(GVector *v, GVector *vn)
{
  return FALSE;
} // end of method TriangleOp
