//************************************************************************************//
// Module       : uzawalapop.cpp
// Date         : 2/26/03 (DLR)
// Copyright    : 2003-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the Laplacian operator derived from the Uzawa
//                decompoosition of the discrete form of the _steady_ 
//                Stokes equations where the pressure resides
//                on a different grid than the velocity, within
//                the same element. It is intended to act on the
//                p-quantity in the p Div v term. Its form is
//                  D1 L^-1 D1^T  + D2 L^-1 D2^T
//                where L is the discrete Laplacian, and Di are the
//                weak pdV derivative operators defined in StokesOp.
//                velem and pelem are the elements where the
//                v-quantity  and p-quantity reside, respectively
//
//
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include "uzawalapop.hpp"
#include "diagop.hpp"
#include "defquad2d.hpp"
#include "rectquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
UzawaLapOp::UzawaLapOp()
: LinOp(),
mConst     (1.0),
ibdy       (NULL),
velem      (NULL),
pelem      (NULL),
vtmpp0     (NULL),
vtmpv0     (NULL),
vtmpv1     (NULL),
mask       (NULL),
iL         (NULL),
pD1        (NULL),
pD2        (NULL),
pDT1       (NULL),
pDT2       (NULL)
{
   vbdy[0] = vbdy[1] = vbdy[2] = NULL;
   vtmpp0 = new GVector();
   vtmpv0 = new GVector();
   vtmpv1 = new GVector();
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
UzawaLapOp::UzawaLapOp(Elem2D *ve, Elem2D *pe)
: LinOp( ), 
mConst     (1.0),
ibdy       (NULL),
velem      (ve),
pelem      (pe),
vtmpp0     (NULL),
vtmpv0     (NULL),
vtmpv1     (NULL),
mask       (NULL),
iL         (NULL),
pD1        (NULL),
pD2        (NULL),
pDT1       (NULL),
pDT2       (NULL)
{
   vtmpp0 = new GVector();
   vtmpv0 = new GVector();
   vtmpv1 = new GVector();
   Init (ve, pe); 
   vbdy[0] = vbdy[1] = vbdy[2] = NULL;

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Destructor
UzawaLapOp::~UzawaLapOp()
{
  DeleteDynamic();
}


//************************************************************************************
// METHOD     : Init
// DESCRIPTION: Initializes operators and data
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void UzawaLapOp::Init(Elem2D *ve, Elem2D *pe)
{
  GINT  NNv, NNp;

  velem = ve;
  pelem = pe;

  // Initialize the interpolated quantities:
  velem->SetInterpBasis(pelem->GetBasisObj(1), pelem->GetBasisObj(2));
 
  // Instantiate the required operators:
  iL   = new iLaplacianOp(velem);
  if ( mask ) iL->SetMask(mask);
  pD1  = new StokesOp(velem,pelem,1);
  pD2  = new StokesOp(velem,pelem,2);

  pDT1 = new StokesOp(velem,pelem,1);
  pDT2 = new StokesOp(velem,pelem,2);
  pDT1->Transpose();
  pDT2->Transpose();

  NNv = (velem->GetOrder(1)+1)*(velem->GetOrder(2)+1);
  NNp = (pelem->GetOrder(1)+1)*(pelem->GetOrder(2)+1);
  vtmpp0->Resize(NNp); 
  vtmpv0->Resize(NNv);
  vtmpv1->Resize(NNv);

}

//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constant
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void UzawaLapOp::SetConst(GDOUBLE c)
{
  mConst = c;
} // end of method SetConst


//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated element
// ARGUMENTS  : Elem2D *
// RETURNS    : none
//************************************************************************************
void UzawaLapOp::SetElem(Elem2D *ve, Elem2D *pe)
{
   if ( ve != velem || pe != pelem )
   {
     DeleteDynamic();
     Init (ve, pe); 
   }

} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetMask
// DESCRIPTION: Sets mas
// ARGUMENTS  : 
// RETURNS    : none
//************************************************************************************
void UzawaLapOp::SetMask(GVector *m)
{
  mask = m;
  iL->SetMask(mask);
} // end of method SetMask


#if 1
//************************************************************************************
// METHOD     : SetVBdyData
// DESCRIPTION: Sets boundary conditions associated with V-components.
//              These bcs are s.t. during the mat/vec product, 
//                M^-1 D_i^T p = vb_i, for component i, where
//                p is the input vector multiplying the operator.
// ARGUMENTS  : idir   : component whose bcs are being specified
//              vb     : boundary values
//              indices: indices of local element for which vb is set. There
//                       must be the same number of these as the dimension of
//                       vb, or the method will seg-fault.
// RETURNS    : none
//************************************************************************************
void UzawaLapOp::SetVBdyData(GINT  idir, GVector *vb, GINT  *indices)
{  

  if ( idir < 1  || idir > 2 ) { 
    cout << "UzawaLapOp::SetVBdyData: Warning: invalid component specified" << endl;
    return;
  }
  if ( vb && !indices || !vb && indices ) { 
    cout << "UzawaLapOp::SetVBdyData: Bdy values and indices required" << endl;
    exit(1);
  }
  vbdy[idir-1] = vb;
  ibdy = indices;

} // end of method SetVBdyData
#endif


//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: 
// ARGUMENTS  : none
// RETURNS    : none
//************************************************************************************
void UzawaLapOp::DeleteDynamic()
{
  if ( iL    != NULL ) delete iL;     iL    = NULL;
  if ( pD1   != NULL ) delete pD1;    pD1   = NULL;
  if ( pD2   != NULL ) delete pD2;    pD2   = NULL;
  if ( pDT1  != NULL ) delete pDT1;   pDT1  = NULL;
  if ( pDT2  != NULL ) delete pDT2;   pDT2  = NULL;
  if ( vtmpp0 != NULL ) delete vtmpp0;  vtmpp0 = NULL;
  if ( vtmpv0 != NULL ) delete vtmpv0;  vtmpv0 = NULL;
  if ( vtmpv1 != NULL ) delete vtmpv1;  vtmpv1 = NULL;
} // end of method DeleteDynamic


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
GVector  UzawaLapOp::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of operator \* 


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector  args.
// RETURNS    : none
void UzawaLapOp::OpVec_prod(GVector &q, GVector &ret)  
{
  if ( velem == NULL || pelem == NULL ) 
  {
    cout << "UzawaLapOp::OpVec_prod: Invalid elements or vector components" << endl;
    exit(1);
  }

  if ( velem->ElemType() != pelem->ElemType() )
  {
    cout << "UzawaLapOp::OpVec_prod: Scalar and vector have imcompatible elements" << endl;
    exit(1);
  }
  if ( iL == NULL || pD1  == NULL || pD2  == NULL || 
                     pDT1 == NULL || pDT2 == NULL  )  
  {
    cout << "UzawaLapOp::OpVec_prod: Derivative operators improperly uninstantiated" << endl;
    exit(1);
  }

  GBOOL    bRet = TRUE;

  switch (velem->ElemType())
  {
    case DEFORMED_QUAD:
    case RECT_QUAD:
      bRet = QuadOp(&q,&ret);
      break;
    case TRIANGULAR:
      bRet = TriangleOp(&q,&ret);
      break;
    default:
      cout << "UzawaLapOp::OpVec_prod: Invalid element type" << endl;
      exit(1);
  }
  if ( !bRet )
  {
    cout << "UzawaLapOp::OpVec_prod: Operator product failed" << endl;
    exit(1);
  }

} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : QuadOp
// DESCRIPTION: Computes operator*vector  for a quad element.
// ARGUMENTS  : p   : vector argument (in the pressure space)
//              pn  : resulting product (in the pressure space)
// RETURNS    : TRUE on success; else FALSE
GBOOL UzawaLapOp::QuadOp(GVector *p, GVector *pn)
{

  if ( p  == NULL || pn  == NULL ) return FALSE;

//*pn =    ( (*pD1) * ( (*iL) * ((*pDT1)*(*p)) ) );
//*pn +=   ( (*pD2) * ( (*iL) * ((*pDT2)*(*p)) ) );

  // Component 1: pD1: vel->press; pD1T: press->vel : D1 L^-1 D1^T p:
  pDT1->OpVec_prod(*p     , *vtmpv0);
  iL  ->OpVec_prod(*vtmpv0, *vtmpv1);
  pD1 ->OpVec_prod(*vtmpv1, *pn    );

  // Component 2: D2 L^-1 D2^T p :
  pDT2->OpVec_prod(*p     , *vtmpv0);
  iL  ->OpVec_prod(*vtmpv0, *vtmpv1);
  pD2 ->OpVec_prod(*vtmpv1, *vtmpp0);
  
  MTK::fvec_add_rep(*pn, *vtmpp0);

  if ( mConst != 1.0 )
     MTK::fvec_const_prod_rep(*pn, mConst);


  return TRUE;
} // end of method QuadOp


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleOp
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : p   : vector argument   (in pressure space)
//              pn  : resulting product (in pressure space)
// RETURNS    : TRUE on success; else FALSE

GBOOL UzawaLapOp::TriangleOp(GVector *p, GVector *pn)
{
  return FALSE;
} // end of method TriangleOp
