//************************************************************************************//
// Module       : vectorop.cpp
// Date         : 6/4/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                defining a (collocated) vector operator object
// Derived From : 
// Modifications:
//************************************************************************************//
#include "vectorop.hpp"
#include "gradop.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
VectorOp::VectorOp()
:
bConvectiveForm_ (TRUE),
elem             (NULL)
{
  v[0] = v[1] = v[2] = NULL;
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
VectorOp::VectorOp(Elem2D *e, GVector *u1, GVector *u2, GVector *u3, GBOOL doconvect)
:
bConvectiveForm_ (doconvect),
elem             (e)
{
  GINT  NN;

  v[0] = u1;  v[1] = u2;  v[2] = u3;
} // end of constructor method (3)



//************************************************************************************
//************************************************************************************
// Destructor
VectorOp::~VectorOp()
{
}

//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets field element
// RETURNS    :
//************************************************************************************
void  VectorOp::SetElem(Elem2D *e)
{
  elem  = e;
} // end of method SetElem


//************************************************************************************
//************************************************************************************
// METHOD     : GetElem
// DESCRIPTION: Sets field element
// RETURNS    :
//************************************************************************************
Elem2D *VectorOp::GetElem()
{
  return elem;
} // end of method GetElem


//************************************************************************************
//************************************************************************************
// METHOD     : SetVec
// DESCRIPTION: Sets vector components
// RETURNS    :  
//************************************************************************************
void  VectorOp::SetVec(GVector *u1, GVector *u2, GVector *u3)
{
  v[0] = u1; v[1] = u2; v[2] = u3;
} // end of method SetVec


//************************************************************************************
//************************************************************************************
// METHOD     : Div (1)
// DESCRIPTION: Computes divergence of vector field, Div v
// RETURNS    : GVector
//************************************************************************************
GVector VectorOp::Div()
{
  GIndex   sei=v[0]->GetIndex();
  GVector vn(sei);

  Div(vn);

  return vn;

} // end of method Div


//************************************************************************************
//************************************************************************************
// METHOD     : Div (2)
// DESCRIPTION: Computes divergence of vector field, Div v
// RETURNS    : none
//************************************************************************************
void VectorOp::Div(GVector &vn)
{
  GINT    j;
  GVector *vtmp;

  if ( v[0] == NULL && v[1] == NULL && v[2] == NULL ) {
    cout << "VectorOp::Div: NULL vector field" << endl;
    exit(1);
  }
#if defined(DO_BLAS_TIMING)
  tstart = STK::Timer();
#endif
  vtmp = elem->GetTemp();
  for ( j=0; j<elem->Dim(); j++ ) {
    if ( v[j] == NULL ) continue;
    elem->Differentiate(vtmp, v[j], j+1);
    MTK::fvec_add_rep(vn, *vtmp);
  }
#if defined(DO_BLAS_TIMING)
  time_result = STK::Timer() - tstart;
#endif
  elem->TempUnlock(vtmp);
} // end of method Div (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Curl (1)
// DESCRIPTION: Computes the component of the curl specified by
//              idir parameter
// RETURNS    :
//************************************************************************************
GVector VectorOp::Curl(GINT  idir)
{
  GIndex   sei=v[0]->GetIndex();
  GVector vn(sei);

  Curl(idir, vn);

  return vn;
} // end of method Curl (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Curl (2)
// DESCRIPTION: Computes the component of the curl specified by
//              idir parameter 
// RETURNS    :
//************************************************************************************
void VectorOp::Curl(GINT  idir, GVector &vn)
{
  GVector *vtmp;

  if ( v[0] == NULL && v[1] == NULL && v[2] == NULL ) {
    cout << "VectorOp::Curl: NULL vector field" << endl;
    exit(1);
  }
  if ( idir < 0 || idir > 3 ) {
    cout << "VectorOp::Curl: Invalid component specification" << endl;
    exit(1);
  }

#if defined(DO_BLAS_TIMING)
  tstart = STK::Timer();
#endif
  vtmp = elem->GetTemp();
 
  if ( idir == 1 ) {
    if ( v[2] != NULL ) {
      elem->Differentiate(&vn, v[2], 2);
    }
    if ( v[1] != NULL && elem->Dim() > 2 ) {
      elem->Differentiate(vtmp, v[1], 3);
      MTK::fvec_sub_rep(vn,*vtmp);
    }

  }
  else if ( idir == 2 ) {
    if ( v[0] != NULL  && elem->Dim() > 2 ) {
      elem->Differentiate(&vn, v[0], 3);
    }
    if ( v[2] != NULL ) {
      elem->Differentiate(vtmp, v[2], 1);
      MTK::fvec_sub_rep(vn,*vtmp);
    }
  }
  else {
    if ( v[1] != NULL ) {
      elem->Differentiate(&vn, v[1], 1);
    }
    if ( v[0] != NULL ) {
      elem->Differentiate(vtmp, v[0], 2);
      MTK::fvec_sub_rep(vn,*vtmp);
    }
  }
#if defined(DO_BLAS_TIMING)
  time_result = STK::Timer() - tstart;
#endif
  elem->TempUnlock(vtmp);

} // end of method Curl (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Advect (1)
// DESCRIPTION: Computes the advection of quantity c
// RETURNS    :
//************************************************************************************
GVector VectorOp::Advect(GVector &c)
{
  GIndex   sei=v[0]->GetIndex();
  GVector vn(sei);

  Advect(c, vn);

  return vn;
} // end of method Advect (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Advect (2)
// DESCRIPTION: Computes the advection of quantity c, using collocation (point-wise)
//              multipication of the tranport velocity with the derivative of
//              c , returning the solution in vn:
//              vn = Sum_i  v_i * D_i * c and the D_i is in tensor prod. form, with
//              vn, and v_i given in natural ordering. Note that transport velocity is
//              set in call to method SetVec.  
// RETURNS    :
//************************************************************************************
void VectorOp::Advect(GVector &c, GVector &vn)
{
  GSHORT     i;
  GVector  *vtmp=NULL;

  if ( elem == NULL ) {
    cout << "VectorOp::Advect: NULL element data" << endl;
    exit(1);
  }
#if 0
  if ( v[0] == NULL && v[1] == NULL && v[2] == NULL ) {
    cout << "VectorOp::Advect: NULL vector field" << endl;
    exit(1);
  }
#endif
  vtmp = elem->GetTemp();
 
  vn = 0.0;
  for ( i=0; i<elem->Dim(); i++ ) {
    if ( v[i] == NULL ) continue;
    elem->Differentiate(vtmp, &c, i+1);
    MTK::fvec_point_prod_rep(*vtmp, *v[i]);
    MTK::fvec_add_rep(vn,*vtmp);
  }
  elem->TempUnlock(vtmp);

} // end of method Advect (2)

//************************************************************************************
//************************************************************************************
/**
 * METHOD     : Advect_Weak (2)
 * DESCRIPTION: Computes the (weak) advection of quantity c, using collocation (point-wise)
 *              multipication of the tranport velocity with the derivative of
 *              c , returning the solution in vn:
 *              vn = Sum_i  v_i * D_i * c and the D_i is in tensor prod. form, with
 *              vn, and v_i given in natural ordering. Note that transport velocity is
 *              set in call to method SetVec.  
 */
// RETURNS    :
//************************************************************************************
void VectorOp::Advect_Weak(GVector &c, GVector &vn)
{
  GINT     i;
  GVector  *vtmp=NULL;
  GVector  *vtmparr=NULL;

  if ( elem == NULL ) {
    cout << "VectorOp::Advect_DG: NULL element data" << endl;
    exit(1);
  }
#if 0
  if ( v[0] == NULL && v[1] == NULL && v[2] == NULL ) {
    cout << "VectorOp::Advect_DG: NULL vector field" << endl;
    exit(1);
  }
#endif

  vtmp = elem->GetTemp();
  vn = 0.0;
  for ( i=0; i<elem->Dim(); i++ ) {
    if ( v[i] == NULL ) continue;
    vtmparr = elem->GetTemp();
    elem->DifferentiateWeak(vtmp, &c, vtmparr,i+1);
    MTK::fvec_point_prod_rep(*vtmp, *v[i]);
    MTK::fvec_add_rep(vn,*vtmp);
    elem->TempUnlock(vtmparr);
  }
  elem->TempUnlock(vtmp);
} // end of method Advect_Weak (2)


//************************************************************************************
//************************************************************************************
// METHOD     : Advect (3)
// DESCRIPTION: Computes U.del u for all u components
// ARGUMENTS  : ui[] : array of pointers--must be 3 of these
//              N [] : array of pointers containing result
// RETURNS    :
//************************************************************************************
void VectorOp::Advect(GVector *ui[], GVector *N[])
{

  GINT  i;

  if ( bConvectiveForm_ ) {
    // The advecting field, U,  should be set by caller
    // using call to ::SetVec
    for ( i=0; i<3 && ui[i] && N[i]; i++ ) { 
      Advect(*ui[i], *N[i]);
    }
  }
  else {  // Is rotation form of nonlin term, less 0.5*Del(u.u)
    AdvectR(ui, N);
  }
} // end of method Advect (3)


//************************************************************************************
//************************************************************************************
// METHOD     : AdvectR (1)
// DESCRIPTION: Computes -u X (Del X u) -- rotational form for nonlinear advect term--
//              for all u components, excluding the  0.5*Del(u.u) term, which can
//              be absorbed into the pressure in Navier-Stokes.
// ARGUMENTS  : ui[] : array of pointers--must be 3 of these
//              N [] : array of pointers containing result
// RETURNS    :
//************************************************************************************
void VectorOp::AdvectR(GVector *ui[], GVector *N[])
{
  char    *serr = "VectorOp::AdvectR: ";
  GINT    i;
  GVector *dv, *vtmp;


  // The method ::SetVec need not be called for this method.
  vtmp = elem->GetTemp(); dv = elem->GetTemp(); 

#if GDIM == 2
  for ( i=0; i<3; i++ ) if ( N[i] ) *N[i] = 0.0;

  if ( ui[2] ) {
  elem->Differentiate(vtmp, ui[2], 1);         // dv_3/dx_1
  MTK::fvec_point_prod(*dv, *ui[2], *N[0]);    // N_1  =   dv_3/dx_1 u_3 +...
  MTK::fvec_point_prod(*dv, *ui[0], *N[2]);    // N_3  =  -dv_3/dx_1 u_1 +...
  MTK::fvec_const_prod_rep(*N[2], -1.0); 
  }
  
  elem->Differentiate(dv  , ui[1], 1);         // dv_2/dx_1
  elem->Differentiate(vtmp, ui[0], 2);         // dv_1/dx_2
  MTK::fvec_sub_rep(*dv, *vtmp);               // (dv_2/dx_1 - dv_1/dx_2) 
  MTK::fvec_point_prod(*dv, *ui[1], *vtmp);    // vt   =  (dv_2/dx_1 - dv_1/dx_2) u_2 
  MTK::fvec_add_rep(*N[0], *vtmp);             // N_1 +=  (dv_2/dx_1 - dv_1/dx_2) u_2 
  MTK::fvec_point_prod(*dv, *ui[0], *vtmp);    // vt   =  (dv_2/dx_1 - dv_1/dx_2) u_1 +...
  MTK::fvec_sub_rep(*N[1],*vtmp);              // N_2 -=  (dv_2/dx_1 - dv_1/dx_2) u_1

  if ( ui[2] ) {
  elem->Differentiate(dv  , ui[2], 2);         // dv_3/dx_2
  MTK::fvec_point_prod(*dv, *ui[2], *vtmp);    // vt   = dv_3/dx_2 u_3
  MTK::fvec_add_rep(*N[1], *vtmp);             // N_2 += dv_3/dx_2 u_3 
  MTK::fvec_point_prod(*dv, *ui[1], *vtmp);    // vt   = dv_3/dx_2 u_2
  MTK::fvec_sub_rep(*N[2], *vtmp);             // N_3 -= dv_3/dx_2 u_3 
  }
#elif GDIM == 3
  elem->Differentiate(dv  , ui[0], 3);         // dv_1/dx_3
  elem->Differentiate(vtmp, ui[2], 1);         // dv_3/dx_1
  MTK::fvec_sub_rep(*dv, *vtmp);               // (dv_1/dx_3 - dv_3/dx_1) 
  MTK::fvec_point_prod(*dv, *ui[2], *N[0]);    // N_1  = -(dv_1/dx_3 - dv_3/dx_1) u_3 +...
  MTK::fvec_const_prod_rep(*N[0], -1.0);
  MTK::fvec_point_prod(*dv, *ui[1], *N[2]);    // N_3  =  (dv_1/dx_3 - dv_3/dx_1) u_1 +...

  elem->Differentiate(dv  , ui[1], 1);         // dv_2/dx_1
  elem->Differentiate(vtmp, ui[0], 2);         // dv_1/dx_2
  MTK::fvec_sub_rep(*dv, *vtmp);               // (dv_2/dx_1 - dv_1/dx_2) 
  MTK::fvec_point_prod(*dv, *ui[2], *vtmp);    // vt   =  (dv_2/dx_1 - dv_1/dx_2) u_2 
  MTK::fvec_add_rep(*N[0], *vtmp);             // N_1 +=  (dv_2/dx_1 - dv_1/dx_2) u_2 
  MTK::fvec_point_prod(*dv, *ui[1], *N[1]);    // N_2  = -(dv_2/dx_1 - dv_1/dx_2) u_1 +...
  MTK::fvec_const_prod_rep(*N[1],-1.0);

  elem->Differentiate(dv  , ui[2], 2);         // dv_3/dx_2
  elem->Differentiate(vtmp, ui[1], 3);         // dv_2/dx_3
  MTK::fvec_sub_rep(*dv, *vtmp);               // (dv_3/dx_2 - dv_2/dx_3) 
  MTK::fvec_point_prod(*dv, *ui[2], *vtmp);    // vt   = (dv_3/dx_2 - dv_2/dx_3) u_3
  MTK::fvec_add_rep(*N[1], *vtmp);             // N_2  = (dv_3/dx_2 - dv_2/dx_3) u_3 
  MTK::fvec_point_prod(*dv, *ui[1], *vtmp);    // vt   = (dv_3/dx_2 - dv_2/dx_3) u_2
  MTK::fvec_sub_rep(*N[2], *vtmp);             // N_3 -= (dv_3/dx_2 - dv_2/dx_3) u_3 
#endif
  elem->TempUnlock(vtmp); elem->TempUnlock(dv);


} // end of method AdvectR (1)


//************************************************************************************
//************************************************************************************
// METHOD     : DoConvectiveForm
// DESCRIPTION: Sets flag to do convective form of advection term or not.
// RETURNS    :  
//************************************************************************************
void  VectorOp::DoConvectiveForm(GBOOL bflag)
{
  bConvectiveForm_ = bflag;
} // end of method DoConvectiveForm

