//************************************************************************************//
// Module       : pcblockjac_plap.hpp
// Date         : 1/8/03 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a block Jacobi preconditioner object to be used
//                for preconditioning the pseudo-Laplacian (pressure)
//                operator, 
//                E = D_1 M^-1 D_1^T +  D_2 M^-1 D_2^T + ...
// Derived From : LinOp.
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#include "pcblockjac_plap.hpp"
#include "rectquad2d.hpp"
#include "defquad2d.hpp"
#include "mtk.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
PCBlockJac_PLap::PCBlockJac_PLap()
:        LinOp(),
bNeed2Compute (TRUE),
bDeleteHere   (FALSE),
vN1      (0),
vN2      (0),
vNN      (0),
pN1      (0),
pN2      (0),
pNN      (0),
ibdy     (NULL),
lConst   (1.0),
velem    (NULL),
pelem    (NULL),
sop      (NULL)
{
  vbdy[0] = vbdy[1] = vbdy[2] = NULL;
} // end of constructor method (1)

//************************************************************************************
//************************************************************************************
// Constructor Method (2)
PCBlockJac_PLap::PCBlockJac_PLap(LinOp *AA)
:        LinOp(AA),
bNeed2Compute (TRUE),
bDeleteHere   (FALSE),
vN1      (0),
vN2      (0),
vNN      (0),
pN1      (0),
pN2      (0),
pNN      (0),
ibdy     (NULL),
lConst   (1.0),
velem    (NULL),
pelem    (NULL),
sop      (NULL)
{
   vbdy[0] = vbdy[1] = vbdy[2] = NULL;
} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Constructor Method (3)
PCBlockJac_PLap::PCBlockJac_PLap(Elem2D *ve, Elem2D *pe, SchurLapOp *Ein)
:        LinOp(pe),
bNeed2Compute (TRUE),
bDeleteHere   (FALSE),
vN1      (0),
vN2      (0),
vNN      (0),
pN1      (0),
pN2      (0),
pNN      (0),
ibdy     (NULL),
lConst   (1.0),
velem    (NULL),
pelem    (NULL),
sop      (Ein)
{
  vbdy[0] = vbdy[1] = vbdy[2] = NULL;
  if ( sop == NULL ) bDeleteHere = TRUE;
  SetElem(ve,pe); 
} // end of constructor method (3)


//************************************************************************************
//************************************************************************************
// Destructor
PCBlockJac_PLap::~PCBlockJac_PLap()
{
  if ( bDeleteHere && sop  != NULL ) delete sop ; 
}


//************************************************************************************
//************************************************************************************
// METHOD     : SetElem
// DESCRIPTION: Sets associated elements
// ARGUMENTS  : Elem2D * ve -- velocity element
//                       pe == pressure element
// RETURNS    : none
//************************************************************************************
void PCBlockJac_PLap::SetElem(Elem2D *ve, Elem2D *pe)
{

  if ( ve == NULL || pe == NULL  ) {
    cout << "PCBlockJac_PLap::SetElem: NULL element" << endl;
    exit(1);
  }

  if ( ve == velem && vN1 == (ve->GetOrder(1)+1) &&
                      vN2 == (ve->GetOrder(2)+1) &&
       pe == pelem && pN1 == (pe->GetOrder(1)+1) &&
                      pN2 == (pe->GetOrder(2)+1) ) return;

  velem = ve;
  vN1 = velem->GetOrder(1)+1;
  vN2 = velem->GetOrder(2)+1;
  vNN = vN1 * vN2;
  pelem = pe;
  pN1 = pelem->GetOrder(1)+1;
  pN2 = pelem->GetOrder(2)+1;
  pNN = pN1 * pN2;
  L.Resize (pNN,pNN);
  iL.Resize(pNN,pNN);
  icol.Resize(pNN); icol = 0.0;
  rcol.Resize(pNN); rcol = 0.0;
  if ( bDeleteHere ) 
  sop  = new SchurLapOp(velem,pelem);
  if ( !ComputeOp() ) {
    cout << "PCBlockJac_PLap::SetElem: ComputeOp failed" << endl;
    exit(1);
  }
  return;

} // end of method SetElem

#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : SetVBdyData
// DESCRIPTION: Sets boundary conditions associated with V-components.
//              These bcs are s.t. during the mat/vec product,
//                M^-1 D_i^T p = vb_i, for component i, where
//                p is the input vector multiplying the operator.
// ARGUMENTS  : idir   : component whose bcs are being specified
//              vb     : boundary values
//              indices: indices of local element for which vb is set. There
//                       must be the same number of these as the dimension of
//                       vb, or the method will seg-fault.
// RETURNS    : none
//************************************************************************************
void PCBlockJac_PLap::SetVBdyData(GINT  idir, GVector *vb, GINT  *indices)
{
  
  if ( idir < 1  || idir > 2 ) {
    cout << "PCBlockJac_PLap::SetVBdyData: Warning: invalid component specified" << endl;
    return;
  }
  if ( vb && !indices || !vb && indices ) {
    cout << "PCBlockJac_PLap::SetVBdyData: Bdy values and indices required" << endl;
    exit(1);
  }
  vbdy[idir-1] = vb;
  ibdy = indices;
  bNeed2Compute = vb && indices ? TRUE : FALSE;
} // end of method SetVBdydata
#endif


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeOp
// DESCRIPTION: Carries out computation of operator
// ARGUMENTS  :  
// RETURNS    : none
//************************************************************************************
GBOOL PCBlockJac_PLap::ComputeOp()
{
  GBOOL bRet ;

  if ( velem == NULL || pelem == NULL ) return FALSE;

  if ( pN1 <= 0 || pN2 <= 0 || pNN <= 0 ) return FALSE;


  switch (pelem->ElemType()) {
    case DEFORMED_QUAD:
      bRet = DefmQuadL();
      break;
    case RECT_QUAD:
      bRet = RectQuadL();
      break;
    case TRIANGULAR:
      bRet = TriangleL();
      break;
    default:
      cout << "LaplacianOp::SetElem: Invalid element type" << endl;
      exit(1);
  }

  if ( !bRet ) {
    cout << "LaplacianOp::SetElem: Operator computation failed" << endl;
    exit(1);
  }
  if ( bRet ) bNeed2Compute = FALSE;
  return bRet;

} // end of method ComputeOp


//************************************************************************************
//************************************************************************************
// METHOD     : SetConst
// DESCRIPTION: Sets multiplicative constants
// ARGUMENTS  :
// RETURNS    : none
//************************************************************************************
void PCBlockJac_PLap::SetConst(GDOUBLE clc)
{
  lConst = clc;
} // end of method SetConst


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
GVector  PCBlockJac_PLap::operator*(GVector q)
{
  GIndex   sei=q.GetIndex();
  GVector newq(sei);

  OpVec_prod(q, newq);

  return newq;

} // end of method \* oeprator


//************************************************************************************
//************************************************************************************
// METHOD     : Multiplication operation
// DESCRIPTION: Multiplies operator by right hand vector
// ARGUMENTS  : GVector with vector arg.
// RETURNS    : GVector containing product (if successful)
//************************************************************************************
void PCBlockJac_PLap::OpVec_prod(GVector &q, GVector &ret)
{

  if ( pelem == NULL || velem == NULL ) {
    cout << "PCBlockJac_PLap::operator*: NULL operator" << endl;
    exit(1);
  }

  if ( q.dim() != pNN  || ret.dim() != pNN ) {
    cout << "PCBlockJac_PLap::operator*: incompatible vector(s)" << endl;
    exit(1);
  }
  if ( bNeed2Compute && !ComputeOp() ) {
    cout << "PCBlockJac_PLap::operator*: PC computation failed" << endl;
    exit(1);
  }

 MTK::fmatvec_prod(iL, q, ret);
 if ( lConst != 1.0 ) MTK::fvec_const_prod_rep(ret, lConst);

} // end of method OpVec_prod


//************************************************************************************
//************************************************************************************
// METHOD     : DefmQuadL
// DESCRIPTION: Computes Laplacian^-1 operator  for a deformed quad element.
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_PLap::DefmQuadL()
{ 

  GSHORT       i, j;

  if ( !sop ) return FALSE;

#if 0
  sop->SetVBdyData(1, vbdy[0], ibdy);
  sop->SetVBdyData(2, vbdy[1], ibdy);
#endif

  icol = 0.0;
  for ( j=0; j<pNN; j++ ) {
    icol(j) = 1.0;
    sop->OpVec_prod(icol, rcol);
    icol(j) = 0.0;
    for ( i=0; i<pNN; i++ )
      L(i,j) = rcol(i);
  }

  // Now, invert to form preconditioner:
  return L.Inverse(iL);

} // end of method DefmQuadL


//************************************************************************************
//************************************************************************************
// METHOD     : RectQuadL
// DESCRIPTION: Computes Laplacian^-1 operator for a regular quad element.
// ARGUMENTS  : 
//             
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_PLap::RectQuadL()
{ 

  GSHORT       i, j;

  if ( !sop ) return FALSE;

  sop->SetConst(lConst);
  for ( j=0; j<pNN; j++ ) {
    icol(j) = 1.0;
    sop->OpVec_prod(icol, rcol);
    icol(j) = 0.0;
    for ( i=0; i<pNN; i++ )
      L(i,j) = rcol(i);
  }

  // Now, invert to form preconditioner:

  return L.Inverse(iL);

} // end of method RectQuadL


//************************************************************************************
//************************************************************************************
// METHOD     : TriangleL
// DESCRIPTION: Computes operator*vector  for a triangular element
// ARGUMENTS  : v   : vector argument
//              vn  : resulting product
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL PCBlockJac_PLap::TriangleL()
{
  return FALSE;
} // end of method TriangleL

