//************************************************************************************//
// Module       : guser_kovasznay.cpp
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
//************************************************************************************//
#include "gtypes.h"
#include "gaspar_t.h"
#include "guser.h"

//############################################################################################
//############################################################################################
//                                  Problem initialization:
//############################################################################################
//############################################################################################

//************************************************************************************
//************************************************************************************
// METHOD     : GUserConfig
// DESCRIPTION: User-furnished method to configure the code. This method is
//              called during setup, before GUserInit.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserConfig()
{

  // Where appropriate, set the number of quantities to solve for:
  nEvolvedFields_ = 2;

  return TRUE;
} // end of method GUserConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserInit
// DESCRIPTION: User-furnished method to initialize Kovasznay test
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserInit(int argc, char *argv[])
{
  GINT       i, j, k, nlevels, nrefine=0;
  GDOUBLE    dt0, t0;
  GBOOL      bRefining;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

  if ( bFixedTimeStep_ )
    dt_       /= (pow(2.0,(GINT)aux_));
  else
    Courant_  /= (pow(2.0,(GINT)aux_));


  nlevels             = ntimelevels_;
  bAutoTimeLevelFill_ = FALSE;
  bRefining           = TRUE;
  bDoCoarse_          = FALSE;
  dt0                 = 0.0;
  SetGlobals();
  while ( bRefining ) {
    cout << "_____________________________________________________________________________" << endl;
    cout << "                            refine_cycle=" << nrefine << " nelems=" << nelems_ << endl;
    SetAnalytic();
    time_ = 0.0;
    time_ -= dt0;
    for ( k=nlevels-1; k>=0; k-- ) {
      time_ += dt_;
      for ( i=0; i<nelems_; i++ ) {
        u1[i]->SetTimeLevel(k);
        u2[i]->SetTimeLevel(k);
//      p [i]->SetTimeLevel(k);
        *u1[i] = *(u1a[i]->GetExpCoeffs(0));
        *u2[i] = *(u2a[i]->GetExpCoeffs(0));
//      *p [i] = *(pa [i]->GetExpCoeffs(0));
        u1[i]->SetTime    (k,time_);
        u2[i]->SetTime    (k,time_);
//      p [i]->SetTime    (k,time_);
      } // end, element loop

//    InitPress1(k);
      if ( k == (nlevels-1) ) {
        GTimeStep(k);
      }
      dthist_[k] = dt_;
      dt0        = dt_;
    } // end, time level loop

    if ( !GAdaptGrid() ) {
      cout << "GUserInit:: adaption failed" << endl;
      exit(1);
    }
    bRefining = bElemListChange_;
    bElemListChange_ = FALSE;
    if ( bRefining ) {
      cout << "GUserInit: resampling on refined grid; nelems=" << nelems_ << endl;
    }
   cout << "                                                     refine_cycle=" << nrefine << endl;
   cout << "_____________________________________________________________________________" << endl;
   nrefine++;
  } // end, bRefining loop
  bElemListChange_ = TRUE;
  bDoCoarse_       = TRUE;
  GUserTimeDep(time_, 0.0);

  return TRUE;

} // end of method GUserInit
//############################################################################################
//############################################################################################


//************************************************************************************
//************************************************************************************
// METHOD     : GUserStart
// DESCRIPTION: User-furnished method to provide any information required after
//              a start or restart. For example, the method may set an analytic
//              solution, or set bdy conditions from an analytic solution, etc.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserStart()
{
  SetGlobals();

  return TRUE;

} // end of GUserStart



//************************************************************************************
//************************************************************************************
// METHOD     : GUserTimeDep
// DESCRIPTION: User-furnished method to provide any information required 
//              during time stepping. This method is called at the start 
//              of the time step, before the solver takes a step. 
//              For example, time dependent boundary conditions, 
//              or a non-steady analytic solution might be computed here.
// ARGUMENTS  : ptime: current time
//              pdt  : current time step
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTimeDep(GDOUBLE ptime, GDOUBLE pdt)
{
  GINT       i ;
  GVector    *u1v, *u1va, *du1v, *u2v, *u2va, *du2v,
             *pv , *pva , *dpv;
  Elem2D     *e;

  SetAnalytic();
  for ( i=0; i<nelems_; i++ ) {
    u1va = u1a  [i]->GetExpCoeffs(0);
    u1v  = u1   [i]->GetExpCoeffs(0);
    du1v = delu1[i]->GetExpCoeffs(0);
    u2va = u2a  [i]->GetExpCoeffs(0);
    u2v  = u2   [i]->GetExpCoeffs(0);
    du2v = delu2[i]->GetExpCoeffs(0);
    pva  = pa   [i]->GetExpCoeffs(0);
    pv   = p    [i]->GetExpCoeffs(0);
    dpv  = delp [i]->GetExpCoeffs(0);
    *du1v = (*u1va) - (*u1v);
    *du2v = (*u2va) - (*u2v);
    *dpv  = (*pva)  - (*pv);
  }
  ComputeMyStuff(0,ptime+pdt);

  return TRUE;
} // end of method GUserTimeDep


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogConfig
// DESCRIPTION: Provides user with a place to configure static and dynamic components
//              of log file.  Data variables set in the SetStaticParams call must be 
//              globalized, and this may be done by placing them in the 'guser.h' header.

// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogConfig()
{
  GBOOL bOk;
  
  bOk = glogger_.SetDynamicParamDesc("%f %f %f"
                                   , "Energy"
                                   , "Enstrophy"
                                   , "#L2_V_ERR"
                                    );
  bOk = bOk &
        glogger_.SetDynamicData(3 , &KE, &Enstrophy, &errVL2 
                                 );

  return bOk;

} // end of method GUserLogConfig


//************************************************************************************
//************************************************************************************
// METHOD     : GUserLogUpdate
// DESCRIPTION: Provides user with a place to update variables that are
//              printed in the dynamic log component of the log file.
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserLogUpdate()
{
  GINT       i, j, k, NN;
  GDOUBLE    *v, *vke, lE[2], gE[2];


  // Compute KE and Enstrophy:
  for ( k=0; k<GDIM; k++ ) pgfields_[k]->start(NULL);
  utmp.start(NULL);
  for ( i=0; i<uelems.size(); i++ ) {
    NN  = pgfields_[0]->member()->GetExpCoeffs(0)->dim();
    v   = pgfields_[0]->member()->GetExpCoeffs(0)->Data();
    vke = utmp.member()->Data();
    for ( j=0; j<NN; j++ ) vke[j] = v[j]*v[j] ;
    for ( k=1; k<GDIM; k++ ) {
      v  = pgfields_[k]->member()->GetExpCoeffs(0)->Data();
      for ( j=0; j<NN; j++ ) vke[j] += v[j]*v[j] ;
    }
    for ( k=0; k<GDIM; k++ ) pgfields_[k]->next();
    utmp.next();
  }

  GUtils::DoLocal(TRUE);  // turn off internal communication
  lE[0]  = GUtils::ComputeGlobalIntegralV(utmp, uelems);
  lE[1]  = GUtils::ComputeL2Norm(vorticity,0);
  lE[1] *= lE[1];         // because the norm takes sqrt, and we don't want this
  GUtils::DoLocal(FALSE);
  GComm::Allreduce(lE, gE, 2, GC_GDOUBLE, G_OP_SUM);

  KE        = gE[0];
  Enstrophy = gE[1];
  ComputeL2Error();
  return TRUE;
} // end of method GUserLogUpdate


//************************************************************************************
//************************************************************************************
// METHOD     : GUserTerm
// DESCRIPTION: Provids user with a place to perform some activity at the
//              end of time stepping
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GUserTerm()
{
  GINT      pe;
  ofstream  os;
  os.open("../kov.conv",ios::app);

  GUserLogUpdate(); // compute new error

  pe = uelems[0]->GetOrder(1);

  // write error to file:
  os << pe << " " << errVL2 << endl;
  os.close();
  
  return TRUE;
} // end of method GUserTerm


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeL2Error
// DESCRIPTION: Computes L2 error
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void ComputeL2Error()
{

  GINT       i, j, m;
  GDOUBLE    errnumv, errdenv, errnump, errdenp;
  GDOUBLE    *ux, *uy;

  GUserTimeDep(time_, 0.0);  
  for ( i=0; i<nelems_; i++ ) {
    ux = u1a[i]->GetExpCoeffs(0)->Data();
    uy = u2a[i]->GetExpCoeffs(0)->Data(); 
    for ( j=0; j<u1a[i]->GetExpCoeffs(0)->dim(); j++ ) {
      (*utmp[i])[j] = sqrt(ux[j]*ux[j] + uy[j]*uy[j]);
    }
  }
  errdenv = GUtils::ComputeL2Norm(utmp , uelems);

  for ( i=0; i<nelems_; i++ ) {
    ux = delu1[i]->GetExpCoeffs(0)->Data(); 
    uy = delu2[i]->GetExpCoeffs(0)->Data(); 
    for ( j=0; j<u1a[i]->GetExpCoeffs(0)->dim(); j++ ) {
      (*utmp[i])[j] = sqrt(ux[j]*ux[j] + uy[j]*uy[j]);
    }
  }
  errnumv = GUtils::ComputeL2Norm(utmp, uelems);

  errdenp = GUtils::ComputeL2Norm(pa   , 0);
  errnump = GUtils::ComputeL2Norm(delp , 0);

  errVL2 = errnumv / errdenv;
  errPL2 = errnump / errdenp;

}  // end of method ComputeL2Error


//************************************************************************************
//************************************************************************************
// METHOD     : SetGlobals
// DESCRIPTION: Method set user-global variables
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL SetGlobals()
{
  GSHORT     i;

  nu_[0] = nu_[1] = 0.025 / pow(10.0,aux0_);
  Re = 1.0/nu_[0];
cout << "SetGlobals: Re = " << Re << endl;
#if defined(NO_ADVECTION)
  Re = 0.0;
  nu_[0] = nu_[1] = 1.0e-2;
#endif
  lambda = 0.5*Re - sqrt(0.25*Re*Re + 4.0*Pi*Pi);

  u1a  .empty();
  u2a  .empty();
  pa   .empty();
  delu1.empty();
  delu2.empty();
  delp .empty();
  for ( i=0; i<nelems_; i++ ) {
    u1a  .add(1,uelems[i]);
    u2a  .add(1,uelems[i]);
    pa   .add(1,pelems[i]);
    delu1.add(1,uelems[i]);
    delu2.add(1,uelems[i]);
    delp .add(1,pelems[i]);
    vorticity.add(1,uelems[i]);
  }

  // Register user fields:
  GSetUserGrid(&uelems);
  if ( GRegisterUserFields(2, &u1a, "V1A", &u2a, "V2A") == 0 ) {
   cout << "SetGlobals: user analytic field registration failed" << endl;
   exit(1);
 }
  if ( GRegisterUserFields(2, &delu1, "delV1", &delu2, "delV2") == 0 ) {
   cout << "SetGlobals: user analytic field registration failed" << endl;
   exit(1);
 }
  if ( GRegisterUserFields(1, &vorticity, "Zeta") == 0 ) {
    cout << "SetGlobals: user field Zeta registration failed" << endl;
    exit(1);
  } 

  GSetUserGrid(&pelems);
  if ( GRegisterUserFields(2,&pa, "PA", &delp, "delP") == 0 ) {
   cout << "GUserInit: user field divV registration failed" << endl;
   exit(1);
 }

  SetAnalytic();

  return TRUE;
} // end of method SetGlobals


//************************************************************************************
//************************************************************************************
// METHOD     : SetAnalytic
// DESCRIPTION: Method sets analytic solution 
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL SetAnalytic()
{
  GSHORT     i, j, m;
  GDOUBLE    fact, x, y;
  GIBuffer   *bdyindices=NULL;
  GBTBuffer  *bdytype=NULL;
  Elem2D     *e;

  for ( i=0; i<nelems_; i++ ) {
    for ( j=0; j<u1.member(i)->GetExpCoeffs(0)->dim(); j++ ) {
      x = u1.X(i,1,j);
      y = u1.X(i,2,j);
      fact = exp(lambda*x);
      u1a(0,i,j) = 1.0-fact*cos(2.0*Pi*y);
      u2a(0,i,j) = 0.5*lambda/Pi * fact*sin(2.0*Pi*y);
    }
    for ( j=0; j<p.member(i)->GetExpCoeffs(0)->dim(); j++ ) {
      x    = p.X(i,1,j);
      y    = p.X(i,2,j);
      fact = exp(lambda*x);
      pa(0,i,j) = 0.5*(1.0-fact);
    }
//cout << "----------------------------------------------" << endl << endl;
  }

// Set DIRICHLET BC vector
  for ( i=0; i<nelems_; i++ ) {
    e = uelems[i];
    bdyindices = e->GetBdyIndices();
    bdytype    = e->GetBdyTypes  ();
    u1bdyvals[i]->Resize(bdyindices->dim());  // Remember, ubdyvals points to member data in Field
    u2bdyvals[i]->Resize(bdyindices->dim());  
    for ( j=0; j<bdyindices->dim(); j++ ) {
      m = (*bdyindices)(j);
      if ( (*bdytype)(j) == NOSLIP ) {
        u1bdyvals(i,j) = 0.0;
        u2bdyvals(i,j) = 0.0;
      }  
      if ( (*bdytype)(j) == DIRICHLET ) {
        u1bdyvals(i,j) = u1a(0,i,m);
        u2bdyvals(i,j) = u2a(0,i,m);
      }
    }
  } 

  return TRUE;
} // end of method SetGlobals


//************************************************************************************
//************************************************************************************
// METHOD     : ComputeMyStuff
// DESCRIPTION: Computes my user variables
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void ComputeMyStuff(GINT nlevel, GDOUBLE tt)
{ 
  GINT     i, j, k;
  GVector  *t, *w, *vx, *vy;
  Elem2D   *e;
  
  // Compute vorticity:
  uelems.start(NULL); utmp.start(NULL); vorticity.start(NULL);
  for ( k=0; k<GDIM; k++ ) pgfields_[k]->start(NULL);
  for ( i=0; i<nelems_; i++ ) {
    e  = uelems.member();
    w  = vorticity.member()->GetExpCoeffs(nlevel);
    vx = u1       .member()->GetExpCoeffs(nlevel);
    vy = u2       .member()->GetExpCoeffs(nlevel);
    t  = utmp     .member();
    e->Differentiate(w, vy, 1);  // dv_y/dx
    e->Differentiate(t, vx, 2);  // dv_x/dy
    MTK::fvec_sub_rep(*w, *t);   // w = dv_y/dx - dv_x/dy
    uelems.next(); utmp.next(); vorticity.next();
    for ( k=0; k<GDIM; k++ ) pgfields_[k]->next();
  } // end, element loop
  
} // end of method ComputeMyStuff



//************************************************************************************
//************************************************************************************
// METHOD     : InitPress 
// DESCRIPTION: initializes pressure for NS equations, given velocity
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void InitPress(GINT nlevel)
{ 
  GINT      i, j, k, pNN, vNN;
  Point     *pX=NULL;
  GVector   *Mp, *pv, *vrhs, *vj, *vk, *X[GDIM];
  Elem2D    *e, *pe;
  GVecList   prhs, dtmp, dsum;
  GLinOpList Lg;               // GL-Laplacian
  CG         cg(CG_STANDARD);  // linear solver

  // Create operator, rhs lists:
  for ( i=0; i<nelems_; i++ ) {
    pNN = pelems[i]->GetNumNodes();
    vNN = uelems[i]->GetNumNodes();
    Lg  .add(NULL,TRUE); Lg  [i] = new LaplacianOp(pelems[i]);
    prhs.add(NULL,TRUE); prhs[i] = new GVector(pNN);
    dsum.add(NULL,TRUE); dsum[i] = new GVector(vNN);
    dtmp.add(NULL,TRUE); dtmp[i] = new GVector(vNN);
  }

  // Compute RHS = Div.[u.Div u], and interpolate this
  // to the GL grid, and multply this interpolant by the
  // GL mass matrix:
  utmp = 0.0;
  for ( i=0; i<nelems_; i++ ) {
    e     = uelems[i];
    vrhs  = utmp[i]; 
//  e->GetTempMgr()->GetGVec(dsum);
//  e->GetTempMgr()->GetGVec(dtmp);
    for ( j=0; j<GDIM; j++ ) {
      vj = (*pgfields_[j])[i]->GetExpCoeffs(nlevel);
      *(dsum[i]) = 0.0;
      for ( k=0; k<GDIM; k++ ) {
        vk = (*pgfields_[k])[i]->GetExpCoeffs(nlevel);
        e->Differentiate(dtmp[i], vj, k+1);  
        MTK::fvec_point_prod_rep(*(dtmp[i]),*vk);
        MTK::fvec_add_rep       (*(dsum[i]),*(dtmp[i]));
      }
      e->Differentiate (dtmp[i], dsum[i], j+1);
//cout << "dtmp[" << j << "]=" << *(dtmp[i]) << endl;
      MTK::fvec_add_rep(*vrhs, *(dtmp[i]));
    }
//  e->TempUnlock(dsum); e->TempUnlock(dtmp);
  } // end, element loop

  // Interpolate vrhs to pressure grid:
  pNN   = pelems[0]->GetNumNodes();
  for ( i=0; i<nelems_; i++ ) {
    e     = uelems[i];
    pe    = pelems[i];
    vrhs  = utmp.member(); 
    Mp    = pe->GetMassMatrix();
    for ( j=0; j<GDIM; j++ ) X[j] = pe->GetSpNodes(j+1);
    if ( pelems[i]->GetNumNodes() != pNN || pX == NULL ) {
      delete [] pX;  pX = new Point [pNN];
    }
    for ( k=0; k<pNN; k++ ) {
      for ( j=0; j<GDIM; j++ ) pX[k][j] = (*X[j])[k];
    }
    e->Interp(vrhs, pX, NULL, pNN, prhs[i]);
    MTK::fvec_point_prod_rep(*prhs[i],*Mp);  // apply p-mass matrix
    pNN   = pelems[i]->GetNumNodes();
  }

  cg.SetTolerance(7e-2);
  // Set linear system to solve:
  cg(&pelems, &Lg, NULL, &ptmp, &prhs, NULL);  
  if ( !cg.SolveCG() ) {
    cout << "Initialization Pressure solve failed:" << endl;
    cout << "              CG iterations     : " << cg.GetNumIterations()  << endl;
    cout << "              CG error condition: " << cg.ErrNo()             << endl;
    cout << "              CG error          : " << cg.GetError()          << endl;
    cout << "              CG min error      : " << cg.GetMinError()       << endl;
    cout << "              CG max error      : " << cg.GetMaxError()       << endl;
    exit(1);
  }
  cout << "              CG iterations     : " << cg.GetNumIterations()  << endl;
  cout << "              CG error condition: " << cg.ErrNo()             << endl;
  cout << "              CG error          : " << cg.GetError()          << endl;
  cout << "              CG min error      : " << cg.GetMinError()       << endl;
  cout << "              CG max error      : " << cg.GetMaxError()       << endl;

  // Do final assignment to pressure field:
  for ( i=0; i<nelems_; i++ ) {
//  pv  = (*pgfields_[nEvolvedFields_-1])[i]->GetExpCoeffs(nlevel);
    pv  = p[i]->GetExpCoeffs(nlevel);
    *pv = *(ptmp[i]);
    p [i]->SetTime(nlevel,time_);

    pv  = pa[i]->GetExpCoeffs(0);
    *pv = *(ptmp[i]);
  }

} // end of method InitPress


//************************************************************************************
//************************************************************************************
// METHOD     : InitPress 
// DESCRIPTION: initializes pressure for NS equations, given velocity
// ARGUMENTS  :
// RETURNS    : 
//************************************************************************************
void InitPress1(GINT nlevel)
{ 
  GINT      i, j, k, pNN, vNN;
  Point     *pX=NULL;
  GVector   *Mp, *pv, *vrhs, *vj, *vk, *X[GDIM];
  Elem2D    *e, *pe;
  GVecList   prhs, dtmp, dsum;
  GLinOpList E, DT[GDIM];
  CG         cg(CG_STANDARD);  // linear solver

  // Create operator, rhs lists:
  for ( i=0; i<nelems_; i++ ) {
    pNN = pelems[i]->GetNumNodes();
    vNN = uelems[i]->GetNumNodes();
    E     .add(NULL,TRUE); E[i] =  new SchurLapOp(uelems[i],pelems[i]);
    ((SchurLapOp*)E[i])->DoCentralOp(FALSE);
    for ( j=0; j<GDIM; j++ ) {
      DT [j].add(NULL,TRUE); DT[j][i] = new StokesOp(uelems[i],pelems[i],j+1) ;
    }
    prhs.add(NULL,TRUE); prhs[i] = new GVector(pNN);
    dsum.add(NULL,TRUE); dsum[i] = new GVector(vNN);
    dtmp.add(NULL,TRUE); dtmp[i] = new GVector(vNN);
  }

  // Compute RHS = Div.[u.Grad u], where Div = DT_i :
  // GL mass matrix:
  utmp = 0.0;
  for ( i=0; i<nelems_; i++ ) {
    e     = uelems[i];
    vrhs  = utmp[i]; 
    for ( j=0; j<GDIM; j++ ) {
      vj = (*pgfields_[j])[i]->GetExpCoeffs(nlevel);
      *(dsum[i]) = 0.0;
      for ( k=0; k<GDIM; k++ ) {
        vk = (*pgfields_[k])[i]->GetExpCoeffs(nlevel);
        e->Differentiate(dtmp[i], vj, k+1);  
        MTK::fvec_point_prod_rep(*(dtmp[i]),*vk);
        MTK::fvec_add_rep       (*(dsum[i]),*(dtmp[i]));
      }
      DT[j][i]->OpVec_prod(*dsum[i],*ptmp[i]);
      MTK::fvec_add_rep(*prhs[i], *(ptmp[i]));
    }
  } // end, element loop

//cg.SetTolerance(7e-2);
  // Set linear system to solve:
  cg(&pelems, &E, NULL, &ptmp, &prhs, NULL);  
  if ( !cg.SolveCG() ) {
    cout << "Initialization Pressure solve failed:" << endl;
    cout << "              CG iterations     : " << cg.GetNumIterations()  << endl;
    cout << "              CG error condition: " << cg.ErrNo()             << endl;
    cout << "              CG error          : " << cg.GetError()          << endl;
    cout << "              CG min error      : " << cg.GetMinError()       << endl;
    cout << "              CG max error      : " << cg.GetMaxError()       << endl;
    exit(1);
  }
  cout << "              CG iterations     : " << cg.GetNumIterations()  << endl;
  cout << "              CG error condition: " << cg.ErrNo()             << endl;
  cout << "              CG error          : " << cg.GetError()          << endl;
  cout << "              CG min error      : " << cg.GetMinError()       << endl;
  cout << "              CG max error      : " << cg.GetMaxError()       << endl;

  // Do final assignment to pressure field:
  for ( i=0; i<nelems_; i++ ) {
//  pv  = (*pgfields_[nEvolvedFields_-1])[i]->GetExpCoeffs(nlevel);
    pv  = p[i]->GetExpCoeffs(nlevel);
    *pv = *(ptmp[i]);
    p [i]->SetTime(nlevel,time_);

    pv  = pa[i]->GetExpCoeffs(0);
    *pv = *(ptmp[i]);
    cout << "p[" << i << "]=" << *pv << endl;
  }

} // end of method InitPress

