//************************************************************************************//
// Module       : bitblock.cpp
// Date         : 3/12/02 (DLR)
// Copyright    : 2002-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a bit-block data type
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <math.h>
#include "bitblock.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
BitBlock::BitBlock(GINT  NumOfBits=0)
:
iBlock                (NULL),
BitFieldLength        (NumOfBits),
nBits                 (BITSPERBYTE*sizeof(GUSHORT )),
NumIntSubBlocks       (0)
{
  GUSHORT  iOffset = NumOfBits%(nBits) > 0 ? 1 : 0;

  NumIntSubBlocks = (NumOfBits)/(nBits) + iOffset;
  iBlock = new GUSHORT  [NumIntSubBlocks];
  memset(iBlock,0,NumIntSubBlocks*sizeof(GUSHORT ));

} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Destructor
BitBlock::~BitBlock()
{
  if ( iBlock != NULL ) 
  {
    delete [] iBlock;
    iBlock = NULL;
  }
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
BitBlock::BitBlock(const BitBlock &a)
{
  GINT  i;

  // copy member data:
  BitFieldLength = a.BitFieldLength;
  NumIntSubBlocks= a.NumIntSubBlocks;
  iBlock = new GUSHORT  [NumIntSubBlocks];
  for ( i=0; i<NumIntSubBlocks; i++ );
     *(iBlock+i) =  *(a.iBlock+i); 
   
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : operator=
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : BitBlock &
//************************************************************************************
BitBlock &BitBlock::operator=(const BitBlock &a)
{
  GINT  i;

  BitFieldLength = a.BitFieldLength;
  NumIntSubBlocks= a.NumIntSubBlocks;
  if ( iBlock != NULL) delete [] iBlock;
  iBlock = new GUSHORT  [NumIntSubBlocks];
  for ( i=0; i<NumIntSubBlocks; i++ );
     *(iBlock+i) =  *(a.iBlock+i);

  return *this;  

} // end of = operator


//************************************************************************************
//************************************************************************************
// METHOD     : () operator  
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : GUSHORT  bit value (0 or 1)
//************************************************************************************

GUSHORT  BitBlock::operator()(const GINT  j) 
{
  GINT  i=j, iOffset, ksubblock;

  // (position 0 is right-most end)

  // find subblock:
  if ( j<0 || j>BitFieldLength-1 )
  {
    cout << "BitBlock::() : invalid bit index" << endl;
    exit(1);
  }
//if ( j < 0 ) i = 0;
//if ( j > BitFieldLength-1 ) i = BitFieldLength-1;

  iOffset = (i+1)%(nBits) > 0 ? 1 : 0;
  ksubblock = (i+1) / (nBits) + iOffset - 1;
  
  return getbits_(*(iBlock+ksubblock), i-ksubblock*nBits, 1);

} // end of operator () 

//************************************************************************************
//************************************************************************************
// METHOD     : [] operator  
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : GUSHORT  bit value (0 or 1)
//************************************************************************************
  
GUSHORT  BitBlock::operator[](const GINT  j)
{ 
  GINT  i=j, iOffset, ksubblock;
  
  // (position 0 is right-most end)

  // find subblock:
  if ( j<0 || j>BitFieldLength-1 )
  {
    cout << "BitBlock::[] : invalid bit index" << endl;
    exit(1);
  }
  
  iOffset = (i+1)%(nBits) > 0 ? 1 : 0;
  ksubblock = (i+1) / (nBits) + iOffset - 1;
  
  return getbits_(*(iBlock+ksubblock), i-ksubblock*nBits, 1);

} // end of operator []


//************************************************************************************
//************************************************************************************
// METHOD     : operator<<
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : ostream &
//************************************************************************************
ostream &operator<<(ostream &str, BitBlock &a)
{

  str << "\n";

  GINT  i;
  str << " Bits: " << endl;
  for ( i=a.GetBlockSize_InBits()-1; i>=0; i-- )
    str << a(i) << "  ";
  str << endl << endl;

  str << " GUSHORT  Blocks: " << endl;
  for ( i=0; i<a.GetBlockSize_InBlocks(); i++ )
    str << *(a.GetBlock()+i) << " ";
  str << endl << endl;

  return str;
} // end of stream operator


//************************************************************************************
//************************************************************************************
// METHOD     : SetBits
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : none.
//************************************************************************************
void BitBlock::SetBits(GINT  j, const GUSHORT  val )
{
  GINT  i=j, iOffset, ksubblock;
  GUSHORT  yval=0, iret;
  
  if ( val != 0 ) yval = 1;

  // (position 0 is right--most end)

  // find subblock:
  if ( j<0 || j>BitFieldLength-1 )
  {
    cout << "BitBlock::SetBits: invalid bit index" << endl;
    exit(1);
  }
//if ( j < 0 ) i = 0;
//if ( j > BitFieldLength-1 ) i = BitFieldLength-1;

  iOffset = (i+1)%(nBits) > 0 ? 1 : 0;
  ksubblock = (i+1) / (nBits) + iOffset - 1;

  iret  = setbits_(*(iBlock+ksubblock), i-ksubblock*nBits, 1, yval);
  *(iBlock+ksubblock) = iret;

} // end of method SetBits



//************************************************************************************
//************************************************************************************
// METHOD     : getbits_
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : bits for subblock of type GUSHORT 
//************************************************************************************
GUSHORT  BitBlock::getbits_(GUSHORT  x, int p, int n)
{
  GUSHORT  iret;

  iret = ( (x >> (p-n+1)) & ~(~0 << n) ); 

  return iret;

} // end of method getbits_


//************************************************************************************
//************************************************************************************
// METHOD     : setbits_
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : subblock GUSHORT  value
//************************************************************************************
GUSHORT  BitBlock::setbits_(GUSHORT  x, int p, int n, GUSHORT  y)
{
 
  GUSHORT  ytmp = 0;
  GUSHORT  iret ;

  ytmp = getbits_(y,n-1,n);

  // (position 0 is right--most end)


  // zero n bits starting at p by building
  // mask and &'ing with x; then turn on
  // y's bits, left shifted to p:
  // zero'ing mask= ( (~0 << p+1) | ~(~0 << (p-n+1)) )

  iret =  ( ( (~0 << p+1) | ~(~0 << (p-n+1)) ) & x ) | ( ytmp << p );

  return iret;

} // end of method setbits_

//************************************************************************************
//************************************************************************************
// METHOD     : SetBlock (1)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BitBlock::SetBlock(GUSHORT   *newblock, GINT  num_subblocks)
{
  if ( newblock == NULL || num_subblocks < 1 ) return FALSE;
 
  GINT  n, i;

  n = num_subblocks;
   
  delete [] iBlock;
  iBlock = NULL;

  iBlock = new GUSHORT  [n];
  if ( iBlock == NULL ) return FALSE;

  for ( i=0; i<n; i++ )
    iBlock[i] = newblock[i];

  NumIntSubBlocks = n;
  BitFieldLength  = NumIntSubBlocks * (nBits);

  return TRUE;
} // end of method SetBlock (1)

  
//************************************************************************************
//************************************************************************************
// METHOD     : SetBlock (2)
// DESCRIPTION:
// ARGUMENTS  :
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BitBlock::SetBlock(GBuffer<GUSHORT >  *newblock, GINT  num_subblocks)
{
  if ( newblock == NULL || num_subblocks < 1 ) return FALSE;
  
  GINT  n, i;

  n = MIN(num_subblocks, newblock->dim());
    
  delete [] iBlock;
  iBlock = NULL;

  iBlock = new GUSHORT  [n];
  if ( iBlock == NULL ) return FALSE;

  for ( i=0; i<n; i++ )
    iBlock[i] = (*newblock)(i);
  
  NumIntSubBlocks = n;
  BitFieldLength  = NumIntSubBlocks * (nBits);

  return TRUE;
} // end of method SetBlock (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetBlock (3)
// DESCRIPTION: 
// Take bitspersubblock right-most bits of each of num_subblocks sublocks 
// of newblock (each sublock is size_subblocks GUSHORT s long), and contatenate
// them into a single block.
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BitBlock::SetBlock(GUSHORT   *newblock, GINT  num_subblocks, GINT  size_subblocks, GINT  bitspersubblock)
{
  if ( newblock == NULL ) return FALSE;
  if ( bitspersubblock > size_subblocks*nBits ) return FALSE;
 
  GUSHORT  iret, yval=1;
  GINT  i,j, k, ibit, iOffset, ksubblock;

  delete [] iBlock;
  iBlock = NULL;
  
  BitFieldLength = num_subblocks * bitspersubblock;
  

  iOffset = BitFieldLength%(nBits) > 0 ? 1 : 0;
  NumIntSubBlocks = (BitFieldLength)/(nBits) + iOffset;
  iBlock = new GUSHORT  [NumIntSubBlocks];
  if ( iBlock == NULL ) return FALSE;
 
  i = 0;
  for ( k=0; k<num_subblocks*size_subblocks; k+=size_subblocks )
  {
    for ( j=0; j<bitspersubblock; j++ )
    {
      ksubblock = i / nBits;
      yval      = getbits_(*(newblock+k+j/nBits), j%nBits, 1);
      ibit      = i - ksubblock*nBits;
      iret      = setbits_(*(iBlock+ksubblock), ibit, 1, yval);
      *(iBlock+ksubblock) = iret;
      i++;
    }
  }

  return TRUE;
} // end of method SetBlock (3)


//************************************************************************************
//************************************************************************************
// METHOD     : SetBlock (4) 
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL BitBlock::SetBlock(GUSHORT  *hi_block, GUSHORT  *lo_block, GINT  nbsize)
{
  if ( hi_block == NULL || lo_block == NULL ) return FALSE;

  GINT  n, i;

  n = 2*nbsize;

  delete [] iBlock;
  iBlock = NULL;

  iBlock = new GUSHORT  [n];
  if ( iBlock == NULL ) return FALSE;

  for ( i=0; i<n/2; i++ ) {
    iBlock    [i] = lo_block[i];
    iBlock[i+n/2] = hi_block[i];
  }

  NumIntSubBlocks = n;
  BitFieldLength  = NumIntSubBlocks * (nBits);

  return TRUE;
} // end of method SetBlock (4)

//************************************************************************************
//************************************************************************************
// METHOD     : HiWord
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void *BitBlock::HiWord()
{
  if ( NumIntSubBlocks%2 != 0 ) return NULL;
  return (iBlock+NumIntSubBlocks/2);
}


//************************************************************************************
//************************************************************************************
// METHOD     : LoWord
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void *BitBlock::LoWord()
{
  if ( NumIntSubBlocks%2 != 0 ) return NULL;
  return (iBlock);
}

//************************************************************************************
//************************************************************************************
// METHOD     : GetBlock 
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : value of GUSHORT  sublock
//************************************************************************************
GUSHORT  *BitBlock::GetBlock()
{
  return iBlock;
} // end of method GetBlock


//************************************************************************************
//************************************************************************************
// METHOD     : GetBlockSize_InBits
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : GINT  blocksize (num bits)
//************************************************************************************
GINT  BitBlock::GetBlockSize_InBits()
{
 
  return BitFieldLength;

} // end of method GetBlockSize_InBits


//************************************************************************************
//************************************************************************************
// METHOD     : GetBlockSize_InBlocks
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : GINT  number of GUSHORT  subblocks
//************************************************************************************
GINT  BitBlock::GetBlockSize_InBlocks()
{
 
  return NumIntSubBlocks;

} // end of method GetBlockSize_InBlocks


//************************************************************************************
//************************************************************************************
// METHOD     : Reset
// DESCRIPTION: 
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void BitBlock::Reset()
{
  memset(iBlock,0,NumIntSubBlocks*sizeof(GUSHORT ));
} // end of method Reset


//************************************************************************************
//************************************************************************************
// METHOD     : TransferBits
// DESCRIPTION: Transfer bit block to the nelems in_elements of size elem_size. 
// ARGUMENTS  : 
// RETURNS    : none.
//************************************************************************************
void BitBlock::TransferBits(void *in_elems, GINT  nelems, size_t elem_size)
{
  memset(in_elems, 0, nelems*elem_size);
  memcpy(in_elems, iBlock, MIN(NumIntSubBlocks*sizeof(GUSHORT),nelems*elem_size));
#if 0
  for ( GINT j=0; j<MIN(NumIntSubBlocks*sizeof(GUSHORT),nelems*elem_size); j++ ) {
    memcpy( (GUSHORT*)in_elems+j*sizeof(GUSHORT), iBlock+j , sizeof(GUSHORT));
  }
#endif

} // end of method TransferBits


