//************************************************************************************//
// Module       : gcutils.cpp
// Date         : 10/17/05 (DLR)
// Copyright    : 2005-2006 Copyright University Corporation for Atmospheric
//                Research
// Copyright    : 2005, Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                the namespace providing C utility funtions.
// Derived From : none
// Modifications:
//************************************************************************************//
#include <stdlib.h>
#if defined(_LINUX)
#include <string.h>
#endif
#include <strings.h>
#include <math.h>
#include "gcutils.hpp"



//************************************************************************************
//************************************************************************************
// METHOD     : swap (1)
// DESCRIPTION: do endian-swap
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GCUtils::swap(GBYTE *x, GINT  size)
{
  unsigned char        c;
  unsigned short       s;
  unsigned long        l;
  unsigned long long  ll;
  short               csz, ssz, lsz, llsz;


  csz  = sizeof     (unsigned char);
  ssz  = sizeof    (unsigned short);
  lsz  = sizeof     (unsigned long);
  llsz = sizeof(unsigned long long);

  switch ( size )
  {
    case 2: // swap two bytes
      c = *x;
      *x = *(x+1);
      *(x+1) = c;
      break;
    case 4: // swap two shorts (2-byte words)
      s = *(unsigned short *)x;
      *(unsigned short *)x = *((unsigned short *)x + 1);
      *((unsigned short *)x + 1) = s;
      swap ((GBYTE *)x, ssz);
      swap ((GBYTE*)((unsigned short *)x+1), ssz);
      break;
    case 8: // swap two longs (4-bytes words)
      l = *(unsigned long *)x;
      *(unsigned long *)x = *((unsigned long *)x + 1);
      *((unsigned long *)x + 1) = l;
      swap ((GBYTE*)x, lsz);
      swap ((GBYTE*)((unsigned long *)x+1), lsz);
      break;
    case 16: // swap two long-longs (8-byte words)
      ll = *(unsigned long long *)x;
      *(unsigned long long *)x = *((unsigned long long *)x + 1);
      *((unsigned long long *)x + 1) = ll;
      swap ((GBYTE*)x, llsz);
      swap ((GBYTE*)((unsigned long long *)x+1), llsz);
      break;
  }
}  // end of method swap (1)


//************************************************************************************
//************************************************************************************
// METHOD     : swapall 
// DESCRIPTION: do endian-swap for an array of quantities of size, size
// ARGUMENTS  :
// RETURNS    :
//************************************************************************************
void GCUtils::swapall(GBYTE *x, GINT  n, GINT  size)
{
  
  GINT  i;
  
  for ( i=0; i<n; i++ ) {
    swap(x+i*size, size);
  }

} // end of method swapall


//************************************************************************************
//************************************************************************************
// METHOD     : CheckFormat
// DESCRIPTION: Checks format specification for conformity with known types,
//              and outputs them in return array.
// ARGUMENTS  : sFormat    : return linked list of string C-types
//              format_spec: input format specification to be checked, of the form
//                           "%s %c %d %i %l %b %f". The elements of sFormat
//                           are s, c, d, i, l, b, f.
// RETURNS    : TRUE if input string is ok; else FALSE
//************************************************************************************
GBOOL GCUtils::CheckFormat(GStringList *sFormat, char *format_spec)
{
  char serrmsg_[] = "GSetParamD::CheckFormat: ";
  GINT    i, isize, ns, nf, num;
  GBOOL   bOk=TRUE;
  char    *p=NULL, c;

  if ( sFormat == NULL ) {
    cout << serrmsg_ << "NULL format buffer" << endl;
    exit(1);
  }
  if ( format_spec == NULL ) {
    cout << serrmsg_ << "NULL format spec" << endl;
    exit(1);
  }

  // Sort format string: find out how many numeric parameters, and
  // how many string parameters, also do a validity check:
  isize = strlen(format_spec);
  i = 0; ns = nf = num = 0;
  while ( (p=strchr(format_spec+i,'%')) != NULL ) {  //&& i < isize-1 ) {
    ++p;
    i = p - format_spec + 1;
    c = tolower(*p);
    if      ( c == 'c' || c == 's'               ) ns++;
    else if ( c == 'd' || c == 'l' ||
              c == 'f' || c == 'i' ||
              c == 'b'                           ) nf++;
    else {
      bOk = FALSE;
      break;
    }
    sFormat->add();
    *(sFormat->member()) = c;
    num++;
  }

  bOk = bOk && ( nf > 0 || ns > 0 );
  if ( !bOk ) {
    cout << serrmsg_ << "Invalid format string" << endl;
    return FALSE;
  }

  return TRUE;
} // end of method CheckFormat


//************************************************************************************
//************************************************************************************
// METHOD     : SetParams
// DESCRIPTION: Sets parameter descriptions and types. This is intended to
//              be used within a routine that makes use of var_args; hence
//              it requires that the call to this method be wrapped in
//              a va_start and va_end pair. The va_end call is important,
//              since the va_list, ap, has no meaning upon returning from
//              this method. Important: the va_list arg _must_ point at the
//              _first_ valid descriptor string.
//
// ARGUMENTS  : maxParams  : max number of params
//              sParamDesc : string list that is appneded to on call
//            : ap         : va_list representing the unlnown number of string
//                           descriptors.
// RETURNS    : none
//************************************************************************************
GBOOL GCUtils::SetParams(GINT maxParams, GStringList *sParamDesc, va_list ap)
{
  char serrmsg_[] = "GCUtils::SetParams: ";
  GINT    i, isize, num ;
  GBOOL   bOk=TRUE;
  char    *p=NULL, c,  stmp[FILE_NAME_MAX];

  if ( sParamDesc == NULL ) {
    cout << serrmsg_ << "NULL string buffer" << endl;
    exit(1);
  }

  // Get parameter descriptions from input:
  isize = sParamDesc->size();
  num = (ap != NULL ? 1 : 0 );
  for ( i=isize; i<isize+maxParams && num>0; i++ ) {
    num = vsnprintf(stmp,FILE_NAME_MAX-1,"%s",ap);
    sParamDesc->add();
    sParamDesc->member()->assign(stmp,strlen(stmp));
//  cout << serrmsg_ << "string[" << i << "]=" << stmp << " num=" << num << endl;
#if !defined(_LINUX) || !defined(_COMPILER_PG) || !defined(_INT64)
    p = va_arg(ap,char*);
    if ( p == NULL ) break;
#endif
  }

  return bOk;

} // end of method SetParams

