//************************************************************************************//
// Module       : glogger.cpp
// Date         : 12/11/04 (DLR)
// Copyright    : 2004-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a logging class as used in GASpAR
// Derived From : none.
// Modifications:
//************************************************************************************//
#include "glogger.hpp"
#include "gcutils.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GLogger::GLogger()
{
} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Destructor
GLogger::~GLogger()
{
  sstaticdesc_.empty();
  sdynamicdesc_.empty();
  sstaticformat_.empty();
  sdynamicformat_.empty();
  pstaticdata_.empty();
  pdynamicdata_.empty();
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
GLogger::GLogger(const GLogger &a)
{
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : SetStaticParamDesc
// DESCRIPTION: Sets descriptors for static parameters
// ARGUMENTS  : 
//              format: data type list of the form: "%d %i %c %b %f %l ..."
//              ...   : va_list consisting of descriptors for each of the
//                      format specifiers in 'format'
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::SetStaticParamDesc(char *format, ...)
{
  va_list ap;
  char    serrmsg_[] = "GLogger::SetStaticParamDesc: ";

  va_start(ap,format);
  if ( !SetParams(sstaticformat_,sstaticdesc_, format, ap) ) {
    cout << serrmsg_ << "Set of static parameters failed" << endl;
    return FALSE;
  }
  va_end(ap);

  return TRUE;

} // end of method SetStaticParamDesc


//************************************************************************************
//************************************************************************************
// METHOD     : SetDynamicParamDesc
// DESCRIPTION: Sets descriptors for dynamic parameters
// ARGUMENTS  : 
//              format: data type list of the form: "%d %i %c %b %f %l ..."
//              ...   : va_list consisting of descriptors for each of the
//                      format specifiers in 'format'
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::SetDynamicParamDesc(char *format, ...)
{
  va_list ap;
  char    serrmsg_[] = "GLogger::SetDynamicParamDesc: ";

  va_start(ap,format);
  if ( !SetParams(sdynamicformat_,sdynamicdesc_, format, ap) ) {
    cout << serrmsg_ << "Set of dynamic parameters failed" << endl;
    return FALSE;
  }
  va_end(ap);

  return TRUE;

} // end of method SetDynamicParamDesc


//************************************************************************************
//************************************************************************************
// METHOD     : SetParams
// DESCRIPTION: Utility method that sets param descriptors 
// ARGUMENTS  : 
//              flist : format list (returned)
//              dlist : descriptor list (returned)
//              format: data type list of the form: "%d %i %c %b %f %l ..."
//              ...   : va_list consisting of descriptors for each of the
//                      format specifiers in 'format'
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::SetParams(GStringList &flist, GStringList &dlist, char *format, va_list ap) 
{
  char    serrmsg_[] = "GLogger::SetParams: ";
  GINT    maxnewparams;

  if ( format == NULL ) {
    cout << serrmsg_ << "NULL format string" << endl;
    return FALSE;
  }

  // Sort format string: find out how many numeric parameters, and
  // how many string parameters:
  if ( !GCUtils::CheckFormat(&flist, format) ) {
    cout << serrmsg_ << "Invalid format string" << endl;
    exit(1);
  }
  maxnewparams = flist.size()-dlist.size();
  if ( !GCUtils::SetParams(maxnewparams, &dlist, ap) ) {
    cout << serrmsg_ << "Bad descriptor buffer specification" << endl;
    return FALSE;
  }

  if ( dlist.size() != flist.size() ) {
    cout << serrmsg_ << "Number of descriptors does not match number of format specs" << endl;
    return FALSE;
  }

  return TRUE;

} // end of method SetParams


//************************************************************************************
//************************************************************************************
// METHOD     : SetStaticData
// DESCRIPTION: Method to set static data fields
// ARGUMENTS  : 
//              nfields: number of fields being set
//              ...    : va_list of form (char*)v1, (char*)v2, ... (pointers to fields)
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::SetStaticData(GINT nfields, ...)
{
  va_list ap;
  char    serrmsg_[] = "GLogger::SetStaticData: ";

  va_start(ap,nfields);
  if ( !SetData(nfields, pstaticdata_, ap) ) {
    cout << serrmsg_ << "Could not register static data" << endl;
    exit(1);
  }
  va_end(ap);

  return TRUE;
  
} // end of method SetStaticData


//************************************************************************************
//************************************************************************************
// METHOD     : SetDynamicData
// DESCRIPTION: Method to set dynamic data record fields
// ARGUMENTS  : 
//              nfields: number of fields being set
//              ...    : va_list of form (char*)v1, (char*)v2, ... (pointers to fields)
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::SetDynamicData(GINT nfields, ...)
{
  va_list ap;
  char    serrmsg_[] = "GLogger::SetDynamicData: ";

  va_start(ap,nfields);
  if ( !SetData(nfields, pdynamicdata_, ap) ) {
    cout << serrmsg_ << "Could not register dynamic data" << endl;
    exit(1);
  }
  va_end(ap);
   
  return TRUE;

} // end of method SetDynamicData


//************************************************************************************
//************************************************************************************
// METHOD     : SetData
// DESCRIPTION: Utility method to set data fields
// ARGUMENTS  : 
//              nfields: number of fields being set
//              plist  : pointer list that contains char pointers to variables
//              ap     : va_list consisting of char* pointers for each field variable.
//                       Caller must wrap call in va_start/va_end pair, and 
//                       make sure that ap starts at the first valid variable desired.
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::SetData(GINT nfields, GpCList &plist, va_list ap)
{
  char    serrmsg_[] = "GLogger::SetData: ";
  GINT    i, isize;
  char    *p=NULL, c;
  GBOOL   bOk=TRUE;

  isize = plist.size();
  for ( i=isize,p=(char*)ap; i<isize+nfields; i++ ) {
    if ( p == NULL ) {
      bOk = FALSE;
      break;
    }
    plist.add();
//  plist[i] = (char*)ap;
    c = sstaticformat_[i][ sstaticformat_[i].length()-1];
    switch(c) {
      case 'd':
      case 'i':
      case 'l':
        plist[i] = (char*)va_arg(ap,GINT *);
        break;
      case 'b':
        plist[i] = (char*)va_arg(ap,GBOOL *);
        break;
      case 'f':
        plist[i] = (char*)va_arg(ap,GDOUBLE *);
        break;
      case 's':
        plist[i] = (char*)va_arg(ap,char *);
        break;
      default:
        cout << serrmsg_ << "Bad data type" << endl;
        exit(1);
    }
  }

  if ( !bOk ) {
    cout << serrmsg_ << "insufficient number of data items specified in call" << endl;
  }

  return bOk;

} // end of method SetStaticData


//************************************************************************************
//************************************************************************************
// METHOD     : PutStaticData (1)
// DESCRIPTION: Method to put dynamic data record to file
// ARGUMENTS  : 
//              filename: file to write to
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::PutStaticData(char *filename)
{
  char     serrmsg_[] = "GLogger::PutStaticData (1): ";
  ofstream out;
  GBOOL    bOk;

  if ( filename == NULL ) {
    cout << serrmsg_ << "NULL filename " << endl;
    exit(1);
  }
  
  out.open(filename,ios::app);
  bOk = PutStaticData(out);
  out.close();
 
  return bOk;
 
} // end of method PutStaticData (1)


//************************************************************************************
//************************************************************************************
// METHOD     : PutStaticData (2)
// DESCRIPTION: Method to put static data to stream
// ARGUMENTS  : 
//              out: ostream to write to
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::PutStaticData(ostream &out)
{
  char    serrmsg_[] = "GLogger::PutStaticData (2): ";
  GINT    i, isize, lmax;
  char    c ;
  GBOOL   bOk=TRUE;
  string  stmp;

  if ( sstaticdesc_.size() != sstaticformat_.size() ||
       sstaticdesc_.size() != pstaticdata_  .size() ) {
    cout << serrmsg_ << "number of data descriptors does not match number of variables" << endl;
    exit(1);
  }
  
  isize = sstaticdesc_.size();   
  for ( i=0, lmax=0; i<isize; i++ ) {
    lmax = MAX(lmax,sstaticdesc_[i].length());
  }
  stmp.assign(lmax+2,' ');
  
  for ( i=0; i<isize; i++ ) {
//  stmp.assign(sstaticdesc_[i],1,sstaticdesc_[i].length());
    c = sstaticformat_[i][ sstaticformat_[i].length()-1];
    out << sstaticdesc_[i].c_str() << ": " ;
    switch(c) {
      case 'd':
      case 'i':
      case 'l':
        out << (GINT)(*((GINT*)pstaticdata_[i]));
        break;
      case 'b':
        out << (GBOOL)(*((GBOOL*)pstaticdata_[i]));
        break;
      case 'f':
        out << (GDOUBLE)(*((GDOUBLE*)pstaticdata_[i]));
        break;
      case 's':
        out << (char*)pstaticdata_[i];
        break;
      default:
        cout << serrmsg_ << "Bad data type" << endl;
        exit(1);
    }
    out << endl;
  }

  return bOk;

} // end of method PutStaticData (2)


//************************************************************************************
//************************************************************************************
// METHOD     : PutDynamicData (1)
// DESCRIPTION: Method to put dynamic data record to file
// ARGUMENTS  : 
//              filename: file to write to
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::PutDynamicData(char *filename) 
{
  char     serrmsg_[] = "GLogger::PutDynamicData (1): ";
  ofstream out;
  GBOOL    bOk;

  if ( filename == NULL ) {
    cout << serrmsg_ << "NULL filename " << endl;
    exit(1);
  }

  out.open(filename,ios::app);
  bOk = PutDynamicData(out);
  out.close();

  return bOk;

} // end of method PutDynamicData (1)


//************************************************************************************
//************************************************************************************
// METHOD     : PutDynamicData (2)
// DESCRIPTION: Method to put dynamic data record to stream
// ARGUMENTS  : 
//              out: ofnstream to write to
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::PutDynamicData(ostream &out)
{
  char    serrmsg_[] = "GLogger::PutDynamicData (2): ";
  GINT    i, isize;
  char    c ;
  GBOOL   bOk=TRUE;

  if ( sdynamicdesc_.size() != sdynamicformat_.size() ||
       sdynamicdesc_.size() != pdynamicdata_  .size() ) {
    cout << serrmsg_ << "number of data descriptors does not match number of variables" << endl;
    exit(1);
  }
  
  isize = sdynamicdesc_.size();   
  out.width(20);
  out.precision(10);

  for ( i=0; i<isize; i++ ) {
    c = sdynamicformat_[i][ sdynamicformat_[i].length()-1];
    switch(c) {
      case 'd':
      case 'i':
      case 'l':
        out << (GINT)(*((GINT*)pdynamicdata_[i]));
        break;
      case 'b':
        out << (GINT)(*((GBOOL*)pdynamicdata_[i]));
        break;
      case 'f':
        out << (GDOUBLE)(*((GDOUBLE*)pdynamicdata_[i]));
        break;
      case 's':
        out << (char*)pdynamicdata_[i];
        break;
      default:
        cout << serrmsg_ << "Bad data type" << endl;
        exit(1);
    }
    out << "  " ;
  }
  out << endl;

  return bOk;

} // end of method PutDynamicData (2)

//************************************************************************************
//************************************************************************************
// METHOD     : PutHeader (1)
// DESCRIPTION: Method to put dynamic data record header to file
// ARGUMENTS  :
//              filename: file to write to
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::PutHeader(char *filename)
{
  char     serrmsg_[] = "GLogger::PutHeader (1): ";
  ofstream out;
  GBOOL    bOk;

  if ( filename == NULL ) {
    cout << serrmsg_ << "NULL filename " << endl;
    exit(1);
  }

  out.open(filename,ios::app);
  bOk = PutHeader(out);
  out.close();

  return bOk;

} // end of method PutHeader (1)


//************************************************************************************
//************************************************************************************
// METHOD     : PutHeader (2)
// DESCRIPTION: Method to put dynamic data record header to stream
// ARGUMENTS  : 
//              out: ostream to write to
// RETURNS    : TRUE if set successful; else FALSE
//************************************************************************************
GBOOL GLogger::PutHeader(ostream &out)
{
  char    serrmsg_[] = "GLogger::PutHeader (2): ";
  GINT    i, isize;
  GBOOL   bOk=TRUE;

  if ( sdynamicdesc_.size() != sdynamicformat_.size() ||
       sdynamicdesc_.size() != pdynamicdata_  .size() ) {
    cout << serrmsg_ << "number of data descriptors does not match number of variables" << endl;
    exit(1);
  }
  
  isize = sdynamicdesc_.size();   

  for ( i=0; i<isize; i++ ) {
    out << sdynamicdesc_[i].c_str() << "    ";
  }
  out << endl;

  return bOk;

} // end of method PutHeader (2)
