//************************************************************************************//
// Module       : gmemmgr.cpp
// Date         : 10/11/04 (DLR)
// Copyright    : 2004-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a GASpAR element-based memory manager. This class provides
//                access to temp space allocated to be the size of the element member
//                data's data segment size by default. A new vector memory space when
//                allocated, is wrapped by a semaphore so that it cannot be accessed
//                again, until the memory space is made available again. This is
//                accomplished by a call to the method Unlock.
// Derived From : none.
// Modifications:
//************************************************************************************//
#include <math.h>
#include <string.h>
#include "gmemmgr.hpp"
#include "gcomm.hpp"
#include "gtbuffer.hpp"

//************************************************************************************
//************************************************************************************
// Constructor Method (1)
GMemMgr::GMemMgr(GINT numvecs)
:
NN_                   (0)
{
  GINT i;
  AddBlocks(numvecs);

} // end of constructor (1) method


//************************************************************************************
//************************************************************************************
// Destructor
GMemMgr::~GMemMgr()
{
  gveclist_.empty();
  vlocked_.empty();  
}


//************************************************************************************
//************************************************************************************
// Copy constructor method
GMemMgr::GMemMgr(const GMemMgr &a)
{
} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// METHOD     : Lock 
// DESCRIPTION: Locks specified vector explicitly. 
// ARGUMENTS  : vec       : vector pointer
// RETURNS    : TRUE if lock successful; else FALSE
//************************************************************************************
GBOOL GMemMgr::Lock(GVector *vec)
{
  LinkElemTT<GDOUBLE,GTVector> *le;

  if ( vec == NULL || (le=gveclist_.find(vec))==NULL ) return FALSE;
  
  if ( vlocked_.find(vec) == NULL ) { // not in locked list...
    vlocked_.add(vec,FALSE);          // ...so add it
  }
  
  return TRUE;

} // end of method Lock 
 

//************************************************************************************
//************************************************************************************
// METHOD     : Unlock 
// DESCRIPTION: Unlocks specified vector explicitly. 
// ARGUMENTS  : vec       : vector pointer
// RETURNS    : TRUE if unlock successful; else FALSE
//************************************************************************************
GBOOL GMemMgr::Unlock(GVector *vec)
{
  LinkElemTT<GDOUBLE,GTVector> *le;

//if ( vec == NULL || (le=gveclist_.find(vec))==NULL ) return FALSE;
  if ( vec == NULL ) return FALSE;
  
  if ( vlocked_.find(vec) != NULL ) { // in locked list...
    vlocked_.del(vec);                 // ...unlock it:
  }
   
  return TRUE;

} // end of method Unlock 
 

//************************************************************************************
//************************************************************************************
// METHOD     : Delete (1)
// DESCRIPTION: Delete specified memory area (vector)
// 
// ARGUMENTS  : v      : pointer to vector to delete
// RETURNS    : none.
//************************************************************************************
void GMemMgr::Delete(GVector *v)
{

  // delete vector
  gveclist_.del(v);

} // end of method Delete (1)


//************************************************************************************
//************************************************************************************
// METHOD     : Delete (2)
// DESCRIPTION: Delete specified memory area (vector)
// 
// ARGUMENTS  : v      : pointer to vector to delete
// RETURNS    : none.
//************************************************************************************
void GMemMgr::Delete(GINT i)
{

  // delete vector
  gveclist_.del(i);

} // end of method Delete (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetGVec (1)
// DESCRIPTION: Gets a GVector memory space, and locks it. If one doesn't exist,
//              then it's created,and then locked. User is responsible for unlocking
//              this vector when done, by calling Unlock method.
// 
// ARGUMENTS  : vec       : vector pointer, set if successful; else null
// RETURNS    : TRUE if get successful; else FALSE
//************************************************************************************
GBOOL GMemMgr::GetGVec(GVector *&vec)
{
  GINT    i;

  vec = NULL;

  if ( NN_ <= 0 ) {
    cout << "GMemMgr::GetGVec: invalid memory block size" << endl;
    exit(1);
  }

  // Find first available vector index. If there isn't
  // one, add a new vector to the list:
  i = 0;
  while ( i<gveclist_.size() && vlocked_.find(gveclist_[i]) != NULL ) i++;

  // If there is a valid unlocked index, return corresponding
  // vector:
  if ( i < gveclist_.size() ) {
    vec = gveclist_[i];
    vlocked_.add(vec,FALSE);    // add vector to locked list
    return vec != NULL;
  }

  // If there is no available, unlocked index, add a new vector:
  gveclist_.add();
  vec = gveclist_.member();
  if ( vec == NULL ) return FALSE;

  vec->Resize(NN_);
  vlocked_.add(vec,FALSE);    // add new vector to locked list

  return TRUE;

} // end of method GetGVec (1)


//************************************************************************************
//************************************************************************************
// METHOD     : GetGVec (2)
// DESCRIPTION: Gets a specific GVector memory space, and locks it. If one exists,
//              and it's not locked, then it is locked and returned. If specified
//              index is locked, then 'exit' is called. If one doesn't
//              exist, then 'exit' is called.
//              User is responsible for unlocking
//              this vector when done, by calling Unlock method.
// 
// ARGUMENTS  : i      : vector mem space index
// RETURNS    : NULL on failure (or 'exit'), vector pointer on success
//************************************************************************************
GVector *GMemMgr::GetGVec(GINT i)
{
  GVector *vec=NULL;

  if ( NN_ <= 0 ) {
    cout << "GMemMgr::GetGVec: invalid memory block size" << endl;
    exit(1);
  }

  // Find first available vector index. If there isn't
  // one, add a new vector to the list:
  vec = gveclist_[i];

  while ( i<gveclist_.size() && vlocked_.find(vec) != NULL ) {
    cout << "GMemMgr::GetGVec(2): index " << i << " locked. Invalid request" << endl;
    exit(1);
  }

  // Resize vector, and lock the vector:
  vec->Resize(NN_);
  vlocked_.add(vec,FALSE);    // add new vector to locked list

  return vec;

} // end of method GetGVec (2)


//************************************************************************************
//************************************************************************************
// METHOD     : SetSize
// DESCRIPTION: Sets vector (mem space) sizes
// 
// ARGUMENTS  : n      : number of GDOUBLES in vectors (mem.spaces)
// RETURNS    : none.
//************************************************************************************
void GMemMgr::SetSize(GINT n)
{
  GVector *vec=NULL;

  NN_ = n;

  // Resize vectors
  gveclist_.start(NULL);
  while ( (vec=gveclist_.member()) != NULL ) {
    vec->Resize(NN_);
    gveclist_.next();
  }

} // end of method SetSize


//************************************************************************************
//************************************************************************************
// METHOD     : GetNumBlocks
// DESCRIPTION: gets number of memory vectors allowed for access to specific
//              vectors.
// 
// ARGUMENTS  : none.
// RETURNS    : number of vector elements currently in vector list
//************************************************************************************
GINT GMemMgr::GetNumBlocks()
{
  return gveclist_.size();

} // end of method GetNumBlocks


//************************************************************************************
//************************************************************************************
// METHOD     : GetFirstAvail
// DESCRIPTION: Gets a index of first available block
// 
// ARGUMENTS  : none
// RETURNS    : index of first available; else -1
//************************************************************************************
GINT GMemMgr::GetFirstAvail()
{
  GINT    i=0;

  while ( i<gveclist_.size() && vlocked_.find(gveclist_[i]) != NULL ) {
    i++;
  }

  return i<gveclist_.size() ? i : -1;
  
} // end of method GetFirstAvail


//************************************************************************************
//************************************************************************************
// METHOD     : GetIndex
// DESCRIPTION: Gets a index of specified block
// 
// ARGUMENTS  : vec: block (vector) pointer
// RETURNS    : index if found; else -1
//************************************************************************************
GINT GMemMgr::GetIndex(GVector *&vec)
{
  GINT    i=0;

  while ( i<gveclist_.size() && vec != gveclist_[i] ) i++;

  return i<gveclist_.size() ? i : -1;

} // end of method GetIndex



//************************************************************************************
//************************************************************************************
// METHOD     : AddBlocks
// DESCRIPTION: Adds n additional blocks of available space to mgr
// 
// ARGUMENTS  : n      : number of new blocks
// RETURNS    : TRUE on success; else FALSE
//************************************************************************************
GBOOL GMemMgr::AddBlocks(GINT n)
{
  GINT i;

  for ( i=0; i<n; i++ ) {
    gveclist_.add();
  }

  return TRUE;
} // end of method AddBlocks
