//************************************************************************************//
// Module       : gpobjbuffer.cpp
// Date         : 7/9/01 (DLR)
// Copyright    : 2001-2006 Copyright University Corporation for Atmospheric
//                Research
// Description  : Encapsulates the methods and data associated with
//                a buffer template  object holding GASpAR object pointers
// Derived From :
// Modifications:
//************************************************************************************//
#include <memory.h>
#include "gpobjbuffer.hpp"


//************************************************************************************
//************************************************************************************
// Constructor Method (1)
template<class TBUFF> GpObjBuffer<TBUFF>::GpObjBuffer<TBUFF>()
:
n                     (0),
data                  (NULL)
{
} // end of constructor method (1)


//************************************************************************************
//************************************************************************************
// Constructor Method (2)
//************************************************************************************
template<class TBUFF> GpObjBuffer<TBUFF>::GpObjBuffer<TBUFF>(GINT  size)
:
n                     (MAX(size,0)),
data                  (NULL)
{
  if ( n > 0 ) {
    data = new TBUFF * [n];
  }

} // end of constructor method (2)


//************************************************************************************
//************************************************************************************
// Copy constructor method
//************************************************************************************
template<class TBUFF> GpObjBuffer<TBUFF>::GpObjBuffer<TBUFF>(const GpObjBuffer<TBUFF> &a)
{
   GINT i;

   // copy member data:
   n    = a.n;
   if ( n > 0 ) data = new TBUFF *  [a.n];

   //  copy array data:
   for ( i=0; i<n; i++ ) data[i] = a.data[i];

} // end of copy constructor method


//************************************************************************************
//************************************************************************************
// Destructor
//************************************************************************************
template<class TBUFF> GpObjBuffer<TBUFF>::~GpObjBuffer<TBUFF>()
{
   DeleteDynamic();
}


//************************************************************************************
//************************************************************************************
// Assignment operator method 
//************************************************************************************
template<class TBUFF> void GpObjBuffer<TBUFF>::operator=(GpObjBuffer<TBUFF> &a)
{
  GINT i;

  if ( this != &a ) 
  {
    if ( a.n == 0 || a.data == NULL ) return;
    if ( a.n != n ) 
    {
      cout << "GpObjBuffer::=: incompatible assignment" << endl;
      return;
    }


    //  copy array data:
    for ( i=0; i<n; i++ ) data[i] = a.data[i];
  }

} // end = operator


//************************************************************************************
//************************************************************************************
// Assignment operator method
//************************************************************************************
template<class TBUFF> void GpObjBuffer<TBUFF>::operator=(TBUFF  *a)
{
  GINT  i;

  if ( data == NULL && n > 0 )
  {
    cout << "GpObjBuffer::=: illegal assignment" << endl;
    exit(1);
  }

  //  copy array data:
  for ( i=0; i<n; i++ )
  {
    data[i] = a;
  }

} // end = operator


//************************************************************************************
//************************************************************************************
// METHOD     : dim
// DESCRIPTION:
// RETURNS    :  size of vector
//************************************************************************************
template<class TBUFF> GINT  GpObjBuffer<TBUFF>::dim() const
{
  return n ;
} // end of method dim


//************************************************************************************
//************************************************************************************
// METHOD     : Data
// DESCRIPTION: 
// RETURNS    :  pointer to the array data area
//************************************************************************************
template<class TBUFF> TBUFF **GpObjBuffer<TBUFF>::Data()
{
  return data;
} // end of method Data


//************************************************************************************
//************************************************************************************
// METHOD     : Resize
// DESCRIPTION: resizes dynamically allocated quantities
//              if required
// RETURNS    :  TRUE on success, else FALSE
//************************************************************************************
template<class TBUFF> GBOOL GpObjBuffer<TBUFF>::Resize(GINT  newSize)
{
  GINT  j;
  GBOOL bRet = FALSE;

  if ( newSize == n && n > 0 ) return TRUE;

  if ( data != NULL ) {
    delete [] data;
    data = NULL;
  }
  if ( newSize == 0 ) {
    n = newSize;
    return TRUE;
  }
  if ( newSize <  0 ) {
    cout << "GpObjBuffer<TBUFF>::Resize: invalid dimension" << endl;
    exit(1);
  }
  
  data = new TBUFF * [newSize];
  bRet = FALSE;
  if ( data != NULL ) {
    for ( j=0; j<newSize; j++ ) data[j] = NULL;
    n = newSize;
    bRet = TRUE;
  }
  return bRet;

} // end of method Resize


//************************************************************************************
//************************************************************************************
// METHOD     : contains (1)
// DESCRIPTION: Determines if candidate member is in the buffer, and if so,
//              sets the index where it is found. Search continues until all
//              values have been checked.
// RETURNS    :  TRUE if member is in list, else FALSE. On TRUE, set index to
//               index where member is found first; else set index = -1.
//************************************************************************************
template<class TBUFF> GBOOL GpObjBuffer<TBUFF>::contains(TBUFF *imember, GINT  &index)
{
  GINT  i=0;

  if ( data == NULL ) return FALSE;

  index = -1;
  while ( i < n && data[i] != imember ) i++;

  if ( i >= n ) return FALSE;

  index = i;

  return TRUE;

} // end of method contains (1)


//************************************************************************************
//************************************************************************************
// METHOD     : contains (2)
// DESCRIPTION: Determines if candidate member is in the buffer, and if so,
//              sets the index where it is found. Search continues until index
//              is found s.t. data <= floor. Search begins at istart, and continues
//              at most num indices.
// RETURNS    :  TRUE if member is in list, else FALSE. On TRUE, set index to
//               index where member is found first; else set index = -1.
//************************************************************************************
template<class TBUFF> GBOOL GpObjBuffer<TBUFF>::contains(TBUFF *imember, GINT  istart, GINT  num, 
                                                      GINT  &index)
{
  GINT  i=istart;

  if ( data == NULL ) return FALSE;

  index = -1;
  while ( i < (istart+num) && i < n && data[i] != imember ) i++;

  if ( i >= (istart+num) || i >= n ) return FALSE;

  index = i;

  return TRUE;

} // end of method contains (2)


//************************************************************************************
//************************************************************************************
// METHOD     : GetSection (1)
// DESCRIPTION: Gets buffer section using argument indices
// RETURNS    : reference to SEBiffer<TBUFF> buffer created from specified indices
//************************************************************************************
template<class TBUFF> GpObjBuffer<TBUFF> &GpObjBuffer<TBUFF>::GetSection(GINT  *ilist, GINT  nlist)
{
  GpObjBuffer<TBUFF> *tbuff;

  if ( data != NULL ) delete [] data ;
//ldim = ilist == NULL ? n : nlist;
  tbuff = new GpObjBuffer<TBUFF>(0);
  GetSection(*tbuff,ilist,nlist);

  return *tbuff;

} // end of method GetSection (1)


//************************************************************************************ 
//************************************************************************************
// METHOD     : GetSection (2)
// DESCRIPTION: Gets buffer section using argument indices
// RETURNS    : none.
//************************************************************************************
template<class TBUFF> void GpObjBuffer<TBUFF>::GetSection(GpObjBuffer<TBUFF> &tbuff, GINT  *ilist, GINT  nlist)
{
  GINT  i, ldim;

  if ( data != NULL ) delete [] data ;
  ldim = ilist == NULL ? n : nlist; 

  tbuff.Resize(ldim); 
  if ( ilist == NULL ) {
    for ( i=0; i<ldim; i++ ) tbuff[i] = data[i];
  }
  else {
    for ( i=0; i<ldim; i++ ) tbuff[i] = data[ilist[i]];
  }

} // end of method GetSection (2)


//************************************************************************************
//************************************************************************************
// METHOD     : DeleteDynamic
// DESCRIPTION: deletes dynamically allocated quantities
// RETURNS    :  none
//************************************************************************************
template<class TBUFF> void GpObjBuffer<TBUFF>::DeleteDynamic()
{

  if ( data != NULL ) delete [] data ;
  data = NULL;
  return;

} // end of method DeleteDynamic


#if 0
//************************************************************************************
//************************************************************************************
// METHOD     : operator <<
// DESCRIPTION: output buffer of <TBUFF>
// RETURNS    :
//************************************************************************************
template<class TBUFF> ostream &operator<<(ostream &str, GpObjBuffer<TBUFF> &a)
{

  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << (TBUFF*)a[i] << "  ";

  return str;
} // end of << operator <TBUFF>

#else
//************************************************************************************
//************************************************************************************
// METHOD     : operator <<
// DESCRIPTION: output buffer of <GLLBasis>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GpObjBuffer<GLLBasis> &a)
{

  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << (GLLBasis*)a[i] << "  ";

  return str;
} // end of << operator <GLLBasis>
//************************************************************************************
//************************************************************************************
// METHOD     : operator <<
// DESCRIPTION: output buffer of <void>
// RETURNS    :
//************************************************************************************
ostream &operator<<(ostream &str, GpObjBuffer<void> &a)
{

  GINT  i;
  for ( i=0; i<a.dim(); i++ )
    str << (char*)a[i] << "  ";

  return str;
} // end of << operator <void>
#endif
